/*!
 *******************************************************************************
 * \file             spi_tclConnMngr.h
 * \brief            Core implementation for Connection Management
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Core implementation for Connection Management. Handles different
 SPI connections (Ex DiPo, Mirrorlink)
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 11.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 06.04.2014 |  Ramya Murthy                | Initialisation sequence implementation
 31.07.2014 |  Ramya Murthy (RBEI/ECP2)    | SPI feature configuration via LoadSettings()
 10.12.2014 |  Shihabudheen P M            | Changed for blocking device usage 
                                             preference updates during 
                                             select/deselect is in progress. 
 05.11.2014 |  Ramya Murthy (RBEI/ECP2)    | Added callback for Application metadata.
 25.01.2016 |  Rachana L Achar             | Logiscope improvements
 01.01.2017 |  Noopur R K                  | Added changes for Delete device interface implementation
 18.07.2017 |  Noopur R K                  | Added vSetGeneralRestrictions method

 \endverbatim
 ******************************************************************************/

#ifndef SPI_CONNMNGR_H_
#define SPI_CONNMNGR_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "Lock.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclConnection.h"
#include "spi_tclDeviceListIntf.h"
#include "spi_tclMediatorIntf.h"
#include "spi_tclSelectionIntf.h"
#include "spi_tclDiscoveryDataIntf.h"
#include "spi_tclConnSettingsIntf.h"

//! Forward declarations
class spi_tclConnMngrResp;

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/*!
 * \class spi_tclConnMngr
 * \brief Core implementation for Connection Management. Handles different
 *        SPI connections (Ex DiPo, Mirrorlink)
 */
class spi_tclConnMngr : public spi_tclLifeCycleIntf, public spi_tclSelectionIntf
{

   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::spi_tclConnMngr
       ***************************************************************************/
      /*!
       * \fn     spi_tclConnMngr()
       * \brief  parameterized Constructor
       * \param  poRespInterface: pointer to Response interface to post asynchronous
       *         responses
       * \param  poDeviceList: pointer to Devicelist handler
       * \param  poConnections: technology specific connection classes
       * \param  poDeviceSwitcher: pointer to switch USB profiles fo the devices
       * \param  poDeviceDiscoverer: pointer to device discoverer
       * \param  m_poMediator: pointer to mediator
       * \sa     ~spi_tclConnMngr()
       **************************************************************************/
      spi_tclConnMngr(spi_tclConnMngrResp *poRespInterface,spi_tclDeviceListIntf *poDeviceList,
               std::map<tenDeviceCategory, spi_tclConnection*> &poConnections,
               spi_tclMediatorIntf* m_poMediator, spi_tclDiscoveryDataIntf *poDiscoverIntf);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::~spi_tclConnMngr
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclConnMngr()
       * \brief  Destructor
       * \sa     spi_tclConnMngr()
       **************************************************************************/
      ~spi_tclConnMngr();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConnMngr::bInitialize()
       ***************************************************************************/
      /*!
       * \fn      bInitialize()
       * \brief   Method to Initialize connection Manager: Start device detection,
       *          Register for callbacks
       * \retval  true: if the initialization of connection manager is successful,
       *          false: on failure
       * \sa      bUnInitialize()
       **************************************************************************/
      virtual t_Bool bInitialize();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConnMngr::bUnInitialize()
       ***************************************************************************/
      /*!
       * \fn      bUnInitialize()
       * \brief   Method to UnInitialize connection Manager: Stop device detection.
       * \sa      bInitialize()
       **************************************************************************/
      virtual t_Bool bUnInitialize();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConnMngr::vLoadSettings()
       ***************************************************************************/
      /*!
       * \fn      vLoadSettings()
       * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
       * \sa      vSaveSettings()
       **************************************************************************/
      virtual t_Void vLoadSettings();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConnMngr::vLoadSettingsCompleted()
       ***************************************************************************/
      /*!
       * \fn      vLoadSettingsCompleted()
       * \brief   vLoadSettingsCompleted Method. Invoked to notify that load settings has completed
       **************************************************************************/
      virtual t_Void vLoadSettingsCompleted();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConnMngr::vSaveSettings()
       ***************************************************************************/
      /*!
       * \fn      vSaveSettings()
       * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
       * \sa      vLoadSettings()
       **************************************************************************/
      virtual t_Void vSaveSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vRestoreSettings
       ***************************************************************************/
      /*!
       * \fn     vRestoreSettings
       * \brief  Clears device history persistently stored on target and other private data
       **************************************************************************/
      t_Void vRestoreSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vGetDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceList
       * \brief  interface to get SPI devices list
       * \param  rfvecDeviceInfoList:[OUT] reference to the device list which will
       *         be populated with the detected SPI devices
       **************************************************************************/
      //t_Void vGetDeviceList(std::vector<trDeviceInfo>& rfvecDeviceInfoList);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vGetDeviceInfoList
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceInfoList
       * \brief  Interface to get SPI devices list asynchronously
       * \param  corfrUsrCntxt:[IN] User Context Details.
       **************************************************************************/
      t_Void vGetDeviceInfoList(const trUserContext &corfrUsrCntxt);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vGetDeviceList
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceList
       * \brief  Interface to get SPI devices list asynchronously
       * \param  corfrUsrCntxt:[IN] User Context Details.
       **************************************************************************/
      t_Void vGetDeviceList(const t_U32 cou32DeviceHandle, const trUserContext &corfrUsrCntxt);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vSelectDevice
       ***************************************************************************/
      /*!
       * \fn     vSelectDevice
       * \brief  Called when a device is selected/ deselected by the user.
       * \param  corfrSelDeviceRequest : Select request details
       **************************************************************************/
      t_Void vSelectDevice(const trSelectDeviceRequest &corfrSelDeviceRequest);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vSelectDeviceResult
       ***************************************************************************/
      /*!
       * \fn     vSelectDeviceResult
       * \brief  device selection result
       * \param  corfrSelectReq : Device selection result details
       * \param  enErrorCode : Error code
       **************************************************************************/
      t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::enGetDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     enGetDeviceCategory
       * \brief  returns the category of the device specified by cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle whose category is requested
       * \retval e8DEV_TYPE_DIPO : if cou32DeviceHandle is DiPo capable
       *         e8DEV_TYPE_MIRRORLINK: if cou32DeviceHandle is Mirrorlink capable
       *         e8DEV_TYPE_UNKNOWN : if cou32DeviceHandle type is not known
       * @TODO: To be removed: Don't use in new implementation
       **************************************************************************/
      tenDeviceCategory enGetDeviceCategory(const t_U32 cou32DeviceHandle) const;

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vGetBTAddress
       ***************************************************************************/
      /*!
       * \fn     vGetBTAddress
       * \brief  Gets the BT Address of the device : cou32DeviceHandle
       * \param  cou32DeviceHandle: Device handle whose connection status is requested
       * \param  rfszBTAddress: [OUT] BT address for the requested device handle
       * @TODO: To be removed: Don't use in new implementation
       **************************************************************************/
      t_Void vGetBTAddress(const t_U32 cou32DeviceHandle, t_String &rfszBTAddress) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConnMngr::vSetDeviceUsagePreference
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceUsagePreference
       * \brief  Interface to set the preference for the usage of the connected
       *         Mirror Link/DiPO device. This can be set for individual devices or
       *         for all the connected Mirror Link/DiPO devices.
       * \param  cou32DeviceHandle  : Unique handle which identifies the device.
       *         If the value is 0xFFFFFFFF, then this function sets the overall
       *         preference usage for Mirror Link/DiPO.
       * \param  enDeviceCategory : Device Type Information(Mirror Link/iPOD Out).
       * \param  enUsagePref      : Enable Information.
       * \param  rfrUsrCntxt      : User Context Details.
       **************************************************************************/
      t_Void vSetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
               tenDeviceCategory enDeviceCategory,
               tenEnabledInfo enUsagePref,
               const trUserContext &rfrUsrCntxt);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConnMngr::bGetDeviceUsagePreference
       ***************************************************************************/
      /*!
       * \fn     bGetDeviceUsagePreference
       * \brief  Interface to get the preference for the usage of the
       *         connected Mirror Link/iPod Out device during startup.
       * \param  cou32DeviceHandle  : Unique handle which identifies the device.
       *         If the value is 0xFFFFFFFF, then this function sets the overall
       *         preference usage for Mirror Link/DiPO.
       * \param  enDeviceCategory : Device Type Information(Mirror Link/iPOD Out).
       * \param  [OUT] rfenEnabledInfo : Enable Information.
       **************************************************************************/
      t_Bool bGetDeviceUsagePreference(const t_U32 cou32DeviceHandle,
               tenDeviceCategory enDeviceCategory,
               tenEnabledInfo& rfenEnabledInfo) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConnMngr::vSetGeneralRestrictions
       ***************************************************************************/
      /*!
       * \fn     vSetGeneralRestriction
       * \brief  Interface to set the general restrictions for various technologies.
       * \param  enDeviceCategory  : Device category .
       * \param  u16GeneralRestrInfo : General Restriction Info for the technology.
       * \param  [OUT] void.
       **************************************************************************/
      t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory, t_U16 u16GeneralRestrInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vSetSelectError
       ***************************************************************************/
      /*!
       * \fn     vSetSelectError
       * \brief  set to true if the error occurs on device side during selection
	    *         Ex: Role switch failure
       * \param cou32DeviceHandle: Device handle of the selected device
       * \parama bIsError value of the device error flag
       **************************************************************************/
      t_Void vSetSelectError(const t_U32 cou32DeviceHandle, t_Bool bIsError);

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclConnMngr::vSetDevSelectorStatus
      ***************************************************************************/
      /*!
      * \fn     t_Bool vSetDevSelectorStatus
      * \brief  Interface to check if the device specified with device id cou32DeviceID
      *         is same as the last used device
      * \param  [IN] cou32DeviceHandle : Device ID
      * \param  [IN] enSelState : Indicates current selection state
      * \retval NONE
      **************************************************************************/
      virtual t_Void vUpdateSelectionState(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enSelState, tenDeviceConnectionType enDeviceConnectionType);

      /**************************************************************************
      ** FUNCTION   : tVoid spi_tclConnMngr:: vSetTechnologyPreference()
      ***************************************************************************/
     /*!
      * \fn      t_Void vSetTechnologyPreference()
      * \brief   To set the preferred SPI technology for devices which support more than once technology
      * \param   [IN].cou32DeviceHandle - Device handle for which the prefernce is applicable.
      *               if cou32DeviceHandle is 0xFFFF then the preference is applicable for overall setting
      *          [IN] coenDeviceType - Device type information (Android/Apple Device)
      * \param   [IN].enTechnologyPreference - contains the technology preference
      * \retval  t_Void
      **************************************************************************/
     t_Void vSetTechnologyPreference(const t_U32 cou32DeviceHandle, const tenDeviceType coenDeviceType , tenDeviceCategory enTechnologyPreference,
              tenErrorCode &rfenErrorCode) const;

     /**************************************************************************
     ** FUNCTION   : tVoid spi_tclConnMngr:: vGetTechnologyPreference()
     ***************************************************************************/
    /*!
     * \fn      t_Void vGetTechnologyPreference()
     * \brief   To get the  previously set preferred SPI technology
     * \param   [IN].cou32DeviceHandle - Device handle for which the prefernce is applicable.
     *               if cou32DeviceHandle is 0xFFFF then the preference is applicable for overall setting
     *          [IN] -coenDeviceType - Devicetype information(Android/Apple Devices)
     * \param   [OUT].rfenTechnologyPreference - contains the technology preference 
     * \retval  t_Void
     **************************************************************************/
    t_Void vGetTechnologyPreference(const t_U32 cou32DeviceHandle, const tenDeviceType coenDeviceType, tenDeviceCategory &rfenTechnologyPreference) const;

    /***************************************************************************
     ** FUNCTION:  spi_tclConnMngr::bIsConnectionBusy
     ***************************************************************************/
    /*!
     * \fn     bIsConnectionBusy
     * \brief  Returns true if the connection for the device indicated by u32DeviceHandle is busy
     * \param  u32DeviceHandle : Uniquely identifies the target Device.
     **************************************************************************/
    t_Bool bIsConnectionBusy(const t_U32 cou32DeviceHandle);

    /***************************************************************************
    ** FUNCTION:  t_Void spi_tclConnMngr::vUpdateSessionTransport
    ***************************************************************************/
    /*!
    * \fn      t_Void vUpdateSessionTransport
    * \brief   Update session transport once SPI session is established
    * \param   enSessionTransport : [IN] Session is started over USB Transport or WiFi
    ***************************************************************************/
    t_Void vUpdateSessionTransport(const t_U32 cou32DeviceID, tenDeviceCategory enDeviceCategory, tenSessionTransport enSessionTransport);

    /***************************************************************************
     *** FUNCTION:  spi_tclConnMngr::vSelectionProgressStateCb
     ******************************************************************************/
    /*!
     * \fn     vSelectionProgressStateCb
     * \brief  Updates device selection progress state
     * \param  cou32DeviceHandle: [IN] Device ID
     * \param enDeviceSelectionProgressState: Selection Progress state
     ***************************************************************************/
    t_Void vSelectionProgressStateCb(const t_U32 cou32DeviceHandle, tenDeviceSelectionProgressState enDeviceSelectionProgressState);

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclConnMngr::vTriggerDeselection()
     ***************************************************************************/
     /*!
     * \fn      t_Void vTriggerDeselection()
     * \brief   Method to trigger device selector to continue with deselection, if retry fails.
     * \param   u32DeviceID  : [IN] Device Handle.
	 * \param   enDevCategory : [IN] Device category
     * \retval  t_Void
     **************************************************************************/	
    t_Void vTriggerDeselection(t_U32 u32DevID,tenDeviceCategory enDevCat);

    /***************************************************************************
     ** FUNCTION:  t_Void spi_tclConnMngr ::vSetConnsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf)
     ***************************************************************************/
     /*!
     * \fn      t_Void vSetConnsettingsInstance()
     * \brief   Method to set set connection settings instance.
     * \param   poConnSettingsIntf  : pointer to ConnsettingsIntf.
     * \retval  t_Void
     **************************************************************************/
    t_Void vSetConnsettingsInstance(spi_tclConnSettingsIntf* poConnSettingsIntf);


    /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConnMngr::vGetConnMngrConfig()
       ***************************************************************************/
       /*!
       * \fn      t_Void vGetConnMngrConfig()
       * \brief   To print the Connection Manager configuration to read KDS
       * \retval  t_Void
       **************************************************************************/
    t_Void vGetConnMngrConfig();
	
	/***************************************************************************
     ** FUNCTION:  spi_tclDeviceSelector::vOnSetSelectionProgressState()
     ***************************************************************************/
    /*!
     * \fn     vOnSetSelectionProgressState
     * \brief  set selection state to BDCL Connmanger
     * \param  :tenDeviceSelectionState
     * \retval : t_Void
     ***************************************************************************/
    t_Void vOnSetSelectionProgressState(tenDeviceSelectionState enDeviceSelectionState);

   private: 
      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

      /***************************************************************************
      ** FUNCTION: spi_tclConnMngr(const spi_tclConnMngr &corfobjRhs)
      ***************************************************************************/
      /*!
      * \fn      spi_tclConnMngr(const spi_tclConnMngr &corfobjRhs)
      * \brief   Copy constructor not implemented hence made protected to prevent
      *          misuse
      **************************************************************************/
      spi_tclConnMngr(const spi_tclConnMngr &corfobjRhs);

      /***************************************************************************
      ** FUNCTION: const spi_tclConnMngr & operator=(const spi_tclConnMngr &corfobjRhs);
      ***************************************************************************/
      /*!
      * \fn      const spi_tclConnMngr & operator=(const spi_tclConnMngr &corfobjRhs);
      * \brief   assignment operator not implemented hence made protected to
      *          prevent misuse
      **************************************************************************/
      const spi_tclConnMngr & operator=(
         const spi_tclConnMngr &corfobjRhs);

       /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vSelectDeviceResultCb
       ***************************************************************************/
      /*!
       * \fn     vSelectDeviceResultCb
       * \brief  Callback to update the select device result
       * \param  bSelDevResult : Select device result
       * \param  enConnreq: Indicated whether it is a selection or deselection request
       **************************************************************************/
      t_Void vSelectDeviceResultCb(tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vUpdateDeviceNameCb
       ***************************************************************************/
      /*!
       * \fn     vUpdateDeviceNameCb
       * \brief  updates the device name of the device identified by device handle
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param rfrszDeviceName : Device Name to be set
       **************************************************************************/
      t_Void vUpdateDeviceNameCb(t_U32 cou32DeviceHandle, t_String szDeviceName);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vUpdateBTDeviceInfoCb
       ***************************************************************************/
      /*!
       * \fn     vUpdateBTDeviceInfoCb
       * \brief  updates the BT device name and/or BT address of the device identified
       *          by device handle
       * \param  u32DeviceHandle: Device Handle of the device to be added to history
       * \param  szBTDeviceName : Device Name to be set
       * \param  szBTAddress : BT Address to be set
       **************************************************************************/
      t_Void vUpdateBTDeviceInfoCb(t_U32 u32DeviceHandle,
            t_String szBTDeviceName, t_String szBTAddress);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks
       * \brief  Registers for callbacks to ML/DiPo classes
       **************************************************************************/
      t_Void vRegisterCallbacks();

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::bSetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bSetDevProjUsage
       * \brief  Called when the SPI feature is turned ON or OFF by the user
       * \param  enSPIType: indicates the SPI Type (Mirrorlink, Dipo ..)
       * \param  enSPIState : indicates the service status of enSPIType
       * \retval true: if the requested service state was switched
       *         false: on failure
       **************************************************************************/
      t_Bool bSetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo enSPIState, tenDeviceType enDeviceType);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::bGetDevProjUsage
       ***************************************************************************/
      /*!
       * \fn     bGetDevProjUsage
       * \brief  Returns the present SPI Feature State
       * \param  enSPIType: SPI service type for which state is
       *         requested (Mirrorlink, DiPo ..)
       * \param  rfenEnabledInfo:  [OUT]Indicates if enSPIType is enabled or not
       **************************************************************************/
      t_Bool bGetDevProjUsage(tenDeviceCategory enSPIType, tenEnabledInfo &rfenEnabledInfo, tenDeviceType enDeviceType) const;

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vLoadSPITechnologySettings
       ***************************************************************************/
      /*!
       * \fn     vLoadSPITechnologySettings
       * \brief  Load SPI On/Off settings
       **************************************************************************/
      t_Void vLoadSPITechnologySettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vEvaluateErrorStates
       ***************************************************************************/
      /*!
       * \fn     vEvaluateErrorStates
       * \brief  Clears error states for known devices if required
       **************************************************************************/
      t_Void vEvaluateErrorStates(const t_U32 cou32DeviceHandle, tenDeviceCategory enChosenDeviceCategory);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vUpdateSessionIP
       ***************************************************************************/
      /*!
       * \fn     vUpdateSessionIP
       * \brief  updates the Session IP
       * \param cou32DeviceHandle: Device Handle of the device to be added to history
       * \param rfrszSessionIP : Session IP to be set
       **************************************************************************/
      t_Void vUpdateSessionIP(t_U32 cou32DeviceHandle, t_String szSessionIP);

      //! Pointer to the object handling device list of SPI devices
      spi_tclDeviceListIntf *m_poDeviceListHandler;

      //! Pointer to response interface for posting asynchronous responses
      spi_tclConnMngrResp *m_poConnMngrResp;

      //! List of connection handlers
      std::map<tenDeviceCategory, spi_tclConnection*> m_mapoConnHandlers;

      //! Lock for protecting connection handlers
      Lock m_poLockConnections;

      //! Stores the current status of SPI Service
      t_Bool m_bSPIState[e8_CONN_INDEX_SIZE];

      //! Indicates whether the device selector is busy or not.
      t_Bool m_bIsDevSelectorBusy;

      //! Stores the SPI technology enabled in calibration
      trSpiFeatureSupport m_rSPIFeatureSupport;

      //! Stores user context for role switch request for mediaplayer
      std::map<t_U32, trUserContext> m_mapRoleSwitchRespQueue;

      //! Store Device selection request
      trSelectDeviceRequest m_rDeviceSelectionRequest;


      //! pointer to mediator
      spi_tclMediatorIntf *m_poMediator;

      //! Lock to protect switch device critical section
      Lock m_oDeviceSwitchLock;

      //! Device under selection
      t_U32 m_u32DeviceUnderSelection;

      //Lock for role switch queue
      Lock m_oRoleSwitchRespQueueLock;

      spi_tclDiscoveryDataIntf *m_poDiscovrerDataIntf;
      
	  //! Connection settings Interface pointer
      spi_tclConnSettingsIntf* m_poConnSettings;

};

/*! } */
#endif // SPI_CONNMNGR_H_
