/*!
 *******************************************************************************
 * \file             spi_ConnMngrTypeDefines.h
 * \brief            Base Connection class
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Type defines used in Connection Manager
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 17.02.2014 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLCONNMNGRTYPEDEFINES_H_
#define SPI_TCLCONNMNGRTYPEDEFINES_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "SPITypes.h"


//! Supported Connection Media Types
enum tenSupportedConnectionTypes
{
   e8SUPPOTSCONN_USB = 1,
   e8SUPPOTSCONN_WIFI = 2,
   e8SUPPOTSCONN_USB_OR_WIFI = 3,
   e8SUPPOTSCONN_USB_AND_WIFI = 4
};

//! Mirrorlink 1.0 support
enum tenML10Support
{
   e8ML10_ALLDEVICES = 1,
   e8ML10_DAPCERTIFIED = 2,
   e8ML10_NOT_SUPPORTED = 3
};

//! Priority for autoselection
enum tenSelModePriority
{
   e8PRIORITY_DEVICELIST_HISTORY = 1,
   e8PRIORITY_CONNMODE_PREFERENCE = 2,
   e8PRIORITY_DEVICETYPE_PREFERENCE = 3
};

//! Multi session  support
enum tenMultiSessionSupport
{
   e8MULTISESSION_NOT_SUPPORTED = 0,
   e8MULTISESSION_SUPPORTED = 1
};

//! DAP Preference
enum tenDAPPreference
{
   e8DAPPREF_UNKNOWN = 0,
   e8DAPPREF_ON_DEVICECONNECTION = 1,
   e8DAPPREF_ON_REQUEST = 2
};

//! Determines type of SPI Service
typedef enum
{
   e8SPISERVICE_UNKNOWN = 0,
   e8SPISERVICE_MIRRORLINK = 1,
   e8SPISERVICE_DIPO = 2
} tenSPIServiceType;

//! Specifies user preferred SPI technology
enum tenUserPreference
{
   e8PREFERENCE_NOTKNOWN = 0,
   e8ANDROID_AUTO_NOTPREFERRED = 1,
   e8ANDROID_AUTO_PREFERRED = 2,
   e8MIRRORLINK_NOT_PREFERRED = 3,
   e8MIRRORLINK_PREFERRED = 4,
   e8PROJECTION_NOT_PREFERRED = 5
};

typedef enum
{
    e8_DEVICE_NOT_WHITELISTED = 0,
    e8_DEVICE_WHITELISTED = 1
}tenIsDeviceWhiteListed;

typedef enum
{
   e8_DEVICESWITCHSTATE_UNKNONW = 0,
   e8_DEVICESWITCHSTATE_PROJECTION = 1,
   e8_DEVICESWITCHSTATE_DEFAULT = 2
} tenUSBSwitchState;

/*!
 * Stores device info along with extra fields needed for
 * connection management
 */
struct trEntireDeviceInfo
{
      trDeviceInfo rDeviceInfo;
      t_Bool bDeviceValidated;
      t_Bool bIsDeviceUsed;
      t_U32 u32AccessIndex;
      t_Bool bIsUserDeselected;
      t_U8   u8SelectError;
      t_Bool bIsUSBConnected;
      tenUserPreference enUserPreference;
      trDeviceConnectionCountInfo rDeviceConnectionCountInfo;
      tenIsDeviceWhiteListed enIsDeviceWhiteListed;
      t_String szWifiSessionIP;
      t_String szUSBSessionIP;
      tenSessionTransport enLastActiveSessionTransport;
      tenUSBSwitchState enDeviceSwitchState;
      trEntireDeviceInfo() :
         bDeviceValidated(false), bIsDeviceUsed(false), u32AccessIndex(0), bIsUserDeselected(false), u8SelectError(
               0), bIsUSBConnected(false), enUserPreference(e8PREFERENCE_NOTKNOWN), enIsDeviceWhiteListed(e8_DEVICE_NOT_WHITELISTED),
               enLastActiveSessionTransport(e8_SESSION_TRANSPORT_UNKNOWN), enDeviceSwitchState(e8_DEVICESWITCHSTATE_UNKNONW)
      {

      }
      //! overloads the function call operator, operator(), it becomes a FunctionObject type.
      //! Many standard algorithms, from std::sort to std::accumulate accept objects
      //! of such types to customize behavior.
      bool operator()(trEntireDeviceInfo const & corfrLhs, trEntireDeviceInfo const & corfrRhs) const
      {

         t_Bool bRetVal =
                  //! Connected and Selected to be on the top of the list
                  (((corfrLhs.rDeviceInfo.enDeviceConnectionStatus
                           > corfrRhs.rDeviceInfo.enDeviceConnectionStatus)
                           && (corfrLhs.rDeviceInfo.bSelectedDevice == true))
		 
					//! Two or more Devices connected, Order will be Selected first and then the Connected devices
                           || ((corfrLhs.rDeviceInfo.enDeviceConnectionStatus == e8DEV_CONNECTED)
                                    && (corfrLhs.rDeviceInfo.enDeviceConnectionStatus
                                             == corfrRhs.rDeviceInfo.enDeviceConnectionStatus)
                                             && (corfrLhs.rDeviceInfo.bSelectedDevice == true))

					//! Two or more Devices connected which are not selected, order will be from Higher Index to Lower Index
                           || ((corfrLhs.rDeviceInfo.enDeviceConnectionStatus == e8DEV_CONNECTED)
                                    && (corfrLhs.rDeviceInfo.enDeviceConnectionStatus
                                            == corfrRhs.rDeviceInfo.enDeviceConnectionStatus)
                                            && (corfrLhs.rDeviceInfo.bSelectedDevice == false)
                                            && (corfrRhs.rDeviceInfo.bSelectedDevice == false)
                                            && (corfrLhs.u32AccessIndex > corfrRhs.u32AccessIndex))
											
					//! Two or more Devices connected which are not selected - PSA requires Devices Connected over 
					// OTG to be on the top of the list - This code needs to be revisited and rewritten
                       /*    || ((corfrLhs.rDeviceInfo.enDeviceConnectionStatus == e8DEV_CONNECTED)
                                    && (corfrLhs.rDeviceInfo.enDeviceConnectionStatus
                                            == corfrRhs.rDeviceInfo.enDeviceConnectionStatus)
                                            && (corfrLhs.rDeviceInfo.bSelectedDevice == false)
                                            && (corfrRhs.rDeviceInfo.bSelectedDevice == false)
                                            &&((e8_PORT_TYPE_OTG == corfrLhs.rDeviceInfo.rProjectionCapability.enUSBPortType)
													&& (e8DEV_TYPE_DIPO == corfrLhs.rDeviceInfo.enDeviceCategory)))*/

											
                    //! Devices disconnected, order will be from Higher Index to Lower Index
                           || ((corfrLhs.rDeviceInfo.enDeviceConnectionStatus == e8DEV_NOT_CONNECTED)
                                    && (corfrLhs.rDeviceInfo.enDeviceConnectionStatus
                                            == corfrRhs.rDeviceInfo.enDeviceConnectionStatus)
                                            && (corfrLhs.u32AccessIndex > corfrRhs.u32AccessIndex))
											
					//! Connected device to be on the top of the list
                           || (corfrLhs.rDeviceInfo.enDeviceConnectionStatus
                                   > corfrRhs.rDeviceInfo.enDeviceConnectionStatus)
                  );

         return bRetVal;
      }
};


enum tenDiscovererType
{
   e8_DISCOVERER_TYPE_USB = 0,
   e8_DISCOVERER_TYPE_WIFI = 1,
   e8_DISCOVERER_TYPE_BT = 2,
   e8_DISCOVERER_TYPE_ML = 3,
   e8_DISCOVERER_TYPE_APPLE = 4,
   //! Please add the new discoverers above this line to maintain the size
   e8_DISCOVERER_TYPE_SIZE,
   e8_DISCOVERER_TYPE_UNKNOWN = 0xFF
};

struct trDiscoveredDevicesInfo
{
   trDeviceInfo m_prDeviceInfo;
   tenDiscovererType m_enDiscovererType;
   trDiscoveredDevicesInfo(trDeviceInfo rDeviceInfo, tenDiscovererType enDiscovererType) : m_prDeviceInfo(rDeviceInfo),m_enDiscovererType(enDiscovererType)
   {
   }
};

//! Callback signatures definitions: To be registered by the Creator of this class object
//typedef std::function<t_Void(t_U32, const trDeviceInfo&, tenDeviceCategory)> tfvDeviceConnection;
//typedef std::function<t_Void(t_U32, tenDeviceCategory)> tfvDeviceDisconnection;
typedef std::function<t_Void(t_U32, tenDAPStatus)> tfvUpdateDAPStatus;
typedef std::function<t_Void(tenErrorCode)> tfvSelectDeviceResult;
typedef std::function<t_Void(const t_U32, t_Bool)> tfvSelectDeviceError;

typedef std::function<t_Void(t_U32, t_String)> tfvSetDeviceName;


typedef std::function<t_Void(const trDeviceInfo&, tenDiscovererType)> tfvDeviceConnection;
typedef std::function<t_Void(t_U32, tenDiscovererType,tenDeviceConnectionType, tenSPISupport)> tfvDeviceDisconnection;
typedef std::function<t_Void(t_U32, t_Bool)> tfvDeviceSwitchComplete;
typedef std::function<t_Void(t_U32, trUserContext)> tfvEarlyRoleSwitchCall;
typedef std::function<t_Void(const t_U32, tenDeviceCategory, tenSessionTransport)> tfvUpdateSessionTransport;
typedef std::function<t_Void(t_U32,tenDeviceCategory)> tfvTriggerDeselection;

//! Storing the Session IP Address on Session Start Message
typedef std::function<t_Void(t_U32,t_String)> tfvOnSessionMessageStoreIPAddress;

/*!
 * \brief Structure holding the callbacks to be registered by the
 * Creator of this class object
 */
struct trConnCallbacks
{
      //! Informs when a new device is detected
      tfvDeviceConnection fvDeviceConnection;

      //! Property called when there is change in device status
      tfvUpdateDAPStatus fvUpdateDAPStatus;

      //! Informs when a Device is disconnected
      tfvDeviceDisconnection fvDeviceDisconnection;

      //!Inform the select device result
      tfvSelectDeviceResult fvSelectDeviceResult;

      //! Sets error flag to prevent further automatic selection
      tfvSelectDeviceError fvSelectDeviceError;
 
      //! Sets the device name
      tfvSetDeviceName fvSetDeviceName;

      //! Session Transport
      tfvUpdateSessionTransport fvSessionTransport;

      //!Sets the selection progress state
      tfvSelectionProgressState fvSelectionProgressState;
	  
	  //!  Trigger deselect, if retry fails.
      tfvTriggerDeselection fvTriggerDeselection;

      //! Session IP Address
      tfvOnSessionMessageStoreIPAddress fvOnSessionMessageStoreIPAddress;

      trConnCallbacks() :
                        fvDeviceConnection(NULL),
                        fvUpdateDAPStatus(NULL),
                        fvDeviceDisconnection(NULL),
                        fvSelectDeviceResult(NULL),
                        fvSelectDeviceError(NULL),
                        fvSetDeviceName(NULL),
                        fvSessionTransport(NULL),
                        fvSelectionProgressState(NULL),
                        fvTriggerDeselection(NULL),
                        fvOnSessionMessageStoreIPAddress(NULL)
      {

      }
};

//typedef std::function<void(const trDeviceInfo&, tenDiscovererType)> tfvDeviceConnection;
//typedef std::function<void(t_U32, tenDiscovererType)> tfvDeviceDisconnection;
struct trDiscovererCbs
{
      //! Informs when a new device is detected
      tfvDeviceConnection fvDeviceConnectionCb;

      //! Property called when there is change in device status
      tfvDeviceDisconnection fvDeviceDisconnectionCb;

      //!Indicates the device switch is complete
      tfvDeviceSwitchComplete fvDeviceSwitchCompleteCb;

      //! Property called when there is change in device status
      tfvUpdateDAPStatus fvUpdateAuthenticationStatus;

      trDiscovererCbs(): fvDeviceConnectionCb(NULL), fvDeviceDisconnectionCb(NULL), fvDeviceSwitchCompleteCb(NULL),
               fvUpdateAuthenticationStatus(NULL)
      {

      }
};

struct trDeviceSwitcherCbs
{
      //!Indicates the device switch is complete
      tfvDeviceSwitchComplete fvDeviceSwitchCompleteCb;

      trDeviceSwitcherCbs(): fvDeviceSwitchCompleteCb(NULL)
      {

      }
};


struct trDeviceSwitchInfo
{
      t_U32 u32DeviceID;
      t_U32 u32VendorID;
      t_U32 u32ProductID;
      t_String szSerialNumber;
      t_Bool bSwitchInProgress;
      t_S32 s32HeadUnitID;
      t_S32 s32SwitchedDeviceID;
      tenDeviceCategory enDeviceSwitchType;
      t_Bool bIsEarlyDeviceSwitchInProgress;
      tenDeviceType enDeviceType;
      timer_t t_TimerID;
      tenDeviceConnectionType enDeviceConnectionType;
      trDeviceSwitchInfo() :
                        u32DeviceID(0),
                        u32VendorID(0),
                        u32ProductID(0),
                        bSwitchInProgress(false),
                        s32HeadUnitID(0),
                        s32SwitchedDeviceID(0),
                        enDeviceSwitchType(e8DEV_TYPE_UNKNOWN),
                        bIsEarlyDeviceSwitchInProgress(false),
                        enDeviceType(e8_UNKNOWN_DEVICE),
                        t_TimerID(0),
                        enDeviceConnectionType(e8UNKNOWN_CONNECTION)
      {

      }
};

struct trUSBDeviceInfo
{
      t_U32 u32DeviceHandle;
      t_U32 u32VendorID;
      t_U32 u32ProductID;
      t_String szSerialNumber;
      t_String   szProduct;
      t_String   szManufacturer;
      t_U32    u32USBDeviceNumber;
      t_String szSystemPath;
      t_U32    u32Interface;
      t_Bool   bIsAOAPSupported;
      tenDeviceType enDeviceType;
      tenDeviceProfile enUSBProfile;
      tenDeviceConnectionStatus enConnectionStatus;
      tenUSBPortType enUSBPortType;
      trDeviceSwitchInfo rDevSwitchInfo;
      t_U32 u32SwitchRetrialCount;

      trUSBDeviceInfo():u32DeviceHandle(0), u32VendorID(0), u32ProductID(0),u32USBDeviceNumber(0),
               u32Interface(0), bIsAOAPSupported(false), enDeviceType(e8_UNKNOWN_DEVICE),
               enUSBProfile(e8_PROFILE_UNKNOWN), enConnectionStatus(e8DEV_NOT_CONNECTED),
               enUSBPortType(e8_PORT_TYPE_NOT_KNOWN),u32SwitchRetrialCount(0)
      {

      }
};

typedef enum
{
   e32MESSAGEID_DISCOVERER_DEVICEINFO = 1,
   e32MESSAGEID_DISCOVERER_DEVICEDISCONNECTION = 2,
   e32MESSAGEID_DISCOVERER_AUTHENTICATIONSTATUS = 3,
   e32MESSAGEID_DISCOVERER_DEVICENAMEUPDATE = 4,
   e32MESSAGEID_DISCOVERER_DEVICEPROFILE = 5,
   e32MESSAGEID_DISCOVERER_BTLIMITATIONSTATE = 6
} tenDiscovererMsg;

typedef enum
{
   e8DEVICE_SWITCHTYPE_UNKNOWN = 0,
   e8DEVICE_SWITCHTYPE_PROJECTION = 1,
   e8DEVICE_SWITCHTYPE_DEFAULT = 2
} tenDeviceSwitchType;

typedef enum
{
   e8_DEVICESWITCH_RESULT_UNKNOWN = 0,
   e8_DEVICESWITCH_RESULT_FAILED = 1,
   e8_DEVICESWITCH_RESULT_SWITCH_NOTREQUIRED = 2,
   e8_DEVICESWITCH_RESULT_INPROGRESS = 3,
   e8_DEVICESWITCH_RESULT_SWITCH_PARTOF_PROTOCOL = 4,
   e8_DEVICESWITCH_RESULT_SUCCESS = 5,
   e8_DEVICESWITCH_RESULT_DEVICE_RESET_REQUIRED = 6

} tenDeviceSwitchResult;


/*! } */
#endif /* SPI_TCLCONNMNGRTYPEDEFINES_H_ */
