/*!
 *******************************************************************************
 * \file             spi_tclDiscovererFactory.h
 * \brief            Discoverer Factory: Creates discoverers
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Discoverer Factory: Creates discoverers
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/


#ifndef SPI_TCLDISCOVERERFACTORY_H_
#define SPI_TCLDISCOVERERFACTORY_H_


#include <map>
#include "Lock.h"
#include "spi_tclDiscovererBase.h"
#include "GenericSingleton.h"

/**
 * \class spi_tclDiscovererFactory
 * \brief Responsible for discoverer creation
 */

class spi_tclDiscovererFactory: public GenericSingleton<spi_tclDiscovererFactory>
{

   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::spi_tclDiscovererFactory
       ***************************************************************************/
      /*!
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclDiscovererFactory();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::~spi_tclDiscovererFactory
       ***************************************************************************/
      /*!
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclDiscovererFactory();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::poGetDiscovererInstance
       ***************************************************************************/
      /*!
       * \brief   Get the disovererer instance for the discoverer type specified by enDiscovererType
       * \param   enDiscovererType: Type of the discoverer
       * \retval  spi_tclDiscovererBase*: pointer to the discoverer instance
       **************************************************************************/
      spi_tclDiscovererBase* poGetDiscovererInstance(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::vCreateDiscoverer
       ***************************************************************************/
      /*!
       * \brief   Create the discoverer specified by enDiscovererType
       * \param   enDiscovererType: Type of the discoverer
       **************************************************************************/
      t_Void vCreateDiscoverer(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::vCreateDiscoverer
       ***************************************************************************/
      /*!
       * \brief   Destroys the discoverer specified by enDiscovererType
       * \param   enDiscovererType: Type of the discoverer
       **************************************************************************/
      t_Void vDestroyDiscoverer(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::u32GetNoofDiscoverers
       ***************************************************************************/
      /*!
       * \brief   Returns the number of discoverers created
       * \retval   t_U32: No of discoverers
       **************************************************************************/
      t_U32 u32GetNoofDiscoverers();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererFactory::u32GetNoofDiscoverers
       ***************************************************************************/
      /*!
       * \brief   Returns the instances of discoverers created
       * \param   [OUT]rfvecDiscoverers: Vector with the instances of created disocverers
       **************************************************************************/
      t_Void vGetDiscoverers(std::vector<spi_tclDiscovererBase*> &rfvecDiscoverers);

   private:

      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/
      friend class GenericSingleton<spi_tclDiscovererFactory> ;

      //! Stores the created discoverers
      std::map<tenDiscovererType, spi_tclDiscovererBase*> m_mapDeviceDiscoverers;

      //! Lock to protect the discoverer map
      Lock m_oDeviceDiscovererLock;

};
/*! } */
#endif // SPI_TCLDISCOVERERFACTORY_H_
