/*!
 *******************************************************************************
 * \file             spi_tclDeviceDiscoverer.h
 * \brief            Unified Device Discoverer
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Unified Device Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDEVICEDISCOVERER_H_
#define SPI_TCLDEVICEDISCOVERER_H_

#include "Lock.h"
#include "spi_tclDeviceDiscovererIntf.h"
#include "RespBase.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclDiscoveryResp.h"
#include "spi_tclDeviceListIntf.h"
#include "spi_tclDeviceSwitcherIntf.h"
#include "spi_tclDeviceDiscovererIntf.h"
#include "spi_tclDiscoveryDataIntf.h"
#include "spi_tclDiscovererSettingsIntf.h"
#include "spi_tclMediatorIntf.h"
#include "spi_tclSelectionIntf.h"
#include "spi_tclConnMngrResp.h"
#include "spi_tclDiscovererBase.h"

struct trRoleSwitchInfo
{
      trUserContext rUserContext;
      t_U32 u32MplayDeviceID;
      trRoleSwitchInfo():u32MplayDeviceID(0) {}
};

/**
 * \class spi_tclDeviceDiscoverer
 * \brief Unified discoverer responsible for reporting device connections and disconnections
 *        reported from different connection mediums such as USB, Wi-Fi. All devices detected on
 *        these connection mediums will be reported to the connection manager,
 *  \sa USBDiscoverer, WiFiDiscoverer
 */
class spi_tclDeviceDiscoverer: public spi_tclDeviceDiscovererIntf, public spi_tclLifeCycleIntf,
   public spi_tclDiscoveryResp, public spi_tclSelectionIntf
{

   public:
   
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::spi_tclDeviceDiscoverer()
       ***************************************************************************/
      /*!
       * \brief   constructor
       **************************************************************************/
       spi_tclDeviceDiscoverer(spi_tclConnMngrResp *poRespInterface,spi_tclDeviceListIntf *poDeviceList,
               std::map<tenDiscovererType, spi_tclDiscovererBase*> &rfrmapoDiscoverers,
               spi_tclDeviceSwitcherIntf *poDeviceSwitcher,
               spi_tclDiscoveryDataIntf* m_pDiscoveryData,
               spi_tclDiscovererSettingsIntf* poDiscovSettings,
               spi_tclMediatorIntf* m_poMediator);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::~spi_tclDeviceDiscoverer()
       ***************************************************************************/
      /*!
       * \brief   destructor
       **************************************************************************/
      virtual ~spi_tclDeviceDiscoverer();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::bInitialize()
       ***************************************************************************/
      /*!
       * \brief   Creates required discoverers and initializes them
       * \sa      bUnInitialize()
       **************************************************************************/
      virtual t_Bool bInitialize();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::bUnInitialize()
       ***************************************************************************/
      /*!
       * \brief   Uninitialize and Deletes the created discoverers
       * \sa      bInitialize()
       **************************************************************************/
      virtual t_Bool bUnInitialize();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vLoadSettings()
       ***************************************************************************/
      /*!
       * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
       * \param   rfcrSpiFeatureSupp: Indicates feature support for SPI technologies
       * \sa      vSaveSettings()
       **************************************************************************/
      virtual t_Void vLoadSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSaveSettings()
       ***************************************************************************/
      /*!
       * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
       * \sa      vLoadSettings()
       **************************************************************************/
      virtual t_Void vSaveSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vStartDeviceDetection()
       ***************************************************************************/
      /*!
       * \brief   Starts Device Detection.
       *          Invoked after vLoadSettings() of all subcomponents.
       * \sa      vLoadSettings()
       **************************************************************************/
      t_Void vStartDeviceDetection();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::bStartDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Starts Device detection. This will further invoke the discoverers (USB/Wi-Fi)
       *          to start device detection by listening to libusb, connman etc.
       *          Has to be called when device detection for SPI has to be started.
       *          Can be stopped and restarted when system states change toggles between OFF and ON
       * \param   enDiscovererType: Type of the discoverer for which device detection has to be started
       * \retval  true if the discoverer is started successfully false otherwise
       * \sa      vStopDeviceDetection
       **************************************************************************/
      t_Bool bStartDeviceDetection(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStartDeviceReporting
       ***************************************************************************/
      /*!
       * \brief   Starts reporting detected devices. This can be used to start updating the
       *          detected devices to upper layers. Can be used in use cases such as
       *          or ON OFF where devices detected during the OFF period have to be remembered.
       *          Device reporting can be stopped using vStopDeviceReporting although
       *          Device discovery is in progress
       * \param   enDiscovererType: Type of the discoverer for which device reporting has to be started
       * \note    This interface is not implemented and is for future use
       * \warning Calling this interface without starting device discovery will internally start device discovery
       * \sa      vStopDeviceReporting
       **************************************************************************/
      t_Void vStartDeviceReporting(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStopDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Stops Device detection. This will further invoke the discoverers (USB/Wi-Fi)
       *          to stop device detection by listening to libusb, connman etc.
       *          Has to be called when device detection for SPI has to be stopped.
       *          Can be stopped and restarted when system states change toggles between OFF and ON
       * \param   enDiscovererType: Type of the discoverer for which device detection has to be stopped
       * \sa      bStartDeviceDetection
       **************************************************************************/
      t_Void vStopDeviceDetection(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStopDeviceReporting
       ***************************************************************************/
      /*!
       * \brief   Stops reporting detected devices. This can be used to stop updating the
       *          detected devices to upper layers. Can be used in use cases such as
       *          or ON OFF where devices detected during the OFF period have to be remembered.
       *          Device reporting can be started using vStartDeviceReporting
       * \param   enDiscovererType: Type of the discoverer for which device reporting has to be stopped
       * \note    This interface is not implemented and is for future use
       * \sa      vStopDeviceReporting
       **************************************************************************/
      t_Void vStopDeviceReporting(tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSelectDevice
       ***************************************************************************/
      /*!
       * \fn     vSelectDevice
       * \brief  Called when a device is selected/ deselected by the user.
       * \param  corfrSelDeviceRequest : Select request details
       **************************************************************************/
      t_Void vSelectDevice(const trSelectDeviceRequest &corfrSelDeviceRequest);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSelectDeviceResult
       ***************************************************************************/
      /*!
       * \fn     vSelectDeviceResult
       * \brief  device selection result
       * \param  corfrSelectReq : Device selection result details
       * \param  enErrorCode : Error code
       **************************************************************************/
      t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vUpdateSelectionState
       ***************************************************************************/
      /*!
       * \fn     vUpdateSelectionState
       * \brief  Informs if the device selection is not initiated for a reported device
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      t_Void vUpdateSelectionState(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enDevSelectionState, tenDeviceConnectionType enDeviceConnType);

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclConnMngr::bIsDiPoRoleSwitchRequired
      ***************************************************************************/
      /*!
      * \fn     t_Bool bIsDiPoRoleSwitchRequired
      * \brief  Interface to check if the device specified with device id cou32DeviceID
      *         is same as the last used device
      * \param  [IN] coszSerialNo :Serial number
      * \param  [IN] cobDeviceCarplayCapability : Whether device supports carplay or not
      * \retval true if the device is same as last connected device otherwise false
      **************************************************************************/
      t_Bool bIsDiPoRoleSwitchRequired(const t_String coszSerialNo, const trUserContext corUsrCntxt, const t_U32 cou32MplayDeviceID =0, const t_Bool cobDeviceCarplayCapability= false);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vRestoreSettings()
       ***************************************************************************/
      /*!
       * \brief   vRestoreSettings Method. Invoked on factory reset to clear the private data when SPI Session is active
       * \param   None
       **************************************************************************/
       virtual t_Void vRestoreSettings();

       /***************************************************************************
        ** FUNCTION:  spi_tclDeviceDiscoverer::vUpdateDeviceName
        ***************************************************************************/
       /*!
        * \fn     vUpdateDeviceName
        * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
        * \param  [IN] szDeviceName : Device Name
        * \param  [IN] enDiscovererType: Discoverer type
        ***************************************************************************/
        t_Void vUpdateDeviceName(const t_U32 cou32DeviceHandle,t_String szDeviceName, tenDiscovererType enDiscovererType);

       /***************************************************************************
          ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vGetDeviceDiscovererConfig()
          ***************************************************************************/
          /*!
          * \fn      t_Void vGetDeviceDiscovererConfig()
          * \brief   To print the Device Discoverer configuration to read KDS
          * \retval  t_Void
          **************************************************************************/
       t_Void vGetDeviceDiscovererConfig();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::bSetClientState(...
       ***************************************************************************/
      /*!
       * \fn    bSetClientState
       * \brief  Interface to set client state on which SPI will decide whether to
       *  start device detection or not
       * \param  [IN] tenClientState : current state of the SPI Client
       **************************************************************************/
      t_Bool bSetClientState(tenClientState enClientStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vDeleteDevice
       ***************************************************************************/
      /*!
       * \fn     vDeleteDevice
       * \brief  Delete the device from the list
       * \param  vector consisting of device handles that have to be deleted
       * \param  rfcorUsrCntxt : user context
       **************************************************************************/
      t_Void vDeleteDevice(const std::vector<t_U32>& rfvecrDeleteDeviceList, const trUserContext& rfcorUsrCntxt);

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vUpdateBTLimitedFunctionality
      ***************************************************************************/
      /*!
      * \fn       t_Void vUpdateBTLimitedFunctionality
      * \brief   To Post the BT Limitated Functionality
      * \param   cou32DeviceHandle : [IN] Device Handle
      * \param   m_bIsDeviceBTFunctionalityLimited : [IN] BT Limited functionality is set or not
      * \retval  t_Void
      **************************************************************************/
      t_Void vUpdateBTLimitedFunctionality(const t_U32 cou32DeviceHandle, tenDiscovererType enDiscovererType,
               t_Bool bIsDeviceBTFunctionalityLimited);


   private:

      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vReportDeviceConnection
       ***************************************************************************/
      /*!
       * \fn     vReportDeviceConnection
       * \brief  Callback to be Called by ML/ DiPo class when a new device is detected.
       * \param  cou32DeviceHandle: Device Handle of the detected device
       * \param  corfrDeviceInfo: Device info for the detected device
       **************************************************************************/
      t_Void vReportDeviceConnection(const trDeviceInfo& corfrDeviceInfo, tenDiscovererType enDiscovererType = e8_DISCOVERER_TYPE_USB);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vReportDeviceDisconnection
       ***************************************************************************/
      /*!
       * \fn     vReportDeviceDisconnection
       * \brief  Callback to be Called by ML/ DiPo class when a device is disconnected
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       * \param  enCarplaySupport : Carplay Support
       **************************************************************************/
      t_Void vReportDeviceDisconnection(const t_U32 cou32DeviceHandle, tenDiscovererType enDiscovererType, tenDeviceConnectionType enDeviceConnectionType,
               tenSPISupport enCarplaySupport);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vReportAuthenticationtatus
       ***************************************************************************/
      /*!
       * \fn     vReportAuthenticationtatus
       * \brief   Callback to Notify the client about  authentication progress
       *          information for a Mirror Link device.
       * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
       * \param  [IN] enDAPStatus : DAP Authentication Progress Status.
       **************************************************************************/
      t_Void vReportAuthenticationtatus(const t_U32 cou32DeviceHandle,
               tenDAPStatus enDAPStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vUpdateDAPStatusCb
       ***************************************************************************/
      /*!
       * \fn     vUpdateDAPStatusCb
       * \brief   Callback to Notify the client about DAP authentication progress
       *          information for a Mirror Link device.
       * \param  [IN] cou32DeviceHandle : Uniquely identifies the target Device.
       * \param  [IN] enDAPStatus : DAP Authentication Progress Status.
       **************************************************************************/
      t_Void vUpdateDAPStatusCb(const t_U32 cou32DeviceHandle,
               tenDAPStatus enDAPStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::bSendRoleSwitchResponse
       ***************************************************************************/
      /*!
       * \fn     bSendRoleSwitchResponse
       * \brief  Sends role switch response to mediaplayer
       * \param  cou32DeviceHandle: [IN}Device ID
       * \retval enRoleswitchRes: Indicates roleswitch required response
       **************************************************************************/
      t_Bool bSendRoleSwitchResponse(const t_U32 cou32DeviceHandle, tenRoleSwitchResponse enRoleswitchRes);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSelectDeviceResultCb
       ***************************************************************************/
      /*!
       * \fn     vSelectDeviceResultCb
       * \brief  Callback to update the select device result
       * \param  cou32DeviceHandle : Device Handle
       * \param  enErrorCode: Error Code
       **************************************************************************/
      t_Void vSelectDeviceResultCb(t_U32 cou32DeviceHandle, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceConnectionCb
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceConnectionCb
       * \brief  Callback to be Called by ML/ DiPo class when a new device is detected.
       * \param  cou32DeviceHandle: Device Handle of the detected device
       * \param  corfrDeviceInfo: Device info for the detected device
       **************************************************************************/
      t_Void vOnDeviceConnectionCb(const trDeviceInfo& corfrDeviceInfo, tenDiscovererType enDiscovererType = e8_DISCOVERER_TYPE_USB);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceDisconnectionCb
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceDisconnectionCb
       * \brief  Callback to be Called by ML/ DiPo class when a device is disconnected
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      t_Void vOnDeviceDisconnectionCb(const t_U32 cou32DeviceHandle, tenDiscovererType enDiscovererType);


      /***************************************************************************
       ** FUNCTION:  spi_tclConnMngr::vOnDeviceSwitchCompleteCb
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceSwitchCompleteCb
       * \brief  Called when device switch is complete
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      t_Void vOnDeviceSwitchCompleteCb(const t_U32 cou32DeviceHandle, t_Bool  bSwitchStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vProceedwithConnection
       ***************************************************************************/
      /*!
       * \fn     vProceedwithConnection
       * \brief  Continues with connection process after authorization
       * \param  corfrDeviceInfo: Device info for the detected device
       **************************************************************************/
      t_Void vProceedwithConnection(const trDeviceInfo& corfrDeviceInfo, tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::bRequestDeviceSwitch
       ***************************************************************************/
      /*!
       * \fn     bRequestDeviceSwitch
       * \brief  Function to request device to switch to SPI mode or normal USB mode
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enDevCat : Indicates SPI or device mode.
       *          @Note: e8DEV_TYPE_UNKNOWN : Indicates switch to USB mode
       **************************************************************************/
      t_Bool bRequestDeviceSwitch(const t_U32 cou32DeviceHandle, tenDeviceCategory enDevCat);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::enChooseDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     enChooseDeviceCategory
       * \brief  returns the type of switch needed for the specified device handle
       * \param  cou32DeviceHandle: Device handle
       * \retval type of switch needed
       **************************************************************************/
      tenDeviceCategory enChooseDeviceCategory(const t_U32 cou32DeviceHandle,tenDeviceType enDeviceType);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::enChooseAndroidDeviceCategory
       ***************************************************************************/
      /*!
       * \fn     enChooseAndroidDeviceCategory
       * \brief  returns the type of switch needed for android device
       * \retval type of switch needed
       **************************************************************************/
      tenDeviceCategory enChooseAndroidDeviceCategory(tenDeviceCategory enCurrentDeviceCategory);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::enChooseAppleDeviceCategory(const t_U32 cou32DeviceHandle)
       ***************************************************************************/
      /*!
       * \fn     enChooseAppleDeviceCategory
       * \brief  returns the type of switch needed for apple device
       * \retval type of switch needed
       **************************************************************************/
      tenDeviceCategory enChooseAppleDeviceCategory(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vEvaluateErrorStates
       ***************************************************************************/
      /*!
       * \fn     vEvaluateErrorStates
       * \brief  Clears error states for known devices if required
       **************************************************************************/
      t_Void vEvaluateErrorStates(const t_U32 cou32DeviceHandle, tenDeviceCategory enChosenDeviceCategory);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::bRegisterObject(RespBase *poRespBase)
       ***************************************************************************/
      /*!
       * \brief   Registers the response class object
       * \ret      returns true on success
       **************************************************************************/
      t_Bool bRegisterObject(RespBase *poRespBase);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::bUnRegisterObject(RespBase *poRespBase)
       ***************************************************************************/
      /*!
       * \brief    Unregisters the response class object
       * \ret      returns true on success
       **************************************************************************/
      t_Bool bUnRegisterObject(RespBase *poRespBase);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vResetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Resets device switch info
       **************************************************************************/
      t_Void vResetDeviceSwitchInfo(const trDeviceInfo& corfrDeviceInfo);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::vHandleMirrorlinkDevices
       ***************************************************************************/
      /*!
       * \brief   Special handling for mirrorlink devices. Sends mirrorlink command on detection
       **************************************************************************/
      t_Void vHandleMirrorlinkDevices(const trDeviceInfo& corfrDeviceInfo);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclDeviceDiscoverer::bIsCarplayActiveOverWiFi
       ***************************************************************************/
      /*!
       * \brief   Checks if carplay session is already active over WiFi
       **************************************************************************/
      t_Bool bIsCarplayActiveOverWiFi(const trDeviceInfo& corfrDeviceInfo);


	   /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSetDeviceConnectionCount
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceConnectionCount
       * \brief  Sets the device connection count
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enDeviceCategory : Device category
       **************************************************************************/
      t_Void vSetDeviceConnectionCount(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory);


       /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSetDeviceConnectSuccessCount
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceConnectSuccessCount
       * \brief  Sets the device connection success count
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enDeviceCategory : Device category
       **************************************************************************/
      t_Void vSetDeviceConnectSuccessCount(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory);

	   /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vSetDeviceConnectionTime
       ***************************************************************************/
      /*!
       * \fn     vSetDeviceConnectionTime
       * \brief  Sets the time taken for connection establishment
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enDeviceCategory : Device category
       **************************************************************************/
      t_Void vSetDeviceConnectionTime(const t_U32 cou32DeviceHandle, tenDeviceCategory enDeviceCategory);

      /***************************************************************************
      ** FUNCTION:  spi_tclDeviceDiscoverer::vOnSelectDevice
      ***************************************************************************/
      /*!
      * \fn     vOnSelectDevice
      * \brief  Called when an android device is selected 
      * \param  vOnSelectDevice : Select request details for android device
      **************************************************************************/
      t_Void vOnSelectDevice(const trSelectDeviceRequest &rfrSelDeviceRequest);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStartDeviceReporting(...
      ***************************************************************************/
     /*!
      * \fn      vStartDeviceReporting
      * \brief   Start device reporting
      ***************************************************************************/
      t_Void vStartDeviceReporting();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vStartClientAvailabilityTimer(...
       ***************************************************************************/
      /*!
       * \fn      vStartClientAvailabilityTimer
       * \brief   Start timer for  ClientAvailability
       ***************************************************************************/
      t_Void vStartClientAvailabilityTimer();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vCancelClientAvailabilityTimer(...
       ***************************************************************************/
      /*!
       * \fn      vCancelClientAvailabilityTimer
       * \brief   Cancel ClientAvailability timer if device reappears
       ***************************************************************************/
      t_Void vCancelClientAvailabilityTimer();

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::bClientAvailabilityTimerCb
       ***************************************************************************/
      /*!
       * \fn     bClientAvailabilityTimerCb
       * \brief  called on expiry of ClientAvailability timer
       * \param  rTimerID: ID of the timer which has expired
       * \param  pvObject: pointer to object passed while starting the timer
       * \param  pvUserData: data passed during start of the timer
       **************************************************************************/
      static t_Bool bClientAvailabilityTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData);
   
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::vOnFastReConnectionClearDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \fn     vOnFastReConnectionClearDeviceSwitchInfo
       * \brief  Clearing the Device switch info, if there is fast disconnection and reconnection during
       *         role switch/during reverse role switch
       * \param  cou32DeviceHandle: [IN}Device ID
       * \retval None
       **************************************************************************/
      t_Void vOnFastReConnectionClearDeviceSwitchInfo(const t_U32 cou32DeviceHandle);

     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclDeviceDiscoverer::vSetDeviceProfile
      ***************************************************************************/
      /*!
      * \fn       t_Void vSetDeviceProfile
      * \brief   To Post the Device Id to SPI, when a device is disconnected
      * \param   rUserContext : [IN] Context information passed from the caller
      * \param   cou32DeviceHandle    : [IN] Device Id
      * \retval  t_Void
      **************************************************************************/
      t_Void vSetDeviceProfile(const t_U32 cou32DeviceHandle, tenDiscovererType enDiscovererType, tenDeviceProfile enDeviceProfile);

      /***************************************************************************
    ** FUNCTION:  spi_tclDeviceDiscoverer::vOnDeviceReenumeration
     ***************************************************************************/
    /*!
     * \fn     vOnDeviceReenumeration
     * \brief  Setting the Selection info accordingly on re-enumeration and posting device status info
     * \param  cou32DeviceHandle: [IN}Device ID
     * \retval None
     **************************************************************************/
     t_Void vOnDeviceReenumeration(const t_U32 cou32DeviceHandle);
	 
	  /***************************************************************************
       ** FUNCTION:  spi_tclDeviceDiscoverer::bIsWirelessSelectionInProgress
       ***************************************************************************/
      /*!
       * \fn     bIsWirelessSelectionInProgress
       * \brief  Function to find the whether the Selection is in Progress with the Device
       * \param  cou32DeviceHandle: [IN]Device ID
       * \retval None
       **************************************************************************/
      t_Bool bIsWirelessSelectionInProgress(const t_U32 cou32DeviceHandle);

      //! Discoverer callbacks
      trDiscovererCbs m_rDiscovererCallbacks;

      //! Lock to protect m_mapoDiscoverers
      Lock m_LockDiscoverers;

      //! Pointer to response interface for posting asynchronous responses
      spi_tclConnMngrResp *m_poConnMngrResp;

      //! List of discoverer handlers
      std::map<tenDiscovererType, spi_tclDiscovererBase*> m_mapoDiscoverers;

      spi_tclDeviceSwitcherIntf* m_poDeviceSwitcher;

      spi_tclDiscoveryDataIntf* m_pDiscoveryData;

      //! Lock to protect switch device critical section
      Lock m_oDeviceSwitchLock;

      //Lock for role switch queue
      Lock m_oRoleSwitchRespQueueLock;

      //! Stores user context for role switch request for mediaplayer
      std::map<t_U32, trRoleSwitchInfo> m_mapRoleSwitchRespQueue;

      //! pointer to mediator
      spi_tclMediatorIntf *m_poMediator;

      spi_tclDeviceListIntf *m_poDeviceListHandler;
      //! Store Device selection request
      trSelectDeviceRequest m_rDeviceSelectionRequest;
	  
      //! Discoverer settings pointer
      spi_tclDiscovererSettingsIntf *m_poDiscovererSettings;
	  
      //! Device under selection
      t_U32 m_u32DeviceUnderSelection;

      //! SPI  feature support indicator
      trSpiFeatureSupport m_rSpiFeatureSupport;

      struct timeval m_rConnectionStartTime;
      //! Indicates if the Master client is available
      t_Bool m_bIsMasterClientAvailable;
      //! Stores pending Device Connection updates before starting discoverer
      std::vector<trDiscoveredDevicesInfo> m_vecPendingDeviceConnection;
      //! Fail safe timer if client fails to update the initialization status
      timer_t m_rClientAvailTimerID;
      //! Lock for protecting access to m_vecPendingDeviceConnection
      Lock m_oPendingConnectionsLock;
	  
	  
      struct trDeviceSelectionStateDetails
      {
        tenDeviceSelectionState m_enUSBDeviceSelectionState;
        tenDeviceSelectionState m_enWifiDeviceSelectionState;
      };
      //! Stores Device selection sates for the devices that are connected
      std::map<t_U32, trDeviceSelectionStateDetails> m_mapDeviceSelectionState;

      //Lock for Device Selection State Map
      Lock m_oDeviceSelectionStateLock;

/*! } */
};
#endif // SPI_TCLDEVICEDISCOVERER_H_
