/*!
 *******************************************************************************
 * \file             spi_tclAppleDiscoverer.cpp
 * \brief            Apple Discoverer
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Apple Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#include "spi_tclAppleDiscoverer.h"
#include "RespRegister.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiscoveryMsgQInterface.h"
#include "spi_tclDiscoveryDispatcher.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "crc.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdAppleDiscovererIntf.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclAppleDiscoverer.cpp.trc.h"
#endif
#endif

static const t_U8 scou8InvalidDeviceHandle = 0;

static const t_U32 scou32TimerInterval_ms = 1500;

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::spi_tclAppleDiscoverer
 ***************************************************************************/
spi_tclAppleDiscoverer::spi_tclAppleDiscoverer(): m_poDiPODiscoverer(NULL), m_rCompRespTimerID(0), m_u32DeviceUnderDisconnection(0)
{
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::spi_tclAppleDiscoverer entered\n"));
   spi_tclDiPOManager *poDiPOManager = spi_tclDiPOManager::getInstance();
   if (NULL != poDiPOManager)
   {
      m_poDiPODiscoverer = poDiPOManager->poGetDiPODiscInstance();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::~spi_tclAppleDiscoverer
 ***************************************************************************/
spi_tclAppleDiscoverer::~spi_tclAppleDiscoverer()
{
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::~spi_tclAppleDiscoverer entered\n"));
   m_poDiPODiscoverer = NULL;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::bInitialize
 ***************************************************************************/
t_Bool spi_tclAppleDiscoverer::bInitialize()
{
  ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer:: bInitialize() entered\n"));

  t_Bool bRetVal = true;

  RespRegister *pRespRegister = RespRegister::getInstance();
  if(NULL!= pRespRegister)
  {
     pRespRegister->bRegisterObject((spi_tclExtRespAppleDiscoverer*)this);

     pRespRegister->bRegisterObject((spi_tclExtRespWiFi*)this);

     pRespRegister->bRegisterObject((spi_tclExtRespBluetooth*)this);
  }
  spi_tclDiPOManager *poDiPOManager = spi_tclDiPOManager::getInstance();
  if (NULL != poDiPOManager)
  {
     //Register for the callbacks from the required callbacks.
     poDiPOManager->bRegisterObject((spi_tclDiPORespDiscoverer*)this);
  }//if(NULL != poDiPOManager )

  return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::bStartDeviceDetection
 ***************************************************************************/
t_Bool spi_tclAppleDiscoverer::bStartDeviceDetection()
{
   t_Bool bRetVal = true;
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdAppleDiscovererIntf *poExtCmdAppleDisc = NULL;
   if (NULL != poExtCompMgr)
   {
      poExtCmdAppleDisc = poExtCompMgr->poGetCmdAppleDiscovererIntfInst();
   }

   if (NULL != poExtCmdAppleDisc)
   {
      poExtCmdAppleDisc->vStartDeviceDetection();
   }

   if(NULL != m_poDiPODiscoverer)
   {
      m_poDiPODiscoverer->vStartDeviceDiscovery();
   }

   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::bStartDeviceDetection bRetVal = %d \n", bRetVal));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vRegisterCallbacks(const trDiscovererCbs corfrDisccbs)
{
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vRegisterCallbacks entered\n"));
   m_rDiscovererCbs = corfrDisccbs;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vStartDeviceDetection
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vStartDeviceDetection()
{
   //! TODO Unregister to mediplayer callbacks here
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vStartDeviceDetection()"));
  
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vStartDeviceReporting
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vStopDeviceDetection()
{
   //! TODO Unregister to mediplayer callbacks here
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vStopDeviceDetection()"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vPostDeviceConnectedMsg
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vPostDeviceConnectedMsg(const trDeviceInfo &rfcoDeviceInfo,
                                                       tenDiscovererType enDiscovererType)
{
   SPI_INTENTIONALLY_UNUSED(enDiscovererType);
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vPostDeviceConnectedMsg entered\n"));

   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vPostDeviceConnectedMsg-BTMAC:%s",
            rfcoDeviceInfo.szBTAddress.c_str()));
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vPostDeviceConnectedMsg-Device Name:%s",
            rfcoDeviceInfo.szDeviceName.c_str()));
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vPostDeviceConnectedMsg- Device Profile :%d",
            ETG_ENUM(USB_PROFILE, rfcoDeviceInfo.enDeviceProfile)));

   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vPostDeviceConnectedMsg - Device Category: %d",
            ETG_ENUM(DEVICE_CATEGORY, rfcoDeviceInfo.enDeviceCategory)));

   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostDeviceConnectedMsg Device Handle = 0x%x  "
            "has ConnType = %d ", rfcoDeviceInfo.u32DeviceHandle,  ETG_ENUM(CONNECTION_TYPE,rfcoDeviceInfo.enDeviceConnectionType)));

   //! Check if the device has already appeared on Bonjour
   trDeviceInfo rStoredDeviceInfo;

   t_Bool bDeviceExists = bGetDeviceInfoFromEitherBTSerialName(rfcoDeviceInfo.szBTAddress.c_str(), rfcoDeviceInfo.szDeviceName.c_str(), rStoredDeviceInfo, rfcoDeviceInfo.szSerialNumber.c_str());
   
   tenDeviceProfile enDeviceProfile =e8_PROFILE_UNKNOWN;
   if(true == bDeviceExists)
   {
      vUpdateOverallDeviceProfile(rStoredDeviceInfo.u32DeviceHandle ,  rfcoDeviceInfo.enDeviceProfile);
      enDeviceProfile = enGetDeviceProfile(rStoredDeviceInfo.u32DeviceHandle, rfcoDeviceInfo.enDeviceConnectionType);
      if (rfcoDeviceInfo.u32DeviceHandle != rStoredDeviceInfo.u32DeviceHandle)
      {
         m_oLockDiPoList.s16Lock();
         if (m_rDiPODeviceList.end() != m_rDiPODeviceList.find(rStoredDeviceInfo.u32DeviceHandle))
         {
            m_rDiPODeviceList.erase(rStoredDeviceInfo.u32DeviceHandle);
         }
         m_oLockDiPoList.vUnlock();
      }
   }

  vUpdateDeviceProfile(enDeviceProfile, rfcoDeviceInfo);

   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vPostDeviceConnectedMsg- Updated Device Profile :%d",
            ETG_ENUM(USB_PROFILE, enDeviceProfile)));

   if (((false == rfcoDeviceInfo.szDeviceName.empty()) &&
            ((e8_PROFILE_IAP2BT_CPW_FEASIBLE == enDeviceProfile) ||
            (e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE == enDeviceProfile) ||
            (e8_PROFILE_CDCNCM_CARPLAY == enDeviceProfile) ||
            (e8_PROFILE_HOSTMODE_CARPLAY == enDeviceProfile) ||
            (e8_PROFILE_NATIVETRANSPORT_MYSPIN == enDeviceProfile) ||
            (e8_PROFILE_NATIVETRANSPORT_CARLIFE == enDeviceProfile) ||
            (e8_PROFILE_DEFAULT == enDeviceProfile))))
   {
      ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostDeviceConnectedMsg:CarPlay capable device is detected. Device Profile:%d",ETG_ENUM(USB_PROFILE,enDeviceProfile)));

      trDeviceInfo rTempDeviceInfo = rfcoDeviceInfo;
      rTempDeviceInfo.enDeviceProfile = enDeviceProfile;
      if(e8_PROFILE_CDCNCM_CARPLAY == enDeviceProfile)
      {
         rTempDeviceInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
         rTempDeviceInfo.szBTAddress = rStoredDeviceInfo.szBTAddress;
      }
      else
      {
        t_Bool bIsDeviceOverBonjour =  bIsDeviceOverNetwork(rfcoDeviceInfo.szBTAddress, rfcoDeviceInfo.szDeviceName);
        if((e8WIRELESS_CONNECTED == rfcoDeviceInfo.enDeviceConnectionType) && (e8_PROFILE_IAP2BT_CPW_FEASIBLE == enDeviceProfile))
        {
           if(true == bIsDeviceOverBonjour)
           {
              rTempDeviceInfo.enDeviceProfile = e8_PROFILE_CDCNCM_CARPLAY;
              rTempDeviceInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
              rTempDeviceInfo.szBTAddress = rfcoDeviceInfo.szBTAddress;
              //! also set the Bonjour Profile for that device
              vSetSubDeviceProfile(rTempDeviceInfo.u32DeviceHandle, e8_PROFILE_CDCNCM_CARPLAY);
           }
        }
      }

      if(e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE == rfcoDeviceInfo.enDeviceProfile)
      {
         rTempDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_NOTSUPPORTED;
      }

      vCheckBTMacValidity(rTempDeviceInfo);
      //! has to be done only after the BTMacValidation as BT mac can be empty from MP
      rTempDeviceInfo.bIsBTFunctionalityLimited = bGetDeviceBTFunctionalityInfo(rfcoDeviceInfo.szBTAddress);

      vUpdateDeviceInfo(rTempDeviceInfo);

      //! Set the Device Profile Accordingly in the map before dispatching the update to discoverer
      vSetSubDeviceProfile(rTempDeviceInfo.u32DeviceHandle, rfcoDeviceInfo.enDeviceProfile, rTempDeviceInfo.enDeviceConnectionType);


      ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostDeviceConnectedMsg Device Handle = 0x%x  "
               "has ConnType = %d and BT Limited functionality as %d", rTempDeviceInfo.u32DeviceHandle,  ETG_ENUM(CONNECTION_TYPE,rTempDeviceInfo.enDeviceConnectionType),
                                                         ETG_ENUM(BOOL, rTempDeviceInfo.bIsBTFunctionalityLimited)));

      vDispatchDeviceConnectionMsg(rTempDeviceInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vCheckBTMacValidity
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vCheckBTMacValidity(trDeviceInfo& rfrDeviceInfo )
{
   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(rfrDeviceInfo.u32DeviceHandle);
   if(itMap != m_rDiPODeviceList.end())
   {
      if(true == rfrDeviceInfo.szBTAddress.empty())
      {
         ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vCheckBTMacValidity Updating the BT Mac for the Device handle %d",rfrDeviceInfo.u32DeviceHandle));
         rfrDeviceInfo.szBTAddress = itMap->second.rDeviceInfo.szBTAddress;
      }
   }
   m_oLockDiPoList.vUnlock();

   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vCheckBTMacValidity left- Device handle %d having BT Mac %s",rfrDeviceInfo.u32DeviceHandle,
            rfrDeviceInfo.szBTAddress.c_str()));
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vDispatchDeviceConnectionMsg
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vDispatchDeviceConnectionMsg(const trDeviceInfo &rfrTempDeviceInfo)
{
   DeviceInfoMsg oDevInfoMsg;
   oDevInfoMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_APPLE;
   *(oDevInfoMsg.m_prDeviceInfo) = rfrTempDeviceInfo;
   spi_tclDiscoveryMsgQInterface* poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oDevInfoMsg, sizeof(oDevInfoMsg));
   } //if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vDispatchDeviceDisconnectionMsg
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vDispatchDeviceDisconnectionMsg(const t_U32 cou32DeviceHandle,
                               tenDiscovererType enDiscovererType,
                               tenDeviceConnectionType enDevConnType,
                               tenSPISupport enCarplayWirelessSupport)
{
   //! Here just send the disconnection update to device discoverer
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vDispatchDeviceDisconnectionMsg entered\n"));
   DeviceDisconnectionMsg oDeviceDisconnectionMsg;
   oDeviceDisconnectionMsg.m_u32DeviceHandle = cou32DeviceHandle;
   oDeviceDisconnectionMsg.m_enDiscovererType = enDiscovererType;
   oDeviceDisconnectionMsg.m_enDeviceConnectionType = enDevConnType;
   oDeviceDisconnectionMsg.m_enCarplaySupport = enCarplayWirelessSupport;
   spi_tclDiscoveryMsgQInterface* poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
       poMsgQinterface->bWriteMsgToQ(&oDeviceDisconnectionMsg,sizeof(oDeviceDisconnectionMsg));
   } //if (NULL != poMsgQinterface)
}


/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vPostDeviceDisconnectedMsg
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vPostDeviceDisconnectedMsg(const t_U32 cou32DeviceHandle,
                                                           tenDiscovererType enDiscovererType,
                                                           tenDeviceConnectionType enDevConnType,
                                                           tenSPISupport enCarplayWirelessSupport)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vPostDeviceDisconnectedMsg entered\n"));
   t_Bool bIsDeviceFound = false;
   m_oLockDiPoList.s16Lock();
   //! TODO - Dont erase the Device from Map just update the Profile in AppleDeviceInfo Accordingly based on the ConnectionType and DeviceHandle
   auto itMap= m_rDiPODeviceList.find(cou32DeviceHandle);
   if(m_rDiPODeviceList.end() != itMap)
   {
      ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vPostDeviceDisconnectedMsg Connection Type is :%d", ETG_ENUM(CONNECTION_TYPE , itMap->second.rDeviceInfo.enDeviceConnectionType)));
      bIsDeviceFound = true;
   }
   m_oLockDiPoList.vUnlock();

   if(true == bIsDeviceFound)
   {
      vSetSubDeviceProfile(cou32DeviceHandle, e8_PROFILE_UNKNOWN , enDevConnType);
   }

   vDispatchDeviceDisconnectionMsg(cou32DeviceHandle, enDiscovererType,enDevConnType, enCarplayWirelessSupport);
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::vOnDeviceListUpdate(...
***************************************************************************/
t_Void spi_tclAppleDiscoverer::vOnDeviceUpdate(const trDiPODeviceInfo& corfrDiPODeviceInfo,t_Bool bIsConnected)
{
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::OnDeviceUpdate-DeviceName:%s",
      corfrDiPODeviceInfo.szDeviceName.c_str()));
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vOnDeviceUpdate-DeviceVersion:%s",
      corfrDiPODeviceInfo.szDeviceVersion.c_str()));
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vOnDeviceUpdate-BTMAC:%s",
      corfrDiPODeviceInfo.szBTMACAddress.c_str()));
   ETG_TRACE_USR1(("[DESC] spi_tclAppleDiscoverer::vOnDeviceUpdate-IsConnected:%d Device under Disconnection 0x%x",
      ETG_ENUM(BOOL,bIsConnected), m_u32DeviceUnderDisconnection));

   tenDeviceProfile enUSBConnectionProfile = e8_PROFILE_UNKNOWN;
   tenDeviceProfile enBTConnectionProfile = e8_PROFILE_UNKNOWN;
   trDeviceInfo rDeviceInfo;
   spi_tclDeviceIDDataIntf oDeviceIDIntf;

   rDeviceInfo.u32DeviceHandle = oDeviceIDIntf.u32GetDeviceIDFromName(corfrDiPODeviceInfo.szDeviceName.c_str()); /*u16CalcCRC16((const t_UChar*) corfrDiPODeviceInfo.szBTMACAddress.c_str(),
    corfrDiPODeviceInfo.szBTMACAddress.size());*/
   rDeviceInfo.szDeviceName = corfrDiPODeviceInfo.szDeviceName;
   rDeviceInfo.szBTAddress = corfrDiPODeviceInfo.szBTMACAddress;

   //! TODO - Find the Device in the Map and then update the AppleDevice Info - Bonjour Status

   if (true == bIsConnected)
   {
      //! add the the device network list, since the device has appeared over network
      vAddBonjourDevicetoList(rDeviceInfo.szDeviceName , rDeviceInfo.szBTAddress);

      //! Check if the device exists in the list
      trDeviceInfo rStoredDeviceInfo;
      t_Bool bDeviceExists = bGetDeviceInfoFromEitherBTSerialName(corfrDiPODeviceInfo.szBTMACAddress.c_str(),corfrDiPODeviceInfo.szDeviceName.c_str(), rStoredDeviceInfo);

      if(true == bDeviceExists)
      {
         m_oLockDiPoList.s16Lock();
         auto itMapDeviceList = m_rDiPODeviceList.find(rStoredDeviceInfo.u32DeviceHandle);
         if(itMapDeviceList != m_rDiPODeviceList.end())
         {
            enUSBConnectionProfile = itMapDeviceList->second.enUSBConnectionProfile;
            enBTConnectionProfile = itMapDeviceList->second.enBTConnectionProfile;
         }
         m_oLockDiPoList.vUnlock();
      }

      t_Bool bDeviceReportedbyMP = (e8_PROFILE_HOSTMODE_CARPLAY == enUSBConnectionProfile)||
                                    (e8_PROFILE_IAP2BT_CPW_FEASIBLE == enBTConnectionProfile);

      //! Check if the device is in HOST MODE, i.e device is detected by mediaplayer
      if (true == bDeviceReportedbyMP)
      {
         rDeviceInfo = rStoredDeviceInfo;
         rDeviceInfo.szBTAddress = corfrDiPODeviceInfo.szBTMACAddress;
         DeviceInfoMsg oDevInfoMsg;
         oDevInfoMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_APPLE;

         //! Final profile updated based on the Bonjour Status and device being reported by MP either after roleswitch or CPW feasibility - wirelessCarplayUpdate
         rDeviceInfo.enDeviceProfile = e8_PROFILE_CDCNCM_CARPLAY;
         rDeviceInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
         rDeviceInfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
         rDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_SUPPORTED;

         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::OnDeviceUpdate Device Handle = 0x%x  "
                  "has ConnType = %d ", rDeviceInfo.u32DeviceHandle,  ETG_ENUM(CONNECTION_TYPE,rDeviceInfo.enDeviceConnectionType)));

         *(oDevInfoMsg.m_prDeviceInfo) = rDeviceInfo;
         spi_tclDiscoveryMsgQInterface *poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
         if (NULL != poMsgQinterface)
         {
            poMsgQinterface->bWriteMsgToQ(&oDevInfoMsg, sizeof(oDevInfoMsg));
         }   //if (NULL != poMsgQinterface)

      }
      else
      {
         ETG_TRACE_USR4(("[DESC] spi_tclAppleDiscoverer::vOnDeviceUpdate -Device id not yet reported by mediaplayer"));
         rDeviceInfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
         rDeviceInfo.enDeviceConnectionStatus = e8DEV_CONNECTED;
         rDeviceInfo.enDeviceConnectionType = e8UNKNOWN_CONNECTION;
         rDeviceInfo.rProjectionCapability.enUSBPortType = e8_PORT_TYPE_OTG;
         rDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_SUPPORTED;
         rDeviceInfo.szBTAddress = corfrDiPODeviceInfo.szBTMACAddress;
         rDeviceInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
         rDeviceInfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
         rDeviceInfo.enDeviceProfile = e8_PROFILE_CDCNCM_CARPLAY;
         rDeviceInfo.enSessionTransport = e8_SESSION_TRANSPORT_INVALID;
      }
      if(m_u32DeviceUnderDisconnection == rDeviceInfo.u32DeviceHandle)
      {
         vCancelDisconnectionTimer();
      }

      vUpdateDeviceInfo(rDeviceInfo);

      vSetSubDeviceProfile(rDeviceInfo.u32DeviceHandle, e8_PROFILE_CDCNCM_CARPLAY);
   }
   else
   {
      m_u32DeviceUnderDisconnection = rDeviceInfo.u32DeviceHandle;
      vHandlingDisconnectionTimer(corfrDiPODeviceInfo);
   }
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::bGetDeviceInfo...(...
***************************************************************************/
t_Bool spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName(t_String szBTMac,t_String szDeviceName, trDeviceInfo &rfrDeviceInfo, t_String szSerialNumber)
{
   t_Bool bRetVal = false;

   m_oLockDiPoList.s16Lock();

   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName Number of devices in the Map %d",m_rDiPODeviceList.size() ));

   for(auto itmap = m_rDiPODeviceList.begin(); itmap!= m_rDiPODeviceList.end(); itmap++)
   {
      ETG_TRACE_USR4(("[DESC] spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName-Device BT Mac:%s", itmap->second.rDeviceInfo.szBTAddress.c_str()));
      ETG_TRACE_USR4(("[DESC] spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName-Device Serial Number:%s", itmap->second.rDeviceInfo.szSerialNumber.c_str()));
      ETG_TRACE_USR4(("[DESC] spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName-DeviceName:%s", itmap->second.rDeviceInfo.szDeviceName.c_str()));

      if (((!szBTMac.empty()) && (szBTMac == itmap->second.rDeviceInfo.szBTAddress))
               || ((!szSerialNumber.empty()) && (szSerialNumber == itmap->second.rDeviceInfo.szSerialNumber))
               || (szDeviceName == itmap->second.rDeviceInfo.szDeviceName))
      {
         if(("invalid" == itmap->second.rDeviceInfo.szSerialNumber) &&
                  (e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE == itmap->second.enBTConnectionProfile))
         {
            ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName Erasing the Invalid device Handle" ));
            m_rDiPODeviceList.erase(itmap);
            bRetVal = false;
            break;
         }
         else
         {
            rfrDeviceInfo = itmap->second.rDeviceInfo;
            bRetVal = true;
            break;
         }
      }
   }
   m_oLockDiPoList.vUnlock();

   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName returned %d for %s\n", ETG_ENUM(BOOL, bRetVal), szDeviceName.c_str()));
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::enGetDeviceProfile(...
***************************************************************************/
tenDeviceProfile spi_tclAppleDiscoverer::enGetDeviceProfile(const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDeviceConnectionType)
{

   tenDeviceProfile enDeviceProfile = e8_PROFILE_UNKNOWN;

  /* Logic is based on the Below mentioned table:
   * *********************************************************************************************************
   * enUSBConnectionProfile * enBTConnectionProfile   *   enBonjour   * Device Profile that will be sent to Discoverer *
   * *********************************************************************************************************
   *        0        *          0           *      0        *         UNKNOWN                                *
   * *********************************************************************************************************
   *        0        *          0           *      1        *         UNKNOWN                                *
   * *********************************************************************************************************
   *        0        *          1           *      0        *        CPW_FEASIBLE                            *
   * *********************************************************************************************************
   *        0        *          1           *      1        *       CDCNCM_CARPLAY                           *
   * *********************************************************************************************************
   *        1        *          0           *      0        *       HOSTMODE_CARPLAY                         *
   * *********************************************************************************************************
   *        1        *          0           *      1        *        CDCNCM_CARPLAY                          *
   * *********************************************************************************************************
   *        1        *          1           *      0        *    Overwritten with profile that comes last    *
   * *********************************************************************************************************
   *        1        *          1           *      1        *          CDCNCM_CARPLAY                        *
   * *********************************************************************************************************/

   m_oLockDiPoList.s16Lock();
   auto itMap= m_rDiPODeviceList.find(cou32DeviceHandle);
   if(m_rDiPODeviceList.end() != itMap)
   {
      ETG_TRACE_USR1((" spi_tclAppleDiscoverer::enGetDeviceProfile for Device Handle : %d", cou32DeviceHandle ));
      ETG_TRACE_USR1((" spi_tclAppleDiscoverer::enGetDeviceProfile having  itMap->second.enBonjour : %d", ETG_ENUM(DEVICE_PROFILE, itMap->second.enBonjour) ));
      ETG_TRACE_USR1((" spi_tclAppleDiscoverer::enGetDeviceProfile having  itMap->second.enUSBConnectionProfile : %d", ETG_ENUM(DEVICE_PROFILE, itMap->second.enUSBConnectionProfile) ));
      ETG_TRACE_USR1((" spi_tclAppleDiscoverer::enGetDeviceProfile having  itMap->second.enBTConnectionProfile : %d", ETG_ENUM(DEVICE_PROFILE, itMap->second.enBTConnectionProfile) ));
      ETG_TRACE_USR1((" spi_tclAppleDiscoverer::enGetDeviceProfile having  itMap->second.rDeviceInfo.enDeviceProfile : %d", ETG_ENUM(DEVICE_PROFILE, itMap->second.rDeviceInfo.enDeviceProfile) ));

      if((e8_PROFILE_CDCNCM_CARPLAY == itMap->second.enBonjour) && ((e8_PROFILE_IAP2BT_CPW_FEASIBLE == itMap->second.rDeviceInfo.enDeviceProfile)
               || (e8_PROFILE_HOSTMODE_CARPLAY == itMap->second.rDeviceInfo.enDeviceProfile) || (e8_PROFILE_CDCNCM_CARPLAY == itMap->second.rDeviceInfo.enDeviceProfile)))
      {
         enDeviceProfile = itMap->second.enBonjour;
      }
      else  if((e8_PROFILE_UNKNOWN != itMap->second.enUSBConnectionProfile) || (e8_PROFILE_UNKNOWN != itMap->second.enBTConnectionProfile))
      {
         //! Based on Connection type return the profile
         //! if connection type is UNKNOWN - then getDeviceProfile request is from onDeviceUpdate
         switch(enDeviceConnectionType)
         {
            case e8UNKNOWN_CONNECTION:
               if(e8_PROFILE_UNKNOWN != itMap->second.enUSBConnectionProfile)
               {
                  enDeviceProfile = itMap->second.enUSBConnectionProfile;
               }
               else if(e8_PROFILE_UNKNOWN != itMap->second.enBTConnectionProfile)
               {
                  enDeviceProfile = itMap->second.enBTConnectionProfile;
               }
               break;

            case e8USB_CONNECTED:
               enDeviceProfile = itMap->second.enUSBConnectionProfile;
               break;

            case e8WIRELESS_CONNECTED:
               enDeviceProfile = itMap->second.enBTConnectionProfile;
               break;
         }
      }
      else  if((e8_PROFILE_UNKNOWN == itMap->second.enUSBConnectionProfile) && (e8_PROFILE_UNKNOWN == itMap->second.enBTConnectionProfile))
      {
         enDeviceProfile = e8_PROFILE_UNKNOWN;
      }
   }
   m_oLockDiPoList.vUnlock();
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::enGetDeviceProfile for 0x%x is %d having device connection type as %d\n",cou32DeviceHandle, ETG_ENUM(DEVICE_PROFILE, enDeviceProfile), ETG_ENUM(CONNECTION_TYPE, enDeviceConnectionType)));
   return enDeviceProfile;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::vStartDisconnectionTimer(...
***************************************************************************/
t_Void spi_tclAppleDiscoverer::vStartDisconnectionTimer()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vStartDisconnectionTimer entered\n"));
   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer))
   {
      timer_t rTimerID;
      //! Start timer and wait for Bonjour update
      poTimer->StartTimer(rTimerID, scou32TimerInterval_ms, 0, this,
               &spi_tclAppleDiscoverer::bDisconnectionTimerCb, NULL);
      m_rCompRespTimerID = rTimerID;
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::vCancelDisconnectionTimer(...
***************************************************************************/
t_Void spi_tclAppleDiscoverer::vCancelDisconnectionTimer()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vCancelDisconnectionTimer entered"));
   if (0 != m_rCompRespTimerID)
   {
      Timer* poTimer = Timer::getInstance();
      //!  If the device appeared over Bonjour, cancel waiting
      if ((NULL != poTimer))
      {
         poTimer->CancelTimer(m_rCompRespTimerID);
      }
	  m_rCompRespTimerID = 0;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::bDisconnectionTimerCb
 ***************************************************************************/
t_Bool spi_tclAppleDiscoverer::bDisconnectionTimerCb(timer_t rTimerID, t_Void *pvObject,
         const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::bDisconnectionTimerCb"));
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   t_Bool bIsDeviceDisappearedOverBonjour = false;
   //! Device details that is under disconnection
   t_String szDeviceName;
   t_String szDeviceBTMacAddress;

   //  If the device didn't appear over Bonjour, assume disconnection
   spi_tclAppleDiscoverer* poAppleDiscoverer= static_cast<spi_tclAppleDiscoverer*>(pvObject);
   if(NULL != poAppleDiscoverer)
   {
      ETG_TRACE_USR4(("[PARAM]spi_tclAppleDiscoverer::bDisconnectionTimerCb Device under Disconnection 0x%x", poAppleDiscoverer->m_u32DeviceUnderDisconnection));
      poAppleDiscoverer->vCancelDisconnectionTimer();

      poAppleDiscoverer->m_oLockDiPoList.s16Lock();
      auto itMap = poAppleDiscoverer->m_rDiPODeviceList.find(poAppleDiscoverer->m_u32DeviceUnderDisconnection);
      if (poAppleDiscoverer->m_rDiPODeviceList.end() != itMap)
      {
         ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::Clearing Bonjour Discovery update for 0x%x", poAppleDiscoverer->m_u32DeviceUnderDisconnection));
         itMap->second.rDeviceInfo.enDeviceProfile = e8_PROFILE_UNKNOWN;
         itMap->second.enBonjour = e8_PROFILE_UNKNOWN;
         bIsDeviceDisappearedOverBonjour = true;
         szDeviceName = itMap->second.rDeviceInfo.szDeviceName;
         szDeviceBTMacAddress = itMap->second.rDeviceInfo.szBTAddress;
      }
      poAppleDiscoverer->m_oLockDiPoList.vUnlock();

      //! Remove the Device from theSet of Network Devices
      poAppleDiscoverer->vRemoveBonjourDeviceFromList(szDeviceName, szDeviceBTMacAddress);

      //! set the Device Bonjour Profile Accordingly
      poAppleDiscoverer->vSetSubDeviceProfile(poAppleDiscoverer->m_u32DeviceUnderDisconnection, e8_PROFILE_UNKNOWN);

      if(true == bIsDeviceDisappearedOverBonjour)
      {
         poAppleDiscoverer->vDispatchDeviceProfileMsg(poAppleDiscoverer->m_u32DeviceUnderDisconnection, e8_PROFILE_UNKNOWN);
      }
   }
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vHandlingDisconnectionTimer
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vHandlingDisconnectionTimer(const trDiPODeviceInfo& corfrDiPODeviceInfo)
{
   SPI_INTENTIONALLY_UNUSED(corfrDiPODeviceInfo);
   ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vHandlingDisconnectionTimer"));

   t_Bool bCPWEnabled = false;
   t_Bool bIsDeviceDisappearedOverBonjour = false;
   

   if(NULL != m_poDiscovererSettings)
   {
      bCPWEnabled = m_poDiscovererSettings->bReadCPWStatus();
   }
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vHandlingDisconnectionTimer Variant has CPW Enabled - %d",ETG_ENUM(BOOL, bCPWEnabled)));

   //Starting the Disconnection timer only for CPW Variants
   //If the USB variant HU, then no need to call this timer - Since the for only CPW - Device Discovery over Bonjour is not Consistent
   if(true == bCPWEnabled)
   {
      ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vHandlingDisconnectionTimer -  Starting the Device Disconnection timer - Bonjour Discovery for CPW"));
      vStartDisconnectionTimer();
   }
   else
   {
      //Clearing the Bonjour Discovery Flags on reverse role switch
      m_oLockDiPoList.s16Lock();
      auto itMap = m_rDiPODeviceList.find(m_u32DeviceUnderDisconnection);
      if (m_rDiPODeviceList.end() != itMap)
      {
         ETG_TRACE_USR1(("[FUNC]spi_tclAppleDiscoverer::vHandlingDisconnectionTimer Clearing Bonjour Discovery update for 0x%x", m_u32DeviceUnderDisconnection));
         itMap->second.rDeviceInfo.enDeviceProfile = e8_PROFILE_UNKNOWN;
         itMap->second.enBonjour = e8_PROFILE_UNKNOWN;
         bIsDeviceDisappearedOverBonjour = true;
      }
      m_oLockDiPoList.vUnlock();

      //! Remove the Device from theSet of Network Devices
      vRemoveBonjourDeviceFromList(corfrDiPODeviceInfo.szDeviceName, corfrDiPODeviceInfo.szBTMACAddress);

      //! set the Device Bonjour Profile Accordingly
      vSetSubDeviceProfile(m_u32DeviceUnderDisconnection, e8_PROFILE_UNKNOWN);

      if(true == bIsDeviceDisappearedOverBonjour)
      {
         vDispatchDeviceProfileMsg(m_u32DeviceUnderDisconnection, e8_PROFILE_UNKNOWN);
      }
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostDeviceListMsg()
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vPostDeviceListMsg(const trDeviceInfo &rfcoDeviceInfo)
{
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostDeviceListMsg Entered"));
   ETG_TRACE_USR4(("[PARAM] vPostDeviceListMsg::vPostDeviceListMsg():Device ID:%d ",rfcoDeviceInfo.u32DeviceHandle));
   ETG_TRACE_USR4(("[PARAM] vPostDeviceListMsg::vPostDeviceListMsg():Device Type :%d ",
            ETG_ENUM(DEVICE_TYPE, rfcoDeviceInfo.rProjectionCapability.enDeviceType)));
   ETG_TRACE_USR4(("[PARAM] vPostDeviceListMsg::vPostDeviceListMsg(): BTAddress :%s ",
            rfcoDeviceInfo.szBTAddress.c_str()));
   ETG_TRACE_USR4(("[PARAM] vPostDeviceListMsg::vPostDeviceListMsg():serial number %s ",
            rfcoDeviceInfo.szSerialNumber.c_str()));
   ETG_TRACE_USR4(("[PARAM] vPostDeviceListMsg::vPostDeviceListMsg(): Device Name %s ",
            rfcoDeviceInfo.szDeviceName.c_str()));

   t_Bool bIsDeviceNameEmpty = true;
   t_Bool bCheckDeviceDisconnection = false;
   tenDeviceProfile enBTConnectionProfile = e8_PROFILE_UNKNOWN;

   //! Updating Device Name as a Separate Message as there was issue -> Connect the device over USB and then say "NO" on AOC --> Device is palaying as iPOD but
   //! Device is listed without Name in the Common Connection Managerlist.
   //! Dispatch Message only if the Device is not exisiting in the Map, otherwise there will be un-necessary updates to Device Discoverer
   m_oLockDiPoList.s16Lock();
   auto itMapDipoList = m_rDiPODeviceList.find(rfcoDeviceInfo.u32DeviceHandle);
   if(itMapDipoList == m_rDiPODeviceList.end())
   {
      if(false == rfcoDeviceInfo.szDeviceName.empty())
      {
         bIsDeviceNameEmpty = false;
      }
   }
   else
   {
      enBTConnectionProfile = itMapDipoList->second.enBTConnectionProfile;
      bCheckDeviceDisconnection = true;
   }
   m_oLockDiPoList.vUnlock();

   if(false == bIsDeviceNameEmpty)
   {
      UpdateDeviceNameMsg oDevNameMsg(rfcoDeviceInfo.u32DeviceHandle, rfcoDeviceInfo.szDeviceName, e8_DISCOVERER_TYPE_APPLE);
      spi_tclDiscoveryMsgQInterface *poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oDevNameMsg, sizeof(oDevNameMsg));
      }   //if (NULL != poMsgQinterface)
   }

   if (true == bCheckDeviceDisconnection)
   {
      if ((e8WIRELESS_CONNECTED == rfcoDeviceInfo.enDeviceConnectionType)&&(e8SPI_NOTSUPPORTED == rfcoDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)&&
               (e8_PROFILE_IAP2BT_CPW_FEASIBLE ==  enBTConnectionProfile))
      {
         vSetSubDeviceProfile(rfcoDeviceInfo.u32DeviceHandle, e8_PROFILE_IAP2BT_CPW_NOT_FEASIBLE ,e8WIRELESS_CONNECTED);
         vDispatchDeviceDisconnectionMsg(rfcoDeviceInfo.u32DeviceHandle,e8_DISCOVERER_TYPE_APPLE,rfcoDeviceInfo.enDeviceConnectionType,
                  rfcoDeviceInfo.rProjectionCapability.enCarplayWirelessSupport);
      }
   }
}

/**************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vSetSubDeviceProfile
 **************************************************************************/
t_Void spi_tclAppleDiscoverer::vSetSubDeviceProfile(const t_U32 cou32DeviceHandle, tenDeviceProfile enDeviceProfile,
         tenDeviceConnectionType enDeviceConnectionType)
{
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vSetSubDeviceProfile Entered with Device Handle %d with Device profile %d having connection type as %d",\
            cou32DeviceHandle, ETG_ENUM(DEVICE_PROFILE, enDeviceProfile), ETG_ENUM(CONNECTION_TYPE, enDeviceConnectionType )));


   /* 1. Not found in the Map(FreshDevice) set the Device profile accordingly based on the Device Profile and Connection type
    * 2. Set Device Profile to CDCNCM_CARPLAY, only after MP reported and Bonjour Update has come
    * 3. Otherwise set it accordingly based on the Connection Type and Device Profile
    * 4. Whenever the Device is reported First on Bonjour we might not know the Device connection Type, just set it to CDCNCM_CARPLAY as it has already appeared over Bonjour
    * 5. For Carlife and MySpin, Just forward the Connection to Device discoverer based on USB Profile
    */
   switch(enDeviceConnectionType)
   {
      case e8UNKNOWN_CONNECTION:
      {
         //! Bonjour Update
         m_oLockDiPoList.s16Lock();
         auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
         if(itMap != m_rDiPODeviceList.end())
         {
            //! Device is known to Apple Discoverer
            //! set only Bonjour Profile

            if(e8_PROFILE_UNKNOWN != enDeviceProfile)
            {
               itMap->second.enBonjour = enDeviceProfile;
               if(e8_PROFILE_UNKNOWN != itMap->second.enBTConnectionProfile)
               {
                  itMap->second.bIsDeviceConnectedOverWifi = true;
               }
               if(e8_PROFILE_UNKNOWN != itMap->second.enUSBConnectionProfile)
               {
                  itMap->second.bIsDeviceConnectedOverUSB = true;
               }
            }
            else
            {
               itMap->second.enUSBConnectionProfile = e8_PROFILE_UNKNOWN;
               itMap->second.bIsDeviceConnectedOverUSB = false;

               if(false == itMap->second.bIsDeviceConnectedOverWifi)
               {
                  itMap->second.enBonjour = e8_PROFILE_UNKNOWN;
               }
            }
         }
         else
         {
            //! Device is not known to Apple Discoverer
            //! Insert the Device into the map and populate only Bonjour Profile here
            m_rDiPODeviceList[cou32DeviceHandle].enUSBConnectionProfile = e8_PROFILE_UNKNOWN;
            m_rDiPODeviceList[cou32DeviceHandle].enBTConnectionProfile = e8_PROFILE_UNKNOWN;
            m_rDiPODeviceList[cou32DeviceHandle].enBonjour = enDeviceProfile;
         }
         m_oLockDiPoList.vUnlock();
      }
      break;

      case e8USB_CONNECTED:
      {
         m_oLockDiPoList.s16Lock();
         auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
         if(itMap != m_rDiPODeviceList.end())
         {
            //! Device is known to Apple Discoverer
            //! set only USB Profile
            itMap->second.enUSBConnectionProfile = enDeviceProfile;
         }
         else
         {
            //! Device is not known to Apple Discoverer
            //! Insert the Device into the map and populate only USB Profile here
            m_rDiPODeviceList[cou32DeviceHandle].enUSBConnectionProfile = enDeviceProfile;
            m_rDiPODeviceList[cou32DeviceHandle].enBTConnectionProfile = e8_PROFILE_UNKNOWN;
            m_rDiPODeviceList[cou32DeviceHandle].enBonjour = e8_PROFILE_UNKNOWN;
         }
         m_oLockDiPoList.vUnlock();
      }
      break;

      case e8WIRELESS_CONNECTED:
      {
         m_oLockDiPoList.s16Lock();
         auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
         if(itMap != m_rDiPODeviceList.end())
         {
            //! Device is known to Apple Discoverer
            //! set only BT Connection Profile
            itMap->second.enBTConnectionProfile = enDeviceProfile;
         }
         else
         {
            //! Device is not known to Apple Discoverer
            //! Insert the Device into the map and populate only WIFI Profile here
            m_rDiPODeviceList[cou32DeviceHandle].enUSBConnectionProfile = e8_PROFILE_UNKNOWN;
            m_rDiPODeviceList[cou32DeviceHandle].enBTConnectionProfile = enDeviceProfile;
            m_rDiPODeviceList[cou32DeviceHandle].enBonjour = e8_PROFILE_UNKNOWN;
         }
         m_oLockDiPoList.vUnlock();
      }
      break;

      default:
      {
         //Nothing needs to be handled, as all the three cases are already handled (UNKNOWN, USB, WIFI - Connection Type)
      }
   }


   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
   if(itMap != m_rDiPODeviceList.end())
   {
      ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vSetSubDeviceProfile Entered with USB Connection profile %d",\
              ETG_ENUM(DEVICE_PROFILE, itMap->second.enUSBConnectionProfile)));


      ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vSetSubDeviceProfile Entered with BT Connection profile %d",\
              ETG_ENUM(DEVICE_PROFILE, itMap->second.enBTConnectionProfile)));


      ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vSetSubDeviceProfile Entered with Bonjour %d",\
              ETG_ENUM(DEVICE_PROFILE, itMap->second.enBonjour)));
   }
   m_oLockDiPoList.vUnlock();

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vReportWifiDeviceConnection(...)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vReportWifiDeviceConnection(const t_String coszDeviceIP)
{
   //! Get the Device handle from IP address
   const t_U32 cou32DeviceHandle = u32GetDeviceHandle(coszDeviceIP);
   t_Bool bIsReportDeviceConnection = false;
   trDeviceInfo rTempDeviceInfo;

   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection entered With Device handle %d having IP Address as %s\n", cou32DeviceHandle, coszDeviceIP.c_str()));

   if(scou8InvalidDeviceHandle != cou32DeviceHandle)
   {
      m_oLockDiPoList.s16Lock();
      auto itMap= m_rDiPODeviceList.find(cou32DeviceHandle);
      if(m_rDiPODeviceList.end() != itMap)
      {
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection Bonjour Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.enBonjour)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection Device Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.rDeviceInfo.enDeviceProfile)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection Device Connection Status over WIFI %d",ETG_ENUM(BOOL, itMap->second.bIsDeviceConnectedOverWifi)));

         //@ Two cases:
         //@ A) Fresh Device - MP reports the Device --> Device Connectes to AP (found in map) --> Then device Appears over bonjour --> Set the connected flag as TRUE
         //@ B) Known Device - Two Cases:
         //@                   B.1) Out of BT Range - MP doesn't report the device - Need to handle
         //@                   B.2) Within BT Range - MP reports the Device --> Device Connects to AP(But In the Previous device would not disconnected) --> Set the Flags accordingly and
         //@                                          update the same to the Device discoverer after the device connects to AP

         //! Either Fresh Device or the Previously disconnected device
         if(false == itMap->second.bIsDeviceConnectedOverWifi)
         {
            //@ If Fresh Device/Known device which had disappeared over Bonjour previously
            //@   - Both the Profiles - Bonjour Profile and rDeviceInfo.enDeviceProfile will be UNKNOWN
            //@   - Need not handle anything, as the profiles will be set once the device appears over Bonjour

            //@ If known Device - Bonjour Profile will be CDCNCM_CP and rDeviceInfo.enDeviceProfile will be UNKNOWN
            //@                 - Set the rDeviceInfo.enDeviceProfile will be CDCNCM_CP and Update the same to Device discoverer

            itMap->second.bIsDeviceConnectedOverWifi = true;
            if(e8_PROFILE_UNKNOWN == itMap->second.enBonjour)
            {
               //@ Before Dispatching the WifiDeviceConnectionMessage, Check whethher the MP has reported the device or not.
               //@ That is enBTConnectionProfile should be populated with the CPW_FEASIBLE
               itMap->second.enBonjour = e8_PROFILE_CDCNCM_CARPLAY;
            }

            itMap->second.rDeviceInfo.enDeviceProfile = e8_PROFILE_CDCNCM_CARPLAY;

            if(e8_PROFILE_IAP2BT_CPW_FEASIBLE == itMap->second.enBTConnectionProfile)
            {
               itMap->second.rDeviceInfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
               itMap->second.rDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_SUPPORTED;
               itMap->second.rDeviceInfo.enDeviceCategory = e8DEV_TYPE_DIPO;

               rTempDeviceInfo = itMap->second.rDeviceInfo;
               bIsReportDeviceConnection = true;
            }
         }

         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection left with Bonjour Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.enBonjour)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection left with Device Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.rDeviceInfo.enDeviceProfile)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceConnection left with Device Connection Status over WIFI %d",ETG_ENUM(BOOL, itMap->second.bIsDeviceConnectedOverWifi)));
      }
      else
      {
         //@ Need not handle anything when the device is not found in the map
      }
      m_oLockDiPoList.vUnlock();
   }
   else
   {
      //@ Its new device that is connected to Wifi AP while selection in progress - May be for the same device or the Different device
      //@ **(Set the bIsDeviceConnectedOverWifi flag as TRUE once it appears over Bonjour)
   }

   if(true == bIsReportDeviceConnection)
   {
      vDispatchDeviceConnectionMsg(rTempDeviceInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection(const t_String coszDeviceIP)
{
   //! Get the Device handle from IP address
   const t_U32 cou32DeviceHandle = u32GetDeviceHandle(coszDeviceIP);
   t_Bool bIsReportDeviceDisconnection = false;
   tenDeviceProfile enDeviceProfile = e8_PROFILE_UNKNOWN;

   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection entered With Device handle %d having IP Address as %s\n", cou32DeviceHandle, coszDeviceIP.c_str()));

   if(scou8InvalidDeviceHandle != cou32DeviceHandle)
   {
      m_oLockDiPoList.s16Lock();
      auto itMap= m_rDiPODeviceList.find(cou32DeviceHandle);
      if(m_rDiPODeviceList.end() != itMap)
      {

         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection Bonjour Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.enBonjour)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection USB Connection Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.enUSBConnectionProfile)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection BT Connection Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.enBTConnectionProfile)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection Device Profile %d",ETG_ENUM(DEVICE_PROFILE, itMap->second.rDeviceInfo.enDeviceProfile)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection Device Connection Status over WIFI %d",ETG_ENUM(BOOL, itMap->second.bIsDeviceConnectedOverWifi)));
         ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection Device Connection Status over USB %d",ETG_ENUM(BOOL, itMap->second.bIsDeviceConnectedOverUSB)));

         //@ Dispatch the updated Profile only if the Wirless Session was established.
         //@ Special Case  : CPW session Active --> Tap on CP Icon to end the sessin --> Connect the same device over USB --> Session over USB --> WIFI OFF
         //@ In the above case mentioned, we should not update the Device Profile

         if(true == itMap->second.bIsDeviceConnectedOverWifi)
         {

            //@ Set the bIsDeviceConnectedOverWifi flag as FALSE
            //@ Set the rDeviceInfo.enDeviceProfile and  enBTConnectionProfile to UNKNOWN
            //@ Update the Profile to DeviceDiscoverer
            itMap->second.bIsDeviceConnectedOverWifi = false;
            itMap->second.enBonjour = e8_PROFILE_UNKNOWN;
            itMap->second.rDeviceInfo.enDeviceProfile = e8_PROFILE_UNKNOWN;

            enDeviceProfile = itMap->second.rDeviceInfo.enDeviceProfile;
            bIsReportDeviceDisconnection = true;
         }
      }
      //@ Need not handle anything when the device is not found in the map - Means the device had not appeared over Bonjour Previously
      m_oLockDiPoList.vUnlock();
   }
  //@ Need not handle anything when the device handle is INVALID

   if(true == bIsReportDeviceDisconnection)
   {
      vDispatchDeviceProfileMsg(cou32DeviceHandle, enDeviceProfile);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostWiFiConfig()
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo)
{
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig Entered with Number of stations %d", corvecStationsInfo.size()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig szSSID:%s",rfcorWiFiConfig.szSSID.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig szPassphrase:%s",rfcorWiFiConfig.szPassphrase.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig szInterface:%s",rfcorWiFiConfig.szInterface.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig szMode:%s",rfcorWiFiConfig.szMode.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig szType:%s",rfcorWiFiConfig.szType.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig CurrentOperatingChannel:%d",rfcorWiFiConfig.u32CurrentOperatingChannel));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig enFrequency:%d",ETG_ENUM(WIFI_FREQUENCY,rfcorWiFiConfig.enFrequency)));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig szSecurity:%d",ETG_ENUM(WIFI_SECURITY,rfcorWiFiConfig.enSecurity)));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig bVisible:%d",ETG_ENUM(BOOL,rfcorWiFiConfig.bVisible)));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vPostWiFiConfig bPowered:%d",ETG_ENUM(BOOL,rfcorWiFiConfig.bPowered)));

   /*
    ** Points to be Noted while implementing the wifiDicoverer map:**
    * 1. Recently connected will be on the end of the list
    * 2. If the size is 0, all the devices are disconnected from AP or the AP is being setup now
    * 3. Set all the Device connection state to Disconnected when there is change in the AP type too.
    * 4. Set all the Device Connection State to Disconnected when there is change in the bPowered flag from TRUE to FALSE - i.e, AP is shutdown
    * 5. Set the device connection state to Connected only if the AP type is CPW and bPowered flag is TRUE. And only for the Devices that are present in the Vector provided by the WBL
    * 6. ONLY 10 DEVICES can be CONNECTED to CPW AP at a time
    *
   */

   //! Temporary Map, which is used for Comparisions only
   std::map<t_String, t_String> CurrentStationInfo;
   std::vector<t_String> vecDisconnectedIPAddresses;

   for (trStationInfo itVecStations : corvecStationsInfo)
   {
      CurrentStationInfo[itVecStations.szIPAddress] = itVecStations.szDeviceName;
   }


   if((0 == corvecStationsInfo.size()) || ("Normal" == rfcorWiFiConfig.szType) || (false == rfcorWiFiConfig.bPowered))
   {
      //! Set the all the Devices Connections state to DISCONNECTED irrespective of the AP type, Only If there are one or more devices in the wifidiscoverer map
      m_LockSTAInformationMap.s16Lock();
      for(auto itMap = m_StoredSTAInformation.begin() ; itMap != m_StoredSTAInformation.end() ; itMap++)
      {
         if(true == itMap->second.bIsConnected)
         {
            itMap->second.bIsConnected = false;
            vecDisconnectedIPAddresses.push_back(itMap->first);
         }
      }
      m_LockSTAInformationMap.vUnlock();

      //Report all the devices that are disconnected
      for(t_String szDisconnectedIPAddress : vecDisconnectedIPAddresses)
      {
         //dispatch the DeviceDisconnected Message
         vReportWifiDeviceDisconnection(szDisconnectedIPAddress);
      }
   }
   else if(("CarPlayWireless" == rfcorWiFiConfig.szType) && (true == rfcorWiFiConfig.bPowered))
   {
      //! Device Connected logic
      vUpdateDeviceConnection(corvecStationsInfo);

      //! Device Disconnected logic
      vUpdateDeviceDisconnection(CurrentStationInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vUpdateDeviceConnection(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vUpdateDeviceConnection(const std::vector<trStationInfo>& corvecStationsInfo)
{
   std::vector<t_String> vecConnectedIPAddresses;
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vUpdateDeviceConnection Entered"));
   //! Device Connected logic
   for (trStationInfo itVecStations : corvecStationsInfo)
   {
      m_LockSTAInformationMap.s16Lock();
      //! find the DeviceDetails by the corvecStationsInfo[u8Index].szIPAddress
      auto itMap = m_StoredSTAInformation.find(itVecStations.szIPAddress);
      if (itMap != m_StoredSTAInformation.end())
      {
         if (false == itMap->second.bIsConnected)
         {
            /*
             * 1. Set the connected flag as TRUE and dispatch the DeviceConnected Message
             */
            itMap->second.bIsConnected = true;
            vecConnectedIPAddresses.push_back(itMap->first);
         }
         else
         {
            //! Need not do anything as the device is already connected to HU
         }
      }
      else
      {
         //! Set the connected flag as TRUE
         //! Dispacth the DeviceConnected Message
         m_StoredSTAInformation[itVecStations.szIPAddress].szDeviceName =
                  itVecStations.szDeviceName;
         m_StoredSTAInformation[itVecStations.szIPAddress].bIsConnected = true;

         vecConnectedIPAddresses.push_back(itVecStations.szIPAddress);
      }
      m_LockSTAInformationMap.vUnlock();
   }

   //Report the Device connections for all the devices that are reported as connected
   for(t_String szConnectedIPAddress : vecConnectedIPAddresses)
   {
      //! Dispacth the deviceConnectionMsg
      vReportWifiDeviceConnection(szConnectedIPAddress);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vUpdateDeviceDisconnection(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vUpdateDeviceDisconnection(const std::map<t_String, t_String> &CurrentStationInfo)
{
   std::vector<t_String> vecDisconnectedIPAddresses;
   //! Device Disconnected logic
   m_LockSTAInformationMap.s16Lock();
   for (auto itDeviceListMap = m_StoredSTAInformation.begin(); itDeviceListMap != m_StoredSTAInformation.end();
            itDeviceListMap++)
   {
      if (true == itDeviceListMap->second.bIsConnected)
      {
         auto itDeviceListInSecondMap = CurrentStationInfo.find(itDeviceListMap->first);
         if (itDeviceListInSecondMap == CurrentStationInfo.end())
         {
            //! Device is disconnected, Set the Connected Flag as FALSE.
            itDeviceListMap->second.bIsConnected = false;
            vecDisconnectedIPAddresses.push_back(itDeviceListMap->first);
         }
      }
   }
   m_LockSTAInformationMap.vUnlock();

   //Report all the devices that are disconnected
   for(t_String szDisconnectedIPAddress : vecDisconnectedIPAddresses)
   {
      //dispatch the DeviceDisconnected Message
      vReportWifiDeviceDisconnection(szDisconnectedIPAddress);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::u32GetDeviceHandle
 ***************************************************************************/
t_U32 spi_tclAppleDiscoverer::u32GetDeviceHandle(const t_String coszDeviceIP)
{
   //! Get the Device handle from IP address
   t_U32 u32DeviceHandle = 0;

   spi_tclDeviceIDDataIntf oDataIntf;
   t_U32 u32TempDeviceHandle = oDataIntf.u32GetDeviceIDFromIPAddress(coszDeviceIP);

   m_LockSTAInformationMap.s16Lock();
   auto itMap = m_StoredSTAInformation.find(coszDeviceIP);
   if(m_StoredSTAInformation.end() != itMap)
   {
      if(0 == itMap->second.u32DeviceHandle)
      {
         itMap->second.u32DeviceHandle = u32TempDeviceHandle;
      }
      u32DeviceHandle = itMap->second.u32DeviceHandle;
   }
   m_LockSTAInformationMap.vUnlock();
   ETG_TRACE_USR4(("[DESC] spi_tclAppleDiscoverer::u32GetDeviceHandle left With Device handle as %d and IP Address as %s\n",u32DeviceHandle, coszDeviceIP.c_str()));

   return u32DeviceHandle;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vRestoreSettings Entered"));
   m_oLockDiPoList.s16Lock();
   //clearing the private data on factory reset
   m_rDiPODeviceList.clear();
   m_oLockDiPoList.vUnlock();

   m_LockSTAInformationMap.s16Lock();
   //clearing the private data on factory reset
   m_StoredSTAInformation.clear();
   m_LockSTAInformationMap.vUnlock();

   m_LocksetOfDevicesOverNetwork.s16Lock();
   m_mmapsetOfDevicesOverNetwork.clear();
   m_LocksetOfDevicesOverNetwork.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vClearDeviceDetails
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vClearDeviceDetails(const std::vector<t_U32>& rfvecrDeleteDeviceList)
{

   for (auto itvec = rfvecrDeleteDeviceList.begin(); itvec != rfvecrDeleteDeviceList.end(); ++itvec)
   {
      ETG_TRACE_USR1(("spi_tclDeviceDiscoverer::vDeleteDevice - The device handle to be deleted from Discoverer is : %d", *itvec));
      m_oLockDiPoList.s16Lock();
      auto itMap = m_rDiPODeviceList.find(*itvec);
      if(itMap != m_rDiPODeviceList.end())
      {
         m_rDiPODeviceList.erase(itMap);
      }
      m_oLockDiPoList.vUnlock();
   }

   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdAppleDiscovererIntf *poExtCmdAppleDisc = NULL;
   if (NULL != poExtCompMgr)
   {
      poExtCmdAppleDisc = poExtCompMgr->poGetCmdAppleDiscovererIntfInst();
   }

   if (NULL != poExtCmdAppleDisc)
   {
      poExtCmdAppleDisc->vClearDeviceDetails(rfvecrDeleteDeviceList);
   }

}


/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::enGetDeviceSelectionState
 ***************************************************************************/
tenDeviceSelectionState spi_tclAppleDiscoverer::enGetDeviceSelectionState(const t_U32 cou32DeviceHandle)
{
   tenDeviceSelectionState enDevSelectionState = e8_SELECTION_STATE_DEVICE_NOT_SELECTED;
   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
   if(m_rDiPODeviceList.end() != itMap)
   {
      enDevSelectionState = (itMap->second).enDevSelectionState;
   }
   m_oLockDiPoList.vUnlock();

   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::enGetDeviceSelectionState () Device Handle = 0x%x with selection state =%d",
               cou32DeviceHandle, ETG_ENUM(SELECTION_STATE, enDevSelectionState)));

   return enDevSelectionState;
}


/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vUpdateSelectionState
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vUpdateSelectionState(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enDevSelectionState, tenDeviceConnectionType enDeviceConnType)
{
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vUpdateSelectionState () Device Handle = 0x%x with selection state =%d having Device connection type is %d",
            cou32DeviceHandle, ETG_ENUM(SELECTION_STATE, enDevSelectionState), ETG_ENUM(CONNECTION_TYPE, enDeviceConnType)));

   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
   if(m_rDiPODeviceList.end() != itMap)
   {
      (itMap->second).enDevSelectionState = enDevSelectionState;
   }
   else
   {
     //For Ignition Cycle, CP USB MP reports the devie via Service class and AppleDiscovere is unaware of it.
     m_rDiPODeviceList[cou32DeviceHandle].enDevSelectionState = enDevSelectionState;
   }
   m_oLockDiPoList.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vUpdateDeviceInfo
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vUpdateDeviceInfo(const trDeviceInfo& rTempDeviceInfo)
{
   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(rTempDeviceInfo.u32DeviceHandle);
   if (itMap != m_rDiPODeviceList.end())
   {
      itMap->second.rDeviceInfo = rTempDeviceInfo;
   }
   else
   {
      m_rDiPODeviceList[rTempDeviceInfo.u32DeviceHandle].rDeviceInfo = rTempDeviceInfo;
   }
   m_oLockDiPoList.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vUpdateOverallDeviceProfile
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vUpdateOverallDeviceProfile(const t_U32 cou32DeviceHandle , tenDeviceProfile enDeviceProfile)
{
   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
   if (itMap != m_rDiPODeviceList.end())
   {
      if (((e8_PROFILE_IAP2BT_CPW_FEASIBLE == enDeviceProfile) ||
               (e8_PROFILE_HOSTMODE_CARPLAY == enDeviceProfile)) ||
               (e8_PROFILE_NATIVETRANSPORT_MYSPIN == enDeviceProfile) ||
               (e8_PROFILE_NATIVETRANSPORT_CARLIFE == enDeviceProfile) ||
               (e8_PROFILE_DEFAULT == enDeviceProfile))
      {
         if(e8_PROFILE_CDCNCM_CARPLAY != itMap->second.rDeviceInfo.enDeviceProfile)
         {
            itMap->second.rDeviceInfo.enDeviceProfile = enDeviceProfile;
         }
      }
   }
   m_oLockDiPoList.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::HandleBTLimPreparedState()
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::HandleBTLimPreparedState(const t_String &corfrszBTAddress)
{
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::HandleBTLimPreparedState() entered"));

   vSetBTLimitedFunctionality(corfrszBTAddress, true);

   trDeviceInfo rTempDeviceInfo;
   tenDeviceProfile enBTConnectionProfile = e8_PROFILE_UNKNOWN;

   spi_tclDeviceIDDataIntf oDataIntf;
   t_U32 u32TempDeviceHandle = oDataIntf.u32GetDeviceIDFromBTAddress(corfrszBTAddress);

   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(u32TempDeviceHandle);
   if (itMap != m_rDiPODeviceList.end())
   {
      enBTConnectionProfile = itMap->second.enBTConnectionProfile;

      if (e8_PROFILE_IAP2BT_CPW_FEASIBLE == itMap->second.enBTConnectionProfile)
      {
         if (e8_PROFILE_CDCNCM_CARPLAY == itMap->second.enBonjour)
         {
            itMap->second.rDeviceInfo.enDeviceProfile = e8_PROFILE_CDCNCM_CARPLAY;
         }

         itMap->second.rDeviceInfo.rProjectionCapability.enDeviceType = e8_APPLE_DEVICE;
         itMap->second.rDeviceInfo.rProjectionCapability.enCarplaySupport = e8SPI_SUPPORTED;
         itMap->second.rDeviceInfo.enDeviceCategory = e8DEV_TYPE_DIPO;
         rTempDeviceInfo = itMap->second.rDeviceInfo;
         rTempDeviceInfo.bIsBTFunctionalityLimited = true;
      }
   }
   m_oLockDiPoList.vUnlock();

   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::HandleBTLimPreparedState() left with %d ", enBTConnectionProfile));
   if (e8_PROFILE_IAP2BT_CPW_FEASIBLE == enBTConnectionProfile)
   {
      vDispatchDeviceConnectionMsg(rTempDeviceInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostBTLimitationModeMsg()
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo)
{

   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vPostBTLimitationModeMsg() entered with BT Limitation State as %d",rBTLimitationModeInfo.enActionState ));

   if ( (e8BT_TECH_CARPLAY == rBTLimitationModeInfo.enTechnology) &&
            (e8BT_COMM_CHN_BLUETOOTH != rBTLimitationModeInfo.enCommChannel))
   {
      switch(rBTLimitationModeInfo.enActionState)
      {
         case e8BT_LIM_ACTION_STATE_PREPARED:
         {
            HandleBTLimPreparedState(rBTLimitationModeInfo.szBTAddress);
            break;
         }

         case e8BT_LIM_ACTION_STATE_ACTIVE:
         case e8BT_LIM_ACTION_STATE_ERROR:
         case e8BT_LIM_ACTION_STATE_ERROR_USER_DENIED:
         case e8BT_LIM_ACTION_STATE_IDLE:
         {
            vSetBTLimitedFunctionality(rBTLimitationModeInfo.szBTAddress, false);
            break;
         }

         default:
            //Nothing to do
            break;
      }
   }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vSetBTLimitedFunctionality(
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vSetBTLimitedFunctionality(t_String szBTAddress, t_Bool bIsDeviceBTFunctionalityLimited)
{
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vSetBTLimitedFunctionality:BT Mac Address:%s ",szBTAddress.c_str()));
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::vSetBTLimitedFunctionality: left with BT Functionality limited %d", ETG_ENUM(BOOL, bIsDeviceBTFunctionalityLimited)));

   if(false == szBTAddress.empty())
   {
      m_LockBTFunctionalityLimitation.s16Lock();
      auto itMap = m_BTFunctionalityLimitation.find(szBTAddress);
      if(itMap != m_BTFunctionalityLimitation.end())
      {
         itMap->second = bIsDeviceBTFunctionalityLimited;
      }
      else
      {
         m_BTFunctionalityLimitation[szBTAddress] = bIsDeviceBTFunctionalityLimited;
      }
      m_LockBTFunctionalityLimitation.vUnlock();

      if(false == bIsDeviceBTFunctionalityLimited)
      {
         spi_tclDeviceIDDataIntf oDataIntf;
         t_U32 u32TempDeviceHandle = oDataIntf.u32GetDeviceIDFromBTAddress(szBTAddress);

         if(0 != u32TempDeviceHandle)
         {
            vDispatchBTLimitedFunctionalityMsg(u32TempDeviceHandle, bIsDeviceBTFunctionalityLimited);
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::bGetDeviceBTFunctionalityInfo()
 ***************************************************************************/
t_Bool spi_tclAppleDiscoverer::bGetDeviceBTFunctionalityInfo(t_String szBTAddress)
{
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::bGetDeviceBTFunctionalityInfo: Entered with Device BT Mac %s",szBTAddress.c_str()));
   t_Bool bIsDeviceBTFunctionality = false;
   if (false == szBTAddress.empty())
   {
      m_LockBTFunctionalityLimitation.s16Lock();
      auto itMapDevList = m_BTFunctionalityLimitation.find(szBTAddress);
      if (m_BTFunctionalityLimitation.end() != itMapDevList)
      {
            bIsDeviceBTFunctionality = itMapDevList->second;
      } //  if (m_mapDeviceInfoList.end() != itMapDevList)
      m_LockBTFunctionalityLimitation.vUnlock();
   } // if (0 != cou32DeviceHandle)
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::bGetDeviceBTFunctionalityInfo: left with BT Functionality %d", ETG_ENUM(BOOL, bIsDeviceBTFunctionality)));
   return bIsDeviceBTFunctionality;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::enGetDeviceProfile(...
***************************************************************************/
tenDeviceProfile spi_tclAppleDiscoverer::enGetBTConnectionProfile(const t_U32 cou32DeviceHandle)
{
   tenDeviceProfile enBTConnectionProfile = e8_PROFILE_UNKNOWN;

   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
   if(m_rDiPODeviceList.end() != itMap)
   {
      enBTConnectionProfile = (itMap->second).enBTConnectionProfile;
   }
   m_oLockDiPoList.vUnlock();

   return enBTConnectionProfile;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppleDiscoverer::enGetDeviceProfile(...
***************************************************************************/
tenDeviceProfile spi_tclAppleDiscoverer::enGetUSBConnectionProfile(const t_U32 cou32DeviceHandle)
{

   tenDeviceProfile enUSBConnectionProfile = e8_PROFILE_UNKNOWN;
   m_oLockDiPoList.s16Lock();
   auto itMap = m_rDiPODeviceList.find(cou32DeviceHandle);
   if(m_rDiPODeviceList.end() != itMap)
   {
      enUSBConnectionProfile = (itMap->second).enUSBConnectionProfile;
   }
   m_oLockDiPoList.vUnlock();

   return enUSBConnectionProfile;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppleDiscoverer::vUpdateDeviceProfile
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vUpdateDeviceProfile(tenDeviceProfile &rfrenDeviceProfile,
         const trDeviceInfo& rfcoDeviceInfo)
{
   if ((e8_PROFILE_UNKNOWN == rfrenDeviceProfile) || (e8_PROFILE_DEFAULT == rfrenDeviceProfile) || (e8_PROFILE_DEFAULT == rfcoDeviceInfo.enDeviceProfile))
   {
      rfrenDeviceProfile = rfcoDeviceInfo.enDeviceProfile;
   }
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vUpdateDeviceProfile:left with  the DeviceProfile %d", rfrenDeviceProfile));
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vDispatchDeviceProfileMsg(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vDispatchDeviceProfileMsg(const t_U32 cou32DeviceHandle, tenDeviceProfile enDeviceProfile)
{
   DeviceProfileMsg oDeviceProfileMsg;
   oDeviceProfileMsg.m_u32DeviceHandle = cou32DeviceHandle;
   oDeviceProfileMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_APPLE;
   oDeviceProfileMsg.m_enDeviceProfile = enDeviceProfile;
   spi_tclDiscoveryMsgQInterface* poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oDeviceProfileMsg, sizeof(oDeviceProfileMsg));
   } //if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vDispatchBTLimitedFunctionalityMsg(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vDispatchBTLimitedFunctionalityMsg(const t_U32 cou32DeviceHandle,  t_Bool m_bIsDeviceBTFunctionalityLimited)
{
   BTLimitationStateMsg oBTLimitationStateMsg;
   oBTLimitationStateMsg.m_u32DeviceHandle = cou32DeviceHandle;
   oBTLimitationStateMsg.m_enDiscovererType = e8_DISCOVERER_TYPE_APPLE;
   oBTLimitationStateMsg.m_bIsDeviceBTFunctionalityLimited = m_bIsDeviceBTFunctionalityLimited;
   spi_tclDiscoveryMsgQInterface* poMsgQinterface = spi_tclDiscoveryMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oBTLimitationStateMsg, sizeof(oBTLimitationStateMsg));
   } //if (NULL != poMsgQinterface)
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vAddBonjourDevicetoList(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vAddBonjourDevicetoList( const t_String& corfrszDeviceName , const t_String& corfrszBTMacAddress)
{
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vAddBonjourDevicetoList entered with the Device having  Device Name %s", corfrszDeviceName.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vAddBonjourDevicetoList entered with the Device having  BT Mac Address %s", corfrszBTMacAddress.c_str()));

   //If the two devices have same name, then the entry in the map will be overwritten
   m_LocksetOfDevicesOverNetwork.s16Lock();
   auto itMap = m_mmapsetOfDevicesOverNetwork.find(corfrszDeviceName);
   if(itMap != m_mmapsetOfDevicesOverNetwork.end())
   {
      if(corfrszBTMacAddress != itMap->second)
      {
         m_mmapsetOfDevicesOverNetwork.insert(itMap, {corfrszDeviceName, corfrszBTMacAddress});
      }
   }
   else
   {
      m_mmapsetOfDevicesOverNetwork.insert({corfrszDeviceName, corfrszBTMacAddress});
   }
   m_LocksetOfDevicesOverNetwork.vUnlock();
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vRemoveBonjourDeviceFromList(..)
 ***************************************************************************/
t_Void spi_tclAppleDiscoverer::vRemoveBonjourDeviceFromList(const t_String& corfrszDeviceName, const t_String& corfrszBTMacAddress)
{
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vRemoveBonjourDeviceFromList entered with the Device having Device Name %s  "
            "disappeared over netwrok", corfrszDeviceName.c_str()));

   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::vRemoveBonjourDeviceFromList entered with the Device having  BT Mac %s that has "
            "disappeared over netwrok",  corfrszBTMacAddress.c_str()));

   m_LocksetOfDevicesOverNetwork.s16Lock();
   auto iteratorMap = m_mmapsetOfDevicesOverNetwork.equal_range(corfrszDeviceName);
   if(iteratorMap.first != m_mmapsetOfDevicesOverNetwork.end())
   {
      for(auto itr = iteratorMap.first; itr != iteratorMap.second; ++itr)
      {
         if(itr->second == corfrszBTMacAddress)
         {
            m_mmapsetOfDevicesOverNetwork.erase(itr);
            break;
         }
      }
   }
   m_LocksetOfDevicesOverNetwork.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAppleDiscoverer::bIsDeviceOverNetwork(..)
 ***************************************************************************/
t_Bool spi_tclAppleDiscoverer::bIsDeviceOverNetwork( const t_String &corfrszBTMacAddress, const t_String &corfrszDeviceName)
{
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::bIsDeviceOverNetwork entered with the Device having  Device Name %s", corfrszDeviceName.c_str()));
   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::bIsDeviceOverNetwork entered with the Device having  BT Mac Address %s", corfrszBTMacAddress.c_str()));


   t_Bool bIssDeviceOverBonjour = false;
   t_Bool bCPWEnabled = false;
   if(NULL != m_poDiscovererSettings)
   {
      bCPWEnabled = m_poDiscovererSettings->bReadCPWStatus();
   }
   ETG_TRACE_USR1(("spi_tclAppleDiscoverer::bIsDeviceOverNetwork Variant has CPW Enabled - %d",ETG_ENUM(BOOL, bCPWEnabled)));


   m_LocksetOfDevicesOverNetwork.s16Lock();
   auto iteratorMap = m_mmapsetOfDevicesOverNetwork.equal_range(corfrszDeviceName);
   if(iteratorMap.first != m_mmapsetOfDevicesOverNetwork.end())
   {
      if(false == bCPWEnabled)
      {
         bIssDeviceOverBonjour= true;
      }
      else
      {
         if(false == corfrszBTMacAddress.empty())
         {
            for(auto itr = iteratorMap.first; itr != iteratorMap.second; ++itr)
            {
               if(itr->second == corfrszBTMacAddress)
               {
                  bIssDeviceOverBonjour = true;
                  break;
               }
            }
         }
      }
   }
   m_LocksetOfDevicesOverNetwork.vUnlock();

   ETG_TRACE_USR4(("spi_tclAppleDiscoverer::bIsDeviceOverNetwork left with the Device being  over network : %d", ETG_ENUM(BOOL, bIssDeviceOverBonjour)));

   return bIssDeviceOverBonjour;
}

