/*!
 *******************************************************************************
 * \file             spi_tclAppleDiscoverer.h
 * \brief            Apple Discoverer
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Apple Discoverer
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version
 25.01.2017 |  Unmukt Jain                 | Added usage of external Interfaces
  \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLAPPLEDISCOVERER_H_
#define SPI_TCLAPPLEDISCOVERER_H_

#include "SPITypes.h"
#include "spi_tclDiscovererBase.h"
#include "spi_tclExtRespAppleDiscoverer.h"
#include "spi_tclDiPORespDiscoverer.h"
#include "spi_tclDiPOCmdDiscoverer.h"
#include "spi_tclExtRespWiFi.h"
#include "spi_WifiTypedefs.h"
#include "Lock.h"
#include "Timer.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclExtRespBluetooth.h"

/**
 * \class spi_tclAppleDiscoverer
 * \brief Responsible for reporting connection and disconnection of apple devices
 */
class spi_tclAppleDiscoverer: public spi_tclDiscovererBase, public spi_tclExtRespAppleDiscoverer,
public spi_tclDiPORespDiscoverer, public spi_tclExtRespWiFi, public spi_tclExtRespBluetooth
{

   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::spi_tclAppleDiscoverer
       ***************************************************************************/
      /*!
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclAppleDiscoverer();

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::~spi_tclAppleDiscoverer
       ***************************************************************************/
      /*!
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclAppleDiscoverer();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiscovererBase::bInitialize
       ***************************************************************************/
      /*!
       * \brief   Initializes the Discoverer
       * \retval  true if initialization succeeds
       **************************************************************************/
       t_Bool bInitialize();

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::bStartDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Start detecting apple devices
       * \retval Returns true if the initialization of device detection is successful
       **************************************************************************/
      t_Bool bStartDeviceDetection();

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \brief   Register discoverer callbacks which will be invoked on device connection ad disconnections
       * \param  corfrDisccbs: Discoverer callback structure
       **************************************************************************/
      t_Void vRegisterCallbacks(const trDiscovererCbs corfrDisccbs);

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::vStartDeviceDetection
       ***************************************************************************/
      /*!
       * \brief   Start detecting device connections
       **************************************************************************/
      t_Void vStartDeviceDetection();

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::vStartDeviceReporting
       ***************************************************************************/
      /*!
       * \brief   Stop detecting device connections
       **************************************************************************/
      t_Void vStopDeviceDetection();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostWiFiConfig()
       ***************************************************************************/
      /*!
       * \fn      t_Void vPostWiFiConfig()
       * \brief   Called when ActiveWifiSetup structure is sent by WBL Client Handler.
       *          Mandatory interface to be implemented.
       * \param   [IN] rWifiAPConfig: structure of type trWiFiConfig
       * \param   [IN] corfvecStationsInfo: List of connected stations.
       * \sa      None
       **************************************************************************/
       t_Void vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::vRestoreSettings
       ***************************************************************************/
      /*!
       * \fn     t_Void vRestoreSettings()
       * \brief  vRestoreSettings Method. Invoked on factory reset to clear the private data when SPI Session is active
       * \param  None
       **************************************************************************/
      t_Void vRestoreSettings();

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::vClearDeviceDetails
       ***************************************************************************/
      /*!
       * \fn     vClearDeviceDetails
       * \brief  removes device from Apple Discoverer Map
       * \param  rfvecrDeleteDeviceList: list of devices to be deleted
       **************************************************************************/
      t_Void vClearDeviceDetails(const std::vector<t_U32>& rfvecrDeleteDeviceList);

      /***************************************************************************
       ** FUNCTION:  spi_tclAppleDiscoverer::vUpdateSelectionState
       ***************************************************************************/
      /*!
       * \fn     vUpdateSelectionState
       * \brief  Informs if the device selection is not initiated for a reported device
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
       t_Void vUpdateSelectionState(const t_U32 cou32DeviceHandle, tenDeviceSelectionState enDevSelectionState, tenDeviceConnectionType enDeviceConnType);

   protected:

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostDeviceConnectedMsg()
       ***************************************************************************/
      /*!
       * \fn      t_Void vPostDeviceConnectedMsg()
       * \brief   Called when new USB device is detected
       * \param   corfrDeviceInfo: a trDeviceInfo type structure
       * \param   enDiscovererType: Discoverer type
       * \sa      None
       **************************************************************************/
       t_Void vPostDeviceConnectedMsg(const trDeviceInfo &rfcoDeviceInfo,
   		                                  tenDiscovererType enDiscovererType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostDeviceDisconnectedMsg()
       ***************************************************************************/
      /*!
       * \fn      t_Void vPostDeviceDisconnectedMsg()
       * \brief   Called when USB device is disconnected
       * \param   cou32DeviceHandle: device handle
       * \param   enDiscovererType: Discoverer type
       * \param   enDeviceConnectionType : Device connection/disconnection status
       * \param   enCarplayWirelessSupport : Carplay Wireless Support
       * \sa      None
       **************************************************************************/
       t_Void vPostDeviceDisconnectedMsg(const t_U32 cou32DeviceHandle,
   		                                     tenDiscovererType enDiscovererType,
                                              tenDeviceConnectionType enDevConnType,
                                              tenSPISupport enCarplayWirelessSupport);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vOnDeviceListUpdate(...
       ***************************************************************************/
      /*!
       * \fn      t_Void vOnDeviceUpdate(const trDiPODeviceList& corfrDiPODeviceList,
       *                       t_Bool bIsConnected)
       * \brief   To Post the device info to SPI, when the device list changes
       * \param   corfrDiPODeviceList : [IN] CarPlay Device Info
       * \param   bIsConnected : [IN] device connection status
       * \retval  t_Void
       ***************************************************************************/
       t_Void vOnDeviceUpdate(const trDiPODeviceInfo& corfrDiPODeviceInfo,t_Bool bIsConnected);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::enGetDeviceProfile(...
       ***************************************************************************/
      /*!
       * \fn      tenDeviceProfile enGetDeviceProfile
       * \brief   Get the current device profile
       * \param   cou32DeviceHandle : [IN] Device ID
       * \param   enDeviceConnectionType [IN]: Device Connection type
       * \retval  tenDeviceProfile Device Profile
       ***************************************************************************/
       tenDeviceProfile enGetDeviceProfile(const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDeviceConnectionType = e8UNKNOWN_CONNECTION);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vStartDisconnectionTimer(...
       ***************************************************************************/
      /*!
       * \fn      vStartDisconnectionTimer
       * \brief   Start timer for Bonjour Disconnection
       ***************************************************************************/
       t_Void vStartDisconnectionTimer();

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vCancelDisconnectionTimer(...
       ***************************************************************************/
      /*!
       * \fn      vCancelDisconnectionTimer
       * \brief   Cancel Bonjour Disconnection timer if device reappears
       ***************************************************************************/
       t_Void vCancelDisconnectionTimer();

       /***************************************************************************
        ** FUNCTION:  spi_tclAppleDiscoverer::bDisconnectionTimerCb
        ***************************************************************************/
       /*!
        * \fn     bDisconnectionTimerCb
        * \brief  called on expiry of disconnection timer
        * \param  rTimerID: ID of the timer which has expired
        * \param  pvObject: pointer to object passed while starting the timer
        * \param  pvUserData: data passed during start of the timer
        **************************************************************************/
       static t_Bool bDisconnectionTimerCb(timer_t rTimerID, t_Void *pvObject,
                const t_Void *pvUserData);


       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::bGetDeviceInfoFromEitherBTSerialName(...
       ***************************************************************************/
      /*!
       * \fn      t_Void bGetDeviceInfoFromEitherBTSerialName
       * \brief   Returns device info if exists
       * \param   rfrDeviceInfo : [OUT] CarPlay Device Info
       * \param   szDeviceName : [IN] Device Name
       * \retval  t_Void
       ***************************************************************************/
       t_Bool bGetDeviceInfoFromEitherBTSerialName(t_String u32BTMac, t_String szDeviceName, trDeviceInfo &rfrDeviceInfo, t_String u32SerialNumber = "");

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostDeviceListMsg()
        ***************************************************************************/
       /*!
        * \fn      t_Void vPostDeviceListMsg()
        * \brief   Called on devicelist update
        * \param   corfrDeviceInfo: a trDeviceInfo type structure
        * \sa      None
        **************************************************************************/
        t_Void vPostDeviceListMsg(const trDeviceInfo &rfcoDeviceInfo);

        /***************************************************************************
        ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vReportWifiDeviceConnection
        ***************************************************************************/
        /*!
        * \fn      t_Void vReportWifiDeviceConnection
        * \brief   To Post the device details to SPI, when a new device is detected over Wifi
        * \param   coszDeviceIP    : [IN] Device IP
        * \retval  t_Void
        ***************************************************************************/
        t_Void vReportWifiDeviceConnection(const t_String coszDeviceIP);


        /***************************************************************************
        ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vReportWifiDeviceDisconnection
        ***************************************************************************/
        /*!
        * \fn      virtual t_Void vReportWifiDeviceDisconnection
        * \brief   To Post the Device Id to SPI, when a device is disconnected over Wifi
        * \param   coszDeviceIP    : [IN] Device IP
        * \retval  t_Void
        **************************************************************************/
        t_Void vReportWifiDeviceDisconnection(const t_String coszDeviceIP);

        /***************************************************************************
         ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vPostBTLimitationModeMsg()
         ***************************************************************************/
        /*!
         * \fn      t_Void vPostBTLimitationModeMsg()
         * \brief   Called when BT limitation mode message is  sent by BT Client Handler.
         * \sa      None
         **************************************************************************/
         t_Void vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo);
		

   private:

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vHandlingDisconnectionTimer(...
       ***************************************************************************/
      /*!
       * \fn      vHandlingDisconnectionTimer
       * \brief   Cancel Bonjour Disconnection timer if device reappears - for USB and CPW
       ***************************************************************************/
       t_Void vHandlingDisconnectionTimer(const trDiPODeviceInfo& corfrDiPODeviceInfo);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vSetSubDeviceProfile(...
       ***************************************************************************/
      /*!
       * \fn      tenDeviceProfile vSetSubDeviceProfile
       * \brief   Set the current device profile accordingly based on MP update(Based on the Connection Type) or Bonjour update
       * \param   cou32DeviceHandle : [IN] Device ID
       * \Param   enDeviceProfile : [IN] Device Profile
       * \Param   enDeviceConnectionType : [IN] Device connection type
       ***************************************************************************/
       t_Void vSetSubDeviceProfile(const t_U32 cou32DeviceHandle, tenDeviceProfile enDeviceProfile,
                tenDeviceConnectionType enDeviceConnectionType = e8UNKNOWN_CONNECTION);

       /***************************************************************************
        ** FUNCTION:  t_U32 spi_tclAppleDiscoverer::u32GetDeviceHandle(const t_String coszDeviceIP)
        ***************************************************************************/
       /*!
        * \fn      t_U32 u32GetDeviceHandle(const t_String coszDeviceIP)
        * \brief   To GetDeviceHandle
        * \param   [IN] coszDeviceIP: Device IP
        * \sa      None
        **************************************************************************/
       t_U32 u32GetDeviceHandle(const t_String coszDeviceIP);

       /***************************************************************************
        ** FUNCTION:  t_U32 spi_tclAppleDiscoverer::vUpdateDeviceDisconnection(...)
        ***************************************************************************/
       /*!
        * \fn      t_U32 u32GetDeviceHandle(const t_String coszDeviceIP)
        * \brief   To GetDeviceHandle
        * \param   [IN] CurrentStationInfo: Current Station Information
        * \sa      None
        **************************************************************************/
       t_Void vUpdateDeviceDisconnection(const std::map<t_String, t_String> &CurrentStationInfo);

       /***************************************************************************
        ** FUNCTION:  t_U32 spi_tclAppleDiscoverer::vUpdateDeviceConnection(...)
        ***************************************************************************/
       /*!
        * \fn      t_U32 u32GetDeviceHandle(const t_String coszDeviceIP)
        * \brief   To GetDeviceHandle
        * \param   [IN] corvecStationsInfo: Station Information from WBL
        * \sa      None
        **************************************************************************/
       t_Void vUpdateDeviceConnection(const std::vector<trStationInfo>& corvecStationsInfo);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vCheckBTMacValidity(...)
        ***************************************************************************/
       /*!
        * \fn      T_Void vCheckBTMacValidity(...)
        * \brief   To Check the validity of the BT Mac address and If it is empty from MP, then update it from Map
        * \param   [OUT] rfrTempDeviceInfo: DeviceInfo
        * \sa      None
        **************************************************************************/
       t_Void vCheckBTMacValidity(trDeviceInfo& rfrDeviceInfo );

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vDispatchDeviceConnectionMsg(...)
        ***************************************************************************/
       /*!
        * \fn      vDispatchDeviceConnectionMsg(const trDeviceInfo &rfrTempDeviceInfo)
        * \brief   Dipatches the device connection message
        * \param   [OUT] rfrTempDeviceInfo: DeviceInfo
        * \sa      None
        **************************************************************************/
       t_Void vDispatchDeviceConnectionMsg(const trDeviceInfo &rfrTempDeviceInfo);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vDispatchDeviceDisconnectionMsg(...)
        ***************************************************************************/
       /*!
        * \fn      vDispatchDeviceDisconnectionMsg(const t_U32 cou32DeviceHandle,tenDiscovererType enDiscovererType,
                                                  tenDeviceConnectionType enDevConnType,tenSPISupport enCarplayWirelessSupport)
        * \brief   Dipatches the device disconnection message
        * \param   cou32DeviceHandle : [OUT] Device ID
        * \Param   enDiscovererType : [OUT] Discoverer type
        * \Param   enDeviceConnectionType : [OUT] Device connection type
        * \Param   enCarplayWirelessSupport:[OUT] Carplay Wireless Support
        * \sa      None
        **************************************************************************/

       t_Void vDispatchDeviceDisconnectionMsg(const t_U32 cou32DeviceHandle,tenDiscovererType enDiscovererType,
                                              tenDeviceConnectionType enDevConnType,tenSPISupport enCarplayWirelessSupport);

       /***************************************************************************
        ** FUNCTION:  spi_tclAppleDiscoverer::enGetDeviceSelectionState
        ***************************************************************************/
       /*!
        * \fn     enGetDeviceSelectionState
        * \brief  Informs if the device selection is not initiated for a reported device
        * \param  cou32DeviceHandle: Device handle
        **************************************************************************/
       tenDeviceSelectionState enGetDeviceSelectionState(const t_U32 cou32DeviceHandle);

       /***************************************************************************
        ** FUNCTION:  t_U32 spi_tclAppleDiscoverer::vUpdateDeviceInfo(...)
        ***************************************************************************/
       /*!
        * \fn      vUpdateDeviceInfo(trDeviceInfo& rTempDeviceInfo);
        * \brief   Updating the device info
        * \param   [IN] rTempDeviceInfo: Device Info
        * \sa      None
        **************************************************************************/
        t_Void vUpdateDeviceInfo(const trDeviceInfo& rTempDeviceInfo);

        /***************************************************************************
         ** FUNCTION:  t_U32 spi_tclAppleDiscoverer::vUpdateOverallDeviceProfile(...)
         ***************************************************************************/
        /*!
         * \fn      vUpdateOverallDeviceProfile(const t_U32 cou32DeviceHandle , tenDeviceProfile enDeviceProfile)
         * \brief   Updating over all device profile
         * \param   [IN] cou32DeviceHandle: Device handle
         * \param   [IN] enDeviceProfile : MP reported Device Profile
         **************************************************************************/
         t_Void vUpdateOverallDeviceProfile(const t_U32 cou32DeviceHandle , tenDeviceProfile enDeviceProfile);


         /***************************************************************************
          ** FUNCTION:  spi_tclAppleDiscoverer::vSetBTLimitedFunctionality
          ***************************************************************************/
         /*!
          * \fn     vSetBTLimitedFunctionality()
          * \brief  Called by SPI component when a device's BT Functionality is Limited
          * \param  szBTAddress: Bt Mac Address
          * \param  bIsDeviceBTFunctionalityLimited : Bt Limitated Functionality
          **************************************************************************/
         t_Void vSetBTLimitedFunctionality(t_String szBTAddress, t_Bool bIsDeviceBTFunctionalityLimited);


         /***************************************************************************
          ** FUNCTION:  spi_tclAppleDiscoverer::bGetDeviceBTFunctionalityInfo
          ***************************************************************************/
         /*!
          * \fn     bGetDeviceBTFunctionalityInfo()
          * \brief  Function to get  the BT Limitation (PREPARED) or Not PREPARED for Auto Selection
          * \param  szBTAddress: Bt Mac Address
          **************************************************************************/
         t_Bool bGetDeviceBTFunctionalityInfo(t_String szBTAddress);


         /***************************************************************************
         ** FUNCTION:  t_Void spi_tclAppleDiscoverer::enGetBTConnectionProfile(...
         ***************************************************************************/
        /*!
         * \fn      tenDeviceProfile enGetBTConnectionProfile
         * \brief   Get the current device profile
         * \param   cou32DeviceHandle : [IN] Device ID
         * \retval  tenDeviceProfile Device Profile
         ***************************************************************************/
         tenDeviceProfile enGetBTConnectionProfile(const t_U32 cou32DeviceHandle);

         /***************************************************************************
         ** FUNCTION:  t_Void spi_tclAppleDiscoverer::enGetUSBConnectionProfile(...
         ***************************************************************************/
        /*!
         * \fn      tenDeviceProfile enGetUSBConnectionProfile
         * \brief   Get the current USB Connection profile
         * \param   cou32DeviceHandle : [IN] Device ID
         * \retval  tenDeviceProfile Device Profile
         ***************************************************************************/
         tenDeviceProfile enGetUSBConnectionProfile(const t_U32 cou32DeviceHandle);

         /***************************************************************************
          ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vUpdateDeviceProfile(...
          ***************************************************************************/
         /*!
          * \fn      t_Void vUpdateDeviceProfile
          * \brief   Update the Device Profile
          * \param   enDeviceProfile : [IN] Device Profile
          * \param   rfcoDeviceInfo : Device Info
          * \retval  tenDeviceProfile Device Profile
          ***************************************************************************/
         t_Void vUpdateDeviceProfile(tenDeviceProfile &rfrenDeviceProfile, const trDeviceInfo& rfcoDeviceInfo);

         /***************************************************************************
          ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vDispatchDeviceProfileMsg(...)
          ***************************************************************************/
         /*!
          * \fn      vDispatchDeviceProfileMsg(const t_U32 cou32DeviceHandle, tenDeviceProfile enDeviceProfile)
          * \brief   Dipatches the device Profile message
          * \param   [IN] cou32DeviceHandle: Device Handle
          * \param   [IN] enDeviceProfile : Device Profile
          * \sa      None
          **************************************************************************/
         t_Void vDispatchDeviceProfileMsg(const t_U32 cou32DeviceHandle, tenDeviceProfile enDeviceProfile);


         /***************************************************************************
          ** FUNCTION:  t_Void spi_tclAppleDiscoverer::HandleBTLimPreparedState(...
          ***************************************************************************/
         /*!
          * \fn      t_Void HandleBTLimPreparedState
          * \brief   Sets the BT Limitation State and sends the Device Connection update, if the CPW is feasible
          * \param   enDeviceProfile : [IN] Device Profile
          * \param   rfcoDeviceInfo : Device Info
          * \retval  tenDeviceProfile Device Profile
          ***************************************************************************/
         t_Void HandleBTLimPreparedState(const t_String &corfrszBTAddress);

         /***************************************************************************
          ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vDispatchBTLimitedFunctionalityMsg(...)
          ***************************************************************************/
         /*!
          * \fn      vDispatchBTLimitedFunctionalityMsg(const t_U32 cou32DeviceHandle, tenDeviceProfile enDeviceProfile)
          * \brief   Dipatches the BT Limited Functionality
          * \param   [IN] cou32DeviceHandle: Device Handle
          * \param   [IN] m_bIsDeviceBTFunctionalityLimited : True if BT Limited Functionality is Set
          * \sa      None
          **************************************************************************/
         t_Void vDispatchBTLimitedFunctionalityMsg(const t_U32 cou32DeviceHandle,  t_Bool m_bIsDeviceBTFunctionalityLimited);

         /***************************************************************************
          ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vAddBonjourDevicetoList(...
          ***************************************************************************/
         /*!
          * \fn      t_Void vAddBonjourDevicetoList
          * \brief   On Bonjour update, Device is added to the Set
          * \param   corfrszDeviceName : Device name on which crc is applied to get the Device Handle
          * \param   corfrszBTMacAddress : [IN] Device BT Mac address reported by Network
          ***************************************************************************/
         t_Void vAddBonjourDevicetoList(const t_String& corfrszDeviceName, const t_String& corfrszBTMacAddress);

         /***************************************************************************
          ** FUNCTION:  t_Void spi_tclAppleDiscoverer::vRemoveBonjourDeviceFromList(...
          ***************************************************************************/
         /*!
          * \fn      t_Void vRemoveBonjourDeviceFromList
          * \brief   On Device disappeared over Network, Device is removed from the Set
          * \param   corfrszDeviceName : Device name on which crc is applied to get the Device Handle
          * \param   corfrszBTMacAddress : [IN] Device BT Mac address reported by Network
          ***************************************************************************/
         t_Void vRemoveBonjourDeviceFromList(const t_String& corfrszDeviceName, const t_String& corfrszBTMacAddress);


         /***************************************************************************
          ** FUNCTION:  t_Bool spi_tclAppleDiscoverer::bIsDeviceOverNetwork(...
          ***************************************************************************/
         /*!
          * \fn      t_Void bIsDeviceOverNetwork
          * \brief   Checks whether the Device is still over Network
          * \param   corfrszBTMacAddress : [IN] Device BT Mac address reported by Network
          * \param   corfrszDeviceName : Devcice Name
          ***************************************************************************/
         t_Bool bIsDeviceOverNetwork(const t_String& corfrszBTMacAddress, const t_String &corfrszDeviceName);

      /* structure to provide information of a STA connected/Disconnected to HU Carplay/Android Wireless AP*/
       struct trStationDetails
       {
             t_String szDeviceName;
             t_Bool bIsConnected;
             t_U32 u32DeviceHandle;
             trStationDetails():bIsConnected(false), u32DeviceHandle(0){}
       };

       //! Map to hold the Station details
       std::map<t_String, trStationDetails> m_StoredSTAInformation;

       //! lock for the m_mapSTAInformation
       Lock m_LockSTAInformationMap;


       struct trAppleDeviceInfo
       {
             //! stores the Device Info
             trDeviceInfo rDeviceInfo;
             //! set based on the Role switch
             tenDeviceProfile enUSBConnectionProfile;
             //! set based on the Feasibility reported by MP
             tenDeviceProfile enBTConnectionProfile;
             //! set based on Bonjour update
             tenDeviceProfile enBonjour;
             //! Device Connection status over WIFI
             t_Bool bIsDeviceConnectedOverWifi;
             //! Device Connection status over USB
             t_Bool bIsDeviceConnectedOverUSB;
             //! Selection State for the device
             tenDeviceSelectionState enDevSelectionState;
             trAppleDeviceInfo():
                enUSBConnectionProfile(e8_PROFILE_UNKNOWN), enBTConnectionProfile(e8_PROFILE_UNKNOWN), enBonjour(e8_PROFILE_UNKNOWN),
                bIsDeviceConnectedOverWifi(false), bIsDeviceConnectedOverUSB(false){}
       };

      trDiscovererCbs m_rDiscovererCbs;
      spi_tclDiPOCmdDiscoverer *m_poDiPODiscoverer;

      Lock m_oLockDiPoList;
      //! @TODO: To be removed after R26 integration
      std::map<t_U32, trAppleDeviceInfo> m_rDiPODeviceList;
      //! Stores the disconnection response timer ID
      timer_t m_rCompRespTimerID;

      //! Device under Disconnection
      t_U32 m_u32DeviceUnderDisconnection;

      //! BT functionality Limitation
      std::map<t_String, t_Bool> m_BTFunctionalityLimitation;
      //! lock for the m_mapSTAInformation
      Lock m_LockBTFunctionalityLimitation;

      //! set of devices that are over network(Bonjour)
      std::multimap<t_String, t_String> m_mmapsetOfDevicesOverNetwork;
      //! lock for the m_setOfDevicesOverNetwork
      Lock m_LocksetOfDevicesOverNetwork;

};

#endif  /* SPI_TCLAPPLEDISCOVERER_H_ */
