/*!
 *******************************************************************************
 * \file             spi_tclOnCarSwitch.cpp
 * \brief            AAP device switcher
 * \addtogroup       Connectivity
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:     OnCar device switcher
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                 | Modifications
 06.02.2018 |  Unmukt Jain            | Initial Version

 \endverbatim
 ******************************************************************************/
#if  defined (GEN3X86) || defined (GEN4LSIM)
   #include <AoapTransport.h>
#else
   #include <aauto/AoapTransport.h>
#endif // GEN3X86 
#include "spi_usb_discoverer_types.h"

#include "spi_tclOnCarSwitch.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclOnCarSwitch.cpp.trc.h"
#endif
#endif


static const t_U32 scou32MaxSwitchDelay = 10000;
static const t_S32 scos32InvalidID = 0x7FFFFFFF;
static const t_Char* sczHeadUnitDescription = "Install oncar";
static const t_Char* sczMobileManufacturer= "UBridge Co., Ltd.";
static const t_Char* sczMobileModelName = "oncar-RD1";
static const t_Char* sczApplicationURI = "http://oncar.com/install/RD1";
static const t_Char* sczApplicationVersion = "V3.0.0.0_bosch";

using namespace adit::aauto;

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarSwitch::spi_tclOnCarSwitch
 ***************************************************************************/
spi_tclOnCarSwitch::spi_tclOnCarSwitch()
{

}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarSwitch::~spi_tclOnCarSwitch()
 ***************************************************************************/
spi_tclOnCarSwitch::~spi_tclOnCarSwitch()
{

}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarSwitch::bSwitchDeviceMode
 ***************************************************************************/
tenDeviceSwitchResult spi_tclOnCarSwitch::enSwitchDeviceMode(const trUSBDeviceInfo &rfrDeviceInfo, trDeviceSwitchInfo &rfrDeviceSwitchInfo,
         tenDeviceType enDeviceType)
{
   SPI_INTENTIONALLY_UNUSED(enDeviceType);
   ETG_TRACE_USR1(("[DESC] spi_tclOnCarSwitch::bSwitchDeviceMode entered USBProfile = %d", ETG_ENUM(USB_PROFILE,rfrDeviceInfo.enUSBProfile)));
   tenDeviceSwitchResult enRetVal = e8_DEVICESWITCH_RESULT_UNKNOWN;
   //! Populate AOAP device information
   aoapDeviceInfo_t rAoapInfo;
   rAoapInfo.aoapAccessoryInfo.manufacturer = sczMobileManufacturer;
   rAoapInfo.aoapAccessoryInfo.modelName = sczMobileModelName;
   rAoapInfo.aoapAccessoryInfo.description = sczHeadUnitDescription;
   rAoapInfo.aoapAccessoryInfo.version = sczApplicationVersion;
   rAoapInfo.aoapAccessoryInfo.uri = sczApplicationURI;
   rAoapInfo.aoapAccessoryInfo.serial = m_rHeadUnitInfo.szVehicleID.c_str();
   rAoapInfo.aoapAccessoryInfo.enableAudio = 0;

   //! Display Accessory info
   ETG_TRACE_USR4((" Device Accessory Information"));
   ETG_TRACE_USR4((" manufacturer-%s, ", rAoapInfo.aoapAccessoryInfo.manufacturer));
   ETG_TRACE_USR4((" modelName-%s, ", rAoapInfo.aoapAccessoryInfo.modelName));
   ETG_TRACE_USR4((" description-%s, ", rAoapInfo.aoapAccessoryInfo.description));
   ETG_TRACE_USR4((" version-%s, ", rAoapInfo.aoapAccessoryInfo.version));
   ETG_TRACE_USR4((" uri-%s, ", rAoapInfo.aoapAccessoryInfo.uri));
   ETG_TRACE_USR4((" serial-%s, ", rAoapInfo.aoapAccessoryInfo.serial));
   ETG_TRACE_USR4((" enableAudio-%d, ", rAoapInfo.aoapAccessoryInfo.enableAudio));

   //! initialize AOAP transport info. This will be populated after AOAP switch
   aoapTransportInfo_t rTransportInfo;
   rTransportInfo.aoapAccessoryId = scos32InvalidID;
   rTransportInfo.aoapDeviceId = scos32InvalidID;

   rAoapInfo.vendorId = rfrDeviceInfo.u32VendorID;
   rAoapInfo.productId = rfrDeviceInfo.u32ProductID;
   rAoapInfo.pSerial = strdup(rfrDeviceInfo.szSerialNumber.c_str());

   //! Switch the device to AOAP mode. Below call is required even if the device is in AOAP mode
   //! This is to obtain the accessory and device id. If the device is already in aoap mode don't wait for AOAP callback
   ETG_TRACE_USR1(("Device to be switched to aoap mode found in the list and has VendorID=0x%x, ProductID = 0x%x and serial number = %s", rAoapInfo.vendorId, rAoapInfo.productId, rAoapInfo.pSerial));
   AoapDevice oAoapDevice;
   //! /warning : Blocking Call. Timeout 10 seconds
   t_S32 s32Error = oAoapDevice.switchDevice(&rAoapInfo, &rTransportInfo, scou32MaxSwitchDelay);
   enRetVal = (0 == s32Error) ? e8_DEVICESWITCH_RESULT_SUCCESS : e8_DEVICESWITCH_RESULT_FAILED;

   if (e8_DEVICESWITCH_RESULT_SUCCESS == enRetVal)
   {
      //! Pass the AOAP transport info. This information will be used in session class of AAP Wrapper to start transport
      rfrDeviceSwitchInfo.s32HeadUnitID = rTransportInfo.aoapAccessoryId;
      rfrDeviceSwitchInfo.s32SwitchedDeviceID = rTransportInfo.aoapDeviceId;
   }

   if (e8_PROFILE_AOAP != rfrDeviceInfo.enUSBProfile)
   {
      rfrDeviceSwitchInfo.bSwitchInProgress = (e8_DEVICESWITCH_RESULT_SUCCESS == enRetVal);

      ETG_TRACE_USR1(("Switching to AOAP mode: Result = %d for DeviceID =0x%x, AccessoryID=%d Errorcode =%d",
               ETG_ENUM(DEVICESWITCH_RESULT, enRetVal), rTransportInfo.aoapDeviceId, rTransportInfo.aoapAccessoryId, s32Error));
   }
   else
   {
      ETG_TRACE_USR4(("spi_tclOnCarSwitch::enSwitchDeviceMode: Already in AOAP mode"));
      enRetVal = e8_DEVICESWITCH_RESULT_SWITCH_NOTREQUIRED;
   }
   return enRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarSwitch::vSetHeadUnitInfo
 ***************************************************************************/
t_Void spi_tclOnCarSwitch::vSetHeadUnitInfo(const trHeadUnitInfo &corfrHeadUnitInfo)
{
   m_rHeadUnitInfo = corfrHeadUnitInfo;
}

