/*!
 *******************************************************************************
 * \file             spi_tclDeviceSwitcherIntf.h
 * \brief            Device Profile Switcher
 * \addtogroup       Device Switcher
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Device Profile Switcher
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.05.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLDEVICESWTCHERINTF_H_
#define SPI_TCLDEVICESWTCHERINTF_H_

#include <functional> //! C++11
#include "SPITypes.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclDiscovererSettingsIntf.h"


/**
 * \class spi_tclDeviceSwitcherIntf
 * \brief Device switcher is responsible for switching devices to different connection modes
 *          For example in case of USB, Device switcher is responsible to switch device to AOAP mode,
 *          Mirrorlink mode, Role switch for Carplay, USB reset etc.
 *          Class needs to be extended for Wi-Fi in future
 */

class spi_tclDeviceSwitcherIntf
{

   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::spi_tclDeviceSwitcherIntf
       ***************************************************************************/
      /*!
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclDeviceSwitcherIntf()
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::~spi_tclDeviceSwitcherIntf()
       ***************************************************************************/
      /*!
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclDeviceSwitcherIntf()
      {

      }

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \brief   Register discoverer callbacks
       * \brief   corfrDisccbs: Structure containing the function pointers to be invoked
       **************************************************************************/
      virtual t_Void vRegisterCallbacks(const trDeviceSwitcherCbs corfrDisccbs) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vPrepareDeviceSwitch
       ***************************************************************************/
      /*!
       * \brief   Initialize and prepare device switch information
       * \param   enDeviceCat: SPI technology to which the device has to be switched to
       * \param   cou32DeviceHandle: Device Handle
       * \param   enDeviceType: Device Type : Android/ Apple etc
	   * \param   enDeviceConnType : Device connection transport
       **************************************************************************/
      virtual t_Void vPrepareDeviceSwitch(tenDeviceCategory enDeviceCat, const t_U32 cou32DeviceHandle, tenDeviceConnectionType enDeviceConnType) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::bSwitchDeviceMode
       ***************************************************************************/
      /*!
       * \brief   Switch the connection profile of the device
       * \param   enDeviceCat: SPI technology to which the device has to be switched to
       * \param   cou32DeviceHandle: Device Handle
       * \param   enDeviceType: Device Type : Android/ Apple etc
       **************************************************************************/
      virtual t_Bool bSwitchDeviceMode(tenDeviceCategory enDeviceCat, const t_U32 cou32DeviceHandle,
               tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceSwitcherIntf::vSetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \brief   Sets the Headunit information such as model name, vehicle ID etc.
       *          This function has to be called on load settings so that the underlying discoverers
       *          are aware of this info before performing the switch operations
       * \param   corfrHeadUnitInfo: Reference to the structure containing head unit info
       **************************************************************************/
      virtual t_Void vSetHeadUnitInfo(const trHeadUnitInfo &corfrHeadUnitInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::bIsSwitchInProgress
       ***************************************************************************/
      /*!
       * \brief   Checks if Device switch is in progress for the specified devicehandle
       * \param   cou32DeviceHandle: Device Handle
       **************************************************************************/
      virtual t_Bool bIsSwitchInProgress(const t_U32 cou32DeviceHandle) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vResetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Reset device switch information. This method should be invoked once the device switch is complete
       *          i.e when the device reappears in projection mode
       * \param   cou32DeviceHandle: Device Handle
       **************************************************************************/
      virtual t_Void vResetDeviceSwitchInfo(const t_U32 cou32DeviceHandle) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vGetDeviceSwitchInfo
       ***************************************************************************/
      /*!
       * \brief   Provides device switch information for the device specified by the device handle
       * \param   cou32DeviceHandle: Device Handle
       * \param   [OUT]reference to the device switch information structure
       **************************************************************************/
      virtual t_Void vGetDeviceSwitchInfo(const t_U32 cou32Devicehandle, trDeviceSwitchInfo &rfrDeviceSwitchInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::bIsDeviceSwitcherBusy
       ***************************************************************************/
      /*!
       * \fn     bIsDeviceSwitcherBusy
       * \brief  Returns true if the Device switcher is busy in a profile switch
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       **************************************************************************/
      virtual t_Bool bIsDeviceSwitcherBusy(const t_U32 cou32DeviceHandle) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vSetProjectionState
       ***************************************************************************/
      /*!
       * \fn     vSetProjectionState
       * \brief  Set the projection status for a particular device category
       * \param  enProjType : Projection type
       **************************************************************************/
      virtual t_Void vSetProjectionState(tenDeviceCategory enProjType, tenEnabledInfo enEnabledInfo) = 0;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclDeviceSwitcherIntf::vSetEarlyDeviceSwitchInfo
      ***************************************************************************/
      /*!
      * \brief  Interface to set the early device switch info. Indicates if
      *         early switch is requested or not
      * \param  [IN] cou32DeviceID : Device Id to be checked
      * \retval true if the device is same as last connected device otherwise false
      **************************************************************************/
      virtual t_Void vSetEarlyDeviceSwitchInfo(const t_U32 cou32DeviceID, t_Bool bEarlySwitchRequired) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vClearSwitchFailureCount
       ***************************************************************************/
      /*!
       * \fn     vClearSwitchFailureCount
       * \brief  Clear switch failure count
       * \param  cou32DeviceHandle : Device handle of the disconnected device
       **************************************************************************/
      virtual t_Void vClearSwitchFailureCount(const t_U32 cou32DeviceHandle) =0;
	  
      /***************************************************************************
       ** FUNCTION:  t_Void  spi_tclDeviceSwitcherIntf::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
       ***************************************************************************/
       /*!
       * \fn      t_Void vSetDiscoverersettingsInstance()
       * \brief   Method to set discoverer settings instance.
       * \param   poDiscovererSettingsIntf  : pointer to poDiscovererSettingsIntf.
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)= 0;
	  
	        /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcherIntf::vOnSwitchStatusFailed
       ***************************************************************************/
      /*!
       * \brief   This method should be invoked once the device switch is failed, i.e, on the RoleSwitchReq from MP
       * \param   cou32DeviceHandle: Device Handle
       **************************************************************************/
      virtual t_Void vOnSwitchStatusFailed(const t_U32 cou32DeviceHandle, t_Bool bSwitchStatus) = 0;



};
/*! } */

#endif /* SPI_TCLDEVICESWTCHERINTF_H_ */
