/*!
 *******************************************************************************
 * \file             spi_tclDeviceSwitchBase.h
 * \brief            Base class for device switch
 * \addtogroup       Device Switcher
 * \{
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Base class for device switch
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 16.06.2016 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/
#ifndef SPI_TCLDEVICESWITCHBASE_H_
#define SPI_TCLDEVICESWITCHBASE_H_

#include "Lock.h"
#include "SPITypes.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclDiscoveryDataIntf.h"
#include <map>

#include "spi_tclDiscovererSettingsIntf.h"


/**
 * \class spi_tclDeviceSwitchBase
 * \brief Base class for performing device switch
 */

class spi_tclDeviceSwitchBase
{

   public:
      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitchBase::spi_tclDeviceSwitchBase
       ***************************************************************************/
      /*!
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclDeviceSwitchBase(): m_poDiscovererSettings(NULL){}

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitchBase::~spi_tclDeviceSwitchBase()
       ***************************************************************************/
      /*!
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclDeviceSwitchBase(){ m_poDiscovererSettings = NULL;}

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitchBase::bSwitchDeviceMode
       ***************************************************************************/
      /*!
       * \brief   Switch the connection profile of the device
       * \param   enDeviceCat: SPI technology to which the device has to be switched to
       * \param   enDeviceSwitchType: Device Switch type
       * \param   rfrDeviceSwitchInfo: Device switch Information
       * \retval   tenDeviceSwitchResult: Indicates result of device switch
       **************************************************************************/
      virtual tenDeviceSwitchResult enSwitchDeviceMode(const trUSBDeviceInfo &rfrDeviceInfo,trDeviceSwitchInfo &rfrDeviceSwitchInfo, tenDeviceType enDeviceType) = 0;

      /***************************************************************************
       ** FUNCTION:  spi_tclDeviceSwitcher::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \brief   Register discoverer callbacks
       * \brief   corfrDisccbs: Structure containing the function pointers to be invoked
       **************************************************************************/
      virtual t_Void vRegisterCallbacks(const trDeviceSwitcherCbs corfrDisccbs)
      {
         m_rDeviceSwitcherCbs = corfrDisccbs;
      }

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDeviceSwitcher::vSetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \brief   Sets the Headunit information such as model name, vehicle ID etc.
       *          This function has to be called on load settings so that the underlying discoverers
       *          are aware of this info before performing the switch operations
       * \param   corfrHeadUnitInfo: Reference to the structure containing head unit info
       **************************************************************************/
      virtual t_Void vSetHeadUnitInfo(const trHeadUnitInfo &/*corfrHeadUnitInfo*/) {};

      /***************************************************************************
       ** FUNCTION:  t_Void  spi_tclDeviceSwitcher::vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
       ***************************************************************************/
       /*!
       * \fn      t_Void vSetDiscoverersettingsInstance()
       * \brief   Method to set discoverer settings instance.
       * \param   poDiscovererSettingsIntf  : pointer to DiscovererSettingsIntf.
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDiscoverersettingsInstance(spi_tclDiscovererSettingsIntf* poDiscovererSettingsIntf)
      {
         if( NULL != poDiscovererSettingsIntf)
         {
            m_poDiscovererSettings = poDiscovererSettingsIntf;
         }
      }

   protected:
      //! DiscovererSettingsIntf pointer
      spi_tclDiscovererSettingsIntf* m_poDiscovererSettings;
	  
   private:
      //! Stores the function pointer for device switcher callbacks
      trDeviceSwitcherCbs m_rDeviceSwitcherCbs;
};

typedef spi_tclDeviceSwitchBase* GetDeviceSwitch_t(spi_tclDiscoveryDataIntf *poDiscovererDataIntf);
extern std::map<tenDeviceCategory,GetDeviceSwitch_t*> DeviceSwitch;

/*! } */

#endif /* SPI_TCLDEVICESWITCHBASE_H_ */
