  /*!
 *******************************************************************************
 * \file         spi_tclBluetooth.cpp
 * \brief        Bluetooth Manager class that provides interface to delegate
                 the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Bluetooth Manager class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 21.02.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 03.03.2014 |  Ramya Murthy (RBEI/ECP2)         | Included BT DeviceConnection and
                                                  SPI Device Deselection implementation.
 08.03.2014 |  Ramya Murthy (RBEI/ECP2)         | Included call handling.
 06.04.2014 |  Ramya Murthy (RBEI/ECP2)         | Initialisation sequence implementation
 04.06.2014 |  Ramya Murthy (RBEI/ECP2)         | Device switch cancellation handling changes.
 31.07.2014 |  Ramya Murthy (RBEI/ECP2)         | SPI feature configuration via LoadSettings()
 01.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Removed Telephone client handler
 07.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Implementation for BTPairingRequired property
 08.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Impl. of CarPlay DisableBluetooth msg handling
                                                  and SelectedDevice info access protection
 19.05.2015 |  Ramya Murthy (RBEI/ECP2)         | Added vSendBTDeviceName()
 21.06.2016 |  Ramya Murthy (RBEI/ECP2)         | Changed vSendBTDeviceName() to vSendBTDeviceInfo()
                                                  to provide more BT device info
 16.07.2018	|  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetGeneralRestrictions()

 
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLBLUETOOTH_H_
#define _SPI_TCLBLUETOOTH_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclLifeCycleIntf.h"
#include "spi_tclBluetoothIntf.h"
#include "Lock.h"
#include "spi_tclSelectionIntf.h"
#include "RespRegister.h"
#include "spi_tclBluetoothSettingsIntf.h"

class spi_tclExtCmdBluetoothIntf;
class spi_tclExtCmdOOBTIntf;

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_U8 cou8NUM_BT_HANDLERS = 6;


/* Forward Declarations. */
class spi_tclBluetoothRespIntf;
class spi_tclBluetoothDevBase;

/*!
* \class spi_tclBluetooth
* \brief This is the main Bluetooth Manager class that provides interface to delegate 
*        the execution of command and handle response.
*/
class spi_tclBluetooth : public spi_tclLifeCycleIntf, public spi_tclBluetoothIntf, public spi_tclSelectionIntf
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetooth::spi_tclBluetooth(...)
   ***************************************************************************/
   /*!
   * \fn      spi_tclBluetooth()
   * \brief   Default Constructor
   * \param   [IN] poResponseIntf: Pointer to BT Response interface
   ***************************************************************************/
   spi_tclBluetooth(spi_tclBluetoothRespIntf* poResponseIntf);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetooth::~spi_tclBluetooth();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclBluetooth()
   * \brief   Virtual Destructor
   ***************************************************************************/
   virtual ~spi_tclBluetooth();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclBluetooth::bInitialize()
    ***************************************************************************/
   /*!
    * \fn      bInitialize()
    * \brief   Method to Initialize
    * \sa      bUnInitialize()
    **************************************************************************/
   virtual t_Bool bInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclBluetooth::bUnInitialize()
    ***************************************************************************/
   /*!
    * \fn      bUnInitialize()
    * \brief   Method to UnInitialize
    * \sa      bInitialize()
    **************************************************************************/
   virtual t_Bool bUnInitialize();

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBluetooth::vLoadSettings()
    ***************************************************************************/
   /*!
    * \fn      vLoadSettings()
    * \brief   vLoadSettings Method. Invoked during OFF->NORMAL state transition.
    * \sa      vSaveSettings()
    **************************************************************************/
   virtual t_Void vLoadSettings();

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclBluetooth::vSaveSettings()
    ***************************************************************************/
   /*!
    * \fn      vSaveSettings()
    * \brief   vSaveSettings Method. Invoked during  NORMAL->OFF state transition.
    * \sa      vLoadSettings()
    **************************************************************************/
   virtual t_Void vSaveSettings();
   /***************************************************************************
    ** FUNCTION:  spi_tclVideo::vSelectDevice
    ***************************************************************************/
   /*!
    * \fn     vSelectDevice
    * \brief  Device selection request
    * \param  corfrSelectReq : Device selection request details
    **************************************************************************/
   t_Void vSelectDevice(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
    ** FUNCTION:  spi_tclBluetooth::vSelectDeviceResult
    ***************************************************************************/
   /*!
    * \fn     vSelectDeviceResult
    * \brief  device selection result
    * \param  corfrSelectReq : Device selection result details
    * \param  enErrorCode : Error code
    **************************************************************************/
   t_Void vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vGetVehicleBTAddress()
   ***************************************************************************/
   /*!
   * \fn     vGetVehicleBTAddress()
   * \brief  Function to get the vehicle BT address.
   * \param  szVehicleBTAddress:[OUT] Vehicle BT address
   * \retval  None
   **************************************************************************/
   virtual t_Void vGetVehicleBTAddress(t_String &szVehicleBTAddress);

   /*********Start of functions overridden from spi_tclBluetoothIntf*********/

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSendSelectDeviceResult(t_Bool...)
   ***************************************************************************/
   /*!
   * \fn      vSendSelectDeviceResult(t_Bool bSuccess)
   * \brief   Interface to send result of Bluetooth connection processing during
   *          SelectDevice operation.
   *          Mandatory interface to be implemented.
   * \param   [IN] bSelectionSuccess:
   *               TRUE - If BT processing is successful.
   *               FALSE - If BT processing is unsuccessful.
   * \retval  None
   **************************************************************************/
   virtual t_Void vSendSelectDeviceResult(t_Bool bSelectionSuccess);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSendDeselectDeviceRequest()
   ***************************************************************************/
   /*!
   * \fn      vSendDeselectDeviceRequest(const t_Bool &corfbIsUserTrigger)
   * \brief   Interface to send a Device Deselection request.
   *          Mandatory interface to be implemented.
   * \param   corfbIsUserTrigger : [IN] true if user triggered the deselection
   * \retval  None
   **************************************************************************/
   virtual t_Void vSendDeselectDeviceRequest(const t_Bool &corfbIsUserTrigger = true);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSendDeviceSwitchEvent(t_U32 ...)
   ***************************************************************************/
   /*!
   * \fn      vSendDeviceSwitchEvent(t_U32 u32BluetoothDevHandle,
   *             t_U32 u32ProjectionDevHandle, tenBTChangeInfo enBTChange,
   *             t_Bool bCallActive)
   * \brief   Interface to notify clients about a device switch event being performed
   *          between a BT and a Projection device, which has been triggered as a result
   *          of user action or due to device selection strategy applied by
   *          Bluetooth/SPI service.
   *          Mandatory interface to be implemented.
   * \param   [IN] u32BluetoothDevHandle  : Uniquely identifies a Bluetooth Device.
   * \param   [IN] u32ProjectionDevHandle : Uniquely identifies a Projection Device.
   * \param   [IN] enBTChange  : Indicates type of BT device change
    * \param  [IN] bSameDevice : Inidcates whether BT & Projection device are same
    *              or different devices.
   * \param   [IN] bCallActive : Provides call activity status.
   * \retval  None
   **************************************************************************/
   virtual t_Void vSendDeviceSwitchEvent(t_U32 u32BluetoothDevHandle,
         t_U32 u32ProjectionDevHandle,
         tenBTChangeInfo enBTChange,
         t_U8 u8Profiles,
         t_Bool bSameDevice,
         t_Bool bCallActive);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSendBTDeviceInfo(...)
   ***************************************************************************/
   /*!
   * \fn      vSendBTDeviceInfo
   * \brief   Notifies users when BT Device name and/or BT address is available
   *           for the active projection device
   * \param   [IN] u32ProjectionDevHandle: Unique handle of projection device
   * \param   [IN] szBTDeviceName: BT Device name of projection device
   * \param   [IN] szBTAddress: BT Address of projection device
   * \retval  None
   **************************************************************************/
   t_Void vSendBTDeviceInfo(t_U32 u32ProjectionDevHandle,
            t_String szBTDeviceName, t_String szBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclBluetooth::u32GetSelectedDevHandle()
   ***************************************************************************/
   /*!
   * \fn      u32GetSelectedDevHandle()
   * \brief   Interface to retrieve device handle of selected projection device.
   *          Mandatory interface to be implemented.
   * \param   None
   * \retval  DeviceHandle of selected projection device
   **************************************************************************/
   virtual t_U32 u32GetSelectedDevHandle();

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetooth::szGetSelectedDevBTAddress()
   ***************************************************************************/
   /*!
   * \fn      szGetSelectedDevBTAddress()
   * \brief   Interface to retrieve BT Address of selected projection device.
   *          Mandatory interface to be implemented.
   * \param   None
   * \retval  BT Address of selected projection device
   **************************************************************************/
   virtual t_String szGetSelectedDevBTAddress();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBluetooth::bSetBTLimitationMode()
   ***************************************************************************/
   /*!
   * \fn      bSetBTLimitationMode()
   * \brief   Interface to set a Bluetooth limitation in HU.
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool: True - if request to set limitation mode is sent successfully, false
   **************************************************************************/
    t_Bool bSetBTLimitationMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName, tenBTTechnology enTechnology,
              tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction);

   /***************************************************************************
   ** FUNCTION:  tenDeviceStatus spi_tclBluetooth::enGetSelectedDevStatus()
   ***************************************************************************/
   /*!
   * \fn      enGetSelectedDevStatus()
   * \brief   Interface to retrieve current device status of selected projection device.
   *          Mandatory interface to be implemented.
   * \param   None
   * \retval  Device status of selected projection device
   **************************************************************************/
   virtual tenDeviceStatus enGetSelectedDevStatus();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSetSelectedDevStatus()
   ***************************************************************************/
   /*!
   * \fn      vSetSelectedDevStatus()
   * \brief   Interface to set the status of selected projection device.
   *          Mandatory interface to be implemented.
   * \param   [IN] enDevStatus : Device status enumeration
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetSelectedDevStatus(tenDeviceStatus enDevStatus);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSetSelectedDevBTAddress()
   ***************************************************************************/
   /*!
   * \fn      vSetSelectedDevBTAddress()
   * \brief   Interface to set the BT Address of selected projection device.
   *          Mandatory interface to be implemented.
   * \param   [IN] szBTAddress : BT Address string
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetSelectedDevBTAddress(t_String szBTAddress);

   /***************************************************************************
   ** FUNCTION:  tenDeviceCategory spi_tclBluetooth::enGetSelectedDeviceCategory()
   ***************************************************************************/
   /*!
   * \fn      enGetSelectedDeviceCategory()
   * \brief   Interface to retrieve the device category of current selected
   *          projection device.
   *          Mandatory interface to be implemented.
   * \param   None
   * \retval  tenDeviceCategory: Device category of selected device
   **************************************************************************/
   tenDeviceCategory enGetSelectedDeviceCategory();
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vTriggerWiFiCredentialExchange(
   ***************************************************************************/
   /*!
   * \fn      t_Void vTriggerWiFiCredentialExchange(t_U32 u32DeviceID)
   * \brief   Method to noitify component that BTLimitationMode is pre-activated
   *          and it can share the WiFiCredentials with the Phone.
   * \param   [IN] u32DeviceID: Uniuq Device ID
   * \retval  t_Void
   **************************************************************************/  
   t_Void vTriggerWiFiCredentialExchange(t_U32 u32DeviceID);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclBluetooth::vSetGeneralRestrictions
    ***************************************************************************/
   /*!
    * \fn     vSetGeneralRestriction
    * \brief  Interface to set the general restrictions for various technologies.
    * \param  enDeviceCategory  : Device category .
    * \param  u16GeneralRestrInfo : General Restriction Info for the technology.
    * \param  [OUT] void.
    **************************************************************************/
    t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,t_U16 u16GeneralRestrInfo);


   /***************************************************************************
   ** FUNCTION:  spi_tclExtCmdBluetoothIntf spi_tclBluetooth::poFetchExtCmdBluetoothIntf()
   ***************************************************************************/
   /*!
   * \fn      poFetchExtCmdBluetoothIntf()
   * \brief   Gets the ExtCmdBluetoothIntf pointer from ExtCompManager
   * \retval  None
   **************************************************************************/
   spi_tclExtCmdBluetoothIntf* poFetchExtCmdBluetoothIntf();
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vOnDisableBluetoothTrigger(
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnDisableBluetoothTrigger(t_String szBTAddress)
   * \brief   Method to notify component that DisableBT request is received
   * \param   [IN] szBTAddress: Unique Devcie ID
   * \retval  t_Void
   **************************************************************************/
   t_Void vOnDisableBluetoothTrigger(t_String szBTAddress) override;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBluetooth::vSetBluetoothSettingsInstance(spi_tclBluetoothSettingsIntf* poBluetoothSettingsIntf)
    ***************************************************************************/
    /*!
    * \fn      t_Void vSetBluetoothSettingsInstance()
    * \brief   Method to set Bluetooth settings instance.
    * \param   poBluetoothSettingsIntf  : pointer to BluetoothSettingsIntf.
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetBluetoothSettingsInstance(spi_tclBluetoothSettingsIntf* poBluetoothSettingsIntf);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBluetooth::vGetBluetoothConfig()
    ***************************************************************************/
   /*!
    * \fn      t_Void vGetBluetoothConfig()
    * \brief   To print the Bluetooth configuration to read KDS
    * \sa      None
    **************************************************************************/
   virtual t_Void vGetBluetoothConfig();
   
         /***************************************************************************
       ** FUNCTION:  spi_tclBluetooth::vUpdateSelectionState
       ***************************************************************************/
      /*!
       * \fn     vUpdateSelectionState
       * \brief  Informs if the device selection is not initiated for a reported device
       * \param  cou32DeviceHandle: Device handle
       **************************************************************************/
      t_Void vUpdateSelectionState(const t_U32 cou32DeviceHandle,
               tenDeviceSelectionState enDevSelectionState,
               tenDeviceConnectionType enDeviceConnType);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSetBTLimitationModeResult()
   ***************************************************************************/
   /*!
   * \fn      vSetBTLimitationModeResult()
   * \brief   To set the BT LimitationMode result
   * \param   [IN] corBTLimitationModeInfo : BTLimitationMode Info
   * \retval  None
   **************************************************************************/
   t_Void vSetBTLimitationModeResult(const trBTLimitationModeInfo corBTLimitationModeInfo);

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclBluetooth::szGetBTDeviceName()
    ***************************************************************************/
   /*!
    * \fn    t_String szGetBTDeviceName()
    * \brief Method to get the BT device name
    * \retval t_String
    **************************************************************************/
   t_String szGetBTDeviceName();


   /*********End of functions overridden from spi_tclBluetoothIntf***********/

   /**************************************************************************
   ****************************END OF PUBLIC**********************************
   **************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/  

   /***************************************************************************
   ******************************END OF PROTECTED******************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclBluetooth(const spi_tclBluetooth &rfcoBluetooth)
    ***************************************************************************/
   /*!
    * \fn      spi_tclBluetooth(const spi_tclBluetooth &rfcoBluetooth)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   spi_tclBluetooth(const spi_tclBluetooth &rfcoBluetooth);

   /***************************************************************************
    ** FUNCTION: const spi_tclBluetooth & operator=(
    **                                 const spi_tclBluetooth &rfcoBluetooth);
    ***************************************************************************/
   /*!
    * \fn      const spi_tclBluetooth & operator=(const spi_tclBluetooth &rfcoBluetooth);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   const spi_tclBluetooth & operator=(
            const spi_tclBluetooth &rfcoBluetooth);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vOnVehicleBTAddressCb()
   ***************************************************************************/
   /*!
   * \fn     vOnVehicleBTAddressCb()
   * \brief  Function to set the vehicle BT address.
   * \param  szVehicleBTAddress:[OUT] Vehicle BT address
   * \retval  None
   **************************************************************************/
   t_Void vOnVehicleBTAddressCb(t_String szVehicleBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vSaveSettings()
   ***************************************************************************/
   /*!
   * \fn      enGetSelDevBTHandlerIndex()
   * \brief   Provides the index of BT handler for selected projection device.
   * \param   None
   * \retval  Index of BT handler
   **************************************************************************/
   tenBTHandlerIndex enGetSelDevBTHandlerIndex();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vFetchSelectedDeviceDetails(t_U32...)
   ***************************************************************************/
   /*!
   * \fn      vFetchSelectedDeviceDetails(t_U32 u32SelectedDevHandle)
   * \brief   Method to fetch details about Selected Device.
   * \param   [IN] u32SelectedDevHandle: Unique device handle of selected device
   * \retval  None
   **************************************************************************/
   t_Void vFetchSelectedDeviceDetails(t_U32 u32SelectedDevHandle);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetooth::vClearSelectedDevInfo()
   ***************************************************************************/
   /*!
   * \fn      vClearSelectedDevInfo()
   * \brief   Method to clear information stored about Selected Device.
   * \param   None
   * \retval  None
   **************************************************************************/
   t_Void vClearSelectedDevInfo();

   /***************************************************************************
    ** FUNCTION:  spi_tclBluetooth::RegisterVehicleBTMacAddressCallback()
    ***************************************************************************/
   /*!
    * \fn     RegisterVehicleBTMacAddressCallback
    * \brief  Registers for callbacks to get BT Mac address
    **************************************************************************/
   t_Void vRegisterVehicleBTMacAddressCallback();
   
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclAAPBluetooth::vPostBTServiceAvailabilityMsg()
   ***************************************************************************/
   /*!
   * \fn      t_Void vPostBTConnectionChangedMsg()
   * \brief   Called when BT On Service Availability is  sent by BT Client Handler.
   * \sa      None
   **************************************************************************/
   virtual t_Void vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable);


   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** BT Response Interface pointer
   ***************************************************************************/
   spi_tclBluetoothRespIntf*           m_poBTRespIntf;

   /***************************************************************************
   ** List of pointers to Bluetooth Handlers of different device types
   ***************************************************************************/
   spi_tclBluetoothDevBase*            m_poBTHandlersLst[cou8NUM_BT_HANDLERS];

   /***************************************************************************
   ** Structure containing info about Selected Device in SPI
   ***************************************************************************/
   trSpiDeviceDetails                  m_rSelDeviceDetails;

   /***************************************************************************
   ** Lock object for Selected device details structure
   ***************************************************************************/
   Lock      m_SelDevDetailsLock;
   
   /***************************************************************************
    ** Stores pending BT Limitation Mode request info
    ***************************************************************************/
   trBTLimitationActionInfo m_rPendingBTLimModeInfo;

   spi_tclBluetoothSettingsIntf* m_poBluetoothSettings;

   t_String m_szFriendlyName;

};

#endif // _SPI_TCLBLUETOOTH_H_

