/*!
 *******************************************************************************
 * \file             spi_BluetoothTypedefs.h
 * \brief            Type defines used by Bluetooth classes
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Type defines used by Bluetooth classes
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 25.02.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef _SPI_BLUETOOTHTYPEDEFS_H_
#define _SPI_BLUETOOTHTYPEDEFS_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <string>
#include <functional> 

#include "SPITypes.h"

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/
//! Macro to check if a BT string address is valid.
//! Address is valid if the string is not empty.
#define IS_VALID_BT_ADDRESS(BT_ADDRESS)   (false == (BT_ADDRESS).empty())

//! Macro to check if a Projection device handle is valid
#define IS_VALID_DEVHANDLE(DEVHANDLE)  (0 != DEVHANDLE)

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_U32 cou32InvalidDeviceHandle = 0;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/


//! Identifies the connection result from Bluetooth Source.
enum tenBTConnectionResult
{
   //! Default value
   e8BT_RESULT_UNKNOWN = 0,
   //! Device connection success
   e8BT_RESULT_CONNECTED = 1,
   //! Device connection failure
   e8BT_RESULT_CONN_FAILED = 2,
   //! Device disconnection success
   e8BT_RESULT_DISCONNECTED = 3,
   //! Device disconnection failure
   e8BT_RESULT_DISCONN_FAILED = 4,
   //! Device block success
   e8BT_RESULT_BLOCKED = 5,
   //! Device block failure
   e8BT_RESULT_BLOCK_FAILED = 6,
   //! Device block success
   e8BT_RESULT_UNBLOCKED = 7,
   //! Device block failure
   e8BT_RESULT_UNBLOCK_FAILED = 8
};

//! Indicates status of device change
enum tenDeviceStatus
{
   //! Indicates all device changes are complete (default value)
   e8DEVICE_CHANGE_COMPLETE = 0,
   //! Indicates a DeviceSwitch event b/w Projection & BT device is in progress,
   //! and SPI is waiting for user action
   e8DEVICE_SWITCH_IN_PROGRESS = 1,
   //! Indicates a device changes is being processed
   //! (as a result of user or Projection device triggered action)
   e8DEVICE_CHANGE_IN_PROGRESS = 2,
   //! Indicates a Projection device selection is in progress
   e8DEVICE_SELECTION_IN_PROGRESS = 3,
   //! Indicates a Projection device deselection is in progress
   e8DEVICE_DESELECTION_IN_PROGRESS = 4,
   //! Indicates a device change is delayed due to BT Service unavailability
   e8DEVICE_CHANGE_DELAYED = 5
};

//! Indicates BT pairing status of projection device
enum tenDevicePairingStatus
{
   e8DEVICE_PAIRING_NOT_STARTED = 0,
   e8DEVICE_PAIRING_REQUESTED = 1,
   e8DEVICE_PAIRING_RESPONSE_SENT = 2,
   e8DEVICE_PAIRING_CANCELLED = 3,
   e8DEVICE_PAIRING_COMPLETE = 4
};

//! Identifies BT Pairing status types
enum tenBTSetPairingStatus
{
   e8PAIRING_SUCCESSFUL = 0x00,
   e8PAIRING_UNSUCCESSFUL = 0x01,
   e8PAIRING_TIMEOUT = 0x02,
   e8PAIRING_CONNECTIONLOST = 0x03,
   e8PAIRING_AUTH_FAILURE = 0x04,
   e8PAIRING_CANCELLED = 0x05,
   e8PAIRING_RETRY_KEEPSESSION = 0x06,
   e8PAIRING_RETRY_NEWSESSION = 0x07,
   e8PAIRING_READY_TO_PAIR = 0x08, //TODO - not yet in FI!
   e8PAIRING_IDLE = 0xFF,
   //@Note: Please do not change above values!
   //Values are assigned as per BluetoothSettings FI values.
};

//! Identifies BT Pairing Response Type
enum tenPairingResponseType
{
  e8BT_PAIRING_RESPONSE_YES = 0x00,
  e8BT_PAIRING_RESPONSE_NO = 0x01,
  e8BT_PAIRING_RESPONSE_CANCEL = 0x02,
  e8BT_PAIRING_RESPONSE_RETRY = 0x03
};

enum tenBTSetPairingMethod
{
   e8PAIRING_TYPE_LEGACY = 0x00,
   e8PAIRING_TYPE_SSP_NUMERIC_COMPARISON = 0x01,
   e8PAIRING_TYPE_SSP_JUST_WORKS = 0x02,
   e8PAIRING_TYPE_CHANGE_PIN = 0x03
   //@Note: Please do not change above values!
   //Values are assigned as per BluetoothSettings FI values.
};

//! Contains info about pairable & connectable states of HU
struct trBTPairingRequestInfo
{
   t_String szRemoteDevBTAddr;
   //t_String szRemoteDevName;
   t_String szPairingPin;
   tenBTSetPairingMethod enPairingMethod;

   trBTPairingRequestInfo(): enPairingMethod(e8PAIRING_TYPE_LEGACY)
   {}
};

enum tenBTTechnology
{
   e8BT_TECH_IAP = 1,
   e8BT_TECH_CARPLAY = 2,
   e8BT_TECH_ANDROIDAUTO = 3,
   e8BT_TECH_MYSPIN = 4,
   e8BT_TECH_MYSPIN_IOS = 5,
   e8BT_TECH_MIRRORLINK = 6,
   e8BT_TECH_MIRRORLINK_IOS = 7,
   e8BT_TECH_CARLIFE = 8,
   e8BT_TECH_CARLIFE_IOS = 9,
   e8BT_TECH_ONCAR = 10,
   e8BT_TECH_ONCAR_IOS = 11
};
enum tenBTCommunicationChannel
{
   e8BT_COMM_CHN_USB = 1,
   e8BT_COMM_CHN_WIFI_2_4GHZ = 2,
   e8BT_COMM_CHN_WIFI_5GHZ = 3,
   e8BT_COMM_CHN_BLUETOOTH = 4,
   e8BT_COMM_CHN_WIFI = 5
};

enum tenBTLimitationAction
{
   e8BT_LIM_ACTION_UNKNOWN = 0,
   e8BT_LIM_ACTION_PREPARE = 1,
   e8BT_LIM_ACTION_ACTIVATE = 2,
   e8BT_LIM_ACTION_DEACTIVATE = 3,
   e8BT_LIM_ACTION_DEACTIVATE_WAIT = 4,
   e8BT_LIM_ACTION_DEACTIVATE_INTERNAL =5,
   e8BT_LIM_ACTION_SPM_STATE_OFF =6,
   e8BT_LIM_ACTION_PREACTIVATE =7
};

enum tenBTLimitationActionState
{
   e8BT_LIM_ACTION_STATE_IDLE = 0,
   e8BT_LIM_ACTION_STATE_PREPARING = 1,
   e8BT_LIM_ACTION_STATE_PREPARED = 2,
   e8BT_LIM_ACTION_STATE_ACTIVATING = 3,
   e8BT_LIM_ACTION_STATE_ACTIVE = 4,
   e8BT_LIM_ACTION_STATE_DEACTIVATING = 5,
   e8BT_LIM_ACTION_STATE_WAITING = 6,
   e8BT_LIM_ACTION_STATE_ERROR = 7,
   e8BT_LIM_ACTION_STATE_ERROR_USER_DENIED = 8,
   e8BT_LIM_ACTION_STATE_PREACTIVATING =9,
   e8BT_LIM_ACTION_STATE_PREACTIVATED =10
};

enum tenCallStatus
{
   e8IDLE = 0x00,
   e8DIALING = 0x01,
   e8ACTIVE = 0x02,
   e8ON_HOLD = 0x03,
   e8DISCONNECTING = 0x04,
   e8BUSY = 0x05,
   e8CONFERENCE = 0x06,
   e8IN_VOICEMAIL = 0x07,
   e8RINGTONE = 0x08
   //@Note: Please do not change above values!
   //Values are assigned as per Telephone FI values.
};

//! Call info structure
struct trTelCallStatusInfo
{
   t_U16 u16CallInstance;
   tenCallStatus  enCallStatus;
   t_U32 u32BTDeviceHandle;

   trTelCallStatusInfo():
      u16CallInstance(0), enCallStatus(e8IDLE),u32BTDeviceHandle(0){}

   trTelCallStatusInfo& operator=(const trTelCallStatusInfo& corfSrc)
   {
      if(this != &corfSrc)
      {
         u16CallInstance = corfSrc.u16CallInstance;
         enCallStatus = corfSrc.enCallStatus;
         u32BTDeviceHandle = corfSrc.u32BTDeviceHandle;
      }
      return *this;
   }
};

/*!
 * \brief Various external VR states for Myspin
 */
enum tenExtVRState
{
   e8_EXT_VR_STATE_IDLE = 0,
   e8_EXT_VR_STATE_READY = 1,
   e8_EXT_VR_STATE_ACTIVE = 2
};

enum tenVRErrorCode
{
    e8NO_ERROR = 0,
    e8ERROR_IN_STREAMING = 1, //SCO open failed, No audio allocation, native VR activated when SPI VR active etc
    e8ERROR_PHONE_CALL_ACTIVE = 2
};

//! Telephone Client Callback signatures definition
typedef std::function<t_Void(t_Bool)> vOnTelephoneCallActivity;
typedef std::function<t_Void(const std::vector<trTelCallStatusInfo>&) > vOnTelephoneCallStatus;
typedef std::function<t_Void(t_String,tenExtVRState,tenVRErrorCode)> vOnExtVRStateUpdate;

//! \brief   Structure holding the callbacks for Telephone events.
//!          Used by Telephone Client to notify subscriber about Telephone events.
struct trTelephoneCallbacks
{
   //! Called when there is a change in Telephone call activity
   vOnTelephoneCallActivity fvOnTelephoneCallActivity;

   //! Called with the details of each CallInstance
   vOnTelephoneCallStatus fvOnTelephoneCallStatus;
   
   //! Called when there is external VR state update
   vOnExtVRStateUpdate fvOnExtVRStateUpdate;

   trTelephoneCallbacks() :
      fvOnTelephoneCallActivity(NULL), fvOnTelephoneCallStatus(NULL),
      fvOnExtVRStateUpdate(NULL)
   {
   }
};

struct trBTLimitationModeInfo
{
   t_String szBTAddress;
   t_String szBTDeviceName;
   tenBTTechnology enTechnology;
   tenBTCommunicationChannel enCommChannel;
   tenBTLimitationActionState enActionState;
   
   trBTLimitationModeInfo():enTechnology(e8BT_TECH_IAP),enCommChannel(e8BT_COMM_CHN_BLUETOOTH),enActionState(e8BT_LIM_ACTION_STATE_IDLE)
   {}

};

struct trBTLimitationActionInfo
{
   t_String szBTAddress;
   t_String szBTDeviceName;
   tenBTTechnology enTechnology;
   tenBTCommunicationChannel enCommChannel;
   tenBTLimitationAction enAction;
   
   //e8BT_LIM_ACTION_DEACTIVATE is used as default action, since no default action is defined in FI
   trBTLimitationActionInfo():enTechnology(e8BT_TECH_IAP),enCommChannel(e8BT_COMM_CHN_BLUETOOTH),enAction(e8BT_LIM_ACTION_DEACTIVATE)
   {}
};

//! \brief   Structure holding a BT device information
struct trBluetoothDeviceInfo
{
   //Provides Bluetooth Address of device
   std::string szBTAddress;
   //Provides Connection status of device
   t_Bool bDeviceConnected;
   //Provides Device name
   std::string szDeviceName;
   //Provides HFP connection status
   t_Bool bIsHFPConn;
   //Provides A2DP connection status
   t_Bool bIsA2DPConn;   

   trBluetoothDeviceInfo() :
      szBTAddress(), bDeviceConnected(false), szDeviceName(),
      bIsHFPConn(false), bIsA2DPConn(false)
   {
   }
};

//! Callback signatures definition
typedef std::function<t_Void(std::string, tenBTConnectionResult)> vOnBTConnectionStatus;
typedef std::function<t_Void(tenBTSetPairingStatus, t_String)> vOnBTPairingStatus;
typedef std::function<t_Void(trBTPairingRequestInfo)> vOnBTPairingInfo;
typedef std::function<t_Void(t_Bool)> vOnBTServiceAvailability;
typedef std::function<t_Void(trBTLimitationModeInfo)> vOnBTLimitationMode;
typedef std::function<t_Void(t_Bool)> vOnSetBTLimitationModeResult;
typedef std::function<t_Void()> vOnBTProfileStatus;

typedef std::function<t_Void(t_String, t_String)> vOnBTDeviceNameUpdate;

typedef std::function<t_Void()> vOnOOBTComplete;

//! \brief   Structure holding the callbacks to BT Manager. Used by BT Client to notify
//!          BT Manager about Bluetooth Source events.
struct trBluetoothCallbacks
{
   //! Called when BT device connected/disconnection notification is received by BT Client
   vOnBTConnectionStatus fvOnBTConnectionChanged;
   //! Called when BT service availability status changes
   vOnBTServiceAvailability  fvOnBTServiceAvailability;

   vOnBTLimitationMode  fvOnBTLimitationMode;
   vOnSetBTLimitationModeResult fvOnBTLimitationModeResult;

   vOnBTProfileStatus fvOnBTProfileStatus;

   vOnBTPairingStatus fvOnBTPairingStatus;

   trBluetoothCallbacks() :
      fvOnBTConnectionChanged(NULL),
      fvOnBTServiceAvailability(NULL),
      fvOnBTLimitationMode(NULL), 
      fvOnBTLimitationModeResult(NULL),
      fvOnBTProfileStatus(NULL),
      fvOnBTPairingStatus(NULL)
   {
   }
};

//OOBT Pairing Callbacks
struct trOOBTPairingCallbacks
{
   //called when OOBT the conflict is resolved - to invoke the PrepareSetup
   vOnOOBTComplete     fvOnOOBTComplete;
   trOOBTPairingCallbacks():fvOnOOBTComplete(NULL)
   {

   }

};

//! \brief   Structure holding the callbacks to BT Manager. Used by BT Client to notify
//!          BT Manager about Bluetooth Source events.
struct trBluetoothPairingCallbacks
{
   //! Called when there is a pairing request
   vOnBTPairingInfo     fvOnBTPairingInfo;

   trBluetoothPairingCallbacks() :
      fvOnBTPairingInfo(NULL)
   {
   }
};

//! \brief   Structure holding info for BT Device name callback
struct trBTDeviceNameCbInfo
{
   //! BT Address of interested device 
   t_String szBTAddress;

   //! Callback function to get interested device's name
   vOnBTDeviceNameUpdate   fvOnBTDeviceNameUpdate;

   trBTDeviceNameCbInfo(): fvOnBTDeviceNameUpdate(NULL)
   {
   }
};

//! Contains info about a Projection device
struct trSpiDeviceDetails
{
   t_U32 u32DeviceHandle;
   t_String szBTAddress;
   tenDeviceCategory enDeviceCategory;
   tenDeviceStatus enDeviceStatus;

   trSpiDeviceDetails() :
      u32DeviceHandle(0), enDeviceCategory(e8DEV_TYPE_UNKNOWN),
      enDeviceStatus(e8DEVICE_CHANGE_COMPLETE)
   {}
};

enum tenconflictsDetected
{
   e8NONE = 0,
   e8BT_PAIRING = 1,
   e8OOB_BT_PAIRING = 2,
   e8AAP_USB = 3,
   e8AAP_WIRELESS = 4,
   e8CP_USB = 5,
   e8CP_WIRELESS = 6,
   e8CARLIFE_USB = 7,
   e8CARLIFE_IOS_USB = 8,
   e8ONCAR_USB = 9,
   e8ONCAR_IOS_USB =10,
   e8MIRRORLINK_USB = 11,
   e8MIRRORLINK_IOS_USB =12,
   e8MYSPIN_USB = 13,
   e8MYSPIN_IOS_USB = 14,

   //! Add all the values above this , if extended in future
   e8CONFLICTS_DETECTED_INVALID
};

enum tenConflictState
{
   e8CONFLICT_STATE_NEW = 0,
   e8CONFLICT_STATE_UNCHANGED = 1,
   e8CONFLICT_STATE_SOLVED = 2,

   //! Add all the values above this , if extended in future
   e8CONFLICTS_STATE_INVALID
};

enum tenResolutionType
{
   e8RESOLUTION_TYPE_PROCEED = 0,
   e8RESOLUTION_TYPE_CANCEL = 1,
   //! Add all the values above this , if extended in future
   e8CONFLICT_RESOLUTION_INVALID
};


/*!
 * \brief Index to various BT handlers
 */
enum tenBTHandlerIndex
{
   e8_BT_ML_INDEX = 0,
   e8_BT_DIPO_INDEX = 1,
   e8_BT_AAP_INDEX = 2,
   e8_BT_MYSPIN_INDEX = 3,
   e8_BT_CARLIFE_INDEX = 4,
   e8_BT_ONCAR_INDEX = 5,
   e8_BT_INDEX_UNKNOWN
};


typedef std::function<t_Void(std::string)> vOnVehicleBTAddress;

//! \brief   Structure holding the callbacks to BT Manager. Used by BT Client to write and read
//!          BT Mac address to and from the Datapool
struct trVehicleBTAddressCallback
{
   //! Called when BT address is received by BT Client
   vOnVehicleBTAddress fvOnVehicleBTAddress;

   trVehicleBTAddressCallback() :
      fvOnVehicleBTAddress(NULL)
   {
   }
};

//! BT Service Available (BT SPM) state
enum tenBTServiceAvailableStatus
{
    e8SERVICE_UNAVAILABLE = 0,
    e8SERVICE_AVAILABLE_RESTRICTED = 1,
    e8SERVICE_AVAILABLE = 2
};
#endif //_SPI_BLUETOOTHTYPEDEFS_H_
