/*!
 *******************************************************************************
 * \file          spi_tclAudioSrcInfo.cpp
 * \brief         Audio source class that provides interface to store and retrieve
 *                Source details
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Source Information class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                                  | Modifications
 27.03.2014 |  Deepti Samant (RBEI/ECP2)               | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclAudioSrcInfo.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclAudioSrcInfo.cpp.trc.h"
#endif

/***************************************************************************
 ** FUNCTION:  spi_tclAudioSrcInfo::spi_tclAudioSrcInfo();
 ***************************************************************************/
spi_tclAudioSrcInfo::spi_tclAudioSrcInfo()

{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   m_mapAudioSrcInfo.clear();
}//End of spi_tclAudioSrcInfo::spi_tclAudioSrcInfo()

/***************************************************************************
 ** FUNCTION:  spi_tclAudioSrcInfo::~spi_tclAudioSrcInfo();
 ***************************************************************************/
spi_tclAudioSrcInfo::~spi_tclAudioSrcInfo()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   m_mapAudioSrcInfo.clear();
}//End of spi_tclAudioSrcInfo::~spi_tclAudioSrcInfo()

/***************************************************************************
 ** FUNCTION:  tenAudSrcState spi_tclAudioSrcInfo::enGetSrcState(const t_U8 cu8SrcNum)
 ***************************************************************************/
tenAudSrcState spi_tclAudioSrcInfo::enGetSrcState(const t_U8 cu8SrcNum)
{
   tenAudSrcState enSrcState = e8SRC_INACTIVE;

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      enSrcState = (itMapAudio->second).enSrcState;
   }//End of if (AudSrcInfoMapItr != m_mapAudioSrcInfo.end())

   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::enGetSrcState: SrcNum = %d SrcState = %d ",
            cu8SrcNum, enSrcState));
   return enSrcState;

}//End of tenAudSrcState enGetSrcState(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAudioSrcInfo::bGetAudSrcInfo(const t_U8 cu8SrcNum,
 **            trAudSrcInfo& rfrSrcInfo)
 ***************************************************************************/
t_Bool spi_tclAudioSrcInfo::bGetAudSrcInfo(const t_U8 cu8SrcNum,
         trAudSrcInfo& rfrSrcInfo)
{
   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::crfrGetAudSrcInfo: SrcNum = %d ", cu8SrcNum));

   t_Bool bRetVal = false;

   AudSrcInfoMapItr itMapAudio = m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      rfrSrcInfo = (itMapAudio->second).rAudSrcInfo;
      bRetVal = true;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())

   return bRetVal;

}//End of const trAudSrcInfo& crfrGetAudSrcInfo(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:  tenDeviceCategory spi_tclAudioSrcInfo::enGetDeviceCategory
 **            (const t_U8 cu8SrcNum)
 ***************************************************************************/
tenDeviceCategory spi_tclAudioSrcInfo::enGetDeviceCategory(const t_U8 cu8SrcNum)
{
   tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN;

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      enDevCategory = (itMapAudio->second).enDeviceCategory;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())

   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::enGetDeviceCategory: SrcNum = %d, DevCategory = %d ",
            cu8SrcNum, ETG_ENUM(DEVICE_CATEGORY,enDevCategory)));
   return enDevCategory;

}//End of tenDeviceCategory enGetDeviceCategory(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:  tenAudioDir spi_tclAudioSrcInfo::enGetAudDirection
 **            (const t_U8 cu8SrcNum)
 ***************************************************************************/
tenAudioDir spi_tclAudioSrcInfo::enGetAudDirection(const t_U8 cu8SrcNum)
{
   tenAudioDir enAudDir = e8AUD_INVALID;

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      enAudDir = (itMapAudio->second).enAudDir;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())

   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::enGetAudDirection: SrcNum = %d, AudDir = %d ",
            cu8SrcNum, ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir)));
   return enAudDir;

}//End of tenAudioDir enGetDeviceCategory(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:  tenAudioSamplingRate spi_tclAudioSrcInfo::enGetAudSamplingRate
 **            (const t_U8 cu8SrcNum)
 ***************************************************************************/
tenAudioSamplingRate spi_tclAudioSrcInfo::enGetAudSamplingRate(const t_U8 cu8SrcNum)
{
   tenAudioSamplingRate enAudSampleRate = e8AUD_SAMPLERATE_DEFAULT;

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      enAudSampleRate = (itMapAudio->second).enAudSampleRate;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())

   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::enGetAudSamplingRate: SrcNum = %d, AudSampleRate = %d ",
            cu8SrcNum, ETG_ENUM(AUDIO_SAMPLERATE, enAudSampleRate)));

   return enAudSampleRate;

}//End of tenAudioSamplingRate spi_tclAudioSrcInfo::enGetAudSamplingRate(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:  tenAudioSamplingRate spi_tclAudioSrcInfo::enGetAudNativeSamplingRate
 **            (const t_U8 cu8SrcNum)
 ***************************************************************************/
tenAudioSamplingRate spi_tclAudioSrcInfo::enGetAudNativeSamplingRate(const t_U8 cu8SrcNum)
{
   tenAudioSamplingRate enAudSampleRate = e8AUD_SAMPLERATE_DEFAULT;

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      enAudSampleRate = (itMapAudio->second).enAudNativeSampleRate;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())

   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::enGetAudNativeSamplingRate: SrcNum = %d, AudSampleRate = %d ",
            cu8SrcNum, ETG_ENUM(AUDIO_SAMPLERATE, enAudSampleRate)));

   return enAudSampleRate;

}

/***************************************************************************
 ** FUNCTION:  t_U32 spi_tclAudioSrcInfo::u32GetDeviceID(const t_U8 cu8SrcNum)
 ***************************************************************************/
t_U32 spi_tclAudioSrcInfo::u32GetDeviceID(const t_U8 cu8SrcNum)
{
   t_U32 u32DevID = 0;

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      u32DevID = (itMapAudio->second).u32DeviceId;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())

   ETG_TRACE_USR4(("spi_tclAudioSrcInfo::u32GetDeviceID: SrcNum = %d, DevID = 0x%x ", cu8SrcNum, u32DevID));
   return u32DevID;
}//End of t_U32 u32GetDeviceID(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:    t_Void spi_tclAudioSrcInfo::vSetDeviceInfo(const t_U8 cu8SrcNum,
 **      t_U32 u32DeviceID, tenDeviceCategory enDeviceCat, tenAudioDir enAudDir,
 **      tenAudioSamplingRate enAudSampleRate);
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vSetDeviceInfo(const t_U8 cu8SrcNum,
         t_U32 u32DeviceID, tenDeviceCategory enDeviceCat, tenAudioDir enAudDir,
         trAudSampleRate rAudSampleRate)
{
   ETG_TRACE_USR2(("spi_tclAudioSrcInfo::vSetDeviceInfo: SrcNum = %d, DeviceCat = %d, AudDir = %d ",
            cu8SrcNum, ETG_ENUM(DEVICE_CATEGORY,enDeviceCat),
            ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir)));

   ETG_TRACE_USR2(("spi_tclAudioSrcInfo::vSetDeviceInfo: SampleRate = %d, TelSampleRate = %d",
            ETG_CENUM(tenAudioSamplingRate,rAudSampleRate.enSampleRate),
            ETG_CENUM(tenAudioSamplingRate,rAudSampleRate.enNativeSampleRate)));

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      ETG_TRACE_USR4(("spi_tclAudioSrcInfo::vSetDeviceInfo: Src Info exists "));
      (itMapAudio->second).u32DeviceId = u32DeviceID;
      (itMapAudio->second).enDeviceCategory = enDeviceCat;
      (itMapAudio->second).enAudSampleRate = rAudSampleRate.enSampleRate;
      (itMapAudio->second).enAudNativeSampleRate = rAudSampleRate.enNativeSampleRate;
      (itMapAudio->second).enAudDir = enAudDir;
   }
   else
   {
      ETG_TRACE_USR4(("spi_tclAudioSrcInfo::vSetDeviceInfo: Storing new entry "));
      trAudioSrcProperties rAudSrcProperties;
      rAudSrcProperties.u32DeviceId = u32DeviceID;
      rAudSrcProperties.enDeviceCategory = enDeviceCat;
      rAudSrcProperties.enAudDir = enAudDir;
      rAudSrcProperties.enAudSampleRate = rAudSampleRate.enSampleRate;
      rAudSrcProperties.enAudNativeSampleRate = rAudSampleRate.enNativeSampleRate;
      m_mapAudioSrcInfo.insert(std::pair<t_U8, trAudioSrcProperties> (cu8SrcNum, rAudSrcProperties));
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())
}//End of t_Void vSetDeviceInfo(const t_U8 cu8SrcNum, ..)

/***************************************************************************
 ** FUNCTION:    t_Void spi_tclAudioSrcInfo::vSetAudSrcInfo(const t_U8 cu8SrcNum,
 **               trAudSrcInfo& rfrSrcInfo);
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vSetAudSrcInfo(const t_U8 cu8SrcNum,
         trAudSrcInfo& rfrSrcInfo)
{
   ETG_TRACE_USR2(("spi_tclAudioSrcInfo::vSetAudSrcInfo: SrcNum = %d ", cu8SrcNum));

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      (itMapAudio->second).rAudSrcInfo = rfrSrcInfo;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())
   else
   {
      ETG_TRACE_USR4(("[DESC]::vSetDeviceInfo: Storing new entry "));
      trAudioSrcProperties rAudSrcProperties;
      rAudSrcProperties.rAudSrcInfo = rfrSrcInfo;
      m_mapAudioSrcInfo.insert(std::pair<t_U8, trAudioSrcProperties> (cu8SrcNum, rAudSrcProperties));
   }
}//End of t_Void vSetAudSrcInfo(const t_U8 cu8SrcNum, ..)

/***************************************************************************
 ** FUNCTION:    t_Void spi_tclAudioSrcInfo::vSetActiveSrcFlag
 **               (const t_U8 cu8SrcNum, enSrcState);
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vSetActiveSrcFlag(const t_U8 cu8SrcNum,
         tenAudSrcState enSrcState)
{
   ETG_TRACE_USR1(("spi_tclAudioSrcInfo::vSetActiveSrcFlag: SrcNum = %d, SrcState = %d ",
         cu8SrcNum, enSrcState));

   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      (itMapAudio->second).enSrcState = enSrcState;
      ETG_TRACE_USR4(("[DESC]::vSetActiveSrcFlag: Updated SrcState = %d ", (itMapAudio->second).enSrcState));
   }
   else
   {
      ETG_TRACE_USR4(("[DESC]::vSetActiveSrcFlag: Storing new entry "));
      trAudioSrcProperties rAudSrcProperties;
      rAudSrcProperties.enSrcState = enSrcState;
      m_mapAudioSrcInfo.insert(std::pair<t_U8, trAudioSrcProperties> (cu8SrcNum, rAudSrcProperties));
   }
}//End of t_Void vSetActiveSrcFlag(const t_U8 cu8SrcNum, enSrcState)

/***************************************************************************
 ** FUNCTION:  t_Void vSetSamplingRate(const t_U8 cu8SrcNum..
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vSetSamplingRate(const t_U8 cu8SrcNum, tenAudioSamplingRate enAudioSamplingRate)
{
   AudSrcInfoMapItr itMapAudio = m_mapAudioSrcInfo.find(cu8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      (itMapAudio->second).enAudSampleRate = enAudioSamplingRate;
      ETG_TRACE_USR4(("spi_tclAudioSrcInfo::vSetSamplingRate: %d ",
               ETG_ENUM(AUDIO_SAMPLERATE, (itMapAudio->second).enAudSampleRate)));
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())
}

/***************************************************************************
 ** FUNCTION:  t_Void vEraseAudSrcInfo(const t_U8 cu8SrcNum);
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vEraseAudSrcInfo(const t_U8 cu8SrcNum)
{
   ETG_TRACE_USR1(("spi_tclAudioSrcInfo::vEraseAudSrcInfo: SrcNum = %d ", cu8SrcNum));
   AudSrcInfoMapItr itMapAudio= m_mapAudioSrcInfo.find(cu8SrcNum);
   if(m_mapAudioSrcInfo.end() != itMapAudio)
   {
      m_mapAudioSrcInfo.erase(itMapAudio);
   }// End of if(m_mapAudioSrcInfo.end() != itMapAudio)
}//End of t_Void vEraseAudSrcInfo(const t_U8 cu8SrcNum)

/***************************************************************************
 ** FUNCTION:  t_Void vClearAudSrcInfo();
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vClearAudSrcInfo()
{
   ETG_TRACE_USR1(("spi_tclAudioSrcInfo::vClearAudSrcInfo entered"));
   m_mapAudioSrcInfo.clear();
}

/***************************************************************************
 ** FUNCTION:  t_Void vClearDeviceInfo();
 ***************************************************************************/
t_Void spi_tclAudioSrcInfo::vClearDeviceInfo(t_U8 u8SrcNum)
{
   ETG_TRACE_USR1(("spi_tclAudioSrcInfo::vClearDeviceInfo entered: SrcNum = %d ", u8SrcNum));

   AudSrcInfoMapItr itMapAudio = m_mapAudioSrcInfo.find(u8SrcNum);
   if (itMapAudio != m_mapAudioSrcInfo.end())
   {
      (itMapAudio->second).u32DeviceId = 0;
      (itMapAudio->second).enDeviceCategory = e8DEV_TYPE_UNKNOWN;
      (itMapAudio->second).enAudSampleRate = e8AUD_SAMPLERATE_DEFAULT;
   }//End of if (itMapAudio != m_mapAudioSrcInfo.end())
   else
   {
      ETG_TRACE_ERR(("[ERR]::vClearDeviceInfo: No entry found for source num %d ", u8SrcNum));
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void bGetActiveSrc();
 ***************************************************************************/
t_Bool spi_tclAudioSrcInfo::bGetActiveSrc(t_U8 &rfu8ActiveSrc)
{
   t_Bool bRetval = false;
   for (AudSrcInfoMapItr itMapSrcInfo = m_mapAudioSrcInfo.begin(); itMapSrcInfo
            != m_mapAudioSrcInfo.end(); itMapSrcInfo++)
   {
      if (e8SRC_ACTIVE == itMapSrcInfo->second.enSrcState)
      {
         bRetval = true;
         rfu8ActiveSrc = itMapSrcInfo->first;
         break;
      }
   }
   ETG_TRACE_USR2(("spi_tclAudioSrcInfo::bGetActiveSrc left with ActiveSrc = %d Retval = %d ",
            rfu8ActiveSrc, ETG_ENUM(BOOL, bRetval)));
   return bRetval;
}
