/*!
 *******************************************************************************
 * \file             spi_tclAudio.cpp
 * \brief            Main Audio class that provides interface to delegate 
                     the execution of command and handle response
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Main Audio class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                         | Modifications
 28.10.2013 |  Hari Priya E R(RBEI/ECP2)      | Initial Version
 18.11.2013 |  Raghavendra S (RBEI/ECP2)      | Redefinition of Interfaces
 03.01.2014 |  Hari Priya E R(RBEI/ECP2)      | Included variant handling for main
 06.04.2014 |  Ramya Murthy                   | Initialisation sequence implementation
 10.06.2014 |  Ramya Murthy                   | Audio policy redesign implementation.
 31.07.2014 | Ramya Murthy                    | SPI feature configuration via LoadSettings()
 06.05.2015 | Tejaswini H B                   | Lint Errors and warning fixes
 25.06.2015 | Tejaswini HB (RBEI/ECP2)        |Featuring out Android Auto
 07.07.2015 | Ramya Murthy                    | Fix for deallocating maun audio source on device disconnection,
                                                during active phone call or speech.
 18.08.2015 |  Shiva Kumar G                  | Added elements for ML Dynamic Audio   
 24.02.2016 |  Ramya Murthy                   | Implementation of default audio type for CarPlay
 23.05.2016 |  Ramya Murthy                   | Moved ECNR logic to technology classes
 01.03.2017 |  Shiva Kumar G                  | Changes to enable/disable CarPlay feature
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclMediator.h"
#include "spi_tclAudioDevBase.h"
#include "spi_tclAudioIn.h"

#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDiPoAudio.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
#include "spi_tclAAPAudio.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINAudio.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLAudio.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarAudio.h"
#endif

#include "spi_tclAudioPolicyBase.h"
#include "spi_tclAudio.h"
#include "FunctionThreader.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclAudio.cpp.trc.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenDeviceCategory, GetAudio_t*> Audio;
#endif
#define AUD_MAX_SOURCES e8AUD_INVALID
#define AUD_MAX_SAMPLERATE e8AUD_SAMPLERATE_MAX

#ifdef VARIANT_S_FTR_ENABLE_AUDIO_PREPROCESSING_USED
const t_String gszAudDeviceOutName[AUD_MAX_SOURCES][AUD_MAX_SAMPLERATE] =
                                    {
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                   //MAIN_OUT
                                    {"AdevAudioInfoMonoOut", "AdevAudioInfoMonoOut","AdevAudioInfoMonoOut","AdevAudioInfoMonoOut"}, //MIX_OUT
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                   //ALERT_OUT
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //PHONE_IN
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //VR_IN
                                    {"AdevAudioInfoMonoOut", "AdevAudioInfoMonoOut","AdevAudioInfoMonoOut","AdevAudioInfoMonoOut"}, //DUCK
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                   // SPI_TRANSIENT
                                    {"AdevSPIInfo", "AdevSPIInfo", "AdevSPIInfo", "AdevSPIInfo"},                                   // SPI_STEREO MIX
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //DEFAULT
                                    {"AdevVoiceOutStereo", "AdevVoiceOutStereo", "AdevVoiceOutStereo", "AdevVoiceOutStereo"}        //VR_STEREO
                                    };
#else
const t_String gszAudDeviceOutName[AUD_MAX_SOURCES][AUD_MAX_SAMPLERATE] =
                                    {
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                   //MAIN_OUT
                                    {"AdevAudioInfoMonoOut", "AdevAudioInfoMonoOut","AdevAudioInfoMonoOut","AdevAudioInfoMonoOut"}, //MIX_OUT
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                   //ALERT_OUT
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //PHONE_IN
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //VR_IN
                                    {"AdevAudioInfoMonoOut", "AdevAudioInfoMonoOut","AdevAudioInfoMonoOut","AdevAudioInfoMonoOut"}, //DUCK
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                   // SPI_TRANSIENT
                                    {"AdevSPIInfo", "AdevSPIInfo", "AdevSPIInfo", "AdevSPIInfo"},                                   // SPI_STEREO MIX
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //DEFAULT
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"}            						//VR_STEREO
                                    };
#endif

const t_String gszAudDeviceInName[AUD_MAX_SOURCES][AUD_MAX_SAMPLERATE] =
                                    {
                                    {"", "", "", ""},  																				//MAIN_OUT                                                                                                                                                                     //MAIN_OUT
                                    {"", "","",""},     																			//MIX_OUT                                                                                                                                                                                                                                        //MIX_OUT
                                    {"", "", "", ""},       																		//ALERT_OUT                                                                                                                                                                //ALERT_OUT
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //PHONE_IN
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //VR_IN
                                    {"", "","",""},           																		//DUCK                                                                                                                                                                                                                                   //DUCK
                                    {"", "", "", ""},             																	// SPI_TRANSIENT                                                                                                                                                          // SPI_TRANSIENT
                                    {"", "", "", ""},                                                                               // SPI_STEREO MIX                                                                                       // SPI_STEREO MIX
                                    {"AdevSPIInOut", "AdevSPIInOut_8", "AdevSPIInOut_16", "AdevSPIInOut"},                          //DEFAULT
                                    {"AdevVoiceInStereo", "AdevVoiceInStereo", "AdevVoiceInStereo", "AdevVoiceInStereo"}            //VR_STEREO
                                    };

const t_String gszECNRAudDeviceName[AUD_MAX_SOURCES][AUD_MAX_SAMPLERATE] =
                                    {
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                    //MAIN_OUT
                                    {"AdevAudioInfoMonoOut", "AdevAudioInfoMonoOut","AdevAudioInfoMonoOut","AdevAudioInfoMonoOut"},  //MIX_OUT
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                    //ALERT_OUT
                                    {"AdevECNRInOut", "AdevECNRInOut_8", "AdevECNRInOut_16", "AdevECNRInOut"},                       //PHONE_IN
                                    {"AdevECNRInOut", "AdevECNRInOut_8", "AdevECNRInOut_16", "AdevECNRInOut"},                       //VR_IN
                                    {"AdevAudioInfoMonoOut", "AdevAudioInfoMonoOut","AdevAudioInfoMonoOut","AdevAudioInfoMonoOut"},  //DUCK
                                    {"AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out", "AdevEnt1Out"},                                    // SPI_TRANSIENT
                                    {"AdevSPIInfo", "AdevSPIInfo", "AdevSPIInfo", "AdevSPIInfo"},                                    // SPI_STEREO MIX
                                    {"AdevECNRInOut", "AdevECNRInOut_8", "AdevECNRInOut_16", "AdevECNRInOut"},                       //DEFAULT
                                    {"AdevECNRInOut","AdevECNRInOut","AdevECNRInOut","AdevECNRInOut"}                                //VR_STEREO
                                    };

//! Flag that indicates if device information has to be cleared on audio route deallocation
//Unused, hence commenting the variable.
//static t_Bool sbClearDeviceInfoReqd = false;

#define AUD_ECNR_MIC_OUT 2
#define AUD_ECNR_RECV_IN 3
#define AUD_ECNR_RECV_OUT 4

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
 ** FUNCTION:  spi_tclAudio::spi_tclAudio();
 ***************************************************************************/
spi_tclAudio::spi_tclAudio(spi_tclAudioPolicyBase* poPolicyBase) : spi_tclSelectionIntf(e32COMPID_AUDIO),
   m_poAudioPolicyBase(poPolicyBase),
   m_poAudioInHandler(NULL),
   m_poGMainLoopThread(NULL),
   m_enAudioInState(e8_AUDIOIN_UNITIAILZED),
   m_poAudioSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclAudio::spi_tclAudio entered "));
   SPI_NORMAL_ASSERT(NULL == m_poAudioPolicyBase);

   for (t_U8 u8AudClients = 0; u8AudClients < NUM_AUD_CLIENTS; ++u8AudClients)
   {
      m_poAudDevHandlers[u8AudClients] = NULL;
   }

   for (t_U8 u8AudSources = 0; u8AudSources < e8AUD_INVALID; ++u8AudSources)
   {
      m_u8AudSources[u8AudSources] = 0;
   }
}//spi_tclAudio::spi_tclAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclAudio::~spi_tclAudio();
 ***************************************************************************/
spi_tclAudio::~spi_tclAudio()
{
   ETG_TRACE_USR1(("spi_tclAudio::~spi_tclAudio entered "));
   m_poAudioPolicyBase = NULL;
   m_poAudioInHandler  = NULL;
   m_poGMainLoopThread = NULL;
   m_enAudioInState    = e8_AUDIOIN_UNITIAILZED;
   m_poAudioSettings = NULL;

   for (t_U8 u8AudClients = 0; u8AudClients < NUM_AUD_CLIENTS; ++u8AudClients)
   {
      m_poAudDevHandlers[u8AudClients] = NULL;
   }

   for (t_U8 u8AudSources = 0; u8AudSources < e8AUD_INVALID; ++u8AudSources)
   {
      m_u8AudSources[u8AudSources] = 0;
   }
}//spi_tclAudio::~spi_tclAudio()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAudio::bInitialize()
 ***************************************************************************/
t_Bool spi_tclAudio::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclAudio::bInitialize entered "));

   t_Bool bInit = false;
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
   ETG_TRACE_USR1(("spi_tclAudio::bInitialize Audio[e8DEV_TYPE_MIRRORLINK]=%p",Audio[e8DEV_TYPE_MIRRORLINK]));
   if ((NULL != Audio[e8DEV_TYPE_MIRRORLINK]))
   {
      ETG_TRACE_USR1(("spi_tclAudio::bInitialize before obj m_poAudDevHandlers[e8DEV_TYPE_MIRRORLINK] = %p ", m_poAudDevHandlers[e8DEV_TYPE_MIRRORLINK]));
      m_poAudDevHandlers[e8DEV_TYPE_MIRRORLINK] = Audio[e8DEV_TYPE_MIRRORLINK]();
      SPI_NORMAL_ASSERT(NULL == m_poAudDevHandlers[e8DEV_TYPE_MIRRORLINK]);
   }
   ETG_TRACE_USR1(("spi_tclAudio::bInitialize obj success m_poAudDevHandlers[e8DEV_TYPE_MIRRORLINK] = %p ", m_poAudDevHandlers[e8DEV_TYPE_MIRRORLINK]));
#endif
   //! Create handlers
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
   m_poAudDevHandlers[e8DEV_TYPE_DIPO] = new spi_tclDiPoAudio();
   SPI_NORMAL_ASSERT(NULL == m_poAudDevHandlers[e8DEV_TYPE_DIPO]);
#endif


#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
   m_poAudDevHandlers[e8DEV_TYPE_ANDROIDAUTO] = new spi_tclAAPAudio();
   SPI_NORMAL_ASSERT(NULL == m_poAudDevHandlers[e8DEV_TYPE_ANDROIDAUTO]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
      m_poAudDevHandlers[e8DEV_TYPE_MYSPIN] = new (std::nothrow)spi_tclMySPINAudio();
      SPI_NORMAL_ASSERT(NULL == m_poAudDevHandlers[e8DEV_TYPE_MYSPIN]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
      m_poAudDevHandlers[e8DEV_TYPE_CARLIFE] = new (std::nothrow)spi_tclBDCLAudio();
      SPI_NORMAL_ASSERT(NULL == m_poAudDevHandlers[e8DEV_TYPE_CARLIFE]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
      m_poAudDevHandlers[e8DEV_TYPE_ONCAR] = new (std::nothrow) spi_tclOnCarAudio();
      SPI_NORMAL_ASSERT(NULL == m_poAudDevHandlers[e8DEV_TYPE_ONCAR]);
#endif

   //! Initialize the Audio handlers
   for(t_U8 u8Index = 0; u8Index < NUM_AUD_CLIENTS; ++u8Index)
   {
       tenDeviceCategory enDevCat = static_cast<tenDeviceCategory>(u8Index);
       if (NULL != m_poAudDevHandlers[enDevCat])
       {
           bInit = m_poAudDevHandlers[enDevCat]->bInitialise() || bInit;
       }
   }

   //! Audio In Handler for Echo Cancellation and Noise Reduction
   m_poAudioInHandler = new spi_tclAudioIn();
   SPI_NORMAL_ASSERT(NULL == m_poAudioInHandler);

   vRegisterCallbacks();

   //! Create a new thread for GMainLoop
   if (NULL != m_poAudioInHandler)
   {
      m_poGMainLoopThread = new shl::thread::FunctionThreader(
            m_poAudioInHandler);
      if (NULL != m_poGMainLoopThread)
      {
         m_poGMainLoopThread->bRunThread();
      }
   }
   SPI_NORMAL_ASSERT(NULL == m_poGMainLoopThread);
   //! Set the bInit to false, if all the SPI technologies cannot be initialized
   //! else return true if one of the SPI technologies can be initialized
   //! If Mirrorlink supports AudioIn in future modify the logic accordingly
   bInit = bInit && ((NULL != m_poAudioInHandler) && (NULL != m_poGMainLoopThread));

   ETG_TRACE_USR1(("spi_tclAudio::bInitialize left with result = %u ", ETG_ENUM(BOOL, bInit)));
   return bInit;
}//spi_tclAudio::bInitialize()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAudio::bUnInitialize()
 ***************************************************************************/
t_Bool spi_tclAudio::bUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclAudio::bUnInitialize entered "));

   for (t_U8 u8AudDevCnt = 0; u8AudDevCnt < NUM_AUD_CLIENTS; ++u8AudDevCnt)
   {
      RELEASE_MEM(m_poAudDevHandlers[u8AudDevCnt]);
   }
   RELEASE_MEM(m_poGMainLoopThread);
   RELEASE_MEM(m_poAudioInHandler);

   return true;
}//spi_tclAudio::bUnInitialize()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAudio::vLoadSettings()
***************************************************************************/
t_Void spi_tclAudio::vLoadSettings()
{
   ETG_TRACE_USR1(("spi_tclAudio::vLoadSettings entered "));

   if (NULL != m_poAudioSettings)
   {
      for (t_U8 u8AudSources = 0; u8AudSources < e8AUD_INVALID; ++u8AudSources)
      {
         m_u8AudSources[u8AudSources] = m_poAudioSettings->u8GetSourceNumber(
               static_cast<tenAudioDir> (u8AudSources));
      }
      //! Read and set audio pipeline configuration for alsa devices
      tmapAudioPipeConfig mapAudioPipeConfig;
      m_poAudioSettings->vGetAudioPipeConfig(mapAudioPipeConfig);
      tenLastModeSupport enLastModeType = m_poAudioSettings->enGetLastModeSupport();

      for (t_U8 u8AudClients = 0; u8AudClients < NUM_AUD_CLIENTS; ++u8AudClients)
      {
         tenDeviceCategory enDevCat = static_cast<tenDeviceCategory>(u8AudClients);
         if (NULL != m_poAudDevHandlers[enDevCat])
         {
		    m_poAudDevHandlers[enDevCat]->vSetAudioSettingsInstance(m_poAudioSettings);
            t_Bool bIsLastModeSupported = m_poAudioSettings->bGetLastModeSupport(enDevCat);
            m_poAudDevHandlers[enDevCat]->vSetAudioPipeConfig(mapAudioPipeConfig);
            m_poAudDevHandlers[enDevCat]->vSetLastModeSupport(bIsLastModeSupported);
         }
      }//for (t_U8 u8AudClients = 0; ...)

      trAudioConfig rAudioConfig;
      rAudioConfig.bMuteEnabled = (e8LASTMODE_DISABLED == enLastModeType)? (false) : (m_poAudioSettings->bGetAudioMuteEnabled());
      rAudioConfig.bVolLockEnabled = (e8LASTMODE_DISABLED == enLastModeType)? (false) : (m_poAudioSettings->bGetVolumeLockEnabled());
      if (NULL != m_poAudioPolicyBase)
      {
         m_poAudioPolicyBase->vSetAudioConfig(rAudioConfig);
      }
   } //if(NULL != m_poAudioSettings)
}//t_Void spi_tclAudio::vLoadSettings(...)

/***************************************************************************
** FUNCTION:  t_Void spi_tclAudio::vSaveSettings()
***************************************************************************/
t_Void spi_tclAudio::vSaveSettings()
{
   ETG_TRACE_USR1(("spi_tclAudio::vSaveSettings entered "));
   //Add code
}//t_Void spi_tclAudio::vSaveSettings()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclAudio::vRegisterCallbacks()
{
   ETG_TRACE_USR1(("spi_tclAudio::vRegisterCallbacks entered "));
   /*lint -esym(40,fvSelectDeviceResp) Undeclared identifier */
   /*lint -esym(40,fvStartAudioResp) fvStartAudioResp Undeclared identifier */
   /*lint -esym(40,fvStopAudioResp) fvStopAudioResp Undeclared identifier */
   /*lint -esym(40,fvLaunchAudioReq)fvLaunchAudioReq  Undeclared identifier */
   /*lint -esym(40,fvTerminateAudioReq)fvTerminateAudioReq Undeclared identifier */
   /*lint -esym(40,fvInitializeAudioIn) fvInitializeAudioIn Undeclared identifier */
   /*lint -esym(40,fvUninitializeAudioIn) fvUninitializeAudioIn Undeclared identifier */
   /*lint -esym(40,fvSetAlsaDevice) fvSetAlsaDevice Undeclared identifier */
   /*lint -esym(40,fvSetAudioInConfig)fvSetAudioInConfig Undeclared identifier */
   /*lint -esym(40,_1)_1 Undeclared identifier */
   /*lint -esym(40,_2)_2 Undeclared identifier */
   /*lint -esym(40,_3)_1 Undeclared identifier */
   /*lint -esym(40,_4)_2 Undeclared identifier */
   

   trAudioCallbacks rAudCallbacks;

   //! Populate the callback structure
   rAudCallbacks.fvSelectDeviceResp
            = std::bind(&spi_tclAudio::vCbSelectDeviceResp,
                     this,
                     std::placeholders::_1,
                     std::placeholders::_2,
                     std::placeholders::_3);

   rAudCallbacks.fvStartAudioResp
            = std::bind(&spi_tclAudio::vCbStartSrcActivity,
                     this,
                     std::placeholders::_1,
                     std::placeholders::_2);

   rAudCallbacks.fvStopAudioResp = std::bind(&spi_tclAudio::vCbStopSrcActivity,
            this,
            std::placeholders::_1,
            std::placeholders::_2);

   rAudCallbacks.fvLaunchAudioReq = std::bind(&spi_tclAudio::vLaunchAudio,
            this,
            std::placeholders::_1,
            std::placeholders::_2,
            std::placeholders::_3,
            std::placeholders::_4);

   rAudCallbacks.fvTerminateAudioReq = std::bind(&spi_tclAudio::vTerminateAudio,
            this,
            std::placeholders::_1,
            std::placeholders::_2,
            std::placeholders::_3);

   rAudCallbacks.fvStartAudioIn = std::bind(&spi_tclAudio::vStartAudioIn,
            this,
            std::placeholders::_1);

   rAudCallbacks.fvStopAudioIn = std::bind(&spi_tclAudio::vStopAudioIn,
            this,
            std::placeholders::_1);

   rAudCallbacks.fvInitializeAudioIn = std::bind(&spi_tclAudio::vInitializeAudioIn,
            this,
            std::placeholders::_1,
            std::placeholders::_2);

   rAudCallbacks.fvUninitializeAudioIn = std::bind(&spi_tclAudio::vUninitializeAudioIn,
            this,
            std::placeholders::_1);

   rAudCallbacks.fvSetAlsaDevice = std::bind(&spi_tclAudio::vSetAlsaDevice,
            this,
            std::placeholders::_1);

   rAudCallbacks.fvSetAudioInConfig = std::bind(&spi_tclAudio::vSetAudioInConfig,
            this,
            std::placeholders::_1,
            std::placeholders::_2);

   rAudCallbacks.fbSetAudioDucking = std::bind(&spi_tclAudio::bSetAudioDucking,
            this,
            std::placeholders::_1,
            std::placeholders::_2,
            std::placeholders::_3,
            std::placeholders::_4);

   rAudCallbacks.fvPostVRSupportInfo = std::bind(&spi_tclAudio::vPostVRSupportInfo,
            this,
            std::placeholders::_1,
            std::placeholders::_2,
            std::placeholders::_3);

   /*rAudCallbacks.fbSetSrcAvailability = std::bind(&spi_tclAudio::bSetAudioSrcAvailability,
            this,
            std::placeholders::_1,
            std::placeholders::_2,
            std::placeholders::_3);*/

   rAudCallbacks.fbSetSourceMute = std::bind(&spi_tclAudio::bSetAudioSrcMute,
            this,
            std::placeholders::_1,
            std::placeholders::_2);

   rAudCallbacks.fvPostRelativelyActiveMediaAppStatus = std::bind(&spi_tclAudio::vPostRelativelyActiveMediaAppStatus,
            this,
            std::placeholders::_1,
            std::placeholders::_2);

   //! Register for callbacks with SPI connections
   for (t_U8 u8AudDevCnt = 0; u8AudDevCnt < NUM_AUD_CLIENTS; ++u8AudDevCnt)
   {
      if (NULL != m_poAudDevHandlers[u8AudDevCnt])
      {
         m_poAudDevHandlers[u8AudDevCnt]->vRegisterCallbacks(rAudCallbacks);
      } // if(NULL != m_poAudDevHandlers[u8AudDevCnt])
   } // for(t_U8 u8AudDevCnt = 0; u8AudDevCnt < NUM_AUD_CLIENTS; ++u8AudDevCnt)
}//t_Void spi_tclAudio::vRegisterCallbacks()

/***************************************************************************
 ** FUNCTION:  static t_Void vCbSelectDeviceResp(t_U32 u32DeviceId,..)
 ***************************************************************************/
t_Void spi_tclAudio::vCbSelectDeviceResp(t_U32 u32DeviceId,
         tenDeviceConnectionReq enDevConnReq, t_Bool bResult)
{
   ETG_TRACE_USR1(("spi_tclAudio::vCbSelectDeviceResp entered: Result = %u ", ETG_ENUM(BOOL,bResult)));
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   SPI_INTENTIONALLY_UNUSED(enDevConnReq);

   tenErrorCode enErrorCode = (true==bResult)? e8NO_ERRORS : e8SELECTION_FAILED;
   //! Post Select device result
   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_AUDIO, enErrorCode);
   } //if (NULL != poMediator)

}//t_Void spi_tclAudio::vCbSelectDeviceResp(t_U32 u32DeviceId,..)

/***************************************************************************
 ** FUNCTION:  static t_Void vCbStartSrcActivity(tenAudioDir enAudDir,..)
 ***************************************************************************/
t_Void spi_tclAudio::vCbStartSrcActivity(tenAudioDir enAudDir, t_Bool bResult)
{
   ETG_TRACE_USR1(("spi_tclAudio::vCbStartSrcActivity entered: AudioDirection = %u, Result = %u ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(BOOL,bResult)));

   if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vStartSourceActivityResult(m_u8AudSources[enAudDir], bResult);
   }
}//t_Void spi_tclAudio::vCbStartSrcActivity(t_U32 u32DeviceId...)


/***************************************************************************
 ** FUNCTION:  static t_Void vCbStopSrcActivity(tenAudioDir enAudDir,..)
 ***************************************************************************/
t_Void spi_tclAudio::vCbStopSrcActivity(tenAudioDir enAudDir, t_Bool bResult)
{
   ETG_TRACE_USR1(("spi_tclAudio::vCbStopSrcActivity entered: AudioDirection = %u, Result = %u ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(BOOL,bResult)));

   if (NULL != m_poAudioPolicyBase)
   {
      t_U8 u8SrcNum = (e8AUD_INVALID > enAudDir)? (m_u8AudSources[enAudDir]):(0);
      tenAudSrcState enSrcState = m_oAudioSrcInfo.enGetSrcState(u8SrcNum);

      if (e8SRC_PAUSE == enSrcState)
      {
        m_poAudioPolicyBase->vPauseSourceActivityResult(u8SrcNum, bResult);
      }
      else
      {
        m_poAudioPolicyBase->vStopSourceActivityResult(u8SrcNum, bResult);
      }
   }//if (NULL != m_poAudioPolicyBase)
}//t_Void spi_tclAudio::vCbStopSrcActivity(tenAudioDir enAudDir...)

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAudio::bOnRouteAllocateResult(t_U8, trAudSrcInfo&)
 ***************************************************************************/
t_Bool spi_tclAudio::bOnRouteAllocateResult(t_U8 u8SourceNum,
         trAudSrcInfo& rfrSrcInfo)
{
   ETG_TRACE_USR1(("spi_tclAudio::bOnRouteAllocateResult entered for: SourceNum = %u ", u8SourceNum));
   ETG_TRACE_USR4(("[PARAM]::spi_tclAudio::bOnRouteAllocateResult MainAudDevName out = %s ",
         rfrSrcInfo.rMainAudDevNames.szOutputDev.c_str()));
   ETG_TRACE_USR4(("[PARAM]::spi_tclAudio::bOnRouteAllocateResult MainAudDevName in = %s ",
         rfrSrcInfo.rMainAudDevNames.szInputDev.c_str()));
   ETG_TRACE_USR4(("[PARAM]::spi_tclAudio::bOnRouteAllocateResult EcnrAudDevName out = %s ",
         rfrSrcInfo.rEcnrAudDevNames.szOutputDev.c_str()));
   ETG_TRACE_USR4(("[PARAM]::spi_tclAudio::bOnRouteAllocateResult EcnrAudDevName in = %s ",
         rfrSrcInfo.rEcnrAudDevNames.szInputDev.c_str()));

   //Store the details here in Audio Source class if the Source is not active
   if (e8SRC_INACTIVE == m_oAudioSrcInfo.enGetSrcState(u8SourceNum))
   {
       m_oAudioSrcInfo.vSetAudSrcInfo(u8SourceNum, rfrSrcInfo);
   } //if(e8SRC_INACTIVE == m_oAudioSrcInfo.enGetSrcState(u8SourceNum))

   t_Bool bRetVal = false;
   t_U8 u8DevCategory = static_cast<t_U8> (m_oAudioSrcInfo.enGetDeviceCategory(u8SourceNum));
   tenAudioDir enAudDir;

   if (NULL != m_poAudDevHandlers[u8DevCategory])
   {
      t_U32 u32DevID = m_oAudioSrcInfo.u32GetDeviceID(u8SourceNum);
      enAudDir = m_oAudioSrcInfo.enGetAudDirection(u8SourceNum);
      tenAudioSamplingRate enSamplingRate = m_oAudioSrcInfo.enGetAudSamplingRate(u8SourceNum);
      tenAudioSamplingRate enNativeSamplingRate = m_oAudioSrcInfo.enGetAudNativeSamplingRate(u8SourceNum);

      /** enNativeSamplingRate populated to non default value in case of CP Telephony Audio **/
      m_poAudDevHandlers[u8DevCategory]->bInitializeAudioPlayback(u32DevID,enAudDir,enSamplingRate,enNativeSamplingRate);
   }//if (NULL != m_poAudDevHandlers[u8DevCategory])

   ETG_TRACE_USR4(("spi_tclAudio::bOnRouteAllocateResult left with result = %u ", ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}//spi_tclAudio::bOnRouteAllocateResult(t_U8, trAudSrcInfo&)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vOnRouteDeAllocateResult(t_U8 u8SourceNum)
 ***************************************************************************/
t_Void spi_tclAudio::vOnRouteDeAllocateResult(t_U8 u8SourceNum)
{
   ETG_TRACE_USR1(("spi_tclAudio::vOnRouteDeAllocateResult entered for: SourceNum = %u ", u8SourceNum));

   t_U8 u8DevCategory = static_cast<t_U8> (m_oAudioSrcInfo.enGetDeviceCategory(u8SourceNum));
   t_Bool bIsSrcInactive = (e8SRC_INACTIVE == m_oAudioSrcInfo.enGetSrcState(u8SourceNum));
   //@Note: Do not read audio direction from m_oAudioSrcInfo, since on startup it will not contain any info
   t_Bool bIsMainAudSrc = (u8SourceNum == m_u8AudSources[e8AUD_MAIN_OUT]);

   if (NULL != m_poAudDevHandlers[u8DevCategory])
   {
      t_U32 u32DevID = m_oAudioSrcInfo.u32GetDeviceID(u8SourceNum);
      tenAudioDir enAudDir = m_oAudioSrcInfo.enGetAudDirection(u8SourceNum);
      m_poAudDevHandlers[u8DevCategory]->bFinalizeAudioPlayback(u32DevID, enAudDir);
   }
   else if ((bIsSrcInactive) && (bIsMainAudSrc))
   {
      //@Note: To handle startup case, when audio source info is not available in m_oAudioSrcInfo since
      //no device has started any audio (when last mode is enabled)
      ETG_TRACE_USR2(("[DESC]::vOnRouteDeAllocateResult: Setting source unavailable since source info not found"));
      bSetAudioSrcAvailability(e8AUD_MAIN_OUT, e8SRC_NOT_AVAILABLE, e8REASON_NOMEDIA);
   }

   if (bIsMainAudSrc)
   {
      //@Note: Unmute whenever main audio channel is deallocated due to SA_PAUSE or SA_OFF, since
      //volume lock should be removed for next main/transient source
      bSetAudioSrcMute(e8AUD_MAIN_OUT, false);
   }

   //@Note: Clearing device info is avoided so that audio manager requests are handled by technology audio classes
   /*if (bIsSrcInactive)
   {
      m_oAudioSrcInfo.vClearDeviceInfo(u8SourceNum);
   }*/

   ETG_TRACE_USR1(("spi_tclAudio::vOnRouteDeAllocateResult left "));
}//spi_tclAudio::vOnRouteDeAllocateResult(t_U8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vOnStartSourceActivity(t_U8 u8SourceNum)
 ***************************************************************************/
t_Void spi_tclAudio::vOnStartSourceActivity(t_U8 u8SourceNum, t_Bool bResult)
{
   ETG_TRACE_USR1(("spi_tclAudio::vOnStartSourceActivity entered for: SourceNum = %u ", u8SourceNum));

   trAudSrcInfo rSrcInfo;
   t_Bool bRetVal     = m_oAudioSrcInfo.bGetAudSrcInfo(u8SourceNum, rSrcInfo);
   t_U8 u8DevCategory = static_cast<t_U8> (m_oAudioSrcInfo.enGetDeviceCategory(u8SourceNum));
   t_U32 u32DevID     = m_oAudioSrcInfo.u32GetDeviceID(u8SourceNum);

   //Route is allocated for SPI - Set the ActiveSrcFlag to e8SRC_ACTIVE
   m_oAudioSrcInfo.vSetActiveSrcFlag(u8SourceNum, e8SRC_ACTIVE);

   if ((true == bRetVal) && (NULL != m_poAudDevHandlers[u8DevCategory]) && (0 != u32DevID))
   {
      tenAudioDir enAudDir = m_oAudioSrcInfo.enGetAudDirection(u8SourceNum);
      tenAudioSamplingRate enAudSampleRate = m_oAudioSrcInfo.enGetAudSamplingRate(u8SourceNum);

      if ((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir) || (e8AUD_VR_STEREO == enAudDir))
      {
         bRetVal = m_poAudDevHandlers[u8DevCategory]->bStartAudio(u32DevID,
                  (gszAudDeviceOutName[enAudDir][enAudSampleRate]).c_str(),
                  (gszAudDeviceInName[enAudDir][enAudSampleRate]).c_str(), enAudDir);
      }//if ((e8AUD_PHONE_IN == enAudDir) || ...))
      else if( (e8AUD_MAIN_OUT == enAudDir) || (e8AUD_MIX_OUT == enAudDir) || 
         (e8AUD_ALERT_OUT == enAudDir) ||(e8AUD_DUCK == enAudDir) || 
         (e8AUD_TRANSIENT == enAudDir) || ( e8AUD_STEREO_MIX_OUT == enAudDir) )
      {
         if ("" != rSrcInfo.rMainAudDevNames.szOutputDev)
         {
            bRetVal = m_poAudDevHandlers[u8DevCategory]->bStartAudio(u32DevID,
                      rSrcInfo.rMainAudDevNames.szOutputDev, enAudDir);
         }
         else 
         {
            //@Note: Alsa device name is retrieved (using bGetAudSrcInfo method) as NULL when last mode was CarPlay.
            //Hence, using directly from global audio device name array
            bRetVal = m_poAudDevHandlers[u8DevCategory]->bStartAudio(u32DevID,
                      gszAudDeviceOutName[enAudDir][enAudSampleRate], enAudDir);
         }
      }//else if((e8AUD_MAIN_OUT == enAudDir) || (e8AUD_MIX_OUT == enAudDir) ...)
   }//if ((true == bRetVal) && (NULL != m_poAudDevHandlers[u8DevCategory]))

   else if ((NULL != m_poAudioPolicyBase) && (true == bResult))
   {
      //@Note: To handle startup case, when audio source info is not available in m_oAudioSrcInfo since
      //no device has started any audio (when last mode is enabled)
      ETG_TRACE_USR2(("[DESC]::vOnStartSourceActivity: Sending response since source info not found"));
      m_poAudioPolicyBase->vStartSourceActivityResult(u8SourceNum, true);

      //@Note: Do not read audio direction from m_oAudioSrcInfo, since on startup it will not contain any info
      t_Bool bIsMainAudSrc = (u8SourceNum == m_u8AudSources[e8AUD_MAIN_OUT]);
      if (bIsMainAudSrc)
      {
         ETG_TRACE_USR2(("[DESC]::vOnStartSourceActivity: Setting mute since no audio is playing"));
         bSetAudioSrcMute(e8AUD_MAIN_OUT, true);
      }
   }

   ETG_TRACE_USR1(("spi_tclAudio::vOnStartSourceActivity left "));
}//spi_tclAudio::vOnStartSourceActivity(t_U8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vOnPauseSourceActivity(t_U8 u8SourceNum)
 ***************************************************************************/
t_Void spi_tclAudio::vOnPauseSourceActivity(t_U8 u8SourceNum)
{
   ETG_TRACE_USR1(("spi_tclAudio::vOnPauseSourceActivity entered for: SourceNum = %u ", u8SourceNum));

   t_U8 u8DevCategory = static_cast<t_U8> (m_oAudioSrcInfo.enGetDeviceCategory(u8SourceNum));

   //Set the ActiveSrcFlag to PAUSE since source is deallocated
   m_oAudioSrcInfo.vSetActiveSrcFlag(u8SourceNum, e8SRC_PAUSE);

   if (NULL != m_poAudDevHandlers[u8DevCategory])
   {
      t_U32 u32DevID = m_oAudioSrcInfo.u32GetDeviceID(u8SourceNum);
      tenAudioDir enAudDir = m_oAudioSrcInfo.enGetAudDirection(u8SourceNum);
      m_poAudDevHandlers[u8DevCategory]->vStopAudio(u32DevID, enAudDir, true);
   }
   else if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vPauseSourceActivityResult(u8SourceNum, true);
   }

   ETG_TRACE_USR1(("spi_tclAudio::vOnPauseSourceActivity left. "));
}//spi_tclAudio::vOnPauseSourceActivity(t_U8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vOnStopSourceActivity(t_U8 u8SourceNum)
 ***************************************************************************/
t_Void spi_tclAudio::vOnStopSourceActivity(t_U8 u8SourceNum)
{
   ETG_TRACE_USR1(("spi_tclAudio::vOnStopSourceActivity entered for: SourceNum = %u ", u8SourceNum));

   t_U8 u8DevCategory = static_cast<t_U8> (m_oAudioSrcInfo.enGetDeviceCategory(u8SourceNum));
   m_oAudioSrcInfo.vSetActiveSrcFlag(u8SourceNum, e8SRC_INACTIVE);

   if (NULL != m_poAudDevHandlers[u8DevCategory])
   {
      t_U32 u32DevID = m_oAudioSrcInfo.u32GetDeviceID(u8SourceNum);
      tenAudioDir enAudDir = m_oAudioSrcInfo.enGetAudDirection(u8SourceNum);
      m_poAudDevHandlers[u8DevCategory]->vStopAudio(u32DevID, enAudDir, false);
   }
   else if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vStopSourceActivityResult(u8SourceNum, true);
   }

   ETG_TRACE_USR1(("spi_tclAudio::vOnStopSourceActivity left. "));
}//spi_tclAudio::vOnStopSourceActivity(t_U8 u8SourceNum)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vSelectDevice
 ***************************************************************************/
t_Void spi_tclAudio::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclAudio::vSelectDevice request received for: "
         "DeviceHandle = %u, Connection Request type = %u ",
         corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ,corfrSelectReq.m_enDevConnReq) ));

   t_U8 u8DevCategory = static_cast<t_U8> (corfrSelectReq.m_enDevCategory);

   if (NULL != m_poAudDevHandlers[u8DevCategory])
   {
      (e8DEVCONNREQ_SELECT == corfrSelectReq.m_enDevConnReq) ?
      ((t_Void)(m_poAudDevHandlers[u8DevCategory]->bSelectAudioDevice(corfrSelectReq.m_u32DeviceHandle))) :
      (m_poAudDevHandlers[u8DevCategory]->vDeselectAudioDevice(corfrSelectReq.m_u32DeviceHandle));

      //@Note:
      //1. Audio source info is cleared on Device Select/Deselect Callback
      //2. Deactivation of audio channels on device deselection is moved to technology handlers

   }// if (NULL != m_poAudDevHandlers[u8DevCategory])

}//spi_tclAudio::vSelectDevice(t_U32 u32DeviceId, tenEnabledInfo...)

/***************************************************************************
** FUNCTION:  t_Void spi_tclAudio::vSelectDeviceResult(t_U32...)
***************************************************************************/
t_Void spi_tclAudio::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclAudio::vSelectDeviceResult received for: "
         "DeviceHandle = %u, Connection Request type = %u, ErrorCode = %u ",
         corfrSelectReq.m_u32DeviceHandle, ETG_ENUM(CONNECTION_REQ, corfrSelectReq.m_enDevConnReq),
          ETG_ENUM(ERROR_CODE, enErrorCode)));

   tenResponseCode enRespCode = (e8NO_ERRORS == enErrorCode) ? e8SUCCESS : e8FAILURE;

   const tenDeviceCategory cenDevCat = corfrSelectReq.m_enDevCategory;
   if (NULL != m_poAudDevHandlers[cenDevCat])
   {
      m_poAudDevHandlers[cenDevCat]->vUpdateDeviceSelection(corfrSelectReq.m_u32DeviceHandle,
               cenDevCat, corfrSelectReq.m_enDevConnReq, enRespCode, enErrorCode);
   } // if (NULL != m_poAudDevHandlers[cenDevCat])

   if ((e8DEVCONNREQ_DESELECT == corfrSelectReq.m_enDevConnReq) &&
            (e8SRC_INACTIVE == m_oAudioSrcInfo.enGetSrcState(m_u8AudSources[e8AUD_MAIN_OUT])))
   {
      ETG_TRACE_USR2(("[DESC]::vSelectDeviceResult: Setting source unavailable since entertainment source is with HU "));
      bSetAudioSrcAvailability(e8AUD_MAIN_OUT, e8SRC_NOT_AVAILABLE, e8REASON_NOMEDIA);
   }
}//spi_tclAudio::vSelectDeviceResult(t_U32 u32DeviceId, tenEnabledInfo...)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vLaunchAudio(t_U32 u32DeviceId,
 tenDeviceCategory enDevCat, tenAudioDir enAudDir, tenAudioSampleRate enAudSampleRate)
 ***************************************************************************/
t_Void spi_tclAudio::vLaunchAudio(t_U32 u32DeviceId,
         tenDeviceCategory enDevCat, tenAudioDir enAudDir, trAudSampleRate rAudSampleRate)
{
   ETG_TRACE_USR1(("spi_tclAudio::vLaunchAudio request received for: "
         "DeviceHandle = %u, Device Category = %u, Audio Direction = %u",
         u32DeviceId, ETG_ENUM(DEVICE_CATEGORY, enDevCat),
         ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   ETG_TRACE_USR1(("spi_tclAudio::vLaunchAudio request:Audio SampleRate = %u,"
         "Audio Telephony SampleRate = %u",ETG_ENUM(SAMPLING_RATE, rAudSampleRate.enSampleRate),
         ETG_ENUM(SAMPLING_RATE, rAudSampleRate.enNativeSampleRate)));

   t_U8 u8SrcNum = (e8AUD_INVALID > enAudDir)? (m_u8AudSources[enAudDir]):(0);

   if ((0 != u8SrcNum) && (NULL != m_poAudioPolicyBase))
   {
      m_oAudioSrcInfo.vSetDeviceInfo(u8SrcNum, u32DeviceId, enDevCat, enAudDir, rAudSampleRate);

      tenAudSrcState enAudSrcState = m_oAudioSrcInfo.enGetSrcState(u8SrcNum);
      ETG_TRACE_USR2(("[PARAM]::vLaunchAudio: Audio Source State = %d ", enAudSrcState));

      //TODO - temporary solution to mock source activation for all SPI technologies in G3G project. (PSARCCB-8613)
      //Feature switch to be removed once correct solution is found.
      t_Bool bMockSrcActivation = ( e8SRC_ACTIVE == enAudSrcState);

      //Fix for NCG3D-28714, NCG3D-24773. To resume audio in paused state on demuting audio
      if (e8SRC_ACTIVE != enAudSrcState)
      {
         t_Bool bRetVal = m_poAudioPolicyBase->bRequestAudioActivation(u8SrcNum);
         ETG_TRACE_USR2(("[DESC]::vLaunchAudio: Is Audio activation request successful = %u ",
               ETG_ENUM(BOOL, bRetVal)));

      }// if (e8SRC_ACTIVE != enAudSrcState)
      else if (true == bMockSrcActivation)
      {
         ETG_TRACE_USR2(("[DESC]::vLaunchAudio: Mocking source activation since source is already allocated "));

         if (e8AUD_MAIN_OUT == enAudDir)
         {
            bSetAudioSrcMute(enAudDir, false);
         }

         //Since the Source is already active, get the Source Info and use the existing APIs to mock the Audio Sequence
         //@Note: This is not required for CarPlay since it is taken care when SPI receives Audio device info from Mediaplayer
         trAudSrcInfo rSrcInfo;
         m_oAudioSrcInfo.bGetAudSrcInfo(u8SrcNum, rSrcInfo);
         bOnRouteAllocateResult(u8SrcNum, rSrcInfo);
         vOnStartSourceActivity(u8SrcNum, false);
      } //else if (true == bMockSrcActivation)
   }
   ETG_TRACE_USR1(("spi_tclAudio::vLaunchAudio left "));
}//spi_tclAudio::vLaunchAudio(t_U32 u32DeviceId,tenDeviceCategory...)

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vTerminateAudio(t_U32 u32DeviceId...)
 ***************************************************************************/
t_Void spi_tclAudio::vTerminateAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, tenSrcAvailabilityReason enSrcAvlReason)
{
   ETG_TRACE_USR1(("spi_tclAudio::vTerminateAudio request received for: "
         "DeviceHandle = %u, Audio Direction = %u ",
         u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));

   /*Not required to check if the source is active for sending deactivation request.
   Reason: If Audio Manager is in process of restoring source, deactivation request needs to be triggered
   as the source active flag will not be set yet.*/
   if ((NULL != m_poAudioPolicyBase) && (e8AUD_INVALID != enAudDir))
   {
       /*SPI MW improvement for NCG3D-142495,e8AUD_VR_STEREO is also added here
       For NCG3D-142495,activate VR channel request is mocked from SPI since deactivate sequence is not yet finished.
       Because of this ActiveSrcFlag is always active.This is as good as VR is always active from SPI which is not the case from audio*/
       if ((e8AUD_MIX_OUT == enAudDir) || (e8AUD_STEREO_MIX_OUT == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir) || (e8AUD_VR_STEREO == enAudDir))
       {
           /*SPI route is going to be deactivated for Alternate channel - Set the ActiveSrcFlag to FALSE
           This will resolve the issue if back to back requests for MIX channel is triggered by iPhone.*/
           m_oAudioSrcInfo.vSetActiveSrcFlag(m_u8AudSources[enAudDir], e8SRC_INACTIVE);
       }//End of if(e8AUD_MIX_OUT == enAudDir)

      (t_Void) m_poAudioPolicyBase->bRequestAudioDeactivation(m_u8AudSources[enAudDir], enSrcAvlReason);
   }//if (NULL != m_poAudioPolicyBase)

   ETG_TRACE_USR1(("spi_tclAudio::vTerminateAudio left "));
}//spi_tclAudio::vTerminateAudio(t_U32 u32DeviceId,tenAudioDir enAudDir)

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAudio::bGetAudSrcInfo(const t_U8 cu8SourceNum,
 **            trAudSrcInfo& rfrSrcInfo)
 ***************************************************************************/
t_Bool spi_tclAudio::bGetAudSrcInfo(const t_U8 cu8SourceNum, trAudSrcInfo& rfrSrcInfo)
{
   t_Bool bRetVal = m_oAudioSrcInfo.bGetAudSrcInfo(cu8SourceNum, rfrSrcInfo);
   ETG_TRACE_USR3(("spi_tclAudio::bGetAudSrcInfo left with result = %u ", ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::bSetAudioBlockingMode()
***************************************************************************/
t_Bool spi_tclAudio::bSetAudioBlockingMode(t_Bool bGlobalMuteEnabled,
      t_Bool bMediaAudioBlockingEnabled, const trUserContext& rfcorUsrCntxt) const
{
   ETG_TRACE_USR1(("spi_tclAudio::bSetAudioBlockingMode entered "));
   SPI_INTENTIONALLY_UNUSED(rfcorUsrCntxt);

   t_Bool bRet = false;

   for(t_U8 u8Index = 0; u8Index < NUM_AUD_CLIENTS; ++u8Index)
   {
	   if(NULL != m_poAudDevHandlers[u8Index])
	   {
		   bRet = m_poAudDevHandlers[u8Index]->bSetAudioBlockingMode(
				   bGlobalMuteEnabled,bMediaAudioBlockingEnabled);
	   }//if(NULL != m_poAudDevHandlers[u8DevCat])
   }//for(t_U8 u8Index = 0; u8Index < NUM_AUD_CLIENTS; ++u8Index)

   return bRet;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::bSetAudioSrcAvailability()
***************************************************************************/
t_Bool spi_tclAudio::bSetAudioSrcAvailability(tenAudioDir enAudioDir,
         tenSrcAvailability enSrcAvailability, tenSrcAvailabilityReason enAvailReason)
{
   ETG_TRACE_USR1(("spi_tclAudio::bSetAudioSrcAvailability entered for: "
         "Audio Direction = %u, Src Availability = %u, Availability Reason = %d",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudioDir), ETG_ENUM(SRC_AVAILABILITY, enSrcAvailability), ETG_ENUM(AVAIL_REASON, enAvailReason)));

   t_Bool bRetVal = false;
   t_U8 u8SrcNum = (e8AUD_INVALID > enAudioDir)? (m_u8AudSources[enAudioDir]):(0);
   if((NULL != m_poAudioPolicyBase) && (0 != u8SrcNum))
   {
      bRetVal = m_poAudioPolicyBase->bSetSrcAvailability(u8SrcNum, enSrcAvailability, enAvailReason);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::bSetAudioSrcMute()
***************************************************************************/
t_Bool spi_tclAudio::bSetAudioSrcMute(tenAudioDir enAudioDir, t_Bool bIsMute)
{
   ETG_TRACE_USR1(("spi_tclAudio::bSetSourceMute entered for: Audio Direction = %u, IsMute = %u",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudioDir), ETG_ENUM(BOOL, bIsMute)));

   t_Bool bRetVal = false;
   t_U8 u8SrcNum = (e8AUD_INVALID > enAudioDir)? (m_u8AudSources[enAudioDir]):(0);
   if ((NULL != m_poAudioPolicyBase) && (0 != u8SrcNum) && (e8AUD_MAIN_OUT == enAudioDir))
   {
      bRetVal = m_poAudioPolicyBase->bSetSourceMute(u8SrcNum, bIsMute);
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::vSetServiceAvailable()
***************************************************************************/
t_Void spi_tclAudio::vSetServiceAvailable(t_Bool bAvail)
{
   ETG_TRACE_USR1(("spi_tclAudio::vSetServiceAvailable entered with availability = %u ",
         ETG_ENUM(BOOL, bAvail)));

   if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vSetServiceAvailable(bAvail);
   }
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::bSetAudioDucking()
***************************************************************************/
t_Bool spi_tclAudio::bSetAudioDucking(const t_U16 cou16RampDuration,const
      t_U8 cou8VolumeLevelindB, const tenDuckingType coenDuckingType,tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1(("spi_tclAudio::bSetAudioDucking entered"));
   t_Bool bRetVal = false;
   t_U8 u8SrcNum = (enDevCat == e8DEV_TYPE_MIRRORLINK)? m_u8AudSources[e8AUD_STEREO_MIX_OUT]:
      m_u8AudSources[e8AUD_DUCK]; 
   tenDuckingType enDuckingType = coenDuckingType ;
   t_U8 u8MainOutSrcNum = m_u8AudSources[e8AUD_MAIN_OUT];
   /*
    Duck: This is used in case of Carplay where application will set the duck value by XX db (ex: -41db). Navi volume context is loaded.
    DeviceDuck: This is used in case of AAP where AAP device will take care of ducking. Navi volume context is loaded.
    NativeDuck: This is used in case of AAP where native ducking is applied. Navi volume context is loaded.
    UnDuck: Unduck will remove Navi volume context and retain last entertainment volume if it was attenuated.
   */
   if((enDevCat != e8DEV_TYPE_DIPO) &&(enDuckingType != e8_DUCKINGTYPE_UNDUCK))
   {
      enDuckingType = (e8SRC_ACTIVE == m_oAudioSrcInfo.enGetSrcState(u8MainOutSrcNum)) ? e8_DUCKINGTYPE_DEVICEDUCK : e8_DUCKINGTYPE_NATIVEDUCK;
   }
   if(NULL != m_poAudioPolicyBase)
   {
      bRetVal = m_poAudioPolicyBase->bSetAudioDucking(u8SrcNum, cou16RampDuration,
                           cou8VolumeLevelindB, enDuckingType);
   }
   ETG_TRACE_USR4(("spi_tclAudio::bSetAudioDucking sent for: "
         "Ramp Duration = %u, VolumeLevel in dB = %u, Ducking Type = %u ",
         cou16RampDuration, cou8VolumeLevelindB, ETG_ENUM(AUDIO_DUCK_TYPE, enDuckingType)));
   return bRetVal;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vInitializeAudioIn()
***************************************************************************/
t_Void spi_tclAudio::vInitializeAudioIn(tenAudioDir enAudDir, tenAudioInDataSet enAudioDataSet)
{
   ETG_TRACE_USR1(("spi_tclAudio::vInitializeAudioIn entered for: "
         "Audio Direction = %u, Current AudioIn State = %u  ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(AUDIOIN_STATE, m_enAudioInState)));

   m_oAudioInLock.s16Lock();
   if (e32_AUDIOIN_DATASET_UNKNOWN == enAudioDataSet)
   {
      ETG_TRACE_ERR(("[ERR]::vInitializeAudioIn: Invalid AudioIn data set received "));
   }
   else if ((e8_AUDIOIN_UNITIAILZED == m_enAudioInState) && (NULL != m_poAudioInHandler))
   {
      m_poAudioInHandler->vInitializeAudioIn(enAudioDataSet);
      m_enAudioInState = e8_AUDIOIN_INITIALIZED;
   }
   m_oAudioInLock.vUnlock();
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vSetAlsaDevice()
***************************************************************************/
t_Void spi_tclAudio::vSetAlsaDevice(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclAudio::vSetAlsaDevice entered for: "
         "Audio Direction = %u, Current AudioIn State = %u  ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(AUDIOIN_STATE, m_enAudioInState)));

   m_oAudioInLock.s16Lock();
   if ((e8_AUDIOIN_UNITIAILZED != m_enAudioInState) && (NULL != m_poAudioInHandler))
   {
      t_String szALsaDevice = "";
      t_U8 u8SrcNum = m_u8AudSources[enAudDir];
      tenAudioSamplingRate enAudSampleRate = m_oAudioSrcInfo.enGetAudSamplingRate(u8SrcNum);
      //! Alsa Devices to be set to ECNR is different for GM and G3G projects
      if (e8AUD_PHONE_IN == enAudDir)
      {
         szALsaDevice = "AdevVoiceOut";
      }
      else if ((e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir))
      {
         szALsaDevice = "AdevAcousticoutSpeech";
      }
      else if(e8AUD_VR_STEREO == enAudDir)
      {
         szALsaDevice = "";
      }
      /*Set Audio Device changes.*/
      m_poAudioInHandler->vSetAlsaDevice(AUD_ECNR_MIC_OUT, gszECNRAudDeviceName[enAudDir][enAudSampleRate]);
      m_poAudioInHandler->vSetAlsaDevice(AUD_ECNR_RECV_IN, gszECNRAudDeviceName[enAudDir][enAudSampleRate]);
      if(e8AUD_VR_STEREO != enAudDir)
      {
         m_poAudioInHandler->vSetAlsaDevice(AUD_ECNR_RECV_OUT, szALsaDevice);
      }
      else
      {
         ETG_TRACE_USR4(("[DESC]:spi_tclAudio::vSetAlsaDevice set alsa device is not called for e8AUD_VR_STEREO"));
      }
      m_enAudioInState = e8_AUDIOIN_INITIALIZED;
   }
   m_oAudioInLock.vUnlock();
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vUninitializeAudioIn()
***************************************************************************/
t_Void spi_tclAudio::vUninitializeAudioIn(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclAudio::vUninitializeAudioIn entered: "
         "Audio Direction = %u, Current AudioIn State = %u  ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(AUDIOIN_STATE, m_enAudioInState)));

   //Call ECNR StopAudio()
   m_oAudioInLock.s16Lock();
   if ((e8_AUDIOIN_UNITIAILZED != m_enAudioInState) && (NULL != m_poAudioInHandler))
   {
      // Call ECNR Destroy and uninitialize
      m_poAudioInHandler->vUnitializeAudioIn();
      m_enAudioInState = e8_AUDIOIN_UNITIAILZED;
   }
   m_oAudioInLock.vUnlock();
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vStartAudioIn()
***************************************************************************/
t_Void spi_tclAudio::vStartAudioIn(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclAudio::vStartAudioIn entered: "
         "Audio Direction = %u, Current AudioIn State = %u  ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(AUDIOIN_STATE, m_enAudioInState)));

   //Call ECNR StartAudio()
   m_oAudioInLock.s16Lock();
   if ((e8_AUDIOIN_INITIALIZED == m_enAudioInState) && (NULL != m_poAudioInHandler))
   {
      m_poAudioInHandler->vStartAudioIn();
      m_enAudioInState = e8_AUDIOIN_STARTED;
   }
   m_oAudioInLock.vUnlock();
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vStopAudioIn()
***************************************************************************/
t_Void spi_tclAudio::vStopAudioIn(tenAudioDir enAudDir)
{
   ETG_TRACE_USR1(("spi_tclAudio::vStopAudioIn entered: "
         "Audio Direction = %u, Current AudioIn State = %u  ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(AUDIOIN_STATE, m_enAudioInState)));

   //Call ECNR StopAudio()
   m_oAudioInLock.s16Lock();
   if ((e8_AUDIOIN_STARTED == m_enAudioInState) && (NULL != m_poAudioInHandler))
   {
      m_poAudioInHandler->vStopAudioIn();
      m_enAudioInState = e8_AUDIOIN_INITIALIZED;
   }
   m_oAudioInLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudio::vSetAudioInConfig
 ***************************************************************************/
t_Void spi_tclAudio::vSetAudioInConfig(tenAudioDir enAudDir, tenAudioInDataSet enAudioDataSet)
{
   ETG_TRACE_USR1(("spi_tclAudio::vSetAudioInConfig entered: "
         "Audio Direction = %u, Current AudioIn State = %u  ",
         ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir), ETG_ENUM(AUDIOIN_STATE, m_enAudioInState)));

   m_oAudioInLock.s16Lock();
   //Call ECNR vSetAudioConfig()
   if (e32_AUDIOIN_DATASET_UNKNOWN == enAudioDataSet)
   {
      ETG_TRACE_ERR(("[ERR]::vSetAudioInConfig: Invalid AudioIn data set received "));
   }
   else if ((e8_AUDIOIN_INITIALIZED == m_enAudioInState) && (NULL != m_poAudioInHandler))
   {
      m_poAudioInHandler->vSetAudioInConfig(enAudioDataSet);
   }
   m_oAudioInLock.vUnlock();
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vRestoreLastMainAudSrc()
***************************************************************************/
t_Void spi_tclAudio::vRestoreLastMediaAudSrc()
{
   ETG_TRACE_USR1(("spi_tclAudio::vRestoreLastMainAudSrc entered "));

   if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vRestoreLastMediaAudSrc();
   }//End of if (NULL != m_poAudioPolicyBase)
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::bOnReqAVDeActivationResult()
***************************************************************************/
t_Bool spi_tclAudio::bOnReqAVDeActivationResult(const t_U8 cou8SourceNum)
{
   ETG_TRACE_USR1(("spi_tclAudio::bOnReqAVDeActivationResult entered: SourceNum = %u", cou8SourceNum));
   m_oAudioSrcInfo.vSetActiveSrcFlag(cou8SourceNum, e8SRC_INACTIVE);
   return true;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAudio::vOnAudioError()
***************************************************************************/
t_Void spi_tclAudio::vOnAudioError(const t_U8 cou8SourceNum, tenAudioError enAudioError)
{
   ETG_TRACE_USR1(("spi_tclAudio::vOnAudioError entered: SourceNum = %u, AudioError =%u ",
         cou8SourceNum, ETG_ENUM(AUDIO_ERROR,enAudioError)));

   t_U8 u8DevCategory = static_cast<t_U8> (m_oAudioSrcInfo.enGetDeviceCategory(cou8SourceNum));

   if ((NULL != m_poAudDevHandlers[u8DevCategory]))
   {
      tenAudioDir enAudDir = m_oAudioSrcInfo.enGetAudDirection(cou8SourceNum);
      m_poAudDevHandlers[u8DevCategory]->vOnAudioError(enAudDir, enAudioError);
   }//if ((NULL != m_poAudDevHandlers[u8DevCategory]))
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vSendAudioStatusChange(...)
 ***************************************************************************/
t_Void spi_tclAudio::vSendAudioStatusChange(tenAudioStatus enAudioStatus)
{
   //! Forward audio status to policy
   if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vSendAudioStatusChange(enAudioStatus);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAudio::vSetAppAudioInfo()
***************************************************************************/
t_Void spi_tclAudio::vSetAppAudioInfo(tenDeviceCategory enDevCat,
                                      const std::vector<trAppsAudioInfo>& corvecAppsAudioInfo)
{
   t_U8 u8DevCat = static_cast<t_U8>(enDevCat);

   if((u8DevCat < NUM_AUD_CLIENTS) && (NULL != m_poAudDevHandlers[u8DevCat]) )
   {
      m_poAudDevHandlers[u8DevCat]->vSetAppAudioInfo(corvecAppsAudioInfo);
   }//if(u8DevCat < NUM_AUD_CLIENTS) && (NULL != m_poAudDevHandlers[u8DevCat])

}

/***************************************************************************
** FUNCTION: t_Void spi_tclAudio::vPostVRSupportInfo()
***************************************************************************/
t_Void spi_tclAudio::vPostVRSupportInfo(t_U32 cou32DevId, t_U16 u16ValidityOfData,
         t_Bool bVoiceRecognitionSupportInfo)
{
   if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vPostVRSupportInfo(cou32DevId, u16ValidityOfData, bVoiceRecognitionSupportInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vSetAudioSettingsInstance()
***************************************************************************/
t_Void spi_tclAudio::vSetAudioSettingsInstance(spi_tclAudioSettingsIntf* poAudioSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclAudio::vSetAudioSettingsInstance entered"));
   if( NULL != poAudioSettingsIntf)
   {
      m_poAudioSettings = poAudioSettingsIntf;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vDisplayAudioSettings()
***************************************************************************/
t_Void spi_tclAudio::vDisplayAudioSettings()
{
   ETG_TRACE_USR1(("spi_tclAudio::vDisplayAudioSettings entered"));
   if( NULL != m_poAudioSettings)
   {
      m_poAudioSettings->vDisplayAudioSettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vGetAudioConfig()
***************************************************************************/
t_Void spi_tclAudio::vGetAudioConfig()
{
   ETG_TRACE_USR1(("spi_tclAudio::vGetAudioConfig entered"));
   if( NULL != m_poAudioSettings)
   {
      m_poAudioSettings->bGetRTPInSupport();
      m_poAudioSettings->szGetDisableRealTimePriorityAudio();
      m_poAudioSettings->szGetAudThreadsRealTimePriority();
      m_poAudioSettings->szGetAudSinkThresholdMs();
      m_poAudioSettings->bGetVolumeLockEnabled();
      m_poAudioSettings->bGetMLDynAudioApplicability();
      m_poAudioSettings->bGetAudioMuteEnabled();

   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAudio::vPostRelativelyActiveMediaAppStatus()
***************************************************************************/
t_Void spi_tclAudio::vPostRelativelyActiveMediaAppStatus(t_U32 u32RelativelyActiveMediaAppID,tenRelativelyActiveMediaAppCat enRelativelyActiveMediaAppCat)
{
   if (NULL != m_poAudioPolicyBase)
   {
      m_poAudioPolicyBase->vPostRelativelyActiveMediaAppStatus(u32RelativelyActiveMediaAppID, enRelativelyActiveMediaAppCat);
   }
}

//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
