/***********************************************************************/
/*!
* \file  spi_tclAppMngr.cpp
* \brief Main Application Manager class that provides interface to delegate 
*        the execution of command and handle response
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         Shiva Kumar Gurija
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
16.02.2014  | Shiva Kumar Gurija    | Initial Version
20.03.2014  | Shihabudheen P M      | Added vCbPostDeviceDisplayContext()
23.04.2014  | Shiva Kumar Gurija    | Updated with Notifications Impl
06.04.2014  | Ramya Murthy          | Initialisation sequence implementation
31.07.2014  | Ramya Murthy          | SPI feature configuration via LoadSettings()
25.06.2015  | Tejaswini HB          | Featuring out Android Auto
26.02.2016  | Rachana L Achar       | AAP Navigation implementation
10.03.2016  | Rachana L Achar       | AAP Notification implementation
05.05.2016  | Ramya Murthy          | Reverted usage of vOnVehicleData() to determine Drive mode.
                                      Only interface vSetVehicleConfig() will be used instead (fix for GMMY17-14526).
01.03.2017  |  Shiva Kumar G        | Changes to enable/disable CarPlay feature    
\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclRespInterface.h"
#include "spi_tclAppMngrRespIntf.h"
#include "spi_tclAppMngrDev.h"
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
#include "spi_tclDiPoAppMngr.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO	
#include "spi_tclAAPAppMngr.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINAppMngr.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLAppMngr.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarAppMngr.h"
#endif

#include "spi_tclAppMngr.h"
#include "spi_tclMediator.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_APPMNGR
#include "trcGenProj/Header/spi_tclAppMngr.cpp.trc.h"
#endif
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenDeviceCategory,GetAppmngr_t*> Appmngr;
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/***************************************************************************
** FUNCTION:  spi_tclAppMngr::spi_tclAppMngr()
***************************************************************************/
spi_tclAppMngr::spi_tclAppMngr(spi_tclAppMngrRespIntf* poRespIntf): spi_tclSelectionIntf(e32COMPID_APPMANAGER),
   m_poAppMngrRespIntf(poRespIntf), m_u32SelectedDevice(0),
   m_poAppSettings(NULL)
{
   ETG_TRACE_USR1((" spi_tclAppMngr::spi_tclAppMngr"));

   for (t_U8 u8Index = 0; u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
   {
      m_poAppMngrDevBase[u8Index] = NULL;
   }//for(t_U8 u8Index=0; u8Index
}

/***************************************************************************
** FUNCTION:  spi_tclAppMngr::~spi_tclAppMngr()
***************************************************************************/
spi_tclAppMngr::~spi_tclAppMngr()
{
   ETG_TRACE_USR1((" spi_tclAppMngr::~spi_tclAppMngr"));

   m_mapDevConnType.clear();
   m_poAppMngrRespIntf = NULL;
   m_poAppSettings = NULL;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAppMngr::bInitialize()
***************************************************************************/
t_Bool spi_tclAppMngr::bInitialize()
{
#ifdef VARIANT_S_FTR_ENABLE_SPI_DIPO
   m_poAppMngrDevBase[e8DEV_TYPE_DIPO] = new spi_tclDiPoAppMngr();
   SPI_NORMAL_ASSERT(NULL == m_poAppMngrDevBase[e8DEV_TYPE_DIPO]);
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
   ETG_TRACE_USR1(("spi_tclAppMngr::bInitialize Appmngr[e8DEV_TYPE_MIRRORLINK]=%p",Appmngr[e8DEV_TYPE_MIRRORLINK]));
   if ((NULL != Appmngr[e8DEV_TYPE_MIRRORLINK]))
   {
      ETG_TRACE_USR1((" spi_tclAppMngr::bInitialize before obj create m_poAppMngrDevBase[e8DEV_TYPE_MIRRORLINK] = %p", m_poAppMngrDevBase[e8DEV_TYPE_MIRRORLINK]));
      m_poAppMngrDevBase[e8DEV_TYPE_MIRRORLINK] = Appmngr[e8DEV_TYPE_MIRRORLINK]();
      SPI_NORMAL_ASSERT(NULL == m_poAppMngrDevBase[e8DEV_TYPE_MIRRORLINK]);
   }
   ETG_TRACE_USR1((" spi_tclAppMngr::bInitialize obj success m_poAppMngrDevBase[e8DEV_TYPE_MIRRORLINK] = %p", m_poAppMngrDevBase[e8DEV_TYPE_MIRRORLINK]));
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO	
   m_poAppMngrDevBase[e8DEV_TYPE_ANDROIDAUTO] = new spi_tclAAPAppMngr();
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
   m_poAppMngrDevBase[e8DEV_TYPE_MYSPIN] = new (std::nothrow) spi_tclMySPINAppMngr();
   SPI_NORMAL_ASSERT(NULL == m_poAppMngrDevBase[e8DEV_TYPE_MYSPIN]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
  m_poAppMngrDevBase[e8DEV_TYPE_CARLIFE] = new spi_tclBDCLAppMngr();
  SPI_NORMAL_ASSERT(NULL == m_poAppMngrDevBase[e8DEV_TYPE_CARLIFE]);
#endif

#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
   m_poAppMngrDevBase[e8DEV_TYPE_ONCAR] = new (std::nothrow) spi_tclOnCarAppMngr();
   SPI_NORMAL_ASSERT(NULL == m_poAppMngrDevBase[e8DEV_TYPE_ONCAR]);
#endif

   t_Bool bInit = false;
   for(t_U8 u8Index=0; u8Index<NUM_APPMNGR_CLIENTS;u8Index++)
   {
      if(NULL != m_poAppMngrDevBase[u8Index])
      {
         bInit = (m_poAppMngrDevBase[u8Index]->bInitialize()) || bInit;
      }
   }

   //Register for callbacks
   vRegisterCallbacks();

   ETG_TRACE_USR1((" spi_tclAppMngr::bInitialize bInit = %d", ETG_ENUM(BOOL, bInit)));
   return bInit;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAppMngr::bUnInitialize()
***************************************************************************/
t_Bool spi_tclAppMngr::bUnInitialize()
{
   ETG_TRACE_USR1((" spi_tclAppMngr::vUnInitialize() \n"));

   for (t_U8 u8Index = 0; u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
   {
      if (NULL != m_poAppMngrDevBase[u8Index])
      {
         m_poAppMngrDevBase[u8Index]->vUnInitialize();
      }//if((NULL != m_poVideoDevBase[
      RELEASE_MEM(m_poAppMngrDevBase[u8Index]);
   }//for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)

   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vLoadSettings()
***************************************************************************/
t_Void spi_tclAppMngr::vLoadSettings()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   //Read the settings from the XML
   if (NULL != m_poAppSettings)
   {
      tenRegion enRegion = m_poAppSettings->enGetRegion();
      m_poAppSettings->vSetRegion(enRegion);
      m_poAppSettings->vReadSysConfigs();
      ETG_TRACE_USR1(("spi_tclAppMngr::vLoadSettings Populating the structure!"));
      trConfigData rConfigData;
      rConfigData.bAAPNavigationEnable = m_poAppSettings->bGetAAPNavigationEnabledInfo();//bEnableAAPNavigation
      rConfigData.bAAPNotificationEnable = m_poAppSettings->bGetAAPNotificationUpdatesEnabledInfo();//bEnableAAPNotification
      rConfigData.bAAPMediaPlaybackEnable = m_poAppSettings->bGetAAPMediaPlaybackEnabledInfo();//bEnableAAPMediaPlayback
      rConfigData.bTBTUpdateEnabled = m_poAppSettings->bGetTBTUpdatesEnabledInfo();
      rConfigData.bTBTAssisstantGuideEnabled = m_poAppSettings->bGetAssisstantGuideUpdatesEnabledInfo();
      for (t_U8 u8Index = 0; u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
      {
         if (NULL != m_poAppMngrDevBase[u8Index])
            {
               m_poAppMngrDevBase[u8Index]->vSetAppSettingsInstance(m_poAppSettings);
               m_poAppMngrDevBase[u8Index]->vSetConfigData(rConfigData);
            }
      }//app settings instance and call the functions here//if(NULL != m_poAppSettings)
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSaveSettings()
***************************************************************************/
t_Void spi_tclAppMngr::vSaveSettings()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   //Add code
}

/***************************************************************************
** FUNCTION:  t_Bool  spi_tclAppMngr::bValidateClient()
***************************************************************************/
inline t_Bool spi_tclAppMngr::bValidateClient(t_U8 u8Index)
{
   //Assert if the Index is greater than the array size
   SPI_NORMAL_ASSERT( u8Index > NUM_APPMNGR_CLIENTS);

   t_Bool bRet = ( u8Index < NUM_APPMNGR_CLIENTS) && 
      (NULL != m_poAppMngrDevBase[u8Index]);

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void  spi_tclAppMngr::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclAppMngr::vRegisterCallbacks()
{
	/*lint -esym(40,fpvNotifyAppListChangeCb) fpvNotifyAppListChangeCb Undeclared identifier */
	/*lint -esym(40,fpvTerminateAppResult) fpvTerminateAppResult Undeclared identifier */
	/*lint -esym(40,fpvLaunchAppResult) fpvLaunchAppResult Undeclared identifier */
	/*lint -esym(40,fpvCbAppIconDataResult) fpvCbAppIconDataResult Undeclared identifier */
	/*lint -esym(40,fpvSelectDeviceResult) fpvSelectDeviceResult Undeclared identifier */
	/*lint -esym(40,fpvPostDeviceDisplayContext) fpvPostDeviceDisplayContext Undeclared identifier */
	/*lint -esym(40,fpvEnableNotiResult) fpvEnableNotiResult Undeclared identifier */
	/*lint -esym(40,fpvInvokeNotiActionResult) fpvInvokeNotiActionResult Undeclared identifier */
	/*lint -esym(40,fpvNotifyNoticationInfo) fpvNotifyNoticationInfo Undeclared identifier */
	/*lint -esym(40,fpvNotifySessionStatus) fpvNotifySessionStatus Undeclared identifier */
	/*lint -esym(40,fpvNotifyActiveAppInfo) fpvEnableNotiResult Undeclared identifier */
	/*lint -esym(40,_1) _1 Undeclared identifier */
	/*lint -esym(40,_2) _2 Undeclared identifier */
	/*lint -esym(40,_3) _3 Undeclared identifier */
	/*lint -esym(40,_4) _4 Undeclared identifier */
	/*lint -esym(40,_5) _5 Undeclared identifier */


   ETG_TRACE_USR1(("spi_tclAppMngr:vRegisterCallbacks()"));

   trAppMngrCallbacks rAppMngrCbs;

   rAppMngrCbs.fpvNotifyAppListChangeCb = std::bind(
      &spi_tclAppMngr::vCbNotifyAppListChange, 
      this, 
      SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fpvTerminateAppResult = std::bind(
      &spi_tclAppMngr::vCbTerminateAppResult,
      this,
      SPI_FUNC_PLACEHOLDERS_4);

   rAppMngrCbs.fpvLaunchAppResult = std::bind(
      &spi_tclAppMngr::vCbLaunchAppResult,
      this,
      SPI_FUNC_PLACEHOLDERS_5);

   rAppMngrCbs.fpvCbAppIconDataResult = std::bind(
      &spi_tclAppMngr::vCbAppIconDataResult,
      this,
      SPI_FUNC_PLACEHOLDERS_4);

   rAppMngrCbs.fpvSelectDeviceResult = std::bind(
      &spi_tclAppMngr::vCbSelectDeviceResult, 
      this,SPI_FUNC_PLACEHOLDERS_1);

   rAppMngrCbs.fpvPostDeviceDisplayContext = std::bind(
      &spi_tclAppMngr::vCbPostDeviceDisplayContext,
      this, SPI_FUNC_PLACEHOLDERS_4);

   rAppMngrCbs.fpvEnableNotiResult = std::bind(
      &spi_tclAppMngr::vCbPostSetNotiEnabledInfoResult,
      this, SPI_FUNC_PLACEHOLDERS_3);

   rAppMngrCbs.fpvInvokeNotiActionResult = std::bind(
      &spi_tclAppMngr::vCbPostInvokeNotificationActionResult,
      this, SPI_FUNC_PLACEHOLDERS_3);

   rAppMngrCbs.fpvNotifyNoticationInfo = std::bind(
      &spi_tclAppMngr::vCbPostNotificationResult,
      this, SPI_FUNC_PLACEHOLDERS_2);


   rAppMngrCbs.fpvNotifySessionStatus = std::bind(
      &spi_tclAppMngr::vCbSessionStatusUpdate,this,
      SPI_FUNC_PLACEHOLDERS_3);

   rAppMngrCbs.fpvNotifyActiveAppInfo= std::bind(
      &spi_tclAppMngr::vCbActiveAppInfo,this,
      SPI_FUNC_PLACEHOLDERS_4);

   rAppMngrCbs.fvUpdateNavStatusData =
         std::bind(&spi_tclAppMngr::vSendNavigationStatus,
         this,
         SPI_FUNC_PLACEHOLDERS_1);

   rAppMngrCbs.fvUpdateNavNextTurnData =
         std::bind(&spi_tclAppMngr::vSendNavigationNextTurnData,
         this,
         SPI_FUNC_PLACEHOLDERS_1);

   rAppMngrCbs.fvUpdateNavNextTurnDistanceData =
         std::bind(&spi_tclAppMngr::vSendNavigationNextTurnDistanceData,
         this,
         SPI_FUNC_PLACEHOLDERS_1);

   rAppMngrCbs.fvUpdateNotificationData =
         std::bind(&spi_tclAppMngr::vSendNotificationData,
         this,
         SPI_FUNC_PLACEHOLDERS_1);


   rAppMngrCbs.fvAppMediaMetaData =
            std::bind(&spi_tclAppMngr::vApplicationMediaMetaDataCb,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fvAppMediaPlaytime =
            std::bind(&spi_tclAppMngr::vApplicationMediaPlaytimeCb,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fvAppPhoneData =
            std::bind(&spi_tclAppMngr::vApplicationPhoneDataCb,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fvAppPhoneCallData =
            std::bind(&spi_tclAppMngr::vApplicationPhoneCallDataCb,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fvAppMediaAlbumArt =
            std::bind(&spi_tclAppMngr::vAppMediaAlbumArtCb,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fvUpdateTurnByTurnInfo =
            std::bind(&spi_tclAppMngr::vTurnByTurnInfoCb,
                      this,
                      SPI_FUNC_PLACEHOLDERS_1);

   rAppMngrCbs.fvUpdateCarPlayRouteGuidanceUpdate =
            std::bind(&spi_tclAppMngr::vCarPlayRouteGuidanceUpdateCb,
                      this,
                      SPI_FUNC_PLACEHOLDERS_1);

   rAppMngrCbs.fvUpdateCarPlayRouteGuidanceManeuverUpdate =
            std::bind(&spi_tclAppMngr::vCarPlayRouteGuidanceManeuverUpdateCb,
                      this,
                      SPI_FUNC_PLACEHOLDERS_1);
  
   rAppMngrCbs.fvUpdateVendorExtsnData =
           std::bind(&spi_tclAppMngr::vCbPostVendorExtsnData,
                    this, 
                    SPI_FUNC_PLACEHOLDERS_2);

   rAppMngrCbs.fvSelectionProgressState =
            std::bind(&spi_tclAppMngr::vSendSelectionProgressState,
                     this,
                     SPI_FUNC_PLACEHOLDERS_2);
					 
    rAppMngrCbs.fvPostDeviceLockInfo =
            std::bind(&spi_tclAppMngr::vPostDeviceLockInfoCb,
                     this,
                     SPI_FUNC_PLACEHOLDERS_3);

   for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
   {
      if(NULL != m_poAppMngrDevBase[u8Index])
      {
         m_poAppMngrDevBase[u8Index]->vRegisterAppMngrCallbacks(rAppMngrCbs);
      }//if((NULL != m_poAppMngrDevBase[
   }//for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSelectDevice()
***************************************************************************/
t_Void spi_tclAppMngr::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{

   t_U8 u8Index = static_cast<t_U8>(corfrSelectReq.m_enDevCategory);

   ETG_TRACE_USR1(("spi_tclAppMngr::vSelectDevice:Dev-0x%x,enDevCat-%d,coenConnReq-%d",
            corfrSelectReq.m_u32DeviceHandle,u8Index,corfrSelectReq.m_enDevConnReq));

   //set the selected device. this is required to inform AppLauncher 
   //to send error response to HMI, if the Launch/terminate request has come for 
   //non selected device.
   m_u32SelectedDevice = (corfrSelectReq.m_enDevConnReq == e8DEVCONNREQ_SELECT)?corfrSelectReq.m_u32DeviceHandle:0;

   //Store the device connection type in the internally.
   //It is required while posting the App list change update to HMI
   m_oLock.s16Lock();
   m_mapDevConnType.insert(std::pair<t_U32,tenDeviceConnectionType>(corfrSelectReq.m_u32DeviceHandle,
            corfrSelectReq.m_enDevConnType));
   m_oLock.vUnlock();

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vSelectDevice(corfrSelectReq);
   }//if(true == bValidateClient(u8Index))

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vLaunchApp()
***************************************************************************/
t_Void spi_tclAppMngr::vLaunchApp(const t_U32 cou32DevId, 
                                  tenDeviceCategory enDevCat, 
                                  t_U32 u32AppHandle, 
                                  tenDiPOAppType enDiPOAppType, 
                                  t_String szTelephoneNumber, 
                                  tenEcnrSetting enEcnrSetting,
                                  const trUserContext& rfrcUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vLaunchApp:Dev-0x%x,App-0x%x,enDevCat-%d enDiPOAppType-%d",
      cou32DevId,u32AppHandle,ETG_ENUM(DEVICE_CATEGORY,enDevCat),enDiPOAppType));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vLaunchApp(cou32DevId,
         u32AppHandle,
         rfrcUsrCntxt,
         enDiPOAppType,
         szTelephoneNumber,
         enEcnrSetting);
   } //if(true == bValidateClient(u8Index))
   else
   {
      //invalid device handle - post error
      vCbLaunchAppResult(cou32DevId,u32AppHandle,enDiPOAppType,e8INVALID_DEV_HANDLE,
         rfrcUsrCntxt);
   }

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vTerminateApp()
***************************************************************************/
t_Void spi_tclAppMngr::vTerminateApp(const t_U32 cou32DevId,
                                     const t_U32 cou32AppId,
                                     tenDeviceCategory enDevCat,
                                     const trUserContext& rfrcUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vTerminateApp:Dev-0x%x,App-0x%x,enDevCat-%d",
      cou32DevId,cou32AppId,u8Index));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vTerminateApp(cou32DevId,cou32AppId,rfrcUsrCntxt);
   } //if(true == bValidateClient(u8Index))
   else
   {
      //invalid device handle - post error
      vCbTerminateAppResult(cou32DevId,cou32AppId,e8INVALID_DEV_HANDLE,rfrcUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vGetAppDetailsList()
***************************************************************************/
t_Void spi_tclAppMngr::vGetAppDetailsList(const t_U32 cou32DevId,
                                          t_U32& u32NumAppInfoList,
                                          std::vector<trAppDetails>& corfvecrAppDetailsList,
                                          tenDeviceCategory enDevCat)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vGetAppDetailsList:Dev-0x%x, enDevCat-%d",
      cou32DevId,u8Index));

   if( (m_u32SelectedDevice == cou32DevId) &&(true == bValidateClient(u8Index)) )
   {
      m_poAppMngrDevBase[u8Index]->vGetAppDetailsList(cou32DevId,
         u32NumAppInfoList,
         corfvecrAppDetailsList);
   } //if(true == bValidateClient(u8Index))

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vGetAppInfo()
***************************************************************************/
t_Void spi_tclAppMngr::vGetAppInfo(const t_U32 cou32DevId, 
                                   const t_U32 cou32AppId,
                                   tenDeviceCategory enDevCat,
                                   trAppDetails& rfrAppDetails)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vGetAppInfo: Dev-0x%x App-0x%x enDevCat-%d ",
      cou32DevId,cou32AppId,u8Index));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vGetAppInfo(cou32DevId,cou32AppId,
         rfrAppDetails);
   } //if(true == bValidateClient(u8Index))

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vDisplayAllAppsInfo()
***************************************************************************/
t_Void spi_tclAppMngr::vDisplayAllAppsInfo(const t_U32 cou32DevId,
                                           tenDeviceCategory enDevCat)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vDisplayAllAppsInfo: Dev-0x%x enDevCat-%d ",
      cou32DevId,u8Index));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vDisplayAllAppsInfo(cou32DevId);
   } //if(true == bValidateClient(u8Index))
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vGetAppIconData()
***************************************************************************/
t_Void spi_tclAppMngr::vGetAppIconData(t_String szAppIconUrl,
                                       tenDeviceCategory enDevCat,
                                       const trUserContext& rfrcUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vGetAppIconData:u8Index-%d IconURL-%s",
      u8Index,szAppIconUrl.c_str()));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vGetAppIconData(szAppIconUrl,rfrcUsrCntxt);
   } //if(true == bValidateClient(u8Index))
   else
   {
      //invalid handle - Post error
      vCbAppIconDataResult(e8ICON_INVALID,NULL,0,rfrcUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vSetAppIconAttributes()
***************************************************************************/
t_Void spi_tclAppMngr::vSetAppIconAttributes(t_U32 u32DevId,
                                             t_U32 u32AppId,
                                             tenDeviceCategory enDevCat,
                                             const trIconAttributes &rfrIconAttributes,
                                             const trUserContext& rfrcUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);
   ETG_TRACE_USR1(("spi_tclAppMngr::vSetAppIconAttributes:Dev-0x%x,App-0x%x,u8Index-%d",
      u32DevId,u32AppId,u8Index));

   tenResponseCode enRespCode = e8FAILURE;
   tenErrorCode enErrCode = e8UNSUPPORTED_OPERATION;

   if(true == bValidateClient(u8Index))
   {
      if(true == m_poAppMngrDevBase[u8Index]->bSetAppIconAttributes(u32DevId,
         u32AppId,rfrIconAttributes))
      {
         enRespCode = e8SUCCESS;
         enErrCode = e8NO_ERRORS;
      }//if(true == m_poAppMngrDevBase[u8Index]->bSetAppIconAttributes(u32DevId,
   } //if(true == bValidateClient(u8Index))
   else
   {
      enErrCode = e8INVALID_DEV_HANDLE;
   }

   if(NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostSetAppIconAttributesResult(enRespCode,enErrCode,
         rfrcUsrCntxt);
   }//if(NULL != m_poAppMngrRespIntf)

}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vSetVehicleConfig()
***************************************************************************/
t_Void spi_tclAppMngr::vSetVehicleConfig(tenVehicleConfiguration enVehicleConfig,
      t_Bool bSetConfig)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSetVehicleConfig:vSetVehicleConfig-0x%x",
      enVehicleConfig));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();

   if(NULL != poMediator)
   {
      if((e8_DAY_MODE == enVehicleConfig)||(e8_NIGHT_MODE == enVehicleConfig))
      {
         poMediator->vPostDayNightMode(enVehicleConfig);
      }
   }
   if(NULL != m_poAppSettings)
   {
	  if((e8_RIGHT_HAND_DRIVE == enVehicleConfig)||(e8_LEFT_HAND_DRIVE == enVehicleConfig))
	  {
	     m_poAppSettings->vSetDriveSideInfo(enVehicleConfig);
	  }
   }

   for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
   {
      if(NULL != m_poAppMngrDevBase[u8Index])
      {
         m_poAppMngrDevBase[u8Index]->vSetVehicleConfig(enVehicleConfig, bSetConfig);
      }//if((NULL != m_poAppMngrDevBase[
   }//for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
}

/***************************************************************************
** FUNCTION:  t_U32 spi_tclAppMngr::u32GetSelectedDevice()()
***************************************************************************/
t_U32 spi_tclAppMngr::u32GetSelectedDevice()
{
	ETG_TRACE_USR1(("spi_tclAppMngr::u32GetSelectedDevice-0x%x",m_u32SelectedDevice));
   return m_u32SelectedDevice;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAppMngr::bCheckAppValidity()
***************************************************************************/
t_Bool spi_tclAppMngr::bCheckAppValidity(const t_U32 cou32DevId, 
                                         const t_U32 cou32AppId,
                                         tenDeviceCategory enDevCat)
{
   t_Bool bRet=false;

   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::bCheckAppValidity:Dev-0x%x,App-0x%x,enDevCat-%d",
      cou32DevId,cou32AppId,ETG_ENUM(DEVICE_CATEGORY,enDevCat)));

   if(true == bValidateClient(u8Index))
   {
      bRet = m_poAppMngrDevBase[u8Index]->bCheckAppValidity(cou32DevId,
         cou32AppId);
   } //if(true == bValidateClient(u8Index))

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vDisplayAppcertificationInfo()
***************************************************************************/
t_Void spi_tclAppMngr::vDisplayAppcertificationInfo(const t_U32 cou32DevId,
                                                    const t_U32 cou32AppId,
                                                    tenDeviceCategory enDevCat)
{
  t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::bCheckAppValidity:Dev-0x%x,App-0x%x,u8Index-%d",
      cou32DevId,cou32AppId,u8Index));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vDisplayAppcertificationInfo(cou32DevId,
         cou32AppId);
   } //if(true == bValidateClient(u8Index))
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAppMngr::bSetClientProfile()
***************************************************************************/
t_Bool spi_tclAppMngr::bSetClientProfile(const t_U32 cou32DevId,
                                         const trClientProfile& corfrClientProfile,
                                         tenDeviceCategory enDevCat)
{
   t_Bool bRet = false;
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::bSetClientProfile:Dev-0x%x u8Index-%d",
      cou32DevId,u8Index));

   if(true == bValidateClient(u8Index))
   {
      bRet = m_poAppMngrDevBase[u8Index]->bSetClientProfile(cou32DevId,
         corfrClientProfile);
   } //if(true == bValidateClient(u8Index))

   return bRet;

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSetVehicleBTAddress()
***************************************************************************/
t_Void spi_tclAppMngr::vSetVehicleBTAddress(const t_U32 cou32DevId,
                                            t_String szBTAddress,
                                            tenDeviceCategory enDevCat,
                                            const trUserContext& corfrUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   ETG_TRACE_USR1(("spi_tclAppMngr::vSetVehicleBTAddress:Dev-0x%x,u8Index-%d",
      cou32DevId,u8Index));

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vSetVehicleBTAddress(cou32DevId,
         szBTAddress.c_str(),corfrUsrCntxt);
   } //if(true == bValidateClient(u8Index))
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vSetMLNotificationEnabledInfo()
***************************************************************************/
t_Void spi_tclAppMngr::vSetMLNotificationEnabledInfo(const t_U32 cou32DevId,
                                                     t_U16 u16NumNotiEnableList,
                                                     std::vector<trNotiEnable> vecrNotiEnableList,
                                                     tenDeviceCategory enDevCat,
                                                     const trUserContext& corfrUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);
   ETG_TRACE_USR1(("spi_tclAppMngr::vSetMLNotificationEnabledInfo:Dev-0x%x,Num.Apps-%d,u8Index-%d",
      cou32DevId,u16NumNotiEnableList,u8Index));

   //this method is meant for only MirrorLink Notifications.
   //If any request comes for DiPo Device, send error response.
   if((e8DEV_TYPE_MIRRORLINK == u8Index) && (NULL != m_poAppMngrDevBase[u8Index]))
   {
      m_poAppMngrDevBase[u8Index]->vSetMLNotificationEnabledInfo(cou32DevId,u16NumNotiEnableList,
         vecrNotiEnableList,corfrUsrCntxt);
   } //if(e8DEV_TYPE_MIRRORLINK == u8Index)
   else
   {
      //Return Error
      vCbPostSetNotiEnabledInfoResult(cou32DevId,e8INVALID_DEV_HANDLE,corfrUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vInvokeNotificationAction()
***************************************************************************/
t_Void spi_tclAppMngr::vInvokeNotificationAction(t_U32 cou32DevId,
                                                 t_U32 u32NotificationID,
                                                 t_U32 u32NotificationActionID,
                                                 tenDeviceCategory enDevCat,
                                                 const trUserContext& corfrUsrCntxt)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);
   ETG_TRACE_USR1(("spi_tclAppMngr::vInvokeNotificationAction:Dev-0x%x,NoitificatioID-%d,u8Index-%d",
      cou32DevId,u32NotificationID,u8Index));
   if((e8DEV_TYPE_MIRRORLINK == u8Index) && (NULL != m_poAppMngrDevBase[u8Index]))
   {
      m_poAppMngrDevBase[u8Index]->vInvokeNotificationAction(cou32DevId,u32NotificationID,
         u32NotificationActionID,corfrUsrCntxt);
   } //if(e8DEV_TYPE_MIRRORLINK == u8Index)
   else
   {
      //Return Error
      vCbPostInvokeNotificationActionResult(cou32DevId,e8INVALID_DEV_HANDLE,corfrUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vSetMLNotificationOnOff()
***************************************************************************/
t_Void spi_tclAppMngr::vSetMLNotificationOnOff(t_Bool bSetNotificationsOn)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSetMLNotificationOnOff:bSetNotificationsOn-%d",
      ETG_ENUM(BOOL,bSetNotificationsOn)));

   //Set this setting in the Settings class and inform ML & DiPo App Mngrs
   if(NULL != m_poAppSettings)
   {
      m_poAppSettings->vSetMLNotificationOnOff(bSetNotificationsOn);
   }//if(NULL != m_poAppSettings)

   for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
   {
      if(NULL != m_poAppMngrDevBase[u8Index])
      {
         m_poAppMngrDevBase[u8Index]->vSetMLNotificationOnOff(bSetNotificationsOn);
      }//if((NULL != m_poAppMngrDevBase[
   }//for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)

}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAppMngr::bGetMLNotificationEnabledInfo()
***************************************************************************/
t_Bool spi_tclAppMngr::bGetMLNotificationEnabledInfo()
{
   t_Bool bRet = false;

   ETG_TRACE_USR1(("spi_tclAppMngr::bGetMLNotificationEnabledInfo()"));

   if(NULL != m_poAppSettings)
   {
      bRet = m_poAppSettings->bGetMLNotificationEnabledInfo();
   }//if(NULL != m_poAppSettings)

   return bRet;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vSetRegion(.)
***************************************************************************/
t_Void spi_tclAppMngr::vSetRegion(tenRegion enRegion)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSetRegion-%d",ETG_ENUM(REGION_TYPE,enRegion)));

   if(NULL != m_poAppSettings)
   {
      m_poAppSettings->vSetRegion(enRegion);
   }//if(NULL != m_poAppSettings)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vDisplayAppListXml(.)
***************************************************************************/
t_Void spi_tclAppMngr::vDisplayAppListXml(const t_U32 cou32DevId,
                                          tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vDisplayAppListXml: Dev-0x%x ",
      cou32DevId));

   t_U8 u8Index = static_cast<t_U8>(enDevCat);
   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vDisplayAppListXml(cou32DevId);
   } //if(true == bValidateClient(u8Index))
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vCbNotifyAppListChange()
***************************************************************************/
t_Void spi_tclAppMngr::vCbNotifyAppListChange(const t_U32 cou32DevId,
                                              tenAppStatusInfo enAppStatus)
{
	
    /*lint -esym(40,second) second Undeclared identifier */
    ETG_TRACE_USR1(("spi_tclAppMngr::vCbNotifyAppListChange:Dev-0x%x,enAppStatus-%d",
      cou32DevId,enAppStatus));
   tenDeviceConnectionType enDevConnType=e8UNKNOWN_CONNECTION;

   m_oLock.s16Lock();
   if(SPI_MAP_NOT_EMPTY(m_mapDevConnType))
   {
      auto itConnType = m_mapDevConnType.end();
      itConnType = m_mapDevConnType.find(cou32DevId);
      if(itConnType != m_mapDevConnType.end())
      {
         enDevConnType = itConnType->second;
      }//if(itConnType != m_mapDevConnType.end())
   }//if(SPI_MAP_NOT_EMPTY(m_mapDevConnType))
   m_oLock.vUnlock();

   if(NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostAppStatusInfo(cou32DevId,enDevConnType,
         enAppStatus);
   }//if(NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vCbAppIconDataResult()
***************************************************************************/
t_Void spi_tclAppMngr::vCbAppIconDataResult(tenIconMimeType enMimeType,
                                            const t_U8* pcu8AppIconData, 
                                            t_U32 u32Len,
                                            const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbAppIconDataResult:enMimeType-%d",
      ETG_ENUM(ICON_MIME_TYPE,enMimeType)));

   if(NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostAppIconDataResult(enMimeType,pcu8AppIconData,
         u32Len,rfrcUsrCntxt);
   }//if(NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vCbTerminateAppResult()
***************************************************************************/
t_Void spi_tclAppMngr::vCbTerminateAppResult(const t_U32 cou32DevId,
                                             const t_U32 cou32AppId,
                                             tenErrorCode enErrorCode,
                                             const trUserContext& rfrCUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbTerminateAppResult:Dev-0x%x App-0x%x",
      cou32DevId,cou32AppId));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostTerminateAppResult(e32COMPID_APPMANAGER,cou32DevId,cou32AppId,
         enErrorCode,rfrCUsrCntxt);
   } //if(NULL != poMediator)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbLaunchAppResult
***************************************************************************/
t_Void spi_tclAppMngr::vCbLaunchAppResult(
   const t_U32 cou32DevId,
   const t_U32 cou32AppId,
   const tenDiPOAppType coenDiPoAppType,
   tenErrorCode enErrorCode,  
   const trUserContext& rfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbLaunchAppResult:Dev-0x%x App-0x%x",
      cou32DevId,cou32AppId));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostLaunchAppResult(e32COMPID_APPMANAGER,cou32DevId,cou32AppId,
         coenDiPoAppType,enErrorCode,rfrcUsrCntxt);
   } //if(NULL != poMediator)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vCbSelectDeviceResult()
***************************************************************************/
t_Void spi_tclAppMngr::vCbSelectDeviceResult(t_Bool bResult)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbSelectDeviceResult:bResult-%d",
      bResult));
   tenErrorCode enErrorCode = (true==bResult)? e8NO_ERRORS : e8SELECTION_FAILED;
   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   if(NULL != poMediator)
   {
      poMediator->vPostSelectDeviceRes(e32COMPID_APPMANAGER, enErrorCode);
   }//if(NULL != poMediator)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSelectDeviceResult()
***************************************************************************/
t_Void spi_tclAppMngr::vSelectDeviceResult(const trSelectDeviceRequest& corfrSelectReq, tenErrorCode enErrorCode)
{
   t_U8 u8Index = static_cast<t_U8>(corfrSelectReq.m_enDevCategory);

   ETG_TRACE_USR1(("spi_tclAppMngr::vOnSelectDeviceResult:Dev-0x%x,coenConnReq-%d ErrorCode-%d u8Index-%d",
            corfrSelectReq.m_u32DeviceHandle,corfrSelectReq.m_enDevConnReq,enErrorCode,u8Index));

   tenResponseCode enRespCode =(e8NO_ERRORS == enErrorCode)?e8SUCCESS: e8FAILURE ;
   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vOnSelectDeviceResult(corfrSelectReq.m_u32DeviceHandle,
               corfrSelectReq.m_enDevConnReq,enRespCode);
   } //if(true == bValidateClient(u8Index))
   if(corfrSelectReq.m_enDevConnReq == e8DEVCONNREQ_DESELECT )
   {
      //! Clear all metadata upon deselection;
      if (NULL != m_poAppMngrRespIntf)
      {
         trAppMediaMetaData rMediaMetadata;
         trAppPhoneData rfcorApplicationPhoneData;
         trAppPhoneCallData rfcorApplicationPhoneCallData;
         trNavStatusData corfrNavStatusData;
         trNavNextTurnData corfrNavNextTurnData;
         trNavNextTurnDistanceData corfrNavNextTurnDistanceData;
         trAppMediaPlaytime rfcorApplicationMediaPlaytime;
         trTurnByTurnInfo corfrTurnByTurnInfo;
         m_poAppMngrRespIntf->vPostApplicationMediaMetaData(rMediaMetadata, corEmptyUsrContext);
         m_poAppMngrRespIntf->vPostApplicationPhoneData(rfcorApplicationPhoneData, corEmptyUsrContext);
         m_poAppMngrRespIntf->vPostApplicationPhoneCallData(rfcorApplicationPhoneCallData, corEmptyUsrContext);
         m_poAppMngrRespIntf->vPostNavigationStatus(corfrNavStatusData);
         m_poAppMngrRespIntf->vPostNavigationNextTurnDataStatus(corfrNavNextTurnData);
         m_poAppMngrRespIntf->vPostNavigationNextTurnDistanceDataStatus(corfrNavNextTurnDistanceData);
         m_poAppMngrRespIntf->vPostApplicationMediaPlaytime(rfcorApplicationMediaPlaytime, corEmptyUsrContext);
         m_poAppMngrRespIntf->vPostTurnByTurnInfo(corfrTurnByTurnInfo);
         ETG_TRACE_USR1(("spi_tclAppMngr::metadata cleared"));

      } //if (NULL != m_poConnMngrResp)
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbPostSetNotiEnabledInfoResult
***************************************************************************/
t_Void spi_tclAppMngr::vCbPostSetNotiEnabledInfoResult(const t_U32 cou32DevId,
                                                       tenErrorCode enErrorCode, 
                                                       const trUserContext& rfrCUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbPostSetNotiEnabledInfoResult:Dev-0x%x enErrorCode-%d",
      cou32DevId,enErrorCode));

   if(NULL != m_poAppMngrRespIntf)
   {
      tenResponseCode enRespCode = (enErrorCode==e8NO_ERRORS)?e8SUCCESS:e8FAILURE ;
      m_poAppMngrRespIntf->vPostSetMLNotificationEnabledInfoResult(cou32DevId,enRespCode,
         enErrorCode,rfrCUsrCntxt);
   }//if(NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbPostInvokeNotificationActionResult
***************************************************************************/
t_Void spi_tclAppMngr::vCbPostInvokeNotificationActionResult(const t_U32 cou32DevId,
                                                             tenErrorCode enErrorCode, 
                                                             const trUserContext& rfrCUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbPostInvokeNotificationActionResult:Dev-0x%x enErrorCode-%d",
      cou32DevId,enErrorCode));

   if(NULL != m_poAppMngrRespIntf)
   {
      tenResponseCode enRespCode = (enErrorCode==e8NO_ERRORS)?e8SUCCESS:e8FAILURE ;
      m_poAppMngrRespIntf->vPostInvokeNotificationActionResult(enRespCode,
         enErrorCode,rfrCUsrCntxt);
   }//if(NULL != m_poAppMngrRespIntf)
}


/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbPostNotificationResult
***************************************************************************/
t_Void spi_tclAppMngr::vCbPostNotificationResult(const t_U32 cou32DevId,
                                                 const trNotiData& corfrNotidata)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbPostNotificationResult:Dev-0x%x App-%d",
      cou32DevId,corfrNotidata.u32NotiAppID));
   if(NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostNotificationInfo(cou32DevId,corfrNotidata.u32NotiAppID,
         corfrNotidata);
   }//if(NULL != m_poAppMngrRespIntf)
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclAppMngr::vCbPostDeviceDisplayContext(t_U32 u32DeviceHandle,
                                                   t_Bool bDisplayFlag,
                                                   tenDisplayContext enDisplayContext,
                                                   const trUserContext rcUsrCntxt)
{
   ETG_TRACE_USR1((" spi_tclAppMngr::vCbPostDeviceDisplayContext"));
   SPI_INTENTIONALLY_UNUSED(bDisplayFlag);
   SPI_INTENTIONALLY_UNUSED(enDisplayContext);
   SPI_INTENTIONALLY_UNUSED(rcUsrCntxt);
   SPI_INTENTIONALLY_UNUSED(u32DeviceHandle);
   if(NULL != m_poAppMngrRespIntf)
   {
     // m_poAppMngrRespIntf->vPostDeviceDisplayContext(u32DeviceHandle, bDisplayFlag, enDisplayContext, rcUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclAppMngr::bIsLaunchAppReq()
***************************************************************************/
t_Bool spi_tclAppMngr::bIsLaunchAppReq(t_U32 u32DevID,
                                       t_U32 u32AppID,
                                       t_U32 u32NotificationID,
                                       t_U32 u32NotificationActionID,
                                       tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1((" spi_tclAppMngr::bIsLaunchAppReq:Dev-0x%x App-0x%x Noti-0x%x NotiAction-0x%x Cat-%d",
      u32DevID,u32AppID,u32NotificationID,u32NotificationActionID,enDevCat));

   t_U8 u8Index = static_cast<t_U8>(enDevCat);
   t_Bool bIsLaunchReq = false;

   if(true == bValidateClient(u8Index))
   {
      bIsLaunchReq = m_poAppMngrDevBase[u8Index]->bIsLaunchAppReq(u32DevID,u32AppID,
         u32NotificationID,u32NotificationActionID);
   } //if(true == bValidateClient(u8Index))

   return bIsLaunchReq;
}


/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbSessionStatusUpdate()
***************************************************************************/
t_Void spi_tclAppMngr::vCbSessionStatusUpdate(const t_U32 cou32DevId,
                                              const tenDeviceCategory coenDevCat,
                                              const tenSessionStatus coenSessionStatus)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbSessionStatusUpdate:Dev-0x%x Cat-%d Status-%d",
      cou32DevId,ETG_ENUM(DEVICE_CATEGORY,coenDevCat),ETG_ENUM(SESSION_STATUS,coenSessionStatus)));
   if(NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vUpdateAppBlockingInfo(cou32DevId,coenDevCat,coenSessionStatus);
   }//if(NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbActiveAppInfo()
***************************************************************************/
t_Void spi_tclAppMngr::vCbActiveAppInfo(const t_U32 cou32DevId,
                                        const tenDeviceCategory coenDevCat,
                                        const t_U32 cou32AppId,
                                        const tenAppCertificationInfo coenAppCertInfo)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCbActiveAppInfo:Dev-0x%x Cat-%d App-0x%x CertStatus-%d",
      cou32DevId,ETG_ENUM(DEVICE_CATEGORY,coenDevCat),cou32AppId,coenAppCertInfo));
   if(NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vUpdateActiveAppInfo(cou32DevId,coenDevCat,cou32AppId,coenAppCertInfo);
   }//if(NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSendNavigationStatus()
***************************************************************************/
t_Void spi_tclAppMngr::vSendNavigationStatus(const trNavStatusData& corfrNavStatusData)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSendNavigationStatus entered"));

   if(NULL!= m_poAppMngrRespIntf)
   {
	   m_poAppMngrRespIntf->vPostNavigationStatus(corfrNavStatusData);
   }
} //!end of vSendNavigationStatus()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSendNavigationNextTurnData()
***************************************************************************/
t_Void spi_tclAppMngr::vSendNavigationNextTurnData(const trNavNextTurnData& corfrNavNextTurnData)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSendNavigationNextTurnData entered"));

   if(NULL!= m_poAppMngrRespIntf)
   {
	   m_poAppMngrRespIntf->vPostNavigationNextTurnDataStatus(corfrNavNextTurnData);
   }
} //!end of vSendNavigationNextTurnData()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vMock_vSendNavigationNextTurnData() tenAAutoTurnSide,tenAAutoNextTurnEvent,s32TurnAngle
***************************************************************************/
t_Void spi_tclAppMngr::vMock_NavigationNextTurnCallback(tenAAutoTurnSide enAAPNavNextTurnSide,
         tenAAutoNextTurnEvent enAAPNavNextTurnType, t_S32 s32TurnAngle)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vMock_NavigationNextTurnCallback entered"));

   if(NULL != m_poAppMngrDevBase[e8DEV_TYPE_ANDROIDAUTO])
   {
      m_poAppMngrDevBase[e8DEV_TYPE_ANDROIDAUTO]->vMock_NavigationNextTurnCallback(enAAPNavNextTurnSide,enAAPNavNextTurnType,s32TurnAngle);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSendNavigationNextTurnData()
***************************************************************************/
t_Void spi_tclAppMngr::vSendNavigationNextTurnDistanceData(const trNavNextTurnDistanceData& corfrNavNextTurnDistanceData)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSendNavigationNextTurnDistanceData entered"));

   if(NULL!= m_poAppMngrRespIntf)
   {
	   m_poAppMngrRespIntf->vPostNavigationNextTurnDistanceDataStatus(corfrNavNextTurnDistanceData);
   }
} //!end of vSendNavigationNextTurnDistanceData()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSendNotificationData()
***************************************************************************/
t_Void spi_tclAppMngr::vSendNotificationData(const trNotificationData& corfrNotificationData)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSendNotificationData entered"));

   if(NULL!= m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostNotificationData(corfrNotificationData);
   }
} //!end of vSendNotificationData()

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vAckNotification()
***************************************************************************/
t_Void spi_tclAppMngr::vAckNotification(const trNotificationAckData& corfrNotifAckData)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vAckNotification entered"));

   t_U8 u8Index = static_cast<t_U8>(corfrNotifAckData.enDeviceCategory);

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vAckNotification(corfrNotifAckData.u32DeviceHandle, corfrNotifAckData.szNotifId);
   } //if(true == bValidateClient(u8Index))
} //!end of vAckNotification()

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAppMngr::vGetAppAudioInfo()
 ***************************************************************************/
t_Void spi_tclAppMngr::vGetAppAudioInfo(t_U32 u32DevID,
                                        tenDeviceCategory enDevCat,
                                        std::vector<trAppsAudioInfo>& rfvecAppsAudioInfo)
{
   t_U8 u8Index = static_cast<t_U8>(enDevCat);

   if(true == bValidateClient(u8Index))
   {
      m_poAppMngrDevBase[u8Index]->vGetAppAudioInfo(u32DevID,rfvecAppsAudioInfo);
   } //if(true == bValidateClient(u8Index))
}

/***************************************************************************
** FUNCTION: t_Void spi_tclVideo::vSetVideoBlockingMode()
***************************************************************************/
t_Void spi_tclAppMngr::vSetVideoBlockingMode(const t_U32 cou32DevId,
                                           const tenBlockingMode coenBlockingMode,
                                           const tenVideoBlockingReason coenVideoBlockingReason,
                                           const tenDeviceCategory coenDevCat,
                                           const trUserContext& corfrcUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAppMngr:vSetVideoBlockingMode:Device ID-0x%x  BlockingMode-%d VideoBlockingReason=%d",
      cou32DevId,ETG_ENUM(BLOCKING_MODE,coenBlockingMode), ETG_ENUM(VIDEO_BLOCKING_REASON,coenVideoBlockingReason)));

   tenErrorCode enErrorcode = e8INVALID_DEV_HANDLE;
   t_U8 u8Index = static_cast<t_U8>(coenDevCat);
   
   if(true == bValidateClient(u8Index))
   {
      enErrorcode = m_poAppMngrDevBase[u8Index]->enSetVideoBlockingMode(cou32DevId, coenBlockingMode, coenVideoBlockingReason);
   } //if(true == bValidateClient(u8Index))

   //Post the Message to RespIntf
   if(NULL != m_poAppMngrRespIntf)
   {
      tenResponseCode enResponseCode = (enErrorcode == e8NO_ERRORS)?e8SUCCESS:e8FAILURE;
      m_poAppMngrRespIntf->vPostSetVideoBlockingModeResult(enResponseCode,enErrorcode,corfrcUsrCntxt);
   }//if(NULL != m_poVideoRespIntf)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::bGetAppAudioHandlingCaps()
***************************************************************************/
t_Bool spi_tclAppMngr::bGetAppAudioHandlingCaps(const tenDeviceCategory coenDevCat)
{
   //Get the Applications audio handling capability  t_U8 u8Index = static_cast<t_U8>(coenDevCat);

   t_U8 u8Index = static_cast<t_U8>(coenDevCat);
   t_Bool bIsAppAdvAudioCapable = false;

   if(true == bValidateClient(u8Index))
   {
      bIsAppAdvAudioCapable = m_poAppMngrDevBase[u8Index]->bGetAudioHandlingCaps();
   }

   return bIsAppAdvAudioCapable;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vApplicationMediaMetaDataCb(...
***************************************************************************/

t_Void spi_tclAppMngr::vApplicationMediaMetaDataCb(const trAppMediaMetaData& rfcorApplicationMediaMetaData,
   const trUserContext& rfcorUsrCntxt)
{
   //! Forward media metadata to  response interface
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostApplicationMediaMetaData(rfcorApplicationMediaMetaData, rfcorUsrCntxt);
   } //if (NULL != m_poConnMngrResp)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbPostVendorExtsnData(...
***************************************************************************/
t_Void spi_tclAppMngr::vCbPostVendorExtsnData(const t_U8* poVendorExtsnData, size_t u8LengthOfData)
{
    ETG_TRACE_USR1(("spi_tclAppMngr:vCbPostVendorExtsnData entered"));
    m_poAppMngrRespIntf->vPostVendorExtsnData(poVendorExtsnData,u8LengthOfData);
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vCbPostVendorExtsnData(...
***************************************************************************/
t_Void spi_tclAppMngr::vSendVendorExtsn(t_U8* poVendorExtsnData, size_t u8LengthOfData,const trUserContext &rfcorUsrCntxt)
{
    ETG_TRACE_USR1(("spi_tclAppMngr:vSendVendorExtsn entered"));

    for(t_U8 u8Index=0;u8Index < NUM_APPMNGR_CLIENTS; u8Index++)
    {
       if(NULL != m_poAppMngrDevBase[u8Index])
       {
           m_poAppMngrDevBase[u8Index]->vSendVendorExtsn(poVendorExtsnData,u8LengthOfData,rfcorUsrCntxt);
       }//if((NULL != m_poAppMngrDevBase[
    }
    tenErrorCode enErrorcode = e8NO_ERRORS;
    tenResponseCode enResponseCode = e8SUCCESS;
    if(NULL != m_poAppMngrRespIntf)
    {
        m_poAppMngrRespIntf->vPostSendVendorExtsnDataResult(enResponseCode,enErrorcode,rfcorUsrCntxt);
    }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vApplicationPhoneDataCb(...
***************************************************************************/

t_Void spi_tclAppMngr::vApplicationPhoneDataCb(const trAppPhoneData& rfcorApplicationPhoneData,
   const trUserContext& rfcorUsrCntxt)
{
   //! Forward phone data including phone call metadata to connnMngr response interface
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostApplicationPhoneData(rfcorApplicationPhoneData, rfcorUsrCntxt);
   } //if (NULL != m_poConnMngrResp)
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vApplicationPhoneCallDataCb(...
***************************************************************************/

t_Void spi_tclAppMngr::vApplicationPhoneCallDataCb(const trAppPhoneCallData& rfcorApplicationPhoneCallData,
   const trUserContext& rfcorUsrCntxt)
{
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostApplicationPhoneCallData(rfcorApplicationPhoneCallData, rfcorUsrCntxt);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAppMngr::vApplicationMediaPlaytimeCb(...
***************************************************************************/

t_Void spi_tclAppMngr::vApplicationMediaPlaytimeCb(const trAppMediaPlaytime& rfcorApplicationMediaPlaytime,
      const trUserContext& rfcorUsrCntxt)
{
   //! Forward current playing media playtime to  response interface
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostApplicationMediaPlaytime(rfcorApplicationMediaPlaytime, rfcorUsrCntxt);
   } //if (NULL != m_poConnMngrResp)
}

t_Void spi_tclAppMngr::vAppMediaAlbumArtCb(const trAppMediaAlbumArt& rAppMediaAlbumArt, const trUserContext& rUserContext)
{

   //! Forward current media albumart response interface
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostApplicationMediaAlbumArt(rAppMediaAlbumArt, rUserContext);
   } //if (NULL != m_poConnMngrResp)
}


/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::vGetMediaObjectAlbumArt
 ***************************************************************************/
t_Void spi_tclAppMngr::vGetMediaObjectAlbumArt(const t_U32& corfu32DeviceHandle,
        const t_String& corfsAlbumArt, const trUserContext& corfrUsrCntxt) const
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vGetMediaObjectAlbumArt entered. DeviceHandle-%d, albumArt-%s",
          corfu32DeviceHandle, corfsAlbumArt.c_str()));

   // Right now this function only is supported for DiPO
   if(NULL != m_poAppMngrDevBase[e8DEV_TYPE_DIPO])
   {
      m_poAppMngrDevBase[e8DEV_TYPE_DIPO]->vGetMediaObjectAlbumArt(corfu32DeviceHandle,corfsAlbumArt,corfrUsrCntxt);
   }

}


/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vPostDeviceLockInfoCb()
***************************************************************************/
t_Void spi_tclAppMngr::vPostDeviceLockInfoCb(const t_U32& corfu32DeviceHandle,
         const tenDeviceCategory& corfenDevCat, const t_Bool& bDeviceLockEnabled)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vPostDeviceLockInfoCb:DeviceLockEnabled-%d",
      ETG_ENUM(BOOL,bDeviceLockEnabled)));

   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostDeviceLockInfo(corfu32DeviceHandle, corfenDevCat, bDeviceLockEnabled);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::vTurnByTurnInfoCb
 ***************************************************************************/
t_Void spi_tclAppMngr::vTurnByTurnInfoCb(const trTurnByTurnInfo &corfrTurnByTurnInfo)
{
    ETG_TRACE_USR1(("spi_tclAppMngr::vTurnByTurnInfoCb entered"));
    if (NULL != m_poAppMngrRespIntf)
    {
       //@TODO: vPostTurnByTurnInfo to be removed once HMI adopts to technology specific updates.
       m_poAppMngrRespIntf->vPostTurnByTurnInfo(corfrTurnByTurnInfo);

       switch(corfrTurnByTurnInfo.enDeviceCategory)
       {
       case e8DEV_TYPE_ANDROIDAUTO:
       {
           // On receiving Android Auto TBT updates, use AA specific update - vPostAAutoTurnByTurnInfo
           m_poAppMngrRespIntf->vPostAAutoTurnByTurnInfo(corfrTurnByTurnInfo.u32DeviceHandle, corfrTurnByTurnInfo.rAAutoTBTUpdate);
       }
       break;

       case e8DEV_TYPE_MIRRORLINK:
       {
           // On receiving MirrorLink TBT updates, use ML specific update - vPostMLTurnByTurnInfo
           m_poAppMngrRespIntf->vPostMLTurnByTurnInfo(corfrTurnByTurnInfo.u32DeviceHandle, corfrTurnByTurnInfo.rMLTBTUpdate);
       }
       break;

       case e8DEV_TYPE_CARLIFE:
       {
           // On receiving Baidu Carlife TBT updates, use BDCL specific update - vPostBDCLTurnByTurnInfo
           m_poAppMngrRespIntf->vPostBDCLTurnByTurnInfo(corfrTurnByTurnInfo.u32DeviceHandle, corfrTurnByTurnInfo.rBDCLTBTUpdate);
       }
       break;

       default: break;

       }
    } //if (NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAppMngr::vSendSelectionProgressState(...)
***************************************************************************/
t_Void spi_tclAppMngr::vSendSelectionProgressState(const t_U32 cou32DeviceHandle,tenDeviceSelectionProgressState enDeviceSelectionProgressState)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSendSelectionProgressState() entered "
            "cou32DeviceHandle = %d, DeviceSelectionProgressState = %d",
            cou32DeviceHandle, ETG_ENUM(SEL_PROGRESS_STATE,enDeviceSelectionProgressState)));

   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   SPI_NORMAL_ASSERT(NULL == poMediator);
   if ((NULL != poMediator) && (IS_VALID_DEVHANDLE(cou32DeviceHandle)))
   {
      poMediator->vPostSelectionProgressState(cou32DeviceHandle,enDeviceSelectionProgressState);
   }
} //!end of vSendSelectionProgressState()


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppMngr::vSetAppSettingsInstance()
***************************************************************************/
t_Void spi_tclAppMngr::vSetAppSettingsInstance(spi_tclAppSettingsIntf* poAppSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vSetAppSettingsInstance entered"));
   if( NULL != poAppSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclAppMngr::vSetAppSettingsInstance: Setting m_poAppSettings val "));
       m_poAppSettings = poAppSettingsIntf;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppMngr::vDisplayAppSettings()
***************************************************************************/
t_Void spi_tclAppMngr::vDisplayAppSettings()
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vDisplayAppSettings entered"));

   if( NULL != m_poAppSettings)
   {
      m_poAppSettings->vDisplayAppSettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAppMngr::vGetAppManagerConfig()
***************************************************************************/

t_Void spi_tclAppMngr::vGetAppManagerConfig()
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vGetAppManagerConfig entered"));

   if( NULL != m_poAppSettings)
   {
      t_String szVehicleModelNumber;
      m_poAppSettings->vReadSysConfigs();
      m_poAppSettings->bGetAAPNavigationEnabledInfo();
      m_poAppSettings->bGetAAPNotificationUpdatesEnabledInfo();
      m_poAppSettings->bEnableCTSTest();
      m_poAppSettings->szGetVehicleModelNumber(szVehicleModelNumber);

   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::bOnRegRouteGuidanceDisplayConf
 ***************************************************************************/
t_Bool spi_tclAppMngr::bOnRegRouteGuidanceDisplayConf(const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::bOnRegRouteGuidanceDisplayConf entered"));
   t_Bool bRet = false;

   // supported only for DiPO
   if(NULL != m_poAppMngrDevBase[e8DEV_TYPE_DIPO])
   {
       bRet = m_poAppMngrDevBase[e8DEV_TYPE_DIPO]->bOnRegRouteGuidanceDisplayConf(corfvecRouteGuidanceDisplayComponent);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::bOnStartRouteGuidanceUpdates
 ***************************************************************************/
t_Bool spi_tclAppMngr::bOnStartRouteGuidanceUpdates(trStartRouteGuidanceUpdates& corfStartRouteGuidanceUpdates)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::bOnStartRouteGuidanceUpdates entered"));
   t_Bool bRet = false;

   // supported only for DiPO
   if(NULL != m_poAppMngrDevBase[e8DEV_TYPE_DIPO])
   {
       bRet = m_poAppMngrDevBase[e8DEV_TYPE_DIPO]->bOnStartRouteGuidanceUpdates(corfStartRouteGuidanceUpdates);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::bOnStopRouteGuidanceUpdates
 ***************************************************************************/
t_Bool spi_tclAppMngr::bOnStopRouteGuidanceUpdates(trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::bOnStopRouteGuidanceUpdates entered"));
   t_Bool bRet = false;

   // supported only for DiPO
   if(NULL != m_poAppMngrDevBase[e8DEV_TYPE_DIPO])
   {
       bRet = m_poAppMngrDevBase[e8DEV_TYPE_DIPO]->bOnStopRouteGuidanceUpdates(corfStopRouteGuidanceUpdates);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::vCarPlayRouteGuidanceUpdateCb
 ***************************************************************************/
t_Void spi_tclAppMngr::vCarPlayRouteGuidanceUpdateCb(const trDiPORouteGuidanceUpdate &corfrtrDiPORouteGuidanceUpdate)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCarPlayRouteGuidanceUpdateCb entered"));
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostCarPlayRouteGuidanceUpdate(corfrtrDiPORouteGuidanceUpdate);
   } //if (NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::vCarPlayRouteGuidanceManeuverUpdateCb
 ***************************************************************************/
t_Void spi_tclAppMngr::vCarPlayRouteGuidanceManeuverUpdateCb(const trDiPORouteGuidanceManeuverUpdate &corfrtrDiPORouteGuidanceManeuverUpdate)
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vCarPlayRouteGuidanceManeuverUpdateCb entered"));
   if (NULL != m_poAppMngrRespIntf)
   {
      m_poAppMngrRespIntf->vPostCarPlayRouteGuidanceManeuverUpdate(corfrtrDiPORouteGuidanceManeuverUpdate);
   } //if (NULL != m_poAppMngrRespIntf)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppMngr::vRestoreSettings
 ***************************************************************************/
t_Void spi_tclAppMngr::vRestoreSettings()
{
   ETG_TRACE_USR1(("spi_tclAppMngr::vRestoreSettings entered"));
   if (NULL != m_poAppMngrDevBase[e8DEV_TYPE_DIPO])
   {
      m_poAppMngrDevBase[e8DEV_TYPE_DIPO]->vRestoreSettings();
   } //if (NULL != m_poAppMngrRespIntf)
}

//lint ?restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
