/*!
 *******************************************************************************
 * \file             spi_tclBDCLConnection.h
 * \brief            Baidu Carlife Video class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Baidu Carlife Video class
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 21.04.2017  | Ramya Murthy          | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLBDCLVIDEO_H_
#define _SPI_TCLBDCLVIDEO_H_

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <memory>

#include "spi_tclVideoDevBase.h"
#include "spi_tclBDCLRespVideo.h"
#include "spi_tclBDCLRespSession.h"
#include "SPITypes.h"
#include "Lock.h"
#include "Timer.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
//! Forward declarations
class spi_tclBDCLCmdVideo;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
typedef std::shared_ptr< spi_tclBDCLCmdVideo >     t_SptrBDCLCmdVideo;


/****************************************************************************/
/*!
* \class    spi_tclBDCLVideo
* \brief    BDCL Video implementation
*           This class interacts with BDCL Video wrapper to interact with google
*           API's
****************************************************************************/
class spi_tclBDCLVideo: public spi_tclVideoDevBase, public spi_tclBDCLRespVideo,
public spi_tclBDCLRespSession
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLVideo::spi_tclBDCLVideo()
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLVideo()
   * \brief   Default Constructor
   * \param   t_Void
   * \sa      ~spi_tclBDCLVideo()
   **************************************************************************/
   spi_tclBDCLVideo();

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLVideo::~spi_tclBDCLVideo()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclBDCLVideo()
   * \brief   Destructor
   * \param   t_Void
   * \sa      spi_tclBDCLVideo()
   **************************************************************************/
   ~spi_tclBDCLVideo();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBDCLVideo::bInitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitialize()
   * \brief   To Initialize all the BDCL Video related things
   * \retval  t_Bool
   * \sa      vUninitialize()
   **************************************************************************/
   t_Bool bInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLVideo::vUninitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitialize()
   * \brief   To Uninitialize all the BDCL Video related things
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   t_Void vUninitialize();

   /***************************************************************************
   ** FUNCTION:  t_Void  spi_tclBDCLVideo::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRegisterCallbacks(const trVideoCallbacks& corfrVideoCallbacks)
   * \brief   To Register for the asynchronous responses that are required from
   *          BDCL Video
   * \param   corfrVideoCallbacks : [IN] Video callabcks structure
   * \retval  t_Void
   **************************************************************************/
   t_Void vRegisterCallbacks(const trVideoCallbacks& corfrVideoCallbacks);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLVideo::vSelectDevice()
   ***************************************************************************/
   /*!
   * \fn      virtual t_Void vSelectDevice(const t_U32 cou32DevId,
   *          const tenDeviceConnectionReq coenConnReq)
   * \brief   To Initialize/UnInitialize Video setup for the selected device
   * \pram    cou32DevId : [IN] Unique Device Id
   * \param   coenConnReq : [IN] connected/disconnected
   * \retval  t_Void
   **************************************************************************/
   t_Void vSelectDevice(const t_U32 cou32DevId,
         const tenDeviceConnectionReq coenConnReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLVideo::vOnSelectDeviceResult()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSelectDeviceResult(const t_U32 cou32DevId,
   *                 const tenDeviceConnectionReq coenConnReq,
   *                 const tenResponseCode coenRespCode)
   * \brief   To perform the actions that are required, after the select device is
   *           successful
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    coenConnReq : [IN] Identifies the Connection Request.
   * \pram    coenRespCode: [IN] Response code. Success/Failure
   * \retval  t_Void
   **************************************************************************/
   t_Void vOnSelectDeviceResult(const t_U32 cou32DevId,
         const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBDCLVideo::bLaunchVideo()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bLaunchVideo(const t_U32 cou32DevId,
   *                 const t_U32 cou32AppId,
   *                 const tenEnabledInfo coenSelection)
   * \brief   To Launch the Video for the requested app
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    cou32AppId  : [IN] Application Id
   * \pram    coenSelection  : [IN] Enable/disable the video
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bLaunchVideo(const t_U32 cou32DevId, const t_U32 cou32AppId,
         const tenEnabledInfo coenSelection);

   /***************************************************************************
   ** FUNCTION:  t_U32  spi_tclBDCLVideo::vStartVideoRendering()
   ***************************************************************************/
   /*!
   * \fn      t_Void vStartVideoRendering(t_Bool bStartVideoRendering)
   * \brief   Method to inform that the SPI Layer is enabled/disabled
   * \pram    bStartVideoRendering : [IN] True - Layer is enabled
   *                                      False - Layer is Disabled
   * \retval  t_Void
   **************************************************************************/
   t_Void vStartVideoRendering(t_Bool bStartVideoRendering);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclBDCLVideo::vGetVideoSettings()
   ***************************************************************************/
   /*!
   * \fn     t_Void vGetVideoSettings(const t_U32 cou32DevId,
   *                                  trVideoAttributes& rfrVideoAttributes
   * \brief  To get the current Video Settings.
   * \param  u32DeviceHandle    : [IN] Uniquely identifies the target Device.
   * \param  rfrVideoAttributes : [OUT]includes screen size & orientation.
   * \retval t_Void
   * \sa
   **************************************************************************/
   t_Void vGetVideoSettings(const t_U32 cou32DevId, trVideoAttributes& rfrVideoAttributes);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclBDCLVideo::vSetOrientationMode()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetOrientationMode(const t_U32 cou32DevId,
   *                                    const tenOrientationMode coenOrientationMode,
   *                                    const trUserContext& corfrUsrCntxt)
   * \brief  Interface to set the orientation mode of the projected display.
   * \param  cou32DevId          : [IN] Uniquely identifies the target Device.
   * \param  coenOrientationMode : [IN] Orientation Mode Value.
   * \param  corfrUsrCntxt       : [IN] User Context
   * \retval t_Void
   **************************************************************************/
   t_Void vSetOrientationMode(const t_U32 cou32DevId,
         const tenOrientationMode coenOrientationMode,
         const trUserContext& corfrUsrCntxt);
   /***************************************************************************
   *** FUNCTION:  t_Void spi_tclBDCLVideo::vOnMDInfo()
   ***************************************************************************/
   /*!
   * \fn      vOnMDInfo
   * \brief   Triggered when BDCL device sends device info
   * \param   corfrMDInfo [IN]: Structure containing BDCL device info
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vOnMDInfo(const trBdclMDInfo& crfrMDInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLVideo::vOnCoreError()
    ***************************************************************************/
   /*!
    * \fn      vOnCoreError
    * \brief   Triggered when there is an error notified by AditCoreSurrogate endpoint
    * \param   enCarlifeError [IN]: Error type enumeration
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vOnCoreError(tenBdclCarLifeError enCarlifeError);

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLVideo(const spi_tclBDCLVideo...
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLVideo( const spi_tclBDCLVideo& corfoSrc)
   * \brief   Copy constructor - Do not allow the creation of copy constructor
   * \param   corfoSrc : [IN] reference to source data interface object
   * \retval
   * \sa      spi_tclBDCLVideo()
   ***************************************************************************/
   spi_tclBDCLVideo(const spi_tclBDCLVideo& corfoSrc);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLVideo& operator=( const spi_tclBDCL...
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLVideo& operator=(const spi_tclBDCLVideo& corfoSrc))
   * \brief   Assignment operator
   * \param   corfoSrc : [IN] reference to source data interface object
   * \retval
   * \sa      spi_tclBDCLVideo(const spi_tclBDCLVideo& otrSrc)
   ***************************************************************************/
   spi_tclBDCLVideo& operator=(const spi_tclBDCLVideo& corfoSrc);

   /***************************************************************************
    ** FUNCTION:  t_Void vOnVideoEncoderInitDone()
    ***************************************************************************/
   /*!
    * \fn      virtual t_Void vOnVideoEncoderInitDone()
    * \brief   Triggered when VideoEncoderInitDone msg is sent by MD
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vOnVideoEncoderInitDone(t_U32 u32Width,
            t_U32 u32Height, t_U32 u32Framerate);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBDCLVideo::bInitializeVideo()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitializeVideo(t_U32 u32DevID))
   * \brief   Method to initialize the Video resources that are required for session
   * \pram    u32DevID  : [IN] Uniquely identifies the target Device.
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bInitializeVideo(t_U32 u32DevID);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBDCLVideo::vProceedwithSelection(const t_U32 cou32DevId)
   ***************************************************************************/
   /*!
   * \fn      t_Bool vProceedwithSelection(const t_U32 cou32DevId)
   * \brief   Method to perform device selection
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \retval  t_Bool
   **************************************************************************/
   t_Void vProceedWithSelection(const t_U32 cou32DevId);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLVideo::vUninitializeVideo()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitializeVideo(t_U32 u32DevID)
   * \brief   Method to uninitialize the Video resources that were allocated for session
   * \pram    u32DevID  : [IN] Uniquely identifies the target Device.
   * \retval  t_Void
   **************************************************************************/
   t_Void vUninitializeVideo(t_U32 u32DevID);

   /***************************************************************************
    *** FUNCTION: spi_tclBDCLVideo::vStartSDKVersionCheckTimer ()
    ***************************************************************************/
   /*!
    * \fn      vStartSDKVersionCheckTimer ()
    * \brief   Start timer function will begin the timer if MDInfo message is not received from phone.
    * \param   none
    * \return  none
    **************************************************************************/
    t_Void vStartSDKVersionCheckTimer ();

    /***************************************************************************
     ** FUNCTION: spi_tclBDCLVideo::vStopSDKVersionCheckTimer ()
     ***************************************************************************/
    /*!
     * \fn      vStopSDKVersionCheckTimer ()
     * \brief   Stop timer function will stop the timer after MDInfo message is received from phone.
     * \param   none
     * \return  none
     **************************************************************************/
    t_Void vStopSDKVersionCheckTimer ();

    /***************************************************************************
     ** FUNCTION: spi_tclBDCLVideo::bSDKVersionCheckTimerCb()
     ***************************************************************************/
    /*!
     * \fn      bSDKVersionCheckTimerCb()
     * \brief   Callback that will be called on SDKVersionCheck timer expiry
     * \param   none
     * \return  none
     **************************************************************************/
    static t_Bool bSDKVersionCheckTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData);

    /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclBDCLVideo::vCarlifeSupportCheck(const t_U32 cou32DevId)
    ***************************************************************************/
    /*!
    * \fn      t_Bool vCarlifeSupportCheck(const t_U32 cou32DevId)
    * \brief   Method to check wthether Carlife is supported in HU
    * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
    * \retval  t_Bool
    **************************************************************************/
    t_Void vCarlifeSupportCheck(const t_U32 & cou32DevId);

   //! Pointer to Cmd Video class
   t_SptrBDCLCmdVideo m_spoCmdVideo;

   //! Video Callbacks structure
   trVideoCallbacks m_rVideoCallbacks;

   //! Currently selected device
   t_U32 m_u32SelectedDeviceID;
   
   t_Bool m_bMDInfoMessageReceived;

   t_Bool m_SelectionInProgress;

   t_U32 m_u32AndroidSdkVersion;

   t_U32 m_u32SelectedDevId;

   //! Lock to protect Timer flags
   Lock m_oSDKVersionCheckLock;

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/
}; //class spi_tclBDCLVideo

#endif //_SPI_TCLBDCLVIDEO_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
