/*!
 *******************************************************************************
 * \file             spi_tclBDCLVideoResourceMngr.cpp
 * \brief            Video Resource Manager for Baidu Carlife
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Video Resource Manager for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 04.05.2017 |  Ramya Murthy        | Initial Version
 06.03.2018 |  Ramya Murthy        | Added notification of Session error on transport error

 \endverbatim
 ******************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLContextHandler.h"
#include "spi_tclBDCLResourceMngr.h"
#include "spi_tclBDCLVideoResourceMngr.h"
#include "spi_tclMediator.h"
#include "Timer.h"


//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
      #include "trcGenProj/Header/spi_tclBDCLVideoResourceMngr.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

static timer_t srTranportErrorTimerId = 0;
static timer_t srVideoInitDoneWaitTimerId = 0;
static t_Bool bTransportErrorTimerRunning = false;
static t_Bool bVideoInitDoneWaitTimerRunning = false;
static const t_U32 cou32TransportTimerVal = 5000;
static const t_U32 cou32VideoInitDoneWaitTimerVal = 3000;

/***************************************************************************
** FUNCTION:  spi_tclBDCLVideoResourceMngr::spi_tclBDCLVideoResourceMngr()
***************************************************************************/
spi_tclBDCLVideoResourceMngr::spi_tclBDCLVideoResourceMngr():
   m_u32SelectedDeviceID(0),
   m_enCurAccDispCntxt(e8DISPLAY_CONTEXT_NATIVE),
   m_bCurAccDisplayFlag(true),
   m_bInitialMDVideoFocus(false),
   m_bIsVideoInitDoneCbRcvd(false),
   m_bPlaybackStarted(false),
   m_enCurMDFocusState(e8CL_VIDEOFOCUSSTATE_LOSS),
   m_bIsLaunchAppRequestReceived(false)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr() entered "));
}

/***************************************************************************
** FUNCTION:  spi_tclBDCLVideoResourceMngr::~spi_tclBDCLVideoResourceMngr()
***************************************************************************/
spi_tclBDCLVideoResourceMngr::~spi_tclBDCLVideoResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclBDCLVideoResourceMngr() entered"));
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclBDCLVideoResourceMngr::bInitialize()
***************************************************************************/
t_Bool spi_tclBDCLVideoResourceMngr::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::bInitialize() entered"));

   t_Bool bRet = false;

   t_SptrBDCLManager spoBDCLMngr = spi_tclBDCLManager::getInstance();

   SPI_NORMAL_ASSERT(!spoBDCLMngr);
   if (spoBDCLMngr)
   {
      bRet = spoBDCLMngr->bRegisterObject((spi_tclBDCLRespVideo*) this);
      bRet = bRet && spoBDCLMngr->bRegisterObject((spi_tclBDCLRespSession*) this);
   }

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vUninitialize()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vUnInitialize() entered"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayContext()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayContext(const t_U32 cou32DevId,
         t_Bool bDisplayFlag, tenDisplayContext enDisplayContext)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayContext:"
            "Device ID-0x%x Accessory Takes Focus-%d AccDisplayContext-%d",
            cou32DevId,ETG_ENUM(BOOL,bDisplayFlag),ETG_ENUM(DISPLAY_CONTEXT,enDisplayContext)));

   // do not process the update, if the DisplayFlag is false for a context, which is not active currently.
   if ((true == bDisplayFlag) || ((false == bDisplayFlag) && (m_enCurAccDispCntxt == enDisplayContext)))
   {
      m_enCurAccDispCntxt = (bDisplayFlag)? (enDisplayContext) : (e8DISPLAY_CONTEXT_NATIVE);
      m_bCurAccDisplayFlag = bDisplayFlag;

      t_SptrBDCLContextHandler spoBDCLCtxtHandler = spi_tclBDCLContextHandler::getInstance();
      if ((0 != m_u32SelectedDeviceID) && (spoBDCLCtxtHandler))
      {
         tenBdclVideoFocusState enNewAccFocusState = e8CL_VIDEOFOCUSSTATE_UNKNOWN;
         tenBdclVideoFocusState enNewMDFocusState = e8CL_VIDEOFOCUSSTATE_UNKNOWN;

         m_oCurMDFocusStateLock.s16Lock();
         if ((true == spoBDCLCtxtHandler->bGetAccVideoFocusState(enDisplayContext, bDisplayFlag, enNewAccFocusState)) &&
                  (true == spoBDCLCtxtHandler->bGetMDVideoFocusState(m_enCurMDFocusState, enNewAccFocusState, enNewMDFocusState)))
         {
            switch (enNewAccFocusState)
            {
               case e8CL_VIDEOFOCUSSTATE_GAIN:
               case e8CL_VIDEOFOCUSSTATE_GAIN_TRANSIENT:
               {
                  //! HMI does TAKE or BORROW when MD has the screen, pause MD video playback
                  if (e8CL_VIDEOFOCUSSTATE_GAIN == m_enCurMDFocusState)
                  {
                     vSetVideoFocus(e8CL_VIDEOFOCUS_NATIVE);
                  }
               }
                  break;
               case e8CL_VIDEOFOCUSSTATE_LOSS_TRANSIENT:
               {
                  //! HMI does an UNBORROW, start MD video playback
                  if (e8CL_VIDEOFOCUSSTATE_LOSS_TRANSIENT == m_enCurMDFocusState)
                  {
                     vSetVideoFocus(e8CL_VIDEOFOCUS_PROJECTED);
                  }
               }
                  break;
               default:
                  ETG_TRACE_ERR(("[ERR]:vSetAccessoryDisplayContext: default state"));
                  break;
            }//switch (enNewAccFocusState)

            m_enCurMDFocusState = enNewMDFocusState;

         }//if (true == bGetAccVideoFocusState...)
         else
         {
            ETG_TRACE_ERR(("[ERR]:vSetAccessoryDisplayContext: Element not found"));
         }
         m_oCurMDFocusStateLock.vUnlock();
      }//if ((0 != m_u32SelectedDeviceID) && (spoBDCLCtxtHandler))

      //! Populate TAKE Constraint, even when device is not active.
      m_oCurMDFocusStateLock.s16Lock();
      tenBdclResponse enRespType = e8CL_RESP_DENY;
      if ((spoBDCLCtxtHandler) &&
               (true == spoBDCLCtxtHandler->bGetAccRespType(m_enCurAccDispCntxt, m_enCurMDFocusState, enRespType)))
      {
         m_enGrantConstraint = (e8CL_RESP_ACCEPT == enRespType)? e8DIPO_CONSTRAINT_ANYTIME : e8DIPO_CONSTRAINT_NEVER ;
      }
      m_oCurMDFocusStateLock.vUnlock();
   }//if ((true == bDisplayFlag) || ...)
   else
   {
      ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayContext: Current active Context is %d ",
         ETG_ENUM(DISPLAY_CONTEXT, m_enCurAccDispCntxt)));
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayMode(t_U32...
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayMode(const t_U32 cou32DeviceHandle,
         const trDisplayContext corDisplayContext,
         const trDisplayConstraint corDisplayConstraint,
         const tenDisplayInfo coenDisplayInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vSetAccessoryDisplayMode: Resource Mngt handled by %d",
            ETG_ENUM(DISPLAY_INFO, coenDisplayInfo)));
   SPI_INTENTIONALLY_UNUSED(corDisplayConstraint);

   if(e8_DISPLAY_CONTEXT == coenDisplayInfo)
   {
      vSetAccessoryDisplayContext(cou32DeviceHandle,
         corDisplayContext.bDisplayFlag,
         corDisplayContext.enDisplayContext);
   }//if(e8_DISPLAY_CONTEXT == coenDisplayInfo)
   if(e8_DISPLAY_CONSTRAINT == coenDisplayInfo)
   {
      if ((e8DIPO_CONSTRAINT_NEVER == corDisplayConstraint.enTakeConstraint) ||
               (e8DIPO_CONSTRAINT_NEVER == corDisplayConstraint.enBorrowConstraint))
      {
         //Considered Borrow constraint also as in the cases for Emergency e.g RVC,
         //Borrow can be set to Never where take can still be userinitiated/Anytime
         m_enGrantConstraint = e8DIPO_CONSTRAINT_NEVER;
      }
      else
      {
         m_enGrantConstraint = e8DIPO_CONSTRAINT_ANYTIME;
      }

      if (0 != m_u32SelectedDeviceID)
      {
         switch (corDisplayConstraint.enTransferType)
         {
            case e8DIPO_TRANSFERTYPE_TAKE:
            {
               //Take has higher priority, process take even after borrow
               m_oCurMDFocusStateLock.s16Lock();
               m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_LOSS;
               m_oCurMDFocusStateLock.vUnlock();
               m_enPrevTransferType = corDisplayConstraint.enTransferType;
               vSetVideoFocus(e8CL_VIDEOFOCUS_NATIVE);
            }
               break;
            case e8DIPO_TRANSFERTYPE_BORROW:
               //after take, if borrow comes - continue with take only
               //else proceed with borrow
               if (e8DIPO_TRANSFERTYPE_TAKE != m_enPrevTransferType)
               {
                  m_oCurMDFocusStateLock.s16Lock();
                  m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_LOSS_TRANSIENT;
                  m_oCurMDFocusStateLock.vUnlock();
                  m_enPrevTransferType = corDisplayConstraint.enTransferType;
                  vSetVideoFocus(e8CL_VIDEOFOCUS_NATIVE);
               }
               break;
            case e8DIPO_TRANSFERTYPE_UNTAKE:
            {
               //Nothing to do. HMI has to do Launch App
            }
               break;
            case e8DIPO_TRANSFERTYPE_UNBORROW:
               // check for previous state - If the last state was Borrow then only process
               if (e8DIPO_TRANSFERTYPE_BORROW == m_enPrevTransferType)
               {
                  m_oCurMDFocusStateLock.s16Lock();
                  m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_GAIN_TRANSIENT;
                  m_oCurMDFocusStateLock.vUnlock();
                  m_enPrevTransferType = corDisplayConstraint.enTransferType;
                  vSetVideoFocus(e8CL_VIDEOFOCUS_PROJECTED);
               }
               break;
            default:
               ETG_TRACE_ERR(("[ERR]::Accessory Display Context:default state"));
               break;
         }//switch(corDisplayConstraint.enTransferType)
      }//if( 0 != m_u32SelectedDeviceID )
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnSPISelectDeviceResult()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID,
       tenDeviceConnectionReq enDevConnReq,
       tenResponseCode enRespCode,
       tenErrorCode enErrorCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnSPISelectDeviceResult: Result of -%d ",
            ETG_ENUM(CONNECTION_REQ,enDevConnReq)));
   SPI_INTENTIONALLY_UNUSED(u32DevID);
   SPI_INTENTIONALLY_UNUSED(enRespCode);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);

   m_bPlaybackStarted = false;

   if ((e8DEVCONNREQ_SELECT == enDevConnReq) && (e8SUCCESS == enRespCode))
   {
      m_u32SelectedDeviceID = u32DevID;
      m_oVideoSetupLock.s16Lock();
      if(false == m_bIsVideoInitDoneCbRcvd)
      {
	     // Added to handle error case when VideoeEncoderInitDone message doesnot come from phone. If message doesnot come 
		 //before timeout deselection is triggered
         ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnSPISelectDeviceResult() Video encoder init done not received yet. Starting timer"));
         vStartVideoInitDoneWaitTimer();

      }else
      {
         ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnSPISelectDeviceResult() Video encoder init done received. Continue the session"));
      }
      m_oVideoSetupLock.vUnlock();
   }
   else
   {
      m_u32SelectedDeviceID = 0;

      m_oVideoSetupLock.s16Lock();
      m_bIsVideoInitDoneCbRcvd = false;
      m_bInitialMDVideoFocus = false;
      m_oVideoSetupLock.vUnlock();

      m_oCurMDFocusStateLock.s16Lock();
      m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_LOSS;
      m_oCurMDFocusStateLock.vUnlock();

      vStopTransportErrorTimer ();
      vStopVideoInitDoneWaitTimer();
   }

   //@Note: Video is released at start of deselection to immediately remove CL screen and avoid showing last frame/black screen.

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vRequestProjUI()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vRequestProjUI(t_U32 u32DevId,t_U32 u32AppId)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vRequestProjUI:Device ID-0x%x AppID-0x%x",
      u32DevId,u32AppId));

   m_bIsLaunchAppRequestReceived = true;

   m_oVideoSetupLock.s16Lock();
   if (false == m_bIsVideoInitDoneCbRcvd)
   {
      ETG_TRACE_USR2(("[DESC] Projection UI requested before Video encoder init is completed - Delaying Video playback request to MD"));
      m_bInitialMDVideoFocus = true;
   }
   else
   {
      t_SptrBDCLContextHandler spoBDCLCtxtHandler = spi_tclBDCLContextHandler::getInstance();
      if ((spoBDCLCtxtHandler) && (true == spoBDCLCtxtHandler->bProcessRequestProjUI(m_enCurAccDispCntxt)))
      {
         //! Set the Playback state to GAIN
         m_oCurMDFocusStateLock.s16Lock();
         m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_GAIN;
         m_oCurMDFocusStateLock.vUnlock();

         ETG_TRACE_USR2(("[DESC] Starting video on user request "));
         vSetVideoFocus(e8CL_VIDEOFOCUS_PROJECTED);
      }
      else
      {
         //Clearing variable m_bIsLaunchAppRequestReceived as video is not requested from phone(NCG3D-158207).
         m_bIsLaunchAppRequestReceived = false;
         ETG_TRACE_ERR(("[ERR] Do not process the Projection UI request in the current display context"));
      }
   }
   m_oVideoSetupLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnPlaybackStart()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnPlaybackStart()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnPlaybackStart() entered: CurMDFocusState = %d", m_enCurMDFocusState));
   
   m_bPlaybackStarted = true;
   
   m_oCurMDFocusStateLock.s16Lock();
   tenBdclVideoFocusState enCurMDFocusState = m_enCurMDFocusState;
   m_oCurMDFocusStateLock.vUnlock();

   if ((e8CL_VIDEOFOCUSSTATE_LOSS == enCurMDFocusState) || (e8CL_VIDEOFOCUSSTATE_LOSS_TRANSIENT == enCurMDFocusState))
   {
      m_oVideoSetupLock.s16Lock();
      //MD was in GAIN state and video playback was requested.
      //Due to some user interaction, MD's state changed to LOSS. So rquest phone to stop video playback.
      ETG_TRACE_USR2(("[DESC] Accessory has taken Video Focus due to user interactions. Stopping video"));
      vSetVideoFocus(e8CL_VIDEOFOCUS_NATIVE);

      m_oVideoSetupLock.vUnlock();
   }
   //Enable Projection screen
   else if(NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt)
   {
      m_rResMngrCbs.fvUpdateDeviceDisplayCntxt(true, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }//else if(NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt)

   m_oCurMDFocusStateLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnVideoEncoderInitDone()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnVideoEncoderInitDone(t_U32 u32Width,
         t_U32 u32Height, t_U32 u32Framerate)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr:vOnVideoEncoderInitDone entered: Width=%d, Height=%d, Framerate=%d",
            u32Width, u32Height, u32Framerate));

   m_oVideoSetupLock.s16Lock();

   m_bIsVideoInitDoneCbRcvd = true;
   vStopVideoInitDoneWaitTimer();
   m_oVideoSetupLock.vUnlock();

   //send session active message to HMI. m_oVideoSetupLock is released because SessionState setting may take some time
   if ((NULL != m_rResMngrCbs.fvBdclSessionState) && (0 != m_u32SelectedDeviceID))
   {
      m_rResMngrCbs.fvBdclSessionState(e8_SESSION_ACTIVE);
   }

   m_oVideoSetupLock.s16Lock();
   if (m_bInitialMDVideoFocus)
   {
      m_bInitialMDVideoFocus = false;

      ETG_TRACE_USR2(("[DESC] Starting video since request to start projection is already received"));
      vRequestProjUI(m_u32SelectedDeviceID, 0);
   }
   m_oVideoSetupLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnGotoDeskTop()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnGotoDeskTop()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnGotoDeskTop() entered"));

   m_oCurMDFocusStateLock.s16Lock();
   m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_LOSS;
   m_oCurMDFocusStateLock.vUnlock();

   if (NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt)
   {
      m_rResMngrCbs.fvUpdateDeviceDisplayCntxt(false, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_HOME_SCREEN);
   }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnForeground()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnForeground(t_Bool bForeground)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnForeground() entered"));
   if ((NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt) && (FALSE == bForeground))
   {
      m_oCurMDFocusStateLock.s16Lock();
      m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_LOSS;
      m_oCurMDFocusStateLock.vUnlock();
      m_rResMngrCbs.fvUpdateDeviceDisplayCntxt(false, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_PHONE_SCREEN_OFF);
  
   }
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vSetVideoFocus()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vSetVideoFocus(tenBdclVideoFocus enVideoFocus)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vSetVideoFocus:VideoFocusMode-%d ",
      ETG_ENUM(VIDEOFOCUS_MODE,enVideoFocus)));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   t_SptrBDCLCmdVideo spoCmdVideo = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdVideoInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!spoCmdVideo);

   if ((spoCmdVideo) && (NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt) && (0 != m_u32SelectedDeviceID))
   {
      switch(enVideoFocus)
      {
         case e8CL_VIDEOFOCUS_NATIVE:
         {
            //@Note: Pausevideo is commented as an improvement for the ticket NCG3D-159211
            //Reason: According to latest Carlife spec Even when carlife app is in background MD will send video data along with Heartbeat message. So no need to pause MD video
            //spoCmdVideo->vPauseVideo();
            m_rResMngrCbs.fvUpdateDeviceDisplayCntxt(false, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
            //@Note: Not required to wait for device response since there is no acknowledgment
         }
            break;
         case e8CL_VIDEOFOCUS_PROJECTED:
         {
            spoCmdVideo->vStartVideo();
         }
         break;
         default:
            ETG_TRACE_ERR(("[ERR]:Invalid video focus"));
            break;
      }//switch(enVideoFocus)
   }//if ((spoCmdVideo) && (NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt))
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnCoreError()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnCoreError(tenBdclCarLifeError enCarlifeError)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnCoreError entered: %d, Selected DeviceID = %d  IsVideoStarted = %d, LaunchApprequest received state = %d",
            ETG_ENUM(BDCL_ERROR, enCarlifeError), m_u32SelectedDeviceID, ETG_ENUM(BOOL, m_bPlaybackStarted), ETG_ENUM(BOOL, m_bIsLaunchAppRequestReceived)));

   switch (enCarlifeError)
   {
      case e8CL_AOAP_WRITE_TIMEOUT_ERROR:
      case e8CL_AOAP_READ_TIMEOUT_ERROR:
      {
		 //variable m_bIsLaunchAppRequestReceived is added to resolve RTC Ticket:474277
         if (((m_bPlaybackStarted)|| (m_bIsLaunchAppRequestReceived)) && (0 != m_u32SelectedDeviceID)) 
         {
            vStartTransportErrorTimer ();
         }
      }
         break;
      case e8CL_AOAP_NO_ERROR:
      {
            vStopTransportErrorTimer ();
      }
         break;
      case e8CL_AOAP_NO_DEVICE_ERROR:
      {
         //Device ID is cleared to avoid starting video once device disconnected/deselection started
         m_oVideoSetupLock.s16Lock();
         m_u32SelectedDeviceID = 0;
         m_oVideoSetupLock.vUnlock();

         vStopTransportErrorTimer ();
         vStopVideoInitDoneWaitTimer();
      }
         break;
      default:
         //! Nothing to do
         break;

   }//switch (enCarlifeError)
}

/***************************************************************************
 ** FUNCTION: spi_tclBDCLVideoResourceMngr::vSendDeviceDeselect()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vSendDeviceDeselect()
{
   ETG_TRACE_USR4(("[DESC]::vSendDeviceDeselect: Requesting deselection of device "));

   m_bPlaybackStarted = false;

   m_bIsLaunchAppRequestReceived = false;

   //! Post internal deselection
   spi_tclMediator *poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vPostAutoDeviceSelection(m_u32SelectedDeviceID, e8DEVCONNREQ_DESELECT);
   } // if (NULL != poMediator)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vStartTransportErrorTimer()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vStartTransportErrorTimer ()
{
   m_oErrorLock.s16Lock();
   ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::vStartTransportErrorTimer entered: Timer running status : %d",
            ETG_ENUM(BOOL, bTransportErrorTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if((NULL != poTimer) && (false == bTransportErrorTimerRunning))
   {
      poTimer->StartTimer(srTranportErrorTimerId, cou32TransportTimerVal, 0, this,
         &spi_tclBDCLVideoResourceMngr::bTransportErrorTimerCb, NULL);

      ETG_TRACE_USR4(("[DESC]:Transport Error Timer started"));

      bTransportErrorTimerRunning = true;
   }//End of if(NULL != poTimer)

   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vStartTransportErrorTimer left"));
   m_oErrorLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vStopTransportErrorTimer ()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vStopTransportErrorTimer ()
{
   m_oErrorLock.s16Lock();
   ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::vStopTransportErrorTimer  entered: Timer running status : %d",
            ETG_ENUM(BOOL, bTransportErrorTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (true == bTransportErrorTimerRunning))
   {
      poTimer->CancelTimer(srTranportErrorTimerId);
      ETG_TRACE_USR4(("[DESC]:Transport Error Timer Stopped"));

      bTransportErrorTimerRunning = false;
   }//End of if (NULL != poTimer)...

   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vStopTransportErrorTimer  left"));
   m_oErrorLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::bTransportErrorTimerCb()
***************************************************************************/
t_Bool spi_tclBDCLVideoResourceMngr::bTransportErrorTimerCb(timer_t rTimerID,
         t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::bTransportErrorTimerCb entered: Timer running status : %d",
            ETG_ENUM(BOOL, bTransportErrorTimerRunning)));
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srTranportErrorTimerId);
      ETG_TRACE_USR4(("Transport Error Timer expired"));

      bTransportErrorTimerRunning = false;
   }//if (NULL != poTimer)

   spi_tclBDCLVideoResourceMngr* poBDCLVideoRes = static_cast<spi_tclBDCLVideoResourceMngr*>(pvObject);
   if (NULL != poBDCLVideoRes)
   {
      ETG_TRACE_USR4(("Transport Error Timer expired. Hence sending session error and requesting device deslection."));

      if (NULL != poBDCLVideoRes->m_rResMngrCbs.fvBdclSessionState)
      {
         poBDCLVideoRes->m_rResMngrCbs.fvBdclSessionState(e8_SESSION_ERROR);
      }

      poBDCLVideoRes->vSendDeviceDeselect();
   }
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vOnDeviceDeselection()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vOnDeviceDeselection()
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vOnDeviceDeselection() entered"));

   m_bIsLaunchAppRequestReceived = false;
   m_oCurMDFocusStateLock.s16Lock();
   m_enCurMDFocusState = e8CL_VIDEOFOCUSSTATE_LOSS;
   m_oCurMDFocusStateLock.vUnlock();

   //Device ID is cleared to avoid starting video once device disconnected/deselection started
   m_oVideoSetupLock.s16Lock();
   m_u32SelectedDeviceID = 0;
   m_oVideoSetupLock.vUnlock();

   if (NULL != m_rResMngrCbs.fvUpdateDeviceDisplayCntxt)
   {
      m_rResMngrCbs.fvUpdateDeviceDisplayCntxt(false, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }

}

/***************************************************************************
 ** FUNCTION:  t_Void vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vRegisterCallbacks(const trBdclResMngrCbs& corfrResMngrCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vRegisterCallbacks entered"));
   m_rResMngrCbs = corfrResMngrCbs;
}


/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vStartVideoInitDoneWaitTimer()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vStartVideoInitDoneWaitTimer()
{
   m_oVideoInitDoneWaitTimerLock.s16Lock();
   ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::vStartVideoInitDoneWaitTimer entered: Timer running status : %d",
            ETG_ENUM(BOOL, bVideoInitDoneWaitTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if((NULL != poTimer) && (false == bVideoInitDoneWaitTimerRunning))
   {
      poTimer->StartTimer(srVideoInitDoneWaitTimerId, cou32VideoInitDoneWaitTimerVal, 0, this,
         &spi_tclBDCLVideoResourceMngr::bVideoInitDoneWaitTimerCb, NULL);

      ETG_TRACE_USR4(("[DESC]:Video Encoder Init not received. Timer started"));

      bVideoInitDoneWaitTimerRunning = true;
   }//End of if(NULL != poTimer)

   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vStartVideoInitDoneWaitTimer left"));
   m_oVideoInitDoneWaitTimerLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::vStopVideoInitDoneWaitTimer()
***************************************************************************/
t_Void spi_tclBDCLVideoResourceMngr::vStopVideoInitDoneWaitTimer()
{
   m_oVideoInitDoneWaitTimerLock.s16Lock();
   ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::vStopVideoInitDoneWaitTimer entered: Timer running status : %d",
            ETG_ENUM(BOOL, bVideoInitDoneWaitTimerRunning)));

   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (true == bVideoInitDoneWaitTimerRunning))
   {
      poTimer->CancelTimer(srVideoInitDoneWaitTimerId);
      ETG_TRACE_USR4(("[DESC]:Video Encoder Init wait Timer Stopped"));

      bVideoInitDoneWaitTimerRunning = false;
   }//End of if (NULL != poTimer)...

   ETG_TRACE_USR1(("spi_tclBDCLVideoResourceMngr::vStopVideoInitDoneWaitTimer left"));
   m_oVideoInitDoneWaitTimerLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLVideoResourceMngr::bVideoInitDoneWaitTimerCb()
***************************************************************************/
t_Bool spi_tclBDCLVideoResourceMngr::bVideoInitDoneWaitTimerCb(timer_t rTimerID,
         t_Void *pvObject, const t_Void *pvUserData)
{
   ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::bVideoInitDoneWaitTimerCb() entered: Timer running status : %d",
            ETG_ENUM(BOOL, bVideoInitDoneWaitTimerRunning)));
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srVideoInitDoneWaitTimerId);
      ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::bVideoInitDoneWaitTimerCb():: Video Encoder Init wait expired"));

      bVideoInitDoneWaitTimerRunning = false;
   }//if (NULL != poTimer)

   spi_tclBDCLVideoResourceMngr* poBDCLVideoRes = static_cast<spi_tclBDCLVideoResourceMngr*>(pvObject);
   if (NULL != poBDCLVideoRes)
   {
      ETG_TRACE_USR4(("spi_tclBDCLVideoResourceMngr::bVideoInitDoneWaitTimerCb():: Video Encoder Init wait expired. Hence sending session error and requesting device deslection."));

      if (NULL != poBDCLVideoRes->m_rResMngrCbs.fvBdclSessionState)
      {
         poBDCLVideoRes->m_rResMngrCbs.fvBdclSessionState(e8_SESSION_ERROR);
      }

      poBDCLVideoRes->vSendDeviceDeselect();
   }
   return true;
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
