/*!
 *******************************************************************************
 * \file             spi_tclBDCLAudioResourceMngr.cpp
 * \brief            Audio Resource Manager for Baidu Carlife
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Resource Manager for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 04.05.2017 |  Ramya Murthy        | Initial Version
 06.03.2018 |  Ramya Murthy        | Impemented fix for NCG3D-81731, handling of audio take during VR, 
                                     and usage of VR interrupt msg for cleanup
 13.03.2018 |  Ramya Murthy        | Impemented fix for NCG3D-81798

 \endverbatim
 ******************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLContextHandler.h"
#include "spi_tclBDCLAudioResourceMngr.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
      #include "trcGenProj/Header/spi_tclBDCLAudioResourceMngr.cpp.trc.h"
   #endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/***************************************************************************
** FUNCTION:  spi_tclBDCLAudioResourceMngr::spi_tclBDCLAudioResourceMngr()
***************************************************************************/
spi_tclBDCLAudioResourceMngr::spi_tclBDCLAudioResourceMngr():
   m_enMuteState(e8AUD_UNMUTED),
   m_enTakeConstraint(e8CL_CONSTRAINT_ANYTIME),
   m_enBorrowConstraint(e8CL_CONSTRAINT_ANYTIME),
   m_enMixConstraint(e8CL_CONSTRAINT_ANYTIME),
   m_enCurAudioCntxt(e8SPI_AUDIO_MAIN),
   m_bCurCtxtFlag(false),
   m_enMusicModuleStatus(e8CL_MUSIC_STATUS_IDLE),
   m_enVRModuleStatus(e8CL_VR_STATUS_RECORD_IDLE),
   m_enPhoneModuleStatus(e8CL_PHONE_STATUS_IDLE)
{
	/*lint -esym(40,_1)_1 Undeclared identifier */
	/*lint -esym(40,_2)_2 Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr() entered "));

   m_DevAudStates[e8CL_AUDTYPE_MEDIA] = e8CL_AUD_ST_IDLE;
   m_DevAudStates[e8CL_AUDTYPE_TTS] = e8CL_AUD_ST_IDLE;
   m_DevAudStates[e8CL_AUDTYPE_VOICE] = e8CL_AUD_ST_IDLE;
   m_DevAudStates[e8CL_AUDTYPE_MIC] = e8CL_AUD_ST_IDLE;

   m_mapAudStreamState[e8CL_AUDTYPE_MEDIA] = e8CL_AUD_STREAM_CLOSED;
   m_mapAudStreamState[e8CL_AUDTYPE_TTS] = e8CL_AUD_STREAM_CLOSED;
   m_mapAudStreamState[e8CL_AUDTYPE_VOICE] = e8CL_AUD_STREAM_CLOSED;
   m_mapAudStreamState[e8CL_AUDTYPE_MIC] = e8CL_AUD_STREAM_CLOSED;

   //! Register with BDCL manager for Session & Audio callbacks
   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   SPI_NORMAL_ASSERT(!spoBDCLManager);
   if (spoBDCLManager)
   {
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespAudio*) this);
   }
}

/***************************************************************************
** FUNCTION:  spi_tclBDCLAudioResourceMngr::~spi_tclBDCLAudioResourceMngr()
***************************************************************************/
spi_tclBDCLAudioResourceMngr::~spi_tclBDCLAudioResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclBDCLAudioResourceMngr() entered"));

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLAudioResourceMngr::vSetAccessoryAudioContext()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetAccessoryAudioContext(const t_U32 cou32DevId,
         const tenAudioContext coenAudioCntxt, t_Bool bReqFlag)
{
   m_oLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vSetAccessoryAudioContext :: AudioCntxt :%d, bReqFlag : %d",
         ETG_ENUM(AUDIO_CONTEXT, coenAudioCntxt), ETG_ENUM(BOOL, bReqFlag)));

   t_SptrBDCLContextHandler spoBDCLCtxtHandler = spi_tclBDCLContextHandler::getInstance();
   t_Bool bIsAudioContextChanged = (coenAudioCntxt != m_enCurAudioCntxt) || (bReqFlag != m_bCurCtxtFlag);

   //! Process audio context update only if there is a change in audio context or flag.
   if ((spoBDCLCtxtHandler) && (bIsAudioContextChanged))
   {
      trBdclAudioStates rCurAudStates;
      rCurAudStates.enMuteState = enGetCurMuteState();
      rCurAudStates.enTakeConstraint = enGetCurTakeConstraint();
      rCurAudStates.enBorrowConstraint = enGetCurBorrowConstraint();
      rCurAudStates.enMixConstraint = enGetCurMixConstraint();
      rCurAudStates.enMusicState = enGetCurAudioState(e8CL_AUDTYPE_MEDIA);
      rCurAudStates.enVRState = enGetCurAudioState(e8CL_AUDTYPE_VOICE);
      rCurAudStates.enTTSState = enGetCurAudioState(e8CL_AUDTYPE_TTS);

      trBdclAudioStates rNewAudStates;

      if (spoBDCLCtxtHandler->bGetAudioStatus(bReqFlag, coenAudioCntxt, rCurAudStates, rNewAudStates))
      {
         if ((bReqFlag) && (e8SPI_AUDIO_UNKNOWN == coenAudioCntxt))
         {
            m_oModuleStatesLock.s16Lock();
            tenBdclPhoneState enPhoneModState = m_enPhoneModuleStatus;
            m_oModuleStatesLock.vUnlock();

            //@Note: If phone call is active on CL device, HU should not request to stop music since it is handled by phone itself.
            if (e8CL_PHONE_STATUS_IDLE == enPhoneModState)
            {
               ETG_TRACE_USR2(("[DESC] Since current active audio context is unknown, constraints remain unchanged. "
                     "Only updating new media state."));
               vSetCurAudioState(e8CL_AUDTYPE_MEDIA, rNewAudStates.enMusicState, true);
            }
            else
            {
               ETG_TRACE_USR2(("[DESC] Since CL phone call is active, constraints and audio states remains unchanged."
                     "Waiting for phone to pause music"));
            }
         } //if ((bReqFlag) && (e8SPI_AUDIO_UNKNOWN == coenAudioCntxt))
         else if ((!bReqFlag) && (e8SPI_AUDIO_UNKNOWN == coenAudioCntxt))
         {
            if (e8CL_CONSTRAINT_ANYTIME == enGetCurTakeConstraint())
            {
               ETG_TRACE_USR2(("[DESC] Since current active audio context is unknown, constraints remain unchanged. "
                     "Only updating new media state."));
               vSetCurAudioState(e8CL_AUDTYPE_MEDIA, rNewAudStates.enMusicState, true);
            }
            else
            {
               ETG_TRACE_USR2(("[DESC] Since high prio audio is active, constraints and audio states remains unchanged."
                     "Waiting for unborrow of high prio source"));
            }
         } //else if ((!bReqFlag) && (e8SPI_AUDIO_UNKNOWN == coenAudioCntxt))
         else
         {
            //! Update the new device audio states
            vSetCurAudioStates(rNewAudStates, true);

            //! Special usecase: During VR, music is paused and HU takes audio -> set Music state IDLE to phone
            tenBdclAudioContext enBdclAudContext = e8CL_AUDCTXT_MAIN;
            t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
            t_SptrBDCLCmdSession spoCmdSession = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdSessionInstance()) : (nullptr);
            if ((bReqFlag) && (spoBDCLCtxtHandler->bLookupCLAudioContext(coenAudioCntxt, enBdclAudContext)) &&
                     (e8CL_AUDCTXT_MAIN == enBdclAudContext) && (spoCmdSession))
            {
               ETG_TRACE_USR4(("[DESC]::Setting Music state IDLE to phone since HU has taken audio focus"));
               spoCmdSession->vSetMusicState(e8CL_MUSIC_STATUS_IDLE);
            } //if ((bReqFlag) && ...)
         } //else
      } //if (spoBDCLCtxtHandler->bGetAudioStatus(...))
   } //if ((spoBDCLCtxtHandler) && (bIsAudioContextChanged))

   m_enCurAudioCntxt = coenAudioCntxt;
   m_bCurCtxtFlag = bReqFlag;

   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudioResourceMngr::vOnModuleStatusInfo()
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vOnModuleStatusInfo(
         const tBdclModuleStatusList& corfStatusList)
{
   m_oModuleStatesLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vOnModuleStatusInfo entered: List size=%d", corfStatusList.size()));

   for (auto itr = corfStatusList.begin(); itr != corfStatusList.end(); ++itr)
   {
      tenBdclCmdModuleID enModuleID = static_cast<tenBdclCmdModuleID>(itr->u32ModuleID);
      switch (enModuleID)
      {
         case e8CL_PHONE_MODULE_ID:
         {
            tenBdclPhoneState enPhoneModuleStatus = static_cast<tenBdclPhoneState>(itr->u32StatusID);
            ETG_TRACE_USR4(("[PARAM]::Phone module state = %d", ETG_ENUM(BDCL_PHONE_STATE, enPhoneModuleStatus)));
            m_enPhoneModuleStatus = enPhoneModuleStatus;
         }
            break;
         case e8CL_NAVI_MODULE_ID:
            ETG_TRACE_USR4(("[PARAM]::Navigation module state = %d", ETG_ENUM(BDCL_NAV_STATE, itr->u32StatusID)));
            break;
         case e8CL_MUSIC_MODULE_ID:
         {
            tenBdclMusicState enMusicModuleStatus = static_cast<tenBdclMusicState>(itr->u32StatusID);
            ETG_TRACE_USR4(("[PARAM]::Music module state = %d", ETG_ENUM(BDCL_MUSIC_STATE, enMusicModuleStatus)));
            //@Note: No action taken on Music module status since it arrives before VR module state notifications.
            /*if (enMusicModuleStatus != m_enMusicModuleStatus)
            {
               t_Bool bIsMusicActive = (e8CL_MUSIC_STATUS_RUNNING == enMusicModuleStatus);
               vValidateModuleState(e8CL_AUDTYPE_MEDIA, bIsMusicActive);
            }
            m_enMusicModuleStatus = enMusicModuleStatus;*/
         }
            break;
         case e8CL_VR_MODULE_ID:
         {
            tenBdclVRState enVRModuleStatus = static_cast<tenBdclVRState>(itr->u32StatusID);
            ETG_TRACE_USR4(("[PARAM]::VR module state = %d", ETG_ENUM(BDCL_VR_STATE, enVRModuleStatus)));
            if (enVRModuleStatus != m_enVRModuleStatus)
            {
               t_Bool bIsVRActive = (e8CL_VR_STATUS_RECORD_RUNNING == enVRModuleStatus);
               vValidateModuleState(e8CL_AUDTYPE_VOICE, bIsVRActive);
            }
            m_enVRModuleStatus = enVRModuleStatus;
         }
            break;
         case e8CL_CONNECT_MODULE_ID:
            ETG_TRACE_USR4(("[PARAM]::Connect module state = %d", ETG_ENUM(BDCL_CONNECT_STATE, itr->u32StatusID)));
            break;
         case e8CL_MIC_MODULE_ID:
            ETG_TRACE_USR4(("[PARAM]::Mic module state = %d", ETG_ENUM(BDCL_MIC_STATE, itr->u32StatusID)));
            break;
         default:
            ETG_TRACE_ERR(("[ERR]: Invalid module ID %d", ETG_ENUM(BDCL_MODULE_ID, enModuleID)));
            break;
      } //switch (enModuleID)
   } //for (auto itr = corfStatusList.begin();...)

   m_oModuleStatesLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void vOnVRModuleStatus(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vOnVRModuleStatus(tenBdclVRState enVRModuleState)
{
   m_oModuleStatesLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vOnVRModuleStatus entered: %d", ETG_ENUM(BDCL_VR_STATE, enVRModuleState)));

   if (enVRModuleState != m_enVRModuleStatus)
   {
      t_Bool bIsVRActive = (e8CL_VR_STATUS_RECORD_RUNNING == enVRModuleState);
      vValidateModuleState(e8CL_AUDTYPE_VOICE, bIsVRActive);
   }
   m_enVRModuleStatus = enVRModuleState;

   m_oModuleStatesLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLAudioResourceMngr::vOnAudioRequest(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vOnAudioRequest(const trBdclAudioRequest& crfrAudioRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vOnAudioRequest entered: Request = %d, Stream = %d",
         ETG_ENUM(BDCL_AUD_REQUEST, crfrAudioRequest.enAudRequest),
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrAudioRequest.enAudStream)));

   m_oLock.s16Lock();

   switch (crfrAudioRequest.enAudRequest)
   {
      case e8CL_AUD_REQ_START:
      {
         //@Note: Special handling for phone issue - when VR is ended, phone sends VR module status IDLE,
         //closes Mic and after sometime reopens Mic, causing Speech channel to never be released by SPI
         if ((e8CL_AUDTYPE_MIC == crfrAudioRequest.enAudStream) && (e8CL_VR_STATUS_RECORD_IDLE == m_enVRModuleStatus))
         {
            ETG_TRACE_ERR(("[ERR]::vOnAudioRequest: Invalid request. Mic is opened when VR module state is IDLE"));
         }
         else
         {
            m_mapAudStreamState[crfrAudioRequest.enAudStream] = e8CL_AUD_STREAM_OPEN;
            ETG_TRACE_USR4(("Updated %d stream state to %d",
                     ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrAudioRequest.enAudStream),
                     ETG_ENUM(BDCL_AUDSTREAM_STATE, m_mapAudStreamState[crfrAudioRequest.enAudStream])));

            vProcessAudioStartRequest(crfrAudioRequest);
         }
      }
         break;
      case e8CL_AUD_REQ_STOP:
      {
         m_mapAudStreamState[crfrAudioRequest.enAudStream] = e8CL_AUD_STREAM_CLOSED;
         ETG_TRACE_USR4(("Updated %d stream state to %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrAudioRequest.enAudStream),
                  ETG_ENUM(BDCL_AUDSTREAM_STATE, m_mapAudStreamState[crfrAudioRequest.enAudStream])));

         vProcessAudioStopRequest(crfrAudioRequest);
      }
         break;
      case e8CL_AUD_REQ_INTERRUPT:
      {
         m_mapAudStreamState[crfrAudioRequest.enAudStream] = e8CL_AUD_STREAM_CLOSED;
         ETG_TRACE_USR4(("Updated %d stream state to %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrAudioRequest.enAudStream),
                  ETG_ENUM(BDCL_AUDSTREAM_STATE, m_mapAudStreamState[crfrAudioRequest.enAudStream])));

         vProcessAudioInterruptRequest(crfrAudioRequest);
      }
         break;
      default:
      {
         ETG_TRACE_ERR(("[ERR]::vOnAudioRequest: Invalid request type"));
      }
         break;
   }//switch (crfrAudioRequest.enAudRequest)

   m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void vSendAudioRequest(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSendAudioRequest(const trBdclAudioRequest& crfrAudioRequest)
{
   //! TODO - temporary solution to forward request to Audio class via CmdAudio class.
   //! Needs to be redesigned.
   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   t_SptrBDCLCmdAudio spoCmdAudio = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdAudioInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!spoCmdAudio);
   if (spoCmdAudio)
   {
      spoCmdAudio->vProcessAudioRequest(crfrAudioRequest);
   }
}

/***************************************************************************
** FUNCTION:  t_Void vValidateModuleState()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vValidateModuleState(tenBdclAudStreamType enAudStream, t_Bool bModuleActive)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vValidateModuleState entered: Stream=%d, IsActive=%d",
            ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream), ETG_ENUM(BOOL, bModuleActive)));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   t_SptrBDCLCmdAudio spoCmdAudio = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdAudioInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!spoCmdAudio);

   if (spoCmdAudio)
   {
      //! Forward Module inactive state
      if (false == bModuleActive)
      {
         ETG_TRACE_USR1(("VR state = %d, Mic state = %d", m_mapAudStreamState[e8CL_AUDTYPE_VOICE], m_mapAudStreamState[e8CL_AUDTYPE_MIC]));
         if ((e8CL_AUD_STREAM_CLOSED == m_mapAudStreamState[e8CL_AUDTYPE_VOICE]) &&
               (e8CL_AUD_STREAM_CLOSED == m_mapAudStreamState[e8CL_AUDTYPE_MIC]) &&
               ((e8CL_AUDTYPE_VOICE == enAudStream) || (e8CL_AUDTYPE_MIC == enAudStream)))
         {
            vSetCurAudioState(e8CL_AUDTYPE_VOICE, e8CL_AUD_ST_IDLE, false);
         }
         spoCmdAudio->vSetModuleState(enAudStream, bModuleActive);
      }
      //! Check if module active state is valid based on current audio context and only then forward it.
      else
      {
         t_SptrBDCLContextHandler spoBDCLCtxtHandler = spi_tclBDCLContextHandler::getInstance();

         trBdclAudioStates rCurAudStates;
         rCurAudStates.enMuteState = enGetCurMuteState();
         rCurAudStates.enTakeConstraint = enGetCurTakeConstraint();
         rCurAudStates.enBorrowConstraint = enGetCurBorrowConstraint();
         rCurAudStates.enMixConstraint = enGetCurMixConstraint();
         rCurAudStates.enMusicState = enGetCurAudioState(e8CL_AUDTYPE_MEDIA);
         rCurAudStates.enVRState = enGetCurAudioState(e8CL_AUDTYPE_VOICE);
         rCurAudStates.enTTSState = enGetCurAudioState(e8CL_AUDTYPE_TTS);

         trBdclAudioStates rNewAudStates;
         tenBdclAudioResponse enAudResponse = e8CL_AUD_RESP_INVALID;

         trBdclAudioRequest rAudioRequest;
         rAudioRequest.enAudRequest = e8CL_AUD_REQ_START;
         rAudioRequest.enAudStream = enAudStream;

         if (
            (spoBDCLCtxtHandler)
            &&
            (spoBDCLCtxtHandler->bGetAudioResponse(rAudioRequest, rCurAudStates, rNewAudStates, enAudResponse))
            &&
            (e8CL_AUD_RESP_INVALID != enAudResponse)
            )
         {
            ETG_TRACE_USR4(("[DESC]::vValidateModuleState: Audio response found..."));

            //@Note: Update the new device audio states after notifying module state
            if (e8CL_AUD_RESP_DENY == enAudResponse)
            {
               vDenyAudioRequest(rAudioRequest);
            }
            else
            {
               //! TODO - temporary solution to forward request to Audio class via CmdAudio class.
               //! Needs to be redesigned.
               spoCmdAudio->vSetModuleState(enAudStream, bModuleActive);
            }

            vSetCurAudioStates(rNewAudStates, false, enAudStream);
         }
      }//if (bModuleActive)
   }//if (spoCmdAudio)
}

/***************************************************************************
** FUNCTION:  t_Void vProcessAudioStartRequest()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vProcessAudioStartRequest(
      const trBdclAudioRequest& crfrAudioRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vProcessAudioStartRequest entered"));

   t_SptrBDCLContextHandler spoBDCLCtxtHandler = spi_tclBDCLContextHandler::getInstance();

   trBdclAudioStates rCurAudStates;
   rCurAudStates.enMuteState = enGetCurMuteState();
   rCurAudStates.enTakeConstraint = enGetCurTakeConstraint();
   rCurAudStates.enBorrowConstraint = enGetCurBorrowConstraint();
   rCurAudStates.enMixConstraint = enGetCurMixConstraint();
   rCurAudStates.enMusicState = enGetCurAudioState(e8CL_AUDTYPE_MEDIA);
   rCurAudStates.enVRState = enGetCurAudioState(e8CL_AUDTYPE_VOICE);
   rCurAudStates.enTTSState = enGetCurAudioState(e8CL_AUDTYPE_TTS);

   trBdclAudioStates rNewAudStates;
   tenBdclAudioResponse enAudResponse = e8CL_AUD_RESP_INVALID;

   if (
      (spoBDCLCtxtHandler)
      &&
      (spoBDCLCtxtHandler->bGetAudioResponse(crfrAudioRequest, rCurAudStates, rNewAudStates, enAudResponse))
      &&
      (e8CL_AUD_RESP_INVALID != enAudResponse)
      )
   {
      ETG_TRACE_USR4(("[DESC]::vProcessAudioStartRequest: Audio response found...sending response"));

      //! Update the new device audio states
      vSetCurAudioStates(rNewAudStates, false, crfrAudioRequest.enAudStream);

      if (e8CL_AUD_RESP_DENY == enAudResponse)
      {
         vDenyAudioRequest(crfrAudioRequest);
      }
      else
      {
         vSendAudioRequest(crfrAudioRequest);
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void vProcessAudioStartRequest()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vProcessAudioStopRequest(
      const trBdclAudioRequest& crfrAudioRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vProcessAudioStopRequest entered"));

   //! Clear the audio states
   switch (crfrAudioRequest.enAudStream)
   {
      case e8CL_AUDTYPE_MEDIA:
      {
         if (e8CL_AUD_ST_PAUSED != enGetCurAudioState(e8CL_AUDTYPE_MEDIA))
         {
            vSetCurAudioState(e8CL_AUDTYPE_MEDIA, e8CL_AUD_ST_IDLE, false);
         }
      }
         break;
      case e8CL_AUDTYPE_TTS:
      {
         vSetCurAudioState(e8CL_AUDTYPE_TTS, e8CL_AUD_ST_IDLE, false);
      }
         break;
      case e8CL_AUDTYPE_VOICE:
      case e8CL_AUDTYPE_MIC:
      {
         if ((e8CL_AUD_STREAM_CLOSED == m_mapAudStreamState[e8CL_AUDTYPE_VOICE]) &&
               (e8CL_AUD_STREAM_CLOSED == m_mapAudStreamState[e8CL_AUDTYPE_MIC]) &&
               (e8CL_VR_STATUS_RECORD_IDLE == m_enVRModuleStatus))
         {
            vSetCurAudioState(e8CL_AUDTYPE_VOICE, e8CL_AUD_ST_IDLE, false);
         }
      }
         break;
      default:
         ETG_TRACE_USR4(("[ERR]::vProcessAudioStopRequest: Not handled for audio stream %d",
               ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrAudioRequest.enAudStream)));
         break;
   } //switch (crfrAudioRequest.enAudStream)

   vSendAudioRequest(crfrAudioRequest);
}

/***************************************************************************
 ** FUNCTION:  t_Void vProcessAudioInterruptRequest()
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vProcessAudioInterruptRequest(
         const trBdclAudioRequest& crfrAudioRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vProcessAudioInterruptRequest entered: Stream %d ",
            ETG_ENUM(BDCL_AUDSTREAM_TYPE, crfrAudioRequest.enAudStream)));

   if (e8CL_AUDTYPE_VOICE == crfrAudioRequest.enAudStream)
   {
      if ((e8CL_AUD_STREAM_CLOSED == m_mapAudStreamState[e8CL_AUDTYPE_VOICE]) &&
            (e8CL_AUD_STREAM_CLOSED == m_mapAudStreamState[e8CL_AUDTYPE_MIC]) &&
            (e8CL_VR_STATUS_RECORD_IDLE == m_enVRModuleStatus))
      {
         vSetCurAudioState(e8CL_AUDTYPE_VOICE, e8CL_AUD_ST_IDLE, false);
      }
   }

   vSendAudioRequest(crfrAudioRequest);
}

/***************************************************************************
** FUNCTION:  t_Void vDenyAudioRequest()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vDenyAudioRequest(const trBdclAudioRequest& crfrAudioRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vDenyAudioRequest entered"));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   t_SptrBDCLCmdSession spoCmdSession = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdSessionInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!spoCmdSession);
   if (spoCmdSession)
   {
      switch (crfrAudioRequest.enAudStream)
      {
         case e8CL_AUDTYPE_MEDIA:
            spoCmdSession->vSetMusicState(e8CL_MUSIC_STATUS_IDLE);
            break;
         case e8CL_AUDTYPE_VOICE:
         case e8CL_AUDTYPE_MIC:
            spoCmdSession->vSetVRState(e8CL_VR_STATUS_RECORD_IDLE);
            break;
         case e8CL_AUDTYPE_TTS:
            ETG_TRACE_USR4(("[DESC]::Nothing to be done. TTS stream will be closed by phone eventually."));
            break;
         default:
            ETG_TRACE_ERR(("[ERR]::vDenyAudioRequest: Invalid audio stream %d",
                  crfrAudioRequest.enAudStream));
            break;
      }
   } //if (spoCmdSession)
}

/***************************************************************************
** FUNCTION:  trBdclAudioStates enGetCurAudioStates()
***************************************************************************/
trBdclAudioStates spi_tclBDCLAudioResourceMngr::enGetCurAudioStates() const
{
   trBdclAudioStates rAudStates;
   rAudStates.enMusicState = m_DevAudStates.at(e8CL_AUDTYPE_MEDIA);
   rAudStates.enVRState = m_DevAudStates.at(e8CL_AUDTYPE_VOICE);
   rAudStates.enTTSState = m_DevAudStates.at(e8CL_AUDTYPE_TTS);

   ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::enGetCurAudioStates left with: "
         "MusicState : %d, VRState : %d, TTSState : %d)",
         ETG_ENUM(BDCL_AUD_STATE, rAudStates.enMusicState),
         ETG_ENUM(BDCL_AUD_STATE, rAudStates.enVRState),
         ETG_ENUM(BDCL_AUD_STATE, rAudStates.enTTSState)));
   return rAudStates;
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurAudioStates()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurAudioStates(trBdclAudioStates rAudStates,
      t_Bool bSetByAccessory)
{
   ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::vSetCurAudioStates entered"));

   vSetCurMuteState(rAudStates.enMuteState);
   vSetCurTakeConstraint(rAudStates.enTakeConstraint);
   vSetCurBorrowConstraint(rAudStates.enBorrowConstraint);
   vSetCurMixConstraint(rAudStates.enMixConstraint);
   vSetCurAudioState(e8CL_AUDTYPE_VOICE, rAudStates.enVRState, bSetByAccessory);
   vSetCurAudioState(e8CL_AUDTYPE_TTS, rAudStates.enTTSState, bSetByAccessory);
   vSetCurAudioState(e8CL_AUDTYPE_MEDIA, rAudStates.enMusicState, bSetByAccessory);
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurAudioStates()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurAudioStates(trBdclAudioStates rAudStates,
      t_Bool bSetByAccessory, tenBdclAudStreamType enChangedAudStream)
{
   ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::vSetCurAudioStates entered"));

   vSetCurMuteState(rAudStates.enMuteState);
   vSetCurTakeConstraint(rAudStates.enTakeConstraint);
   vSetCurBorrowConstraint(rAudStates.enBorrowConstraint);
   vSetCurMixConstraint(rAudStates.enMixConstraint);

   trBdclAudioStates rCurAudStates = enGetCurAudioStates();
   switch (enChangedAudStream)
   {
      case e8CL_AUDTYPE_MEDIA:
      {
         vSetCurAudioState(e8CL_AUDTYPE_VOICE, rAudStates.enVRState, true);
         vSetCurAudioState(e8CL_AUDTYPE_TTS, rAudStates.enTTSState, true);
         vSetCurAudioState(e8CL_AUDTYPE_MEDIA, rAudStates.enMusicState, bSetByAccessory);
      }
         break;
      case e8CL_AUDTYPE_TTS:
      {
         vSetCurAudioState(e8CL_AUDTYPE_VOICE, rAudStates.enVRState, true);
         vSetCurAudioState(e8CL_AUDTYPE_TTS, rAudStates.enTTSState, bSetByAccessory);
         vSetCurAudioState(e8CL_AUDTYPE_MEDIA, rAudStates.enMusicState, true);
      }
         break;
      case e8CL_AUDTYPE_VOICE:
      case e8CL_AUDTYPE_MIC:
      {
         vSetCurAudioState(e8CL_AUDTYPE_VOICE, rAudStates.enVRState, bSetByAccessory);
         vSetCurAudioState(e8CL_AUDTYPE_TTS, rAudStates.enTTSState, true);
         vSetCurAudioState(e8CL_AUDTYPE_MEDIA, rAudStates.enMusicState, true);
      }
         break;
      default:
         ETG_TRACE_ERR(("[ERR]::vSetCurAudioStates: Invalid stream type"));
         break;
   } //switch (enChangedAudStream)
}

/***************************************************************************
** FUNCTION:  tenBdclAudioState enGetCurAudioState()
***************************************************************************/
tenBdclAudioState spi_tclBDCLAudioResourceMngr::enGetCurAudioState(
      tenBdclAudStreamType enAudStream) const
{
   tenBdclAudioState enAudState = m_DevAudStates.at(enAudStream);
   /*ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::enGetCurAudioState left with: %d (for Stream %d)",
         ETG_ENUM(BDCL_AUD_STATE, enAudState),
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));*/
   return enAudState;
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurAudioState()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurAudioState(
      tenBdclAudStreamType enAudStream,
      tenBdclAudioState enAudState,
      t_Bool bSetByAccessory)
{
   /*ETG_TRACE_USR4(("spi_tclBDCLAudioResourceMngr::vSetCurAudioState entered: %d for Stream %d",
         ETG_ENUM(BDCL_AUD_STATE, enAudState),
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));*/

   if ((e8CL_AUD_ST_DEFAULT != enAudState) && (m_DevAudStates[enAudStream] != enAudState))
   {
      ETG_TRACE_USR2(("[DESC]::Changing audio state from %d to %d for stream %d, IsSetByAccessory = %d",
            ETG_ENUM(BDCL_AUD_STATE, m_DevAudStates[enAudStream]),
            ETG_ENUM(BDCL_AUD_STATE, enAudState),
            ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream),
            ETG_ENUM(BOOL, bSetByAccessory)));

      t_Bool bIsUnpause = (e8CL_AUD_ST_PAUSED == m_DevAudStates[enAudStream]) && (e8CL_AUD_ST_PLAYING == enAudState);

      switch (enAudStream)
      {
         case e8CL_AUDTYPE_MEDIA:
         case e8CL_AUDTYPE_TTS:
         {
            m_DevAudStates[enAudStream] = enAudState;
         }
            break;
         case e8CL_AUDTYPE_VOICE:
         case e8CL_AUDTYPE_MIC:
         {
            //@Note: Mic & Voice states are kept same since they are used as a single VR state.
            m_DevAudStates[e8CL_AUDTYPE_MIC] = enAudState;
            m_DevAudStates[e8CL_AUDTYPE_VOICE] = enAudState;

            tenBdclVRState enVRState = (e8CL_AUD_ST_PLAYING == enAudState) ?
                  (e8CL_VR_STATUS_RECORD_RUNNING) : (e8CL_VR_STATUS_RECORD_IDLE);
            vPostVRStatus(enVRState);
         }
            break;
         default:
            ETG_TRACE_ERR(("[ERR]::vSetCurAudioState: Invalid audio stream %d",
                     ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
            break;
      }//switch (enAudStream)

      t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
      t_SptrBDCLCmdSession spoCmdSession = (spoBDCLManager) ? (spoBDCLManager->spoGetCmdSessionInstance()) : (nullptr);
      SPI_NORMAL_ASSERT(!spoCmdSession);
      if ((spoCmdSession) && ((bSetByAccessory) || (bIsUnpause)))
      {
         if (bIsUnpause)
         {
            ETG_TRACE_USR4(("[DESC]::Setting audio state to phone since audio was paused by accessory"));
         }

         switch (enAudStream)
         {
            case e8CL_AUDTYPE_MEDIA:
            {
               tenBdclMusicState enMusicState = (e8CL_AUD_ST_PLAYING == enAudState) ?
                          (e8CL_MUSIC_STATUS_RUNNING) : (e8CL_MUSIC_STATUS_IDLE);
               ETG_TRACE_USR4(("[DESC]::Setting Music state %d to phone", ETG_ENUM(BDCL_MUSIC_STATE, enMusicState)));
               spoCmdSession->vSetMusicState(enMusicState);
            }
               break;
            case e8CL_AUDTYPE_VOICE:
            case e8CL_AUDTYPE_MIC:
            {
               if (e8CL_AUD_ST_IDLE == enAudState)
               {
                  ETG_TRACE_USR4(("[DESC]::Setting VR state IDLE to phone"));
                  spoCmdSession->vSetVRState(e8CL_VR_STATUS_RECORD_IDLE);
               }
               else
               {
                  ETG_TRACE_ERR(("[ERR]::Usecase not handled"));
               }
            }
               break;
            case e8CL_AUDTYPE_TTS:
            {
               if (e8CL_AUD_ST_IDLE == enAudState)
               {
                  ETG_TRACE_USR4(("[DESC]::Force closing TTS stream "));

                  //@Note: Mock TTS interrupt request from phone since there is no way to request phone to stop TTS audio.
                  trBdclAudioRequest rTTSInterruptRequest = {e8CL_AUDTYPE_TTS, e8CL_AUD_REQ_INTERRUPT};
                  vOnAudioRequest(rTTSInterruptRequest);
               }
               else
               {
                  ETG_TRACE_ERR(("[ERR]::Usecase not handled"));
               }
            }
               break;
            default:
               ETG_TRACE_ERR(("[ERR]::vSetCurAudioState: Invalid audio stream %d",
                        ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
               break;
         } //switch (enAudStream)
      } //if ((bSetByAccessory) && (spoCmdSession))
   } //if ((e8CL_AUD_ST_DEFAULT != enAudState) && ...)
}

/***************************************************************************
** FUNCTION:  tenAudioMuteState enGetCurMuteState()
***************************************************************************/
tenAudioMuteState spi_tclBDCLAudioResourceMngr::enGetCurMuteState() const
{
   /*ETG_TRACE_USR4(("spi_tclBDCLAudioResourceMngr::enGetCurMuteState left with %d",
         ETG_ENUM(AUD_MUTE_STATE, m_enMuteState)));*/
   return m_enMuteState;
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurMuteState()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurMuteState(
      tenAudioMuteState enMuteState)
{
   /*ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::vSetCurMuteState entered with %d",
         ETG_ENUM(AUD_MUTE_STATE, enMuteState)));*/
   if ((e8AUD_MUTE_DEFAULT != enMuteState) && (m_enMuteState != enMuteState))
   {
      ETG_TRACE_USR2(("[DESC]::Changing Mute state to %d ", ETG_ENUM(AUD_MUTE_STATE, enMuteState)));
      m_enMuteState = enMuteState;
   }
}

/***************************************************************************
** FUNCTION:  tenAudioContextType enGetCurTakeConstraint()
***************************************************************************/
tenBdclAudioConstraint spi_tclBDCLAudioResourceMngr::enGetCurTakeConstraint() const
{
   ETG_TRACE_USR4(("spi_tclBDCLAudioResourceMngr::enGetCurTakeConstraint left with %d",
         ETG_ENUM(BDCL_AUD_CONSTRAINT, m_enTakeConstraint)));
   return m_enTakeConstraint;
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurTakeConstraint()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurTakeConstraint(
      tenBdclAudioConstraint enAudConstraint)
{
   /*ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::vSetCurTakeConstraint entered with %d",
         ETG_ENUM(BDCL_AUD_CONSTRAINT, enAudConstraint)));*/
   if ((e8CL_CONSTRAINT_DEFAULT != enAudConstraint) && (m_enTakeConstraint != enAudConstraint))
   {
      ETG_TRACE_USR2(("[DESC]::Changing Take constraint to %d ", ETG_ENUM(BDCL_AUD_CONSTRAINT, enAudConstraint)));
      m_enTakeConstraint = enAudConstraint;
   }
}

/***************************************************************************
** FUNCTION:  tenAudioContextType enGetCurBorrowConstraint()
***************************************************************************/
tenBdclAudioConstraint spi_tclBDCLAudioResourceMngr::enGetCurBorrowConstraint() const
{
   ETG_TRACE_USR4(("spi_tclBDCLAudioResourceMngr::enGetCurBorrowConstraint left with %d",
         ETG_ENUM(BDCL_AUD_CONSTRAINT, m_enBorrowConstraint)));
   return m_enBorrowConstraint;
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurBorrowConstraint()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurBorrowConstraint(
      tenBdclAudioConstraint enAudConstraint)
{
   /*ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::vSetCurBorrowConstraint entered with %d",
         ETG_ENUM(BDCL_AUD_CONSTRAINT, enAudConstraint)));*/
   if ((e8CL_CONSTRAINT_DEFAULT != enAudConstraint) && (m_enBorrowConstraint != enAudConstraint))
   {
      ETG_TRACE_USR2(("[DESC]::Changing Borrow constraint to %d ", ETG_ENUM(BDCL_AUD_CONSTRAINT, enAudConstraint)));
      m_enBorrowConstraint = enAudConstraint;
   }
}

/***************************************************************************
** FUNCTION:  tenAudioContextType enGetCurMixConstraint()
***************************************************************************/
tenBdclAudioConstraint spi_tclBDCLAudioResourceMngr::enGetCurMixConstraint() const
{
   ETG_TRACE_USR4(("spi_tclBDCLAudioResourceMngr::enGetCurMixConstraint left with %d",
         ETG_ENUM(BDCL_AUD_CONSTRAINT, m_enMixConstraint)));
   return m_enMixConstraint;
}

/***************************************************************************
** FUNCTION:  t_Void vSetCurMixConstraint()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vSetCurMixConstraint(
      tenBdclAudioConstraint enAudConstraint)
{
   /*ETG_TRACE_USR2(("spi_tclBDCLAudioResourceMngr::vSetCurMixConstraint entered with %d",
         ETG_ENUM(BDCL_AUD_CONSTRAINT, enAudConstraint)));*/
   if ((e8CL_CONSTRAINT_DEFAULT != enAudConstraint) && (m_enMixConstraint != enAudConstraint))
   {
      ETG_TRACE_USR2(("[DESC]::Changing Mix constraint to %d ", ETG_ENUM(BDCL_AUD_CONSTRAINT, enAudConstraint)));
      m_enMixConstraint = enAudConstraint;
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vRegisterCallbacks(const trBdclResMngrCbs& corfrResMngrCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vRegisterCallbacks entered"));
   m_rResMngrCbs = corfrResMngrCbs;
}

/***************************************************************************
** FUNCTION:  t_Void vPostVRStatus()
***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vPostVRStatus(tenBdclVRState enBdclVRState)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vPostVRStatus entered"));

   if (NULL != m_rResMngrCbs.fvBdclVRState)
   {
      m_rResMngrCbs.fvBdclVRState(enBdclVRState);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void vOnDeviceDeselection()
 ***************************************************************************/
t_Void spi_tclBDCLAudioResourceMngr::vOnDeviceDeselection()
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioResourceMngr::vOnDeviceDeselection() entered"));

   //! Clear module states
   m_enMusicModuleStatus = e8CL_MUSIC_STATUS_IDLE;
   m_enVRModuleStatus = e8CL_VR_STATUS_RECORD_IDLE;
   m_enPhoneModuleStatus = e8CL_PHONE_STATUS_IDLE;

}

//lint restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
