/*!
 *******************************************************************************
 * \file              spi_tclBDCLInputHandler.cpp
 * \brief             SPI input handler for BDCL devices
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Input handler class to send input events from Head Unit to
                BDCL mobile device
 AUTHOR:        Sameer Chandra (RBEI/ECP2)
 COPYRIGHT:     &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 06.03.2015 |  Sameer Chandra              | Initial Version
 07.07.2015 |  Shiva Kumar G               | Dynamic display configuration
 17.07.2015	|  Sameer Chandra              | Knob Encoder Implementation
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "BDCLTypes.h"
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLInputHandler.h"
#include "spi_tclMediator.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclBDCLInputHandler.cpp.trc.h"
#endif
#endif

static const t_U8 cou8EnableVerbose   = 0;
static const t_U8 cou8TouchMaximum    = 2;
static const t_U8 cou8TriggerInterval = 50;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/***************************************************************************
** FUNCTION:  spi_tclBDCLInputHandler::spi_tclBDCLInputHandler()
***************************************************************************/
spi_tclBDCLInputHandler::spi_tclBDCLInputHandler()
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::spi_tclBDCLInputHandler entered "));

   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();
   if (spoBDCLManager)
   {
      m_spoCmdInput = spoBDCLManager->spoGetCmdInputInstance();
   }
   SPI_NORMAL_ASSERT((!spoBDCLManager) || (!m_spoCmdInput));

   trNotifyDevAppStateCallbacks rAppStateCbs;
   rAppStateCbs.fvAppStateStatus = std::bind(
         &spi_tclBDCLInputHandler::vOnSpeechAppUpdateCb, this,
         std::placeholders::_1,
         std::placeholders::_2,
         std::placeholders::_3);
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vRegisterCallbacks(rAppStateCbs);
   } // if (NULL != poMediator)

}

/***************************************************************************
** FUNCTION:  spi_tclBDCLInputHandler::~spi_tclBDCLInputHandler()
***************************************************************************/
spi_tclBDCLInputHandler::~spi_tclBDCLInputHandler()
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::~spi_tclBDCLInputHandler entered"));
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLInputHandler::vProcessKeyEvents()
***************************************************************************/
t_Void spi_tclBDCLInputHandler::vProcessKeyEvents(t_U32 u32DeviceHandle,
         tenKeyMode enKeyMode, tenKeyCode enKeyCode) const
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::vProcessKeyEvents entered"));
   ETG_TRACE_USR2(("[PARAM]::vProcessKeyEvents: Key Mode = %d Key Code = %d",
      ETG_ENUM(KEY_MODE, enKeyMode), ETG_ENUM(SPI_KEY_CODE, enKeyCode)));

   if (NULL != m_spoCmdInput)
   {
         m_spoCmdInput->vReportKey(enKeyMode, enKeyCode);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLInputHandler::vProcessKnobKeyEvents()
***************************************************************************/
t_Void spi_tclBDCLInputHandler::vProcessKnobKeyEvents(t_U32 u32DeviceHandle,t_S8 s8EncoderDeltaCnt) const
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::vProcessKnobKeyEvents entered "));
   ETG_TRACE_USR2(("[DESC]:vProcessKnobKeyEvents: Device ID = 0x%x Encoder Delta Count = %d",
             u32DeviceHandle, s8EncoderDeltaCnt));

   ETG_TRACE_ERR(("[ERR]::spi_tclBDCLInputHandler::vProcessKnobKeyEvents: Not implemented "));
   //TODO - send knob events via key code?
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLInputHandler::vRegisterInputCallbacks()
***************************************************************************/
t_Void spi_tclBDCLInputHandler::vRegisterInputCallbacks(const trInputCallbacks& corfrInputCallbacks)
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::vRegisterVideoCallbacks entered "));

   m_rInputCallbacks = corfrInputCallbacks;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLInputHandler::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclBDCLInputHandler::vSelectDevice(const t_U32 cou32DevId,
      const tenDeviceConnectionReq coenConnReq)
{
	/*lint -esym(40,fvSelectDeviceResp) fvSelectDeviceResp is not declared */
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler:vSelectDevice Device Id - 0x%x Connection Request - %d",
            cou32DevId, ETG_ENUM(CONNECTION_REQ, coenConnReq)));

   tenErrorCode enErrCode = e8UNKNOWN_ERROR;
   if (m_spoCmdInput)
   {
      if (e8DEVCONNREQ_SELECT == coenConnReq)
      {
         //! Initialize the Input Source
         enErrCode = (m_spoCmdInput->bInitialize()) ? e8NO_ERRORS : e8SELECTION_FAILED;
      }
      else if (e8DEVCONNREQ_DESELECT == coenConnReq)
      {
         enErrCode = e8NO_ERRORS;
         m_spoCmdInput->vUninitialize();
      }
   }

   ETG_TRACE_USR2(("[DESC]::spi_tclBDCLInputHandler::vSelectDevice left with Error Code - %d",
         ETG_ENUM(ERROR_CODE, enErrCode)));

   if (NULL != m_rInputCallbacks.fvSelectDeviceResp)
   {
      (m_rInputCallbacks.fvSelectDeviceResp)(cou32DevId, enErrCode);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclBDCLInputHandler::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclBDCLInputHandler::vOnSelectDeviceResult(const t_U32 cou32DevId,
            const tenDeviceConnectionReq coenConnReq,
            const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::vOnSelectDeviceResult:"
         "Device handle-0x%x Connection Request-%d, Response Code-%d",
         cou32DevId, ETG_ENUM(CONNECTION_REQ,coenConnReq), ETG_ENUM(RESPONSE_CODE,coenRespCode)));

   if ((e8FAILURE == coenRespCode) && (e8DEVCONNREQ_SELECT == coenConnReq) && (m_spoCmdInput))
   {
      //! Un-initialize the InputSource
      m_spoCmdInput->vUninitialize();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLInputHandler::vOnSpeechAppUpdateCb()
**************************************************************************/
t_Void spi_tclBDCLInputHandler::vOnSpeechAppUpdateCb(tenSpeechAppState enSpeechAppState,
                                                     tenPhoneAppState enPhoneAppState,
                                                     tenNavAppState enNavAppState)
{
   ETG_TRACE_USR1(("spi_tclBDCLInputHandler::vCbPostSpeechAppState() entered "));
   SPI_INTENTIONALLY_UNUSED(enPhoneAppState);
   SPI_INTENTIONALLY_UNUSED(enNavAppState);
   if (NULL != m_spoCmdInput)
   {
      m_spoCmdInput->vReportVRStatus(enSpeechAppState);
   }

}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
