/*!
 *******************************************************************************
 * \file             spi_tclBDCLDataService.cpp
 * \brief            Implements the data service functionality for Baidu Carlife
                     using interface to CL Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Data service implementation for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 04.05.2017 |  pok6kor             | Initial Version

 \endverbatim
 ******************************************************************************/
#include "spi_tclBDCLManager.h"
#include "spi_tclBDCLDataService.h"

#include <chrono>
#include <boost/bind.hpp>
#include "Timer.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
#include "trcGenProj/Header/spi_tclBDCLDataService.cpp.trc.h"
#endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
#define dPItoDegree (180/3.14159265358979323846)         //used for converting pi to decimal degrees

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
//! Vehicle GPS timer data
static timer_t srVehGPSTimerID;
static t_Bool sbVehGPSTimerRunning = false;

//! Vehicle Speed timer data
static timer_t srVehSpeedTimerID;
static t_Bool sbVehSpeedTimerRunning = false;

//! Vehicle Gyroscope timer data
static timer_t srVehGyroscopeTimerID;
static t_Bool sbVehGyroscopeTimerRunning = false;

//! Vehicle Acceleration timer data
static timer_t srVehAccelerationTimerID;
static t_Bool sbVehAccelerationTimerRunning = false;

//! Vehicle Gear timer data
static timer_t srVehGearTimerID;
static t_Bool sbVehGearTimerRunning = false;

static const trBDCLSelectedDeviceInfo corEmptyDeviceInfo;

//Default frequencey is 1 hz. So Default timer interval is (1/1)*1000

static const t_U32 scoDefaultFrequencyInHz = 1;
static const t_U32 scoDefaultTimerIntervalInMs = 1000;

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::spi_tclBDCLDataService()
 ***************************************************************************/
spi_tclBDCLDataService::spi_tclBDCLDataService(const trDataServiceCb& rfcorDataServiceCb):
                                    m_rDataServiceCb(rfcorDataServiceCb),
                                    m_s16VehicleSpeed(0),
                                    m_enGearPosition(e8GEAR_POS_UNKNOWN)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService() entered "));
   m_vecGyroscopeData.clear();
   m_vecAccelerationData.clear();
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::~spi_tclBDCLDataService()
 ***************************************************************************/
spi_tclBDCLDataService::~spi_tclBDCLDataService()
{
   ETG_TRACE_USR1(("[DESC]:~spi_tclBDCLDataService() entered "));
   vStopVehicleUpdate(e8CAR_DATA_VELOCITY);
   vStopVehicleUpdate(e8CAR_DATA_GEAR);
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLDataService::bInitialise()
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bInitialise()
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::bInitialise() entered "));
   t_Bool bInitSuccess = false;
   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();

   if (NULL != spoBDCLManager)
   {
      bInitSuccess = true;
      spoBDCLManager->bRegisterObject((spi_tclBDCLRespCarData*)this);
      // Initializations to be done which are not specific to device connected/selected
      m_spoCmdCarData = spoBDCLManager->spoGetCmdCarDataInstance();
   }// if (NULL != spoBDCLManager)

   SPI_NORMAL_ASSERT((!spoBDCLManager) || (!m_spoCmdCarData));

   ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::bInitialise() left with result = %u ", bInitSuccess));
   return bInitSuccess;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLDataService::bUninitialise()
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bUninitialise()
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::bUninitialise() entered "));

   t_Bool bUnInitSuccess = false;
   t_SptrBDCLManager spoBDCLManager = spi_tclBDCLManager::getInstance();

   if ((spoBDCLManager) &&
            (spoBDCLManager->bUnRegisterObject((spi_tclBDCLRespCarData*) this))
            && (m_rSelDevInfo.u32DeviceHandle))
   {
      //unsubscribe from previous subscriptions and stop timers.
      //! Clear Selected device's info
      m_rSelDevInfo = corEmptyDeviceInfo;

      //! Stop timers for fixed frequency updates
      vStopVehicleUpdate(e8CAR_DATA_GPS);
      vStopVehicleUpdate(e8CAR_DATA_VELOCITY);
      vStopVehicleUpdate(e8CAR_DATA_GYROSCOPE);
      vStopVehicleUpdate(e8CAR_DATA_ACCELERATION);
      vStopVehicleUpdate(e8CAR_DATA_GEAR);

      //! Unsubscribe data
      vDataSubscribe(false);

      bUnInitSuccess = true;
      ETG_TRACE_USR4(("spi_tclBDCLDataService::bUninitialise():unsubscribed data and stopped timers."));
   }
   ETG_TRACE_USR2(("[DESC]:bUninitialise() left with result = %u", bUnInitSuccess));

   return bUnInitSuccess;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclBDCLDataService::vSetDataSubscription(...)
***************************************************************************/
t_Void spi_tclBDCLDataService::vDataSubscribe(t_Bool bSubscriptionState)
{
   /*lint -esym(40,fvSubscribeForData) fvSubscribeForData is not declared */
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vDataUnSubscribe() entered: bSubscriptionState = %u ",
      ETG_ENUM(BOOL, bSubscriptionState)));

   if (NULL != m_rDataServiceCb.fvSubscribeForData)
   {
      m_rDataServiceCb.fvSubscribeForData(bSubscriptionState, e8VEHICLE_DATA);

      if ((true == m_rBDCLDataServiceConfigData.bLocDataAvailable) && (
            false == m_rBDCLDataServiceConfigData.bDeadReckonedData))
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionState, e8GNSS_DATA);
      }
      if (m_rBDCLDataServiceConfigData.bAcclData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionState, e8ACCELEROMETER_DATA);
      }
      if (m_rBDCLDataServiceConfigData.bGyroData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionState, e8GYROSCOPE_DATA);
      }
   }//if (NULL != m_rDataServiceCb.fvSubscribeForData)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vSelectDevice(t_U32...)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vSelectDevice(const t_U32 cou32DevId,
         tenDeviceConnectionReq enConnReq)
{
   ETG_TRACE_USR1(("[PARAM]:spi_tclBDCLDataService::vSelectDevice: Device ID = 0x%x Selection Type = %d ",
            cou32DevId, ETG_ENUM(CONNECTION_REQ, enConnReq)));

   if (m_spoCmdCarData)
   {
      if (e8DEVCONNREQ_SELECT == enConnReq)
      {
         //Initialize CarData session
         t_Bool bResult = m_spoCmdCarData->bInitialize();
         ETG_TRACE_USR4(("[DESC]::spi_tclBDCLDataService:vSelectDevice: Initialize result = %d", ETG_ENUM(BOOL, bResult)));
         vDataSubscribe(false);
      }//if( coenConnReq == e8DEVCONNREQ_SELECT )
      else
      {
         vDataSubscribe(false);
         //! Stop timers for fixed frequency updates
         vStopVehicleUpdate(e8CAR_DATA_GPS);
         vStopVehicleUpdate(e8CAR_DATA_VELOCITY);
         vStopVehicleUpdate(e8CAR_DATA_GYROSCOPE);
         vStopVehicleUpdate(e8CAR_DATA_ACCELERATION);
         vStopVehicleUpdate(e8CAR_DATA_GEAR);
         m_spoCmdCarData->vUninitialize();
       }//else
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLDataService::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vOnSelectDeviceResult(t_U32 u32DeviceHandle,
         tenDeviceConnectionReq coenConnReq,
         tenResponseCode coenRespCode)
{ 
   ETG_TRACE_USR1(("[PARAM]:spi_tclBDCLDataService:vOnSelectDeviceResult: Device ID = 0x%x",u32DeviceHandle));
   m_rSelDevInfo = corEmptyDeviceInfo;
   m_rSelDevInfo.u32DeviceHandle = u32DeviceHandle;

   if ((e8DEVCONNREQ_SELECT == coenConnReq) && (e8FAILURE == coenRespCode) && (m_spoCmdCarData))
   {
      m_spoCmdCarData->vUninitialize();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnDeselectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vOnDeselectDeviceResult() entered: u32DeviceHandle = %u ",
            u32DeviceHandle));
   //! Clear Selected device's info
   m_rSelDevInfo = corEmptyDeviceInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vSetSensorDataAvailablility(...)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vSetSensorDataAvailablility(
         const trDataServiceConfigData& rfrDataServiceConfigInfo)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vSetSensorDataAvailablility() entered "));

   //! Store config data
   m_rBDCLDataServiceConfigData = rfrDataServiceConfigInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnData(const trSensorData& rfcorSensorData)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vOnData(const trSensorData& rfcorSensorData)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vOnData for GPS entered"));
   t_Bool bIsGPSFixAvailable = (e8GNSSQUALITY_NOFIX != rfcorSensorData.enGnssQuality);
   if ((true == m_rBDCLDataServiceConfigData.bLocDataAvailable) &&
            (false == m_rBDCLDataServiceConfigData.bDeadReckonedData) && (0 != m_rSelDevInfo.u32DeviceHandle)
            && (true == bIsGPSFixAvailable))
   {
      ETG_TRACE_USR4(("[PARAM]::spi_tclBDCLDataService::vOnData(): Raw data "
               "Latitude = %f, Longitude = %f, Heading = %f, Speed = %f, "
               "Altitude = %f, SatellitesUsed = %d, PosixTime = 0x%x,  ",
               rfcorSensorData.dLatitude, rfcorSensorData.dLongitude, rfcorSensorData.fHeading, rfcorSensorData.fSpeed,
               rfcorSensorData.fAltitude, rfcorSensorData.u16SatellitesUsed, rfcorSensorData.PosixTime));
      m_rGPS.s32AntennaState = m_rBDCLDataServiceConfigData.bLocDataAvailable;
      m_rGPS.s32SignalQuality = 0;
      m_rGPS.s32Latitude  = static_cast<t_S32>((fabs(((rfcorSensorData.dLatitude)*(dPItoDegree))))*1000000);
      m_rGPS.s32Longitude  = static_cast<t_S32>((fabs(((rfcorSensorData.dLongitude)*(dPItoDegree))))*1000000);

      //!fabs converts negative values to positive. Retain the negative values of location.
      m_rGPS.s32Latitude  = (0 > rfcorSensorData.dLatitude)?((-1)*(m_rGPS.s32Latitude )):(m_rGPS.s32Latitude);
      m_rGPS.s32Longitude = (0 > rfcorSensorData.dLongitude)?((-1)*(m_rGPS.s32Longitude)):(m_rGPS.s32Longitude);

      m_rGPS.s32Height = static_cast<t_S32>((rfcorSensorData.fAltitude) * 10);
      m_rGPS.u32Speed = static_cast<t_U32>((rfcorSensorData.fSpeed) * 100 * 3.6);// sent in km/h
      m_rGPS.u32Heading = static_cast<t_U32>((fmod((((rfcorSensorData.fHeading)*(dPItoDegree)) + 360.0), 360.0))*10);
      m_rGPS.u32Year = 0;
      m_rGPS.u32Month = 0;
      m_rGPS.u32Day = 0;
      m_rGPS.u32Hrs = 0;
      m_rGPS.u32Min = 0;
      m_rGPS.u32Sec = 0;
      m_rGPS.u32Fix = 0;
      m_rGPS.u32hdop = rfcorSensorData.dHDOP;
      m_rGPS.u32Pdop = 0;
      m_rGPS.u32Vdop = 0;
      m_rGPS.u32SatsUsed = static_cast<t_U32>(rfcorSensorData.u16SatellitesUsed);
      m_rGPS.u32SatsVisible = 0;
      m_rGPS.u32HorPosError = 0;
      m_rGPS.u32VertPosError = 0;
      m_rGPS.s32NorthSpeed = 0;
      m_rGPS.s32EastSpeed = 0 ;
      m_rGPS.s32VertSpeed = 0;
      m_rGPS.u64TimeStamp = (rfcorSensorData.PosixTime) * 1000000000;

      ETG_TRACE_USR4(("[PARAM]::spi_tclBDCLDataService::vOnData(): After calculation "
               "Latitude = %d, Longitude = %d, Heading = %d, Speed = %d, "
               "Altitude = %d, SatellitesUsed = %d, PosixTime = %d,  ",
               m_rGPS.s32Latitude, m_rGPS.s32Longitude,m_rGPS.u32Heading, m_rGPS.u32Speed ,
               m_rGPS.s32Height, m_rGPS.u32SatsUsed, m_rGPS.u64TimeStamp ));

      //! Send Vehicle GPS data if timer is not running
      if (!sbVehGPSTimerRunning)
      {
         vReportVehicleGPSInfo();
      }
   }//if ((true == m_rBDCLDataServiceConfigData.bLocDataAvailable)&&...)
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleGPSInfo()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vReportVehicleGPSInfo()
{
   //! Send Vehicle GPS data
   if ((NULL != m_spoCmdCarData) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      t_Bool bRetValue =  m_spoCmdCarData->bSendCarGPS(m_rGPS);
      ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vReportVehicleGPSInfo return value on sending GPS data:%d",bRetValue));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnAccSensorData
 ** (const std::vector<trAccSensorData>& corfvecrAccSensorData)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData)
{
   if ((0 != m_rSelDevInfo.u32DeviceHandle))
   {
      ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vOnAccSensorData Acc list size = %d", corfvecrAccSensorData.size()));
      m_vecAccelerationData = corfvecrAccSensorData;

      //! Send Vehicle Acceleration data if timer is not running
      if(!sbVehAccelerationTimerRunning)
      {
         vReportVehicleAccInfo();
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleAccInfo()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vReportVehicleAccInfo()
{
   //! Send Vehicle Acceleration data
   if ((NULL != m_spoCmdCarData) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      for(tU16 u16Index = 0 ; (u16Index < m_vecAccelerationData.size()) && (u16Index < 20) ; ++u16Index)
      {
         ETG_TRACE_USR4_CLS((TR_CLASS_SMARTPHONEINT_SENSORDATA, "[PARAM]:spi_tclBDCLDataService::vOnAccSensorData Acc Values"
                  " s32AccZE3 = %d, s32AccYE3 = %d, s32AccXE3 = %d", m_vecAccelerationData[u16Index].fAccZ,
                  m_vecAccelerationData[u16Index].fAccY, m_vecAccelerationData[u16Index].fAccX));
         t_Bool bRetVal = m_spoCmdCarData->bSendCarAcceleration(m_vecAccelerationData[u16Index]);
         ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vOnAccSensorData:Status of Acceleration message:%d:",bRetVal));
      }
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnGyroSensorData
 ** (const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
{
   if ((0 != m_rSelDevInfo.u32DeviceHandle))
   {
      ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vOnGyroSensorData Gyro list size = %d", corfvecrGyroSensorData.size()));
      m_vecGyroscopeData = corfvecrGyroSensorData;

      //! Send Vehicle Gyro data if timer is not running
      if(!sbVehGyroscopeTimerRunning)
      {
         vReportVehicleGyroscopeInfo();
      }
   }//if ((true == m_rAAPDataServiceConfigData.bGyroData) &&...)
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleGyroscopeInfo()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vReportVehicleGyroscopeInfo()
{
   //! Send Vehicle Gyroscope data
   if ((NULL != m_spoCmdCarData) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      for (tU16 u16Index = 0; (u16Index < m_vecGyroscopeData.size()) && (u16Index < 20); ++u16Index)
      {
         // Convert to axes requested by Google and rad/s
         ETG_TRACE_USR4_CLS((TR_CLASS_SMARTPHONEINT_SENSORDATA, "[PARAM]:spi_tclAAPDataService::vOnGyroSensorData Gyro Values"
                  " s32GyroZE3 = %d, s32GyroYE3 = %d, s32GyroXE3 = %d", m_vecGyroscopeData[u16Index].fGyroZ,
                  m_vecGyroscopeData[u16Index].fGyroY, m_vecGyroscopeData[u16Index].fGyroX));
         t_Bool bRetVal = m_spoCmdCarData->bSendCarGyro(m_vecGyroscopeData[u16Index]);
         ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vOnGyroSensorData:Status of Gyro message:%d:",bRetVal));
      }
   }
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::bVehicleVelocityTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bVehicleVelocityTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1(("spi_tclBDCLDataService::bVehicleSpeedTimerCb entered "));

   spi_tclBDCLDataService* poBDCLDataService = static_cast<spi_tclBDCLDataService*> (pvObject);
   if (NULL != poBDCLDataService)
   {
      poBDCLDataService->vReportVehicleVelocityInfo();
   }

   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::vReportVehicleVelocityInfo()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vReportVehicleVelocityInfo()
{
   //! Send Vehicle speed data
   if ((NULL != m_spoCmdCarData) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      // convert speed cm/s to km/hr
      trBdclCarVelocity rCarVelocityInfo;
      rCarVelocityInfo.u32speed = ((m_s16VehicleSpeed) * 0.036);
      //rCarVelocityInfo.u64timestamp = 0; // Timestamp not expected by ADIT
      t_Bool bRetVal = m_spoCmdCarData->bSendCarVelocity(rCarVelocityInfo); //TODO: Check if the velocity needs to be recalculated based on gear position.
      ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vReportVehicleVelocityInfo:Status of Velocity message:%d:",bRetVal));
   }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vOnData(const trVehicleData rfrcVehicleData)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vOnData(const trVehicleData& rfrcVehicleData, t_Bool bSolicited)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vOnData() entered with Vehicle Data"));
   m_enGearPosition = rfrcVehicleData.enGearPosition;
   m_s16VehicleSpeed = rfrcVehicleData.s16Speed;
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::vStartVehicleUpdate
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vStartVehicleUpdate(tenBdclCarDataModuleID enModuleID)
{
   //! Start timer to send Vehicle data at the rate subscribed by the MD
   Timer* poTimer = Timer::getInstance();
   ETG_TRACE_USR1(("[PARAM]:spi_tclBDCLDataService::vStartVehicleUpdate()  Module ID : %d" ,enModuleID));
   switch(enModuleID)
   {
      case e8CAR_DATA_GPS:
      {
         ETG_TRACE_USR1(("spi_tclBDCLDataService::vStartVehicleUpdate()  timer value is %d",m_rBDCLDataServiceConfigData.u32GPSTimerIntervalInMs));
         if ((false == sbVehGPSTimerRunning) && (NULL != poTimer) && (scoDefaultTimerIntervalInMs != m_rBDCLDataServiceConfigData.u32GPSTimerIntervalInMs))
         {
            poTimer->StartTimer(srVehGPSTimerID,m_rBDCLDataServiceConfigData.u32GPSTimerIntervalInMs,
                     m_rBDCLDataServiceConfigData.u32GPSTimerIntervalInMs, this,
                     &spi_tclBDCLDataService::bVehicleGPSTimerCb, NULL);
            sbVehGPSTimerRunning = true;
            ETG_TRACE_USR2(("[DESC]:Vehicle Speed timer started"));
         }
         else if (true == sbVehGPSTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer already running"));
         }
      }
      break;
      case e8CAR_DATA_VELOCITY:
      {
         ETG_TRACE_USR1(("spi_tclBDCLDataService::vStartVehicleSpeedUpdate()  timer value is %d"
                  ,m_rBDCLDataServiceConfigData.u32SpeedTimerIntervalInMs));
         if ((false == sbVehSpeedTimerRunning) && (NULL != poTimer))
         {
            poTimer->StartTimer(srVehSpeedTimerID,m_rBDCLDataServiceConfigData.u32SpeedTimerIntervalInMs,
                     m_rBDCLDataServiceConfigData.u32SpeedTimerIntervalInMs, this,
                     &spi_tclBDCLDataService::bVehicleVelocityTimerCb, NULL);
            sbVehSpeedTimerRunning = true;

            ETG_TRACE_USR2(("[DESC]:Vehicle Speed timer started"));

         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (true == sbVehSpeedTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer already running"));
         }
      }
      break;
      case e8CAR_DATA_GYROSCOPE:
      {
         ETG_TRACE_USR1(("spi_tclBDCLDataService::vStartVehicleUpdate()  timer value is %d"
                  ,m_rBDCLDataServiceConfigData.u32GyroscopeTimerIntervalInMs));
         if ((false == sbVehGyroscopeTimerRunning) && (NULL != poTimer) && (scoDefaultTimerIntervalInMs != m_rBDCLDataServiceConfigData.u32GyroscopeTimerIntervalInMs))
         {
            poTimer->StartTimer(srVehGyroscopeTimerID,m_rBDCLDataServiceConfigData.u32GyroscopeTimerIntervalInMs,
                     m_rBDCLDataServiceConfigData.u32GyroscopeTimerIntervalInMs, this,
                     &spi_tclBDCLDataService::bVehicleGyroscopeTimerCb, NULL);
            sbVehGyroscopeTimerRunning = true;

            ETG_TRACE_USR2(("[DESC]:Vehicle Gyroscope timer started"));

         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (true == sbVehGyroscopeTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:Vehicle Gyroscope timer already running"));
         }
      }
      break;
      case e8CAR_DATA_ACCELERATION:
      {
         ETG_TRACE_USR1(("spi_tclBDCLDataService::vStartVehicleUpdate()  timer value is %d"
                  ,m_rBDCLDataServiceConfigData.u32AccelerationTimerIntervalInMs));
         if ((false == sbVehAccelerationTimerRunning) && (NULL != poTimer) && (scoDefaultTimerIntervalInMs != m_rBDCLDataServiceConfigData.u32AccelerationTimerIntervalInMs))
         {
            poTimer->StartTimer(srVehAccelerationTimerID,m_rBDCLDataServiceConfigData.u32AccelerationTimerIntervalInMs,
                     m_rBDCLDataServiceConfigData.u32AccelerationTimerIntervalInMs, this,
                     &spi_tclBDCLDataService::bVehicleAccelerationTimerCb, NULL);
            sbVehAccelerationTimerRunning = true;

            ETG_TRACE_USR2(("[DESC]:Vehicle Acceleration timer started"));

         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (true == sbVehAccelerationTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:Vehicle Acceleration timer already running"));
         }
      }
      break;
      case e8CAR_DATA_GEAR:
      {
         ETG_TRACE_USR1(("spi_tclBDCLDataService::vStartVehicleUpdate()  timer value is %d"
                  ,m_rBDCLDataServiceConfigData.u32GearTimerIntervalInMs));
         if ((false == sbVehGearTimerRunning) && (NULL != poTimer))
         {
            poTimer->StartTimer(srVehGearTimerID,m_rBDCLDataServiceConfigData.u32GearTimerIntervalInMs,
                     m_rBDCLDataServiceConfigData.u32GearTimerIntervalInMs, this,
                     &spi_tclBDCLDataService::bVehicleGearTimerCb, NULL);
            sbVehGearTimerRunning = true;

            ETG_TRACE_USR2(("[DESC]:Vehicle Gear timer started"));

         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (true == sbVehGearTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:Vehicle Gear timer already running"));
         }
      }
      break;
      default:
         ETG_TRACE_ERR(("[ERR]: Unsupported Module Id:%d",enModuleID));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::vStopVehicleUpdate
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vStopVehicleUpdate(tenBdclCarDataModuleID enModuleID)
{
   ETG_TRACE_USR1(("spi_tclBDCLDataService::vStopVehicleUpdate() entered.  Module ID : %d" ,enModuleID));
   //! Stop timer
   Timer* poTimer = Timer::getInstance();

   switch(enModuleID)
   {
      case e8CAR_DATA_GPS:
      {
         if ((true == sbVehGPSTimerRunning) && (NULL != poTimer))
         {
            poTimer->CancelTimer(srVehGPSTimerID);
            sbVehGPSTimerRunning = false;
            ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vStopVehicleUpdate Vehicle GPS timer stopped "));
         }
         else if (false == sbVehGPSTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclBDCLDataService::vStopVehicleUpdate Vehicle GPS timer not running"));
         }
      }
      break;
      case e8CAR_DATA_VELOCITY:
      {
         if ((true == sbVehSpeedTimerRunning) && (NULL != poTimer))
         {
            poTimer->CancelTimer(srVehSpeedTimerID);
            sbVehSpeedTimerRunning = false;
            ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vStopVehicleUpdate Vehicle Speed timer stopped "));
         }
         else if (false == sbVehSpeedTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer not running"));
         }
      }
      break;
      case e8CAR_DATA_GYROSCOPE:
      {
         if ((true == sbVehGyroscopeTimerRunning) && (NULL != poTimer))
         {
            poTimer->CancelTimer(srVehGyroscopeTimerID);
            sbVehGyroscopeTimerRunning = false;
            ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vStopVehicleUpdateVehicle Gyroscope timer stopped "));
         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (false == sbVehGyroscopeTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclBDCLDataService::vStopVehicleUpdateVehicle Gyroscope timer not running"));
         }
      }
      break;
      case e8CAR_DATA_ACCELERATION:
      {
         if ((true == sbVehAccelerationTimerRunning) && (NULL != poTimer))
         {
            poTimer->CancelTimer(srVehAccelerationTimerID);
            sbVehAccelerationTimerRunning = false;
            ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vStopVehicleUpdateVehicle Acceleration timer stopped "));
         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (false == sbVehAccelerationTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclBDCLDataService::vStopVehicleUpdateVehicle Acceleration timer not running"));
         }
      }
      break;
      case e8CAR_DATA_GEAR:
      {
         if ((true == sbVehGearTimerRunning) && (NULL != poTimer))
         {
            poTimer->CancelTimer(srVehGearTimerID);
            sbVehGearTimerRunning = false;
            ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vStopVehicleUpdate Vehicle Gear timer stopped "));
         }//if ((false == sbVehSpeedTimerRunning) &&...)
         else if (false == sbVehGearTimerRunning)
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclBDCLDataService::vStopVehicleUpdate Vehicle Gear timer not running"));
         }
      }
      break;
      default:
         ETG_TRACE_ERR(("[ERR]: spi_tclBDCLDataService::vStopVehicleUpdate Unsupported Module Id:%d",enModuleID));
   }
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::bVehicleGPSTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bVehicleGPSTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::bVehicleGPSTimerCb entered "));

   spi_tclBDCLDataService* poBDCLDataService = static_cast<spi_tclBDCLDataService*> (pvObject);
   if (NULL != poBDCLDataService)
   {
      poBDCLDataService->vReportVehicleGPSInfo();
   }

   return true;
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::bVehicleGyroscopeTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bVehicleGyroscopeTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::bVehicleGyroscopeTimerCb entered "));

   spi_tclBDCLDataService* poBDCLDataService = static_cast<spi_tclBDCLDataService*> (pvObject);
   if (NULL != poBDCLDataService)
   {
      poBDCLDataService->vReportVehicleGyroscopeInfo();
   }

   return true;
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::bVehicleAccelerationTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bVehicleAccelerationTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::bVehicleAccelerationTimerCb entered "));

   spi_tclBDCLDataService* poBDCLDataService = static_cast<spi_tclBDCLDataService*> (pvObject);
   if (NULL != poBDCLDataService)
   {
      poBDCLDataService->vReportVehicleAccInfo();
   }

   return true;
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclBDCLDataService::bVehicleGearTimerCb
 ***************************************************************************/
t_Bool spi_tclBDCLDataService::bVehicleGearTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::bVehicleGearTimerCb entered "));

   spi_tclBDCLDataService* poBDCLDataService = static_cast<spi_tclBDCLDataService*> (pvObject);
   if (NULL != poBDCLDataService)
   {
      poBDCLDataService->vReportGear();
   }

   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vReportGear()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vReportGear()
{
   ETG_TRACE_USR1(("[PARAM]:spi_tclBDCLDataService::vReportGear() entered with Gear Position = %d ",ETG_ENUM(GEARPOSITION, m_enGearPosition)));
   t_Bool bRetVal = false;
   if((NULL != m_spoCmdCarData) && (e8GEAR_POS_UNKNOWN != m_enGearPosition))
   {
      switch(m_enGearPosition)
      {
         case e8GEAR_POS_NEUTRAL_GEAR:
            bRetVal = m_spoCmdCarData->bSendGearInfo(e8GEAR_NEUTRAL);
            break;
         case e8GEAR_POS_PARK_GEAR:
            bRetVal = m_spoCmdCarData->bSendGearInfo(e8GEAR_PARK);
            break;
         case e8GEAR_POS_SECOND_GEAR:
         case e8GEAR_POS_THIRD_GEAR:
         case e8GEAR_POS_FOURTH_GEAR:
         case e8GEAR_POS_FIFTH_GEAR:
         case e8GEAR_POS_SIXTH_GEAR:
            bRetVal = m_spoCmdCarData->bSendGearInfo(e8GEAR_DRIVE);
            break;
         case e8GEAR_POS_FIRST_GEAR:
         case e8GEAR_POS_FORWARD_GEAR:
            bRetVal = m_spoCmdCarData->bSendGearInfo(e8GEAR_LOW);
            break;
         case e8GEAR_POS_REVERSE_GEAR:
            bRetVal = m_spoCmdCarData->bSendGearInfo(e8GEAR_REVERSE);
            break;
         default: ETG_TRACE_USR1(("spi_tclBDCLDataService::vReportGear(): Unknown Gear position "));
         break;
      }
      ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vReportGear:Status of Gear message:%d:",bRetVal));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLDataService::vCarDataSubscriptionCallback()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vCarDataSubscriptionCallback(const tBdclCarDataInfoList& rCarDataSubscription)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionCallback() entered"));
   tBdclCarDataInfoList vecSubscribedModInfo;

   t_Bool bIsGPSsupported = false;
   if(NULL != m_poDataServiceSettings)
   {
      bIsGPSsupported = m_poDataServiceSettings->bGetCarlifeGPSDataSupportflag();
   }

   for (t_U8 u8Index = 0; u8Index < rCarDataSubscription.size(); ++u8Index)
   {
      trBdclCarDataInfo rCarDataInfo;
      rCarDataInfo.enCarDataModuleID = rCarDataSubscription[u8Index].enCarDataModuleID;

      ETG_TRACE_USR2(("[PARAM]:spi_tclBDCLDataService::vCarDataSubscriptionCallback() Received Module ID = %d, "
               "Support Flag = %d, Frequency = %d", rCarDataSubscription[u8Index].enCarDataModuleID,
               rCarDataSubscription[u8Index].s32SupportFlag, rCarDataSubscription[u8Index].u32Frequency));

      if((rCarDataSubscription[u8Index].s32SupportFlag)&&(scoDefaultFrequencyInHz >= rCarDataSubscription[u8Index].u32Frequency))
      {
         switch(rCarDataInfo.enCarDataModuleID)
         {
            case e8CAR_DATA_VELOCITY:
            {
               rCarDataInfo.s32SupportFlag = (m_rBDCLDataServiceConfigData.bSpeedData)?true:false;
               rCarDataInfo.u32Frequency = rCarDataSubscription[u8Index].u32Frequency;
            }
               break;
            case e8CAR_DATA_GEAR:
            {
               rCarDataInfo.s32SupportFlag = (m_rBDCLDataServiceConfigData.bGearStatus);
               rCarDataInfo.u32Frequency = rCarDataSubscription[u8Index].u32Frequency;
            }
               break;
            case e8CAR_DATA_GPS:
            {
               /**
                * As per Baidu Spec, SupportFlag should be populated as below for GPS :
                * 0 : Not Supported
                * 1 : WGS-84
                * 2 : GCJ-02
                * **/
               rCarDataInfo.s32SupportFlag = ((m_rBDCLDataServiceConfigData.bLocDataAvailable) &&
                                             (false == m_rBDCLDataServiceConfigData.bDeadReckonedData) &&
                                             (true == bIsGPSsupported))
                                             ? (m_rBDCLDataServiceConfigData.enGeoCoordinateSystemType)
                                             : (e8GEO_COORDINATE_SYSTEM_UNKNOWN);
               rCarDataInfo.u32Frequency = rCarDataSubscription[u8Index].u32Frequency;
            }
            break;
            case e8CAR_DATA_GYROSCOPE:
            {
               rCarDataInfo.s32SupportFlag = m_rBDCLDataServiceConfigData.bGyroData;
               rCarDataInfo.u32Frequency = rCarDataSubscription[u8Index].u32Frequency;
            }
               break;
            case e8CAR_DATA_ACCELERATION:
            {
               rCarDataInfo.s32SupportFlag = m_rBDCLDataServiceConfigData.bAcclData;
               rCarDataInfo.u32Frequency = rCarDataSubscription[u8Index].u32Frequency;
            }
               break;
            default:
            {
               rCarDataInfo.s32SupportFlag = false;
               rCarDataInfo.u32Frequency = 0;
               ETG_TRACE_USR4(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionCallback()  Module ID not supported"));
            }
         }
      }
      else
      {
         rCarDataInfo.s32SupportFlag = false;
         rCarDataInfo.u32Frequency = 0;
      }
      vecSubscribedModInfo.push_back(rCarDataInfo);
   }
   vSendSubscriptionDoneMsg(vecSubscribedModInfo);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLDataService::vSendSubscriptionDoneMsg(...)
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vSendSubscriptionDoneMsg(
         const tBdclCarDataInfoList& rCarDataSubscriptionDoneList)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vSendSubscriptionDoneMsg() entered "));
   if((m_spoCmdCarData) && (!rCarDataSubscriptionDoneList.empty()))
   {
      m_spoCmdCarData->bSendDataSubscribeDone(rCarDataSubscriptionDoneList);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLDataService::vCarDataSubscriptionStartCallback()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vCarDataSubscriptionStartCallback(const tBdclCarDataInfoList& rCarDataSubscriptionStart)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionStartCallback() entered"));
   //Assuming that start subscription is sent for Modules that are supported by HU.(as per the DONE message)
   t_Bool bIsGPSsupported = false;

   if(NULL != m_poDataServiceSettings)
   {
      bIsGPSsupported = m_poDataServiceSettings->bGetCarlifeGPSDataSupportflag();
   }

   if (NULL != m_rDataServiceCb.fvSubscribeForData)
   {
      for (t_U8 u8Index = 0; u8Index < rCarDataSubscriptionStart.size(); ++u8Index)
      {
         if((rCarDataSubscriptionStart[u8Index].s32SupportFlag) && (scoDefaultFrequencyInHz >= rCarDataSubscriptionStart[u8Index].u32Frequency))
         {
            switch(rCarDataSubscriptionStart[u8Index].enCarDataModuleID)
            {
               case e8CAR_DATA_GPS :
                  if(((true == m_rBDCLDataServiceConfigData.bLocDataAvailable) && (false == m_rBDCLDataServiceConfigData.bDeadReckonedData)
                           && (true == bIsGPSsupported)))
                  {
                     m_rDataServiceCb.fvSubscribeForData(true, e8GNSS_DATA);
                     m_rBDCLDataServiceConfigData.u32GPSTimerIntervalInMs = (1000/rCarDataSubscriptionStart[u8Index].u32Frequency);
                     vStartVehicleUpdate(rCarDataSubscriptionStart[u8Index].enCarDataModuleID);
                  }
                  break;
               case e8CAR_DATA_VELOCITY :
                  {
                     m_rDataServiceCb.fvSubscribeForData(true, e8VEHICLE_DATA);
                     m_rBDCLDataServiceConfigData.u32SpeedTimerIntervalInMs = (1000/rCarDataSubscriptionStart[u8Index].u32Frequency);
                     vStartVehicleUpdate(rCarDataSubscriptionStart[u8Index].enCarDataModuleID);
                  }
                  break;
               case e8CAR_DATA_GYROSCOPE :
                  if((m_rBDCLDataServiceConfigData.bGyroData))
                  {
                     m_rDataServiceCb.fvSubscribeForData(true, e8GYROSCOPE_DATA);
                     m_rBDCLDataServiceConfigData.u32GyroscopeTimerIntervalInMs = (1000/rCarDataSubscriptionStart[u8Index].u32Frequency);
                     vStartVehicleUpdate(rCarDataSubscriptionStart[u8Index].enCarDataModuleID);
                  }
                  break;
               case e8CAR_DATA_ACCELERATION :
                  if((m_rBDCLDataServiceConfigData.bAcclData))
                  {
                     m_rDataServiceCb.fvSubscribeForData(true, e8ACCELEROMETER_DATA);
                     m_rBDCLDataServiceConfigData.u32AccelerationTimerIntervalInMs = (1000/rCarDataSubscriptionStart[u8Index].u32Frequency);
                     vStartVehicleUpdate(rCarDataSubscriptionStart[u8Index].enCarDataModuleID);
                  }
                  break;
               case e8CAR_DATA_GEAR :
                  if((m_rBDCLDataServiceConfigData.bGearStatus))
                  {
                     m_rDataServiceCb.fvSubscribeForData(true, e8VEHICLE_DATA);
                     m_rBDCLDataServiceConfigData.u32GearTimerIntervalInMs = (1000/rCarDataSubscriptionStart[u8Index].u32Frequency);
                     vStartVehicleUpdate(rCarDataSubscriptionStart[u8Index].enCarDataModuleID);
                  }
                  break;
               default :
                  ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionStartCallback() : Module ID =%d  Not Supported",
                           rCarDataSubscriptionStart[u8Index].enCarDataModuleID));
            }
         }else
         {
            ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionStartCallback() could not send update.Error occurred"));
         }
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclBDCLDataService::vCarDataSubscriptionStopCallback()
 ***************************************************************************/
t_Void spi_tclBDCLDataService::vCarDataSubscriptionStopCallback(const tBdclCarDataInfoList& rCarDataSubscriptionStop)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionStopCallback() entered"));
   if (NULL != m_rDataServiceCb.fvSubscribeForData)
   {
      for (t_U8 u8Index = 0; u8Index < rCarDataSubscriptionStop.size(); ++u8Index)
      {
         switch(rCarDataSubscriptionStop[u8Index].enCarDataModuleID)
         {
            case e8CAR_DATA_GPS :
               {
                  m_rDataServiceCb.fvSubscribeForData(false, e8GNSS_DATA);
                  vStopVehicleUpdate(e8CAR_DATA_GPS);
               }
               break;
            case e8CAR_DATA_VELOCITY :
               {
                  vStopVehicleUpdate(e8CAR_DATA_VELOCITY);
               }
               break;
            case e8CAR_DATA_GYROSCOPE :
               {
                  m_rDataServiceCb.fvSubscribeForData(false, e8GYROSCOPE_DATA);
                  vStopVehicleUpdate(e8CAR_DATA_GYROSCOPE);
               }
               break;
            case e8CAR_DATA_ACCELERATION :
               {
                  m_rDataServiceCb.fvSubscribeForData(false, e8ACCELEROMETER_DATA);
                  vStopVehicleUpdate(e8CAR_DATA_ACCELERATION);
               }
               break;
            case e8CAR_DATA_GEAR :
               {
                  vStopVehicleUpdate(e8CAR_DATA_GEAR);
               }
               break;
            default :
               ETG_TRACE_USR2(("[DESC]:spi_tclBDCLDataService::vCarDataSubscriptionStopCallback() : Module ID =%d  Not Supported",
                        rCarDataSubscriptionStop[u8Index].enCarDataModuleID));
         }
      }
   }
   if((!sbVehGearTimerRunning) && (!sbVehSpeedTimerRunning)) //unsubscribe from Gear and Velocity Info update
      m_rDataServiceCb.fvSubscribeForData(false, e8VEHICLE_DATA);
}

//lint -restore

///////////////////////////////////////////////////////////////////////////////
// <EOF>
