/*!
 *******************************************************************************
 * \file             spi_tclBDCLConnection.h
 * \brief            Baidu Carlife Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Baidu Carlife Connection class to handle devices capable of Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 21.04.2017  | Ramya Murthy          | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLBDCLCONNECTION_H_
#define SPI_TCLBDCLCONNECTION_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <memory>

#include "spi_tclConnection.h"
#include "spi_tclBDCLRespSession.h"
#include "spi_tclBDCLRespVideo.h"
#include "BDCLTypes.h"
#include "SPITypes.h"
#include "Lock.h"
#include "spi_tclExtRespNativeTransport.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
//! Forward declarations
class spi_tclBDCLCmdSession;

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
typedef std::shared_ptr< spi_tclBDCLCmdSession >     t_SptrBDCLCmdSession;


class spi_tclBDCLConnection:
   public spi_tclConnection, //! Base Connection Class
   public spi_tclBDCLRespSession,
   public spi_tclBDCLRespVideo,
   public spi_tclExtRespNativeTransport
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::spi_tclBDCLConnection
       ***************************************************************************/
      /*!
       * \fn     spi_tclBDCLConnection()
       * \brief  Default Constructor
       * \sa      ~spi_tclBDCLConnection()
       **************************************************************************/
      spi_tclBDCLConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::~spi_tclBDCLConnection
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclBDCLConnection()
       * \brief  Destructor
       * \sa     spi_tclBDCLConnection()
       **************************************************************************/
      ~spi_tclBDCLConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::bInitializeConnection
       ***************************************************************************/
      /*!
       * \fn     bInitializeConnection()
       * \brief  required initializations.
       *         Mandatory interface
       * \retval returns true on successful initialization and false on failure
       **************************************************************************/
      virtual t_Bool bInitializeConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::vOnLoadSettings
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettings()
       * \brief  Called on loadsettings
       *         Optional interface
       * \param rfrheadUnitInfo : Head unit information
       * \param enCertificateType : CertificateType to be used for authentication
       **************************************************************************/
      t_Void vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo,
               tenCertificateType enCertificateType);

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  interface for the creator class to register for the required
       *        callbacks.
       * \param rfrConnCallbacks : reference to the callback structure
       *        populated by the caller
       **************************************************************************/
      t_Void vRegisterCallbacks(trConnCallbacks& ConnCallbacks);

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::vSetDeviceSwitchInfo
       ***************************************************************************/
      /*!
      * \brief  Sets device switch information for Android Auto devices.
      * \param  u32DeviceID  : [IN] Device Id
      * \retval t_Bool
      **************************************************************************/
      t_Void vSetDeviceSwitchInfo(trDeviceSwitchInfo &rfrDeviceSwitchInfo);

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnSelectDevice()
      ***************************************************************************/
      /*!
       * \fn     vOnSelectDevice
       * \brief  Called when a device is selected by the user. Optional interface
       * \param  u32DeviceHandle : Uniquely identifies the target Device.
       * \param  enConnType   : Identifies the Connection Type.
       * \param  enConnReq    : Identifies the Connection Request.
       * \param  enDAPUsage      : Identifies Usage of DAP for the selected ML device.
       *              This value is not considered for de-selection of device.
       * \param  enCDBUsage      : Identifies Usage of CDB for the selected ML device.
       *              This value is not considered for de-selection of device
       * \param  enSelectReason   :  [IN] Describes whether the device Selection or Deselection is
       *               triggered due to a user action.
       * \param  corUsrCntxt      : User Context Details.
       **************************************************************************/
      virtual t_Void vOnSelectDevice(t_U32 u32DeviceHandle,
               tenDeviceConnectionType enConnType, tenDeviceConnectionReq enConnReq,
               tenEnabledInfo enDAPUsage, tenEnabledInfo enCDBUsage,
               tenSelectReason enSelectionReason, trUserContext corUsrCntxt,
               tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnSelectDeviceResult()
       ***************************************************************************/
       /*!
       * \fn      t_Void vOnSelectDeviceResult
       * \brief   To perform the actions that are required, after the select device is
       *           successful/failed
       * \pram    u32DeviceHandle  : [IN] Uniquely identifies the target Device.
       * \pram    enConnReq : [IN] Identifies the Connection Request.
       * \pram    enRespCode: [IN] Response code. Success/Failure
       * \pram    enDeviceType : [IN] Device type
       * \retval  t_Void
       **************************************************************************/
       t_Void vOnSelectDeviceResult(t_U32 u32DeviceHandle,
                tenDeviceConnectionReq enConnReq, tenResponseCode enRespCode,
                tenDeviceCategory enDevCat, tenSelectReason enSelectionReason,
                tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

   //protected:

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnExitInfo()
        ***************************************************************************/
       /*!
        * \fn      virtual t_Void vOnExitInfo()
        * \brief   Triggered when BDCL device sends exit message
        * \retval  t_Void
        **************************************************************************/
       virtual t_Void vOnExitInfo();

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnCoreError()
        ***************************************************************************/
       /*!
        * \fn      vOnCoreError
        * \brief   Triggered when there is an error notified by AditCoreSurrogate endpoint
        * \param   enCarlifeError [IN]: Error type enumeration
        * \retval  t_Void
        **************************************************************************/
       virtual t_Void vOnCoreError(tenBdclCarLifeError enCarlifeError);

       /***************************************************************************
        **** FUNCTION: t_Void spi_tclBDCLConnection::vPostNativeTransportStartResult()
        **************************************************************************/
       /*!
        * \fn     t_Void vPostNativeTransportStartResult(const t_U32 cou32DevID)
        * \brief  Method called on native transport start
        * \param  cou32DevID : [IN] device ID
        * \retval t_Void
        **************************************************************************/
       t_Void vPostNativeTransportStartResult(const t_U32 cou32DevID);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclBDCLConnection::vPostNativeTransportStopResult()
        ***************************************************************************/
       /*!
        * \fn      t_Void vPostNativeTransportStopResult()
        * \brief   Called by MPlay client handler to inform native transport has stoppped
        * \param   [IN] u32DeviceId: Device ID
        * \sa      None
        **************************************************************************/
       t_Void vPostNativeTransportStopResult(t_U32 u32DeviceId) ;

	    /***************************************************************************
        ** FUNCTION: t_Void spi_tclBDCLConnection::vUpdateSelectionProgressState()
        ***************************************************************************/
       /*!
        * \fn     t_Void vUpdateSelectionProgressState(tenDeviceSelectionProgressState enDeviceSelectionProgressState)
        * \brief  Updates device selection progress state
        * \param  cou32DeviceHandle: [IN] Device ID
        * \param  enDeviceSelectionProgressState : [IN] Selection Progress state
        * \retval t_Void
        **************************************************************************/
       t_Void vUpdateSelectionProgressState(const t_U32 cou32DeviceHandle, tenDeviceSelectionProgressState enDeviceSelectionProgressState);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnProtocolVersionMatchStatus()
        ***************************************************************************/
       /*!
        * \fn      vOnProtocolVersionMatchStatus
        * \brief   Triggered when BDCL device updates protocol version match status
        * \param   bProtocolMatchStatus [IN]: TRUE - if protocol version match is successful, else FALSE
        * \retval  t_Void
        **************************************************************************/
       t_Void vOnProtocolVersionMatchStatus(t_Bool bProtocolMatchStatus);

       /***************************************************************************
        ** FUNCTION:  spi_tclBDCLConnection::vUpdateDeviceConnectionCountInfo
        ***************************************************************************/
       /*!
        * \fn     vUpdateDeviceConnectionCountInfo
        * \brief  Update device Connection Count information
        * \param  rDeviceConnectionCountInfo : Structure containing Device connection count info
        **************************************************************************/
       t_Void vUpdateDeviceConnectionCountInfo(trDeviceConnectionCountInfo &rfrDeviceConnectionCountInfo);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclBDCLConnection::vOnFeatureConfigRequest()
        ***************************************************************************/
       /*!
        * \fn      virtual t_Void vOnFeatureConfigRequest()
        * \brief   Triggered when BDCL device sends Feature config request message
        * \retval  t_Void
        **************************************************************************/
       virtual t_Void vOnFeatureConfigRequest();

       /***************************************************************************
        ** FUNCTION:  spi_tclBDCLConnection::vSetSelectionProgressState
        **************************************************************************/
       /*!
        * \fn      vSetSelectionProgressState()
        * \brief   set selection state to BDCL Connmanger
        * \retval  t_Void
        *************************************************************************/
       t_Void vOnSetSelectionProgressState(tenDeviceSelectionState enDeviceSelectionState) override;

       /***************************************************************************
       ** FUNCTION: t_Void spi_tclBDCLConnection::vReportSessionErrorStatus()
       ***************************************************************************/
       /*!
        * \fn     vReportSessionErrorStatus
        * \param   enDevCategory : Device Category
        * \param   tenSessionStatus : Session status
        * \brief   vReportSessionErrorStatus reports session error status to resource manager
        **************************************************************************/
       t_Void vReportSessionErrorStatus(tenDeviceCategory coenDevCat, tenSessionStatus coenSessionStatus);


	   
   private:

      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLConnection::vSendHUInfo()
       ***************************************************************************/
      /*!
       * \fn     vSendHUInfo
       * \brief  Sends HU Information to BDCL device
       * \retval None
       **************************************************************************/
      t_Void vSendHUInfo();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLConnection::vGetStatisticsInfo()
       ***************************************************************************/
      /*!
       * \fn     vGetStatisticsInfo
       * \brief  Sends Statistics information to BDCL device
       * \retval None
       **************************************************************************/
      t_Void vGetStatisticsInfo(trBdclStatisticsInfo& rfrStatisticsInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLConnection::vSendSelectDeviceResult()
       ***************************************************************************/
      /*!
       * \fn     t_Void vSendSelectDeviceResult()
       * \brief  Function to post the device selection status.
       * \param  enErrorType : [IN] error value.
       * \retval None
       **************************************************************************/
      t_Void vSendSelectDeviceResult(tenErrorCode enErrorType);

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::enPrepareSession
       ***************************************************************************/
      /*!
       * \fn     enPrepareSession()
       * \brief  Prepares session with BDCL device
       * \retval Selection error value.
       **************************************************************************/
      tenErrorCode enPrepareSession(tenDeviceType enDeviceType);

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::bTeardownSession
       ***************************************************************************/
      /*!
       * \fn     bTeardownSession()
       * \brief  Terminates session with BDCL device
       * \retval true if session teardown is successful
       **************************************************************************/
      t_Bool bTeardownSession();

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLConnection::vGetInitInfo()
       ***************************************************************************/
      /*!
       * \fn     vGetInitInfo()
       * \brief  Provides initialisation info to prepare BDCL session
       * \param  enDeviceType [IN]: Indicates whether device is Android or iOS device
       * \param  rfrInitInfo [OUT]: Structure containing initialisation info
       * \sa     bPrepareSession()
       * \retval None
       **************************************************************************/
      t_Void vGetInitInfo(tenDeviceType enDeviceType, trBdclInitInfo& rfrInitInfo);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclBDCLConnection::vOnNativeTransportStart()
             ***************************************************************************/
      /*!
      * \fn     t_Void vOnNativeTransportStart(const t_U32 cou32DevID)
      * \brief  Method called on native transport start
      * \param  cou32DevID : [IN] device ID
      * \retval t_Void
      **************************************************************************/
      t_Void vOnNativeTransportStart(const t_U32 cou32DevID);


      /***************************************************************************
      ** FUNCTION: t_Void spi_tclBDCLConnection::vSetSelectedDevice()
      ***************************************************************************/
      /*!
      * \fn     t_Void vSetSelectedDevice(const t_U32 cou32DevID)
      * \brief  Method to set
      * \param  cou32DevID : [IN] device ID
      * \retval t_Void
      **************************************************************************/
      t_Void vSetSelectedDevice(const t_U32 cou32DevID);

      /***************************************************************************
      ** FUNCTION: t_U32 spi_tclBDCLConnection::vSetSelectedDevice()
      ***************************************************************************/
      /*!
      * \fn     t_U32 u32GetSelectedDevice()
      * \brief  Method to get the selected device
      * \retval t_Void
      **************************************************************************/
      t_U32 u32GetSelectedDevice();

      /***************************************************************************
       ** FUNCTION: spi_tclBDCLConnection::vSendDeviceDeselect()
       ***************************************************************************/
      /*!
       * \fn      vSendDeviceDeselect()
       * \brief   Requests deselection of current active device
       * \param   none
       * \return  none
       **************************************************************************/
      t_Void vSendDeviceDeselect(tenSelectReason enSelectReason = e8_REASON_UNKNOWN);

      t_Bool m_bStartTransportAfterSwitch;

      t_U32 m_u32SelectedDevice;

      tenDeviceType m_enDeviceType;

      //! Pointer to Cmd Session class
      t_SptrBDCLCmdSession m_spoCmdSession;

      //! Store head unit information
      trHeadUnitInfo m_rHeadUnitInfo;

      //! Structure containing device switch details
      trDeviceSwitchInfo m_rDeviceSwitchInfo;

      //! Callbacks for ConnMngr to register.
      trConnCallbacks m_rConnCallbacks;

      t_U32 m_u32ConnectCount, m_u32ConnectSuccessCount, m_u32ConnectTime;

      tenBdclCarLifeError m_enDeviceErrorDuringSelection;

      t_Bool m_bWaitingForFeatureConfigRequest;

      Lock m_oSessionPrepareLock;

      t_Bool m_bIsPreparesessioncalled;

      tenDeviceSelectionState m_enDeviceSelectionState;

};

#endif // SPI_TCLBDCLCONNECTION_H_
