  /*!
 *******************************************************************************
 * \file         spi_tclBDCLBluetooth.h
 * \brief        BDCL Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    BDCL Bluetooth handler class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 22.06.2017 |  Pooja Krishnan (RBEI/ECO2)       | Initial Version
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLBDCLBLUETOOTH_H_
#define _SPI_TCLBDCLBLUETOOTH_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include <memory>
#include <algorithm>

#include "BDCLTypes.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclBluetoothDevBase.h"
#include "spi_tclBDCLRespBluetooth.h"
#include "spi_tclExtRespBluetooth.h"
#include "spi_tclBDCLRespSession.h"
#include "Lock.h"

/* Forward Declarations. */
class spi_tclExtCmdBluetoothIntf;
class spi_tclBluetoothIntf;
class spi_tclBDCLCmdBluetooth;

/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
typedef std::shared_ptr< spi_tclBDCLCmdBluetooth >     t_SptrBDCLCmdBluetooth;

/*!
* \class spi_tclBDCLBluetooth
* \brief This is the BDCL Bluetooth class that handles the Bluetooth
*        connection logic during a BDCL device session
*/
class spi_tclBDCLBluetooth : public spi_tclBluetoothDevBase, public spi_tclBDCLRespBluetooth, public spi_tclBDCLRespSession,
public spi_tclExtRespBluetooth
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLBluetooth::spi_tclBDCLBluetooth(...)
   ***************************************************************************/
   /*!
   * \fn      spi_tclBDCLBluetooth(spi_tclBluetoothIntf* poBTInterface,
   *             spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf)
   * \brief   Parameterized Constructor
   * \param   [IN] poBTInterface: Pointer to Bluetooth manager interface
   * \param   [IN] poExtCmdBluetoothIntf: Pointer to Extended Command Bluetooth Interface
   ***************************************************************************/
   spi_tclBDCLBluetooth(spi_tclBluetoothIntf* poBTInterface);

   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLBluetooth::~spi_tclBDCLBluetooth();
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclBDCLBluetooth()
   * \brief   Virtual Destructor
   ***************************************************************************/
   virtual ~spi_tclBDCLBluetooth();

   /*********Start of functions overridden from spi_tclBluetoothDevBase*******/

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclBDCLBluetooth::bInitialise();
    ***************************************************************************/
   /*!
    * \fn      bInitialise()
    * \brief   Method to initialises the service handler. (Performs initialisations which
    *          are not device specific.)
    *          Optional interface to be implemented.
    * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
    * \sa      bUninitialise()
    ***************************************************************************/
   t_Bool bInitialise();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceRequest(...)
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(t_U32 u32ProjectionDevHandle,
   *             tenDeviceConnectionReq enDevConnReq)
   * \brief   Called when SelectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceResponse(...)
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
   *             tenResponseCode enRespCode)
   * \brief   Called when SelectDevice operation is complete, with the result
   *          of the operation.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceRequest(t_U32...)
   ***************************************************************************/
   /*!
   * \fn       vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);
   * \brief   Called when DeselectDevice request is received by SPI.
   *          Mandatory interface to be implemented.
   * \param   [IN] corfrSelectReq: Device Selection Info
   * \retval  None
   **************************************************************************/
   t_Void vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnSPISelectDeviceResponse(t_U32...)
   ***************************************************************************/
   /*!
   * \fn      vOnSPISelectDeviceResponse(t_U32 u32ProjectionDevHandle,
   *             tenResponseCode enRespCode)
   * \brief   Called when DeselectDevice operation is complete, with the result
   *          of the operation.
   *          Mandatory interface to be implemented.
   *   * \param   [IN] corfrSelectReq: Device Selection Info
   * \param   [IN] enRespCode: Response code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

   /*********End of functions overridden from spi_tclBluetoothDevBase*********/

   /********Start of functions overridden from spi_tclBDCLRespBluetooth********/

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnBluetoothIdentifyResultIndMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vOnBluetoothIdentifyResultIndMsg()
    * \brief   Called when Identity result is sent by BDCL device.
    ** \param   [IN] cprBTIdentityResultIndInfo: Structure contains Identify Result
    * \sa      None
    **************************************************************************/
   t_Void vOnBluetoothIdentifyResultIndMsg(const trBdclBluetoothIdentifyResultInd* cprBTIdentityResultIndInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnStartBluetoothAutoPairMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vOnStartBluetoothAutoPairMsg()
    * \brief   Called when Pairing request is sent by BDCL device.
    * \param   [IN] cprStartBTAutopairInfo : Structure contains Autopair information
    * \sa      None
    **************************************************************************/
   t_Void vOnStartBluetoothAutoPairMsg(const trBdclStartBluetoothAutopair* cprStartBTAutopairInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnBluetoothOOBInfoMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vOnBluetoothOOBInfoMsg()
    * \brief   Called when Pairing OOB information is sent by BDCL device.
    * \param   [IN] cprInfo: Structure contains OOB information
    ** \sa      None
    **************************************************************************/
   t_Void vOnBluetoothOOBInfoMsg(const trBdclBluetoothOOBInfo* cprInfo);

   /**********End of functions overridden from spi_tclBDCLRespBluetooth********/

   /**************************************************************************
   ****************************END OF PUBLIC**********************************
   **************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/

   /*********Start of functions overridden from spi_tclBluetoothDevBase*******/
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnBTDeviceNameUpdate(...)
   ***************************************************************************/
   /*!
   * \fn      vOnBTDeviceNameUpdate()
   * \brief   Interface to receive notification on device name of BT paired devices.
   *          Optional interface to be implemented.
   * \param   [IN] szBTDeviceAddress: BT Address of paired device
   * \param   [IN] szBTDeviceName: BT Device name of paired device
   * \retval  None
   **************************************************************************/
   /*t_Void vOnBTDeviceNameUpdate(t_String szBTDeviceAddress,
         t_String szBTDeviceName);*/

   /***************************************************************************
   *** FUNCTION:  t_Void spi_tclBDCLBluetooth::vOnMDInfo()
   ***************************************************************************/
   /*!
   * \fn      vOnMDInfo
   * \brief   Triggered when BDCL device sends device info
   * \param   corfrMDInfo [IN]: Structure containing BDCL device info
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vOnMDInfo(const trBdclMDInfo& crfrMDInfo);

   /*********End of functions overridden from spi_tclBluetoothDevBase*********/

   /***************************************************************************
   ******************************END OF PROTECTED******************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclBDCLBluetooth(const spi_tclBDCLBluetooth &rfcoBluetooth)
    ***************************************************************************/
   /*!
    * \fn      spi_tclBDCLBluetooth(const spi_tclBDCLBluetooth &rfcoBluetooth)
    * \brief   Copy constructor not implemented hence made private
    **************************************************************************/
   spi_tclBDCLBluetooth(const spi_tclBDCLBluetooth &rfcoBluetooth);

   /***************************************************************************
    ** FUNCTION: const spi_tclBDCLBluetooth & operator=(
    **                                 const spi_tclBDCLBluetooth &rfcoBluetooth);
    ***************************************************************************/
   /*!
    * \fn      const spi_tclBDCLBluetooth & operator=(const spi_tclBDCLBluetooth &rfcoBluetooth);
    * \brief   assignment operator not implemented hence made private
    **************************************************************************/
   const spi_tclBDCLBluetooth & operator=(
            const spi_tclBDCLBluetooth &rfcoBluetooth);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vStopBTPairing()
   ***************************************************************************/
   /*!
   * \fn      enGetPairingStatus()
   * \brief   Interface to read current pairing status of BDCL device
   * \retval  tenDevicePairingStatus
   **************************************************************************/
   tenDevicePairingStatus enGetPairingStatus();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vSetPairingStatus()
   ***************************************************************************/
   /*!
   * \fn      vSetPairingStatus()
   * \brief   Interface to set pairing status of BDCL device
   * \param   enPairingStatus : [IN] Pairing state enum
   **************************************************************************/
   t_Void vSetPairingStatus(tenDevicePairingStatus enPairingStatus);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vRestoreHUDefaultState()
   ***************************************************************************/
   /*!
   * \fn      vRestoreHUDefaultState()
   * \brief   Resets the HU BT stack state to default condition
   * \retval  None
   **************************************************************************/
   t_Void vRestoreHUDefaultState();

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclDiPoBluetooth::vDeselectBDCLDevice()
   ***************************************************************************/
   /*!
   * \fn      vDeselectBDCLDevice()
   * \brief   Triggers deselection request for BDCL device
   * \retval  None
   **************************************************************************/
   t_Void vDeselectBDCLDevice();

   /***************************************************************************
   ** FUNCTION:   t_Void spi_tclBDCLBluetooth::vSendDevicePairingResponse()
   ***************************************************************************/
   /*!
   * \fn      vSendDevicePairingResponse(t_Bool enBTStatus)
   * \brief   Sends a BluetoothOOBInfo message to BDCL device
   * \param   enBTStatus [IN]: Bluetooth connection status
   * \retval  None
   **************************************************************************/
   t_Void vSendDevicePairingResponse(tenBdclHUBTStatus enBTStatus);

   /***************************************************************************
   ** FUNCTION:   t_Bool spi_tclBDCLBluetooth::bCompareBTDevices()
   ***************************************************************************/
   /*!
   * \fn      bCompareBTDevices()
   * \brief   Compares BT Address of AA P device and other device to determine if they are same.
   * \param   None
   * \retval  True - if BT devices are same, else False.
   **************************************************************************/
   t_Bool bCompareBTDevices(const t_String& rfcszBDCLBTDevAddress,
         const t_String& rfcszBTDevAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vSendIdentificationRequest(...)
   ***************************************************************************/
   /*!
   * \fn      vSendIdentificationRequest()
   * \brief   Sends BT_START_IDENTIFY_REQ message to BDCL device
   * \param   None
   * \retval  None
   **************************************************************************/
   t_Void vSendIdentificationRequest();
   
   /***************************************************************************
   ** FUNCTION:  spi_tclBDCLBluetooth::bFormatBTToMacAddress
   ***************************************************************************/
   /*!
   * \fn     bFormatBTToMacAddress()
   * \brief  This function is used to format the BT address to MAC address
   * \param  rfszMACAddress: [IN] MAC address
   * \retVal Bool : True if successfully transferred.
   **************************************************************************/
   t_Bool bFormatBTToMacAddress(t_String &rfszMACAddress);

   /***************************************************************************
    *** FUNCTION:  spi_tclBDCLBluetooth::bFormatMacToBTAddress
    ******************************************************************************/
   /*!
    * \fn     bFormatMacToBTAddress()
    * \brief  This function is used to format the BT address to MAC address
    * \param  rfszMACAddress: [IN] MAC address
    * \retVal Bool : True if successfully transferred.
      **************************************************************************/
   t_Bool bFormatMacToBTAddress(t_String &rfszMACAddress);
   
   /***************************************************************************
    *** FUNCTION:  spi_tclBDCLBluetooth::bIsValidBTAddress
    ******************************************************************************/
   /*!
    * \fn     bIsValidBTAddress()
    * \brief  This function is used to validate the correct MAC address
    * \param  rfszMACAddress: [IN] MAC address
    * \retVal Bool : True if BT MAC address is valid.
      **************************************************************************/
   t_Bool bIsValidBTAddress(const t_String &rfszMACAddress);
   
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTConnectionChangedMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTConnectionChangedMsg()
    * \brief   Called when BT Connection change result is sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTConnectionChangedMsg(t_String szBTDeviceAddress,tenBTConnectionResult enBTConnResult);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTPairingInfoMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTPairingInfoMsg()
    * \brief   Called when BT Pairing message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTPairingInfoMsg(trBTPairingRequestInfo rBTPairingReqInfo);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTLimitationModeMsg()
    ***************************************************************************/
   /*!
    * \fn      t_Void vPostBTLimitationModeMsg()
    * \brief   Called when BT limitation mode message is  sent by BT Client Handler.
    * \sa      None
    **************************************************************************/
   virtual t_Void vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBDCLBluetooth::vPostBTServiceAvailabilityMsg()
   ***************************************************************************/
   /*!
   * \fn      t_Void vPostBTServiceAvailabilityMsg()
   * \brief   Called when BT On Service Availability is  sent by BT Client Handler.
   * \sa      None
   **************************************************************************/
   t_Void vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable);

   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** BT Manager interface pointer
   ***************************************************************************/
   spi_tclBluetoothIntf* const   m_cpoBTInterface;

   /***************************************************************************
   ** Pairing status of BDCL device
   ***************************************************************************/
   tenDevicePairingStatus  m_enPairingStatus;

   /***************************************************************************
   ** Lock object for device pairing status
   ***************************************************************************/
   Lock  m_oPairingStateLock;

   /***************************************************************************
   ** Lock object for HU BT stack state
   ***************************************************************************/
   Lock  m_oHUBTStateLock;

   spi_tclExtCmdBluetoothIntf *m_poExtCmdBluetoothIntf;

   //! Pointer to Cmd Session class
   t_SptrBDCLCmdBluetooth   m_spoCmdBluetooth;

   //String holds MD BT Address
   t_String m_szMDBTAddress;

   t_Bool m_bIdentificationRequestSent;

   //! Action of pending SetBTLimitationMode request
   tenBTLimitationAction m_enPendingBTLimAction;
   
   //! Technology of pending SetBTLimitationMode request
   tenBTTechnology m_enPendingBTLimTechnology;

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

};
#endif // _SPI_TCLBDCLBLUETOOTH_H_

