/***********************************************************************/
/*!
* \file  spi_tclBDCLCmdSession.cpp
* \brief  Implementation of the Class spi_tclBDCLCmdSession
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         rur1kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <algorithm>
#include "spi_tclBDCLMsgQInterface.h"
#include "spi_tclBDCLSessionDispatcher.h"
#include "spi_tclBDCLCmdSession.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLCmdSession.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/


/***************************************************************************
 ** FUNCTION:   spi_tclBDCLCmdSession()
 ***************************************************************************/
spi_tclBDCLCmdSession::spi_tclBDCLCmdSession(): m_bInitInProgress(false), m_enInitStatus(e8CL_INIT_UNKNOWN_ERR), m_u16ProtocolStatusTimeOut(8)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::spi_tclBDCLCmdSession entered "));
}

/***************************************************************************
 ** FUNCTION:   ~spi_tclBDCLCmdSession()
 ***************************************************************************/
spi_tclBDCLCmdSession::~spi_tclBDCLCmdSession()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::~spi_tclBDCLCmdSession entered "));

   vUninitialize();
}

/***************************************************************************
 ** FUNCTION:   t_Bool enInitialize()
 ***************************************************************************/
tenBdclInitStatus spi_tclBDCLCmdSession::enInitialize(const trBdclInitInfo& corfrInitInfo,
        const trBdclStatisticsInfo& corfrStatisticsInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::enInitialize entered "));
   m_enInitStatus = e8CL_INIT_UNKNOWN_ERR;

   t_SptrBDCLProxyManager spoProxyManager = spi_tclBDCLProxyManager::getInstance();
   t_SptrBDCLCoreDataIntf spoCoreDataIntf = (spoProxyManager)?
            (spoProxyManager->spoGetCoreDataIntf()) : (nullptr);
   t_CoreCallbackDealer* poCallbackDealer = (spoCoreDataIntf)?
            (spoCoreDataIntf->poGetCoreCallbackDealer()) : (NULL);
   SPI_NORMAL_ASSERT(NULL == poCallbackDealer);

   m_spoSessionProxy = (spoProxyManager)? (spoProxyManager->spoGetSessionProxyInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!m_spoSessionProxy);

   if ((m_spoSessionProxy) && (NULL != poCallbackDealer))
   {
      m_spoCoreEndpoint = std::move(t_SptrBDCLCoreEndpoint(new adit::bdcl::AditCoreSurrogate(this, poCallbackDealer)));
      SPI_NORMAL_ASSERT(!m_spoCoreEndpoint);
      if (m_spoCoreEndpoint)
      {
         vRegisterCallbacks();

         vSetStatisticsInfo(corfrStatisticsInfo);
         m_spoCoreEndpoint->setProtocolStatusTimeout(m_u16ProtocolStatusTimeOut);

         m_oLock.s16Lock();
         m_bInitInProgress = true;
         if (corfrInitInfo.bIsAoapDevice)
         {
            m_enInitStatus = enGetBdclInitState(m_spoCoreEndpoint->initialize(
                     corfrInitInfo.s32AoapAccessoryId, corfrInitInfo.s32AoapDeviceId));
         }
         else
         {
            m_enInitStatus = enGetBdclInitState(m_spoCoreEndpoint->initialize(
                     corfrInitInfo.szWriteDevice.c_str(), corfrInitInfo.szReadDevice.c_str()));
         }
         m_bInitInProgress = false;
         m_oLock.vUnlock();

         ETG_TRACE_USR4(("[PARAM]::enInitialize: Init Status = %d", ETG_ENUM(BDCL_INIT_STATUS, m_enInitStatus)));
      }

   }

   ETG_TRACE_USR1(("[DESC]::spi_tclBDCLCmdSession::enInitialize left with result = %d", ETG_ENUM(BDCL_INIT_STATUS, m_enInitStatus)));
   return m_enInitStatus;
}

/***************************************************************************
 ** FUNCTION:   t_Void vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vUninitialize entered"));

   if (m_spoCoreEndpoint)
   {
      if (m_bInitInProgress)
      {
         m_spoCoreEndpoint->abortInitialization();
         ETG_TRACE_USR1(("[DESC]::vUninitialize: Init still in progress. Hence requested to abort session"));
      }

      m_oLock.s16Lock();
      if (e8CL_INIT_NO_ERR == m_enInitStatus)
      {
         ETG_TRACE_USR1(("[DESC]::vUninitialize: Init was not aborted. Hence requesting teardown."));
         m_spoCoreEndpoint->teardown();
      }
      m_oLock.vUnlock();
   }

   //! Clear flags
   m_enInitStatus = e8CL_INIT_NO_ERR;
   m_bInitInProgress = false;
   m_spoCoreEndpoint = nullptr;
   m_spoSessionProxy = nullptr;

   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vUninitialize left"));
}

/***************************************************************************
 ** FUNCTION:   t_Void vAbortInit()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vAbortInit()
{
   if (m_spoCoreEndpoint)
   {
      ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vAbortInit entered "));
      m_spoCoreEndpoint->abortInitialization();
   }
}

/***************************************************************************
 ** FUNCTION:   t_Void vSetProtocolStatusTimeOut()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetProtocolStatusTimeOut(t_U16 u16TimeOut)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetProtocolStatusTimeOut entered "));
   m_u16ProtocolStatusTimeOut = u16TimeOut;
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetHeadUnitInfo( )
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetHeadUnitInfo(const trBdclHUInfo& corfrHUInfo)
{
   if (m_spoSessionProxy)
   {
      ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetHeadUnitInfo entered "));
      m_spoSessionProxy->vSendHUInfo(corfrHUInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void vSendStatisticsInfo()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetStatisticsInfo(const trBdclStatisticsInfo& corfrStatisticsInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetStatisticsInfo entered "));

   if (m_spoCoreEndpoint)
   {
      S_STATISTICS_INFO* prInfo = new S_STATISTICS_INFO;
      prInfo->cuid = corfrStatisticsInfo.szCuid;
      prInfo->versionName = corfrStatisticsInfo.szVersionName;
      prInfo->versionCode = corfrStatisticsInfo.u32VersionCode;
      prInfo->channel = corfrStatisticsInfo.szChannel;
      prInfo->connectCount = corfrStatisticsInfo.u32ConnectCount;
      prInfo->connectSuccessCount = corfrStatisticsInfo.u32ConnectSuccessCount;
      prInfo->connectTime = corfrStatisticsInfo.u32ConnectTime;
      prInfo->crashLog = corfrStatisticsInfo.szCrashLog;

      m_spoCoreEndpoint->setCarLifeStaticsInfo(prInfo);

      delete prInfo;
   }

   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetStatisticsInfo left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetFeatureConfigInfo()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetFeatureConfigInfo(const tBdclFeatureConfigList& corfrFeatConfigList)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetFeatureConfigInfo entered: List size = %d ",
            corfrFeatConfigList.size()));
   
   if (m_spoSessionProxy)
   {  
      m_spoSessionProxy->vSendFeatureConfigList(corfrFeatConfigList);
   }
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetFeatureConfigInfo left "));
}

/***************************************************************************
 ** FUNCTION:   t_Void vSetMusicState()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetMusicState(tenBdclMusicState enMusicState)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetMusicState entered: %d ", enMusicState));
   if (m_spoSessionProxy)
   {
      m_spoSessionProxy->vSendCmdModuleControl(
            static_cast<t_U32>(e8CL_MUSIC_MODULE_ID),
            static_cast<t_U32>(enMusicState));
   }
}

/***************************************************************************
 ** FUNCTION:   t_Void vSetVRState()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetVRState(tenBdclVRState enVRState)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetVRState entered: %d ", enVRState));
   if (m_spoSessionProxy)
   {
      m_spoSessionProxy->vSendCmdModuleControl(
            static_cast<t_U32>(e8CL_VR_MODULE_ID),
            static_cast<t_U32>(enVRState));
   }
}

/***************************************************************************
 ** FUNCTION:   t_Void vSetNavigationState()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetNavigationState(tenBdclNaviState enNavState)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetNavigationState entered: %d ", enNavState));
   if (m_spoSessionProxy)
   {
      m_spoSessionProxy->vSendCmdModuleControl(
            static_cast<t_U32>(e8CL_NAVI_MODULE_ID),
            static_cast<t_U32>(enNavState));
   }
}

/***************************************************************************
 ** FUNCTION:   t_Void vSetPhoneState()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vSetPhoneState(tenBdclPhoneState enPhoneState)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vSetPhoneState entered: %d ", enPhoneState));
   if (m_spoSessionProxy)
   {
      m_spoSessionProxy->vSendCmdModuleControl(
            static_cast<t_U32>(e8CL_PHONE_MODULE_ID),
            static_cast<t_U32>(enPhoneState));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vRegisterCallbacks()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vRegisterCallbacks entered"));

   if (m_spoSessionProxy)
   {
      trBdclSessionCbs rCallbacks;
      rCallbacks.fvOnMDInfo = vOnMDInfoCb;
      rCallbacks.fvOnModuleStatus = vOnModuleStatusCb;
      rCallbacks.fvOnExit = vOnExitCb;
      rCallbacks.fvOnFeatureConfigRequest = vOnFeatureConfigRequestCb;
      m_spoSessionProxy->vRegisterSessionCallbacks(rCallbacks);
   }
}

/***************************************************************************
 ** FUNCTION:   void onProtocolVersionMatchStatus()
 ***************************************************************************/
void spi_tclBDCLCmdSession::onProtocolVersionMatchStatus(bool status)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::onProtocolVersionMatchStatus entered: %d", ETG_ENUM(BOOL, status)));
   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLProtocolVersionMatchStatusMsg oProtocolVersionMatchStatusMsg;
      oProtocolVersionMatchStatusMsg.bProtocolMatchStatus = status;
      poMsgQInterface->bWriteMsgToQ(&oProtocolVersionMatchStatusMsg, sizeof(oProtocolVersionMatchStatusMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:   void onAuthenticationResponse()
 ***************************************************************************/
void spi_tclBDCLCmdSession::onAuthenticationResponse(bool status)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::onAuthenticationResponse entered: %d", ETG_ENUM(BOOL, status)));

   //TODO
}

/***************************************************************************
 ** FUNCTION:   void onError()
 ***************************************************************************/
void spi_tclBDCLCmdSession::onError(tBdclErrorCodes inErrorCode)
{
   tenBdclCarLifeError enError = static_cast<tenBdclCarLifeError>(inErrorCode);
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::onError entered: %d", ETG_ENUM(BDCL_ERROR, enError)));

   if ((m_bInitInProgress) && (e8CL_AOAP_NO_DEVICE_ERROR == enError) && (NULL != m_spoCoreEndpoint))
   {
      m_bInitInProgress = false;
      //Whenever device gets disconnected immediately call abortinit to come out of bdclinit as it is  a blocking call
      m_spoCoreEndpoint->abortInitialization();
      ETG_TRACE_USR2(("[DESC]::onError: Init still in progress. Hence requested to abort session"));
   }
   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLCoreErrorMsg oErrorMsg;
      oErrorMsg.m_enCarlifeError = enError;
      poMsgQInterface->bWriteMsgToQ(&oErrorMsg, sizeof(oErrorMsg));
   }
}

/***************************************************************************
 ** FUNCTION:  static t_Void vOnMDInfoCb(...)
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vOnMDInfoCb(S_MD_INFO* prInfo)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vOnMDInfoCb entered"));
   SPI_NORMAL_ASSERT(NULL == prInfo);

   if (NULL != prInfo)
   {
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: os = %s", prInfo->os.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: board = %s", prInfo->board.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: bootloader = %s", prInfo->bootloader.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: brand = %s", prInfo->brand.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: cpu_abi = %s", prInfo->cpu_abi.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: cpu_abi2 = %s", prInfo->cpu_abi2.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: device = %s", prInfo->device.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: display = %s", prInfo->display.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: fingerprint = %s", prInfo->fingerprint.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: hardware = %s", prInfo->hardware.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: host = %s", prInfo->host.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: cid = %s", prInfo->cid.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: manufacturer = %s", prInfo->manufacturer.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: model = %s", prInfo->model.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: product = %s", prInfo->product.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: serial = %s", prInfo->serial.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: codename = %s", prInfo->codename.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: incremental = %s", prInfo->incremental.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: release = %s", prInfo->release.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: sdk = %s", prInfo->sdk.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: sdk_int = %d", prInfo->sdk_int));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: token = %s", prInfo->token.c_str()));
      ETG_TRACE_USR4(("[PARAM]::vOnMDInfoCb: btaddress = %s", prInfo->btaddress.c_str()));

      spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
      if (NULL != poMsgQInterface)
      {
         BDCLMDInfoMsg oMDInfoMsg;
         oMDInfoMsg.m_prMDInfo->szOS = prInfo->os;
         oMDInfoMsg.m_prMDInfo->szBoard = prInfo->board;
         oMDInfoMsg.m_prMDInfo->szBootloader = prInfo->bootloader;
         oMDInfoMsg.m_prMDInfo->szBrand = prInfo->brand;
         oMDInfoMsg.m_prMDInfo->szCpu_abi = prInfo->cpu_abi;
         oMDInfoMsg.m_prMDInfo->szCpu_abi2 = prInfo->cpu_abi2;
         oMDInfoMsg.m_prMDInfo->szDevice = prInfo->device;
         oMDInfoMsg.m_prMDInfo->szDisplay = prInfo->display;
         oMDInfoMsg.m_prMDInfo->szFingerprint = prInfo->fingerprint;
         oMDInfoMsg.m_prMDInfo->szHardware = prInfo->hardware;
         oMDInfoMsg.m_prMDInfo->szHost = prInfo->host;
         oMDInfoMsg.m_prMDInfo->szCid = prInfo->cid;
         oMDInfoMsg.m_prMDInfo->szManufacturer = prInfo->manufacturer;
         oMDInfoMsg.m_prMDInfo->szModel = prInfo->model;
         oMDInfoMsg.m_prMDInfo->szProduct = prInfo->product;
         oMDInfoMsg.m_prMDInfo->szSerial = prInfo->serial;
         oMDInfoMsg.m_prMDInfo->szCodename = prInfo->codename;
         oMDInfoMsg.m_prMDInfo->szIncremental = prInfo->incremental;
         oMDInfoMsg.m_prMDInfo->szRelease = prInfo->release;
         oMDInfoMsg.m_prMDInfo->szSdk = prInfo->sdk;
         oMDInfoMsg.m_prMDInfo->u32Sdk = prInfo->sdk_int;
         oMDInfoMsg.m_prMDInfo->szToken = prInfo->token;

         t_String szBTAddress = prInfo->btaddress.c_str();
         std::transform(szBTAddress.begin(), szBTAddress.end(), szBTAddress.begin(), ::toupper);
         oMDInfoMsg.m_prMDInfo->szBTAddress = szBTAddress;

         poMsgQInterface->bWriteMsgToQ(&oMDInfoMsg, sizeof(oMDInfoMsg));
      }//if (NULL != poMsgQinterface)
   }//if (NULL != prInfo)
}

/***************************************************************************
 ** FUNCTION:  static t_Void vOnModuleStatusCb(...)
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vOnModuleStatusCb(S_MODULE_STATUS_LIST_MOBILE* prModuleStatusList)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vOnModuleStatusCb entered"));
   SPI_NORMAL_ASSERT(NULL == prModuleStatusList);

   if (NULL != prModuleStatusList)
   {
      ETG_TRACE_USR1(("[PARAM]::vOnModuleStatusCb: Number of modules changed = %d", prModuleStatusList->cnt));

      tBdclModuleStatusList moduleStatusList;

      S_MODULE_STATUS_MOBILE* prModuleStatus = prModuleStatusList->moduleStatus;
      for (t_U32 u32ListIndex = 0; u32ListIndex < (prModuleStatusList->cnt); ++u32ListIndex)
      {
         if (NULL != prModuleStatus)
         {
            ETG_TRACE_USR4(("[PARAM]::vOnModuleStatusCb: moduleID = %d, statusID = %d",
                  prModuleStatus->moduleID, prModuleStatus->statusID));

            trBdclModuleStatusInfo rModuleStatus;
            rModuleStatus.u32ModuleID = prModuleStatus->moduleID;
            rModuleStatus.u32StatusID = prModuleStatus->statusID;
            moduleStatusList.push_back(rModuleStatus);

            prModuleStatus = prModuleStatus->next;
         }
      }//for (t_U32 u32ListIndex = 0; ...)

      spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
      if ((NULL != poMsgQInterface) && (!moduleStatusList.empty()))
      {
         BDCLModuleStatusMsg oModStatusMsg;
         *(oModStatusMsg.m_pStatusList) = moduleStatusList;

         poMsgQInterface->bWriteMsgToQ(&oModStatusMsg, sizeof(oModStatusMsg));
      }//if (NULL != poMsgQinterface)
   }//if (NULL != prMobileStatusList)
}

/***************************************************************************
 ** FUNCTION:  static t_Void vOnExitCb(...)
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vOnExitCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vOnExitCb entered"));

   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLExitMsg oExitMsg;
      poMsgQInterface->bWriteMsgToQ(&oExitMsg, sizeof(oExitMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  static t_Void vOnFeatureConfigRequestCb(...)
 ***************************************************************************/
t_Void spi_tclBDCLCmdSession::vOnFeatureConfigRequestCb()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdSession::vOnFeatureConfigRequestCb entered"));

   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLFeatureConfigRequestMsg oFeatureConfigRequestMsg;
      poMsgQInterface->bWriteMsgToQ(&oFeatureConfigRequestMsg, sizeof(oFeatureConfigRequestMsg));
   }//if (NULL != poMsgQinterface)
}
/***************************************************************************
 ** FUNCTION:   tenBdclInitStatus enGetBdclInitState()
 ***************************************************************************/
tenBdclInitStatus spi_tclBDCLCmdSession::enGetBdclInitState(tBdclInitStatus enStatus)
{
   tenBdclInitStatus enInitStatus = e8CL_INIT_UNKNOWN_ERR;
   switch (enStatus)
   {
      case tBdclInitStatus::BDCLInit_NoError: {enInitStatus = e8CL_INIT_NO_ERR; } break;
      case tBdclInitStatus::BDCLInit_HUVersionError: {enInitStatus = e8CL_INIT_HUVERSION_ERR; } break;
      case tBdclInitStatus::BDCLInit_StatisticInfoError: {enInitStatus = e8CL_INIT_STATISTICINFO_ERR; } break;
      case tBdclInitStatus::BDCLInit_TimeoutError: {enInitStatus = e8CL_INIT_TIMEOUT_ERR; } break;
      case tBdclInitStatus::BDCLInit_ParamError: {enInitStatus = e8CL_INIT_PARAM_ERR; } break;
      case tBdclInitStatus::BDCLInit_NoMemory: {enInitStatus = e8CL_INIT_NOMEM_ERR; } break;
      case tBdclInitStatus::BDCLInit_FileIOError: {enInitStatus = e8CL_INIT_FILEIO_ERR; } break;
      case tBdclInitStatus::BDCLInit_HBThreadFail: {enInitStatus = e8CL_INIT_HBTHREAD_FAIL; } break;
   }
   return enInitStatus;
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
