/***********************************************************************/
/*!
* \file  spi_tclBDCLCmdInput.cpp
* \brief  Implementation of the Class spi_tclBDCLCmdInput
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         rur1kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <bdcl/UspiInputSource.h>
using namespace adit::bdcl;

#include "spi_tclBDCLCmdInput.h"
#include "SPITypes.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLCmdInput.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

static trBDCLKeyCode aKeyCode[]=
{
      {e32DEV_HOME, e8CL_KEYCODE_MAIN},
      {e32DEV_PHONE_CALL, e8CL_KEYCODE_PHONE_CALL},
      {e32DEV_PHONE_END, e8CL_KEYCODE_PHONE_END},
      {e32MULTIMEDIA_MUTE, e8CL_KEYCODE_MUTE},
      {e32DEV_SOFT_LEFT, e8CL_KEYCODE_SELECTOR_PREVIOUS},
      {e32DEV_SOFT_RIGHT, e8CL_KEYCODE_SELECTOR_NEXT},
      {e32DEV_BACKWARD, e8CL_KEYCODE_BACK},
      {e32MULTIMEDIA_NEXT, e8CL_KEYCODE_SEEK_ADD},
      {e32MULTIMEDIA_PREVIOUS, e8CL_KEYCODE_SEEK_SUB},
      {e32DEV_MEDIA, e8CL_KEYCODE_MEDIA},
      {e32DEV_NAV, e8CL_KEYCODE_NAVI},
      {e32DEV_MENU, e8CL_KEYCODE_OK},
      {e32DEV_OK, e8CL_KEYCODE_OK},
      {e32APP_KEYCODE_TELEPHONY, e8CL_KEYCODE_TEL},
      {e32MULTIMEDIA_PLAY, e8CL_KEYCODE_MEDIA_START},
      {e32MULTIMEDIA_PAUSE, e8CL_KEYCODE_MEDIA_STOP},
      {e32MULTIMEDIA_STOP, e8CL_KEYCODE_MEDIA_STOP},
      {e32DEV_PTT, e8CL_KEYCODE_VR_START},
      {e32DEV_DELETE, e8CL_KEYCODE_NUM_DEL},
      {e32DEV_CLEAR, e8CL_KEYCODE_NUM_CLEAR},
      {e32DEV_KEY_0, e8CL_KEYCODE_NUM_0},
      {e32DEV_KEY_1, e8CL_KEYCODE_NUM_1},
      {e32DEV_KEY_2, e8CL_KEYCODE_NUM_2},
      {e32DEV_KEY_3, e8CL_KEYCODE_NUM_3},
      {e32DEV_KEY_4, e8CL_KEYCODE_NUM_4},
      {e32DEV_KEY_5, e8CL_KEYCODE_NUM_5},
      {e32DEV_KEY_6, e8CL_KEYCODE_NUM_6},
      {e32DEV_KEY_7, e8CL_KEYCODE_NUM_7},
      {e32DEV_KEY_8, e8CL_KEYCODE_NUM_8},
      {e32DEV_KEY_9, e8CL_KEYCODE_NUM_9},
      {e32DEV_KEY_PLUS, e8CL_KEYCODE_NUM_PLUS},
      {e32DEV_KEY_STAR, e8CL_KEYCODE_NUM_10},
      {e32DEV_KEY_HASH, e8CL_KEYCODE_NUM_11},
      {e32TWODKNOB0_SHIFT_RIGHT, e8CL_KEYCODE_MOVE_RIGHT},
      {e32TWODKNOB0_SHIFT_LEFT, e8CL_KEYCODE_MOVE_LEFT},
      {e32TWODKNOB0_SHIFT_UP, e8CL_KEYCODE_MOVE_UP},
      {e32TWODKNOB0_SHIFT_DOWN, e8CL_KEYCODE_MOVE_DOWN},
      {e32TWODKNOB0_SHIFT_UP_RIGHT, e8CL_KEYCODE_MOVE_UP_RIGHT },
      {e32TWODKNOB0_SHIFT_UP_LEFT, e8CL_KEYCODE_MOVE_UP_LEFT},
      {e32TWODKNOB0_SHIFT_DOWN_RIGHT, e8CL_KEYCODE_MOVE_DOWN_RIGHT},
      {e32TWODKNOB0_SHIFT_DOWN_LEFT, e8CL_KEYCODE_MOVE_DOWN_LEFT}
};

static trBDCLVRState aVRState[]=
{
      {e8SPI_SPEECH_UNKNOWN, e8CL_VR_STATUS_NOT_SUPPORT},
      {e8SPI_SPEECH_SPEAKING, e8CL_VR_STATUS_RECORD_RUNNING},
      {e8SPI_SPEECH_END, e8CL_VR_STATUS_RECORD_IDLE}
};


/***************************************************************************
 ** FUNCTION:   spi_tclBDCLCmdInput()
 ***************************************************************************/
spi_tclBDCLCmdInput::spi_tclBDCLCmdInput():m_enBdclVRState (e8CL_VR_STATUS_NOT_SUPPORT)
{
   
   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::spi_tclBDCLCmdInput entered "));
   vSetBDCLKeyCodes();
   vMapBDCLSpeechAppState();
}

/***************************************************************************
 ** FUNCTION:   ~spi_tclBDCLCmdInput()
 ***************************************************************************/
spi_tclBDCLCmdInput::~spi_tclBDCLCmdInput()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::~spi_tclBDCLCmdInput entered "));
   m_KeyCodeMap.clear();
}

/***************************************************************************
 ** FUNCTION:   t_Bool bInitialize()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdInput::bInitialize()
{
   m_oEndpointLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::bInitialize entered "));

   t_Bool bResult = false;

   t_SptrBDCLProxyManager spoProxyManager = spi_tclBDCLProxyManager::getInstance();
   t_SptrBDCLCoreDataIntf spoCoreDataIntf = (spoProxyManager)?
            (spoProxyManager->spoGetCoreDataIntf()) : (nullptr);
   t_CoreCallbackDealer* poCallbackDealer = (spoCoreDataIntf)?
            (spoCoreDataIntf->poGetCoreCallbackDealer()) : (NULL);
   SPI_NORMAL_ASSERT(NULL == poCallbackDealer);

   m_spoInputProxy = (spoProxyManager)? (spoProxyManager->spoGetInputProxyInstance()) : (nullptr);
   SPI_NORMAL_ASSERT(!m_spoInputProxy);

   if ((m_spoInputProxy) && (NULL != poCallbackDealer))
   {
      m_spoInputSource = std::move(t_SptrBDCLInputSource(new adit::bdcl::UspiInputSource(this, poCallbackDealer)));
      SPI_NORMAL_ASSERT(!m_spoInputSource);

      bResult = (m_spoInputSource)? (m_spoInputSource->initialize()) : (false);
   }

   m_oEndpointLock.vUnlock();

   ETG_TRACE_USR1(("[DESC]::spi_tclBDCLCmdInput::bInitialize left with result = %d ", ETG_ENUM(BOOL, bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:   t_Void vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLCmdInput::vUninitialize()
{
   m_oEndpointLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::vUninitialize entered "));

   if (m_spoInputSource)
   {
      m_spoInputSource->teardown();
   }
   m_spoInputSource = nullptr;
   m_spoInputProxy = nullptr;

   m_oEndpointLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::vUninitialize left "));
}

/***************************************************************************
** FUNCTION:  t_Void vReportKey()
***************************************************************************/
t_Void spi_tclBDCLCmdInput::vReportKey(tenKeyMode enKeyMode, tenKeyCode enKeyCode)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclBDCLCmdInput::vReportKey entered: KeyMode = %d, KeyCode = %d",
            ETG_ENUM(KEY_MODE,enKeyMode), ETG_ENUM(SPI_KEY_CODE, enKeyCode)));

   tenBdclKeyCode enBdclKeyCode = enGetKeyCode(enKeyCode);

   if ((nullptr != m_spoInputProxy) && (e8CL_KEYCODE_UNKNOWN != enBdclKeyCode))
   {
      switch (enKeyMode)
      {
         case e8KEY_LONGPRESS:
         {
            t_Bool bVRStartedStatus = bGetVRStartStatus();
            if(e8CL_KEYCODE_VR_START == enBdclKeyCode)
            {
               if (true == bVRStartedStatus)
               {
                  //Workaround for NCG3D-104460. Restart VR on Longpress
                  m_spoInputProxy->vSendKeyCode(static_cast<t_U32> (e8CL_KEYCODE_VR_STOP));
                  ETG_TRACE_USR1(("[DESC]::spi_tclBDCLCmdInput::vReportKey VR Stopped and now Restarting"));
                  m_spoInputProxy->vSendKeyCode(static_cast<t_U32> (e8CL_KEYCODE_VR_START));

               }else
               {
                  m_spoInputProxy->vSendKeyCode(static_cast<t_U32> (e8CL_KEYCODE_VR_START));
               }
            }
            else
            {
               ETG_TRACE_USR4(("[DESC]::vReportKey: Long press of unsupported key %d reported", enBdclKeyCode));
            }

         }
            break;
         case e8KEY_PRESS:
         {
            t_Bool bVRStartedStatus = bGetVRStartStatus();
            if(e8CL_KEYCODE_VR_START == enBdclKeyCode)
            {
               if (true == bVRStartedStatus)
               {
                  m_spoInputProxy->vSendKeyCode(static_cast<t_U32> (e8CL_KEYCODE_VR_STOP));
               }else
               {
                  m_spoInputProxy->vSendKeyCode(static_cast<t_U32> (e8CL_KEYCODE_VR_START));
               }
            }
         }
            break;
         case e8KEY_RELEASE:
         {
            //For VR operations(e32DEV_PTT) only PRESS is considered not RELEASE
            if( e8CL_KEYCODE_VR_START != enBdclKeyCode)
            {
               m_spoInputProxy->vSendKeyCode(static_cast<t_U32> (enBdclKeyCode));
            }
         }
            break;
         default:
            ETG_TRACE_ERR(("[ERR]::vReportKey: entered with wrong key %d", enBdclKeyCode));
            break;
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]::vReportKey: No valid BDCL KeyCode is found or InputProxy Instance is NULL "));
   }
}

/***************************************************************************
** FUNCTION: tenBdclKeyCode spi_tclBDCLInputHandler::enGetKeyCode()
***************************************************************************/
tenBdclKeyCode spi_tclBDCLCmdInput::enGetKeyCode(tenKeyCode enKeyCode) const
{
   //ETG_TRACE_USR1(("spi_tclBDCLCmdInput::enGetKeyCode entered "));
   tenBdclKeyCode enBdclKeyCode = e8CL_KEYCODE_UNKNOWN;

   auto itrKeyCode = m_KeyCodeMap.find(enKeyCode);
   if(itrKeyCode != m_KeyCodeMap.end())
   {
	   enBdclKeyCode = itrKeyCode->second;
   }
   ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLCmdInput::enGetKeyCode: Key code = %d ", enBdclKeyCode));
   return enBdclKeyCode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLInputHandler::vSetBDCLKeyCodes()
 ***************************************************************************/
t_Void spi_tclBDCLCmdInput::vSetBDCLKeyCodes()
{
	ETG_TRACE_USR1(("spi_tclBDCLCmdInput::vSetBDCLKeyCodes entered "));

	//!  mapping between SPI and BDCL key-codes
	for (tU32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aKeyCode)) / sizeof(trBDCLKeyCode)); ++u32KeyIndex)
	{
		m_KeyCodeMap.insert(std::pair<tenKeyCode, tenBdclKeyCode>(
				aKeyCode[u32KeyIndex].enKeyCode, aKeyCode[u32KeyIndex].enBDCLKeyCode));
	}
}

/***************************************************************************
 ** FUNCTION:  void spi_tclBDCLCmdInput::onError()
 ***************************************************************************/
void spi_tclBDCLCmdInput::onError(tBdclErrorCodes inErrorCode)
{
   tenBdclCarLifeError enError = static_cast<tenBdclCarLifeError>(inErrorCode);
   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::onError entered: %d", ETG_ENUM(BDCL_ERROR, enError)));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLCmdInput::vMapBDCLSpeechAppState()
 ***************************************************************************/
t_Void spi_tclBDCLCmdInput::vMapBDCLSpeechAppState()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::vMapBDCLSpeechAppState entered "));

   //!  mapping between SPI and BDCL speech App State
   for (tU32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aVRState)) / sizeof(trBDCLVRState)); ++u32KeyIndex)
   {
      m_SpeechStateMap.insert(std::pair<tenSpeechAppState, tenBdclVRState>(
                           aVRState[u32KeyIndex].enSpeechAppState, aVRState[u32KeyIndex].enBdclVRState));
   }
}

/***************************************************************************
** FUNCTION: tenSpeechAppState spi_tclBDCLCmdInput::enGetBDCLVRState()
***************************************************************************/
tenBdclVRState spi_tclBDCLCmdInput::enGetBDCLVRState(tenSpeechAppState enSpeechAppState) const
{
   //ETG_TRACE_USR1(("spi_tclBDCLCmdInput::enGetBDCLVRState entered "));
   tenBdclVRState enBdclVRState = e8CL_VR_STATUS_NOT_SUPPORT;
   auto itrVRState = m_SpeechStateMap.find(enSpeechAppState);
   if(itrVRState != m_SpeechStateMap.end())
   {
      enBdclVRState = itrVRState->second;
   }
   ETG_TRACE_USR2(("[PARAM]::spi_tclBDCLCmdInput::enGetBDCLVRState: VR State = %d ", enBdclVRState));
   return enBdclVRState;
}

/***************************************************************************
 ** FUNCTION:  void spi_tclBDCLCmdInput::bGetVRStartStatus()
 ***************************************************************************/
t_Void spi_tclBDCLCmdInput::vReportVRStatus(tenSpeechAppState enSpeechAppState)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdInput::vReportVRStatus entered"));
   m_enBdclVRState = enGetBDCLVRState(enSpeechAppState);
}

/***************************************************************************
 ** FUNCTION:  void spi_tclBDCLCmdInput::bGetVRStartStatus()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdInput::bGetVRStartStatus()
{
   ETG_TRACE_USR2(("[DESC]:spi_tclBDCLCmdInput::bGetVRStartStatus(): VR state = %d ", m_enBdclVRState));
   return (e8CL_VR_STATUS_RECORD_RUNNING == m_enBdclVRState);
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
