/***********************************************************************/
/*!
* \file  spi_tclBDCLCmdInput.h
* \brief  Implementation of the Class spi_tclBDCLCmdInput
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         rur1kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version

\endverbatim
*************************************************************************/
#ifndef SPI_TCLBDCLCMDINPUT_
#define SPI_TCLBDCLCMDINPUT_

#include <memory>
#include <bdcl/AditInputSource.h>

#include "SPITypes.h"
#include "BaseTypes.h"
#include "BDCLTypes.h"
#include "BDCL_WrapperTypes.h"
#include "Lock.h"
#include "spi_tclBDCLProxyManager.h"


//structure containing the map between the SPI_KEYCODES and BDCL_KEYCODES
struct trBDCLKeyCode
{
	//! corresponds to SPI Key Codes
	tenKeyCode enKeyCode;
	//!corresponds to BDCL key codes
	tenBdclKeyCode enBDCLKeyCode;

};

//structure containing the map between the SPI_SPEECHSTATE and BDCL_SPEECHSTATE
struct trBDCLVRState
{
   //! corresponds to SPI Speech App State
   tenSpeechAppState enSpeechAppState;

   //!corresponds to BDCL Speech App State
   tenBdclVRState enBdclVRState;

};

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
typedef std::unique_ptr< adit::bdcl::AditInputSource >   t_SptrBDCLInputSource;


/*!
 * \class spi_tclBDCLCmdInput
 * \brief
 */
class spi_tclBDCLCmdInput: public adit::bdcl::IAditInputSourceCallbacks
{
   public:

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLCmdInput()
       ***************************************************************************/
      /*!
       * \fn       spi_tclBDCLCmdInput()
       * \brief    Constructor
       * \sa       ~spi_tclBDCLCmdInput()
       **************************************************************************/
      spi_tclBDCLCmdInput();

      /***************************************************************************
       ** FUNCTION:   ~spi_tclBDCLCmdInput()
       ***************************************************************************/
      /*!
       * \fn       ~spi_tclBDCLCmdInput()
       * \brief    Destructor
       * \sa       spi_tclBDCLCmdInput()
       **************************************************************************/
     virtual ~spi_tclBDCLCmdInput();

      /***************************************************************************
       ** FUNCTION:  t_Bool bInitialize()
       ***************************************************************************/
      /*!
       * \fn      bInitialize()
       * \brief   Creates and registers an input source for BDCL device
       * \sa      vUninitialize()
       **************************************************************************/
      t_Bool bInitialize();

      /***************************************************************************
       ** FUNCTION:  t_Bool vUninitialize()
       ***************************************************************************/
      /*!
       * \fn      vUninitialize()
       * \brief   Performs cleanup related to input source
       * \sa      bInitialize()
       **************************************************************************/
      t_Void vUninitialize();

      /***************************************************************************
       ** FUNCTION:  t_Void vReportKey()
       ***************************************************************************/
      /*!
       * \fn      vReportKey
       * \brief   Receives hard key events and forwards it to BDCL device
       * \param   enKeyCode : [IN] unique key code identifier
       * \param   enKeyMode :[IN] unique key mode identifier
       * \retval  NONE
       **************************************************************************/
       t_Void vReportKey(tenKeyMode enKeyMode, tenKeyCode enKeyCode);

      /***************************************************************************
       ** FUNCTION:  t_Void vReportVRStatus()
       ***************************************************************************/
      /*!
       * \fn      vReportVRStatus
       * \brief   Reporting VR status
      * \param   enSpeechAppState : [IN] Speech App State
       * \retval  NONE
       **************************************************************************/
      t_Void vReportVRStatus(tenSpeechAppState enSpeechAppState);

   private:

      /***************************************************************************
       ** FUNCTION: spi_tclBDCLCmdInput(const spi_tclBDCLCmdInput&)
       ***************************************************************************/
      /*!
       * \fn      spi_tclBDCLCmdInput(const spi_tclBDCLCmdInput&)
       * \brief   Copy constructor not implemented
       **************************************************************************/
      spi_tclBDCLCmdInput(const spi_tclBDCLCmdInput&);

      /***************************************************************************
       ** FUNCTION: const spi_tclBDCLCmdInput & operator=(const spi_tclBDCLCmdInput&);
       ***************************************************************************/
      /*!
       * \fn      const spi_tclBDCLCmdInput & operator=(const spi_tclBDCLCmdInput&);
       * \brief   assignment operator not implemented
       **************************************************************************/
      const spi_tclBDCLCmdInput & operator=(const spi_tclBDCLCmdInput&);

      /***************************************************************************
       ** FUNCTION:  t_Void vSetBDCLKeyCodes()
       ***************************************************************************/
      /*!
       * \fn      vSetBDCLKeyCodes
       * \brief   maps the SPI_KEYCODES to BDCL_KEYCODES
       * \param   NONE
       * \retval  NONE
       **************************************************************************/
      t_Void vSetBDCLKeyCodes();

      /***************************************************************************
       ** FUNCTION:  t_Void enGetKeyCode()
       ***************************************************************************/
      /*!
       * \fn      enGetKeyCode
       * \brief   Returns the corresponding BDCL_KEYCODE of SPI_KEYCODE
       * \param   NONE
       * \retval  tenBdclKeyCode
       **************************************************************************/
      tenBdclKeyCode enGetKeyCode(tenKeyCode enKeyCode) const;

      /***************************************************************************
       ** FUNCTION:  void spi_tclBDCLCmdInput::onError()
       ***************************************************************************/
      /*!
       * \fn      onError()
       * \brief   Callback to receive errors from Input endpoint
       * \param   [IN] inErrorCode: Error type enumeration
       **************************************************************************/
      virtual void onError(tBdclErrorCodes inErrorCode);

      /***************************************************************************
       ** FUNCTION:  t_Void vMapBDCLSpeechAppState()
       ***************************************************************************/
      /*!
       * \fn      vMapBDCLSpeechAppState
       * \brief   maps the SPI_SPEECHAPPSTATE to BDCL_VRSTATE
       * \param   NONE
       * \retval  NONE
       **************************************************************************/
      t_Void vMapBDCLSpeechAppState();

      /***************************************************************************
       ** FUNCTION:  tenSpeechAppState enGetBDCLVRState(tenBdclVRState enBdclVRState) const
       ***************************************************************************/
      /*!
       * \fn      enGetBDCLVRState
       * \brief  get the mapped value from mapping of SPI_SPEECHAPPSTATE and BDCL_VRSTATE
       * \param   NONE
       * \retval  NONE
       **************************************************************************/
      tenBdclVRState enGetBDCLVRState(tenSpeechAppState enSpeechAppState) const;

      /***************************************************************************
       ** FUNCTION:  t_Bool bGetVRStartStatus()
       ***************************************************************************/
      /*!
       * \fn      bGetVRStartStatus
       * \brief  get VR Start Status
       * \param   NONE
       * \retval  NONE
       **************************************************************************/
      t_Bool bGetVRStartStatus();


      //! Mapping between SPI and AAP keycodes
      std::map<tenKeyCode, tenBdclKeyCode> m_KeyCodeMap;

      //! Mapping between SPI and BDCL Speech AppStates
      std::map<tenSpeechAppState, tenBdclVRState> m_SpeechStateMap;

      //! Pointer to Input proxy
      t_SptrBDCLInputProxy m_spoInputProxy;

      //! Pointer to input source
      t_SptrBDCLInputSource m_spoInputSource;

      //! Lock object for Input source endpoint
      Lock  m_oEndpointLock;

      tenBdclVRState m_enBdclVRState;

};
#endif //SPI_TCLBDCLCMDINPUT_
