/***********************************************************************/
/*!
* \file  spi_tclBDCLCmdAudio.cpp
* \brief  Implementation of the Class spi_tclBDCLCmdAudio
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         rur1kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "spi_tclBDCLAudioSinkAdapt.h"
#include "spi_tclBDCLAudioSourceAdapt.h"
#include "spi_tclBDCLCmdAudio.h"
#include "spi_tclBDCLMsgQInterface.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLCmdAudio.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION:   spi_tclBDCLCmdAudio()
 ***************************************************************************/
spi_tclBDCLCmdAudio::spi_tclBDCLCmdAudio():
   m_poMicAudioSource(NULL),m_poAudioSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::spi_tclBDCLCmdAudio entered "));

   for(t_U8 u8AudSinkIndex = 0; u8AudSinkIndex < scu8MaxNumAudioSinks; ++u8AudSinkIndex)
   {
      m_mapAudioSinks[u8AudSinkIndex] = NULL;
   }
}

/***************************************************************************
 ** FUNCTION:   ~spi_tclBDCLCmdAudio()
 ***************************************************************************/
spi_tclBDCLCmdAudio::~spi_tclBDCLCmdAudio()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::~spi_tclBDCLCmdAudio entered "));
   m_poAudioSettings=NULL;
   vUninitialize();
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetAudioPipeConfig()
 ***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vSetAudioPipeConfig(const tmapAudioPipeConfig& crfmapAudioPipeConfig)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vSetAudioPipeConfig entered "));
   ETG_TRACE_USR2(("[DESC]:Set Audio pipeline configuration size = %d", crfmapAudioPipeConfig.size()));
   m_mapAudioPipeConfig = crfmapAudioPipeConfig;
}

/***************************************************************************
 ** FUNCTION:   t_Bool bInitialize()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdAudio::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::bInitialize entered "));

   t_Bool bResult = true;
   t_String szAudioPipeConfig;

   for (t_U8 u8AudSinkIndex = 0; u8AudSinkIndex < scu8MaxNumAudioSinks; ++u8AudSinkIndex)
   {
      tenBdclAudStreamType enAudioStream = static_cast<tenBdclAudStreamType>(u8AudSinkIndex);
      m_mapAudioSinks[u8AudSinkIndex] = new spi_tclBDCLAudioSinkAdapt(enAudioStream);
      SPI_NORMAL_ASSERT(NULL == m_mapAudioSinks[u8AudSinkIndex]);
      if( NULL!= m_mapAudioSinks[u8AudSinkIndex])
      {
          m_mapAudioSinks[u8AudSinkIndex]->vSetAudioSettingsInstance(m_poAudioSettings);
      }
      bResult = bResult && ((NULL != m_mapAudioSinks[u8AudSinkIndex]) &&
            (bGetAudioPipeConfig(enAudioStream, szAudioPipeConfig)) &&
            (m_mapAudioSinks[u8AudSinkIndex]->bInitialize(szAudioPipeConfig)));
   }//for (t_U8 u8NumAudSinks = 0...)

   m_poMicAudioSource = new spi_tclBDCLAudioSourceAdapt();
   SPI_NORMAL_ASSERT(NULL == m_poMicAudioSource);
   if( NULL!= m_poMicAudioSource)
   {
	   m_poMicAudioSource->vSetAudioSettingsInstance(m_poAudioSettings);
   }
   bResult = bResult && ((NULL != m_poMicAudioSource) &&
            (bGetAudioPipeConfig(e8CL_AUDTYPE_MIC, szAudioPipeConfig)) &&
            (m_poMicAudioSource->bInitialize(szAudioPipeConfig)));

   ETG_TRACE_USR1(("[PARAM]::spi_tclBDCLCmdAudio::bInitialize left with result = %d", ETG_ENUM(BOOL, bResult)));
	return bResult;
}

/***************************************************************************
 ** FUNCTION:   t_Void vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vUninitialize entered "));

   //! Note: AudioSink & AudioSource memory cleanup to be done by SPI.
   for (t_U8 u8AudSinkIndex = 0; u8AudSinkIndex < scu8MaxNumAudioSinks; ++u8AudSinkIndex)
   {
      RELEASE_MEM(m_mapAudioSinks[u8AudSinkIndex]);
   }//for (t_U8 u8AudSinkIndex = 0...)

   RELEASE_MEM(m_poMicAudioSource);

   //! Notify upper layers for cleanup
   //! @Note: Logic moved from Sink/Source endpoint classes to here to ensure endpoints are destroyed
   //! before notifying upper layers, to prevent invalid memory access of Sink/Source 
   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance();
   if (NULL != poMsgQInterface)
   {
      BDCLAudioRequestMsg oAudioMsg;
      oAudioMsg.m_rAudioRequest.enAudRequest = e8CL_AUD_REQ_STOP;

      for (t_U8 u8AudSinkIndex = 0; u8AudSinkIndex < scu8MaxNumAudioSinks; ++u8AudSinkIndex)
      {
         oAudioMsg.m_rAudioRequest.enAudStream = static_cast<tenBdclAudStreamType>(u8AudSinkIndex);
         poMsgQInterface->bWriteMsgToQ(&oAudioMsg, sizeof(oAudioMsg));
      }

      oAudioMsg.m_rAudioRequest.enAudStream = e8CL_AUDTYPE_MIC;
      poMsgQInterface->bWriteMsgToQ(&oAudioMsg, sizeof(oAudioMsg));
   }//if (NULL != poMsgQinterface)

   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vUninitialize left"));
}

/***************************************************************************
 ** FUNCTION:  t_Bool bStartAudioPlayback()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdAudio::bStartAudioPlayback(tenBdclAudStreamType enAudStream)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::bStartAudioPlayback entered for AudStreamType = %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));

   t_Bool bRetVal = false;
   switch (enAudStream)
   {
      case e8CL_AUDTYPE_MEDIA:
      case e8CL_AUDTYPE_TTS:
      case e8CL_AUDTYPE_VOICE:
      {
         if ((enAudStream < scu8MaxNumAudioSinks) && (NULL != m_mapAudioSinks[enAudStream]))
         {
            bRetVal = m_mapAudioSinks[enAudStream]->bStartAudioPlayback();
         }
      }
         break;
      case e8CL_AUDTYPE_MIC:
      {
         if (NULL != m_poMicAudioSource)
         {
            bRetVal = m_poMicAudioSource->bMicrophoneRequestCompleted(true);
         }
      }
         break;
      default:
         ETG_TRACE_ERR(("[ERR]::bStartAudioPlayback: Invalid audio stream type %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
         break;
   }

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void bStopAudioPlayback()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdAudio::bStopAudioPlayback(tenBdclAudStreamType enAudStream, t_Bool bFlushBuffer)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::bStopAudioPlayback entered for AudStreamType = %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));

   t_Bool bRetVal = false;
   switch (enAudStream)
   {
      case e8CL_AUDTYPE_MEDIA:
      case e8CL_AUDTYPE_TTS:
      case e8CL_AUDTYPE_VOICE:
      {
         if ((enAudStream < scu8MaxNumAudioSinks) && (NULL != m_mapAudioSinks[enAudStream]))
         {
            bRetVal = m_mapAudioSinks[enAudStream]->bStopAudioPlayback(bFlushBuffer);
         }
      }
         break;
      case e8CL_AUDTYPE_MIC:
      {
         if (NULL != m_poMicAudioSource)
         {
            bRetVal = m_poMicAudioSource->bMicrophoneRequestCompleted(false);
         }
      }
         break;
      default:
         ETG_TRACE_ERR(("[ERR]::bStopAudioPlayback: Invalid audio stream type %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream)));
         break;
   }

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetAudioStreamConfig(...)
 ***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vSetAudioStreamConfig(tenBdclAudStreamType enAudStream,
      t_String szConfigKey, tenAudioStreamConfig enAudioStreamConfig)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vSetAudioStreamConfig Entered"));
   if ((enAudStream < scu8MaxNumAudioSinks) && (NULL != m_mapAudioSinks[enAudStream]))
   {
      m_mapAudioSinks[enAudStream]->vSetAudioStreamConfig(
            szConfigKey, m_mapAudioPipeConfig[enAudioStreamConfig]);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclBDCLCmdAudio::vRegisterAudioCbs()
 ***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vRegisterAudioCbs(trBdclAudioCbs rAudioCbs)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vRegisterAudioCbs entered"));
   m_rAudioCbs = rAudioCbs;
}

/***************************************************************************
 ** FUNCTION: t_Void vSendAudioResponse()
 ***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vProcessAudioRequest(const trBdclAudioRequest& crfrRequest)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vProcessAudioRequest entered"));

   if (NULL != m_rAudioCbs.fvAudioRequestCb)
   {
      m_rAudioCbs.fvAudioRequestCb(crfrRequest);
   }
}

/***************************************************************************
** FUNCTION:  t_Void vSetModuleState()
***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vSetModuleState(tenBdclAudStreamType enAudStream, t_Bool bModuleActive)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vSetModuleState entered: Module=%d, IsActive=%d",
            ETG_ENUM(BDCL_AUDSTREAM_TYPE, enAudStream), ETG_ENUM(BOOL, bModuleActive)));

   if (NULL != m_rAudioCbs.fvAudioModStateCb)
   {
      m_rAudioCbs.fvAudioModStateCb(enAudStream, bModuleActive);
   }
}

/***************************************************************************
 ** FUNCTION:   t_Bool bGetAudioPipeConfig()
 ***************************************************************************/
t_Bool spi_tclBDCLCmdAudio::bGetAudioPipeConfig(tenBdclAudStreamType enAudioStream,
      t_String& rfszAudioPipeConfig)
{
   t_Bool bRetVal = true;
   switch (enAudioStream)
   {
      case e8CL_AUDTYPE_MEDIA:
         rfszAudioPipeConfig = m_mapAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE_BDCL];
         ETG_TRACE_USR4((" Audio media_standalone value for BDCL - %s",  m_mapAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_MEDIA_STANDALONE_BDCL].c_str()));
         break;
      case e8CL_AUDTYPE_TTS:
         rfszAudioPipeConfig = m_mapAudioPipeConfig[e8AUDIOCONFIG_ALTERNATEAUDIO];
         break;
      case e8CL_AUDTYPE_VOICE:
         rfszAudioPipeConfig = m_mapAudioPipeConfig[e8AUDIOCONFIG_MAINAUDIO_SPEECH];
         break;
      case e8CL_AUDTYPE_MIC:
         rfszAudioPipeConfig = m_mapAudioPipeConfig[e8AUDIOCONFIG_AUDIOIN_16kHz];
         break;
      default:
         bRetVal = false;
         break;
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclBDCLCmdAudio::vSetAudioSettingsInstance()
***************************************************************************/
t_Void spi_tclBDCLCmdAudio::vSetAudioSettingsInstance(spi_tclAudioSettingsIntf* poAudioSettingsIntf)
{
   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vSetAudioSettingsInstance entered"));
   if( NULL != poAudioSettingsIntf)
   {
	   ETG_TRACE_USR1(("spi_tclBDCLCmdAudio::vSetAudioSettingsInstance: Setting m_poAudioSettings val"));
       m_poAudioSettings = poAudioSettingsIntf;
   }
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
