/***********************************************************************/
/*!
* \file  spi_tclBDCLAudioSinkAdapt.h
* \brief  Implementation of the Class spi_tclBDCLAudioSinkAdapt
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    
AUTHOR:         pok6kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
21.04.2017  | Ramya Murthy          | Initial Version
06.03.2018  | Ramya Murthy          | Implemented notification of Audio interrupt msg (for VR_INTERRUPT)

\endverbatim
*************************************************************************/


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <bdcl/AilAudioSink.h>
using namespace adit::bdcl;

#include "spi_tclBDCLAudioSinkAdapt.h"
#include "spi_tclBDCLMsgQInterface.h"
#include "spi_tclBDCLAudioDispatcher.h"
#include "spi_tclBDCLProxyManager.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BDCLWRAPPER
      #include "trcGenProj/Header/spi_tclBDCLAudioSinkAdapt.cpp.trc.h"
   #endif
#endif

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#define DISPATCH_AUD_REQ(REQ_TYPE) \
   spi_tclBDCLMsgQInterface* poMsgQInterface = spi_tclBDCLMsgQInterface::getInstance(); \
   if (NULL != poMsgQInterface)  \
   {  \
      BDCLAudioRequestMsg oAudioMsg;   \
      oAudioMsg.m_rAudioRequest.enAudStream = m_cenStreamType; \
      oAudioMsg.m_rAudioRequest.enAudRequest = REQ_TYPE; \
      poMsgQInterface->bWriteMsgToQ(&oAudioMsg, sizeof(oAudioMsg)); \
   }//if (NULL != poMsgQinterface)

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const t_String sczAudSinkDeviceNameKey="audio-playback-device";

static const t_String sczAudSinkChnKey="audio-sink-channels";
static const t_String sczAudSinkSampleRateKey="audio-sink-sampling-rate";
static const t_String sczAudSinkBitsPerSampleKey="audio-sink-bits-per-sample";


/***************************************************************************
 ** FUNCTION:   spi_tclBDCLAudioSinkAdapt()
 ***************************************************************************/
spi_tclBDCLAudioSinkAdapt::spi_tclBDCLAudioSinkAdapt(tenBdclAudStreamType enAudStream):
         m_cenStreamType(enAudStream),m_poAudioSettings(NULL)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::spi_tclBDCLAudioSinkAdapt entered "));
   ETG_TRACE_USR2(("[DESC]::AudioSinkpoint created with Audio StreamType : %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  ~spi_tclBDCLAudioSinkAdapt()
 ***************************************************************************/
spi_tclBDCLAudioSinkAdapt::~spi_tclBDCLAudioSinkAdapt()
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::~spi_tclBDCLAudioSinkAdapt entered "));
   m_poAudioSettings = NULL;
   vUninitialize();
}

/***************************************************************************
 ** FUNCTION:  t_Bool bInitialize()
 ***************************************************************************/
t_Bool spi_tclBDCLAudioSinkAdapt::bInitialize(const t_String& crfszAudioPipeConfig)
{
   m_oAudSinkLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt:bInitialize entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   t_Bool bResult = false; 

   t_SptrBDCLProxyManager spoProxyManager = spi_tclBDCLProxyManager::getInstance();
   t_SptrBDCLCoreDataIntf spoCoreDataIntf = (spoProxyManager)?
            (spoProxyManager->spoGetCoreDataIntf()) : (nullptr);
   t_CoreCallbackDealer* poCallbackDealer = (spoCoreDataIntf)?
            (spoCoreDataIntf->poGetCoreCallbackDealer()) : (NULL);
   SPI_NORMAL_ASSERT(NULL == poCallbackDealer);

   if (NULL != poCallbackDealer)
   {
      m_spoAudioSink = std::move(tSptrAditAudioSink(new adit::bdcl::AilAudioSink(this, poCallbackDealer)));
      SPI_NORMAL_ASSERT(!m_spoAudioSink);

      vSetAudioStreamConfig(crfszAudioPipeConfig);
      bResult = m_spoAudioSink->initialize();
   }

   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR1(("[PARAM]::spi_tclBDCLAudioSinkAdapt::bInitialize left with result = %d (for StreamType %d)",
            ETG_ENUM(BOOL, bResult), ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Void vUninitialize()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSinkAdapt::vUninitialize()
{
   m_oAudSinkLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::vUninitialize entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));

   if (m_spoAudioSink)
   {
      m_spoAudioSink->playbackStop(true);
      ETG_TRACE_USR1(("[DESC] Audio sink playbackStop with buffer flush complete"));

      m_spoAudioSink->teardown();
      ETG_TRACE_USR1(("[DESC] Audio sink teardown complete"));
   }
   m_spoAudioSink = nullptr;

   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::vUninitialize left for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  t_Bool bStartAudioPlayback()
 ***************************************************************************/
t_Bool spi_tclBDCLAudioSinkAdapt::bStartAudioPlayback()
{
   m_oAudSinkLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::bStartAudioPlayback entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   t_Bool bResult = false;

   if (m_spoAudioSink)
   {
      ETG_TRACE_USR2(("[DESC] Before playbackStart (for StreamType %d)",
               ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
      bResult = m_spoAudioSink->playbackStart();

      ETG_TRACE_USR2(("[DESC] playbackStart completed with result %d (for StreamType %d)",
            ETG_ENUM(BOOL, bResult), ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   } //if (m_spoAudioSink)
   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::bStartAudioPlayback left with result %d (for StreamType %d)",
            ETG_ENUM(BOOL, bResult), ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Void bStopAudioPlayback()
 ***************************************************************************/
t_Bool spi_tclBDCLAudioSinkAdapt::bStopAudioPlayback(t_Bool bFlushBuffer)
{
   m_oAudSinkLock.s16Lock();

   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::bStopAudioPlayback entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   t_Bool bResult = false;

   if (m_spoAudioSink)
   {
      ETG_TRACE_USR2(("[DESC] Before playbackStop (for StreamType %d, FlushBuffer = %d)",
               ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType), ETG_ENUM(BOOL, bFlushBuffer)));
      bResult = m_spoAudioSink->playbackStop(bFlushBuffer);

      ETG_TRACE_USR2(("[DESC] playbackStop completed with result %d (for StreamType %d, FlushBuffer = %d)",
            ETG_ENUM(BOOL, bResult), ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType), ETG_ENUM(BOOL, bFlushBuffer)));

   } //if (m_spoAudioSink)
   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::bStopAudioPlayback left with result %d (for StreamType %d)",
            ETG_ENUM(BOOL, bResult), ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetAudioStreamConfig(...)
 ***************************************************************************/
t_Void spi_tclBDCLAudioSinkAdapt::vSetAudioStreamConfig(
      t_String szConfigKey, t_String szConfigValue)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::vSetAudioStreamConfig Entered "));
   ETG_TRACE_USR2(("[DESC]::Set Audio stream configuration with Config Key = %s ", szConfigKey.c_str()));
   ETG_TRACE_USR2(("[DESC]::Set Audio stream configuration with Config Value = %s ", szConfigValue.c_str()));
   //TO DO:Check if enStreamType parameter is required

   m_oAudSinkLock.s16Lock();
   if (m_spoAudioSink)
   {
      m_spoAudioSink->setConfigItem(szConfigKey.c_str(), szConfigValue.c_str());
      t_Bool bResult = m_spoAudioSink->initialize();
      ETG_TRACE_USR2(("[DESC]::Initialized Audio stream configuration with result %d ", ETG_ENUM(BOOL, bResult)));
   } //if (m_spoAudioSink)
   m_oAudSinkLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  void onPlaybackInit()
 ***************************************************************************/
void spi_tclBDCLAudioSinkAdapt::onPlaybackInit(unsigned int inSampleRate,
      unsigned int inChannelConfig, unsigned int inSampleFormat)
{
   m_oAudSinkLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::onPlaybackInit entered for "
         "StreamType %d, inSampleRate %d, inChannelConfig %d, inSampleFormat %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType),
         inSampleRate, inChannelConfig, inSampleFormat));

   if (m_spoAudioSink)
   {
      m_spoAudioSink->setConfigItem(sczAudSinkSampleRateKey, std::to_string(inSampleRate));
      m_spoAudioSink->setConfigItem(sczAudSinkChnKey, std::to_string(inChannelConfig));
      m_spoAudioSink->setConfigItem(sczAudSinkBitsPerSampleKey, std::to_string(inSampleFormat));

      //t_Bool bResult = m_spoAudioSink->initialize(); //Not required on track change
      //ETG_TRACE_USR2(("[DESC]::Initialised Audio stream configuration with result %d ", ETG_ENUM(BOOL, bResult)));

      DISPATCH_AUD_REQ(e8CL_AUD_REQ_START);
   } //if (m_spoAudioSink)
   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::onPlaybackInit left for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  void onPlaybackPause()
 ***************************************************************************/
void spi_tclBDCLAudioSinkAdapt::onPlaybackPause()
{
   m_oAudSinkLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::onPlaybackPause entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));

   if (m_spoAudioSink)
   {
      DISPATCH_AUD_REQ(e8CL_AUD_REQ_STOP);
   }
   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::onPlaybackPause left for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  void onPlaybackResume()
 ***************************************************************************/
void spi_tclBDCLAudioSinkAdapt::onPlaybackResume()
{
   m_oAudSinkLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::onPlaybackResume entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));

   if (m_spoAudioSink)
   {
      DISPATCH_AUD_REQ(e8CL_AUD_REQ_START);
   }

   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::onPlaybackResume left for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  void onPlaybackStop()
 ***************************************************************************/
void spi_tclBDCLAudioSinkAdapt::onPlaybackStop()
{
   m_oAudSinkLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::onPlaybackStop entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));

   if (m_spoAudioSink)
   {
      DISPATCH_AUD_REQ(e8CL_AUD_REQ_STOP);
   }
   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::onPlaybackStop left for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  void onPlaybackStop()
 ***************************************************************************/
void spi_tclBDCLAudioSinkAdapt::onPlaybackInterrupt()
{
   m_oAudSinkLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::onPlaybackInterrupt entered for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));

   //@Note: ADIT will take care of flushing buffer. SPI should use this msg for cleanup purpose only
   //bStopAudioPlayback(true);

   if (m_spoAudioSink)
   {
      DISPATCH_AUD_REQ(e8CL_AUD_REQ_INTERRUPT);
   }
   m_oAudSinkLock.vUnlock();

   ETG_TRACE_USR2(("spi_tclBDCLAudioSinkAdapt::onPlaybackInterrupt left for StreamType %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
}

/***************************************************************************
 ** FUNCTION:  void onError()
 ***************************************************************************/
void spi_tclBDCLAudioSinkAdapt::onError(tBdclErrorCodes inErrorCode)
{
   tenBdclCarLifeError enError = static_cast<tenBdclCarLifeError>(inErrorCode);
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::onError entered for StreamType %d, Error %d",
         ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType), ETG_ENUM(BDCL_ERROR, enError)));

}

/***************************************************************************
 ** FUNCTION:  t_Void vSetAudioStreamConfig()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSinkAdapt::vSetAudioStreamConfig(const t_String& crfszAudioPipeConfig)
{
   ETG_TRACE_USR1(("spi_tclBDCLAudioSinkAdapt::vSetAudioStreamConfig Entered: AudioPipeConfig = %s ", crfszAudioPipeConfig.c_str()));

   
   if ((m_spoAudioSink) && (NULL != m_poAudioSettings))
   {
      std::map< t_String, t_String > mapAudSinkConfig;

      switch (m_cenStreamType)
      {
         case e8CL_AUDTYPE_MEDIA:
            m_poAudioSettings->vGetBdclMediaSinkConfig(mapAudSinkConfig);
            break;
         case e8CL_AUDTYPE_TTS:
            m_poAudioSettings->vGetBdclTTSSinkConfig(mapAudSinkConfig);
            break;
         case e8CL_AUDTYPE_VOICE:
            m_poAudioSettings->vGetBdclVRSinkConfig(mapAudSinkConfig);
            break;
         default:
            ETG_TRACE_ERR(("[ERR]::vSetAudioStreamConfig: Invalid Stream Type %d",
                  ETG_ENUM(BDCL_AUDSTREAM_TYPE, m_cenStreamType)));
            break;
      } //switch (m_cenStreamType)

      if (!mapAudSinkConfig.empty())
      {
         m_spoAudioSink->setConfigItem(sczAudSinkDeviceNameKey, crfszAudioPipeConfig.c_str());

         for (auto itr = mapAudSinkConfig.begin(); itr != mapAudSinkConfig.end(); ++itr)
         {
            if ((false == itr->first.empty()) && (false == itr->second.empty()))
            {
               m_spoAudioSink->setConfigItem(itr->first, itr->second);
            }
            else
            {
               ETG_TRACE_ERR(("[ERR]::vSetAudioStreamConfig: Skipping empty entry in map"));
            }
         }
         ETG_TRACE_USR2(("[DESC] Audio source configuration complete"));
      }
   } //if ((m_spoAudioSink) && (NULL != poAudSettings))
}

/***************************************************************************
 ** FUNCTION:  t_Void vSetAudioSettingsInstance()
 ***************************************************************************/
t_Void spi_tclBDCLAudioSinkAdapt:: vSetAudioSettingsInstance (spi_tclAudioSettingsIntf* poAudioSettingsIntf)
{
   ETG_TRACE_USR1(("vSetAudioSettingsInstance::vSetBDCLAudioSinkConfig Entered:"));
   if( NULL != poAudioSettingsIntf)
   {
	   ETG_TRACE_USR1(("vSetAudioSettingsInstance::vSetBDCLAudioSinkConfig: Setting m_poAudioSettings Value"));
       m_poAudioSettings = poAudioSettingsIntf;
   }
}
