/*!
 *******************************************************************************
 * \file              BDCLTypes.h
 * \brief             Baidu Carlife Types
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Common Baidu Carlife Types
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 19.04.2017 |  Ramya Murthy                | Initial Version

 \endverbatim
 ******************************************************************************/
#ifndef BDCLTYPES_H_
#define BDCLTYPES_H_

#include <vector>
#include <cstdlib>
#include <cstring>
#include <functional>

#include "BaseTypes.h"
#include "SPITypes.h"

#define STR_METADATA_LENGTH 200
#define STR_ALBUMART_MAX_SIZE 1048576 // 1 MByte
#define STR_ROADNAME_LENGTH 200

enum tenBDCLModuleID
{
   e32MODULEID_BDCLDISCOVERER = 1,
   e32MODULEID_BDCLSESSION = 2,
   e32MODULEID_BDCLAUDIO = 3,
   e32MODULEID_BDCLVIDEO = 4,
   e32MODULEID_BDCLTOUCH = 5,
   e32MODULEID_BDCLSENSOR = 6,
   e32MODULEID_BDCLBLUETOOTH = 7,
   e32MODULEID_BDCLMEDIADATA = 8,
   e32MODULEID_BDCLNAVIGATIONTBT = 9,
   e32MODULEID_BDCLCARDATA = 10,
   e32MODULEID_BDCLNOTIFICATION = 11
};

//!Identifies various Audio streams from BDCL device.
enum tenBdclAudStreamType
{
   e8CL_AUDTYPE_MEDIA = 0,
   e8CL_AUDTYPE_TTS = 1,
   e8CL_AUDTYPE_VOICE = 2,
   e8CL_AUDTYPE_MIC = 3
};

//TODO - generic, move to SPI Types?
enum tenAudioMuteState
{
   //! Indicates default value or no change
   e8AUD_MUTE_DEFAULT = 0,
   //! Indicates mute is not active
   e8AUD_UNMUTED = 1,
   //! Indicates mute is active
   e8AUD_MUTED = 2,
};

enum tenBdclAudioState
{
   //! Indicates default value or no change
   e8CL_AUD_ST_DEFAULT = 0,
   //! Indicates audio playback has stopped
   e8CL_AUD_ST_IDLE = 1,
   //! Indicates audio playback has been paused by HU due to an active transient source
   e8CL_AUD_ST_PAUSED = 2,
   //! Indicates audio is playing
   e8CL_AUD_ST_PLAYING = 3,
};

enum tenBdclStreamState
{
   e8CL_AUD_STREAM_CLOSED = 0, // Playback pause/stop received. Audio streaming ended (default state)
   e8CL_AUD_STREAM_OPEN = 1,   // Playback init/resume received. Blocked until Audio Activation.
   e8CL_AUD_STREAMING = 2,     // Audio Activation completed. Device configured. Audio streaming in progress.
};

enum tenBdclAudioConstraint
{
   //! Indicates default value or no change
   e8CL_CONSTRAINT_DEFAULT = 0,
   //! Indicates audio can be requested anytime
   e8CL_CONSTRAINT_ANYTIME = 1,
   //! Indicates audio cannot be requested
   e8CL_CONSTRAINT_NEVER = 2,
};

//! Indicates audio context type
enum tenBdclAudioContext
{
   e8CL_AUDCTXT_MAIN = 0,
   e8CL_AUDCTXT_TRANS = 1,
   e8CL_AUDCTXT_TRANS_MIX = 2,
   e8CL_AUDCTXT_TRANS_HIGHPRIO = 3,
   e8CL_AUDCTXT_MUTE = 4
};

enum tenBdclAudioRequestType
{
   e8CL_AUD_REQ_START = 0,
   e8CL_AUD_REQ_STOP = 1,
   e8CL_AUD_REQ_INTERRUPT = 2
};

enum tenBdclAudioResponse
{
   e8CL_AUD_RESP_INVALID = 0,
   e8CL_AUD_RESP_ACCEPT = 1,
   e8CL_AUD_RESP_DENY = 2,
};

//! Indicates audio resource owner
enum tenBdclAudioResourceRequest
{
   e8RSRC_REQUEST = 0,
   e8RSRC_RELEASE = 1
};

struct trBdclAudioStates
{
      tenAudioMuteState enMuteState;
      tenBdclAudioConstraint enTakeConstraint;
      tenBdclAudioConstraint enBorrowConstraint;
      tenBdclAudioConstraint enMixConstraint;

      tenBdclAudioState enMusicState;
      tenBdclAudioState enVRState;
      tenBdclAudioState enTTSState;

      trBdclAudioStates& operator=(const trBdclAudioStates &crfrStates)
      {
         enMuteState = crfrStates.enMuteState;
         enTakeConstraint = crfrStates.enTakeConstraint;
         enBorrowConstraint = crfrStates.enBorrowConstraint;
         enMixConstraint = crfrStates.enMixConstraint;
         enMusicState = crfrStates.enMusicState;
         enVRState = crfrStates.enVRState;
         enTTSState = crfrStates.enTTSState;
         return *this;
      }
      bool operator==(const trBdclAudioStates &crfrStates)
      {
         return ((enMuteState == crfrStates.enMuteState) &&
               (enTakeConstraint == crfrStates.enTakeConstraint) &&
               (enBorrowConstraint == crfrStates.enBorrowConstraint) &&
               (enMixConstraint == crfrStates.enMixConstraint) &&
               (enMusicState == crfrStates.enMusicState) &&
               (enVRState == crfrStates.enVRState) &&
               (enTTSState == crfrStates.enTTSState));
      }
};

struct trBdclAudioTypeLookup
{
      tenAudioContext enAudContext;
      tenBdclAudioContext enCLAudContext;

      trBdclAudioTypeLookup& operator=(const trBdclAudioTypeLookup &crfrLookup)
      {
         enAudContext = crfrLookup.enAudContext;
         enCLAudContext = crfrLookup.enCLAudContext;
         return *this;
      }
};

struct trBdclAudioResponseLookup
{
      tenBdclAudioRequestType enAudRequest;

      tenAudioMuteState enCurMuteState;
      tenBdclAudioConstraint enCurAudConstraint;
      tenBdclAudioState enCurMusicState;
      tenBdclAudioState enCurVRState;
      tenBdclAudioState enCurTTSState;

      tenBdclAudioState enNewMusicState;
      tenBdclAudioState enNewVRState;
      tenBdclAudioState enNewTTSState;
      tenBdclAudioResponse enAudResponse;

      trBdclAudioResponseLookup& operator=(const trBdclAudioResponseLookup &crfrResponseLookup)
      {
         enAudRequest = crfrResponseLookup.enAudRequest;
         enCurMuteState = crfrResponseLookup.enCurMuteState;
         enCurAudConstraint = crfrResponseLookup.enCurAudConstraint;
         enCurMusicState = crfrResponseLookup.enCurMusicState;
         enCurVRState = crfrResponseLookup.enCurVRState;
         enCurTTSState = crfrResponseLookup.enCurTTSState;
         enNewMusicState = crfrResponseLookup.enNewMusicState;
         enNewVRState = crfrResponseLookup.enNewVRState;
         enNewTTSState = crfrResponseLookup.enNewTTSState;
         enAudResponse = crfrResponseLookup.enAudResponse;
         return *this;
      }

};

struct trBdclAudioStatusNotifLookup
{
      tenBdclAudioResourceRequest enAudRsrcRequest;
      tenBdclAudioContext enAudContext;

      tenAudioMuteState enCurMuteState;
      tenBdclAudioState enCurMusicState;
      tenBdclAudioState enCurVRState;
      tenBdclAudioState enCurTTSState;

      tenAudioMuteState enNewMuteState;
      tenBdclAudioState enNewMusicState;
      tenBdclAudioState enNewVRState;
      tenBdclAudioState enNewTTSState;

      tenBdclAudioConstraint enNewTakeConstraint;
      tenBdclAudioConstraint enNewBorrowConstraint;
      tenBdclAudioConstraint enNewMixConstraint;

      trBdclAudioStatusNotifLookup& operator=(const trBdclAudioStatusNotifLookup &crfrStatusLookup)
      {
         enAudRsrcRequest = crfrStatusLookup.enAudRsrcRequest;
         enAudContext = crfrStatusLookup.enAudContext;
         enCurMuteState = crfrStatusLookup.enCurMuteState;
         enCurMusicState = crfrStatusLookup.enCurMusicState;
         enCurVRState = crfrStatusLookup.enCurVRState;
         enCurTTSState = crfrStatusLookup.enCurTTSState;
         enNewMuteState = crfrStatusLookup.enNewMuteState;
         enNewMusicState = crfrStatusLookup.enNewMusicState;
         enNewVRState = crfrStatusLookup.enNewVRState;
         enNewTTSState = crfrStatusLookup.enNewTTSState;
         enNewTakeConstraint = crfrStatusLookup.enNewTakeConstraint;
         enNewBorrowConstraint = crfrStatusLookup.enNewBorrowConstraint;
         enNewMixConstraint = crfrStatusLookup.enNewMixConstraint;
         return *this;
      }
};

struct trBdclAudioRequest
{
      tenBdclAudStreamType enAudStream;
      tenBdclAudioRequestType enAudRequest;
      trBdclAudioRequest() :
         enAudStream(e8CL_AUDTYPE_MEDIA), enAudRequest(e8CL_AUD_REQ_STOP)
      {
      }
      trBdclAudioRequest(tenBdclAudStreamType enStream, tenBdclAudioRequestType enRequest) :
         enAudStream(enStream), enAudRequest(enRequest)
      {
      }
};

//! Callback types
typedef std::function<void(const trBdclAudioRequest&)> tfvBdclAudioReqCb;
typedef std::function<void(tenBdclAudStreamType, t_Bool)> tfvBdclAudioModStateCb;

struct trBdclAudioCbs
{
      tfvBdclAudioReqCb fvAudioRequestCb;
      tfvBdclAudioModStateCb fvAudioModStateCb;

      trBdclAudioCbs() :
         fvAudioRequestCb(NULL), fvAudioModStateCb(NULL)
      {
      }
};

enum tenBdclPlayMode
{
   e8PLAYMODE_REPEATE_ONE = 0,
   e8PLAYMODE_SHUFFLE = 1,
   e8PLAYMODE_REPEATE_ALL = 2
};

enum tenBdclInitStatus
{
   e8CL_INIT_NO_ERR = 0,
   e8CL_INIT_HUVERSION_ERR = -1,
   e8CL_INIT_STATISTICINFO_ERR = -2,
   e8CL_INIT_TIMEOUT_ERR = -3,
   e8CL_INIT_PARAM_ERR = -4,
   e8CL_INIT_NOMEM_ERR = -5,
   e8CL_INIT_FILEIO_ERR = -4,
   e8CL_INIT_HBTHREAD_FAIL = -5,
   //! Above values are as per ADIT. Dont change.
   e8CL_INIT_UNKNOWN_ERR = 255
};

enum tenBdclCarLifeError
{
   e8CL_AOAP_ERROR_UNKNOWN = 0,
   /* Audio Errors */
   e8CL_AUDIO_SINK_CONFIGURATION_ERROR  = 101,
   e8CL_AUDIO_SINK_START_ERROR          = 102,
   e8CL_AUDIO_SINK_WRITE_ERROR          = 103,
   e8CL_AUDIO_SOURCE_READ_ERROR         = 151,
   /* Video Errors */
   e8CL_VIDEO_SINK_CONFIGURATION_ERROR  = 201,
   e8CL_VIDEO_SINK_START_ERROR          = 202,
   e8CL_VIDEO_SINK_RENDER_ERROR         = 203,
   /* Input Touch Errors */
   e8CL_ATTACHED_TOUCH_INIT_ERROR       = 301,
   e8CL_INPUT_EVENT_DISPATCH_ERROR      = 302,
   /* AOAP Errors */
   e8CL_AOAP_NO_ERROR                   = 400,
   e8CL_AOAP_IO_ERROR                   = 401,
   e8CL_AOAP_INVALID_PARAM_ERROR        = 402,
   e8CL_AOAP_ACCESS_ERROR               = 403,
   e8CL_AOAP_NO_DEVICE_ERROR            = 404,
   e8CL_AOAP_NOT_FOUND_ERROR            = 405,
   e8CL_AOAP_BUSY_ERROR                 = 406,
   e8CL_AOAP_TIMEOUT_ERROR              = 407,
   e8CL_AOAP_OVERFLOW_ERROR             = 408,
   e8CL_AOAP_PIPE_ERROR                 = 409,
   e8CL_AOAP_INTERRUPTED_ERROR          = 410,
   e8CL_AOAP_NO_MEM_ERROR               = 411,
   e8CL_AOAP_NOT_SUPPORTED_ERROR        = 412,
   e8CL_AOAP_READ_TIMEOUT_ERROR         = 413,
   e8CL_AOAP_WRITE_TIMEOUT_ERROR        = 414
   //! Above values are as per ADIT. Dont change.
};

//! BDCL Hard Key values
enum tenBdclKeyCode
{
   e8CL_KEYCODE_UNKNOWN = 0,
   e8CL_KEYCODE_HOME = 0x00000001,
   e8CL_KEYCODE_PHONE_CALL = 0x00000002,
   e8CL_KEYCODE_PHONE_END = 0x00000003,
   //e8CL_KEYCODE_PHONE_END_MUTE = 0x00000004, //!Abandoned as per spec
   e8CL_KEYCODE_HFP = 0x00000005,
   e8CL_KEYCODE_SELECTOR_NEXT = 0x00000006,
   e8CL_KEYCODE_SELECTOR_PREVIOUS = 0x00000007,
   //e8CL_KEYCODE_SETTING = 0x00000008, //!Abandoned as per spec
   e8CL_KEYCODE_MEDIA = 0x00000009,
   //e8CL_KEYCODE_RADIO = 0x0000000A, //!Abandoned as per spec
   e8CL_KEYCODE_NAVI = 0x0000000B,
   //e8CL_KEYCODE_SRC = 0x0000000C, //!Abandoned as per spec
   //e8CL_KEYCODE_MODE = 0x0000000D, //!Abandoned as per spec
   e8CL_KEYCODE_BACK = 0x0000000E,
   e8CL_KEYCODE_SEEK_SUB = 0x0000000F,
   e8CL_KEYCODE_SEEK_ADD = 0x00000010,
   //e8CL_KEYCODE_VOLUME_SUB = 0x00000011, //!Abandoned as per spec
   //e8CL_KEYCODE_VOLUME_ADD = 0x00000012, //!Abandoned as per spec
   e8CL_KEYCODE_MUTE = 0x00000013,
   e8CL_KEYCODE_OK = 0x00000014,
   e8CL_KEYCODE_MOVE_LEFT = 0x00000015,
   e8CL_KEYCODE_MOVE_RIGHT = 0x00000016,
   e8CL_KEYCODE_MOVE_UP = 0x00000017,
   e8CL_KEYCODE_MOVE_DOWN = 0x00000018,
   e8CL_KEYCODE_MOVE_UP_LEFT = 0x00000019,
   e8CL_KEYCODE_MOVE_UP_RIGHT = 0x0000001A,
   e8CL_KEYCODE_MOVE_DOWN_LEFT = 0x0000001B,
   e8CL_KEYCODE_MOVE_DOWN_RIGHT = 0x0000001C,
   e8CL_KEYCODE_TEL = 0x0000001D,
   e8CL_KEYCODE_MAIN = 0x0000001E, //TODO
   e8CL_KEYCODE_MEDIA_START = 0x0000001F,
   e8CL_KEYCODE_MEDIA_STOP = 0x00000020,
   e8CL_KEYCODE_VR_START = 0x00000021,
   e8CL_KEYCODE_VR_STOP = 0x00000022,
   e8CL_KEYCODE_NUM_0 = 0x00000023,
   e8CL_KEYCODE_NUM_1 = 0x00000024,
   e8CL_KEYCODE_NUM_2 = 0x00000025,
   e8CL_KEYCODE_NUM_3 = 0x00000026,
   e8CL_KEYCODE_NUM_4 = 0x00000027,
   e8CL_KEYCODE_NUM_5 = 0x00000028,
   e8CL_KEYCODE_NUM_6 = 0x00000029,
   e8CL_KEYCODE_NUM_7 = 0x0000002A,
   e8CL_KEYCODE_NUM_8 = 0x0000002B,
   e8CL_KEYCODE_NUM_9 = 0x0000002C,
   e8CL_KEYCODE_NUM_10 = 0x0000002D,
   e8CL_KEYCODE_NUM_11 = 0x0000002E,
   e8CL_KEYCODE_NUM_DEL = 0x0000002F,
   e8CL_KEYCODE_NUM_CLEAR = 0x00000030,
   e8CL_KEYCODE_NUM_PLUS = 0x00000031,
};

enum tenBdclAction
{
   e32BDCL_ACTION_INVALID         =  0,
   e32BDCL_ACTION_SHOW            =  1,
   e32BDCL_ACTION_UPDATE          =  2,
   e32BDCL_ACTION_HIDE            =  3
};

enum tenBdclTurnKind
{
   e32BDCL_TURN_KIND_UNKNOWN = 0x00000000,
   e32BDCL_TURN_KIND_FRONT =  0x00000001,
   e32BDCL_TURN_KIND_RIGHT_FRONT  =  0x00000002,
   e32BDCL_TURN_KIND_RIGHT  =  0x00000003,
   e32BDCL_TURN_KIND_RIGHT_BACK  =  0x00000004,
   e32BDCL_TURN_KIND_BACK  =  0x00000005,
   e32BDCL_TURN_KIND_LEFT_BACK  =  0x00000006,
   e32BDCL_TURN_KIND_LEFT  =  0x00000007,
   e32BDCL_TURN_KIND_LEFT_FRONT  =  0x00000008,
   e32BDCL_TURN_KIND_LEFT_SIDE  =  0x00000009,
   e32BDCL_TURN_KIND_RIGHT_SIDE  =  0x0000000A,
   e32BDCL_TURN_KIND_BRANCH_LEFT  =  0x0000000B,
   e32BDCL_TURN_KIND_BRANCH_CENTER  =  0x0000000C,
   e32BDCL_TURN_KIND_BRANCH_RIGHT  =  0x0000000D,
   e32BDCL_TURN_KIND_RING  =  0x0000000E,
   e32BDCL_TURN_KIND_INFERRY  =  0x0000000F,
   e32BDCL_TURN_KIND_OUTFERRY  =  0x000000010,
   e32BDCL_TURN_KIND_TOLLGATE  =  0x000000011,
   e32BDCL_TURN_KIND_LEFT_SIDE_MAIN  =  0x000000012,
   e32BDCL_TURN_KIND_RIGHT_SIDE_MAIN  =  0x000000013,
   e32BDCL_TURN_KIND_RIGHT_SIDE_IC  =  0x000000014,
   e32BDCL_TURN_KIND_LEFT_SIDE_IC  =  0x000000015,
   e32BDCL_TURN_KIND_LEFT_SIDE_STRAIGHT_IC  =  0x000000016,
   e32BDCL_TURN_KIND_RIGHT_SIDE_STRAIGHT_IC  =  0x000000017,
   e32BDCL_TURN_KIND_LEFT_SIDE_STRAIGHT  =  0x000000018,
   e32BDCL_TURN_KIND_RIGHT_SIDE_STRAIGHT  =  0x000000019,
   e32BDCL_TURN_KIND_BRANCH_LEFT_STRAIGHT  =  0x00000001A,
   e32BDCL_TURN_KIND_BRANCH_CENTER_STRAIGHT  =  0x00000001B,
   e32BDCL_TURN_KIND_BRANCH_RIGHT_STRAIGHT  =  0x00000001C,
   e32BDCL_TURN_KIND_BRANCH_LEFT_IC  =  0x00000001D,
   e32BDCL_TURN_KIND_BRANCH_CENTER_IC  =  0x00000001E,
   e32BDCL_TURN_KIND_BRANCH_RIGHT_IC  =  0x00000001F,
   e32BDCL_TURN_KIND_BRANCH_LEFT_IC_STRAIGHT  =  0x000000020,
   e32BDCL_TURN_KIND_BRANCH_CENTER_IC_STRAIGHT  =  0x000000021,
   e32BDCL_TURN_KIND_BRANCH_RIGHT_IC_STRAIGHT  =  0x000000022,
   e32BDCL_TURN_KIND_STRAIGHT_2BRANCH_LEFT_BASE  =  0x000000023,
   e32BDCL_TURN_KIND_STRAIGHT_2BRANCH_RIGHT_BASE  =  0x000000024,
   e32BDCL_TURN_KIND_STRAIGHT_3BRANCH_LEFT_BASE  =  0x000000025,
   e32BDCL_TURN_KIND_STRAIGHT_3BRANCH_MIDDLE_BASE  =  0x000000026,
   e32BDCL_TURN_KIND_STRAIGHT_3BRANCH_RIGHT_BASE  =  0x000000027,
   e32BDCL_TURN_KIND_LEFT_2BRANCH_LEFT_BASE =  0x000000028,
   e32BDCL_TURN_KIND_LEFT_2BRANCH_RIGHT_BASE =  0x000000029,
   e32BDCL_TURN_KIND_LEFT_3BRANCH_LEFT_BASE =  0x00000002A,
   e32BDCL_TURN_KIND_LEFT_3BRANCH_MIDDLE_BASE =  0x00000002B,
   e32BDCL_TURN_KIND_LEFT_3BRANCH_RIGHT_BASE =  0x00000002C,
   e32BDCL_TURN_KIND_RIGHT_2BRANCH_LEFT_BASE =  0x00000002D,
   e32BDCL_TURN_KIND_RIGHT_2BRANCH_RIGHT_BASE =  0x00000002E,
   e32BDCL_TURN_KIND_RIGHT_3BRANCH_LEFT_BASE =  0x00000002F,
   e32BDCL_TURN_KIND_RIGHT_3BRANCH_MIDDLE_BASE =  0x000000030,
   e32BDCL_TURN_KIND_RIGHT_3BRANCH_RIGHT_BASE =  0x000000031,
   e32BDCL_TURN_KIND_LEFT_FRONT_2BRANCH_LEFT_BASE =  0x000000032,
   e32BDCL_TURN_KIND_LEFT_FRONT_2BRANCH_RIGHT_BASE =  0x000000033,
   e32BDCL_TURN_KIND_RIGHT_FRONT_2BRANCH_LEFT_BASE =  0x000000034,
   e32BDCL_TURN_KIND_RIGHT_FRONT_2BRANCH_RIGHT_BASE =  0x000000035,
   e32BDCL_TURN_KIND_BACK_2BRANCH_LEFT_BASE =  0x000000036,
   e32BDCL_TURN_KIND_BACK_2BRANCH_RIGHT_BASE =  0x000000037,
   e32BDCL_TURN_KIND_BACK_3BRANCH_LEFT_BASE =  0x000000038,
   e32BDCL_TURN_KIND_BACK_3BRANCH_MIDDLE_BASE =  0x000000039,
   e32BDCL_TURN_KIND_BACK_3BRANCH_RIGHT_BASE =  0x00000003A
};


enum tenBdclAssistantType
{
   e32BDCL_ASSISTANT_TYPE_INVALID   =  0,
   e32BDCL_ASSISTANT_TYPE_TRAFFIC_SIGN   =  1,
   e32BDCL_ASSISTANT_TYPE_SPEED_CAMERA  =  2,
   e32BDCL_ASSISTANT_TYPE_TRAFFIC_LIGHT_CAMERA  = 3,
   e32BDCL_ASSISTANT_TYPE_PECCANRY_CAMERA = 4,
   e32BDCL_ASSISTANT_TYPE_INTERVAL_CAMERA  = 5
};

enum tenBdclRequestType
{
   e8REQUEST_STARTCALL =1 ,
   e8REQUEST_TERMINATECALL = 2,
   e8REQUEST_ANSWERCALL = 3,
   e8REQUEST_REJECTCALL = 4,
   e8REQUEST_DTMFCODE = 5,
   e8REQUEST_MUTEMIC = 6
};

enum tenBdclOSType
{
   e8CL_OSTYPE_ANDROID = 0,
   e8CL_OSTYPE_IOS = 1
};

enum tenBdclStatusType
{
	e8CL_STATUS_MIC = 1,
	e8CL_STATUS_UNKNOWN = 255
};

enum tenBdclBTIdentificationStatus
{
   e8CL_BT_IDENTIFY_SUCCESS = 1,
   e8CL_BT_IDENTIFY_FAILURE = 0
};

enum tenBdclIndicationType
{
   e8CL_INDICATION_NEWCALL = 1,
   e8CL_INDICATION_OUTCALL = 2,
   e8CL_INDICATION_CALLACTIVE = 3,
   e8CL_INDICATION_NOCALLACTIVE = 4,
   e8CL_INDICATION_MULTICALLACTIVE = 5,
   e8CL_INDICATION_MULTICALLINACTIVE = 6,
   e8CL_INDICATION_UNKNOWN = 255 // added as default value
};

enum tenBdclConnectionState
{
   e8CL_HFP_DISCONNECTED = 0,
   e8CL_HFP_CONNECTING = 1,
   e8CL_HFP_CONNECTED = 2
};

enum tenBdclGearState
{
   e8GEAR_NEUTRAL = 1,
   e8GEAR_PARK = 2,
   e8GEAR_DRIVE = 3,
   e8GEAR_LOW = 4,
   e8GEAR_REVERSE = 5
};

enum tenBdclCarDataModuleID
{
   e8CAR_DATA_GPS = 0,
   e8CAR_DATA_VELOCITY = 1,
   e8CAR_DATA_GYROSCOPE = 2,
   e8CAR_DATA_ACCELERATION = 3,
   e8CAR_DATA_GEAR = 4,
   e8CAR_DATA_OIL = 5
};

enum tenBdclCarLifeDataModuleID
{
   e8CARLIFE_DATA_TURNBYTURN = 0,
   e8CARLFIE_DATA_ASSISTANTGUIDE = 1
};

enum tenBdclHUBTStatus
{
	e8CL_BT_STATUS_IDLE = 0,
	e8CL_BT_STATUS_READY = 1,
	e8CL_BT_STATUS_CONNECTED = 2
};

struct trBdclHUInfo
{
      t_String szOS;
      t_String szBoard;
      t_String szBootloader;
      t_String szBrand;
      t_String szCpu_abi;
      t_String szCpu_abi2;
      t_String szDevice;
      t_String szDisplay;
      t_String szFingerprint;
      t_String szHardware;
      t_String szHost;
      t_String szCid;
      t_String szManufacturer;
      t_String szModel;
      t_String szProduct;
      t_String szSerial;
      t_String szCodename;
      t_String szIncremental;
      t_String szRelease;
      t_String szSdk;
      t_U32 u32Sdk;
      t_String szToken;
      t_String szBTAddress;

      trBdclHUInfo() :
         u32Sdk(0)
      {
      }
};

struct trBdclMDInfo
{
      t_String szOS;
      t_String szBoard;
      t_String szBootloader;
      t_String szBrand;
      t_String szCpu_abi;
      t_String szCpu_abi2;
      t_String szDevice;
      t_String szDisplay;
      t_String szFingerprint;
      t_String szHardware;
      t_String szHost;
      t_String szCid;
      t_String szManufacturer;
      t_String szModel;
      t_String szProduct;
      t_String szSerial;
      t_String szCodename;
      t_String szIncremental;
      t_String szRelease;
      t_String szSdk;
      t_U32 u32Sdk;
      t_String szToken;
      t_String szBTAddress;

      trBdclMDInfo() :
         u32Sdk(0)
      {
      }
};

struct trBdclStatisticsInfo
{
      t_String szCuid;
      t_String szVersionName;
      t_U32 u32VersionCode;
      t_String szChannel;
      t_U32 u32ConnectCount;
      t_U32 u32ConnectSuccessCount;
      t_U32 u32ConnectTime;
      t_String szCrashLog;

      trBdclStatisticsInfo() :
         u32VersionCode(0), u32ConnectCount(0), u32ConnectSuccessCount(0), u32ConnectTime(0)
      {
      }
};

struct trBdclFeatureConfigInfo
{
      t_String szKey;
      t_U32 u32Value;

      trBdclFeatureConfigInfo(t_String szKeyVal, t_U32 u32Val) :szKey(szKeyVal),
          u32Value(u32Val)
       {
       }
     trBdclFeatureConfigInfo() :szKey(),
         u32Value()
      {
      }
};

typedef std::vector< trBdclFeatureConfigInfo > tBdclFeatureConfigList;

struct trBdclInitInfo
{
      t_U32 s32AoapAccessoryId;
      t_U32 s32AoapDeviceId;
      t_Bool bIsAoapDevice;
      t_String szReadDevice;
      t_String szWriteDevice;

      trBdclInitInfo() :
         s32AoapAccessoryId(0), s32AoapDeviceId(0), bIsAoapDevice(false)
      {
      }
};
struct trBdclModuleStatusInfo
{
      t_U32 u32ModuleID; //TODO - convert to enums
      t_U32 u32StatusID;
      trBdclModuleStatusInfo() :
         u32ModuleID(0), u32StatusID(0)
      {
      }
};

typedef std::vector<trBdclModuleStatusInfo> tBdclModuleStatusList;

struct trBdclOOBInfo
{
      t_String strAddress;
      t_String strPassKey;
      t_String strHash;
      t_String strRandomizer;
      t_String strUuid;
      t_String strName;
      t_U32 u32Status;

      trBdclOOBInfo() :
         strAddress(""), strPassKey(""), strHash(""), strRandomizer(""), strUuid(""), strName(""), u32Status(0)
      {
      }
};

struct trBdclNaviNextTurnInfo
{
      tenBdclAction enAction;
      tenBdclTurnKind enNextTurn;
      t_Char cRoadName[STR_ROADNAME_LENGTH];
      t_Char cTurnIcon[STR_ALBUMART_MAX_SIZE];
      t_U32 u32TurnIconSize;
      t_U32 u32TotalDistance;
      t_U32 u32RemainingDistance;

      trBdclNaviNextTurnInfo():enAction(e32BDCL_ACTION_INVALID),enNextTurn(e32BDCL_TURN_KIND_UNKNOWN),
                               u32TurnIconSize(0),u32TotalDistance(0),u32RemainingDistance(0), cRoadName(""), cTurnIcon("")
      {
         //memset(cRoadName, 0, STR_ROADNAME_LENGTH);
         //memset(pszTurnIcon, 0, STR_ALBUMART_MAX_SIZE);
      }
};

struct trBdclNaviAssitantGuideInfo
{
      tenBdclAction enAction;
      tenBdclAssistantType enAssistantType;
      t_U32 u32TrafficSignType;
      t_U32 u32TotalDistance;
      t_U32 u32RemainingDistance;
      t_U32 u32CameraSpeed;
      trBdclNaviAssitantGuideInfo():enAction(e32BDCL_ACTION_INVALID),enAssistantType(e32BDCL_ASSISTANT_TYPE_INVALID),u32TrafficSignType(0),u32TotalDistance(0),u32RemainingDistance(0),u32CameraSpeed(0)
      {
      }
};

struct trBdclHfpIndicationInfo
{
      //TODO
};

struct trBdclBluetoothOOBInfo
{
      t_String szBTAddress;
      t_String szPassKey;
      t_String szHash;
      t_String szRamdomizer;
      t_String szUUID;
      t_String szName;
      tenBdclHUBTStatus enBTStatus;
      t_Bool bIsPaired;
      trBdclBluetoothOOBInfo():szBTAddress(""),szPassKey(""),szHash(""),szRamdomizer(""),szUUID(""),szName(""),enBTStatus(e8CL_BT_STATUS_IDLE),
                               bIsPaired(false)
      {
      }
};

struct trBdclBluetoothHFPRequest
{
      tenBdclRequestType enCommand;
      t_String szPhoneNum;
      t_U32 u32DTMFCode;
      trBdclBluetoothHFPRequest():enCommand(),szPhoneNum(""),u32DTMFCode() //check for default value enCommand
      {
      }
};

struct trBdclStartBluetoothAutopair
{
      tenBdclOSType enOSType;
      t_String szMDBTAddress;
      trBdclStartBluetoothAutopair():enOSType(e8CL_OSTYPE_ANDROID),szMDBTAddress("")
      {
      }
};

struct trBdclBluetoothIdentifyResultInd
{
      tenBdclBTIdentificationStatus enStatus;
      t_String szHUBTAddress;
      trBdclBluetoothIdentifyResultInd():enStatus(e8CL_BT_IDENTIFY_FAILURE),szHUBTAddress("")
      {
      }
};

struct trBdclBluetoothHFPIndication
{
      tenBdclIndicationType enIndicationType;
      t_String szPhoneNum;
      t_String szName;
      t_String szBTHUAddress;
      trBdclBluetoothHFPIndication():enIndicationType(e8CL_INDICATION_UNKNOWN),szPhoneNum(""),szName(""),szBTHUAddress("")
      {
      }
};

struct trBdclBluetoothHFPConn
{
      tenBdclConnectionState enConnectionState;
      t_String szHUBTAddress;
      t_String szName;
      trBdclBluetoothHFPConn():enConnectionState(e8CL_HFP_DISCONNECTED),szHUBTAddress(""),szName("")
      {
      }
};

struct trBdclBluetoothHFPResponse
{
      t_U32 u32Status;
      tenBdclRequestType enRequestType;
      t_U32 u32DtmfCode;
      trBdclBluetoothHFPResponse():u32Status(0),enRequestType(),u32DtmfCode(0)
      {
      }
};

struct trBdclBluetoothHFPStatusResponse
{
      t_U32 u32Status;
      tenBdclStatusType enStatustype;
      trBdclBluetoothHFPStatusResponse():u32Status(0),enStatustype(e8CL_STATUS_UNKNOWN)
      {
      }
};

struct trBdclHFPStatusRequest
{
      tenBdclStatusType enStatustype;
      trBdclHFPStatusRequest():enStatustype(e8CL_STATUS_UNKNOWN)
      {
      }
};

struct trBdclMediaMetadata
{
      t_Char cSource[STR_METADATA_LENGTH];
      t_Char cSong[STR_METADATA_LENGTH];
      t_Char cArtist[STR_METADATA_LENGTH];
      t_Char cAlbum[STR_METADATA_LENGTH];
      t_Char cAlbumArt[STR_ALBUMART_MAX_SIZE];
      t_Char cSongID[STR_METADATA_LENGTH];
      t_U32 u32Duration;
      t_U32 u32PlayListNumber;
      size_t sImageSize;
      tenBdclPlayMode enPlayMode;

      trBdclMediaMetadata() :
         u32Duration(0), u32PlayListNumber(0), enPlayMode(e8PLAYMODE_REPEATE_ONE), sImageSize(0),
         cSource(""), cSong(""), cArtist(""), cAlbum(""), cAlbumArt(""), cSongID("")
      {
         /*memset(cSource, 0, STR_METADATA_LENGTH);
          memset(cSong, 0, STR_METADATA_LENGTH);
          memset(cArtist, 0, STR_METADATA_LENGTH);
          memset(cAlbum, 0, STR_METADATA_LENGTH);
          memset(cAlbumArt, 0, STR_ALBUMART_MAX_SIZE);
          memset(cSongID, 0, STR_METADATA_LENGTH);*/
      }
};

struct trBdclVideoConfig
{
   t_U16 u16LayerID;
   t_U16 u16SurfaceID;
   t_U32 u32ScreenWidth;
   t_U32 u32ScreenHeight;
   t_U8 u8Fps;
   t_U16 u16ProjScreenWidth;
   t_U16 u16ProjScreenHeight;


   trBdclVideoConfig() :
         u16LayerID(0), u16SurfaceID(0), u32ScreenWidth(800), u32ScreenHeight(480), u8Fps(30),
                  u16ProjScreenWidth(800), u16ProjScreenHeight(480)
      {
      }
};

struct trBdclCarDataInfo
{
      tenBdclCarDataModuleID enCarDataModuleID;
      t_S32 s32SupportFlag;
      t_U32 u32Frequency;
      trBdclCarDataInfo() :
         enCarDataModuleID(e8CAR_DATA_GPS), s32SupportFlag(0),u32Frequency(0)
      {
      }
};

typedef std::vector<trBdclCarDataInfo> tBdclCarDataInfoList;

struct trBdclCarLifeInfo
{
      tenBdclCarLifeDataModuleID enCarlifeDataModuleID;
      t_Bool bSupportFlag;
      t_U32 u32Frequency;
      trBdclCarLifeInfo() :
         enCarlifeDataModuleID(e8CARLIFE_DATA_TURNBYTURN), bSupportFlag(false),u32Frequency(0)
      {
      }
};

typedef std::vector<trBdclCarLifeInfo> tBdclCarLifeInfoList;

struct trBdclCarVelocity
{
      t_U32 u32speed;
      t_U64 u64timestamp;

      trBdclCarVelocity() :
         u32speed(0), u64timestamp(0)
      {

      }
};

struct trBdclCarOil
{
      t_S32 s32level;
      t_S32 s32range;
      t_Bool bLowFuelWarning;

      trBdclCarOil() :
         s32level(0), s32range(0), bLowFuelWarning(false)
      {

      }
};

struct trBdclCarGPS
{
      t_S32 s32AntennaState;
      t_S32 s32SignalQuality;//in dBHz
      t_S32 s32Latitude;//in 10^6 degrees
      t_S32 s32Longitude;//in 10^6 degrees
      t_S32 s32Height;//in 10 meters
      t_U32 u32Speed;//in 100 km/h units
      t_U32 u32Heading;//in 10 degrees
      t_U32 u32Year;
      t_U32 u32Month;
      t_U32 u32Day;
      t_U32 u32Hrs;
      t_U32 u32Min;
      t_U32 u32Sec;
      t_U32 u32Fix;
      t_U32 u32hdop;
      t_U32 u32Pdop;
      t_U32 u32Vdop;
      t_U32 u32SatsUsed;
      t_U32 u32SatsVisible;
      t_U32 u32HorPosError;
      t_U32 u32VertPosError;
      t_S32 s32NorthSpeed;
      t_S32 s32EastSpeed;
      t_S32 s32VertSpeed;
      t_U64 u64TimeStamp;

      trBdclCarGPS() :
         s32AntennaState(0), s32SignalQuality(0), s32Latitude(0), s32Longitude(0), s32Height(0), u32Speed(0),
                  u32Heading(0), u32Year(0), u32Month(0), u32Day(0), u32Hrs(0), u32Min(0), u32Sec(0), u32Fix(0),
                  u32hdop(0), u32Pdop(0), u32Vdop(0), u32SatsUsed(0), u32SatsVisible(0), u32HorPosError(0),
                  u32VertPosError(0), s32NorthSpeed(0), s32EastSpeed(0), s32VertSpeed(0), u64TimeStamp(0)
      {
      }

};

enum tenBdclCmdModuleID
{
   e8CL_PHONE_MODULE_ID = 1,
   e8CL_NAVI_MODULE_ID = 2,
   e8CL_MUSIC_MODULE_ID = 3,
   e8CL_VR_MODULE_ID = 4,
   e8CL_CONNECT_MODULE_ID = 5,
   e8CL_MIC_MODULE_ID = 6,
   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclPhoneState
{
   e8CL_PHONE_STATUS_IDLE = 0,
   e8CL_PHONE_STATUS_INCOMING = 1,
   e8CL_PHONE_STATUS_OUTING = 2,
   e8CL_PHONE_STATUS_CONNECTED = 3

   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclNaviState
{
   e8CL_NAVI_STATUS_IDLE = 0,
   e8CL_NAVI_STATUS_START = 1,
   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclMusicState
{
   e8CL_MUSIC_STATUS_IDLE = 0,
   e8CL_MUSIC_STATUS_RUNNING = 1,
   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclVRState
{
   e8CL_VR_STATUS_RECORD_IDLE = 0,
   e8CL_VR_STATUS_RECORD_RUNNING = 1,
   e8CL_VR_STATUS_NOT_SUPPORT = 2,
   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclConnectState
{
   e8CL_CONNECT_STATUS_ADB = 1,
   e8CL_CONNECT_STATUS_AOA = 2,
   e8CL_CONNECT_STATUS_NCM_ANDROID = 3,
   e8CL_CONNECT_STATUS_NCM_IOS = 4,
   e8CL_CONNECT_STATUS_WIFI = 5,
   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclMicState
{
   e8CL_MIC_STATUS_USE_VEHICLE_MIC = 0,
   e8CL_MIC_STATUS_USE_MOBILE_MIC = 1,
   e8CL_MIC_STATUS_NOT_SUPPORTED = 2,
   //! Do not change above values since they are as per BDCL protocol.
};

enum tenBdclResponse
{
   e8CL_RESP_INVALID = 0,
   e8CL_RESP_ACCEPT = 1,
   e8CL_RESP_DENY = 2,
};

enum tenBdclVideoState
{
   e8CL_VID_UNINITIALISED = 0,
   e8CL_VID_INITIALISED = 1,
   e8CL_VID_REQUESTED = 2,
   e8CL_VID_STREAMING = 3,
   e8CL_VID_PAUSED = 4
};

enum tenBdclVideoFocusState
{
   e8CL_VIDEOFOCUSSTATE_UNKNOWN = 0,
   e8CL_VIDEOFOCUSSTATE_GAIN = 1,
   e8CL_VIDEOFOCUSSTATE_GAIN_TRANSIENT = 2,
   e8CL_VIDEOFOCUSSTATE_LOSS = 3,
   e8CL_VIDEOFOCUSSTATE_LOSS_TRANSIENT = 4
};

enum tenBdclVideoFocus
{
   e8CL_VIDEOFOCUS_UNKNOWN = 0,
   e8CL_VIDEOFOCUS_PROJECTED = 1,
   e8CL_VIDEOFOCUS_NATIVE = 2
};

enum tenBdclVideoPlaybackState
{
   e8VID_PB_STATE_START_RQSTD =0,
   e8VID_PB_STATE_STARTED =1,
   e8VID_PB_STATE_STOP_RQSTD =2,
   e8VID_PB_STATE_STOPPED =3
};

//It is used to hold the data mentioned in the config file,
//to derive the current HU Video focus state based on the context & display flag
struct trBdclAccVideoFocusNoti
{
   tenDisplayContext enAccDispCntxt;
   //This state is set, when HMI requests for Video Focus
   tenBdclVideoFocusState enAccFocusReqType;
   //This state is set, when HMI releases for Video Focus
   tenBdclVideoFocusState enAccFocusRelType;

   trBdclAccVideoFocusNoti& operator=(const trBdclAccVideoFocusNoti& corfrBdclAccVideoFocusNoti)
   {
      if(&corfrBdclAccVideoFocusNoti != this)
      {
         enAccDispCntxt = corfrBdclAccVideoFocusNoti.enAccDispCntxt;
         enAccFocusReqType = corfrBdclAccVideoFocusNoti.enAccFocusReqType;
         enAccFocusRelType = corfrBdclAccVideoFocusNoti.enAccFocusRelType;
      } //if(&corfrBdclAccVideoFocusNoti != this)
      return *this;
   }
};

struct trBdclVideoFocusState
{
   //Current MD Focus State
   tenBdclVideoFocusState enCurMDFocusState;
   //Accessory or MD Requested Focus State
   tenBdclVideoFocusState enReqFocusState;
   //New MD Focus State
   tenBdclVideoFocusState enUpdatedMDFocusState;

   trBdclVideoFocusState& operator=(const trBdclVideoFocusState& corfrBdclVideoFocusState)
   {
      if( &corfrBdclVideoFocusState != this )
      {
         enCurMDFocusState = corfrBdclVideoFocusState.enCurMDFocusState;
         enReqFocusState = corfrBdclVideoFocusState.enReqFocusState;
         enUpdatedMDFocusState = corfrBdclVideoFocusState.enUpdatedMDFocusState;
      }   //if( &corfrBdclVideoFocusState != this )
      return *this;
   }
};


struct trBdclMDVideoFocusReq
{
   //Current Accessory display context
   tenDisplayContext enAccDispCntxt;
   tenBdclResponse enMDFocusReqResp_StateLoss;
   tenBdclResponse enMDFocusReqResp_StateLossTransient;

   trBdclMDVideoFocusReq& operator=(const trBdclMDVideoFocusReq& corfrBdclMDVideoFocusReq)
   {
      if( &corfrBdclMDVideoFocusReq != this )
      {
         enAccDispCntxt = corfrBdclMDVideoFocusReq.enAccDispCntxt;
         enMDFocusReqResp_StateLoss = corfrBdclMDVideoFocusReq.enMDFocusReqResp_StateLoss;
         enMDFocusReqResp_StateLossTransient = corfrBdclMDVideoFocusReq.enMDFocusReqResp_StateLossTransient;
      }   //if( &corfrBdclMDVideoFocusReq != this )
      return *this;
   }
};

//! Callback for updating Speech States.

typedef std::function<void(tenBdclVRState)> tfvBdclVRState;
typedef std::function<void(tenSessionStatus)> tfvBdclSessionState;
typedef std::function<void(t_Bool, tenDisplayContextReason)> tfvUpdateDeviceDisplayCntxt;

struct trBdclResMngrCbs
{
      tfvBdclVRState fvBdclVRState;
      tfvBdclSessionState fvBdclSessionState;
      tfvUpdateDeviceDisplayCntxt fvUpdateDeviceDisplayCntxt;


      trBdclResMngrCbs() :
         fvBdclVRState(NULL), fvBdclSessionState(NULL),fvUpdateDeviceDisplayCntxt(NULL)
      {
      }
};

#endif //BDCLTYPES_H_
