/*!
 *******************************************************************************
 * \file             spi_tclBDCLAudio.h
 * \brief            Implements the Audio functionality for Baidu Carlife using
                     interface to CL Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for Baidu Carlife
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author              | Modifications
 18.04.2017 |  Ramya Murthy        | Initial Version

 \endverbatim
 ******************************************************************************/
#ifndef SPI_TCL_BDCLAUDIO_H
#define SPI_TCL_BDCLAUDIO_H

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include <memory>

#include "spi_tclAudioDevBase.h"
#include "SPITypes.h"
#include "BDCLTypes.h"
#include "Timer.h"
#include "Lock.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

//! Forward declarations
class spi_tclBDCLCmdAudio;

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
typedef std::shared_ptr< spi_tclBDCLCmdAudio >     t_SptrBDCLCmdAudio;

/**
 *  class definitions.
 * This class implements Audio functionality for Android Auto.
 */
class spi_tclBDCLAudio: public spi_tclAudioDevBase
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLAudio::spi_tclBDCLAudio();
       ***************************************************************************/
      /*!
       * \fn      spi_tclBDCLAudio()
       * \brief   Default Constructor
       **************************************************************************/
      spi_tclBDCLAudio();

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLAudio::~spi_tclBDCLAudio();
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclBDCLAudio()
       * \brief   Virtual Destructor
       **************************************************************************/
      virtual ~spi_tclBDCLAudio();

      /*************Start of functions overridden from spi_tclAudioDevBase********/

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLAudio::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  interface for the creator class to register for the required
       *         callbacks.
       * \param rfrAudCallbacks : reference to the callback structure
       *        populated by the caller
       **************************************************************************/
      t_Void vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bInitializeAudioPlayback(t_U32)
       ***************************************************************************/
      /*!
       * \fn      bInitializeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
       * \brief   Perform necessary actions to prepare for an Audio Playback.
       *          Function will be called prior to a Play Command from Audio Manager.
       *          Mandatory Interface to be implemented by Device Class.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device
       * \param   [enAudDir]: Audio route being allocated
       * \param   [enAudSampleRate]: Specifies the Audio Sample Rate.
       * \retval  Bool value
       **************************************************************************/
      t_Bool bInitializeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir,
            tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bFinalizeAudioPlayback(t_U32,tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      bFinalizeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir)
       * \brief   Perform necessary actions on completion of Audio Playback.
       *          Function will be called after to a Stop Command from Audio Manager.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
      * \param   [enAudDir]: Audio route being deallocated
       * \retval  Bool value
       **************************************************************************/
      t_Bool bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vSetAudioPipeConfig()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetAudioPipeConfig()
       * \brief   Set the Audio pipeline configuration for alsa devices
       * \param   crfmapAudioPipeConfig: Contains audio pipeline configuration
       **************************************************************************/
      t_Void vSetAudioPipeConfig(const tmapAudioPipeConfig& crfmapAudioPipeConfig);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bStartAudio(t_U32,t_String)
       ***************************************************************************/
      /*!
       * \fn      bStartAudio(t_U32 u32DeviceId, t_String szAudioDev)
       * \brief   Start Streaming of Audio from the CE Device to the Audio Output
       *          Device assigned by the Audio Manager for the Source.
       *          Mandatory Interface to be implemented.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       *         [szAudioDev]: ALSA Audio Device
       *          [enAudDir]    :Specify the Audio Direction(Alternate or Main Audio).
       * \retval  Bool value
       **************************************************************************/
      t_Bool bStartAudio(t_U32 u32DeviceId, t_String szAudioDev,
            tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bStartAudio(t_U32,t_String, t_String,
      **             tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
       *          t_String szInputAudioDev, tenAudioDir enAudDir)
       * \brief   Overloaded method to handle audio stream for Phone and VR.
       *          Start Streaming of Audio from the CE Device to the Audio Output
       *          Device assigned by the Audio Manager for the Source.
       *          Mandatory Interface to be implemented.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       *          [szOutputAudioDev]: Output ALSA Audio Device
       *          [szInputAudioDev] : Input ALSA Audio Device
       *           [enAudDir]        : Specify the Audio Direction(Phone or VR Audio).
       * \retval  Bool value
       **************************************************************************/
      t_Bool bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev,
              t_String szInputAudioDev, tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vStopAudio(t_U32)
       ***************************************************************************/
      /*!
       * \fn      vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir)
       * \brief   Stop Streaming of Audio from the CE Device to the Audio Output
       *          Device assigned by the Audio Manager for the Source.
       *          Mandatory Interface to be implemented.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       * \param   [enAudDir]   : Specify the Audio Direction.
       * \param   [bIsPaused]  : Indicates if source is paused or stopped
       * \retval  None
       **************************************************************************/
      t_Void vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bSelectAudioDevice(t_U32)
       ***************************************************************************/
      /*!
       * \fn      bSelectAudioDevice(t_U32 u32DeviceId)
       * \brief   Perform necessary actions specific to a device selection like
       *          obtaining audio capabilities of device, supported modes etc
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       * \retval  Bool value
       **************************************************************************/
      t_Bool bSelectAudioDevice(t_U32 u32DeviceId);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vDeselectAudioDevice()
       ***************************************************************************/
      /*!
       * \fn      vDeselectAudioDevice(t_U32 uu32DeviceId
       * \brief   Perform necessary actions specific to a device on de-selection.
       *          Optional Interface for implementation.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       * \retval  NONE
       **************************************************************************/
      t_Void vDeselectAudioDevice(t_U32 u32DeviceId);

      /***************************************************************************
      ** FUNCTION:  t_Void  spi_tclBDCLAudio::vUpdateDeviceSelection()
      ***************************************************************************/
      /*!
      * \fn      t_Void vUpdateDeviceSelection()
      * \brief   To update the device selection.
      * \param   u32DevID : [IN] Resource Manager callbacks structure.
      * \param   enDevCat : [IN] Category of the device
      * \param   enDeviceConnReq : [IN] Select/ deselect.
      * \param   enRespCode : [IN] Response code (success/failure)
      * \param   enErrorCode : [IN] Error
      * \retval  t_Void
      **************************************************************************/
      t_Void vUpdateDeviceSelection(t_U32 u32DevID, tenDeviceCategory enDevCat,
                              tenDeviceConnectionReq enDeviceConnReq,
                              tenResponseCode enRespCode, tenErrorCode enErrorCode);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bIsAudioLinkSupported(t_U32,
       *                                            tenAudioLink)
       ***************************************************************************/
      /*!
       * \fn      bIsAudioLinkSupported(t_U32 u32DeviceId)
       * \brief   Perfom necessary actions specific to a device on de-selection.
       *          Optional Interface for implementation.
       * \param   [u32DeviceId]: Unique Identifier for the Connected Device.
       *          [enLink]: Specify the Audio Link Type for which Capability is
       *          requested. Mandatory interface to be implemented.
       * \retval  Bool value, TRUE if Supported, FALSE otherwise
       **************************************************************************/
      t_Bool bIsAudioLinkSupported(t_U32 u32DeviceId,
               tenAudioLink enLink)
      {
         return true;
      }

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclBDCLAudio::vOnAudioError()
       ***************************************************************************/
      /*!
       * \fn    t_Void vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError)
       * \brief  Interface to set the audio error.
       * \param  enAudDir       : [IN] Uniquely identifies the target Device.
       * \param  enAudioError : [IN] Audio Error
       **************************************************************************/
      t_Void vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError);

      /***************************************************************************
      ** FUNCTION: t_Bool spi_tclBDCLAudio::vSetLastModeSupport()
      ***************************************************************************/
      /*!
      * \fn     t_Void vSetLastModeSupport()
      * \brief  Interface to set last mode setting
      * \param  bSupported : True if last mode audio is supported, else False
      **************************************************************************/
      virtual t_Void vSetLastModeSupport(t_Bool bSupported)
      {
         //@Note: Last mode audio is disabled. To be enabled after last mode video is implemented
         m_bIsLastModeSupported = false;
      }

      /************End of functions overridden from spi_tclAudioDevBase***********/

   private:

      /***************************************************************************
       *********************************PRIVATE************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessAudioRequest(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessAudioRequest
       * \brief   Triggered when BDCL device starts or stops audio streaming
       * \param   crfrRequest [IN]: Structure containing audio request details
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessAudioRequest(const trBdclAudioRequest& crfrRequest);

      /***************************************************************************
      ** FUNCTION:  t_Void vOnModuleState()
      ***************************************************************************/
      /*!
       * \fn      vOnModuleState
       * \brief   Interface to receive different audio module state notifications
       * \param   enAudStream: [IN] Audio stream type
       * \param   bModuleActive: [IN] TRUE if module is active, else FALSE
       * \retval  t_Void
       **************************************************************************/
      t_Void vOnModuleState(tenBdclAudStreamType enAudStream, t_Bool bModuleActive);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vEvaluateVRStates()
       ***************************************************************************/
      /*!
       * \fn      vEvaluateVRStates
       * \brief   Evaluates VR, Mic stream states & VoiceSession states to
       *             determine whether VR is ended.
       * \retval  t_Void
       **************************************************************************/
      t_Void vEvaluateVRStates();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bGetBdclAudioStreamStype()
       ***************************************************************************/
      /*!
       * \fn      bGetBdclAudioStreamStype
       * \brief  Function to convert Audio direction to BDCL audio stream type.
       * \param  enAudDir : Audio Direction of type tenAudioDir.
       * \param  rfenAudioStream : reference to BDCL Audio stream type.
       * \retval  Bool value, TRUE if retrieved, FALSE otherwise
       **************************************************************************/
      t_Bool bGetBdclAudioStreamStype(tenAudioDir enAudDir,
            tenBdclAudStreamType& rfenAudioStream);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclBDCLAudio::bGetAudioDir()
       ***************************************************************************/
      /*!
       * \fn      bGetAudioDir
       * \brief  Function to convert BDCL Audio stream type to Audio direction.
       * \param  enAudStreamType : BDCL Audio stream type.
       * \param  rfenAudioDir : reference to Audio direction type.
       * \retval  Bool value, TRUE if retrieved, FALSE otherwise
       **************************************************************************/
      t_Bool bGetAudioDir(tenBdclAudStreamType enAudStreamType,
            tenAudioDir& rfenAudioDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vActivateChannel(tenAudioDir enAudDir)
       ***************************************************************************/
      /*!
       * \fn     vActivateChannel(tenAudioDir enAudDir)
       * \brief  Function to validate and request for activation of Guidance channel
       * \param  enAudDir : Audio direction type
       * \retval None
       **************************************************************************/
      t_Void vActivateChannel(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vDeactivateChannel(tenAudioDir enAudDir)
       ***************************************************************************/
      /*!
       * \fn     vDeactivateChannel(tenAudioDir enAudDir)
       * \brief  Function to request for deactivation of Guidance channel
       * \param  enAudDir : Audio direction type
       * \retval None
       **************************************************************************/
      t_Void vDeactivateChannel(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vMuteAudio(...)
       ***************************************************************************/
      /*!
       * \fn     vMuteAudio(t_Bool bMute)
       * \brief  Function to request for audio source mute
       * \param  bMute : TRUE if Mute required, else FALSE for Unmute
       * \retval None
       **************************************************************************/
      t_Void vMuteAudio(t_Bool bMute);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessPlaybackStart(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessPlaybackStart
       * \brief   Handles an audio playback start request from device
       * \param   enAudStream [IN]: Audio stream type for which request is received
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessPlaybackStart(tenBdclAudStreamType enAudStream);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessPlaybackStop(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessPlaybackStop
       * \brief   Handles an audio playback stop request from device
       * \param   enAudStream [IN]: Audio stream type for which request is received
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessPlaybackStop(tenBdclAudStreamType enAudStream);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessInterrupt(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessInterrupt
       * \brief   Handles an audio interrupt request from device
       * \param   enAudStream [IN]: Audio stream type for which request is received
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessInterrupt(tenBdclAudStreamType enAudStream);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessMediaStart(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessMediaStart
       * \brief   Handles media playback start request from device
       * \param   enAudDir [IN]: Audio direction for the media stream
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessMediaStart(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessTTSStart(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessTTSStart
       * \brief   Handles TTS playback start request from device
       * \param   enAudDir [IN]: Audio direction for the TTS stream
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessTTSStart(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessVoiceStart(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessVoiceStart
       * \brief   Handles VR playback start request from device
       * \param   enAudDir [IN]: Audio direction for the VR stream
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessVoiceStart(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vProcessMicStart(...)
       ***************************************************************************/
      /*!
       * \fn      vProcessMicStart
       * \brief   Handles Mic start request from device
       * \param   enAudDir [IN]: Audio direction for the Mic stream
       * \retval  t_Void
       **************************************************************************/
      t_Void vProcessMicStart(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vStartAudioRespTimer()
       ***************************************************************************/
      /*!
       * \fn     vStartAudioRespTimer
       * \brief  Function to start the StopAudio response timer
       * \param  None
       * \retval None
       **************************************************************************/
      t_Void vStartAudioRespTimer();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vStopAudioRespTimer()
       ***************************************************************************/
      /*!
       * \fn     vStopAudioRespTimer
       * \brief  Function to stop the StopAudio response timer
       * \param  None
       * \retval None
       **************************************************************************/
      t_Void vStopAudioRespTimer();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclBDCLAudio::vSendStopAudioResponse(...)
       ***************************************************************************/
      /*!
       * \fn     vSendStopAudioResponse
       * \brief  Sends response for a StopAudio request.
       * \param  enAudDir: Audio direction for which StopAudio response is to be sent
       **************************************************************************/
      t_Void vSendStopAudioResponse(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAAPAudio::vStartSpeechRecRelTimer()
       ***************************************************************************/
      /*!
       * \fn     vStartSpeechRecRelTimer
       * \brief  Function to start the speech rec channel timer based on Speech channel
       *         status.
       * \param  None
       * \retval None
       **************************************************************************/
      t_Void vStartSpeechRecRelTimer();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAAPAudio::vStopSpeechRecRelTimer()
       ***************************************************************************/
      /*!
       * \fn     vStopSpeechRecRelTimer
       * \brief  Function to stop the speech rec channel timer based on Speech channel
       *         status.
       * \param  None
       * \retval None
       **************************************************************************/
      t_Void vStopSpeechRecRelTimer();

      /***************************************************************************
       ** FUNCTION:  spi_tclBDCLAudio::bStopAudioTimerCb
       ***************************************************************************/
      /*!
       * \fn     bStopAudioTimerCb
       * \brief  called on expiry of Stop Audio timer
       * \param  rTimerID: ID of the timer which has expired
       * \param  pvObject: pointer to object passed while starting the timer
       * \param  pvUserData: data passed during start of the timer
       **************************************************************************/
      static t_Bool bStopAudioTimerCb(timer_t rTimerID, t_Void *pvObject,
               const t_Void *pvUserData);

      /***************************************************************************
       ** FUNCTION:  spi_tclAAPAudio::bSpeechRecRelTimerCb
       ***************************************************************************/
      /*!
       * \fn     bSpeechRecRelTimerCb
       * \brief  called on expiry of Speech rec release timer
       * \param  rTimerID: ID of the timer which has expired
       * \param  pvObject: pointer to object passed while starting the timer
       * \param  pvUserData: data passed during start of the timer
       **************************************************************************/
      static t_Bool bSpeechRecRelTimerCb(timer_t rTimerID, t_Void *pvObject,
               const t_Void *pvUserData);

      enum
      {
         e8CL_MAX_AUDIO_STREAMS = 4
      };

      //! Pointer to Cmd Audio class
      t_SptrBDCLCmdAudio      m_spoCmdAudio;

      //! Structure object for Function pointers .
      //! This will be used by Audio Manager to register for response callbacks from CL Audio
      trAudioCallbacks        m_rAudCallbacks;

      //! To remove hard coding of number of streams.
      //! For every audio stream from MD, maintain the state of playback
      tenBdclStreamState      m_aenStreamState[e8CL_MAX_AUDIO_STREAMS];

      //! Audio Channel Status used for every stream
      tenAudioChannelStatus   m_aenChannelStatus[e8AUD_INVALID];

      //! Lock for Audio Streams and Activations
      Lock                    m_oAudioStreamLock;
};
#endif // SPI_TCL_BDCLAUDIO_H
