/***********************************************************************/
/*!
 * \file    spi_tclAAPVideo.cpp
 * \brief   AAP Video implementation
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    AAP Video implementation
 AUTHOR:         Shiva Kumar Gurija
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date         | Author                 | Modification
 20.03.2015   | Shiva Kumar Gurija     | Initial Version
 04.02.2015   | Shiva Kumar Gurija     | Select Device Error Handling & Launch Video Impl
 03.07.2015   | Shiva Kumar Gurija     | Dynamic display configuration
 04.02.2016   | Shiva Kumar Gurija    | Moved LaunchApp handling from Video to RsrcMngr

 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "AAPTypes.h"
#include "spi_tclVideoTypedefs.h"
#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
#include "spi_tcl_MockAAPManager.h"
#else
#include "spi_tclAAPManager.h"
#endif
#include "spi_tclAAPCmdVideoIntf.h"
#include "spi_tclAAPVideo.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_VIDEO
#include "trcGenProj/Header/spi_tclAAPVideo.cpp.trc.h"
#endif
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
#define MM_TO_INCHES 0.0393701d

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
 ** FUNCTION:  spi_tclAAPVideo::spi_tclAAPVideo()
 ***************************************************************************/
spi_tclAAPVideo::spi_tclAAPVideo() :
         m_u32SelectedDeviceID(0)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo() entered"));

#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   if (NULL != poAAPManager)
   {
      //! Register with AAP manager for Video callbacks
      poAAPManager->bRegisterObject((spi_tclAAPRespVideo*) this);
   } //if(NULL != poAAPManager)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPVideo::~spi_tclAAPVideo()
 ***************************************************************************/
spi_tclAAPVideo::~spi_tclAAPVideo()
{
   ETG_TRACE_USR1(("~spi_tclAAPVideo() entered"));
   m_u32SelectedDeviceID = 0;
   m_poCmdVideo = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPVideo::bInitialize()
 ***************************************************************************/
t_Bool spi_tclAAPVideo::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::bInitialize() entered"));

   t_Bool bRet = false;

#if (defined(_LINUXX86MAKE_) || defined (_LINUXX86_64_))
   spi_tcl_MockAAPManager* poAAPManager = spi_tcl_MockAAPManager::getInstance();
#else
   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
#endif
   SPI_NORMAL_ASSERT(NULL == poAAPManager);

   if (NULL != poAAPManager)
   {
      m_poCmdVideo = poAAPManager->poGetVideoInstance();
      bRet = true;

   } //if (NULL != poAAPMngr)

   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vUninitialize()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::vUninitialize() entered"));
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclMLVideo::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vRegisterCallbacks(const trVideoCallbacks& corfrVideoCallbacks)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::vRegisterCallbacks() entered"));
   m_rVideoCallbacks = corfrVideoCallbacks;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vSelectDevice(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq)
{
   /*lint -esym(40,fpvSelectDeviceCb) fpvSelectDeviceCb is not declared */
   ETG_TRACE_USR1(("spi_tclAAPVideo:vSelectDevice: Device ID-0x%x Selection Type - %d ", cou32DevId, ETG_ENUM(CONNECTION_REQ,
            coenConnReq)));

   tenErrorCode enErrCode = e8NO_ERRORS;
   if (e8DEVCONNREQ_SELECT == coenConnReq)
   {
      //Initialize Video session
      enErrCode = (bInitVideoSession(cou32DevId)) ? e8NO_ERRORS : e8UNKNOWN_ERROR;
   } //if( coenConnReq == e8DEVCONNREQ_SELECT )
   else
   {
      vUnInitVideoSession(cou32DevId);
   } //else

   if (NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
   {
      (m_rVideoCallbacks.fpvSelectDeviceCb)(cou32DevId, enErrCode);
   } //if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
}
/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPVideo::bInitVideoSession()
 ***************************************************************************/
t_Bool spi_tclAAPVideo::bInitVideoSession(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::bInitVideoSession:Device ID-0x%x", u32DevID));

   t_Bool bRet = false;

   if ((m_u32SelectedDeviceID != u32DevID) && (0 != u32DevID))
   {
      //un initilize video session for the earlier connected device, in case if the clean up was not performed.
      vUnInitVideoSession(m_u32SelectedDeviceID);

      //create the VideoSink end point using ADIT interfaces
      trAAPVideoConfig rAAPVideoConfig;

      if ((NULL != m_poCmdVideo) && (NULL != m_poVideoSettings))
      {
         tvecVideoConfigList vecVideoConfigList;
         m_poVideoSettings->vGetVideoConfigData(e8DEV_TYPE_ANDROIDAUTO, vecVideoConfigList);
         tmapAAPVideoConfigList mapAAPVideoConfigList;
         for (t_U8 u8Index = 0; u8Index < vecVideoConfigList.size(); u8Index++)
         {
            mapAAPVideoConfigList[vecVideoConfigList[u8Index].u16PriorityIndex] = vecVideoConfigList[u8Index];
         }

         tvecAAPVideoPipeLineConfigList vecAAPVideoPipeLineConfigList;

         //function set the video config data
         vSetAAPVideoConfig(mapAAPVideoConfigList, rAAPVideoConfig);

         //function to set the video pipeline configuration for AAP
         vSetAAPVideoPipeLineConfig(mapAAPVideoConfigList, vecAAPVideoPipeLineConfigList);

         bRet = m_poCmdVideo->bInitialize(rAAPVideoConfig,vecAAPVideoPipeLineConfigList);

         if (false == bRet)
         {
            ETG_TRACE_USR1(("spi_tclAAPVideo::bInitVideoSession: Video configuration intialization has failed"));
         }

         m_u32SelectedDeviceID = u32DevID;
      } //if((NULL != poSettings) && (NULL != spoCmdVideo) )
   } //if( (m_u32SelectedDeviceID != u32DevID) && (0 != u32DevID) )
   else
   {
      //return true if the currently selected device is same as already selected device.
      //return an error, if the device ID is zero(invalid device).
      bRet = ((0 != u32DevID) && (m_u32SelectedDeviceID == u32DevID));
   }

   ETG_TRACE_USR4(("spi_tclAAPVideo::bInitVideoSession left with %d", ETG_ENUM(BOOL, bRet)));

   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vSetAAPVideoConfig()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vSetAAPVideoConfig(const tmapAAPVideoConfigList& rfmapAAPVideoConfigList,
         trAAPVideoConfig& rfrAAPVideoConfig)
{
   ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig entered"));
   std::map<t_U16, trVideoConfigData>::const_iterator coitVideoConfigList;

   coitVideoConfigList = rfmapAAPVideoConfigList.begin();

   if (m_poVideoSettings != NULL)
   {
      rfrAAPVideoConfig.u32ScreenHeight = coitVideoConfigList->second.u32Screen_Height;
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - Screen height = %d", rfrAAPVideoConfig.u32ScreenHeight));

      rfrAAPVideoConfig.u16LayerID = static_cast<t_U16>(coitVideoConfigList->second.u32LayerId);
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - Layer id = %d", rfrAAPVideoConfig.u16LayerID));

      rfrAAPVideoConfig.u16SurfaceID = static_cast<t_U16>(coitVideoConfigList->second.u32SurfaceId);
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - Surface ID = %d", rfrAAPVideoConfig.u16SurfaceID));

      rfrAAPVideoConfig.bAutoStartProjection = m_poVideoSettings->bGetAutoStartProjection();
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - Auto start Projection = %d", ETG_ENUM(BOOL,
               rfrAAPVideoConfig.bAutoStartProjection)));

      rfrAAPVideoConfig.szVideoCodec = m_poVideoSettings->szGetVideoCodecType().c_str();
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - Video codec = %s", rfrAAPVideoConfig.szVideoCodec.c_str()));

      rfrAAPVideoConfig.u8MaxUnAckedFrames = m_poVideoSettings->u8GetMaxUnackedFrames();
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - max unacked frames = %d", rfrAAPVideoConfig.u8MaxUnAckedFrames));

      rfrAAPVideoConfig.u32ScreenWidth = coitVideoConfigList->second.u32Screen_Width;
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - Screen width = %d", rfrAAPVideoConfig.u32ScreenWidth));

      rfrAAPVideoConfig.szDisplayDev = coitVideoConfigList->second.szDisplayDev;
      ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig - video device = %s", rfrAAPVideoConfig.szDisplayDev.c_str()));
   }
   ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoConfig left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vSetAAPVideoPipeLineConfig()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vSetAAPVideoPipeLineConfig(const tmapAAPVideoConfigList& rfmapAAPVideoConfigList,
         tvecAAPVideoPipeLineConfigList& rfvecAAPVideoPipeLineConfigList)
{
   ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig entered"));

   //vector has to be filled with the video pipeline configuration

   std::map<t_U16, trVideoConfigData>::const_iterator coitVideoConfigList;
   trAAPVideoPipeLineConfig rAAPVideoPipeLineConfig;
   if (m_poVideoSettings != NULL)
   {
      for (coitVideoConfigList = rfmapAAPVideoConfigList.begin(); coitVideoConfigList != rfmapAAPVideoConfigList.end();
               ++coitVideoConfigList)
      {
         rAAPVideoPipeLineConfig.u16DpiDensity = coitVideoConfigList->second.u16dpi;
         ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig - DPI - %d", rAAPVideoPipeLineConfig.u16DpiDensity));

         rAAPVideoPipeLineConfig.u8Fps = m_poVideoSettings->u8GetFramesPerSec();
         ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig - FPS - %d", rAAPVideoPipeLineConfig.u8Fps));

         rAAPVideoPipeLineConfig.szPixelAspectRatio = coitVideoConfigList->second.szPixelAspectRatio;
         ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig - Aspect ratio - %s", rAAPVideoPipeLineConfig.szPixelAspectRatio.c_str()));

         rAAPVideoPipeLineConfig.u16ProjScreenWidth = (t_U16) coitVideoConfigList->second.u32ProjScreen_Width;
         ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig - Pro screen width in pixels : %d",rAAPVideoPipeLineConfig.u16ProjScreenWidth));

         rAAPVideoPipeLineConfig.u16ProjScreenHeight = (t_U16) coitVideoConfigList->second.u32ProjScreen_Height;
         ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig - Proj screen height in pixels : %d",rAAPVideoPipeLineConfig.u16ProjScreenHeight));

         t_U32 u32VerticalCodecResolution = 480;

         if(rAAPVideoPipeLineConfig.u16ProjScreenHeight <= 480)
         {
            u32VerticalCodecResolution = 480;
         }
         else if(rAAPVideoPipeLineConfig.u16ProjScreenHeight > 480 && rAAPVideoPipeLineConfig.u16ProjScreenHeight <= 720)
         {
            u32VerticalCodecResolution = 720;
         }
         else
         {
            u32VerticalCodecResolution = 1080;
         }

         t_U32 u32HeightMargin = u32VerticalCodecResolution - coitVideoConfigList->second.u32ProjScreen_Height;

         rAAPVideoPipeLineConfig.u16RealDensity = (t_U16) ((u32VerticalCodecResolution - u32HeightMargin)/(coitVideoConfigList->second.u32ProjScreen_Height_Mm * MM_TO_INCHES));

         ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig - u16RealDensity  %d",rAAPVideoPipeLineConfig.u16RealDensity));

         rfvecAAPVideoPipeLineConfigList.push_back(rAAPVideoPipeLineConfig);
      }
   }
   ETG_TRACE_USR4(("spi_tclAAPVideo::vSetAAPVideoPipeLineConfig left"));

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vUnInitVideoSession()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vUnInitVideoSession(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::vUnInitVideoSession:Device ID-0x%x", u32DevID));
   if ( (0 != m_u32SelectedDeviceID) && (NULL != m_poCmdVideo) )
   {
      m_poCmdVideo->vUninitialize();
      m_u32SelectedDeviceID = 0;
   }      //if (NULL != spoCmdVideo)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPVideo::bLaunchVideo()
 ***************************************************************************/
t_Bool spi_tclAAPVideo::bLaunchVideo(const t_U32 cou32DevId, const t_U32 cou32AppId, const tenEnabledInfo coenSelection)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::bLaunchVideo:Device ID-0x%x AppID-0x%x", cou32DevId, cou32AppId));

   SPI_INTENTIONALLY_UNUSED(coenSelection);

   return true;
}

/***************************************************************************
 ** FUNCTION:  t_U32  spi_tclAAPVideo::vStartVideoRendering()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vStartVideoRendering(t_Bool bStartVideoRendering)
{
   /*lint -esym(40,fpvVideoRenderStatusCb) fpvVideoRenderStatusCb identifier*/
   // Send response to the set request
   if (NULL != m_rVideoCallbacks.fpvVideoRenderStatusCb)
   {
      (m_rVideoCallbacks.fpvVideoRenderStatusCb)(bStartVideoRendering, e8DEV_TYPE_ANDROIDAUTO);
   }      //if(NULL != m_rVideoCallbacks.fpvVideoRenderStatusCb )
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAAPVideo::vGetVideoSettings()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vGetVideoSettings(const t_U32 cou32DevId, trVideoAttributes& rfrVideoAttributes)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(rfrVideoAttributes);
   //unused
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vOnSelectDeviceResult(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo:vOnSelectDeviceResult: Device ID-0x%x Device Selection Type - %d "
            "Response Code - %d", cou32DevId, ETG_ENUM(CONNECTION_REQ, coenConnReq), ETG_ENUM(RESPONSE_CODE,
            coenRespCode)));

   //Device selection is failed. Clear the video resources,if already allocated
   if ((e8FAILURE == coenRespCode) && (e8DEVCONNREQ_SELECT == coenConnReq))
   {
      vUnInitVideoSession(m_u32SelectedDeviceID);
   } //if ( (e8FAILURE == coenRespCode) && (e8DEVCONNREQ_SELECT == coenConnReq) )
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclAAPVideo::vSetOrientationMode()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vSetOrientationMode(const t_U32 cou32DevId, const tenOrientationMode coenOrientationMode,
         const trUserContext& corfrUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::vSetOrientationMode"));

   /*lint -esym(40,fpvSetOrientationModeCb) fpvSetOrientationModeCb Undeclared identifier */
   SPI_INTENTIONALLY_UNUSED(coenOrientationMode);
   if (NULL != m_rVideoCallbacks.fpvSetOrientationModeCb)
   {
      (m_rVideoCallbacks.fpvSetOrientationModeCb)(cou32DevId,
               e8UNSUPPORTED_OPERATION,
               corfrUsrCntxt,
               e8DEV_TYPE_ANDROIDAUTO);
   } // if(NULL != m_rVideoCallbacks.fpvCbSetVideoBlockingMode)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vTriggerVideoFocusCb()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vTriggerVideoFocusCb(t_S32 s32Focus, t_S32 s32Reason)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::vTriggerVideoFocusCb: Focus-%d", s32Focus));
   if (NULL != m_poCmdVideo)
   {
      m_poCmdVideo->videoFocusCallback(s32Focus, s32Reason);
   }//if (NULL != m_poCmdVideo)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPVideo::vVideoConfigCallback()
 ***************************************************************************/
t_Void spi_tclAAPVideo::vVideoConfigCallback(t_S32 s32LogicalUIWidth, t_S32 s32LogicalUIHeight)
{
   ETG_TRACE_USR1(("spi_tclAAPVideo::vVideoConfigCallback entered"));
   trProjectionVideoAttributes rProjVideoAttr;
   rProjVideoAttr.u16ProjectionUIWidth = static_cast<t_U16>(s32LogicalUIWidth);
   rProjVideoAttr.u16ProjectionUIHeight = static_cast<t_U16>(s32LogicalUIHeight);
   if (NULL != m_rVideoCallbacks.fpvProjectionVideoAttributes)
   {
      (m_rVideoCallbacks.fpvProjectionVideoAttributes)(m_u32SelectedDeviceID, rProjVideoAttr);
   }
}
//lint restore

///////////////////////////////////////////////////////////////////////////////
// <EOF>
