/***********************************************************************/
/*!
* \file  spi_tclAAPDataService.cpp
* \brief AAP Data Service Class
*************************************************************************
\verbatim

PROJECT        :   Gen3
SW-COMPONENT   :   Smart Phone Integration
DESCRIPTION    :   Android Auto Data Service class implements Data Service Info Management for
                    Android Auto capable devices. This class must be derived from base Data Service class.
AUTHOR         :   SHITANSHU SHEKHAR (RBEI/ECP2)
COPYRIGHT      :   &copy; RBEI

HISTORY:
Date        | Author                         | Modification
24.03.2015  | SHITANSHU SHEKHAR (RBEI/ECP2)  | Initial Version
30.10.2015  | Shiva Kumar G                  | Implemented ReportEnvironment Data feature
05.05.2016  | Ramya Murthy                   | Replaced usage of VehicleMovementState with 
                                               GearPosition, to send Gear data to phone.
17.01.2017  | Noopur R Kalawatia             | Policy based configuration of Speed update frequency
                                               for AAP.
 16.01.2019 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetGeneralRestrictions method 
 16.01.2019 |  Roveena Francy Lobo (RBEI/ECO2)  | Added vSetETCStatus method 											  
 
\endverbatim
*************************************************************************/
#include <cmath>

#include "spi_tclMediator.h"
#include "Timer.h"
#include "spi_tclAAPManager.h"
#include "spi_tclAAPDataService.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
      #include "trcGenProj/Header/spi_tclAAPDataService.cpp.trc.h"
   #endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
static const trAAPSelectedDeviceInfo corEmptyDeviceInfo;

//! Temperature timer data
static timer_t srOutsideTempUpdateTimerID;
static const t_U32 scou32OutsideTempUpdateTimerVal = 10000;
static t_Bool sbOutsideTempUpdateTimerRunning = false;

//! Vehicle Speed timer data
static timer_t srVehSpeedTimerID;
static t_Bool sbVehSpeedTimerRunning = false;
static const t_U32 scou32VehSpeedUpdateTimerVal = 100;
static const t_U16 scu16AAPETCInfoBit = 4;
/****************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::spi_tclAAPDataService()
 ***************************************************************************/
spi_tclAAPDataService::spi_tclAAPDataService(const trDataServiceCb& rfcorDataServiceCb)
: m_poCmdSensor(NULL),
  m_rDataServiceCb(rfcorDataServiceCb),
  m_enDayNightMode(e8_AAP_DAY_MODE),
  m_bParkBrakeActive(false),
  m_enVehMovState(e8VEHICLE_MOVEMENT_STATE_INVALID),
  m_enGearPosition(e8GEAR_POS_UNKNOWN),
  m_bLocDataAvailable(false),
  m_bDeadReckonedLocData(false),  
  m_s32AAPParkModeRestrictionInfo(static_cast<int>(e8_DRIVE_STATUS_UNRESTRICTED)),
  m_s16VehicleSpeed(0),
  m_dOutsideTemp(0),
  m_bOutsideTempValidityFlag(false),
  m_dBarometricPressure(0),
  m_bBaroPressureValidityFlag(false)
{
   ETG_TRACE_USR1(("spi_tclAAPDataService() entered "));

}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::~spi_tclAAPDataService()
 ***************************************************************************/
spi_tclAAPDataService::~spi_tclAAPDataService()
{
   ETG_TRACE_USR1(("~spi_tclAAPDataService() entered "));
   vStopVehicleSpeedUpdate();
   m_poCmdSensor = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPDataService::bInitialise()
 ***************************************************************************/
t_Bool spi_tclAAPDataService::bInitialise()
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::bInitialise() entered "));

   //! If data service is not initialised, perform initializations which are
   // not device dependent.
   t_Bool bInitSuccess = false;

   spi_tclAAPManager* poAAPManager = spi_tclAAPManager::getInstance();
   m_poCmdSensor = (NULL != poAAPManager)? (poAAPManager->poGetSensorInstance()) : (NULL);

   if (NULL != m_poCmdSensor)
   {
      bInitSuccess = true;
      vRegisterCallbacks();
   }// if (NULL != m_poCmdSensor)

   ETG_TRACE_USR2(("[DESC]:spi_tclAAPDataService::bInitialise() left with result = %u ", bInitSuccess));
   return bInitSuccess;
}
/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPDataService::bUninitialise()
 ***************************************************************************/
t_Bool spi_tclAAPDataService::bUninitialise()
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::bUninitialise() entered "));

   m_poCmdSensor = NULL;
   return true;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAAPDataService::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclAAPDataService::vRegisterCallbacks()
{
   /*lint -esym(40,fvSetDayNightMode) fvSetDayNightMode is not declared */
   /*lint -esym(40,_1) _1is not declared */
   ETG_TRACE_USR1((" spi_tclAAPDataService::vRegisterCallbacks() Entered "));

   trAAPSensorCallback rDayNightCbs;
   rDayNightCbs.fvSetDayNightMode = std::bind(
      &spi_tclAAPDataService::vOnSetDayNightMode, this,
      std::placeholders::_1);
   spi_tclMediator* poMediator = spi_tclMediator::getInstance();
   if (NULL != poMediator)
   {
      poMediator->vRegisterCallbacks(rDayNightCbs);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPDataService::vOnSetDayNightMode()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vOnSetDayNightMode(tenVehicleConfiguration enVehicleConfig)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vOnSetDayNightMode() entered "
         "with DayNight mode =%d ", enVehicleConfig));

   m_enDayNightMode = (e8_DAY_MODE == enVehicleConfig)?(e8_AAP_DAY_MODE):(e8_AAP_NIGHT_MODE);

   vReportDayNightMode();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPDataService::vSelectDevice(t_U32...)
 ***************************************************************************/
t_Void spi_tclAAPDataService::vSelectDevice(const t_U32 cou32DevId,
      tenDeviceConnectionReq enConnReq)
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vSelectDevice() entered "));

   if(e8DEVCONNREQ_DESELECT == enConnReq)
   {
      if ((NULL != m_poCmdSensor) && (cou32DevId == m_rSelDevInfo.u32DeviceHandle))
      {
         //! Clear Selected device's info
         m_rSelDevInfo = corEmptyDeviceInfo;

         //! Stop timers for fixed frequency updates
         vStopVehicleSpeedUpdate();
         vStopOutsideTempUpdate();

         //! Unsubscribe for interested data
         vSetDataSubscription(false);

         m_poCmdSensor->bUnInitializeSensor();
      }//if ((u32DeviceHandle == m_rSelDevInfo.u32DeviceHandle))
   }//if(e8DEVCONNREQ_DESELECT == enConnReq)
   else if(e8DEVCONNREQ_SELECT == enConnReq)
   {
      //! Clear Selected device's info & update the newly selected device's handle.
      m_rSelDevInfo = corEmptyDeviceInfo;
      m_rSelDevInfo.u32DeviceHandle = cou32DevId;

      // Initialize Sensor Endpoint
      if ((NULL != m_poCmdSensor) &&
            (true == m_poCmdSensor->bInitializeSensor(m_rAAPDataServiceConfigData)))
      {
         //! Subscribe for interested data
         vSetDataSubscription(true);

         //! Update current data values
         vReportDayNightMode();
         vReportParkingBrake();
         vReportGear();
         vParkRestrictionInfo();
         vReportDrivingStatus();
         vReportEnvironmentData();
         //! Start timers for fixed frequency updates
         vStartOutsideTempUpdate();
         vStartVehicleSpeedUpdate();

      }//if ((NULL != m_poCmdSensor) && ...)
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPDataService::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vOnSelectDeviceResult(t_U32 u32DeviceHandle,
         const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vOnSelectDeviceResult() entered: u32DeviceHandle = %u ",
         u32DeviceHandle));

   t_Bool bIsSelectFailure = ((e8FAILURE == coenRespCode) && (e8DEVCONNREQ_SELECT == coenConnReq));

   if ((bIsSelectFailure) && (NULL != m_poCmdSensor) && (u32DeviceHandle == m_rSelDevInfo.u32DeviceHandle))
   {
         //! Clear Selected device's info
         m_rSelDevInfo = corEmptyDeviceInfo;

         //! Stop timers for fixed frequency updates
         vStopVehicleSpeedUpdate();
         vStopOutsideTempUpdate();

         //! Unsubscribe for interested data
         vSetDataSubscription(false);

         m_poCmdSensor->bUnInitializeSensor();
   }//if ((bIsSelectFailure) && ...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPDataService::vOnDeselectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclAAPDataService::vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vOnDeselectDeviceResult() entered: u32DeviceHandle = %u ",
         u32DeviceHandle));
   //! Nothing to be done
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vSetDataSubscription(...)
***************************************************************************/
t_Void spi_tclAAPDataService::vSetDataSubscription(t_Bool bSubscriptionOn)
{
   /*lint -esym(40,fvSubscribeForData) fvSubscribeForData is not declared */
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vSetDataSubscription() entered: SubscriptionOn = %u ",
      ETG_ENUM(BOOL, bSubscriptionOn)));

   if (NULL != m_rDataServiceCb.fvSubscribeForData)
   {
      m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8VEHICLE_DATA);

      if ((true == m_rAAPDataServiceConfigData.bLocDataAvailable) && (
            false == m_rAAPDataServiceConfigData.bDeadReckonedData) &&
            (e8GEO_COORDINATE_SYSTEM_WGS == m_rAAPDataServiceConfigData.enGeoCoordinateSystemType))
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8GNSS_DATA);
      }
      if (m_rAAPDataServiceConfigData.bAcclData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8ACCELEROMETER_DATA);
      }
      if (m_rAAPDataServiceConfigData.bGyroData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8GYROSCOPE_DATA);
      }
      if (m_rAAPDataServiceConfigData.bEnvData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8ENVIRONMENT_DATA);
      }
      if (m_rAAPDataServiceConfigData.bParkBrakeData)
      {
         m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8PARKBRAKE_DATA);
      }
   }//if (NULL != m_rDataServiceCb.fvSubscribeForData)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vSetSensorDataAvailablility(...)
***************************************************************************/
t_Void spi_tclAAPDataService::vSetSensorDataAvailablility(
      const trDataServiceConfigData& rfrDataServiceConfigInfo)
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vSetSensorDataAvailablility() entered "));

   //! Store config data
   m_rAAPDataServiceConfigData = rfrDataServiceConfigInfo;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vOnData(const trGPSData& rfcorGpsData)
***************************************************************************/
t_Void spi_tclAAPDataService::vOnData(const trGPSData& rfrcGpsData)
{
   SPI_INTENTIONALLY_UNUSED(rfrcGpsData);
//	t_S32 s32GyroZE3 = static_cast<t_S32>((((t_Double)rfrcGpsData.s16TurnRate/100)* 0.0174532925)* 1000);    // gyro sent in rad/s
//	t_S32 s32AccelerationYE3 = (rfrcGpsData.s16Acceleration/100) *1000 ;										// acceleration is in m/s2
//                                     //TODO Commented for legality permission
//	if(NULL != spoAAPManager)
//	{
//	   if(NULL != spoAAPManager->poGetSensorInstance())
//	   {
//	       spoAAPManager->poGetSensorInstance()->vReportAccelerometerData(false,0,true,s32AccelerationYE3,false,0);
//	       spoAAPManager->poGetSensorInstance()->vReportGyroscopeData(false,0,false,0,true,s32GyroZE3);
//	   }
//   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vOnData(const trSensorData& rfcorSensorData)
***************************************************************************/
t_Void spi_tclAAPDataService::vOnData(const trSensorData& rfcorSensorData)
{
   if ((true == m_rAAPDataServiceConfigData.bLocDataAvailable) &&
         (false == m_rAAPDataServiceConfigData.bDeadReckonedData) &&
         (NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
       m_poCmdSensor->vReportGpsData(rfcorSensorData);

	   t_S32 numberInUse = static_cast<int>(rfcorSensorData.u16SatellitesUsed);
	   t_Bool hasNumberInView = false;		//TODO Need to decide what to pass as an arguement
	   t_S32 numberInView = 0;				//TODO Need to decide what to pass as an arguement
	   const t_S32* prns = NULL;
	   const t_S32* snrsE3 = NULL;
	   const t_Bool* usedInFix = NULL;
	   const t_S32* azimuthsE3 = NULL;
	   const t_S32* elevationsE3 = NULL;
	   m_poCmdSensor->vReportGpsSatelliteData(numberInUse, hasNumberInView,
                                          numberInView,  prns, snrsE3,
                                          usedInFix,  azimuthsE3, elevationsE3);

   }//if ((true == m_rAAPDataServiceConfigData.bLocDataAvailable)&&...)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vOnAccSensorData
** (const std::vector<trAccSensorData>& corfvecrAccSensorData)
***************************************************************************/
t_Void spi_tclAAPDataService::vOnAccSensorData(const std::vector<trAccSensorData>& corfvecrAccSensorData)
{
   if ((true == m_rAAPDataServiceConfigData.bAcclData) &&
         (NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
	  ETG_TRACE_USR2(("[DESC]:spi_tclAAPDataService::vOnAccSensorData Acc list size = %d", corfvecrAccSensorData.size()));

      for(t_U16 u16Index = 0 ; (u16Index < corfvecrAccSensorData.size()) && (u16Index < 20) ; ++u16Index)
      {
    	  // Convert to axes requested by Google
         t_S32 s32AccZE3 = static_cast<t_S32>((corfvecrAccSensorData[u16Index].fAccZ) * (1000));
         t_S32 s32AccYE3 = static_cast<t_S32>((corfvecrAccSensorData[u16Index].fAccX) * (1000));
         t_S32 s32AccXE3 = (-1) * (static_cast<t_S32>((corfvecrAccSensorData[u16Index].fAccY) * (1000)));

         ETG_TRACE_USR4_CLS((TR_CLASS_SMARTPHONEINT_SENSORDATA, "[PARAM]:spi_tclAAPDataService::vOnAccSensorData Acc Values"
  	   		" s32AccZE3 = %d, s32AccYE3 = %d, s32AccXE3 = %d", s32AccZE3,
  	   		s32AccYE3, s32AccXE3));
	     m_poCmdSensor->vReportAccelerometerData(true,s32AccXE3,true,s32AccYE3,true,s32AccZE3);
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vOnGyroSensorData
** (const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
***************************************************************************/
t_Void spi_tclAAPDataService::vOnGyroSensorData(const std::vector<trGyroSensorData>& corfvecrGyroSensorData)
{
   if ((true == m_rAAPDataServiceConfigData.bGyroData) &&
         (NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      ETG_TRACE_USR2(("[DESC]:spi_tclAAPDataService::vOnGyroSensorData Gyro list size = %d", corfvecrGyroSensorData.size()));

      for (t_U16 u16Index = 0; (u16Index < corfvecrGyroSensorData.size()) && (u16Index < 20); ++u16Index)
      {
         // Convert to axes requested by Google and rad/s
         t_S32 s32GyroZE3 = static_cast<t_S32> ((corfvecrGyroSensorData[u16Index].fGyroZ) * (1000));
         t_S32 s32GyroYE3 = static_cast<t_S32> ((corfvecrGyroSensorData[u16Index].fGyroX) * (1000));
         t_S32 s32GyroXE3 = (-1) * (static_cast<t_S32> ((corfvecrGyroSensorData[u16Index].fGyroY) * (1000)));

         ETG_TRACE_USR4_CLS((TR_CLASS_SMARTPHONEINT_SENSORDATA, "[PARAM]:spi_tclAAPDataService::vOnGyroSensorData Gyro Values"
                  " s32GyroZE3 = %d, s32GyroYE3 = %d, s32GyroXE3 = %d", s32GyroZE3, s32GyroYE3, s32GyroXE3));
         m_poCmdSensor->vReportGyroscopeData(true, s32GyroXE3, true, s32GyroYE3, true, s32GyroZE3);
      }
   }//if ((true == m_rAAPDataServiceConfigData.bGyroData) &&...)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vOnData(const trVehicleData rfrcVehicleData)
***************************************************************************/
t_Void spi_tclAAPDataService::vOnData(const trVehicleData& rfrcVehicleData, t_Bool bSolicited)
{
   //! Store Vehicle speed info
   m_s16VehicleSpeed = rfrcVehicleData.s16Speed;

   //! Send data to selected phone on vehicle data change
   if (false == bSolicited)
   {
      vSetVehicleData(rfrcVehicleData.bParkBrakeActive,
            rfrcVehicleData.enVehMovState, rfrcVehicleData.enGearPosition);
   }//if (false == bSolicited)
}

/***************************************************************************
** FUNCTION:  spi_tclAAPDataService::vOnEnvironmentData()
***************************************************************************/
t_Void spi_tclAAPDataService::vOnEnvironmentData(t_Bool bValidTempUpdate,
      t_Double dTemp,
      t_Bool bValidPressureUpdate,
      t_Double dPressure)
{
   //ETG_TRACE_USR1(("spi_tclAAPDataService::vOnEnvironmentData() entered"));

   //Pressure & Temperature updates needs to be stored, because in case of G3G projects,
   //HMI would set this before the AAP device becomes active and then doesn't update later, if there is no change.
   //@Note: Only valid updates are stored to prevent sending invalid values

   if (true == bValidTempUpdate)
   {
      m_dOutsideTemp = dTemp;
      m_bOutsideTempValidityFlag = bValidTempUpdate;
   }//if (true == bValidTempUpdate)

   if (true == bValidPressureUpdate)
   {
      m_dBarometricPressure = dPressure;
      m_bBaroPressureValidityFlag = bValidPressureUpdate;
   }//if (true == bValidPressureUpdate)

   //! Notify data since it is changed
   if ((true == bValidTempUpdate) || (true == bValidPressureUpdate))
   {
      vReportEnvironmentData();
   }//if ((true == bValidTempUpdate) || (true == bValidPressureUpdate))
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPDataService::vSetVehicleData()
***************************************************************************/
t_Void spi_tclAAPDataService::vSetVehicleData(t_Bool bParkBrakeActive,
      tenVehicleMovementState enVehMovState, tenGearPosition enGearPosition)
{
   //! Send changed ParkBrake and/or restriction data
   if ((bParkBrakeActive != m_bParkBrakeActive) || (enVehMovState != m_enVehMovState)) /*if states have changed*/
   {
      if ((bParkBrakeActive != m_bParkBrakeActive) &&
               (true == m_rAAPDataServiceConfigData.bParkBrakeData))
      {
         //! Store latest state
         m_bParkBrakeActive = bParkBrakeActive;
         vReportParkingBrake();
      }
      //! Store latest state
      m_enVehMovState = enVehMovState;

      vReportDrivingStatus();
   }//if ((bParkBrakeActive != m_bParkBrakeActive) || (enVehMovState != m_enVehMovState))

   //! Send changed Gear data
   if (enGearPosition != m_enGearPosition)
   {
      //! Report the latest Gear Position
      m_enGearPosition = enGearPosition;
      vReportGear();
   }//if (enGearPosition != m_enGearPosition)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vReportParkingBrake()
***************************************************************************/
t_Void spi_tclAAPDataService::vReportParkingBrake()
{
	ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vReportParkingBrake() entered: ParkBrakeActive = %d ",
	      ETG_ENUM(BOOL, m_bParkBrakeActive)));

   //! Send park mode data
   if ((NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle) &&
            (true == m_rAAPDataServiceConfigData.bParkBrakeData))
   {
      m_poCmdSensor->vReportParkingBrakeData(m_bParkBrakeActive);
	}
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vReportGear()
***************************************************************************/
t_Void spi_tclAAPDataService::vReportGear()
{
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vReportGear() entered: Gear Position = %d ",
         ETG_ENUM(GEARPOSITION, m_enGearPosition)));

   //! Send Gear data
   if ((true == m_rAAPDataServiceConfigData.bGearStatus) &&
         (e8GEAR_POS_UNKNOWN != m_enGearPosition) &&
         (NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      m_poCmdSensor->vReportGearData(static_cast<int>(m_enGearPosition));
   }//if ((true == rfrDataServiceConfigData.bGearStatus) && ...)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPDataService::vParkRestrictionInfo()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vParkRestrictionInfo()
{
	ETG_TRACE_USR1(("spi_tclAAPDataService::vParkRestrictionInfo() entered "));
	if(NULL != m_poDataServiceSettings)
	{
		m_s32AAPParkModeRestrictionInfo= static_cast<t_S32>(m_poDataServiceSettings->u8GetRestrictionsInfo(e8DEV_TYPE_ANDROIDAUTO,false));
	}
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPDataService::vReportDrivingStatus()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vReportDrivingStatus()
{
   //! Send park mode data
   if ((NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      //! Set driving status
      tenDrivingMode enDrivingMode =
            ((m_bParkBrakeActive) || (e8VEHICLE_MOVEMENT_STATE_PARKED == m_enVehMovState)) ?
            (e8_PARK_MODE) : (e8_DRIVE_MODE);

      t_S32 s32AAPRestrictionInfo = m_s32AAPParkModeRestrictionInfo;

      //! Read drive level restriction info
      if ((e8_DRIVE_MODE == enDrivingMode) && (NULL != m_poDataServiceSettings))
      {
         s32AAPRestrictionInfo = static_cast<t_S32>(m_poDataServiceSettings->u8GetRestrictionsInfo(e8DEV_TYPE_ANDROIDAUTO,true));
      }

      ETG_TRACE_USR4(("[PARAM]:spi_tclAAPDataService::vReportDrivingStatus(): DriveMode = %d, AAPRestrictionInfo = %d ",
            ETG_ENUM(AAP_DRIVING_MODE, enDrivingMode), s32AAPRestrictionInfo));

      m_poCmdSensor->vReportDrivingStatusData(s32AAPRestrictionInfo);
   }//if ((NULL != m_poCmdSensor) && ...)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vReportVehicleSpeedInfo()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vReportVehicleSpeedInfo()
{
   //! Send Vehicle speed data
   if ((NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle))
   {
      // convert speed cm/s to m/s and multiply by E3 as per AAUTO spec
      // (m_s16VehicleSpeed) * 10) == (m_s16VehicleSpeed / 100) * 1000)
	  t_S32 s32SpeedE3 = ((m_s16VehicleSpeed) * 10);
      s32SpeedE3 = (e8GEAR_POS_REVERSE_GEAR == m_enGearPosition) ? ((-1)*(abs(s32SpeedE3))):(s32SpeedE3);
      m_poCmdSensor->vReportSpeedData(s32SpeedE3);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vReportEnvironmentData()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vReportEnvironmentData()
{
   //To avoid Gen4Compiler warnings converting double to float. Reason: ETG trace does not support for double values
   ETG_TRACE_USR1(("[PARAM]:spi_tclAAPDataService::vReportEnvironmentData() entered: "
         "Temp = %f, IsValidTemp = %u, Pressure = %f, IsValidPressure = %u ",
         static_cast<t_Float>(m_dOutsideTemp), ETG_ENUM(BOOL, m_bOutsideTempValidityFlag),
         static_cast<t_Float>(m_dBarometricPressure), ETG_ENUM(BOOL, m_bBaroPressureValidityFlag)));

   //! Send Environment data
   if ((true == m_rAAPDataServiceConfigData.bEnvData) &&
            (0 != m_rSelDevInfo.u32DeviceHandle) && (NULL != m_poCmdSensor))
   {
      m_poCmdSensor->vReportEnvironmentData(m_bOutsideTempValidityFlag, m_dOutsideTemp,
            m_bBaroPressureValidityFlag, m_dBarometricPressure);
   }//if ((true == m_rAAPDataServiceConfigData.bEnvData) &&...)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vReportDayNightMode()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vReportDayNightMode()
{
   //! Send Day/Night mode data
   if ((0 != m_rSelDevInfo.u32DeviceHandle) && (NULL != m_poCmdSensor))
   {
      m_poCmdSensor->vReportDayNightMode(m_enDayNightMode);
   }//if ((0 != m_rSelDevInfo.u32DeviceHandle))
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vStartVehicleSpeedUpdate
 ***************************************************************************/
t_Void spi_tclAAPDataService::vStartVehicleSpeedUpdate()
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vStartVehicleSpeedUpdate() entered "));

   //! Start timer to send VehicleSpeed data at 10Hz rate
   Timer* poTimer = Timer::getInstance();
   ETG_TRACE_USR1(("spi_tclAAPDataService::vStartVehicleSpeedUpdate()  timer value is %d"
   ,m_rAAPDataServiceConfigData.u32SpeedTimerIntervalInMs));
   if ((false == sbVehSpeedTimerRunning) && (NULL != poTimer))
   {
      poTimer->StartTimer(srVehSpeedTimerID,m_rAAPDataServiceConfigData.u32SpeedTimerIntervalInMs,
    		  m_rAAPDataServiceConfigData.u32SpeedTimerIntervalInMs, this,
            &spi_tclAAPDataService::bVehicleSpeedTimerCb, NULL);
      sbVehSpeedTimerRunning = true;

      ETG_TRACE_USR2(("[DESC]:Vehicle Speed timer started"));

   }//if ((false == sbVehSpeedTimerRunning) &&...)
   else if (true == sbVehSpeedTimerRunning)
   {
      ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer already running"));
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vStopVehicleSpeedUpdate
 ***************************************************************************/
t_Void spi_tclAAPDataService::vStopVehicleSpeedUpdate()
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vStopVehicleSpeedUpdate() entered "));

   //! Stop timer
   Timer* poTimer = Timer::getInstance();
   if ((true == sbVehSpeedTimerRunning) && (NULL != poTimer))
   {
      poTimer->CancelTimer(srVehSpeedTimerID);
      sbVehSpeedTimerRunning = false;

      ETG_TRACE_USR2(("[DESC]:Vehicle Speed timer stopped "));

   }//if ((true == sbVehSpeedTimerRunning) && ...)
   else if (false == sbVehSpeedTimerRunning)
   {
      ETG_TRACE_ERR(("[ERR]:Vehicle Speed timer not running"));
   }
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::bVehicleSpeedTimerCb
 ***************************************************************************/
t_Bool spi_tclAAPDataService::bVehicleSpeedTimerCb(
         timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   //ETG_TRACE_USR1(("spi_tclAAPDataService::bVehicleSpeedTimerCb entered "));

   spi_tclAAPDataService* poAAPDataService = static_cast<spi_tclAAPDataService*> (pvObject);
   if (NULL != poAAPDataService)
   {
      poAAPDataService->vReportVehicleSpeedInfo();
   }

   return true;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclAAPDataService::bTriggerOutsideTempUpdateCb
***************************************************************************/
t_Bool spi_tclAAPDataService::bTriggerOutsideTempUpdateCb(timer_t rTimerID, 
                                                          t_Void *pvObject,
                                                          const t_Void *pvUserData)
{
   //ETG_TRACE_USR1(("spi_tclAAPDataService::bTriggerOutsideTempUpdateCb() entered"));

   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   spi_tclAAPDataService* poAAPDataService = static_cast<spi_tclAAPDataService*> (pvObject);
   if (NULL != poAAPDataService)
   {
      poAAPDataService->vReportEnvironmentData();
   }

   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vStartOutsideTempUpdate()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vStartOutsideTempUpdate()
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vStartOutsideTempUpdate() entered "));

   if (true == m_rAAPDataServiceConfigData.bEnvData)
   {
      Timer* poTimer = Timer::getInstance();

      //Outside Temperature needs to be updated with a frequency of 0.1 Hz (Once in every 10sec).
      if ((false == sbOutsideTempUpdateTimerRunning) && (NULL != poTimer))
      {
         poTimer->StartTimer(srOutsideTempUpdateTimerID,
                  scou32OutsideTempUpdateTimerVal, scou32OutsideTempUpdateTimerVal, this,
                  &spi_tclAAPDataService::bTriggerOutsideTempUpdateCb, NULL);
         sbOutsideTempUpdateTimerRunning = true ;

         ETG_TRACE_USR2(("[DESC]:OutsideTemp Update timer started"));

      }//if ((false == sbOutsideTempUpdateTimerRunning) &&...)
      else if (true == sbOutsideTempUpdateTimerRunning)
      {
         ETG_TRACE_ERR(("[ERR]:OutsideTemp Update timer already running"));
      }
   }//if (true == m_rAAPDataServiceConfigData.bEnvData)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPDataService::vStopOutsideTempUpdate()
 ***************************************************************************/
t_Void spi_tclAAPDataService::vStopOutsideTempUpdate()
{
   ETG_TRACE_USR1(("spi_tclAAPDataService::vStopOutsideTempUpdate() entered "));

   //@Note: Environment data setting is not checked here to ensure timer is stopped
   //even if environment data availability changes.
   Timer* poTimer = Timer::getInstance();

   if ((true == sbOutsideTempUpdateTimerRunning) && (NULL != poTimer))
   {
      poTimer->CancelTimer(srOutsideTempUpdateTimerID);
      sbOutsideTempUpdateTimerRunning = false;

      ETG_TRACE_USR2(("[DESC]:OutsideTemp Update timer Stopped"));

   }//if ((true == sbOutsideTempUpdateTimerRunning) && ...)
   else if (false == sbOutsideTempUpdateTimerRunning)
   {
      ETG_TRACE_ERR(("[ERR]:OutsideTemp Update timer not running"));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPDataService::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclAAPDataService::vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
      const t_U8 cou8DriveModeRestrictionInfo)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vSetFeatureRestrictions() entered with"
         "Park mode Restriction = %d, Drive mode Restriction = %d ",
         cou8ParkModeRestrictionInfo, cou8DriveModeRestrictionInfo));

   SPI_INTENTIONALLY_UNUSED(cou8DriveModeRestrictionInfo);

   m_s32AAPParkModeRestrictionInfo = cou8ParkModeRestrictionInfo;
   //Send Driving Status
   vReportDrivingStatus();
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPDataService::vSetGeoCoordinateSystem(...)
 ******************************************************************************************************/
t_Void spi_tclAAPDataService::vSetGeoCoordinateSystem(tenGeoCoordinateSystemType enGeoCoordinateSystemType)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclAAPDataService::vSetGeoCoordinateSystem() Entered with Geo Coordinate System = %d",enGeoCoordinateSystemType));

   m_rAAPDataServiceConfigData.enGeoCoordinateSystemType = enGeoCoordinateSystemType;
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vSetTollCardStatus()
***************************************************************************/
t_Void spi_tclAAPDataService::vSetGeneralRestrictions(t_U16 u16GeneralRestrInfo)
{

   //fetching the 5th bit from the GeneralRestriction - If 1 - Enabled 0 - Disabled
   m_rAAPDataServiceConfigData.bTollCardStatus = (u16GeneralRestrInfo >> scu16AAPETCInfoBit )& 1 ;
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vSetGeneralRestrictions() entered and has Tollcard status = %u ",
        ETG_ENUM(BOOL, m_rAAPDataServiceConfigData.bTollCardStatus)));
}

/***************************************************************************
** FUNCTION: t_Void spi_tclAAPDataService::vSetETCStatus(...
***************************************************************************/
t_Void spi_tclAAPDataService::vSetETCStatus(t_Bool bETCStatus)
{
   //TODO - Based on the bETCStatus value - Carry on the implementation
   ETG_TRACE_USR1(("spi_tclAAPDataService::vSetETCStatus entered  with Tollcard status = %u ",
           ETG_ENUM(BOOL, bETCStatus)));
   vReportTollCard(bETCStatus);
   ETG_TRACE_USR1(("spi_tclAAPDataService::vSetETCStatus left"));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclAAPDataService::vReportTollCard()
***************************************************************************/
t_Void spi_tclAAPDataService::vReportTollCard(t_Bool bETCStatus)
{
   ETG_TRACE_USR1(("[DESC]:spi_tclAAPDataService::vReportTollCard() entered"));
   //! Send  Toll Card status
   if ((NULL != m_poCmdSensor) && (0 != m_rSelDevInfo.u32DeviceHandle) &&
            (true == m_rAAPDataServiceConfigData.bTollCardStatus))
   {
      m_poCmdSensor->vReportTollCardData(bETCStatus);
   }
}

///////////////////////////////////////////////////////////////////////////////
//lint restore
// <EOF>
