/*!
 *******************************************************************************
 * \file             spi_tclAautoConnection.h
 * \brief            Android auto Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Android auto Connection class to handle android devices capable of android auto
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 14.03.2015 |  Pruthvi Thej Nagaraju       | Initial Version
 27.05.2015 |  Vinoop U                    | Extend for handling Media metadata
 25.01.2016 |  Rachana L Achar             | Logiscope improvements
 08.11.2016 |  Noopur R K                  | Session Configuration for AAP
 18.07.2017 |  Noopur R K                  | Added vSetGeneralRestrictions method

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLAAUTOCONNECTION_H_
#define SPI_TCLAAUTOCONNECTION_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclConnection.h"
#include "spi_tclAAPCmdSession.h"
#include "spi_tclAAPRespSession.h"
/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
 //! Stores infomation about the last session with the AAP Device
struct trSessionHistory
{
   t_U32 u32DeviceID; 
   t_U8  u8ErrorCnt;
   tenSessionStatus enSessionStatus;
   trSessionHistory():u32DeviceID(0),u8ErrorCnt(0),enSessionStatus(e8_SESSION_UNKNOWN)
   {
   }
};
/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/


class spi_tclAautoConnection: public spi_tclConnection, //! Base Connection Class
    public spi_tclAAPRespSession
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::spi_tclAautoConnection
       ***************************************************************************/
      /*!
       * \fn     spi_tclAautoConnection()
       * \brief  Default Constructor
       * \sa      ~spi_tclAautoConnection()
       **************************************************************************/
      spi_tclAautoConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::~spi_tclAautoConnection
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclAautoConnection()
       * \brief  Destructor
       * \sa     spi_tclAautoConnection()
       **************************************************************************/
      ~spi_tclAautoConnection();

      /***************************************************************************
       ** FUNCTION:  spi_tclConnection::vOnLoadSettings
       ***************************************************************************/
      /*!
       * \fn     vOnLoadSettings()
       * \brief  Called on loadsettings
       *         Optional interface
       * \param rfrheadUnitInfo : Head unit information
       * \param enCertificateType : CertificateType to be used for authentication
       **************************************************************************/
      t_Void vOnLoadSettings(trHeadUnitInfo &rfrheadUnitInfo, tenCertificateType enCertificateType);

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vOnAddDevicetoList
       ***************************************************************************/
      /*!
       * \fn     vOnAddDevicetoList()
       * \brief  To be called when a new device has to be added to device list
       * \param  cou32DeviceHandle: Device handle of the device to be added to
       *         the device list
       **************************************************************************/
      t_Void vOnAddDevicetoList(const t_U32 cou32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vRegisterCallbacks
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks()
       * \brief  interface for the creator class to register for the required
       *        callbacks.
       * \param rfrConnCallbacks : reference to the callback structure
       *        populated by the caller
       **************************************************************************/
      t_Void vRegisterCallbacks(trConnCallbacks &ConnCallbacks);

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vSetDeviceSwitchInfo
       ***************************************************************************/
      /*!
      * \brief  Sets device switch information for Android Auto devices.
      * \param  u32DeviceID  : [IN] Device Id
      * \retval t_Bool
      **************************************************************************/
      t_Void vSetDeviceSwitchInfo(trDeviceSwitchInfo &rfrDeviceSwitchInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vSetGeneralRestrictions
       ***************************************************************************/
      /*!
      * \brief  Sets general restrictions for the session.
      * \param  u16GeneralRestrictionInfo  : General Restriction Info
      * \retval t_Void
      **************************************************************************/
      t_Void vSetGeneralRestrictions(t_U16 u16GeneralRestrictionInfo);


      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclAautoConnection::vOnSelectDevice()
      ***************************************************************************/
      /*!
      * \fn     t_Void vOnSelectDevice()
      * \brief  Function to get the deviec selection.
      * \param  cou32DeviceHandle : [IN] Device handle
      * \param  enDevConnReq : [IN] Selection status
      **************************************************************************/
      virtual t_Void vOnSelectDevice(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionType enDevConnType,
               tenDeviceConnectionReq enDevSelReq, tenEnabledInfo enDAPUsage,
               tenEnabledInfo enCDBUsage, tenSelectReason enSelectionReason,
               const trUserContext corUsrCntxt, tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAautoConnection::vOnSelectDeviceResult()
       ***************************************************************************/
       /*!
       * \fn      t_Void vOnSelectDeviceResult
       * \brief   To perform the actions that are required, after the select device is
       *           successful/failed
       * \pram    cou32DeviceHandle  : [IN] Uniquely identifies the target Device.
       * \pram    enDevSelReq : [IN] Identifies the Connection Request.
       * \pram    coenRespCode: [IN] Response code. Success/Failure
       * \pram    enDevCat    : [IN] Device Category. ML/DiPo
       * \retval  t_Void
       **************************************************************************/
       t_Void vOnSelectDeviceResult(const t_U32 cou32DeviceHandle,
          const tenDeviceConnectionReq coenConnReq,
          const tenResponseCode coenRespCode,
          tenDeviceCategory enDevCat, tenSelectReason enSelectionReason, tenDeviceType enDeviceType = e8_UNKNOWN_DEVICE);

   private:
      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/
     /***************************************************************************
      ** FUNCTION:  t_Void spi_tclAautoConnection::vPostSelctDeviceResult()
      ***************************************************************************/
      /*!
      * \fn     t_Void vPostSelctDeviceResult()
      * \brief  Function to post the device selection statu.
      * \pram    enDevSelReq : [IN] Identifies the Connection Request.
      * \param  enResponse : [IN] Operation status.
      * \param  enErrorType : [IN] error value.
      **************************************************************************/
      virtual t_Void vPostSelectDeviceResult(tenDeviceConnectionReq coenConnReq,
               tenResponseCode enResponse,  tenErrorCode enErrorType);

      //! Methods overwriteen from Andorid auto discoverer response class
      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclAautoConnection::vPostDeviceInfo
      ***************************************************************************/
      /*!
      * \fn      t_Void vPostDeviceInfo
      * \brief   To Post the device info to SPI, when a new device is detected
      * \param   cou32DeviceHandle     : [IN] Device Id
      * \param   corfrDeviceInfo : [IN] const reference to the DeviceInfo structure.
      * \retval  t_Void
      * \sa      vPostDeviceDisconncted(const t_U32 cou32DeviceHandle)
      ***************************************************************************/
      virtual t_Void vPostDeviceInfo(
          const t_U32 cou32DeviceHandle,
          const trDeviceInfo &corfrDeviceInfo);

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclAautoConnection::vPostDeviceDisconnected
      ***************************************************************************/
      /*!
      * \fn      virtual t_Void vPostDeviceDisconnected
      * \brief   To Post the Device Id to SPI, when a device is disconnected
      * \param   rUserContext : [IN] Context information passed from the caller
      * \param   cou32DeviceHandle    : [IN] Device Id
      * \retval  t_Void
      **************************************************************************/
      virtual t_Void vPostDeviceDisconnected(const t_U32 cou32DeviceHandle);

      //! Methods from android auto session response class
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclAAPSession::vUnrecoverableErrorCallback()
       ***************************************************************************/
      /*!
       * \fn      vUnrecoverableErrorCallback
       * \brief   called when phone detects a unrecoverable error:
       * advice is to reset USB on this call
       **************************************************************************/
       t_Void vUnrecoverableErrorCallback();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclAAPSession::vByeByeRequestCb()
       ***************************************************************************/
      /*!
       * \fn      vByeByeRequestCb
       * \brief   Called when ByeByeRequest is received from phone. After taking necessary steps,
       * car side should send ByeByeResponse.
       * \param   enReason  The reason for the disconnection request.
       **************************************************************************/
       t_Void vByeByeRequestCb(tenAAPByeByeReason enReason);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclAAPSession::vByeByeResponseCb()
       ***************************************************************************/
      /*!
       * \fn      vByeByeResponseCb
       * \brief  Called when ByeByeResponse is received from phone. Normally this is a reply for
       * ByeByeRequest message sent from car.
       **************************************************************************/
       t_Void vByeByeResponseCb();

     /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclAAPResourceMngr::vSessionStatusInfo()
      ***************************************************************************/
     /*!
      * \fn      vSessionStatusInfo
      * \brief  informs the current session status of android auto session. function
      *         overridden from spi_tclAAPRespSession.h
      * \param  enSessionStatus : indicates current status of android auto session
      **************************************************************************/
      t_Void vSessionStatusInfo(tenSessionStatus enSessionStatus, t_Bool bSessionTimedOut);

      //! Callbacks for ConnMngr to register. These callbacks will be used to
      //! inform device detection and device disconnection to connection manager
      trConnCallbacks m_rAautoConnCallbacks;

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vOnDeviceConnection
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceConnection()
       * \brief  to be called on Android auto device connection
       **************************************************************************/
      t_Void vOnDeviceConnection(const t_U32 cou32DeviceHandle,
               const trDeviceInfo &corfrDeviceInfo);

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vOnDeviceDisconnection
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceDisconnection()
       * \brief to be called on android auto device disconnection
       **************************************************************************/
      t_Void vOnDeviceDisconnection(const t_U32 cou32DeviceHandle);


      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::bPrepareSession
       ***************************************************************************/
      /*!
       * \fn     bPrepareSession()
       * \brief  Prepares for android auto session
       * \brief  true if session preparation successful
       **************************************************************************/
      t_Bool bPrepareSession();

      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vProcessDeviceSelection(const t_U32...
       ***************************************************************************/
      /*!
       * \fn     vProcessDeviceSelection(const t_U32& corfu32DeviceHandle)
       * \brief  Triggers AAP switch if the AAP connection is not up
       *         Posts SelectDevice result
       * \param  corfu32DeviceHandle  : [IN] Device handle
       **************************************************************************/
      t_Void vProcessDeviceSelection(const t_U32& corfu32DeviceHandle);
  
      /***************************************************************************
       ** FUNCTION:  spi_tclAautoConnection::vProcessDeviceDeselection()
       ***************************************************************************/
      /*!
       * \fn     vProcessDeviceDeselection(const t_U32& corfu32DeviceHandle,
       *         tenSelectReason enSelectionReason)
       * \brief  Checks if device is deselected from HMI.
       *	     If yes, sends Bye Bye message.
       *         Else stops transport and posts SelectDevice result
       * \param  corfu32DeviceHandle : [IN] Device handle
       * \param  enSelectionReason   : [IN] Device deselection from HMI check
       **************************************************************************/
      t_Void vProcessDeviceDeselection(const t_U32& corfu32DeviceHandle,
             tenSelectReason enSelectionReason);

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclAAPRespSession::vServiceDiscoveryRequestCb
      ***************************************************************************/
      /*!
      * \fn     vServiceDiscoveryRequestCb
      * \brief   Called when service discovery request is received.
      *          This call sends across an icon set and a label that can be
      *          used by the native UI to display a button that allows
      *          users to switch back to projected mode.
      * \param     szSmallIcon   32x32 png image.
      * \param     szMediumIcon  64x64 png image.
      * \param     szLargeICon   128x128 png image.
      * \param     szLabel  A label that may be displayed alongside the icon.
      * \param     szDeviceName Name of Device
      ***************************************************************************/
       t_Void vServiceDiscoveryRequestCb(t_String szSmallIcon, t_String szMediumIcon, t_String szLargeICon , t_String szLabel , t_String szDeviceName);

      /***************************************************************************
      ** FUNCTION:  spi_tclAautoConnection::vSetSessionConfiguration
      ***************************************************************************/
      /*!
      * \fn     vSetSessionConfiguration()
      * \brief  Set the session configuration value
      * \param
      *\retval : t_Void
      **************************************************************************/
      t_Void vSetSessionConfigurationInfo(trSessionConfigurationInfo &rfrSessionConfigurationInfo);

      /***************************************************************************
      ** FUNCTION:  spi_tclAautoConnection::vStopPingTimer
      ***************************************************************************/
      /*!
      * \fn     vStopPingTimer()
      * \brief  Stop the Ping Timer
      * \param
      *\retval : t_Void
      **************************************************************************/
      t_Void vStopPingTimer();

      /***************************************************************************
      ** FUNCTION:  spi_tclAautoConnection::bAAPPingTimerCb
      ***************************************************************************/
      /*!
      * \fn     bAAPPingTimerCb()
      * \brief  called on expiry of ping timer
       * \param  rTimerID: ID of the timer which has expired
       * \param  pvObject: pointer to object passed while starting the timer
       * \param  pvUserData: data passed during start of the timer
      **************************************************************************/
      static t_Bool bAAPPingTimerCb(timer_t rTimerID, t_Void *pvObject,
                  const t_Void *pvUserData);

      /***************************************************************************
      ** FUNCTION:  spi_tclAautoConnection::vTriggerPingRequest
      ***************************************************************************/
      /*!
      * \fn     vTriggerPingRequest()
      * \brief  To trigger the sendping request to the endpoint
      *\retval : t_Void
      **************************************************************************/
      t_Void vTriggerPingRequest();

      /***************************************************************************
      ** FUNCTION:  spi_tclAautoConnection::vStopPingTimer
      ***************************************************************************/
      /*!
      * \fn     vPingResponseCb()
      * \brief  Called when the response for the ping request is received
      * \param
      *\retval : t_Void
      **************************************************************************/
      t_Void vPingResponseCb(t_S64 s64timestamp);

      //! Current selected device
      t_U32 m_u32CurrSelectedDevice; 

      //! Indicates if device switch request is in progress
      t_Bool m_bSwitchInProgress;

      //! Store head unit information
      trHeadUnitInfo m_rHeadUnitInfo;

      //! Store the Certificate Configuration
      trAAPCertConfigData  m_rCertsConfigData;

      //! Maintain the status of last session
      trSessionHistory m_rSessionHistory;

      //! Certificate type
      tenCertificateType m_enCertificateType;

      trDeviceSwitchInfo m_rDeviceSwitchInfo;

      timer_t m_rPingTimerId;

      //! counter for maximum count of requests to be sent without reponse
      t_U8 m_u8PingRespCounter;

      //!Ping weak connection counter
      t_U8 m_u8PingWeakConnCounter;

      //!Variable to store the connection type
      tenDeviceConnectionType m_enDevConnType;

      //!Variable to store the connection request
      tenDeviceConnectionReq m_enDevConnReq;
};
#endif // SPI_TCLAAUTOCONNECTION_H_
