/*******************************************************************************
 * \file              spi_tclAAPSessionCbsCbs.h
 * \brief             Class implementing callbacks from GALreceiver
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Class implementing callbacks from GALreceiver
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 17.03.2015 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/
/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclAAPMsgQInterface.h"
#include "spi_tclAAPSessionDispatcher.h"
#include "spi_tclAAPSessionCbs.h"
#include "spi_tclAAPSession.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AAPWRAPPER
#include "trcGenProj/Header/spi_tclAAPSessionCbs.cpp.trc.h"
#endif
#endif


//! Galreceiver callbacks
/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::serviceDiscoveryRequestCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::serviceDiscoveryRequestCallback(const std::string& smallIcon, const std::string& mediumIcon,
      const std::string& largeIcon, const std::string& label, const std::string& deviceName)
{
   ETG_TRACE_USR1((" serviceDiscoveryRequestCallback smallIcon = %d mediumIcon = %d largeIcon = %d\n",
            smallIcon.size(), mediumIcon.size(), largeIcon.size()));
   ETG_TRACE_USR1((" serviceDiscoveryRequestCallback label = %s \n", label.c_str()));
   ETG_TRACE_USR1((" serviceDiscoveryRequestCallback Device Name = %s \n", deviceName.c_str()));

   //! TODO check with Google if string is appropriate type for ICon since
   //! Icon with null entry inside the JPEG image will cause issues
   //! Callbacks for the registered objects
   AAPServiceDiscoveryMsg oServiceDiscMsg;
   if((NULL != oServiceDiscMsg.m_pszSmallIcon) && (NULL != oServiceDiscMsg.m_pszMediumIcon) &&
            (NULL != oServiceDiscMsg.m_pszLargeIcon) && (NULL != oServiceDiscMsg.m_pszLabel) && (NULL != oServiceDiscMsg.m_pszDeviceName))
   {
      *(oServiceDiscMsg.m_pszSmallIcon) = smallIcon.c_str();
      *(oServiceDiscMsg.m_pszMediumIcon) = mediumIcon.c_str();
      *(oServiceDiscMsg.m_pszLargeIcon) = largeIcon.c_str();
      *(oServiceDiscMsg.m_pszLabel) = label.c_str();
      *(oServiceDiscMsg.m_pszDeviceName) = deviceName.c_str();
   }

   spi_tclAAPMsgQInterface *poMsgQinterface =
            spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oServiceDiscMsg, sizeof(oServiceDiscMsg));
   }//if (NULL != poMsgQinterface)

   //! Send session status as active if this message is received from phone
   //! This message is considered to be initialization message (suggested from google)
   AAPSessionStatusMsg oSessionStatus;
   oSessionStatus.m_enSessionstatus = e8_SESSION_ACTIVE;
   if (NULL != poMsgQinterface)
     {
        poMsgQinterface->bWriteMsgToQ(&oSessionStatus, sizeof(oSessionStatus));
     }//if (NULL != poMsgQinterface)

   //! Stop the session timer after receiving the session init message
   spi_tclAAPSession* poAAPSession = spi_tclAAPSession::getInstance() ;
   if(NULL != poAAPSession)
   {
      poAAPSession->vStopSessionTimer();
   }  // if(NULL != poAAPSession)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::unrecoverableErrorCallback(MessageStatus err)
 ***************************************************************************/
void spi_tclAAPSessionCbs::unrecoverableErrorCallback(MessageStatus err)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::unrecoverableErrorCallback entered\n"));
   ETG_TRACE_ERR(("[ERR]:spi_tclAAPSessionCbs::unrecoverableErrorCallback received from phone with error - %d\n", ETG_ENUM(AAP_STATUS_RETURNED, err)));
   //! TODO Create messages and send to response class

   //! Send session status as error if this message is received from phone
   AAPSessionStatusMsg oSessionStatus;
   spi_tclAAPMsgQInterface *poMsgQinterface =
            spi_tclAAPMsgQInterface::getInstance();

   switch(err)
   {
    case STATUS_AUTHENTICATION_FAILURE:
    case STATUS_AUTHENTICATION_FAILURE_CERT_NOT_YET_VALID:
         oSessionStatus.m_enSessionstatus = e8_SESSION_AUTHENTICATION_FAILURE;
         break;
    default:
         oSessionStatus.m_enSessionstatus = e8_SESSION_ERROR;
   }
   
   oSessionStatus.m_bSessionTimedOut = false;
   if ((err != STATUS_FRAMING_ERROR) && (NULL != poMsgQinterface))
   {
      poMsgQinterface->bWriteMsgToQ(&oSessionStatus, sizeof(oSessionStatus));
   }//if (NULL != poMsgQinterface)

   //! Stop the session timer after receiving the session error message
   spi_tclAAPSession* poAAPSession = spi_tclAAPSession::getInstance() ;
   if(NULL != poAAPSession)
   {
      poAAPSession->vStopSessionTimer();
   }  // if(NULL != poAAPSession)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::pingRequestCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::pingRequestCallback(int64_t timestamp, bool bugReport)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::pingRequestCallback entered\n"));
   ETG_TRACE_USR2(("[DESC]:On receiving a Ping request with timestamp = %d,  bugReport = %d\n", timestamp, ETG_ENUM(BOOL, bugReport) ));
   //! TODO Create messages and send to response class
   //! Currently not used
}


/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::pingResponseCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::pingResponseCallback(int64_t timestamp)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::pingResponseCallback entered\n"));
   ETG_TRACE_USR2(("[DESC]:Response call back with timestamp = %s \n", std::to_string(timestamp).c_str()));
   AAPPingResponseMsg oPingResponseMsg;
   (oPingResponseMsg.m_s64timestamp) = timestamp;
   spi_tclAAPMsgQInterface *poMsgQinterface =
               spi_tclAAPMsgQInterface::getInstance();
      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oPingResponseMsg, sizeof(oPingResponseMsg));
      }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::navigationFocusCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::navigationFocusCallback(NavFocusType focusType)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::navigationFocusCallback entered\n"));
   ETG_TRACE_USR2(("[DESC]:Navigation focus callback with focusType = %d\n", ETG_ENUM(NAV_FOCUS_TYPE, focusType) ));

   //! Callbacks for the registered objects
   AAPNavigationFocusMsg oNavFocusMsg;
   oNavFocusMsg.m_enNaviFocusType = static_cast<tenAAPNavFocusType>(focusType);

   spi_tclAAPMsgQInterface *poMsgQinterface =
            spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oNavFocusMsg, sizeof(oNavFocusMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::byeByeRequestCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::byeByeRequestCallback(ByeByeReason reason)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::byeByeRequestCallback entered\n"));
   ETG_TRACE_USR2(("[DESC]:Received byebye request from phone - ByeByeReason = %d\n", ETG_ENUM(AAP_BYEBYEREASON, reason) ));
   AAPByeByeRequestMsg oByeByeRequest;
   oByeByeRequest.m_enByeByeReason =  static_cast<tenAAPByeByeReason>(reason);

   spi_tclAAPMsgQInterface *poMsgQinterface =
            spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oByeByeRequest, sizeof(oByeByeRequest));
   }//if (NULL != poMsgQinterface)
}



/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::byeByeResponseCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::byeByeResponseCallback()
{
   ETG_TRACE_USR1(("spi_tclAAPSessionCbs::byeByeResponseCallback entered  \n"));

   //! Session status can be sent as inactive when the phone has sent byebyeresponse as there will
   //! be no communication between the device and the Head unit. This will also prevent
   //! wrong session active status update to HMI during deselection.
   //! Send session status as inactive
   AAPSessionStatusMsg oSessionStatus;
   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   oSessionStatus.m_enSessionstatus = e8_SESSION_INACTIVE;
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oSessionStatus, sizeof(oSessionStatus));
   }//if (NULL != poMsgQinterface)


   AAPByeByeResponseMsg oByeByeResp;
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oByeByeResp, sizeof(oByeByeResp));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::voiceSessionNotificationCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::voiceSessionNotificationCallback(VoiceSessionStatus status)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::voiceSessionNotificationCallback entered  \n"));
   ETG_TRACE_USR2(("[DESC]:A voice session notification is received - VoiceSessionStatus = %d\n", ETG_ENUM(AAP_VOICESESSION_STATUS, status) ));
   AAPVoiceSessionNotifMsg oVoiceNoti;
   oVoiceNoti.m_enVoiceSessionStatus = static_cast<tenAAPVoiceSessionStatus>(status);

   spi_tclAAPMsgQInterface *poMsgQinterface =
            spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oVoiceNoti, sizeof(oVoiceNoti));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSessionCbs::audioFocusRequestCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::audioFocusRequestCallback(AudioFocusRequestType request)
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::audioFocusRequestCallback entered  \n"));
   ETG_TRACE_USR2(("[DESC]:Setting audio focus request with AudioFocusRequestType = %d\n", ETG_ENUM(DEVICE_AUDIOFOCUS_REQ, request) ));
   AAPAudioFocusRequestMsg oAudioFocusMsg;
   oAudioFocusMsg.m_enDevAudFocusRequest = static_cast<tenAAPDeviceAudioFocusRequest>(request);

   spi_tclAAPMsgQInterface *poMsgQinterface =
            spi_tclAAPMsgQInterface::getInstance();
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oAudioFocusMsg, sizeof(oAudioFocusMsg));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  void spi_tclAAPSessionCbs::bugreportRequestCallback(int64_t...)
 ***************************************************************************/
void spi_tclAAPSessionCbs::bugreportRequestCallback(int64_t timestamp)
{
   SPI_INTENTIONALLY_UNUSED(timestamp);
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::bugreportRequestCallback entered  \n"));
}

/***************************************************************************
 ** FUNCTION:  void spi_tclAAPSessionCbs::carConnectedDevicesRequestCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::carConnectedDevicesRequestCallback()
{
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::carConnectedDevicesRequestCallback entered  \n"));

}

/***************************************************************************
 ** FUNCTION:  void spi_tclAAPSessionCbs::userSwitchRequestCallback()
 ***************************************************************************/
void  spi_tclAAPSessionCbs:: userSwitchRequestCallback(const ConnectedDeviceStruct& selectedDevice)
{
   SPI_INTENTIONALLY_UNUSED(selectedDevice);
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::userSwitchRequestCallback entered  \n"));

}

/***************************************************************************
 ** FUNCTION:  void spi_tclAAPSessionCbs::batteryStatusNotificationCallback()
 ***************************************************************************/
void spi_tclAAPSessionCbs::batteryStatusNotificationCallback(int32_t batteryLevel,
           int32_t timeRemainingS, bool batteryCritical)
{
   SPI_INTENTIONALLY_UNUSED(batteryLevel);
   SPI_INTENTIONALLY_UNUSED(timeRemainingS);
   SPI_INTENTIONALLY_UNUSED(batteryCritical);
   ETG_TRACE_USR1((" spi_tclAAPSessionCbs::batteryStatusNotificationCallback entered  \n"));

}
