/*!
 *******************************************************************************
 * \file              spi_tclAAPSession.cpp
 * \brief             Device session core class for Android Auto
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Device session core class for Android Auto
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 27.02.2015 |  Pruthvi Thej Nagaraju       | Initial Version
 08.11.2016 |  Noopur R K                  | Session Configuration for AAP
 26.06.2018 |  Roveena Francy Lobo         | Added function vValidateVehicleModelYear()
 \endverbatim
 ******************************************************************************/
/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <aauto/DevelopersAuthenticator.h>
#include <aauto/AoapTransport.h>
#if defined GEN3ARM || defined GEN3X86
  #include <aauto/SdcAuthenticator.h>
#endif
#include <aauto/FSAuthenticator.h>
#include "crc.h"
#include "StringHandler.h"
#include "Timer.h"
#include "SPITypes.h"
#include "spi_tclAAPSession.h"
#include "spi_tclAAPMsgQInterface.h"
#include "spi_tclAAPSessionDispatcher.h"
#include "spi_tclAAPSessionCbs.h"
#include "spi_tclAAPTransportCbs.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AAPWRAPPER
#include "trcGenProj/Header/spi_tclAAPSession.cpp.trc.h"
#endif
#endif

using namespace adit::aauto;
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
static const t_U32 scou32TimeforSessionMsginms = 20000;
static const t_U16 MAXLENGTH = 4;
/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclAAPSession::spi_tclAAPSession();
 ***************************************************************************/
spi_tclAAPSession::spi_tclAAPSession() : m_poTransport(NULL), m_oSessionTimerId(0), m_bShutdownInProgress(false), m_u32SessionConfiguration(0)
{
   //! TODO remove this trace in later stage of devp
   ETG_TRACE_USR1(("spi_tclAAPSession::spi_tclAAPSession entered"));
}

/***************************************************************************
 ** FUNCTION:  virtual spi_tclAAPSession::~spi_tclAAPSession()
 ***************************************************************************/
spi_tclAAPSession::~spi_tclAAPSession()
{
   //! TODO remove this trace in later stage of devp
   ETG_TRACE_USR1(("spi_tclAAPSession::~spi_tclAAPSession entered"));
   //! Destroy Transport
   m_poTransport = NULL;
   m_oSessionTimerId = 0;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::bInitializeSession()
 ***************************************************************************/
t_Bool spi_tclAAPSession::bInitializeSession()
{
   ETG_TRACE_USR1(("spi_tclAAPSession::bInitializeSession entered"));
	/*lint -esym(40,nullptr) nullptr is not declared */
	/*lint -esym(55,nullptr) Badtype */
   t_Bool bRetval =  false;

   m_bShutdownInProgress = false;
   //! Create GalReceiver object
   m_spoGalReceiver = new GalReceiver();
   SPI_NORMAL_ASSERT(m_spoGalReceiver == nullptr);

   //! Create object to handle galreceiver calbbacks
   m_spoSessionCbs = new spi_tclAAPSessionCbs();
   SPI_NORMAL_ASSERT(m_spoSessionCbs == nullptr);

   /*lint -esym(40,nullptr) nullptr is not declared */
   
   //! Register callbacks and initialize receiver library
   if((m_spoGalReceiver != nullptr) && (m_spoSessionCbs != nullptr))
   {
      bRetval = m_spoGalReceiver->init(m_spoSessionCbs);
   }

   //! Start the timer on initializing session to wait for the
   //!first message from Mobile Device. if the timer expires before receiving the
   //! first session message from phone, then the phone doesn't support AAP
   Timer* poAAPSessionTimer = Timer::getInstance();
   if(NULL != poAAPSessionTimer)
   {
      timer_t rTimerID;
      //! Start timer
      poAAPSessionTimer->StartTimer(rTimerID, scou32TimeforSessionMsginms, 0, this,
               &spi_tclAAPSession::bSessionTimerCb, NULL);
      m_oSessionTimerId = rTimerID;
   };

   ETG_TRACE_USR2(("[DESC]:Creating and Initializing GALReceiver result = %d",
         ETG_ENUM(BOOL, bRetval)));
   ETG_TRACE_USR1(("spi_tclAAPSession::bInitializeSession exited"));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::vUnInitializeSession()
 ***************************************************************************/
t_Void spi_tclAAPSession::vUnInitializeSession()
{ 
   /*lint -esym(40,nullptr) nullptr is not declared */
   /*lint -esym(55,nullptr) Badtype */
   ETG_TRACE_USR1(("spi_tclAAPSession::vUnInitializeSession(): Shutting down and destroying GALReceiver"));
   if (m_spoGalReceiver != nullptr)
   {
      m_spoGalReceiver->shutdown();
   }
   /*lint -esym(40,nullptr) nullptr is not declared */
   
   //! Assign the pointer to null to release the memory as this is a shared pointer
   m_spoGalReceiver = nullptr;

   m_bShutdownInProgress = false;
   /*lint -esym(40,nullptr) nullptr is not declared */
   
   m_spoSessionCbs = nullptr;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::bStartTransport
 ***************************************************************************/
t_Bool spi_tclAAPSession::bStartTransport(const trAAPSessionInfo &rfrSessionInfo)
{
   ETG_TRACE_USR1(("spi_tclAAPSession::bStartTransport entered"));
   t_Bool bRetval =  false;
   /*lint -esym(40,nullptr) nullptr is not declared */
   
   //! Start transport once all services are registered
   if (m_spoGalReceiver != nullptr)
   {
      //Send Session Configuration value.
	  m_spoGalReceiver->setSessionConfiguration(m_u32SessionConfiguration);
	  
      m_spoGalReceiver->start();

      //! create Transport with AOAP specific implementation
      //! to establish data transfer between MD and HU.
      aoapTransportInfo_t rAoapTransportInfo;
      rAoapTransportInfo.aoapAccessoryId = rfrSessionInfo.u32HeadUnitID;
      rAoapTransportInfo.aoapDeviceId = rfrSessionInfo.u32DeviceID;
      m_poTransport = new AoapTransport(m_spoGalReceiver, &rAoapTransportInfo);
      
      //!Create object to handle transport callbacks
      m_spoTransportCbs = new spi_tclAAPTransportCbs();
      SPI_NORMAL_ASSERT(m_spoTransportCbs == nullptr);

      if (NULL != m_poTransport)
      {
         //! Start the GalReceiver transport
         m_poTransport->registerCallbacks(m_spoTransportCbs);
         bRetval = m_poTransport->start();
      }
   }
   ETG_TRACE_USR4(("[PARAM]::bStartTransport - Starting AOAP transport result = %d",
         ETG_ENUM(BOOL, bRetval)));
   return bRetval;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::vStopTransport()
 ***************************************************************************/
t_Void spi_tclAAPSession::vStopTransport()
{
   ETG_TRACE_USR1(("spi_tclAAPSession::vStopTransport(): Stopping AOAP transport"));
   if(NULL != m_poTransport)
   {
      m_bShutdownInProgress = true;
      //! First Stop galreceiver transport and wait till its completely stopped
      //! request for stop and then Join the reader and writer threads  
      //! request for stop will internally invoke prepareshutdwon of galreceiver
      ETG_TRACE_USR1(("spi_tclAAPSession::requestStop"));
      m_poTransport->requestStop();

      ETG_TRACE_USR1(("spi_tclAAPSession::waitForExit"));
      m_poTransport->waitForExit();
   }

   ETG_TRACE_USR1(("Releasing memory allocated for aoap transport"));
   //! Destroy Transport
   RELEASE_MEM(m_poTransport);
   
   m_spoTransportCbs = nullptr;
   //! Session status can be sent as inactive when the transport is stopped as there will
   //! be no communication between the device and the Head unit. This will also prevent
   //! wrong session active status update to HMI during deselection.
   //! Send session status as inactive
   AAPSessionStatusMsg oSessionStatus;
   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   oSessionStatus.m_enSessionstatus = e8_SESSION_INACTIVE;
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oSessionStatus, sizeof(oSessionStatus));
   }//if (NULL != poMsgQinterface)
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::bSetCertificates()
 ***************************************************************************/
t_Bool spi_tclAAPSession::bSetCertificates(trAAPCertConfigData &rfrCertsConfigData, tenCertificateType enCertificatetype)
{ 
ETG_TRACE_USR1(("spi_tclAAPSession::bSetCertificates entered"));
/*lint -esym(40,nullptr) nullptr is not declared */

   t_Bool bRetVal = false;

   ETG_TRACE_USR1(( "Certificate type is %d",ETG_ENUM(AAP_CERTIFICATE_TYPE,enCertificatetype)));
   ETG_TRACE_USR1((" Key Id is %s",rfrCertsConfigData.szKeyId.c_str()));
   t_String szRootCertificate = rfrCertsConfigData.szCertificateFolderPath+"/rootcert.wrap";
   ETG_TRACE_USR1(("RootCertificate Path is %s",szRootCertificate.c_str()));
   t_String szClientCertificate = rfrCertsConfigData.szCertificateFolderPath+"/clientcert.wrap";
   ETG_TRACE_USR1(("ClientCertificate Path is %s",szClientCertificate.c_str()));
   t_String szPrivateKey = rfrCertsConfigData.szCertificateFolderPath+"/privatekey.wrap";
   ETG_TRACE_USR1(("PrivateKey Path is %s",szPrivateKey.c_str()));

   //! Use the authenticator based on the project configuration
   if (m_spoGalReceiver != nullptr)
   {
      switch(enCertificatetype)
      {
         case e8_CERTIFICATETYPE_DEVELOPER:
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclAAPSession::bSetCertificates Using developer certificates"));
            //! Use the developer authenticator for using the
            //! developer certificates embedded in code
            DevelopersAuthenticator oDevpAuthenticator;
            bRetVal = oDevpAuthenticator.setCertificates(m_spoGalReceiver);
            break;
         }

         case e8_CERTIFICATETYPE_SDC:
         {
            ETG_TRACE_ERR(("spi_tclAAPSession::bSetCertificates using certificates in SDC"));
            //! Use the SDC authenticator for more secure certificate storage
            #if defined GEN3ARM || defined GEN3X86
              bRetVal = false;
            if(!rfrCertsConfigData.szKeyId.empty()) //! KeyId should not be empty otherwise reset would be seen in SDC
            {
               SdcAuthenticator oSdcAuthenticator;

               oSdcAuthenticator.setConfigItem("keyId"            , rfrCertsConfigData.szKeyId);
               oSdcAuthenticator.setConfigItem("rootCertificate"  , szRootCertificate  );
               oSdcAuthenticator.setConfigItem("clientCertificate", szClientCertificate);
               oSdcAuthenticator.setConfigItem("privateKey"       , szPrivateKey       );
               bRetVal = oSdcAuthenticator.setCertificates(m_spoGalReceiver);
            }
            #endif
            break;
         }
         case e8_CERTIFICATETYPE_FFS:
         {
             ETG_TRACE_ERR(("spi_tclAAPSession::bSetCertificates using certificates in FFS"));
             //! Use the FFS certificates for using raw certificates
             FSAuthenticator oFSAuthenticator;

             oFSAuthenticator.setConfigItem("rootCertificate"  , szRootCertificate  );
             oFSAuthenticator.setConfigItem("clientCertificate", szClientCertificate);
             oFSAuthenticator.setConfigItem("privateKey"       , szPrivateKey       );
             bRetVal = oFSAuthenticator.setCertificates(m_spoGalReceiver);
             break;
         }
         default:
         {
            ETG_TRACE_ERR(("[ERR]:spi_tclAAPSession::bSetCertificates Certificate type not supported"));
         }

      }
   }
   ETG_TRACE_USR4(("[PARAM]::bSetCertificates -  Certificatetype = %d Return Value = %d",
         ETG_ENUM(CERTIFICATE_TYPE,enCertificatetype), ETG_ENUM(BOOL, bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPSession::vValidateVehicleModelYear
 ***************************************************************************/
t_Void spi_tclAAPSession::vValidateVehicleModelYear(t_String & rfszVehicleModelYear)
{
   ETG_TRACE_USR1(("spi_tclAAPSession::vValidateVehicleModelYear entered"));
   if((true == rfszVehicleModelYear.empty()) || (rfszVehicleModelYear.length() != MAXLENGTH) || ( false == std::all_of(rfszVehicleModelYear.begin(), rfszVehicleModelYear.end(), ::isdigit)))
   {
      rfszVehicleModelYear = "multi";
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::vSetHeadUnitInfo()
 ***************************************************************************/
t_Void spi_tclAAPSession::vSetHeadUnitInfo(const trAAPHeadUnitInfo & rfrHeadUnitInfo)
{
   ETG_TRACE_USR1(("spi_tclAAPSession::vSetHeadUnitInfo entered"));
  // if(NULL != m_spoGalReceiver)
   if (m_spoGalReceiver != nullptr)
   {
      ETG_TRACE_USR2(("Vehicle Model Year before validating = %s", rfrHeadUnitInfo.szYear.c_str()));
      t_String rfszVehicleModelYear = rfrHeadUnitInfo.szYear;
      //Validate th emodel year to set it to multi if the format is incorrect.
      vValidateVehicleModelYear(rfszVehicleModelYear);
      ETG_TRACE_USR2(("[DESC]:Setting vehicle identity info : Vehicle Manufacturer = %s", rfrHeadUnitInfo.szVehicleManufacturer.c_str()));
      ETG_TRACE_USR2(("[DESC]:Setting vehicle identity info : Vehicle Model = %s", rfrHeadUnitInfo.szVehicleModel.c_str()));
      ETG_TRACE_USR2(("[DESC]:Setting vehicle identity info : Vehicle Model Year = %s", rfszVehicleModelYear.c_str()));
      ETG_TRACE_USR2(("[DESC]:Setting vehicle identity info : Serial Number = %s", rfrHeadUnitInfo.szSerial.c_str()));
      //! Set the vehicle and Headunit information
      m_spoGalReceiver->setIdentityInfo(rfrHeadUnitInfo.szVehicleManufacturer.c_str(), rfrHeadUnitInfo.szVehicleModel.c_str(),
               rfszVehicleModelYear.c_str(), rfrHeadUnitInfo.szSerial.c_str());
      //! Sets the driver position - left/right/center.
      //! This influences the layout of the screen (currently no change seen)
      m_spoGalReceiver->setDriverPosition(static_cast<DriverPosition>(rfrHeadUnitInfo.enDriverPos));

      ETG_TRACE_USR2(("[DESC]: Setting Headunit info : Manufacturer = %s", rfrHeadUnitInfo.szManufacturer.c_str()));
      ETG_TRACE_USR2(("[DESC]: Setting Headunit info : Head Unit Model = %s", rfrHeadUnitInfo.szModelName.c_str()));
      ETG_TRACE_USR2(("[DESC]: Setting Headunit info : Head unit software version = %s", rfrHeadUnitInfo.szSoftwareVersion.c_str()));
      ETG_TRACE_USR2(("[DESC]: Setting Headunit info : Head unit software build = %s", rfrHeadUnitInfo.szSoftwareBuild.c_str()));

      //! software build is not known hence left as empty string
      m_spoGalReceiver->setHeadUnitInfo(rfrHeadUnitInfo.szManufacturer.c_str()/*Make*/,rfrHeadUnitInfo.szModelName.c_str(),
            rfrHeadUnitInfo.szSoftwareBuild.c_str(), rfrHeadUnitInfo.szSoftwareVersion.c_str());

      //! TODO: REMOVE: Call not needed since our system maintains time
      //m_spoGalReceiver->setCertificateVerificationTime(1417449318);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPSession::vSendByeByeMessage
 ***************************************************************************/
t_Void spi_tclAAPSession::vSendByeByeMessage()
{
   ETG_TRACE_USR1(("  spi_tclAAPSession::vSendByeByeMessage entered \n"));
   //! TODO extend interface for future ByeBye reasons
   //! Send Byebye message to phone
   if (m_spoGalReceiver != nullptr)
   {
      m_spoGalReceiver->sendByeByeRequest(USER_SELECTION);
   }
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPSession::vSendByeByeResponse
 ***************************************************************************/
t_Void spi_tclAAPSession::vSendByeByeResponse()
{
   /*lint -esym(40,nullptr) nullptr is not declared */
   //!
   ETG_TRACE_USR1(("  spi_tclAAPSession::vSendByeByeResponse(): entered\n"));
   if (m_spoGalReceiver != nullptr)
   {
      m_spoGalReceiver->sendByeByeResponse();
   }
}

/***************************************************************************
 ** FUNCTION:  shared_ptr<GalReceiver> spi_tclAAPSession::poGetGalReceiver();
 ***************************************************************************/
::shared_ptr<GalReceiver> spi_tclAAPSession::poGetGalReceiver()
{
   /*lint -esym(40,nullptr) nullptr is not declared */
   ETG_TRACE_USR1(("spi_tclAAPSession::poGetGalReceiver entered"));
   t_Bool bValid = (m_spoGalReceiver != nullptr);
   ETG_TRACE_USR2(("[DESC]:Getting GAL receiver,  m_spoGalReceiver valid = %d", ETG_ENUM(BOOL, bValid)));
   return m_spoGalReceiver;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::vSetNavigationFocus()
 ***************************************************************************/
t_Void spi_tclAAPSession::vSetNavigationFocus(tenAAPNavFocusType enNavFocusType)
{
   /*lint -esym(40,nullptr) nullptr is not declared */
   ETG_TRACE_USR1(("spi_tclAAPSession::vSetNavigationFocus entered"));
   ETG_TRACE_USR2(("[DESC]:Setting navigation focus - enNavFocusType = %d",
         ETG_ENUM(NAV_FOCUS_TYPE, enNavFocusType)));

   if ((m_spoGalReceiver != nullptr)&& (false == m_bShutdownInProgress))
   {
      m_spoGalReceiver->setNavigationFocus(enNavFocusType);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::vSetAudioFocus()
 ***************************************************************************/
t_Void spi_tclAAPSession::vSetAudioFocus(tenAAPDeviceAudioFocusState enDevAudFocusState, bool bUnsolicited)
{
   ETG_TRACE_USR1(("spi_tclAAPSession::vSetAudioFocus entered"));
   ETG_TRACE_USR2(("[DESC]:Setting audio focus - enDevAudFocusState %d, bUnsolicited %d",
         ETG_ENUM(DEVICE_AUDIOFOCUS, enDevAudFocusState), ETG_ENUM(BOOL, bUnsolicited)));
   /*lint -esym(40,nullptr) nullptr is not declared */
   
   if ((m_spoGalReceiver != nullptr) && (false == m_bShutdownInProgress))
   {
      m_spoGalReceiver->setAudioFocus(
            static_cast<AudioFocusStateType>(enDevAudFocusState),
            bUnsolicited);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclAAPSession::vStopSessionTimer()
 ***************************************************************************/
t_Void spi_tclAAPSession::vStopSessionTimer()
{
   ETG_TRACE_USR1(("spi_tclAAPSession::vStopSessionTimer entered"));
   Timer* poAAPSessionTimer = Timer::getInstance();
   if((NULL != poAAPSessionTimer) && (0 != m_oSessionTimerId))
   {
      ETG_TRACE_USR1(("spi_tclAAPSession::vStopSessionTimer Canceling timer %p", m_oSessionTimerId));
      //! Stop the timer if session message is received
      poAAPSessionTimer->CancelTimer(m_oSessionTimerId);
      m_oSessionTimerId = 0;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAAPSession::bSessionTimerCb
 ***************************************************************************/
t_Bool spi_tclAAPSession::bSessionTimerCb(timer_t rTimerID, t_Void *pvObject,
         const t_Void *pvUserData)
{
   ETG_TRACE_USR1(("spi_tclAAPSession::bSessionTimerCb entered rTimerID = %p", rTimerID));
   ETG_TRACE_USR1(("spi_tclAAPSession::bSessionTimerCb :Android Auto Session timer expired:"
         " No Session message received from phone"));
   SPI_INTENTIONALLY_UNUSED(pvObject);
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   //! Send session status as inactive if no message is received from phone(timer expiry)
   AAPSessionStatusMsg oSessionStatus;
   spi_tclAAPMsgQInterface *poMsgQinterface = spi_tclAAPMsgQInterface::getInstance();
   oSessionStatus.m_enSessionstatus = e8_SESSION_ERROR;
   oSessionStatus.m_bSessionTimedOut = true;
   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oSessionStatus, sizeof(oSessionStatus));
   }//if (NULL != poMsgQinterface)

   return true;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclAAPSession::vSendPingRequest
***************************************************************************/
t_Void spi_tclAAPSession::vSendPingRequest()
{
   ETG_TRACE_USR1((" spi_tclAAPSession::vSendPingRequest entered"));
   if(m_spoGalReceiver != nullptr)
   {
      struct timespec start;
      if (clock_gettime(CLOCK_REALTIME, &start) != -1)
      {
         m_spoGalReceiver->sendPingRequest(start.tv_sec, false);
      }
      else
      {
         ETG_TRACE_ERR(("[ERR]:vSendPingRequest::generateTimeStamp"
               "Failed with error:%d ", errno));
      }
   }
}
/***************************************************************************
 ** FUNCTION:  spi_tclAAPSession::vSetSessionConfiguration(t_U32 u32SessionConfiguration)
 ***************************************************************************/

t_Void spi_tclAAPSession::vSetSessionConfiguration(t_U32 u32SessionConfiguration)
{
   ETG_TRACE_USR1(("spi_tclAAPSession::vSetSessionConfiguration : Session Configuration = %d", u32SessionConfiguration));
   m_u32SessionConfiguration = u32SessionConfiguration;
}
//lint -restore
