//  *****************************************************************************
//  * FILE:         DataTypes.h
//  * SW_COMPONENT: VD-Sensor
//  * DESCRIPTION:  defines datatypes used in messages
//  * AUTHOR:       CM-DI/ESA1-Fischer
//  * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
//  * HISTORY:
//  * 26.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
//  *          Initial Revision.
//  * 05.07.02 Rev. 1.0 CM-DI/ESA1-Fischer
//  *          added new types (vds_tclGyroData, ...).
// * 02.06.09  Rev. 1.1 RBEI/ECF1 - Jeryn Mathew
// *           Added new #define MACROs for vds_trOdometerDefSetData
//  *****************************************************************************

#ifndef DataTypes_h
#define DataTypes_h 1

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define VDS_C_U8_GPS_MAX_NUM_OF_CHANNELS            ((tU8)OSAL_C_S32_GPS_NO_CHANNELS)

// defines for use-tyre-value (used in vds_tclCalcOdometerCalibration..)
#define VDS_C_U8_CALCODOMCALIB_USETYRETYP           (tU8)0x01
#define VDS_C_U8_CALCODOMCALIB_USETYRECIRCUM        (tU8)0x02
#define VDS_C_U8_CALCODOMCALIB_RESET                (tU8)0x03

// defines for u8OdometerStatus in vds_trOdometerData
#define VDS_C_U16_GYROSTATE_CONNECTED_NORMAL               (tU16)0x0000 // gyro-value is "trustworthy"
#define VDS_C_U16_GYROSTATE_CONNECTED_DATA_INVALID     (tU16)0xFFFE // data is invalid, do not use!!!
#define VDS_C_U16_GYROSTATE_CONNECTED_INTERNALERROR   (tU16)0xFFFF  // consistence of gyro-informationen not guaranteed
                                                                                // (e.g. device-driver-buffer overflow)

// defines for u8OdometerStatus in vds_trOdometerData
#define VDS_C_U8_ODOMSTATE_CONNECTED_NORMAL         (tU8)0x00   // odometer-counter is "trustworthy"
#define VDS_C_U8_ODOMSTATE_HW_ODOMETER_CONNECTED   (tU8)0xEE    // Only used by Ford TPEX, if HW Odometer is connected
                                                               // instead of OdometerViaCan.
#define VDS_C_U8_ODOMSTATE_VEH_SPEED_BUT_NO_OVC     (tU8)0xF7   // data inconsistent, do not use!!!
#define VDS_C_U8_ODOMSTATE_NO_CALIBRATION   (tU8)0xF8   // data normal, do not calibatrate the odometer!!!
#define VDS_C_U8_ODOMSTATE_CONNECTED_DATA_INVALID   (tU8)0xF9   // data is invalid, do not use!!!
#define VDS_C_U8_ODOMSTATE_CONNECTED_INTERNALERROR  (tU8)0xFA   // consistence of odometer-informationen not guaranteed
                                                                               // (e.g. device-driver-buffer overflow)
#define VDS_C_U8_ODOMSTATE_CONNECTED_NOINFO         (tU8)0xFB   // odometer-information not available or incomplete
#define VDS_C_U8_ODOMSTATE_CONNECTED_NOMOVEMENT     (tU8)0xFC   // vehicle engine off (vehicle stands still)
#define VDS_C_U8_ODOMSTATE_CONNECTED_ERROR          (tU8)0xFD   // errorflags (from odometer-source via network) active
#define VDS_C_U8_ODOMSTATE_DISCONNECTED             (tU8)0xFE   // network not active (vehicle stands still)
#define VDS_C_U8_ODOMSTATE_UNKNOWN                  (tU8)0xFF   // vehicle standstill- or movin-status unknown

// defines for u8OdometerCountDirection in vds_trOdometerData
#define VDS_C_U8_ODOMCOUNT_DIR_UNKNOWN  (tU8)0x00
#define VDS_C_U8_ODOMCOUNT_DIR_FORWARD  (tU8)0x01
#define VDS_C_U8_ODOMCOUNT_DIR_REVERSE  (tU8)0x02

#define VDS_C_U8_ODOMETERRANGE_UNKNOWN          0
#define VDS_C_U8_ODOMETERRANGE_ONLYSPEC         1
#define VDS_C_U8_ODOMETERRANGE_MINLIMITVALID    2
#define VDS_C_U8_ODOMETERRANGE_MAXLIMITVALIS    3
#define VDS_C_U8_ODOMETERRANGE_MINMAXLIMITVALID 4

typedef struct
{
    tU32    u32Timestamp;               // sensor-tics (wrap around counter)
    tU16    u16OdometerCount;           // odometer puls counter value (wrap around counter)
    tU8     u8OdometerStatus;           // odometer-state, see defines VDS_C_U8_ODOMSTATE_...
    tU8     u8OdometerCountDirection;   // (also known as rfls-status), see VDS_C_U8_ODOMCOUNT_DIR_...
} vds_trOdometerData;                   // size 8 byte


// defines for DEFSET_GROUP_MAINTENANCE
#define DEFSET_GROUP_MAINTENANCE (tU8)0x0D

// defines for DEFSET_SRC_CODING
#define DEFSET_SRC_CODING        (tU8)0x05

typedef struct
{
    tU8     u8DefSetGroup;                    // Group ID: e.g. DEFSET_GROUP_MAINTENANCE.
    tU8     u8DefSetMode;                      // Mode: e.g. DEFSET_SRC_CODING.
    tBool   bDefSetStateSuccess;           // TRUE or FALSE
} vds_trOdometerDefSetData;                 // size 8 byte


// defines for gyro-manufacturer
#define VDS_C_U8_GYROMANUFACTURER_MATSUSHITA    (tU8)0
#define VDS_C_U8_GYROMANUFACTURER_BOSCH         (tU8)1
#define VDS_C_U8_GYROMANUFACTURER_AD            (tU8)2
#define VDS_C_U8_GYROMANUFACTURER_MURATA        (tU8)3
#define VDS_C_U8_GYROMANUFACTURER_AE            VDS_C_U8_GYROMANUFACTURER_BOSCH
#define VDS_C_U8_GYROMANUFACTURER_PANASONIC     (tU8)4
#define VDS_C_U8_GYROMANUFACTURER_Axx100AQ      (tU8)5

#define VDS_C_U8_GYROMANUFACTURER_INVALID    (tU8)255


// Defines for OSAL clock events:
#define VDS_C_U8_OSAL_CLOCK_NOT_SET           (tU8)0  // Clock was not touched
#define VDS_C_U8_OSAL_CLOCK_SET_TO_DEFAULT    (tU8)1  // Clock set to default: 1.1.2000 00:00:00
#define VDS_C_U8_OSAL_CLOCK_SET_TO_USER_INPUT (tU8)2  // Clock set to user entry on 1.2.1970
#define VDS_C_U8_OSAL_CLOCK_ADJ_BY_USER_RTC   (tU8)3  // Clock set to user-adjusted RTC on 1.2.1970
#define VDS_C_U8_OSAL_CLOCK_ADJ_BY_USER       (tU8)4  // Clock set to user entry, valid date
#define VDS_C_U8_OSAL_CLOCK_ADJ_BY_GPS_RTC    (tU8)5  // Clock set to GPS-adjusted RTC, valid date
#define VDS_C_U8_OSAL_CLOCK_ADJ_BY_GPS        (tU8)6  // Clock set to RTC, valid date


// defines for gyro-type
// (note: theoretically, we could start with 0 for each gyro manufacturer here, but for safety
// we use unique numbers for each gyro type)
// for Matsushita:
#define VDS_C_U8_GYROTYPE_MATSUSHITA_EWTS4  (tU8)0
#define VDS_C_U8_GYROTYPE_MATSUSHITA_EWTZGJ    (tU8)3
// for BOSCH:
#define VDS_C_U8_GYROTYPE_BOSCH_SMG045      (tU8)1
#define VDS_C_U8_GYROTYPE_BOSCH_SMG045_SELFTEST (tU8)2
#define VDS_C_U8_GYROTYPE_AE_SMG045     VDS_C_U8_GYROTYPE_BOSCH_SMG045
#define VDS_C_U8_GYROTYPE_AE_SMG045_SELFTEST    VDS_C_U8_GYROTYPE_BOSCH_SMG045_SELFTEST
// for Panasonic:
#define VDS_C_U8_GYROTYPE_PANASONIC_EWTZGJ  (tU8)5

//for Gen2: Axx100AQ
#define VDS_C_U8_GYROTYPE_Axx100AQ (tU8)6

#define VDS_C_U8_GYROTYPE_INVALID            (tU8)255


typedef struct
{
    tU32    u32Timestamp;               // sensor-tics (wrap around counter)
    tU32    u32GyroValue;               // adc-value (range depends on used ad-converter bit-range)
    tU16    u16GyroStatus;              // tbd
    tU8 u8FillByte1;                // for 32bit-alignment
    tU8 u8FillByte2;                // for 32bit-alignment
} vds_trGyroData;                       // size 12 byte

// defines for u8ChannelStatus in vds_trGpsBigVarData
#define VDS_C_U8_GPS_CHANNELSTATUS_NOTUSED      (tU8)0x00
#define VDS_C_U8_GPS_CHANNELSTATUS_SEARCHING    (tU8)0x01
#define VDS_C_U8_GPS_CHANNELSTATUS_TRACKING     (tU8)0x02
#define VDS_C_U8_GPS_CHANNELSTATUS_EPHEMERIS    (tU8)0x03
#define VDS_C_U8_GPS_CHANNELSTATUS_USEDFORFIX   (tU8)0x04

typedef struct
{
    tU8     u8SatId;                    // 0..32
    tU8     u8ChannelStatus;            // 0..255, see VDS_C_U8_GPS_STATUS_...
    tU16    u16ExtendedStatus;          // 0..65535 (receiver depend, additional info)
    tF32    f32Azimuth;                 // 0..2PI rad
    tF32    f32Elevation;               // -PI/2..+PI/2 rad
    tF32    f32CarrierToNoiseRatio;     // -20..+80dBHz
} vds_trGpsBigVarData;                  // size 16 byte

// defines for u8Manufacturer in vds_trGpsBigConstData
#define VDS_C_U8_GPS_MANUFACTURER_UNKNOWN   (tU8)0x00   // unknown
#define VDS_C_U8_GPS_MANUFACTURER_BP        (tU8)0x01   // Bosch-Blaupunkt
#define VDS_C_U8_GPS_MANUFACTURER_JRC       (tU8)0x02   // JRC
#define VDS_C_U8_GPS_MANUFACTURER_ST        (tU8)0x03   //ST TESEO

// defines for u8HardwareVersion in vds_trGpsBigConstData
#define VDS_C_U8_GPS_HW_VERSION_BP_GPS1_4    (tU8)0x00   // if manufacturer BP: GPS1-4
#define VDS_C_U8_GPS_HW_VERSION_BP_GPS5      (tU8)0x01   // if manufacturer BP: GPS5
#define VDS_C_U8_GPS_HW_VERSION_BP_GPS6_7    (tU8)0x02   // if manufacturer BP: GPS6-7
#define VDS_C_U8_GPS_HW_VERSION_BP_ELENA     (tU8)0x0A   // if manufacturer BP: Elena/Tuareg
#define VDS_C_U8_GPS_HW_VERSION_BP_ARION     (tU8)0x0B   // if manufacturer BP: Arion
#define VDS_C_U8_GPS_HW_VERSION_BP_DRAGON    (tU8)0x0D   // if manufacturer BP: Dragon
#define VDS_C_U8_GPS_HW_VERSION_NMEA         (tU8)0x0E   // standard NMEA GPS receiver
#define VDS_C_U8_GPS_HW_VERSION_ST_TESEO_II  (tU8)0x10   // ST Teseo II (STA8088) GPS receiver
#define VDS_C_U8_GPS_HW_VERSION_UNKNOWN      (tU8)0xFF   // Manufacturer Unknown


// defines for u8FixStatus-bitfield in vds_trGpsBigConstData
#define VDS_C_U8_GPS_FIXSTATUS_BIT_TIMEOK           (tU8)0x01   // bit set: time ok
#define VDS_C_U8_GPS_FIXSTATUS_BIT_2DFIX            (tU8)0x02   // bit set: 2D-Fix
#define VDS_C_U8_GPS_FIXSTATUS_BIT_3DFIX            (tU8)0x04   // bit set: 3D-Fix
#define VDS_C_U8_GPS_FIXSTATUS_BIT_COLDSTARTFLAG    (tU8)0x80   // bit set: coldstart-flag

// define for u8AntennaStatus in vds_trGpsBigConstData
#define VDS_C_U8_GPS_ANTENNA_NORMAL     (tU8)0x00
#define VDS_C_U8_GPS_ANTENNA_OPEN       (tU8)0x01
#define VDS_C_U8_GPS_ANTENNA_SHORT      (tU8)0x02
#define VDS_C_U8_GPS_ANTENNA_BURNING        (tU8)0x03

// Defines for u32ValidFlags in GpsExtBigConstData
#define VDS_C_U32_GPS_VALID_POSITION                (tU32)0x00000001
#define VDS_C_U32_GPS_VALID_FILTERED_POSITION       (tU32)0x00000002
#define VDS_C_U32_GPS_VALID_VELOCITY                (tU32)0x00000004
#define VDS_C_U32_GPS_VALID_FILTERED_VELOCITY       (tU32)0x00000008
#define VDS_C_U32_GPS_VALID_POSITION_COVARIANCE     (tU32)0x00000010
#define VDS_C_U32_GPS_VALID_POSITION_RANGE_RESIDUAL (tU32)0x00000020
#define VDS_C_U32_GPS_VALID_VELOCITY_COVARIANCE     (tU32)0x00000040
#define VDS_C_U32_GPS_VALID_VELOCITY_RANGE_RESIDUAL (tU32)0x00000080
#define VDS_C_U32_GPS_VALID_EPHEMERIS_CHECKED       (tU32)0x00000100
#define VDS_C_U32_GPS_VALID_ORIGINAL_TIME_FIX       (tU32)0x00000200
#define VDS_C_U32_GPS_VALID_ORIGINAL_2D_FIX         (tU32)0x00000400
#define VDS_C_U32_GPS_VALID_ORIGINAL_3D_FIX         (tU32)0x00000800
#define VDS_C_U32_GPS_VALID_RAIM_ALARM_POSITION     (tU32)0x40000000
#define VDS_C_U32_GPS_VALID_RAIM_ALARM_VELOCITY     (tU32)0x80000000

// Defines for Odometer usage
#define VDS_C_U8_AUTOMATIC_ODOMETER_USAGE  (tU8)0x00
#define VDS_C_U8_USE_ODOMETER              (tU8)0x01
#define VDS_C_U8_DONT_USE_ODOMETER         (tU8)0x02
#define VDS_C_U8_INVALID_ODOMETER_USAGE    (tU8)0xFF


#ifdef _PACK_FOR_ALIGNMENT_
#pragma pack(push,saveVal_vds_trGpsBigConstData)         /* __packed_for_8byte_alignment__ */
#pragma pack(1)
#endif /* _PACK_FOR_ALIGNMENT_ */

#ifndef COMPILER_STRUCT_PREFIX
   #define COMPILER_STRUCT_PREFIX
#endif

#ifndef COMPILER_STRUCT_POSTFIX
   #define COMPILER_STRUCT_POSTFIX
#endif

COMPILER_STRUCT_PREFIX
struct  vds_stGpsBigConstData
{
    tU16    u16Version;                     // 0 (reserved)
    tU16    u16Year;                        // 0..(e.g.)2002..65535
    tU8     u8Month;                        // 1..12
    tU8     u8Day;                          // 1..31
    tU8     u8Hour;                         // 0..23
    tU8     u8Minute;                       // 0..59
    tU8     u8Second;                       // 0..59
    tU8     u8HundredthSecond;              // 0..99
    tU8     u8Manufacturer;                 // 0..255, see VDS_C_U8_GPS_MANU...
    tU8     u8HardwareVersion;              // 0..255, see VDS_C_U8_GPS_HW_VERSION
    tU32    u32SoftwareVersion;             // 0..4294967295, see document
    tF64    f64Latitude;                    // -PI/2..+PI/2 rad
    tF64    f64Longitude;                   // -PI..+PI rad
    tF32    f32AltitudeWGS84;               // -1000..+18000 m (0==WGS84 ellipsoid)
    tF32    f32AltitudeAboveSealevel;       // -1000..+18000 m (0==approx. Sealevel)
    tF32    f32Speed;                       // 0..515 m/s
    tF32    f32Heading;                     // 0..2PI rad
    tF32    f32PDOP;                        // 1..3.4e38 (only 2D-Fix: == HDOP)
    tF32    f32HDOP;                        // 1..3.4e38 (for 3D-Fix)
    tF32    f32EstAccuracyPlane;            // (0)..3.4e38 m (0==not available)
    tF32    f32EstAccuracyHeight;           // (0)..3.4e38 m (0==not available)
    tU8     u8SatellitesHealthy;            // 0..32
    tU8     u8SatellitesVisible;            // 0..32
    tU8     u8SatellitesReceived;           // 0..32
    tU8     u8SatellitesUsed;               // 0..32
    tU32    u32ReceivedAlmanac;             // 0..429467295 (bitfield: sat-id 1..32)
    tU32    u32ReceivedEphemeris;           // 0..429467295 (bitfield: sat-id 1..32)
    tU32    u32UpdatesSameConstellation;    // 0..429467295 valid upd
    tU8     u8FixStatus;                    // 0..255 (bitfield), see VDS_C_U8_GPS_FIXSTATUS_...
    tU8     u8AntennaStatus;                // 0..255, see VDS_C_U8_GPS_ANTENNA_...
    tU8     u8UpdateFrequency;              // 1.255 Hz
    tU8     u8NumberOfChannels;             // 0..255
    tF32    f32EstAccuracySpeed;            // (0)..515 m/s (0==not available)
    tF32    f32EstAccuracyCourse;           // (0)..2PI rad (0==not available)
}
COMPILER_STRUCT_POSTFIX;


typedef vds_stGpsBigConstData vds_trGpsBigConstData;            // size 92 byte        /*Aenderung nv 18.01.05POSTFIX */


COMPILER_STRUCT_PREFIX
struct  stGpsCovMatrix
{
   tF32 f32Elem0;
   tF32 f32Elem4;
   tF32 f32Elem5;
   tF32 f32Elem8;
   tF32 f32Elem9;
   tF32 f32Elem10;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsCovMatrix tGpsCovMatrix;

COMPILER_STRUCT_PREFIX
struct  vds_stGpsExtBigConstData
{
   tU16          u16Version;    // 0 (reserved)
   tU16          u16Year;   // 0..(e.g.)2002..65535
   tU8           u8Month;   // 1..12
   tU8           u8Day;     // 1..31
   tU8           u8Hour;    // 0..23
   tU8           u8Minute;  // 0..59
   tU8           u8Second;  // 0..59
   tU8           u8HundredthSecond; // 0..99
   tU8           u8Manufacturer; // 0..255, see VDS_C_U8_GPS_MANU...
   tU8           u8HardwareVersion; // 0..255, see VDS_C_U8_GPS_HW_VERSION
   tU32          u32SoftwareVersion; // 0..4294967295, see document
   tF64          f64Latitude;   // -PI/2..+PI/2 rad
   tF64          f64Longitude;  // -PI..+PI rad
   tF32          f32AltitudeWGS84; // -1000..+18000 m (0==WGS84 ellipsoid)
   tF64          f64LatitudeFiltered;
   tF64          f64LongitudeFiltered;
   tF32          f32AltitudeWGS84Filtered;
   tF32          f32GeoidalSeparation;
   tF32          f32VelocityNorth;
   tF32          f32VelocityEast;
   tF32          f32VelocityUp;
   tF32          f32VelocityNorthFiltered;
   tF32          f32VelocityEastFiltered;
   tF32          f32VelocityUpFiltered;
   tGpsCovMatrix rPositionCovarianceMatrix;
   tGpsCovMatrix rVelocityCovarianceMatrix;
   tF32          f32PositionRangeResMax;
   tF32          f32VelocityRangeResMax;
   tU32          u32ValidFlags;
   tF32          f32PDOP;   // 1..3.4e38 (only 2D-Fix: == HDOP)
   tF32          f32HDOP;   // 1..3.4e38 (for 3D-Fix)
   tU8           u8SatellitesHealthy; // 0..32
   tU8           u8SatellitesVisible; // 0..32
   tU8           u8SatellitesReceived; // 0..32
   tU8           u8SatellitesUsed; // 0..32
   tU32          u32ReceivedAlmanac; // 0..429467295 (bitfield: sat-id 1..32)
   tU32          u32ReceivedEphemeris; // 0..429467295 (bitfield: sat-id 1..32)
   tU32          u32UpdatesSameConstellation; // 0..429467295 valid upd
   tU8           u8FixStatus;   // 0..255 (bitfield), see VDS_C_U8_GPS_FIXSTATUS_...
   tU8           u8AntennaStatus; // 0..255, see VDS_C_U8_GPS_ANTENNA_...
   tU8           u8UpdateFrequency; // 1.255 Hz
   tU8           u8NumberOfChannels; // 0..255
}
COMPILER_STRUCT_POSTFIX;

typedef vds_stGpsExtBigConstData vds_trGpsExtBigConstData;

#ifdef _PACK_FOR_ALIGNMENT_
#pragma pack(pop,saveVal_vds_trGpsBigConstData)
#endif /* _PACK_FOR_ALIGNMENT_ */

/* Comment this out and see where errors occur */
typedef struct
{
    tU32                    u32Timestamp;
    vds_trGpsBigConstData   rGpsBigConst;
    vds_trGpsBigVarData     arGpsBigVar[VDS_C_U8_GPS_MAX_NUM_OF_CHANNELS];
}   vds_trGpsBigData;                       // size 96 + u8NumberOfChannels * 16 byte
                                            // for 12-channel-receiver: 288 byte

typedef struct
{
    tU32                     u32Timestamp;
    vds_trGpsExtBigConstData rGpsExtBigConst;
    vds_trGpsBigVarData      arGpsBigVar[VDS_C_U8_GPS_MAX_NUM_OF_CHANNELS];
}   vds_trGpsExtBigData;


/* -- data types for GpsHints method ------------------------------------ */

COMPILER_STRUCT_PREFIX
struct  stGpsVectorLLA
{
   tF64 f64Latitude;
   tF64 f64Longitude;
   tF64 f64Altitude;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsVectorLLA tGpsVectorLLA;

COMPILER_STRUCT_PREFIX
struct  stGpsVectorNEU
{
   tF32 f32North;
   tF32 f32East;
   tF32 f32Up;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsVectorNEU tGpsVectorNEU;

COMPILER_STRUCT_PREFIX
struct  stGpsCovarianceMatrixDiag
{
   tF32 f32Elem11;
   tF32 f32Elem22;
   tF32 f32Elem33;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsCovarianceMatrixDiag tGpsCovarianceMatrixDiag;

COMPILER_STRUCT_PREFIX
struct  stGpsHints
{
   tGpsVectorLLA            rForwardPosition;
   tGpsVectorLLA            rBackwardPosition;
   tGpsVectorNEU            rForwardVelocity;
   tGpsVectorNEU            rBackwardVelocity;
   tGpsCovarianceMatrixDiag rPositionCovariances;
   tGpsCovarianceMatrixDiag rVelocityCovariances;
   tU32                     u32MapMatchingHints;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsHints tGpsHints;

COMPILER_STRUCT_PREFIX
struct  stGpsExtHints
{
   tGpsVectorLLA            rForwardPosition;
   tGpsVectorLLA            rBackwardPosition;
   tGpsVectorNEU            rForwardVelocity;
   tGpsVectorNEU            rBackwardVelocity;
   tGpsCovarianceMatrixDiag rForwardPositionCovariances;
   tGpsCovarianceMatrixDiag rBackwardPositionCovariances;
   tGpsCovarianceMatrixDiag rForwardVelocityCovariances;
   tGpsCovarianceMatrixDiag rBackwardVelocityCovariances;
   tU32                     u32MapMatchingHints;
   tU16                     u16Year;
   tU8                      u8Month;
   tU8                      u8Day;
   tU8                      u8Hour;
   tU8                      u8Minute;
   tU8                      u8Second;
   tU8                      u8HundredthSecond;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsExtHints tGpsExtHints;

/* ---------------------------------------------------------------------- */


COMPILER_STRUCT_PREFIX
struct  stGpsNormalModeTestResult
{
   tU32 u32TimeStamp;
   tU32 u32PinToggle;
   tS32 s32OscillatorFrequencyOffset;
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsNormalModeTestResult vds_trGpsNormalModeTestResult;

/* ---------------------------------------------------------------------- */


COMPILER_STRUCT_PREFIX
struct  stGpsNoiselevelTimeValidityData
{
   tF32 f32NoiseLevel;          /* calculated noise level */
   tU32 u32TimePosValid;        /* bit mask for time and position validity */
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsNoiselevelTimeValidityData vds_trGpsNoiselevelTimeValidityData;


/* ---------------------------------------------------------------------- */

COMPILER_STRUCT_PREFIX
struct  stGpsOscillatorData
{
   tF32 f32OscFreq;             /* oscillator frequency */
   tF32 f32OscFreqOffset;       /* oscillator frequency offset */
   tF32 f32CalcFreqOffset;      /* calculated frequency offset */
   tF32 f32IFOffset;            /* IF offset */
}
COMPILER_STRUCT_POSTFIX;

typedef struct stGpsOscillatorData vds_trGpsOscillatorData;

/* ---------------------------------------------------------------------- */


// define for u8GpsProtocol in vds_trGpsCommProtocol
#define VDS_C_U8_GPS_PROTOCOL_UNSET                 (tU8)0x00
#define VDS_C_U8_GPS_PROTOCOL_NMEA                  (tU8)0x01
#define VDS_C_U8_GPS_PROTOCOL_UTSIP                 (tU8)0x02
#define VDS_C_U8_GPS_PROTOCOL_JRC                   (tU8)0x03
#define VDS_C_U8_GPS_PROTOCOL_BIG                   (tU8)0x04
#define VDS_C_U8_GPS_PROTOCOL_HOSTBASEDSOLUTION     (tU8)0x05
#define VDS_C_U8_GPS_PROTOCOL_COULDNT_DETECTED      (tU8)0xFF

// define for u8CommBaudrate in vds_trGpsCommProtocol
#define VDS_C_U8_COMM_BAUDRATE_4800     (tU8)0x00
#define VDS_C_U8_COMM_BAUDRATE_9600     (tU8)0x01
#define VDS_C_U8_COMM_BAUDRATE_19200    (tU8)0x02
#define VDS_C_U8_COMM_BAUDRATE_38400    (tU8)0x03
#define VDS_C_U8_COMM_BAUDRATE_57600    (tU8)0x04
#define VDS_C_U8_COMM_BAUDRATE_115200   (tU8)0x05

// define for u8CommParity in vds_trGpsCommProtocol
#define VDS_C_U8_COMM_PARITY_NONE       (tU8)0x00
#define VDS_C_U8_COMM_PARITY_ODD        (tU8)0x01
#define VDS_C_U8_COMM_PARITY_EVEN       (tU8)0x02

// define for u8CommStopBits in vds_trGpsCommProtocol
#define VDS_C_U8_COMM_STOPBITS_1        (tU8)0x00
#define VDS_C_U8_COMM_STOPBITS_2        (tU8)0x01

typedef struct
{
    tU8     u8GpsProtocol;                  // see VDS_C_U8_GPS_PROTOCOL_...
    tU8     u8CommBaudrate;                 // see VDS_C_U8_COMM_BAUDRATE...
    tU8     u8CommParity;                   // see VDS_C_U8_COMM_PARITY...
    tU8     u8CommStopbits;                 // see VDS_C_U8_COMM_STOPBITS...
} vds_trGpsCommProtListElem; // internal message-format

typedef struct
{
    tU8     u8GpsProtocol;                  // see VDS_C_U8_GPS_PROTOCOL_...
    tU8     u8CommBaudrate;                 // see VDS_C_U8_COMM_BAUDRATE...
    tU8     u8CommParity;                   // see VDS_C_U8_COMM_PARITY...
    tU8     u8CommStopbits;                 // see VDS_C_U8_COMM_STOPBITS...
    tU8     u8GpsChannels;
    tU8     u8FillByte1;
    tU8     u8FillByte2;
    tU8     u8FillByte3;
} vds_trGpsCommProtocol;                    // size 8 byte

#define VDS_C_U8_TYRECONDITION_UNDEFINED (tU8)0x00
#define VDS_C_U8_TYRECONDITION_NEWTYRE   (tU8)0x01  // more than 5 mm tread depth
#define VDS_C_U8_TYRECONDITION_USEDTYRE  (tU8)0x02  // less tham 5 mm tread depth

// Structures for Testmode3

typedef struct {
    tU32        ulClkOffset;         // Estimated Clock Offset
    tU8         ucSvId;              // SV number of satellite used for test
    tU8         ucNoPasses;          // Number of passes for test
} tSetTestmode;

typedef struct {
     tU32       ulClockOffset;       // Clock Offset [Hz]
     tU32       ulClockDrift;        // Clock Drift [Hz]
     tU32       ulTimeToBitSync;     // Time to first Bit sync [s]
     tU32       ulIsignal;           // Averaged I-channel
     tU32       ulQsignal;           // Averaged Q-channel
     tU16       usPoorStatusCount;
     tU16       usBadBitCount;       // Wrong Bits od of 500
     tU16       usCnoMean;           // Mean Value of CNo during test [10*dB]
     tU16       usCnoSigma;          // Std.Deviation of CNo during test [10*dB]
     tU16       usCnoEffective;      // Effective Value of CNo during test [10*dB]
     tU8        ucSvId;              // SV number of satellite used for test
} tTestmodeData;


// Structures for extended testmode

#define VDS_C_U8_CHANNEL_SNR_SIZE       (tU8)13

typedef struct {
        tS32 s32CurrentClkOffset;
    tU32 u32CurrentClkValue;
      tU32 u32GpsDebugChanStatusA;
      tU32 u32GpsDebugChanStatusB;
      tU32 u32GpsDebugChanStatusC;
      tU32 u32GpsDebugChanStatusD;
    tU8 u8CalcTestResult;
    tU8 u8TestResult;
    tS16 s16GpsAverageChannelSNR;
      tS16 s16GpsActualChannelPower;
      tS16 as16GpsActualChannelSNR[VDS_C_U8_CHANNEL_SNR_SIZE];
} vds_trExtTestModeData;


/* Structure for XO compensation */

typedef struct {
    tU32        ulTow;               // Time of week of last parameter change
    tF32        fltTemperature;      // Measured Temperature [K]
    tF32        fltC0;               // 1. Coefficient of polynomial [ppm]
    tF32        fltC1;               // 2. Coefficient of polynomial [ppm/K]
    tF32        fltC2;               // 3. Coefficient of polynomial [ppm/K/K]
    tF32        fltC3;               // 4. Coefficient of polynomial [ppm/K/K/K]
    tU16        usWeek;              // GPS week of last parameter change
} tXoCompensationData;

/* -- data types for GPS front end -------------------------------------- */

#define VDS_C_U8_GPS_DATA_PATTERN_NONE 0
#define VDS_C_U8_GPS_DATA_PATTERN_0000 4
#define VDS_C_U8_GPS_DATA_PATTERN_0101 5
#define VDS_C_U8_GPS_DATA_PATTERN_1010 6
#define VDS_C_U8_GPS_DATA_PATTERN_1111 7

#define VDS_C_U8_GPS_FRONT_END_OFF   0
#define VDS_C_U8_GPS_FRONT_END_ON    1
#define VDS_C_U8_GPS_FRONT_END_ERROR 2

/* -- definitions for gyro error status --------------------------------- */

#define VDS_C_U8_GYRO_ERROR_UNKNOWN  0
#define VDS_C_U8_GYRO_ERROR_ACTIVE   1
#define VDS_C_U8_GYRO_ERROR_INACTIVE 2

/* -- definitions for GPSTCDATA -- */

typedef struct
{
   /* ECEF satellite position */
   tF64 f64SatPosX;
   tF64 f64SatPosY;
   tF64 f64SatPosZ;

   /* ECEF satellite velocity */
   tF64 f64SatVelX;
   tF64 f64SatVelY;
   tF64 f64SatVelZ;

   /* ECEF satellite acceleration */
   tF64 f64SatAccX;
   tF64 f64SatAccY;
   tF64 f64SatAccZ;

   /* corrected data transmission time */
   tF64 f64TimeSentCorr;

   /* uncorrected data transmission time */
   tF64 f64TimeSentUncorr;

   /* carrier phase difference */
   tF64 f64CarrierPhaseDiff;

   /* pseudo range */
   tF64 f64PseudoRange;

   /* delta pseudo range */
   tF64 f64DeltaPseudoRange;

   /* variance of pseudo range measurements */
   tF32 f32VarPseudoRange;

   /* variance of delta pseudo range measurements */
   tF32 f32VarDeltaPseudoRange;

   /* iono/tropo correction of propagation time */
   tF32 f32PropTimeCorrection;

   /* usable for calculation */
   tU32 u32ChannelStatus;
} vds_trGpsTightCouplingChannelData;

typedef struct
{
   /* /dev/auxclock time stamp */
   tU32 u32TimeStamp;

   /* hardware sample rate */
   tU32 u32SampleRate;

   /* GPS time at interrupt */
   tF64 f64TimeOfWeek;

   /* current oscillator frequency */
   tF64 f64OscFreq;

   /* default frequency mix factor */
   tF32 f32FreqMixFactor;

   /* calculation interval */
   tU32 u32CalcInterval;
} vds_trGpsTightCouplingCommonData;

typedef struct
{
   /* channel-independent data */
   vds_trGpsTightCouplingCommonData rCommon;

   /* per-channel data */
   vds_trGpsTightCouplingChannelData arChannel[OSAL_C_S32_GPS_NO_CHANNELS];
} vds_trGpsTightCouplingData;

/* -- insert generated code here ---------------------------------------- */

/*!
 * \brief Estimator state for fake odometer
 */
typedef struct
{
   tF32  f32AuxclockFrequency;   // Frequency of AuxClock [Hz]
   tU32  u32T1;                  // Start time of extrapolation interval [ticks]
   tF32  f32V1;                  // Start speed of extrapolation interval [m/s]
   tU32  u32T2est;               // Estimated stop time of extrapolation interval [ticks]
   tF32  f32A12est;              // Estimated average acceleration of extrapolation interval [m/s/s]
   tF32  f32V2est;               // Estimated stop speed of extrapolation interval [m/s]
   tF32  f32dS12est;             // Estimated distance of extrapolation interval [m]
   tF32  f32K12est;              // Estimated correction value for travelled distance [1]
   tF32  f32deltaSpeedCorrection;      // Speed correction for next extrapolation interval [m/s]
   tBool bOdometerFakeInitialized;
} vds_trOdometerFakeState;

/* -- definitions for ABS data ------------------------------------------ */

typedef struct
{
   tU32 u32TimeStamp;

   tU16 u16CounterFrontLeft;
   tU16 u16CounterFrontRight;
   tU16 u16CounterRearLeft;
   tU16 u16CounterRearRight;

   tU8  u8StatusFrontLeft;
   tU8  u8StatusFrontRight;
   tU8  u8StatusRearLeft;
   tU8  u8StatusRearRight;

   tU8  u8DirectionFrontLeft;
   tU8  u8DirectionFrontRight;
   tU8  u8DirectionRearLeft;
   tU8  u8DirectionRearRight;
} vds_trAbsData;

/* -- definitions for steering data ------------------------------------- */

typedef struct
{
   tU32 u32TimeStamp;
   tS32 s32WheelAngle;
   tS32 s32WheelRate;
   tU8 u8StatusWheelAngle;
   tU8 u8StatusWheelRate;
} vds_trSteeringData;

/* -- definitions for VD Diaglog interface ------------------------------ */

#define VDS_C_U8_DIAGLOG_PASSED               0
#define VDS_C_U8_DIAGLOG_FAILED               1
#define VDS_C_U8_DIAGLOG_NO_RESULT            2
#define VDS_C_U8_DIAGLOG_PASSED_DIAGNOSTIC    3
#define VDS_C_U8_DIAGLOG_FAILED_DIAGNOSTIC    4
#define VDS_C_U8_DIAGLOG_NO_RESULT_DIAGNOSTIC 5

struct vds_stDiagLogTestResult
{
   tU16 u16TroubleCode;
   tU8  u8Result;
};

typedef struct vds_stDiagLogTestResult vds_trDiagLogTestResult;

#define VDS_C_U8_DIAGLOG_REPLY_ACCEPTED 0
#define VDS_C_U8_DIAGLOG_REPLY_REJECTED 1


/* -- Definitions for SPM interface ------------------------------------- */

#define VDS_C_U32_CVM_CRITICAL_LOW_VOLTAGE_START  0
#define VDS_C_U32_CVM_CRITICAL_LOW_VOLTAGE_END    1
#define VDS_C_U32_CVM_LOW_VOLTAGE_START           2
#define VDS_C_U32_CVM_LOW_VOLTAGE_END             3
#define VDS_C_U32_CVM_HIGH_VOLTAGE_START          4
#define VDS_C_U32_CVM_HIGH_VOLTAGE_END            5
#define VDS_C_U32_CVM_CRITICAL_HIGH_VOLTAGE_START 6
#define VDS_C_U32_CVM_CRITICAL_HIGH_VOLTAGE_END   7
#define VDS_C_U32_CVM_NO_EVENT                    15

typedef amt_tclArrayN<vds_trGpsCommProtListElem> vds_tclGpsCommProtList;
typedef amt_tclMappedIntegralType<vds_trOdometerData> vds_tclOdometerData;
typedef amt_tclArrayN<vds_trOdometerData> vds_tclOdometerList;
typedef amt_tclMappedIntegralType<vds_trGpsBigConstData> vds_tclGpsBigConstData;
typedef amt_tclMappedIntegralType<vds_trGpsExtBigConstData> vds_tclGpsExtBigConstData;
typedef amt_tclMappedIntegralType<vds_trGpsBigVarData> vds_tclGpsBigVarData;
typedef amt_tclArrayN<vds_trGpsBigVarData> vds_tclGpsBigVarList;
typedef amt_tclMappedIntegralType<tSetTestmode> vds_tclGpsSetData;
typedef amt_tclMappedIntegralType<tTestmodeData> vds_tclGpsTestmodeData;
typedef amt_tclMappedIntegralType<tU16> vds_tclSetOdometerManualCalibration;
typedef amt_tclMappedIntegralType<tU16> vds_tclOdometerManualCalibrationData;
typedef amt_tclMappedIntegralType<tXoCompensationData> vds_tclGpsXoCompensationData;
typedef amt_tclMappedIntegralType<tGpsHints> vds_tclGpsHints;
typedef amt_tclMappedIntegralType<tGpsExtHints> vds_tclGpsExtHints;
typedef amt_tclMappedIntegralType<vds_trGpsNormalModeTestResult> vds_tclGpsNormalModeTestResult;
typedef amt_tclMappedIntegralType<vds_trGpsTightCouplingCommonData> vds_tclGpsTightCouplingCommonData;
typedef amt_tclArrayN<vds_trGpsTightCouplingChannelData> vds_tclGpsTightCouplingChannelDataList;
typedef amt_tclMappedIntegralType<tU8> vds_tclU8;
typedef amt_tclMappedIntegralType<tU16> vds_tclU16;
typedef amt_tclMappedIntegralType<tS16> vds_tclS16;
typedef amt_tclMappedIntegralType<tU32> vds_tclU32;
typedef amt_tclMappedIntegralType<tS32> vds_tclS32;
typedef amt_tclArrayN<tS16> vds_tclS16List;

/* -- insert generated amt classes here ---------------------------------------- */

// now defined in vds_basemessages!!!
//typedef amt_tclMappedIntegralType<vds_trGyroData> vds_tclGyroData;
//typedef amt_tclArrayN<vds_trGyroDataListElem> vds_tclGyroList;
/* typedef amt_tclMappedIntegralType<vds_trDiagLogTestResult> vds_tclDiagLogTestResult; */
/* typedef amt_tclArrayN<vds_trDiagLogTestResult> vds_tclDiagLogTestResultList; */

#endif
