/************************************************************************
* FILE:         vds_trace.cpp
* PROJECT:      VASCO
* SW-COMPONENT: VDSensor
*----------------------------------------------------------------------
* DESCRIPTION:  handle traces
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2002 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      |   Modification                         | Author
* 20.09.02  |   Initial revision                     | CM-DI/ESA1-Fischer
* 28.08.09  | New Trace function vTraceMsgReadableVal| RBEI/ECF1-Sainath K.
*           | has been added                         |
* 06.01.17  |   Port to Gen4                     | RBEI/ECF12 - Arun Magi(amg6kor)
*************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#define VDS_S_IMPORT_INTERFACE_SENSOR_DATATYPES
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/* Used to get KDS and errmem definitions */
#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#include "system_pif.h"

//-----------------------------------------------------------------------------
// Trace Device
//-----------------------------------------------------------------------------
static OSAL_tIODescriptor oTrace = OSAL_ERROR;

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Opens the Trace Device
//!
//! \return
//!   Returns the status regarding the Opening of Trace Device\n
//!   Success code in case of success is:
//!   - \c  TRUE : Success\n
//!   Error code in case of failure are:\n
//!   - \c FALSE : Failed
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool bOpenTraceDevice
(
   //!  (O): pointer to store the Trace Device handle
   OSAL_tIODescriptor *pTrace
)
{
   tBool bTraceDeviceOpened = TRUE;
   oTrace = OSAL_IOCreate( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );

   if ( oTrace == OSAL_ERROR )
   {
      /* impossible to create trace device -> already opened? */
      oTrace = OSAL_IOOpen( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );

      if ( oTrace == OSAL_ERROR )
      {
         /* no change -> no trace any more */
         bTraceDeviceOpened = FALSE;
      }
      else
      {
         *pTrace = oTrace;
      }
   }
   else
   {
      *pTrace = oTrace;
   }

   return bTraceDeviceOpened;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Close the Trace Device
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vCloseTraceDevice
(
   //!  (I): pointer containing Trace Device handle
   OSAL_tIODescriptor *pTrace
)
{
   if ( OSAL_ERROR != oTrace )
   {
      OSAL_s32IOClose( oTrace );
      oTrace = OSAL_ERROR;
      *pTrace = oTrace;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Write trace message to trace device
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vTraceMsg
(
   //!  (I): Trace Level
   TR_tenTraceLevel enTraceLevel,
   //!  (I): Trace Message and formatting parameters
   const tChar *pcFormatString, ...
)
{
   if ( OSAL_ERROR != oTrace )
   {
      tBool bActive = FALSE;
      OSAL_trIOCtrlActivTrace rActive;
      static tChar acBuffer[256];
      va_list argList;  /*lint -e530*//* Variable is initialized with macro va_start, to be compiler independent*/
      rActive.enTraceClass = TR_CLASS_VD_SENSOR;
      rActive.enTraceLevel = enTraceLevel;
      rActive.bIsActive = FALSE;

      if ( OSAL_OK == OSAL_s32IOControl( oTrace,
                                         OSAL_C_S32_IOCTRL_ISACTIVE,
                                         ( tLong )&rActive ) )
      {
         bActive = rActive.bIsActive;
      }

      if ( bActive )
      {
         OSAL_VarArgStart( argList, pcFormatString );
         OSAL_s32VarNPrintFormat( acBuffer, 256, pcFormatString, argList );
         OSAL_VarArgEnd( argList );
         tU32 u32NumOfBytes = (tU32) OSAL_u32StringLength( acBuffer );

         OSALUTIL_s32TraceWrite
         (
            oTrace,
            (tU32)enTraceLevel,
            (tU32)TR_CLASS_VD_SENSOR,
            ( tPCS8 )acBuffer,
            u32NumOfBytes
         );

         if( enTraceLevel == VDS_C_TRACELEVEL_FATAL )
         {
            vWriteToErrorMemory( acBuffer, u32NumOfBytes );
         }
      }
   }
}

// ***************** F U N C T I O N  H E A D E R ***********************
//
//  DESCRIPTION:
//
//! \brief
//!   Write message to Error Memory device
//!
//! \return
//!   - \c TRUE  : Success\n
//!   - \c FALSE : Failure
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ---------------------------------------------------------------------
//**********************************************************************
/**************************************************************************//**
  @CAUTION:::: Don't add FATAL trace in this function since it 
               causes stack over flow because of recurssion.
 ******************************************************************************/
tVoid vWriteToErrorMemory
(
   //Buffer holding data
   const tChar *pcBuffer,
   //Size of data
   tU32 u32NumOfBytes
)
{
   OSAL_tIODescriptor fd_errmem;
   trErrmemEntry  rErrmemEntry = {0};

   rErrmemEntry.u16Entry = (tU16)TR_CLASS_VD_SENSOR;
   rErrmemEntry.eEntryType = eErrmemEntryFatal;
   rErrmemEntry.u16EntryLength = (tU16)( u32NumOfBytes + sizeof("VDS_EM:") );

   OSAL_pvMemoryCopy( rErrmemEntry.au8EntryData ,"VDS_EM:",sizeof("VDS_EM:") );
   (void)OSAL_szStringNConcat( rErrmemEntry.au8EntryData,
                               pcBuffer,
                               ERRMEM_MAX_ENTRY_LENGTH - sizeof("VDS_EM:") );

   fd_errmem = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ERRMEM, OSAL_EN_WRITEONLY );

   if ( fd_errmem != OSAL_ERROR )
   {
      (void)OSAL_s32IOWrite( fd_errmem, (tPCS8)&rErrmemEntry, sizeof( rErrmemEntry ) );
      OSAL_s32IOClose( fd_errmem );
   }
}

//***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Sanity check for gyro data.
//!
//!  If the sanity check fails, an error memory entry is made.
//!
//! \return
//!   None
//  HISTORY:
// Date         |  Author                | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void vGyroDataSanityCheck
(
   //!  (I): used to identify the call site.
   tS32 s32Pass,
   //!  (I): pointer to odometer data.
   const vds_trGyroData *prGyroData,
   //!  (I): number of data records pointed to by prOdoData.
   tU32 u32Elems
)
{
   for ( tU32 i = 0; i < u32Elems; ++i )
   {
      if ( ( prGyroData[i].u16GyroStatus != VDS_C_U16_GYROSTATE_CONNECTED_NORMAL ) &&
            ( prGyroData[i].u16GyroStatus != VDS_C_U16_GYROSTATE_CONNECTED_INTERNALERROR ) &&
            ( prGyroData[i].u16GyroStatus != VDS_C_U16_GYROSTATE_CONNECTED_DATA_INVALID ) )
      {
         et_vErrmemStringNormal( TR_COMP_SENSOR, "InvGyro: p=%d, ts=%d, val=%d, st=%04x",
                                 s32Pass,
                                 prGyroData[i].u32Timestamp,
                                 prGyroData[i].u32GyroValue,
                                 prGyroData[i].u16GyroStatus );
      }
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Sanity check for odometer data.
//!
//!  If the sanity check fails, an error memory entry is made.
//!
//! \return
//!   None
//  HISTORY:
// Date         |  Author                | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void vOdoDataSanityCheck
(
   //!  (I): used to identify the call site.
   tS32 s32Pass,
   //!  (I): pointer to odometer data.
   const vds_trOdometerData *prOdoData,
   //!  (I): number of data records pointed to by prOdoData.
   tU32 u32Elems
)
{
   for ( tU32 i = 0; i < u32Elems; ++i )
   {
      if ( ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_NORMAL ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_HW_ODOMETER_CONNECTED ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_VEH_SPEED_BUT_NO_OVC ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_NO_CALIBRATION ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_DATA_INVALID ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_INTERNALERROR ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_NOINFO ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_NOMOVEMENT ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_CONNECTED_ERROR ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_DISCONNECTED ) &&
            ( prOdoData[i].u8OdometerStatus != VDS_C_U8_ODOMSTATE_UNKNOWN ) )
      {
         et_vErrmemStringNormal( TR_COMP_SENSOR, "InvOdo: p=%d, ts=%d, val=%d, st=%04x",
                                 s32Pass,
                                 prOdoData[i].u32Timestamp,
                                 prOdoData[i].u16OdometerCount,
                                 prOdoData[i].u8OdometerStatus );
      }
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  This Function frames a TTFis buffer based on the sensor type.\n
//!  This buffer contains the names of active sensor devices.
//!
//! \return
//!   None
//  HISTORY:
// Date         |  Author                | MODIFICATION
// ----------------------------------------------------------------------------
// 28.08.2009   | sak9kor (RBEI/ECF1)    | XML generated FI has been adapted
//                                         instead manual generated FI
//******************************************************************************
tVoid vTraceMsgReadableVal
(
   //!  (I): Trace Level
   TR_tenTraceLevel enTraceLevel,
   //!  (I): Trace Message
   const tChar *pcFormatString,
   //!  (I): Length of Trace Message
   tU32 u32NumofBytes
)
{
   if ( OSAL_ERROR != oTrace )
   {
      tBool bActive = FALSE;
      OSAL_trIOCtrlActivTrace rActive;
      rActive.enTraceClass = TR_CLASS_VD_SENSOR;
      rActive.enTraceLevel = enTraceLevel;
      rActive.bIsActive = FALSE;

      if ( OSAL_OK == OSAL_s32IOControl( oTrace,
                                         OSAL_C_S32_IOCTRL_ISACTIVE,
                                         ( tLong )&rActive ) )
      {
         bActive = rActive.bIsActive;
      }

      if ( bActive )
      {
         //OSAL_pvMemoryCopy(acBuffer,pcFormatString,u32NumofBytes);
         OSALUTIL_s32TraceWrite
         (
            oTrace,
            (tU32)enTraceLevel,
            (tU32)TR_CLASS_VD_SENSOR,
            ( tPCS8 )pcFormatString,
            u32NumofBytes
         );
      }
   }
}
