/************************************************************************
* FILE:         vds_tcltracehandler.cpp
* PROJECT:
* SW-COMPONENT: VDSensor
*----------------------------------------------------------------------
* DESCRIPTION:  handle traces
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2002 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      |   Modification                         | Author
* 20.09.02  |   Initial revision                     | CM-DI/ESA1-Fischer
* 30.09.09  |   Added doxygen headers for all        | RBEI/ECF1-Sainath K
*           |   classes/funcs                        |
* 06.01.17  |   port to Gen4                     | RBEI/ECF12 Arun Magi(amg6kor)
*************************************************************************/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#define VDS_S_IMPORT_INTERFACE_SENSOR_DATATYPES
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/* Used to get KDS and errmem definitions */
#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#include "system_pif.h"

#include "vds_tcltracehandler.h"

#define MAKE_KDSENTRYWORD  ((tU16) KDS_TYPE_ODOM_CHANNEL| ((tU16) ((tU8)KDS_TARGET_ODOMETER) << 8))
#define MAKE_KDSENTRYWORD_GPS_ANTENNA  ((tU16) KDS_TYPE_GPS_ANTENNA| ((tU16) ((tU8)KDS_TARGET_GPS) << 8))

/*!
 * \brief KDS byte value for odometer via cable
 */
#define ODOMETER_C_U8_ODO_VIA_CABLE 1

/*!
 * \brief KDS byte value for odometer via CAN
 */
#define ODOMETER_C_U8_ODO_VIA_CAN 2

/*!
 * \brief KDS byte value for odometer via cable and direction via CAN
 */
#define ODOMETER_C_U8_ODO_VIA_CABLE_PLUS_DIR_VIA_CAN 3

/*!
 * \brief KDS byte value for odometer via CAN in Opel Vectra
 */
#define ODOMETER_C_U8_ODO_VIA_CAN_VECTRA 4

tVoid Conf_odo2kds( const tUChar*data );
tVoid vds_conf_GpsAntennaCoeff2Kds( const tUChar *data );


vds_tclTraceHandler *vds_tclTraceHandler::poTraceHandler = OSAL_NULL;

//-----------------------------------------------------------------------------
// Tracehandler
//-----------------------------------------------------------------------------
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Constructor.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
vds_tclTraceHandler:: vds_tclTraceHandler
(
   //!  pointer to GPS Parameter message
   tclMsgGpsIf *poMsgGpsIfParam
) :
      //!  Trace Device Handle
      inithTrace( OSAL_ERROR ),
      //!  Gps Handle
      hGps( OSAL_ERROR ),
      //!  Odometer Handle
      hOdo( OSAL_ERROR ),
      //!  Gyro Handle
      hGyro( OSAL_ERROR ),
      //!  ABS handle
      hAbs( OSAL_ERROR ),
      //!  Steering Wheel Handle
      hSteer( OSAL_ERROR )
{
   poTraceHandler = this;
   this->poMsgGpsIf = poMsgGpsIfParam;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Handles the incomming message from TTFis. Depending on the device for
//!   which the instruction (from the incoming mesage) is meant for,
//!   appropriate calls are made to the device and trace messages are also sent
//!   to TTFis.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vds_tclTraceHandler::vChannelMgr
(
   //!  pointer to incoming message
   const tUChar* puchData
)
{
   tU8 u8MsgCode = puchData[1]; //byte 1 -> start of transmitted data
   const tUChar* PureMessageData;
   tS32 s32Retval = OSAL_ERROR;
   PureMessageData = ( puchData + 2 );

   switch ( u8MsgCode )
   {
      case VDS_EN_TRACE_HANDLER_SINK_VD_SENSOR:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_VD_SENSOR" );
         vds_conf_GpsAntennaCoeff2Kds( PureMessageData + 1 );
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_GPS:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_GPS" );
         s32Retval = poMsgGpsIf->bExecuteGpsTraceCommand( PureMessageData );

         if ( OSAL_ERROR != s32Retval )
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "command for VDS_EN_TRACE_HANDLER_SINK_GPS done" );
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "can't perform command for VDS_EN_TRACE_HANDLER_SINK_GPS : 0x%x",
                       OSAL_u32ErrorCode( ) );
         }
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_ODO:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_VD_ODO" );
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_GYRO:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_GYRO" );
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_ABS:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_ABS" );
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_STEER:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_STEER" );
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_KDS:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_KDS" );
         Conf_odo2kds( PureMessageData );
      }
      break;
      case VDS_EN_TRACE_HANDLER_SINK_INTERNAL:
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "Received trace command for VDS_EN_TRACE_HANDLER_SINK_INTERNAL" );
         ET_TRACE_ERROR_BIN( ( int ) TR_CLASS_VD_SENSOR, ET_EN_T8 _ 5 _ ET_EN_T8 _ 1 _ ET_EN_DONE );
      }
      break;
      default:
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "no valid messagecode for tracehandler: u8MsgCode = %d",
                    u8MsgCode );
         break;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Wrapper function to the vChannelMgr method.  Passes the argument to the
//!   method only if the Trace handler is not NULL.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vds_tclTraceHandler::vChannelMgrWrapper
(
   //!  Pointer to the Data for vChannelMgr
   const tUChar* puchData
)
{
   if ( poTraceHandler != OSAL_NULL )
      poTraceHandler->vChannelMgr( puchData );
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Registers to the TTFis Device
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vds_tclTraceHandler::vRegisterToTtfis
(
   //!  Trace device handle
   OSAL_tIODescriptor hTrace
)
{
   tS32  s32Error;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;
   oTraceChannel.enTraceChannel = TR_TTFIS_SENSOR;
   oTraceChannel.pCallback = ( OSAL_tpfCallback )vChannelMgrWrapper;
   s32Error = OSAL_s32IOControl ( hTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, ( tLong ) & oTraceChannel );

   if ( OSAL_OK != s32Error )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Unable to register tracehandler: 0x%x",
                 OSAL_u32ErrorCode( ) );
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Unregisters from TTFis Device
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vds_tclTraceHandler::vUnregisterToTtfis
(
   //!  Trace device handle
   OSAL_tIODescriptor hTrace
)
{
   tS32 s32Error;
   OSAL_trIOCtrlLaunchChannel oTraceChannel;
   oTraceChannel.enTraceChannel = TR_TTFIS_SENSOR;
   oTraceChannel.pCallback = ( OSAL_tpfCallback )vChannelMgrWrapper;
   s32Error = OSAL_s32IOControl ( hTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, ( tLong ) & oTraceChannel );

   if ( OSAL_OK != s32Error )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Unable to unregister tracehandler: 0x%x",
                 OSAL_u32ErrorCode( ) );
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Writes configuration data for the evaluation of the Odometer Data into KDS.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid Conf_odo2kds
(
   //!  Pointer to the Odometer Data
   const tUChar *data
)
{
   /* commented out because of warnings */
   /*   tU32 u32Result = 0; */
   OSAL_tIODescriptor hDevice;
   /* commented out because of warnings */
   /*   tsKDS_Info rKDSInfo; */
   tsKDSEntry rEntryInfo;
   tsKDSEntry rCheckEntry;
   tU16 u16KDSEnableWrite = 1;

   if ( ( data[0] == ODOMETER_C_U8_ODO_VIA_CABLE )
         || ( data[0] == ODOMETER_C_U8_ODO_VIA_CAN )
         || ( data[0] == ODOMETER_C_U8_ODO_VIA_CAN_VECTRA )
         || ( data[0] == ODOMETER_C_U8_ODO_VIA_CABLE_PLUS_DIR_VIA_CAN ) )
   {
      vTraceMsg( VDS_C_TRACELEVEL_DATA,
                 "VDS_EN_TRACE_HANDLER_SINK_KDS: Excute Conf_odo2kds: Source %d", data[0] );
      hDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE );

      if ( hDevice == OSAL_ERROR )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "Failed to open KDS - error: 0x%x",
                    OSAL_u32ErrorCode( ) );
         return;
      }

      if ( OSAL_ERROR == OSAL_s32IOControl( hDevice, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE,
                                            ( tLong )u16KDSEnableWrite ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "Failed to enable KDS writing - error: 0x%x",
                    OSAL_u32ErrorCode( ) );
         OSAL_s32IOClose( hDevice );
         return;
      }

      rEntryInfo.u16Entry = MAKE_KDSENTRYWORD;
      rEntryInfo.u16EntryLength = 1;
      rEntryInfo.au8EntryData[0] = data[0];
      rEntryInfo.u16EntryFlags = 0;

      if ( OSAL_ERROR == OSAL_s32IOWrite( hDevice, ( tS8 * )&rEntryInfo,
                                          ( tS32 )sizeof( rEntryInfo ) ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "Failed to write to KDS - error: 0x%x",
                    OSAL_u32ErrorCode( ) );
      }

      if ( OSAL_ERROR == OSAL_s32IOControl( hDevice, ( OSAL_C_S32_IOCTRL_KDS_WRITE_BACK ),
                                            TRUE ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "Failed to directly store KDS changes for choice of Odometer source - error: 0x%x",
                    OSAL_u32ErrorCode( ) );
         OSAL_s32IOClose( hDevice );
         return;
      }

      /* Check: if no error occured the result from reading should
         be the data written before */
      rCheckEntry.u16Entry = ( ( tU16 ) KDS_TYPE_ODOM_CHANNEL | ( ( tU16 ) ( ( tU8 )KDS_TARGET_ODOMETER ) << 8 ) );
      rCheckEntry.u16EntryFlags = 0;//no flag
      rCheckEntry.u16EntryLength = 1;

      if ( OSAL_s32IORead ( hDevice, ( tS8 * )&rCheckEntry,
                            ( tS32 )sizeof( rEntryInfo ) ) != OSAL_ERROR )
      {
         if ( rCheckEntry.au8EntryData[0] != rEntryInfo.au8EntryData[0] )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "Failed to write to KDS - error: 0x%x",
                       OSAL_u32ErrorCode( ) );
         }
         else if ( rCheckEntry.au8EntryData[0] == ODOMETER_C_U8_ODO_VIA_CABLE )
         {
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Wrote ODOMETER_C_U8_ODO_VIA_CABLE to KDS" );
         }
         else if ( rCheckEntry.au8EntryData[0] == ODOMETER_C_U8_ODO_VIA_CAN )
         {
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Wrote ODOMETER_C_U8_ODO_VIA_CAN to KDS" );
         }
         else if ( rCheckEntry.au8EntryData[0] == ODOMETER_C_U8_ODO_VIA_CABLE_PLUS_DIR_VIA_CAN )
         {
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Wrote ODOMETER_C_U8_ODO_VIA_CABLE_PLUS_DIR_VIA_CAN to KDS" );
         }
         else if ( rCheckEntry.au8EntryData[0] == ODOMETER_C_U8_ODO_VIA_CAN_VECTRA )
         {
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                       "Wrote ODOMETER_C_U8_ODO_VIA_CAN_VECTRA to KDS" );
         }
      }
      OSAL_s32IOClose( hDevice );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Wrong input paramter for ODO2KDS" );
   }
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Writes configuration data for the evaluation of the GPS antenna current in KDS
//!   All values are raw ADC values for the limits.
//!   a = minimum allowable voltage (default 2792)
//!   b = maximum allowable voltage (default 3413)
//!   c = minimum allowable current (default 410)
//!   d = maximum allowable current (default 2668)
//!   e = ADC value corresponding to 0 mA current (default 320)
//!   f = ADC value corresponding to 50 mA current (default 2358)
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid vds_conf_GpsAntennaCoeff2Kds
(
   //!  pointer to GPS Data
   const tUChar *data
)
{
   OSAL_tIODescriptor hDevice;
   tsKDSEntry rEntryInfo;
   tU16 u16KDSEnableWrite = 1;
   hDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE );

   if ( hDevice == OSAL_ERROR )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Failed to open KDS - error: 0x%x",
                 OSAL_u32ErrorCode( ) );
      return;
   }

   if ( OSAL_ERROR == OSAL_s32IOControl( hDevice, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE,
                                         ( tLong )u16KDSEnableWrite ) )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Failed to enable KDS writing - error: 0x%x",
                 OSAL_u32ErrorCode( ) );
      OSAL_s32IOClose( hDevice );
      return;
   }

   rEntryInfo.u16Entry = MAKE_KDSENTRYWORD_GPS_ANTENNA;
   rEntryInfo.u16EntryLength = 24;
   OSAL_pvMemoryCopy( rEntryInfo.au8EntryData, data, 24 );
   rEntryInfo.u16EntryFlags = 0;

   if ( OSAL_ERROR == OSAL_s32IOWrite( hDevice, ( tS8 * )&rEntryInfo,
                                       ( tS32 )sizeof( rEntryInfo ) ) )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Failed to write to KDS - error: 0x%x",
                 OSAL_u32ErrorCode( ) );
   }

   if ( OSAL_ERROR == OSAL_s32IOControl( hDevice, ( OSAL_C_S32_IOCTRL_KDS_WRITE_BACK ),
                                         TRUE ) )
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "Failed to directly store KDS changes for GpsAntennaCoeff - error: 0x%x",
                 OSAL_u32ErrorCode( ) );
      OSAL_s32IOClose( hDevice );
      return;
   }

   OSAL_s32IOClose( hDevice );
}
