/*!
 * \file vds_tclRingBuffer.h
 */

/*
  Invariants: the buffer always stores the elements
  [max(0, u32NumElems -  u32BufferSize) .. u32NumElems - 1]
*/


/*!
 * \brief A general ring buffer for sensor data.
 *
 * The functions of this class are not thread safe! It is the
 * responsibility of the user to ensure that this class is only
 * accessed from within a critical section, or from only one thread.
 */
template<class T>
class vds_tclRingBuffer
{
   private:
      T* parDataBuffer;
      tU32 u32BufferSize;
      tU32 u32NumElems;

   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default Constructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclRingBuffer() :
            parDataBuffer( OSAL_NULL ),
            u32BufferSize( 0 ),
            u32NumElems( 0 )
      {
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default Destructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      ~vds_tclRingBuffer()
      {
         vDeallocate();
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Deallocate the buffer.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vDeallocate()
      {
         if ( (parDataBuffer != OSAL_NULL)
              &&
              (0 < u32BufferSize) )
         {
            delete [] parDataBuffer;
            parDataBuffer = OSAL_NULL;
         }

         u32BufferSize = 0;
         u32NumElems = 0;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Allocate memory for the buffer
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      void vAllocate
      (
         //! (I) number of elements that the buffer will hold.
         tU32 u32RequestedSize
      )
      {
         vDeallocate();
         parDataBuffer = new T[u32RequestedSize];

         if ( parDataBuffer != OSAL_NULL )
         {
            u32BufferSize = u32RequestedSize;
            u32NumElems = 0;
         }
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Is the buffer correctly set up?
      //!
      //! \return
      //!   true  if the buffer was set up correctly;\n
      //!   false if there was a memory allocation problem.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tBool bIsValid()
      {
         return u32BufferSize != 0;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Add one element to the buffer.
      //!
      //! \return
      //!   the new maximum element index for the buffer.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tU32 u32AddOne
      (
         //! (I) new element to be added
         T rData
      )
      {
         if ( !bIsValid() )
            return 0;

         parDataBuffer[( u32NumElems++ )%u32BufferSize] = rData;
         return u32NumElems;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Add elements to the buffer.
      //!
      //! \return
      //!   the new maximum element index for the buffer.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tU32 u32AddList
      (
         //!  (I) an array of new elements
         const T* parData,
         //!  (I) number of elements in parData
         tU32 u32Elems
      )
      {
         tU32 u32Index = 0;

         if ( !bIsValid() || ( parData == OSAL_NULL ) )
            return u32NumElems;

         if ( u32Elems > u32BufferSize )
         {
            u32Index = u32Elems - u32BufferSize;
         }

         for ( ; u32Index < u32Elems; ++u32Index )
         {
            u32AddOne( parData[u32Index] );
         }

         return u32NumElems;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Get the maximum element index for the buffer.
      //!
      //!   The maximum element index works like the size of a C array, that
      //!   is, the buffer contains elements [0..buf.u32GetNumElems()-1] (in
      //!   the sense that it is legal for clients to ask for these
      //!   elements; the older elements may of course no longer be
      //!   available.)
      //!
      //! \return
      //!   The Total Elements in the list
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tU32 u32GetNumElems()
      {
         return u32NumElems;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Get number of available elements starting from u32MinElem
      //!
      //! \return
      //!   Returns the available elements present.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tU32 u32GetElemsFrom( tU32 u32MinElem )
      {
         tU32 u32Available = u32NumElems - u32MinElem;
         return u32Available > u32BufferSize ? u32BufferSize : u32Available;
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Get buffer contents.
      //!
      //! \return
      //!   the number of elements read
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tU32 u32GetList
      (
         //! (O) pointer to an array of data elements
         T* parData,
         //! (I) the maximum number of elements to read
         tU32 u32Elems,
         //! (I) the element number of the first element to
         //! read. If this element is no longer present in the buffer,
         //! reading starts from the oldest available element. This parameter
         //! is updated to refer to the next element after the last one that
         //! was read by this operation (that is, the new value can be used
         //! on the next call to read elements sequentially.)
         tU32& rfu32NextElem
      )
      {
         if ( !bIsValid() ||
               ( parData == OSAL_NULL ) || ( rfu32NextElem >= u32NumElems ) )
            return 0;

         tU32 u32ElemsToRead = u32NumElems - rfu32NextElem;

         if ( u32ElemsToRead > u32BufferSize )
         {
            u32ElemsToRead = u32BufferSize;
            rfu32NextElem = u32NumElems - u32BufferSize;
         }

         if ( u32Elems < u32ElemsToRead  )
         {
            u32ElemsToRead = u32Elems;
         }

         for ( tU32 u32Index = 0; u32Index < u32ElemsToRead; ++u32Index )
         {
            parData[u32Index] =
               parDataBuffer[( rfu32NextElem++ )%u32BufferSize];
         }

         return u32ElemsToRead;
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Get the last element in the buffer.
      //!
      //! \return
      //!   true if data were copied; false if the buffer is empty or prData is 0.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tBool bGetLastElem
      (
         //!  (O) pointer to variable which will hold the data
         T* prData
      )
      {
         if ( !bIsValid() || u32NumElems == 0 )
         {
            return false;
         }

         if ( prData == OSAL_NULL )
         {
            return false;
         }

         *prData = parDataBuffer[( u32NumElems-1 )%u32BufferSize];
         return true;
      }
};
