//#include "osal_if_abs.h"
#include "vds_tclScratchBuffer.h"
#include "vds_tclHistoryProperty.h"

class vds_tclMsgAbsIf;
class vds_tclAbsSynchListener;

//!   \brief  ABS Property Class, derived from History Property class.
class vds_tclAbsProperty : public vds_tclHistoryProperty<vds_trAbsData>
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Default Constructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclAbsProperty
      (
         //! (I) Function ID
         tU16 u16FnctId,
         //! (I) reference to the critical section to be used
         //! to protect the ring buffer.  The critical section must have
         //! constructed, but it is not necessary for it to have been
         //! initialized yet.
         vds_tclCriticalSection& rfoCSParam,
         //! (I) Reference to ABS IF parameter
         vds_tclMsgAbsIf *poMsgAbsIfParam
      ) : vds_tclHistoryProperty<vds_trAbsData>( u16FnctId, rfoCSParam ),
            poMsgAbsIf( poMsgAbsIfParam )
      {
      }

   protected:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Perform actions when data are added to the ring buffer.
      //!
      //!   Derived classes, such as this one, may use this function to implement
      //!   sensor data synchronisation actions, such as triggering a synch event
      //!   or checking for a synch event time out.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vDataAdded( )
      {
         /* vTraceMsg( VDS_C_TRACELEVEL_DATA, */
         /*            "AbsProperty: data added, elems = %d", */
         /*            oRingBuffer.u32GetNumElems() ); */
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Mark data to signal that a buffer overflow occured.
      //!
      //!   The usual way to do this is to set an INTERNAL_ERROR status flag
      //!   or similar on the first data element.
      //!   When this function is called, it is guaranteed that the first
      //!   data element of prData is the oldest remaining element in the
      //!   ring buffer.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vMarkBufferOverflow
      (
         //! (I) Ring Buffer with ABS Data
         vds_trAbsData* prData,
         //! (I) Number of Elements
         tU32 u32Elems
      )
      {
         OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( prData );
         OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Elems );
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "AbsProperty: buffer overflow detected" );
      }


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Mark data as invalid.
      //!
      //!   The usual way to do this is to set an INVALID status flag on all
      //!   elements.
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual void vMarkAsInvalid
      (
         //! Unused Variable
         vds_trAbsData* prData,
         //! Unused Variable
         tU32 u32Elems
      )
      {
         OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( prData );
         OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32Elems );
      }

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Create a CCA status message of the proper type.
      //!
      //!    The message contents should be filled with prData.  The target
      //!    application ID should be set to u16TargetAppId.  The sender
      //!    application ID should be set to CCA_C_U16_APP_SENSOR.
      //!
      //! \return
      //!   Pointer to an object containing the CCA Status Message
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual amt_tclServiceData* poCreateMessage
      (
         //!  (I) ABS Data
         vds_trAbsData* prData,
         //!  (I) Number of Elements
         tU32 u32Elems,
         //!  (I) Target APP ID
         tU16 u16TargetAppId
      );

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    May CCA messages be sent?
      //!
      //!    This function may e.g. look at the internal state of an
      //!    associated message interface class.
      //!
      //! \return
      //!   True if permission granted, else False.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      virtual tBool bMaySend();

   private:
      vds_tclMsgAbsIf *poMsgAbsIf;
};

//!  \brief  ABS Interface class. Handles the common sensor-CCA interfaces.
class vds_tclMsgAbsIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Default Constructor
      //
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      vds_tclMsgAbsIf();
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Default destructor
      //
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      ~vds_tclMsgAbsIf();

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //     Initializes members of this class
      //
      //   return
      //    Incase of Success
      //    - VDS_E_NOERROR - Success.
      //    Incase of Failure
      //    - VDS_E_ALLOC_ERROR - Error in allocating the Ring Buffer.
      //    - VDS_E_INVALID_PARAMETER - Error because the Function pointer is NULL
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32Init
      (
         tS32( *pfs32SendSubscriberMessageParam )
         ( amt_tclServiceData *, tclSubscriberManager * )
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Start the thread for ABS Interface class
      //
      //   return
      //    Success code in case of success is:
      //    - VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //    - VDS_E_GYROIF_THREADSTART_ERROR : ABS Interface Start Error
      //    - VDS_E_GYROIF_NOT_INITIALIZED   : ABS Interface object uninitialized
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32StartThread();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Stops the thread for ABS Interface class
      //
      //   return
      //    Success code in case of success is:
      //   - VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //    - VDS_E_GYROIF_THREADSTOP_ERROR : ABS Interface Thread Stop Error
      //    - VDS_E_GYROIF_NOT_INITIALIZED   : ABS Interface object uninitialized
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32StopThread();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Stops the thread for ABS IF class and deletes the associated
      //    resources
      //
      //   return
      //    Succes code in case of success is:
      //   - VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //    - VDS_E_GYROIF_THREADSTOP_ERROR : ABS IF Thread Stop Error
      //    - VDS_E_GYROIF_NOT_INITIALIZED  : ABS IF Thread object uninitialized
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //*****************************************************************************
      tS32 s32DeleteThread();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Create a CCA message and fill it with ABS data from the ring buffer.
      //
      //   return
      //  The number of data elements in the CCA message (a
      //  positive value or zero) or an error code (a negative value).
      //     0 : Success. No Error.\n
      //     VDS_E_GYROIF_ALLOC_ERROR : Error in Allocation of Data.\n
      //     VDS_E_GYROIF_NOT_INITIALIZED : Uninitialized pointer.\n
      //     VDS_E_INVALID_PARAMETER : Invalid Parameter.\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32GetAbsMessage (
         amt_tclServiceData **ppoServiceData,
         tU32 &rfu32NextElemToRead,
         tU16 u16TargetAppId,
         tBool bSendInitMessage,
         tBool bCreateInvalidMessage = FALSE );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Create a CCA message and fill it with ABS data from the ring buffer.
      //
      //
      //   return
      //  The number of data elements in the CCA message (a
      //  positive value or zero) or an error code (a negative value).
      //    - VDS_E_NO_ERROR : Success. No Error.\n
      //    - VDS_E_POST_MESSAGE_FAILED : Subscriber Message Posting failed.\n
      //    - VDS_E_INVALID_PARAMETER : Invalid Parameter.\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //****************************************************************************
      void vSynch();

      vds_tclCriticalSection oCS;
      vds_tclAbsProperty oAbsData;

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //   Gets the last element index.
      //   return
      //    The index of last element in the buffer.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //****************************************************************************	  
      tU32 u32GetLastAbsElemIndex(tVoid);

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //   Tells number of elements present in the buffer starting from the requested 
	  //   element.
      //   return
      //    Number of elements available in the buffer from "u32FromElem" element.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //****************************************************************************
      tU32 u32GetNumOfElmFrom(tU32 u32FromElem);

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Collects ABS records for "AllSensorData" property. 
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  VDS_E_NO_ERROR : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_GYROIF_NOT_INITIALIZED :if vdsensor is not initialized.
      //!   - \c VDS_E_GYROIF_ALLOC_ERROR :if failed to allocate memory.
      //!   - \c VDS_E_GYROIF_GETGYRODATA_FAILED: if unable to get data.
      //!   - \c VDS_E_GYROIF_DATA_UNAVAILABLE:if no ABS data is available. 
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32GetAbsRecords
      (
         sensor_fi_tcl_AllSensorData *pCmbFiAbsData,
   
          tU32 &u32NextelementToRead
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Changes/Updates the Internal State with that from the argument
      //
      //
      //   return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ---------------------------------------------------------------------------
      //****************************************************************************
      void vSetState( tS32 s32NewState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Gets the current Internal State
      //
      //
      //   return
      //    Returns current internal state
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ---------------------------------------------------------------------------
      //****************************************************************************
      tS32 s32GetState( );

   private:
      static vds_tclMsgAbsIf *poThisInstance;
      tS32 s32InternalState;
      tclSensorThread *poAbsThread;
      tS32( *pfs32SendSubscriberMessage )( amt_tclServiceData *, tclSubscriberManager * );
      vds_tclAbsSynchListener* poAbsSynchListener;

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //    Wrapper for the s32AddList() method, to hold the ABS Data
      //
      //   return
      //    Returns the actual number of elements added
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      static tS32 s32AddAbsDataWrapper( vds_trAbsData *prData, tU32 u32NData );
};
