// *****************************************************************************
// * FILE:         tclSensorThread.h
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-declaration: creates, activates, deletes
// *               Sensor-Threads
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// *****************************************************************************

#ifndef tclSensorThread_h
#define tclSensorThread_h 1

#define VDS_C_S32_THREAD_UNINITIALIZED 0
#define VDS_C_S32_THREAD_INITIALIZED   1
#define VDS_C_S32_THREAD_RUNNING       2
#define VDS_C_S32_THREAD_STOP        3
#define VDS_C_S32_THREAD_ERROR          -1

// error-values
#define VDS_E_THREAD_IOOPEN_ERROR    -1
#define VDS_E_THREAD_SEM_ERROR       -2
#define VDS_E_THREAD_CREATE_ERROR    -3
#define VDS_E_THREAD_ALREADY_INITIALIZED -4
#define VDS_E_THREAD_NOT_INITIALIZED     -5
#define VDS_E_THREAD_ACTIVATE_ERROR      -6
#define VDS_E_THREAD_NOT_RUNNING         -7
#define VDS_E_THREAD_ERROR_UNKNOWN       -8

//!  \brief  Sensor Thread Class. General thread class used for all sensors
class tclSensorThread
{

   public:
      tclSensorThread();

      virtual ~tclSensorThread();

      // *****************************************************************************
      // * METHOD:      tclSensorThread::s32ThreadStart
      // * DESCRIPTION: activates thread, which is created in s32ThreadInit.
      // *              function waits, 'til thread is started from OS
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_C_S32_THREAD_RUNNING (thread activated)
      // *                 error: VDS_E_THREAD_NOT_INITIALIZED,
      // *                        VDS_E_THREAD_ACTIVATE_ERROR
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32ThreadStart ();

      // *****************************************************************************
      // * METHOD:      tclSensorThread::s32ThreadStop
      // * DESCRIPTION: stops thread.
      // *              function waits, 'til thread stopped
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_C_S32_THREAD_INITIALIZED (thread stopped)
      // *                 error: VDS_E_THREAD_NOT_RUNNING
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32ThreadStop ();

      // *****************************************************************************
      // * METHOD:      tclSensorThread::vThreadClose
      // * DESCRIPTION: virtual method, inherited classes have to fill this function.
      // *              closes handles, deletes semaphores, deletes allocated memory
      // *
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tVoid
      // *
      // * HISTORY:
      // * 09.04.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      virtual tVoid vThreadClose ();

      // *****************************************************************************
      // * METHOD:      tclSensorThread::s32GetInternalState
      // * DESCRIPTION: returns internal thread-state
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tS32
      // *                 internal thread-state. possible values are:
      // *                 VDS_C_S32_THREAD_UNINITIALIZED,
      // *                 VDS_C_S32_THREAD_INITIALIZED,
      // *                 VDS_C_S32_THREAD_RUNNING,
      // *                 VDS_C_S32_THREAD_STOP
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32GetInternalState ();

      // *****************************************************************************
      // * METHOD:      tclSensorThread::u32GetSensorCycleTime
      // * DESCRIPTION: returns sensor-cycle-time
      // *
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tU32
      // *              sensor-cycle-time
      // *
      // * HISTORY:
      // * 23.10.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tU32 u32GetSensorCycleTime ();

      // Additional Public Declarations

   protected:

      // *****************************************************************************
      // * METHOD:      tclSensorThread::vSetInternalState
      // * DESCRIPTION: sets internal state
      // * PARAMETER:
      // *              tS32 s32InternalState: (I)
      // *                 new internal state. possible values are (not checked!):
      // *                 VDS_C_S32_THREAD_UNINITIALIZED,
      // *                 VDS_C_S32_THREAD_INITIALIZED,
      // *                 VDS_C_S32_THREAD_RUNNING,
      // *                 VDS_C_S32_THREAD_STOP
      // *
      // * RETURNVALUE: tVoid
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tVoid vSetInternalState ( tS32 s32InternalState );

      // *****************************************************************************
      // * METHOD:      tclSensorThread::bCritSectionBegin
      // * DESCRIPTION: locks access to critical sections.
      // *              only when TRUE is returned, access is allowed
      // *
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tBool
      // *                 TRUE : access allowed
      // *                 FALSE: error, access is not allowed
      // *
      // * HISTORY:
      // * 19.04.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tBool bCritSectionBegin ();

      // *****************************************************************************
      // * METHOD:      tclSensorThread::vCritSectionEnd
      // * DESCRIPTION: unlocks access to critical sections.
      // *
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tVoid
      // *
      // * HISTORY:
      // * 19.04.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tVoid vCritSectionEnd ();

      tU16 u16Frequency;
      OSAL_tIODescriptor SensorFd;
      OSAL_tSemHandle hSensorSemaphore;
      tCString coszSensorSemName;
      OSAL_tThreadID SensorThreadId;
      tU32 u32SensorCycleTime;

   private:
      tS32 s32InternalState;
};

#endif
