// *****************************************************************************
// * FILE:         tclSensorThread.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: creates, activates, deletes
// *               Sensor-Threads
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// *****************************************************************************

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SENSOR_THREADS
#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#include "vds_internal_if.h"

// Class tclSensorThread
// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default constructor
//!
//! \return
//!   None
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tclSensorThread::tclSensorThread()
      : u16Frequency(0), 
      SensorFd( OSAL_ERROR ),
      hSensorSemaphore( OSAL_C_INVALID_HANDLE ),
      coszSensorSemName(OSAL_NULL),
      SensorThreadId( OSAL_ERROR ),
      u32SensorCycleTime( 62500000 ),
      s32InternalState(VDS_C_S32_THREAD_UNINITIALIZED)
{
   vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default destructor
//!
//! \return
//!   None
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tclSensorThread::~tclSensorThread()
{
   // delete thread
   if ( VDS_C_S32_THREAD_RUNNING == s32GetInternalState() )
   {
      if ( VDS_E_NO_ERROR != s32ThreadStop() )
      {
         // lediglich Fehlerausgabe, da Instanz geloescht wird.
         tclSensorThread::vThreadClose();
      }
   }
   coszSensorSemName = OSAL_NULL;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   activates thread, which is created in s32ThreadInit.
//!   function waits, 'til thread is started from OS
//!
//! \return
//!   -  \c  ok   : VDS_C_S32_THREAD_RUNNING (thread activated)
//!   -  \c  error: VDS_E_THREAD_NOT_INITIALIZED,
//!   -  \c  VDS_E_THREAD_ACTIVATE_ERROR
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tS32 tclSensorThread::s32ThreadStart ()
{
   tS32 s32RetVal = VDS_E_THREAD_ERROR_UNKNOWN;

   if ( VDS_C_S32_THREAD_INITIALIZED == s32GetInternalState() )
   {
      // activate sensor-thread
      if ( OSAL_ERROR != OSAL_s32ThreadActivate( SensorThreadId ) )
      {
         tS32 s32TimeOutCounter = 50;  //50*100ms= 5s timeout

         while ( VDS_C_S32_THREAD_RUNNING != s32GetInternalState() && s32TimeOutCounter > 0 )
         {
            OSAL_s32ThreadWait( 100 );
            --s32TimeOutCounter;
         }

         s32RetVal = s32GetInternalState();
      }
      else
         s32RetVal = VDS_E_THREAD_ACTIVATE_ERROR;
   }
   else
      s32RetVal = VDS_E_THREAD_NOT_INITIALIZED;

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   stops thread.
//!   function waits, 'til thread stopped
//!
//! \return
//!   -  \c  ok   : VDS_C_S32_THREAD_INITIALIZED (thread stopped)
//!   -  \c  error: VDS_E_THREAD_NOT_RUNNING
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tS32 tclSensorThread::s32ThreadStop ()
{
   tS32 s32RetVal = VDS_E_THREAD_ERROR_UNKNOWN;

   if ( VDS_C_S32_THREAD_RUNNING == s32GetInternalState() )
   {
      vSetInternalState( VDS_C_S32_THREAD_STOP );
      tS32 s32TimeOutCounter = 100; //100*100ms= 10s timeout

      while ( VDS_C_S32_THREAD_INITIALIZED != s32GetInternalState() && s32TimeOutCounter > 0 )
      {
         OSAL_s32ThreadWait( 100 );
         --s32TimeOutCounter;
      }

      s32RetVal = s32GetInternalState();
   }
   else if ( s32GetInternalState() == VDS_C_S32_THREAD_INITIALIZED )
   {
      s32RetVal = VDS_C_S32_THREAD_INITIALIZED;
   }
   else
      s32RetVal = VDS_E_THREAD_NOT_RUNNING;

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   virtual method, inherited classes have to fill this function.
//!   closes handles, deletes semaphores, deletes allocated memory
//!
//! \return
//!   None
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tVoid tclSensorThread::vThreadClose ()
{
   // delete semaphore
   if ( OSAL_C_INVALID_HANDLE != hSensorSemaphore )
   {
      OSAL_s32SemaphoreClose( hSensorSemaphore );
      OSAL_s32SemaphoreDelete( coszSensorSemName );
      hSensorSemaphore = OSAL_C_INVALID_HANDLE;
   }

   // close sensor-device
   if ( OSAL_ERROR != SensorFd )
   {
      OSAL_s32IOClose( SensorFd );
      SensorFd = OSAL_ERROR;
   }
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   sets internal state
//!
//! \return
//!   None
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tVoid tclSensorThread::vSetInternalState
(
   //!  (I) new internal state. possible values are (not checked!):
   //!  VDS_C_S32_THREAD_UNINITIALIZED,
   //!  VDS_C_S32_THREAD_INITIALIZED,
   //!  VDS_C_S32_THREAD_RUNNING,
   //!  VDS_C_S32_THREAD_STOP
   tS32 s32InternalStateParam
)
{
   this->s32InternalState = s32InternalStateParam;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns internal thread-state
//!
//! \return
//!   internal thread-state. possible values are:
//!   VDS_C_S32_THREAD_UNINITIALIZED,
//!   VDS_C_S32_THREAD_INITIALIZED,
//!   VDS_C_S32_THREAD_RUNNING,
//!   VDS_C_S32_THREAD_STOP
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tS32 tclSensorThread::s32GetInternalState ()
{
   return s32InternalState;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns sensor-cycle-time
//!
//! \return
//!   sensor-cycle-time
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tU32 tclSensorThread::u32GetSensorCycleTime ()
{
   return u32SensorCycleTime;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   locks access to critical sections.
//!   only when TRUE is returned, access is allowed
//!
//! \return
//!   TRUE : access allowed
//!   FALSE: error, access is not allowed
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tBool tclSensorThread::bCritSectionBegin ()
{
   tBool bLocked = FALSE;

   if ( OSAL_C_INVALID_HANDLE != hSensorSemaphore )
      if ( OSAL_ERROR != OSAL_s32SemaphoreWait( hSensorSemaphore, VDS_C_U32_SENSORTHREAD_WAITTIME ) )
         bLocked = TRUE;

   return bLocked;
}

// ***************** F U N C T I O N  H E A D E R **************************
//
//  DESCRIPTION:
//
//! \brief
//!   unlocks access to critical sections.
//!
//! \return
//!   tVoid
//  HISTORY:
//        Date       |       Author         |   MODIFICATION
// -------------------------------------------------------------------------
//        21.03.02   |  CM-DI/ESA1-Fischer  |   Initial Revision.
//**************************************************************************
tVoid tclSensorThread::vCritSectionEnd ()
{
   if ( OSAL_C_INVALID_HANDLE != hSensorSemaphore )
   {
      OSAL_s32SemaphorePost( hSensorSemaphore );
   }
}

// Additional Declarations
