// *****************************************************************************
// * FILE:         tclMsgOdometerIf.h
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-declaration: handles access to odometer-data,
// *               handles subscriber-messages
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
//  * 19.06.09 - sak9kor - APIs "s32GetDistcalMessage","s32GetOdomMessage", and
//  *                      "s32GetEmptyOdomMessage" have been modified to
//  *                      to adopt the XML generated FI
//  * 27.07.09 - sak9kor - APIs "s32GetDriveDirectionMessage" and "s32GetGalaSpeedMessage"
//  *                      have been modified to adapt the XML generated FI
//  * 10.08.09 - sak9kor - new parameter poSubscriber has been added
//  *                      in to 's32GetDistcalMessage" and "s32GetOdomMessage" functions
//  * 12.08.09 - sak9kor - new parameter poSubscriber has been added
//  *                      in to 's32GetDriveDirectionMessage" and "s32GetGalaSpeedMessage" functions
// *****************************************************************************

#ifndef tclMsgOdometerIf_h
#define tclMsgOdometerIf_h 1

#define VDS_C_U32_ODOMIF_RINGBUFFERSIZE      (tU16)160

#define VDS_C_S32_ODOMIF_UNINITIALIZED (tS32)0
#define VDS_C_S32_ODOMIF_NORMAL        (tS32)1
#define VDS_C_S32_ODOMIF_PENDING       (tS32)2

// errors
#define VDS_E_ODOMIF_ALLOC_ERROR                 -1
#define VDS_E_ODOMIF_THREADINIT_ERROR            -2
#define VDS_E_ODOMIF_SEM_ERROR                   -3
#define VDS_E_ODOMIF_THREADSTART_ERROR           -4
#define VDS_E_ODOMIF_NOT_INITIALIZED             -5
#define VDS_E_ODOMIF_THREADSTOP_ERROR            -6
#define VDS_E_ODOMIF_GETODOMDATA_FAILED          -7
#define VDS_E_ODOMIF_SETODOMMSG_FAILED           -8
#define VDS_E_ODOMIF_SUBSCRIBER_ERROR            -9
#define VDS_E_ODOMIF_VALUE_UNSET                 -10
#define VDS_E_ODOMIF_GET_DISTCAL_FAILED          -11
#define VDS_E_ODOMIF_PERSISTENT_ERROR            -12
#define VDS_E_ODOMIF_VALUES_UNSET                -13
#define VDS_E_ODOMIF_DISTCAL_OUTOFRANGE          -14
#define VDS_E_ODOMIF_DATA_UNAVAILABLE            -15
#define VDS_E_ODOMIF_PARAM_OUT_OF_RANGE          -16
#define VDS_E_ODOMIF_INVALID_TYRETYPE            -17
#define VDS_E_ODOMIF_INVALID_TYRECIRCUMFERENCE   -18
#define VDS_E_ODOMIF_INVALID_PULSESPERTYRECIRCUM -19
#define VDS_E_ODOMIF_UNKNOWN_ERROR               -99

//!  Interface class for Odometer Device
class tclMsgOdometerIf
{
   public:
      tclMsgOdometerIf();
      ~tclMsgOdometerIf();

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32Init
      // * DESCRIPTION: initializes odometer-interface.
      // *              create and initialize instance of odometer-thread, create
      // *              semaphore, create ringbuffer for odometer-data,
      // *              initialize member-attributes
      // * PARAMETER:
      // *              tS32(*)( amt_tclServiceData *, tclSubscriberManager * )
      // *                 pfs32SendSubscriberMessage: (->I)
      // *                 pointer to function, which sends given answer-
      // *                 message with given parameters to subscribers
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_ODOMIF_ALLOC_ERROR,
      // *                        VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_THREADINIT_ERROR,
      // *                        VDS_E_ODOMIF_SEM_ERROR
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // * 17.04.02 CM-DI/ESA1-Fischer
      // *          parameter changed
      // *****************************************************************************
      tS32 s32Init (
         tS32( *pfs32SendSubscriberMessage )( amt_tclServiceData *, tclSubscriberManager * ));

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32StartThread
      // * DESCRIPTION: start odometer-thread
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_ODOMIF_THREADSTART_ERROR,
      // *                        VDS_E_ODOMIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32StartThread ();

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32StopThread
      // * DESCRIPTION: stop odometer-thread
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_ODOMIF_THREADSTOP_ERROR,
      // *                        VDS_E_ODOMIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32StopThread ();

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32DeleteThread
      // * DESCRIPTION: delete odometer-thread
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_ODOMIF_NOT_INITIALIZED,
      // *                        VDS_E_ODOMIF_THREADSTOP_ERROR
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32DeleteThread ();


      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::vSetState
      // * DESCRIPTION: set internal state
      // *              (normal or pending)
      // * PARAMETER:
      // *              tS32 s32NewState: (I)
      // *                 new state (VDS_C_S32_ODOMIF_NORMAL
      // *                 or VDS_C_S32_ODOMIF_PENDING
      // *
      // * RETURNVALUE: tVoid
      // *
      // * HISTORY:
      // * 28.05.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tVoid vSetState ( tS32 s32NewState );

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32AddOdometerData
      // * DESCRIPTION: stores odometer-data in ringbuffer, triggers check
      // *              for subscribers (and send data)
      // * PARAMETER:
      // *              vds_trOdometerData *prOdometerData: (->I)
      // *                 odometer-data to be stored in ringbuffer
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32AddOdometerData ( vds_trOdometerData* prOdometerData );

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32AddOdometerList
      // * DESCRIPTION: stores list of odometer-data in ringbuffer, triggers check
      // *              for subscribers (and send data)
      // * PARAMETER:
      // *              vds_trOdometerData *prOdometerData: (->I)
      // *                 list of odometer-data to be stored in ringbuffer
      // *              tU32 u32NumOfData: (I)
      // *                 num of datas in odometer-list
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32AddOdometerList ( const vds_trOdometerData *prOdometerData, tU32 u32NumOfData );

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32CheckForOdomSubscriber
      // * DESCRIPTION: check for subscribers and send message.
      // *              the actual list of odometer-data is given, so we normally
      // *              don't need to create a temporary storage for the
      // *              answer-message.
      // * PARAMETER:
      // *              vds_trOdometerData *prOdometerData: (->I)
      // *                 received list of odometer-data
      // *              tU32 u32NumOfData: (I)
      // *                 num of odometer-data in list
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_ALLOC_ERROR,
      // *                        VDS_E_ODOMIF_GETODOMDATA_FAILED,
      // *                        VDS_E_ODOMIF_SETODOMMSG_FAILED,
      // *                        VDS_E_POST_MESSAGE_FAILED
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32CheckForOdomSubscriber ( vds_trOdometerData *prOdometerData, tU32 u32NumOfData );

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32GetOdomInitMessage
      // * DESCRIPTION: creates odometer-init-message.
      // *
      // * PARAMETER:
      // *              amt_tclServiceData **ppoServiceData: (->O)
      // *                 pointer to created message
      // *              tU16 u16TargetAppId: (I)
      // *                 target-application-id
      // *
      // * RETURNVALUE: tS32
      // *                 [0..n] : num of data in message
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_ALLOC_ERROR,
      // *                        VDS_E_ODOMIF_GETODOMDATA_FAILED,
      // *                        VDS_E_ODOMIF_SETODOMMSG_FAILED
      // *
      // * HISTORY:
      // * 03.03.03 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32GetOdomInitMessage ( amt_tclServiceData **ppoServiceData,
                                   tU16 u16TargetAppId );


      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32GetOdomMessage
      // * DESCRIPTION: creates odometer-status-message.
      // *
      // * PARAMETER:
      //    *             amt_tclServiceData *poRequestMessage
      //    *              pointer to service data
      //    *             tclSubscriberManager *poSubscriber
      //    *             pointer to subscriber data
      // *              amt_tclServiceData **ppoServiceData: (->O)
      // *                 pointer to created message
      // *              vds_trOdometerData *prOdometerData: (->I)
      // *                 list of odometer-data
      // *              tU32 u32NumOfData: (I)
      // *                 num of odometer-data in list
      // *              tU32 &rfu32NextElemToRead: (I)
      // *                 num of next elem to read
      // *              tBool bSendInitMessage: (I)
      // *                 TRUE: init-message, FALSE: update-message
      // *              tBool bCreateInvalidMessage: (I)
      // *                 TRUE: create message with state=invalid
      // *
      // * RETURNVALUE: tS32
      // *                 [0..n] : num of data in message
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_ALLOC_ERROR,
      // *                        VDS_E_ODOMIF_GETODOMDATA_FAILED,
      // *                        VDS_E_ODOMIF_SETODOMMSG_FAILED
      // *
      // * HISTORY:
      // * 17.05.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      //    *  19.06.09: RBEI/ECF1 - Interface has been changed to adapt the XML generated FI
      //    * 10.08.2009 | sak9kor (RBEI/ECF1) | New parameter poSubscriber has been added
      // *****************************************************************************
      tS32 s32GetOdomMessage ( const amt_tclServiceData *poRequestMessage,
                               tclSubscriberManager *poSubscriber,
                               amt_tclServiceData **ppoServiceData,
                               vds_trOdometerData *prOdometerData,
                               tU32 u32NumOfData, tU32 &rfu32NextElemToRead,
                               tBool bSendInitMessage,
                               tBool bCreateInvalidMessage = FALSE );

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::getOdometerValue
      // * DESCRIPTION: returns odometer-value of last read data-set
      // *
      // * PARAMETER:
      // *              tPU16 pu16OdometerValue: (->O)
      // *                 storage for odometer-value
      // *
      // * RETURNVALUE: tU8
      // *                 ok   : VDSENSOR_MSG_ODOMETER_OK
      // *                 error: VDSENSOR_MSG_ODOMETER_ERROR
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tU8 getOdometerValue ( tPU16 pu16OdometerValue );

      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32GetOdometerData
      // * DESCRIPTION: returns a list of odometer-datas from ringbuffer.
      // *              the number of the element from where to be read could be
      // *              determined from rfu32NextElemToRead. if the number is
      // *              smaller than the ones in the list, the oldest datas will be
      // *              returned, if the number is bigger than the ones in the list,
      // *              no data is returned. rfu32NextElemToRead will be set to
      // *              the number of the last returned data-set + 1.
      // * PARAMETER:
      // *              vds_trOdometerData *prOdometerData: (->O)
      // *                 storage for odometer-list
      // *              tU32 &rfu32NextElemToRead: (->IO)
      // *                 number of element that has to be read the next time
      // *              tU32 u32ElemsToRead: (I)
      // *                 number of elements to be read
      // *
      // * RETURNVALUE: tS32
      // *                 [0..n]   : num of elems copied
      // *                 error : VDS_E_INVALID_PARAMETER,
      // *                         VDS_E_ODOMIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tS32 s32GetOdometerData ( vds_trOdometerData *prOdometerData, 
                                tU32 &rfu32NextElemToRead, 
                                tU32 u32ElemsToRead );


      // *****************************************************************************
      // * METHOD:      tclMsgOdometerIf::s32AddDataWrapper
      // * DESCRIPTION: stores list of odometer-data in ringbuffer, triggers check
      // *              for subscribers (and send data)
      // * PARAMETER:
      // *              vds_trOdometerData *prOdometerData: (->I)
      // *                 list of odometer-data to be stored in ringbuffer
      // *              tU32 u32NumOfData: (I)
      // *                 num of datas in odometer-list
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_ODOMIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // * 19.04.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      static tS32 s32AddDataWrapper ( vds_trOdometerData *prOdometerData, tU32 u32NumOfData );


      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //     Synchronises the Data. Updates Gala Speed, Checks for new Odometer
      //     and Gala Speed subscribers.
      //
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // -------------------------------------------------------------------------
      //     04.04.02    |  CM-DI/ESA1-Fischer           |   Initial Revision.
      //**************************************************************************
      void vSynch( void );
      vds_tclCriticalSection oCS;

      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //      Gets Odometer Range
      //
      //  return
      //    None
      //
      //  HISTORY:
      //      Date       |        Author        | MODIFICATION
      // ---------------------------------------------------------------------
      //*********************************************************************
      void vGetOdometerRange( tF32& rfrf32Min, tF32& rfrf32Max, tU8& rfru8Status );


      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //   brief
      //   Gets the last element index.
      //   return
      //    The index of last element in the buffer.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //****************************************************************************	                                    
      tU32 u32GetLastOdoElemIndex(tVoid)
      {
         return(oRingBuffer.u32GetNumElems() - 1);
      }
      
      // ***************** F U N C T I O N  H E A D E R **************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!     Updates Odo data in "AllSensorData" cmb fi object.
      //!
      //! \return
      //!   Success  :  VDS_E_NO_ERROR
      //!   Error    :  VDS_E_ODOMIF_NOT_INITIALIZED
      //!               VDS_E_ODOMIF_GETODOMDATA_FAILED
      //!               VDS_E_ODOMIF_DATA_UNAVAILABLE
      //!               VDS_E_ODOMIF_ALLOC_ERROR
      //!
      //  HISTORY:
      //      Date       |        Author        | MODIFICATION
      // -------------------------------------------------------------------------
      //**************************************************************************
      tS32 s32GetOdoRecords
      (
         sensor_fi_tcl_AllSensorData *pCmbFiGnssData,
         tU32 &NextelementToRead
      );
   private:
      // Data Members for Class Attributes
      tclOdometerThread *poOdometerThread;
      tS32( *pfs32SendSubscriberMessage )( amt_tclServiceData *, tclSubscriberManager * );
      vds_tclRingBuffer<vds_trOdometerData> oRingBuffer;
      static tclMsgOdometerIf *poThisMsgOdometerIf;
      tS32 s32InternalState;
      tU8 u8DriveDirection;
};

// Class tclMsgOdometerIf
#endif
