// *****************************************************************************
// * FILE:         tclMsgGpsIf.h
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-declaration: handles access to gps-data,
// *               handles subscriber-messages
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 20.07.09 - sak9kor - APIS have been modified to adapt the XML generated FI
// * 11.08.09 - sak9kor - Interfaces have been modified to avoid the usage of additional
//                         message pool memory
// * 30.09.09 - sak9kor - Added doxygen headers for all classes/funcs
// * 16.12.10 RBEI/ECF1 - sak9kor - Diaglog related modifications have been done
// * 09.11.12 RBEI/ECF5 - sak9kor - Fix for the issue NIKAI-231 is added
// * 19.09.13 RBEI/ECF5 - sga5kor - Modifications done to adopt VDSensor according to
//                                  the CmbFi version-5.0.0
// *  05.11.15 RBEI/ECF5 - rmm1kor- Added support for Diagnosis satellite system control
// * 10.11.15 RBEI/ECF5 - amg6kor - Adding Get operation for Gnss Epoch property
// * 06.01.17 RBEI/ECF12 - amg6kor - Port to Gen4
// *****************************************************************************

#ifndef tclMsgGpsIf_h
#define tclMsgGpsIf_h 1

#define VDS_C_U32_GPSIF_RINGBUFFERSIZE    (tU32)20


#define VDS_C_S32_GPSIF_UNINITIALIZED     (tS32)0
#define VDS_C_S32_GPSIF_NORMAL            (tS32)1
#define VDS_C_S32_GPSIF_PENDING           (tS32)2

// errors
#define VDS_E_GPSIF_ALLOC_ERROR           -1
#define VDS_E_GPSIF_THREADINIT_ERROR      -2
#define VDS_E_GPSIF_SEM_ERROR             -3
#define VDS_E_GPSIF_THREADSTART_ERROR     -4
#define VDS_E_GPSIF_NOT_INITIALIZED       -5
#define VDS_E_GPSIF_THREADSTOP_ERROR      -6
#define VDS_E_GPSIF_GETGPSDATA_FAILED     -7
#define VDS_E_GPSIF_SETGPSMSG_FAILED      -8
#define VDS_E_GPSIF_SUBSCRIBER_ERROR      -9
#define VDS_E_GPSIF_DATA_UNAVAILABLE      -10
#define VDS_E_GPSIF_UNKNOWN_ERROR         -99

#define VDSENSOR_MSG_GPS_OK    (tU8) 0
#define VDSENSOR_MSG_GPS_ERROR (tU8) 1

/*!
 * \brief Structure containing all kinds of GPS data.
 *
 * The slots of the ring buffer contain one of these structures.  The
 * members are expected to have identical time stamp fields, so that
 * CCA clients, which recieve them as separate messages, can relate
 * them to each other again.
 */
typedef struct
{
   vds_trGpsExtBigData rExtBigData;
   vds_trGpsNormalModeTestResult rTestResult;
   /*    vds_trGpsTightCouplingData rTCData; */
} vds_trGpsAllData;

/*!
* \brief Defines the interface for GPS messages
*/
class tclMsgGpsIf
{

   public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //  Default Constructor
      // PARAMETERS:
      // None
      // return
      //   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tclMsgGpsIf();
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //  Default Destructor
      //
      //  return
      //  None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual ~tclMsgGpsIf();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //    brief
      //       initializes gps-interface.
      //       create and initialize instance of gps-thread, create semaphore, create ringbuffer
      //       for gps-big-data, initialize member-attributes
      //  PARAMETER:
      //               tS32(*)( amt_tclServiceData *, tclSubscriberManager * )
      //                  pfs32SendSubscriberMessage: (->I)
      //                  pointer to function, which sends given answer-
      //                  message with given parameters to subscribers
      //   return
      //   Succes code in case of success is:
      //     VDS_E_NO_ERROR : Success\n
      //   Error code in case of failure are:\n
      //     VDS_E_INVALID_PARAMETER :   Invalid Parameter
      //     VDS_E_GPSIF_ALLOC_ERROR :  GPS interface allocation error
      //     VDS_E_GPSIF_SEM_ERROR : GPS interface Semaphore error
      //     VDS_E_GPSIF_THREADINIT_ERROR : GPS interface thread init error
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02          CM-DI/ESA1-Fischer               Initial Revision.
      // 17.04.02          CM-DI/ESA1-Fischer               parameter changed
      //******************************************************************************
      tS32 s32Init (
         tS32( *pfs32SendSubscriberMessage )( amt_tclServiceData *, tclSubscriberManager * )
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //  start gps-thread
      //  PARAMETERS:
      //  nONE
      //  return
      //  Succes code in case of success is:
      //  VDS_E_NO_ERROR : Success\n
      //  Error code in case of failure are:\n
      //     VDS_E_GPSIF_THREADSTART_ERROR :   GPS interface thread start error
      //     VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02          CM-DI/ESA1-Fischer               Initial Revision.
      //******************************************************************************
      tS32 s32StartThread ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //  stop gps-thread
      //  Parameters:
      //    None
      //  return
      //   Succes code in case of success is:
      //      VDS_E_NO_ERROR : Success\n
      //   Error code in case of failure are:\n
      //     VDS_E_GPSIF_THREADSTOP_ERROR :   GPS interface thread stop error
      //     VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02          CM-DI/ESA1-Fischer               Initial Revision.
      //******************************************************************************
      tS32 s32StopThread ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   delete gps-thread
      //  PARAMETERS: None
      //  return
      //   Succes code in case of success is:
      //     VDS_E_NO_ERROR : Success\n
      //   Error code in case of failure are:\n
      //     VDS_E_GPSIF_THREADSTOP_ERROR :   GPS interface thread stop error
      //     VDS_E_GPSIF_NOT_INITIALIZED :  GPS interface not initialized
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      // 21.03.02          CM-DI/ESA1-Fischer               Initial Revision.
      //******************************************************************************
      tS32 s32DeleteThread ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   set internal state  (normal or pending)
      //  PARAMETERS:
      //     tS32 s32NewState - New state to be set i.e. VDS_C_S32_STATE_NORMAL or
      //                        VDS_C_S32_STATE_NORMAL_DIAG or VDS_C_S32_STATE_PENDING
      //  return
      //   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      // 28.05.02          CM-DI/ESA1-Fischer               Initial Revision.
      //******************************************************************************
      tVoid vSetState ( tS32 s32NewState );

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::bCritSectionBegin
      // * DESCRIPTION: locks access to critical sections.
      // *              only when TRUE is returned, access is allowed
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tBool
      // *                 TRUE : access allowed
      // *                 FALSE: error, access is not allowed
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tBool bCritSectionBegin ();

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::vCritSectionEnd
      // * DESCRIPTION: unlocks access to critical sections.
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tVoid
      // *
      // * HISTORY:
      // * 21.03.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tVoid vCritSectionEnd ();

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::bCritSectionLocked
      // * DESCRIPTION: does caller locked GpsIf?
      // *              only when TRUE is returned, access is allowed
      // * PARAMETER:
      // *              tVoid
      // *
      // * RETURNVALUE: tBool
      // *                 TRUE : access allowed
      // *                 FALSE: error, access is not allowed
      // *
      // * HISTORY:
      // * 08.04.02 CM-DI/ESA1-Fischer
      // *          Initial Revision.
      // *****************************************************************************
      tBool bCritSectionLocked ();

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::s32AddGnssDataWrapper
      // * DESCRIPTION: wrapper function which calls actual function which
      // *              stores gnss data in ringbuffer.
      // *              
      // * PARAMETER:
      // *              OSAL_trGnssFullRecord* rGnssFullRecord: (->I)
      // *                 gnss data to be stored in ringbuffer
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_GPSIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // *****************************************************************************         
      static tS32 s32AddGnssDataWrapper
      (
         OSAL_trGnssFullRecord* rGnssFullRecord
      );
      
      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::s32AddGnssDataWrapper
      // * DESCRIPTION: wrapper function which calls actual function which
      // *              stores gnss data in ringbuffer.
      // *              
      // * PARAMETER:
      // *              OSAL_trGnssFullRecord* rGnssFullRecord: (->I)
      // *                 gnss data to be stored in ringbuffer
      // *
      // * RETURNVALUE: tS32
      // *                 ok   : VDS_E_NO_ERROR
      // *                 error: VDS_E_INVALID_PARAMETER,
      // *                        VDS_E_GPSIF_NOT_INITIALIZED
      // *
      // * HISTORY:
      // *****************************************************************************    
      tS32 s32AddGnssDataToBuffer( const OSAL_trGnssFullRecord *prGnssFullRecord );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    performs the synch operation of AllSensorData property
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vSynch( void );   

      // Additional Public Declarations

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Compose satus message for the property GnssData.
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  Number of elements copied : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
      //!   - \c VDS_E_GPSIF_NOT_INITIALIZED :if GPS interface not initialized.
      //!   - \c VDS_E_GPSIF_DATA_UNAVAILABLE :if no data is available.
      //!   - \c VDS_E_GPSIF_ALLOC_ERROR :if failed to create visitor mesage object.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32GetGnssDataMsg
      (
         //! (I) : Pointer to Service data
         amt_tclServiceData const *poServiceData,
         //! (I) : Pointer to Subscriber data
         tclSubscriberManager *poSubscriber,
         //! (O) : pointer to a location where a pointer to the reply message is available
         amt_tclServiceData **ppoResponseMessage,
         //! (I) : Next element to read.
         tU32 &rfu32NextElemToRead
      );
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Reads GNSS data from ring buffer.
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  Number of elements copied : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
      //!   - \c VDS_E_GPSIF_NOT_INITIALIZED :if GPS interface not initialized.
      //!
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************       
      tS32 s32GetGnssDataFromBuffer
      (
         //! (O) : Pointer to Gnss data
         OSAL_trGnssFullRecord *prGnssData,
         //! (I) : Next elements to read
         tU32 &rfu32NextElemToRead,
         //! (I) : Number of elements to read.  
         tU32 NumOfElmToRead
      );
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Get the information about number of elements available in the GNSS ring buffer.
      //! \return
      //!   Number of elements available in ring buffer. 
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************        
      tU32 u32GetNumOfGNSSrecordsAvailable();
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Compose satus message for the property GnssConfigData.
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  VDS_E_NO_ERROR : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
      //!   - \c VDS_E_GPSIF_NOT_INITIALIZED :if GPS interface not initialized.
      //!   - \c VDS_E_INTERNAL_ERROR :if failed to get config data.
      //!   - \c VDS_E_GPSIF_ALLOC_ERROR :if failed to create visitor mesage object.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************      
      tS32 s32GetGnssConfigDataMsg
      ( 
          //! (I) : Pointer to service data
          amt_tclServiceData const *poRequestMessage,
          //! (O) : pointer to a location where a pointer to the reply message is available
          amt_tclServiceData **ppoResponseMessage
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief Sets different flags of GpsThread.
      //!   
      //! \return None.
      // ******************************************************************************       
      tVoid vSetGpsThreadFlags( vds_tenGpsThreadFlags enFlag, tBool bFlagStatus );
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   check for Gnssdata subscribers and send message.
      //!
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  VDS_E_NO_ERROR : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_GPSIF_UNKNOWN_ERROR : GPS interface unknown error
      //!   - \c VDS_E_POST_MESSAGE_FAILED : Message post failed
      //  HISTORY:
      // Date        |  Author             | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32CheckForGNSSdataSubscribers
      (
        //!None.
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Collects GNSS records for "AllSensorData" property. 
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  VDS_E_NO_ERROR : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_INVALID_PARAMETER :if the received argument parameters are invalid.
      //!   - \c VDS_E_GPSIF_DATA_UNAVAILABLE :if no data is available.
      //!   - \c VDS_E_GPSIF_ALLOC_ERROR :if failed to allocate memory.
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32GetGnssRecords
      (
         sensor_fi_tcl_AllSensorData *pCmbFiGnssData,
   
         tU32 &NextelementToRead
      );

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::bGetTimeOfDay
      // * DESCRIPTION: returns current time of day
      // *
      // * PARAMETER:
      // *              tU16 &rfu16TimeOfDay: (->O)
      // *
      // * RETURNVALUE: tBool
      // *                 TRUE : filter-mode is available
      // *                 FALSE: filter-mode is not available
      // *
      // * HISTORY:
      // *  8.04.03 CM-CR/EES4-Bode
      // *          Initial Revision.
      // *****************************************************************************
      tBool bGetTimeOfDay ( tU16 &rfu16TimeOfDay ) const;

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::bSetTimeOfDay
      // * DESCRIPTION: set time of day
      // *
      // * PARAMETER:
      // *              tU16 &rfu8TimeOfDay: (->I)
      // *
      // * RETURNVALUE: tBool
      // *                 TRUE : time of day changed
      // *                 FALSE: error while setting time of day
      // *                        or feature not available for this receiver
      // *
      // * HISTORY:
      // *  8.04.03 CM-CR/EES4-Bode
      // *          Initial Revision.
      // *****************************************************************************
      tBool bSetTimeOfDay ( const tU16 &rfu16TimeOfDay );

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::bStopDevGps
      // * DESCRIPTION: Stop /dev/gps
      // *
      // * PARAMETER:
      // *              none
      // *
      // * RETURNVALUE: tBool
      // *                 TRUE : if stopped successfully or already stopped
      // *
      // * HISTORY:
      // *  6.08.03 CM-CR/EES4-Bode
      // *          Initial Revision.
      // *****************************************************************************
      tBool bStopDevGps ( tVoid );

      // *****************************************************************************
      // * METHOD:      tclMsgGpsIf::vSetGpsThreadEvent
      // * DESCRIPTION: Send Evnt to GPS
      // *
      // * PARAMETER:
      // *              VDS_GPS_EVENT_NOEVENT,
      // *              VDS_GPS_EVENT_RESTART,
      // *              VDS_GPS_EVENT_START,
      // *              VDS_GPS_EVENT_STOP
      // *              VDS_GPS_EVENT_SEND_OSAL_TIME_UPDATE
      // *
      // * RETURNVALUE: none
      // *
      // *
      // * HISTORY:
      // *  4.03.04 CM-DI/EES4-Bode
      // *          Initial Revision.
      // *****************************************************************************
      tVoid vSetGpsThreadEvent ( tU8 u8Event );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      // brief
      //    Reset factory settings
      //
      //    All persistently saved sensor data should be deleted and
      //    replaced with standard values.  A GPS cold start is performed.
      // return
      //    None
      //  HISTORY:
      // Date        |  Author             | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vFactorySettings();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      // brief
      //   Sets App state
      // return
      //    None
      //  HISTORY:
      // Date        |  Author             | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tVoid vSetAppState( tU32 u32AppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      // brief
      //    Static wrapper for trace commands.
      // return
      //    None
      //  HISTORY:
      // Date        |  Author             | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 bExecuteGpsTraceCommand( const tUChar* puchData );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      // brief
      //   Static wrapper for enable/disable the china encoder.
      // return
      //    None
      //  HISTORY:
      // Date        |  Author             | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vEnDisChinaEnc( tBool bInhibit );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief get the GNSS satellite system used
      //!   
      //! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
      //  used satellite systems. If no satellite system could be determined it retruns 
      //  0.
      //*****************************************************************************
      tU8 u8GetGnssSatSys ( tVoid );
 
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief set the GNSS satellite system used Persistently
      //!   
      //! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
      //  used satellite systems. If no satellite system could be determined it retruns 
      //  0.
      // ******************************************************************************
     tU8 u8SetGnssSatSys ( tU8 u8TargetGnssSatSys );

       // ***************** F U N C T I O N  H E A D E R *****************************
       //
       //  DESCRIPTION:
       //
       //! \brief set the GNSS satellite system used NON-Persistently
       //!   
       //! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
       //  used satellite systems. If no satellite system could be determined it retruns 
       //  0.
       // ******************************************************************************
      tU8 u8SetDiagGnssSatSys ( tU8 u8TargetGnssSatSys );


      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief set the GNSS WN Epoch
      //!   
      //! \return 
      // ******************************************************************************
     tBool bSetWnEpoch ( const OSAL_trGnssTimeUTC &rTargetEpochUTC, 
                         OSAL_trGnssTimeUTC &rReturnActualDateUTC  );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief Get the GNSS WN Epoch
      //!   
      //! \return 
      //       TRUE  : Get Gnss Epoch successful
      //       FALSE : Get Gnss Epoch failed
      // ******************************************************************************
     tBool bGetWnEpoch ( const OSAL_trGnssTimeUTC &rActualDateUTC  );                      

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief set the timemode (auto/manual) and in case of manual the time
      //!   
      //! \return 
      // ******************************************************************************
     tBool bSetTimeMode ( const vds_tenTimeMode &enTargetTimeMode, 
                          const OSAL_trTimeDate &rfTargetTimeDate );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief get actualTimeMode 
      //!   
      //! \return 
      // ******************************************************************************
     tBool bGetTimeMode ( vds_tenTimeMode & rfenActualTimeMode );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief get the last element index in the GNSS ring buffer. 
      //!   
      //! \return 
      // ******************************************************************************     
      tU32 u32GetLastGnssElemIndex(tVoid)
      {
         tU32 u32LastElemIndex=0;

         u32LastElemIndex = oGnssDataBuffer.u32GetNumElems();

         if ( 0 < u32LastElemIndex )
         {
            --u32LastElemIndex;
         }

         return(u32LastElemIndex);
      }

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   check for AllSensorData subscribers and send message.
      //!
      //! \return
      //!   Succes code in case of success is:
      //!   - \c  VDS_E_NO_ERROR : Success\n
      //!   Error code in case of failure are:\n
      //!   - \c VDS_E_GPSIF_UNKNOWN_ERROR : GPS interface unknown error
      //!   - \c VDS_E_POST_MESSAGE_FAILED : Message post failed
      //  HISTORY:
      // Date        |  Author             | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32CheckForAllSensorDataSubscribers(tVoid);
      
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!    Copy GNSS data to CMB FI object.
      //! \return
      //!     None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tVoid vCopyGnssDataToCmbFiObject
      (
         //! (O) : GNSS Data cmb fi object
         sensor_fi_tcl_GnssData *prCmbfiGnssData,
         //! (I) : GNSS data
         const OSAL_trGnssFullRecord *prGnssData
      );
   private:
      // Data Members for Class Attributes
      tclGpsThread *poGpsThread;
      tS32( *pfs32SendSubscriberMessage )( amt_tclServiceData *, tclSubscriberManager * );
      OSAL_tSemHandle hGpsIfSemaphore;
      tCString coszGpsIfSemName;
      vds_tclRingBuffer<OSAL_trGnssFullRecord> oGnssDataBuffer;
      tBool bCritSectionIsLocked;
      static tclMsgGpsIf *poThisMsgGpsIf;
      tS32 s32InternalState;
};

#endif
