// *****************************************************************************
// * FILE:         tclAccThread.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition:Opens the accelerometer Device and 
// *               provides an interface to read the accelerometer raw data 
// * AUTHOR:       RBEI/ECF1-Sainath Kalpuri
// * COPYRIGHT:    (c) 2011 Robert Bosch
// * HISTORY:
// * 24.05.2011   Sainath Kalpuri(RBEI CM-AI/PJ-CF31) - Added the implementation
// *                                                    related to acc3dDataUpdate
// *                                                    property
// *****************************************************************************
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_ERRMEM_DEF
#include "system_pif.h"

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_ERRORCODES
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_FUNCTIONIDS
#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_SERVICEINFO
#include "sensor_fi_if.h"
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "fi_msgfw_if.h"
#define SENSOR_S_IMPORT_INTERFACE_SERVER
#include "vds2_if.h"

#define VDS_S_IMPORT_INTERFACE_SUBSCRIBER_MANAGER
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#define VDS_S_IMPORT_INTERFACE_SENSOR_DATATYPES
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

extern void vGetTimeStamp( tPU32 pu32TimeStamp );
/*!
* \brief Defines the interface for Accsynchlistner.
*/

class vds_tclAccSynchListener : public vds_tclSynchListenerIf
{
   private:
      tclMsgAccIf *poAccIf;
   public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Default constructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      vds_tclAccSynchListener
      (
         //! pointer to Acc interface argument
         tclMsgAccIf *poAccIfArg
      )
      {
         poAccIf = poAccIfArg;
      }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Performs the synch operation
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual void vSynch
      (
         //! None
         void
      )
      {
         vTraceMsg( VDS_C_TRACELEVEL_EVENT, "Synch (Acc)" );

         if ( poAccIf != OSAL_NULL )
         {
            poAccIf->vSynch( );
         }
      }
};

tclMsgAccIf *tclMsgAccIf::poThisMsgAccIf;
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default constructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author         | MODIFICATION
// ----------------------------------------------------------------------------
//  08.03.11         sak9kor(RBEI/ECF1)    Initial Revision.
//******************************************************************************
tclMsgAccIf::tclMsgAccIf()
      : oCS( "VDS_ACC_IF_SEM" ),
      poAccThread( OSAL_NULL ),
      s32InternalState( VDS_C_S32_STATE_UNINITIALIZED )
{
   poThisMsgAccIf = this;
   pfs32SendSubscriberMessage = NULL;
   bIsSelftestRunning = FALSE;
   vds_goSensorPropertySynchManager.bAddListener( new vds_tclAccSynchListener( this ) );
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Destructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author         | MODIFICATION
// ----------------------------------------------------------------------------
//  08.03.11         sak9kor(RBEI/ECF1)    Initial Revision.
// ----------------------------------------------------------------------------
//******************************************************************************
tclMsgAccIf::~tclMsgAccIf()
{
   oRingBuffer3d.vDeallocate();   
   
   // delete acc-thread
   if ( NULL != poAccThread )
   {
      delete poAccThread;
      poAccThread = NULL;
   }
   poThisMsgAccIf = NULL;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   initializes Accelerometer-interface create and initialize instance of Acc-thread,
//!   create semaphore, create ringbuffer for acc-data, initialize
//!   member-attributes
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_ALLOC_ERROR : Acc interface allocation error
//!   - \c VDS_E_ACCIF_THREADINIT_ERROR : Acc interface error
//  HISTORY:
// Date            |  Author         | MODIFICATION
// ----------------------------------------------------------------------------
//  08.03.11         sak9kor(RBEI/ECF1)    Initial Revision.
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32Init
(
   //! (I) : pointer to function, which sends given answer message with given
   //! parameters to subscribers
   tS32( *pfs32SendSubscriberMessageParam )( amt_tclServiceData *, tclSubscriberManager * )

)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   tU32 u32AccCycleTime = 0;

   //Parameter check
   if ( NULL != pfs32SendSubscriberMessageParam )
   {
      //Save ail-sensor-interface
      this->pfs32SendSubscriberMessage = pfs32SendSubscriberMessageParam;

      //Create accelerometer thread instance
      poAccThread = new tclAccThread();

      if ( NULL != poAccThread )
      {
         if ( VDS_E_NO_ERROR 
              == 
              poAccThread->s32ThreadInit( this, tclMsgAccIf::s32Add3dDataWrapper ))
         {
            //Create semaphore
            if ( oCS.bInit( ) )
            {
               tU32 u32RingBufferSize;

               u32AccCycleTime = poAccThread->u32GetAccCycleTime();
               if(u32AccCycleTime != 0)
               {
               //Create Accelerometer-storage (seconds * num_of_sensordata_per_second)
               u32RingBufferSize = (tU32) (VDS_C_U32_BUFFERSIZE_SECONDS * 
                                         ( 1000000000.0 / ( tF64 )u32AccCycleTime ) );                                         

               vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "AccIf:RingBufferSize = %d", u32RingBufferSize ); 
               }
               else 
               {
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                             "AccIf:s32Init:VDS_E_ALLOC_ERROR :Invalid Acc cycle time .");
                  return VDS_E_ALLOC_ERROR;
               }

               oRingBuffer3d.vAllocate( u32RingBufferSize );

              // 3dAcc ring buffer
               if ( !oRingBuffer3d.bIsValid() )
               {
                  s32RetVal = VDS_E_ACCIF_ALLOC_ERROR;
                  oCS.bShutdown();
                  vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccIf:s32Init:Alloc failed!" );
               }
            }
            else
            {
               vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccIf:s32Init:semaphore create failed" );
               s32RetVal = VDS_E_ACCIF_SEM_ERROR;
            }
            if ( VDS_E_NO_ERROR != s32RetVal )
            {
               poAccThread->vThreadClose();
            }
         }
         else
         {
            s32RetVal = VDS_E_ACCIF_THREADINIT_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_FATAL, "AccIf:s32Init:ThreadInit Failed" );
         }
         if ( VDS_E_NO_ERROR != s32RetVal )
         {
            delete poAccThread;
            poAccThread = OSAL_NULL;
         }
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccIf:s32Init:Thread instance creation failed" );
         s32RetVal = VDS_E_ACCIF_ALLOC_ERROR;
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,"AccIf:s32Init:Invalid parameter" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   start Acc-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_THREADSTART_ERROR : Thread start error
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32StartThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Start accelerometer thread
   if ( OSAL_NULL != poAccThread )
   {
      if ( VDS_C_S32_THREAD_RUNNING != poAccThread->s32ThreadStart() )
      {
         /*Return value is commented to ensure that VDSensor works fine in the hardwares
         which doesn't have the accelerometer i.e. EPSON Gyro on board*/
         //s32RetVal = VDS_E_ACCIF_THREADSTART_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf:Couldn't start AccThread" );
      }
   }
   else
   {
      s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf:poAccThread is NULL" );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stop acc-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_THREADSTOP_ERROR : Thread stop error
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32StopThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Stop accelerometer thread
   if ( OSAL_NULL != poAccThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poAccThread->s32ThreadStop() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf:Couldn't stop AccThread" );
         s32RetVal = VDS_E_ACCIF_THREADSTOP_ERROR;
      }
   }
   else
   {
      s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   delete acc-thread
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_THREADSTOP_ERROR : Thread stop error
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32DeleteThread
(
   //! None
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Delete accelerometer thread
   if ( OSAL_NULL != poAccThread )
   {
      if ( VDS_C_S32_THREAD_INITIALIZED != poAccThread->s32ThreadStop() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf:Couldn't stop AccThread" );
         s32RetVal = VDS_E_ACCIF_THREADSTOP_ERROR;
      }

      delete poAccThread;
      poAccThread = OSAL_NULL;
   }
   else
   {
      s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   set internal state (normal or pending)
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgAccIf::vSetState
(
   //! (I) : new state
   //! (VDS_C_S32_STATE_NORMAL, VDS_C_S32_STATE_NORMAL_DIAG, VDS_C_S32_STATE_PENDING)
   tS32 s32NewState
)
{
   if ( ( VDS_C_S32_STATE_NORMAL      == s32NewState )||
        ( VDS_C_S32_STATE_NORMAL_DIAG == s32NewState )||
        ( VDS_C_S32_STATE_PENDING     == s32NewState )
      )
   {
      s32InternalState = s32NewState;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Set Application state
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclMsgAccIf::vSetAppState
(
   //! (I) : Application state to be set
   tU32 u32AppState
)
{
   (void)u32AppState;
   // not need yet
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Performs the synch operation
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void tclMsgAccIf::vSynch
(
   //! None
   void
)
{
   s32CheckFor3dAccSubscriber( OSAL_NULL, 0 ) ;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stores list of 3D acc-data in ringbuffer, triggers check for
//!   subscribers (and send data)
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_SUBSCRIBER_ERROR : Subscriber error
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32Add3dAccList
(
   //! (I) : list of acc-data to be stored in ringbuffer
   const sensor_fi_tcl_3dAccData *pr3dAccData,
   //! (I) : num of datas in acc-list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   //Parameter check
   if ( ( OSAL_NULL != pr3dAccData ) && ( 0 != u32NumOfData ) )
   {
      //Consistence check
      if ( ( oRingBuffer3d.bIsValid() ) && ( oCS.bEnter() ) )
      {
         oRingBuffer3d.u32AddList( pr3dAccData, u32NumOfData );
         oCS.vLeave();
         //  if ( VDS_E_NO_ERROR != s32CheckFor3dAccSubscriber( pr3dAccData, u32NumOfData ) )
         //     s32RetVal = VDS_E_ACCIF_SUBSCRIBER_ERROR;

         // This has to be done after we have left the critical
         // section.  If it does cause a synchronisation event, we
         // will enter the critical section again when we try to send
         // off our data.
         vds_goSensorPropertySynchManager.bCheckTimeOut( );
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"AccIf:s32Add3dAccList:Not initialized" );
         s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"AccIf:s32Add3dAccList:Invalid parameters" );
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   creates 3dAcc-status-message.
//! \return
//!   Succes code in case of success is:
//!   - Number of elements created\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_GETACCDATA_FAILED : Get ACC data failed
//!   - \c VDS_E_ACCIF_ALLOC_ERROR : Allocation error
//!   - \c VDS_E_ACCIF_DATA_UNAVAILABLE : Data unavailable
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED : Data unavailable
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32Get3dAccMessage
(
   //! (I) : pointer to service data
   amt_tclServiceData const *poRequestMessage,
   //! (I) : pointer to subscriber data
   tclSubscriberManager *poSubscriber,
   //! (O) : pointer to a location where a pointer to the reply message is available
   amt_tclServiceData **ppoServiceData,
   //! (I) : pointer to Acc data
   sensor_fi_tcl_3dAccData *pr3dAccData,
   //! (I) : Number of data
   tU32 u32NumOfData,
   //! (IO) : number of element that has to be read the next time
   tU32 &rfu32NextElemToRead,
   //! (I) :  TRUE: init-message, FALSE: update-message
   tBool /*bSendInitMessage*/,
   //! (I) : TRUE: create message with state=invalid
   tBool bCreateInvalidMessage
)
{
   tS32 s32RetVal = OSAL_OK;
   sensor_fi_tcl_3dAccData *parLocal3dAccData = OSAL_NULL; // points to temp buffer or param "prAccData"
   sensor_fi_tcl_3dAccData *parTemp3dAccData = OSAL_NULL;
   tU32 u32NumOfElemToSend = 0;
   tU32 u32TempNumOfData = 0;
   tU16 u16RegisterID = 0;
   tU16 u16CmdCounter = 0;
   tU16 u16TargetAppID = 0;
   tU16 u16ServiceID = 0;

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parLocal3dAccData);
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dAccData);

   if ( OSAL_NULL != poRequestMessage )
   {
      u16TargetAppID = poRequestMessage->u16GetSourceAppID();
      u16RegisterID  = poRequestMessage->u16GetRegisterID();
      u16CmdCounter  = poRequestMessage->u16GetCmdCounter();
      u16ServiceID   = poRequestMessage->u16GetServiceID();
   }
   else if ( OSAL_NULL != poSubscriber )
   {
      u16TargetAppID = poSubscriber->u16GetTargetAppId();
      u16RegisterID  = poSubscriber->u16GetRegisterId();
      u16CmdCounter  = poSubscriber->u16GetCommandCntr();
      u16ServiceID   = poSubscriber->u16GetServiceId();
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   // param-check
   if ( ( OSAL_OK == s32RetVal ) && ( OSAL_NULL != ppoServiceData ) )
   {
      *ppoServiceData = OSAL_NULL;

      if ( oCS.bLocked() )
      {
         u32NumOfElemToSend = 
            oRingBuffer3d.u32GetElemsFrom( rfu32NextElemToRead );

         // vTraceMsg( VDS_C_TRACELEVEL_ERROR,
         //            "AccIf: s32Get3dAccMessage: elements:%d",
         //            u32NumOfElemToSend );
   
         // elems available ?
         if ( u32NumOfElemToSend > 0 )
         {
            // *** get data to send ***

            // more data in buffer (to send) than in params ?
            if ( u32NumOfElemToSend > u32NumOfData )
            {
               // there are data in ringbuffer left, create temp.buffer to send
               // data
               u32TempNumOfData = u32NumOfElemToSend;

               parTemp3dAccData = 
                  OSAL_NEW sensor_fi_tcl_3dAccData[ u32TempNumOfData ];

               if ( OSAL_NULL != parTemp3dAccData )
               {
                  if ( ( tS32 )u32NumOfElemToSend 
                       == 
                       s32Get3dAccData( parTemp3dAccData, 
                                         rfu32NextElemToRead, 
                                         u32NumOfElemToSend ) )
                  {
                     parLocal3dAccData = parTemp3dAccData;

                     if ( bCreateInvalidMessage )
                     {
                        for ( tU32 ctu32AccNumber = 0;
                              ctu32AccNumber < u32NumOfElemToSend;
                              ctu32AccNumber++ )
                        {
                           parLocal3dAccData[ctu32AccNumber].RStatus.enType = 
                              sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;

                           parLocal3dAccData[ctu32AccNumber].SStatus.enType = 
                              sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;

                           parLocal3dAccData[ctu32AccNumber].TStatus.enType = 
                              sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;
                        }
                     }
                  }
                  else
                  {
                     vTraceMsg( 
                        VDS_C_TRACELEVEL_ERROR,
                        "AccIf: s32GetAccessage: couldn't get requested (%d) 3dAcc-data",
                        u32NumOfElemToSend );

                     s32RetVal = VDS_E_ACCIF_GETACCDATA_FAILED;
                  }
               }
               else
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf: s32Get3dAccMessage: alloc-error" );

                  s32RetVal = VDS_E_ACCIF_ALLOC_ERROR;
               }
            }
            else
            {
               // no data in ringbuffer left, send only actual data
               parLocal3dAccData = pr3dAccData;
               rfu32NextElemToRead += u32NumOfElemToSend;
            }

            // are there elems to send?
            if ( OSAL_NULL != parLocal3dAccData )
            {
               // *** create update-message ***
               sensor_locationfi_tclMsgAcc3dData_UpdateStatus oAcc3dDataUpdateStatus;
               //oAcc3dDataUpdateStatus.AccData.resize( u32NumOfElemToSend );

               for ( tU32 u32Loop = 0; u32Loop < u32NumOfElemToSend; u32Loop++ )
               { 
                  // copy values from array to list
                  oAcc3dDataUpdateStatus.Acc3dData.push_back(parLocal3dAccData[u32Loop]);
               }

               tU32 u32TS1 = parLocal3dAccData[0].TimeStamp3dAcc;
               tU32 u32TS2 = parLocal3dAccData[u32NumOfElemToSend-1].TimeStamp3dAcc;

               vTraceMsg( VDS_C_TRACELEVEL_EVENT,
                          "3D-Acc message ts = %d--%d", u32TS1, u32TS2 );

               // vGyroDataSanityCheck( 100, parLocal3dGyroData, u32NumOfElemToSend );
               /* Create the object of visitor message and insert the FI object*/
               fi_tclVisitorMessage* poResultMessage = 
                  OSAL_NEW fi_tclVisitorMessage( oAcc3dDataUpdateStatus, VDS_C_U16_SRV_SENSORS_MAJOR_VERSION );

               if ( OSAL_NULL != poResultMessage )
               {
                  // Prepare response message
                  if ( poResultMessage->bIsValid() )
                  {
                     tU32 u32TimeStamp = 0;
                     vGetTimeStamp( &u32TimeStamp );
                     poResultMessage->vInitServiceData
                     (  CCA_C_U16_APP_SENSOR, // Source
                        u16TargetAppID,       // Target
                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,     // StreamType
                        0,                            // StreamCounter
                        u16RegisterID,    // RegisterID
                        u16CmdCounter,    // nCmdCounter,
                        u16ServiceID,     // nServiceID,
                        SENSOR_LOCATIONFI_C_U16_ACC3DDATA_UPDATE, // nFunctionID,
                        AMT_C_U8_CCAMSG_OPCODE_STATUS,         // OpCode
                        0,                            // Asynchronous Completion Token (ACT)
                        AMT_C_U16_DEFAULT_NULL,       // Source Sub
                        AMT_C_U16_DEFAULT_NULL,       // Target Sub
                        u32TimeStamp                  //TimeStamp
                     );
                     *ppoServiceData = poResultMessage;/*static_cast<amt_tclServiceData *>(poResultMessage);*/
                     s32RetVal = ( tS32 )u32NumOfElemToSend;
                  }
                  else
                  {
                     s32RetVal = VDS_E_ACCIF_ALLOC_ERROR;;
                  }

                  if ( s32RetVal < ( tS32 )u32NumOfElemToSend )
                  {
                     OSAL_DELETE poResultMessage;
                     poResultMessage = OSAL_NULL;
                  }
               }
               else
               {
                  s32RetVal = VDS_E_ACCIF_ALLOC_ERROR;
               }
               
               VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(poResultMessage);

               parLocal3dAccData = OSAL_NULL;
            }

            // else no data to send (nor data in ringbuffer neither in params)

            // (if exists) delete temp buffer
            if ( parTemp3dAccData != OSAL_NULL
                 && 
                 parTemp3dAccData != pr3dAccData )
            {
               // delete temp.buffer
               OSAL_DELETE [] parTemp3dAccData;
               parTemp3dAccData = OSAL_NULL;
            }
         } // end if elems available
         else
         {
            // no elem to send
            s32RetVal = VDS_E_ACCIF_DATA_UNAVAILABLE;
         }
      } // end if oCS.bLocked
      else
      {
         s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;
      }
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parLocal3dAccData);
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dAccData);

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns a list of acc-datas from ringbuffer.the number of the element
//!   from where to be read could be determined from rfu32NextElemToRead.
//!   if the number is smaller than the ones in the list, the oldest datas
//!   will be returned, if the number is bigger than the ones in the list,
//!   no data is returned. rfu32NextElemToRead will be set to
//!   the number of the last returned data-set + 1.
//! \return
//!   Elements copied in case of success
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32Get3dAccData
(
   //! (O) : storage for acc-list
   sensor_fi_tcl_3dAccData *prAccData,
   //! (IO) : number of element that has to be read the next time
   tU32 &rfu32NextElemToRead,
   //! (I) : number of elements to be read
   tU32 u32ElemsToRead
)
{
   tS32 s32ElemsCopied = 0;
   sensor_fi_tcl_3dAccData *prData = prAccData;

   //Parameter check
   if ( ( OSAL_NULL != prData )&& ( 0 != u32ElemsToRead ) )
   {
      //Consistence check
      if ( ( oRingBuffer3d.bIsValid() ) && oCS.bLocked() )
      {
         tU32 u32Elems;
         u32Elems = oRingBuffer3d.u32GetList( prData, u32ElemsToRead, rfu32NextElemToRead );

         // elems lost? -> set status in first copied acc-data
         if ( u32Elems != u32ElemsToRead )
         {
            if ( prData->RStatus.enType != 
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR )
            {
               prData->RStatus.enType = 
                  sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;

               prData->SStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;

              prData->TStatus.enType =
                     sensor_fi_tcl_e16_AccStatus::FI_EN_ACCSTATE_CONNECTED_INTERNALERROR;
            }
         }
         s32ElemsCopied = ( tS32 )u32Elems;
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf: s32Get3dAccData: not init." );
         s32ElemsCopied = VDS_E_ACCIF_NOT_INITIALIZED;
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR, "AccIf: s32Get3dAccData: invalid param" );
      s32ElemsCopied = VDS_E_INVALID_PARAMETER;
   }

   return s32ElemsCopied;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   check for subscribers and send message. the actual list of acc-data is given,
//!   so we normally don't need to create a temporary storage for the answer-message.
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_UNKNOWN_ERROR :   Unknown error
//!   - \c VDS_E_POST_MESSAGE_FAILED :  Message posting failed
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32CheckFor3dAccSubscriber
(
   //! (I) : received list of Acc-data
   sensor_fi_tcl_3dAccData *pr3dAccData,
   //! (I) : num of Acc-data in list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( VDS_C_S32_STATE_NORMAL == s32InternalState
        || 
        VDS_C_S32_STATE_NORMAL_DIAG == s32InternalState
      )
   {
      // get subscriber
      if ( vds_goMasterLock.bEnter() )
      {
         tclSubscriberManager *poSubscriber = 
            tclSubscriberManager::poGetSubscriberWithFId( OSAL_NULL,
                                                          CCA_C_U16_SRV_SENSOR_LOCATION,
                                                          VDS_C_U16_FKTID_ACC3DDATAUPDATE );

         // while subscribers are available...
         while ( OSAL_NULL != poSubscriber )
         {
            // get num of data to send
            tU32 u32NextElemToRead = poSubscriber->u32GetNextElemToRead();
            tS32 s32TempErrorValue = VDS_E_NO_ERROR;
            tS32 s32ElemsCreated = 0;

            do
            {
               amt_tclServiceData *poServiceData = OSAL_NULL;

               if ( oCS.bEnter() )
               {
                  // get Acc-message
                  s32ElemsCreated = s32Get3dAccMessage(
                                       OSAL_NULL,
                                       poSubscriber,
                                       &poServiceData,
                                       pr3dAccData,
                                       u32NumOfData,
                                       u32NextElemToRead,
                                       !poSubscriber->bGetInitMessageSent() );
                  oCS.vLeave();
               }
               else
                  s32RetVal = VDS_E_ACCIF_UNKNOWN_ERROR;

               if ( ( s32ElemsCreated > 0 )
                    && 
                    ( OSAL_NULL != poServiceData )
                     && 
                    ( OSAL_NULL != pfs32SendSubscriberMessage ) )
               {
                  // send message
                  if ( VDS_E_NO_ERROR 
                       == 
                       pfs32SendSubscriberMessage( poServiceData,
                                                   poSubscriber ) )
                  {
                     poSubscriber->vSetInitMessageSent( TRUE );
                     // set next elem to read
                     poSubscriber->vSetNextElemToRead( u32NextElemToRead );
                  }
                  else
                  {
                     s32TempErrorValue = VDS_E_POST_MESSAGE_FAILED;
                  }

                  OSAL_DELETE poServiceData;
               }
               else if ( s32ElemsCreated < 0 )
               {
                  s32TempErrorValue = s32ElemsCreated;
               }
            }
            while ( s32ElemsCreated > 0
                    && 
                    VDS_E_NO_ERROR == s32TempErrorValue );

            // if error occurs in loop and no error is set, take over temp error-value
            if ( VDS_E_NO_ERROR == s32RetVal
                 && 
                 VDS_E_NO_ERROR != s32TempErrorValue )
            {
               s32RetVal = s32TempErrorValue;
            }
            // get next subscriber with this function-id
            poSubscriber = tclSubscriberManager::poGetSubscriberWithFId( poSubscriber,
                           CCA_C_U16_SRV_SENSOR_LOCATION,
                           VDS_C_U16_FKTID_ACC3DDATAUPDATE );
         } // end while (subscribers available)

         // leave critical section
         vds_goMasterLock.vLeave();
      } // end if lock subscriber successful

      // else no subscriber exists, return VDS_E_NO_ERROR
   }

   return s32RetVal;
}




   
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   stores list of 3D acc-data in ringbuffer, triggers check for
//!   subscribers (and send data)
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_SUBSCRIBER_ERROR : Subscriber error
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED : Acc interface not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclMsgAccIf::s32Add3dDataWrapper
(
   //! (I) : list of acc-data to be stored in ring buffer
   const sensor_fi_tcl_3dAccData *pr3dAccData,
   //! (I) : num of datas in acc-list
   tU32 u32NumOfData
)
{
   tS32 s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;

   if ( OSAL_NULL != poThisMsgAccIf )
   {
      s32RetVal = poThisMsgAccIf->s32Add3dAccList( pr3dAccData, u32NumOfData );
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Collects Acc records for "AllSensorData" property and updates in cmb fi object. 
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_ACCIF_NOT_INITIALIZED :if vdsensor is not initialized.
//!   - \c VDS_E_ACCIF_ALLOC_ERROR :if failed to allocate memory.
//!   - \c VDS_E_ACCIF_GETACCDATA_FAILED: if unable to get data.
//!   - \c VDS_E_ACCIF_DATA_UNAVAILABLE:if no acc data is available. 
//  HISTORY:
//  Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//****************************************************************************** 
tS32 tclMsgAccIf::s32GetAccRecords
(
   //! (O) :cmb-fi object to copy ACC data.
   sensor_fi_tcl_AllSensorData *pCmbFiAccData,
   //! (I) :Next element to read
   tU32 &rfu32NextelementToRead
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   sensor_fi_tcl_3dAccData *parTemp3dAccData = OSAL_NULL;

   if ( oCS.bLocked() )
   {
      tU32 u32NumOfElemToSend = oRingBuffer3d.u32GetElemsFrom( rfu32NextelementToRead );

      //Elements available ?
      if ( u32NumOfElemToSend > 0 )
      {
         parTemp3dAccData = OSAL_NEW sensor_fi_tcl_3dAccData[ u32NumOfElemToSend ];

         if ( OSAL_NULL != parTemp3dAccData )
         {
            if( ( tS32 )u32NumOfElemToSend  != s32Get3dAccData( parTemp3dAccData,
                                                                rfu32NextelementToRead,
                                                                u32NumOfElemToSend ) )
            {
               s32RetVal = VDS_E_ACCIF_GETACCDATA_FAILED;
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,"AccIf:AccAllSen:Unable to get Acc data.");
            }
         }
         else
         {
            s32RetVal = VDS_E_ACCIF_ALLOC_ERROR;
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,"AccIf:AccAllSen:Allocation error");
         }

         //Are there elements to send?
         if( ( OSAL_NULL != parTemp3dAccData ) && ( s32RetVal == VDS_E_NO_ERROR ) )
         {
            for ( tU32 u32Loop = 0; u32Loop < u32NumOfElemToSend; u32Loop++ )
            {
               // copy values from array to list
               pCmbFiAccData->AccUpdate.push_back(parTemp3dAccData[u32Loop]);
            }

            tU32 u32TS1 = parTemp3dAccData[0].TimeStamp3dAcc;
            tU32 u32TS2 = parTemp3dAccData[u32NumOfElemToSend-1].TimeStamp3dAcc;
            vTraceMsg( VDS_C_TRACELEVEL_EVENT,
                          "AccIf:3D-Acc message ts = %d--%d", u32TS1, u32TS2 );
         }

         //(if exists) delete temp buffer
         if ( parTemp3dAccData != OSAL_NULL )
         {
            OSAL_DELETE [] parTemp3dAccData;
            parTemp3dAccData = OSAL_NULL;
         }
      }
      else
      {
         //No elements to send
         s32RetVal = VDS_E_ACCIF_DATA_UNAVAILABLE;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"AccIf:AccAllSen:Data Unavailable");
      }
   }
   else
   {
      s32RetVal = VDS_E_ACCIF_NOT_INITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"AccIf:AccAllSen:Not Initialized");
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dAccData);

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//! \brief
//!       Start Acc self test.
//! \return
//!   - \c TRUE  : Self test finished successfully.
//!   - \c FALSE : Unable to start self test.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclMsgAccIf::bStartSelftest
(
   tS32 *ps32SelfTestResult
)
{
   tBool bRetVal = FALSE;

   if( NULL != poAccThread )
   {
      //Start self test.
      bRetVal = poAccThread->bStartSelftest( ps32SelfTestResult );
   }

   return bRetVal;
}
//EOF

