// *****************************************************************************
// * FILE:         tclGyroThread.cpp
// * SW_COMPONENT: VD-Sensor
// * DESCRIPTION:  class-definition: inherits from tclSensorThread,
// *               thread-main-loop, reads data from gyro-device
// * AUTHOR:       CM-DI/ESA1-Fischer
// * COPYRIGHT:    (c) 2002 Blaupunkt GmbH
// * HISTORY:
// * 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
// *          Initial Revision.
// * 30.09.09 sak9kor Added doxygen headers for all classes/funcs.
// * 27.08.10 sak9kor Gyro Read related functionalities have been modified to
// *          support gyro 3d
// * 06.05.11 fsj2hi  added handling of 3d Data for sending messages
// * 24.02.15 fsj2hi  removed dead code, get hwinfo via ioctrl
// * 06.01.17 amg6kor  Port to Gen4
// *****************************************************************************
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SENSOR_THREADS
#define VDS_S_IMPORT_INTERFACE_SENSOR_DEFINES
#define VDS_S_IMPORT_INTERFACE_TRACE
#define VDS_S_IMPORT_INTERFACE_MESSAGE_INTERFACES
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#include "vds_internal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define CCA_S_IMPORT_INTERFACE_GENERIC // CCA_C_U16_APP_SENSOR
#include "cca_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC // scd_s32GetThreadConfiguration
#include "scd_if.h"
#include "regkeys.h"    // THREADNAME_C_STRING_SENSOR_VDS_GYRO
#include "vds_hwinfo.h"

#define MIN_CYCLE_TIME (10 * 1000000) // 10ms = 100Hz

// Class tclGyroThread
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Constructor
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclGyroThread::tclGyroThread()
   : parGyroData( OSAL_NULL ),
     u32NumOfGyroData( 0 ),
     par3dGyroStorage( OSAL_NULL ),
     pfs32Add3dGyroList( OSAL_NULL ),
     poMsgGyroIf( OSAL_NULL ),
     u32GyroSensorCycleTimeInNS( 50000000 ),
     f32TempKelvin(0.f)
{
   coszSensorSemName = OSAL_NULL;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Default Destructor
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclGyroThread::~tclGyroThread()
{
   tclGyroThread::vThreadClose();

   parGyroData = OSAL_NULL;
   par3dGyroStorage = OSAL_NULL;
   poMsgGyroIf = OSAL_NULL;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread. open gyro-device, create semaphore,
//!   create gyro-thread, set function-pointer, set internal state
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROTHREAD_DEVICE_NOT_FOUND :   Device not found
//!   - \c VDS_E_GYROTHREAD_ALLOC_ERROR :  Allocation error
//!   - \c VDS_E_GYROTHREAD_THREADCREATE_FAILED : Thread create failed
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclGyroThread::s32ThreadInit
(
   //! (I) : Pointer to gyroif parameter
   tclMsgGyroIf    *poMsgGyroIfParam,
   //! (I) : this is a pointer to the function, where sensor-data will be
   //! stored.  this function has to be called, when sensor-data is read.
   tS32( *pfs32Add3dGyroListParam )( const sensor_fi_tcl_3dGyroData*, tU32 )
   )
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( VDS_C_S32_THREAD_UNINITIALIZED == s32GetInternalState() )
   {
      this->poMsgGyroIf = poMsgGyroIfParam;

      //Pointer to the function which updates collected data to ring buffer.
      this->pfs32Add3dGyroList = pfs32Add3dGyroListParam;

      if( OSAL_NULL == pfs32Add3dGyroList )
      {
         s32RetVal = VDS_E_INVALID_PARAMETER;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GyroThread:Init:Invalid argument");
      }

      //Open gyro-device
      SensorFd = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO,OSAL_EN_READONLY );
      if ( OSAL_ERROR  ==  SensorFd )
      {
         s32RetVal = VDS_E_GYROTHREAD_DEVICE_NOT_FOUND;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GyroThread:Init:Device open failed error = 0x%x",
                    OSAL_u32ErrorCode() );
      }



      // Ask the device about the gyro type and update the system
      // information structure.
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         tS32 s32GyroCycleTime = 0;//lint !e774
         tS32 s32GyroCycleRetVal = OSAL_s32IOControl( SensorFd,
                                                      OSAL_C_S32_IOCTRL_GYRO_GETCYCLETIME,
                                                      (tLong)&s32GyroCycleTime );

         //check return value of OSAL_s32IOControl and cycle time value, must be
         //!= 0, because of division with it in tclMsgGyroIf
         if( ( OSAL_ERROR != s32GyroCycleRetVal ) 
            && ( s32GyroCycleTime >= MIN_CYCLE_TIME ) ) //lint !e774 !e845 PQM_authorized_multi_527 PQM_authorized_multi_549 */
         {
            u32GyroSensorCycleTimeInNS = (tU32)s32GyroCycleTime;
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
            "GyroThread:OSAL_C_S32_IOCTRL_GYRO_GETCYCLETIME SUCCESS s32GyroCycleRetVal-%d s32GyroCycleTime-%d "
            ,s32GyroCycleRetVal,s32GyroCycleTime); 
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "GyroThread:Invalid gyro cycletime %d, used init default value %d",
                       s32GyroCycleTime, u32GyroSensorCycleTimeInNS );
         }
      }

      tclSystemInformation::rInfo.rGyro.r3dGyroHWInfo.SampleRate =
         (1000000000u / u32GyroSensorCycleTimeInNS);

//OSAL_trSensorHwInfo
      OSAL_trIOCtrlHwInfo rIOCtrlHwInfo = {0};

      if ( OSAL_ERROR ==
           OSAL_s32IOControl( SensorFd,
                              OSAL_C_S32_IOCTRL_GYRO_GET_HW_INFO,
                              (tLong)&rIOCtrlHwInfo ))

      {

         vTraceMsg( VDS_C_TRACELEVEL_FATAL, "Error while calling OSAL_C_S32_IOCTRL_GYRO_GET_HW_INFO");
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "sucess OSAL_C_S32_IOCTRL_GYRO_GET_HW_INFO");

         vVdsCopyOsalToFiHwInfo(
            rIOCtrlHwInfo,
            tclSystemInformation::rInfo.rGyro.r3dGyroHWInfo.MountAngles,
            tclSystemInformation::rInfo.rGyro.r3dGyroHWInfo.RAxes,
            tclSystemInformation::rInfo.rGyro.r3dGyroHWInfo.SAxes,
            tclSystemInformation::rInfo.rGyro.r3dGyroHWInfo.TAxes );

         vTraceMsg( VDS_C_TRACELEVEL_DATA,"f32DriftOffset= %f",rIOCtrlHwInfo.rRAxes.f32DriftOffset);

      }

      //Allocate memory for gyro data.
      // storage for reading from gyro-device
      u32NumOfGyroData = tclSystemInformation::rInfo.rGyro.u16BlockSize;
      
      vTraceMsg( VDS_C_TRACELEVEL_DATA,"Gyro Data Size = %d",u32NumOfGyroData);

      parGyroData = new OSAL_trIOCtrl3dGyroData[ u32NumOfGyroData ];

      //Storage for sending data to gyro-if
      par3dGyroStorage = new sensor_fi_tcl_3dGyroData[ u32NumOfGyroData ];

      // if error occured...
      if ( (OSAL_NULL == parGyroData) || (OSAL_NULL == par3dGyroStorage) )
      {
         s32RetVal = VDS_E_GYROTHREAD_ALLOC_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GyroThread:Init:Alloc failed");
      }

      //GYRO main thread attributes.
      OSAL_trThreadAttribute rThreadAttribute = {0};
      rThreadAttribute.szName  = const_cast<tString>(THREADNAME_C_STRING_SENSOR_VDS_GYRO); // Name
      rThreadAttribute.pfEntry = ( OSAL_tpfThreadEntry ) tclGyroThread::vThreadMainWrapper;
      rThreadAttribute.pvArg   = ( tPVoid ) this; // Argument der Startroutine

      //Get GYRO main thread configuration
      if ( OSAL_ERROR ==
           scd_s32GetThreadConfiguration( CCA_C_U16_APP_SENSOR,
                                          THREADNAME_C_STRING_SENSOR_VDS_GYRO,
                                          &rThreadAttribute.u32Priority,
                                          &rThreadAttribute.s32StackSize ) )
      {
         s32RetVal = VDS_E_GYROTHREAD_THREADCREATE_FAILED;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GyroThread:Init:Unable to get thread config");
      }

      //Create GYRO main thread
      if ( s32RetVal == VDS_E_NO_ERROR )
      {
         if ( OSAL_ERROR == ( SensorThreadId = OSAL_ThreadCreate( &rThreadAttribute ) ) )
         {
            s32RetVal = VDS_E_GYROTHREAD_THREADCREATE_FAILED;
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "GyroThread:Init:Creating thread failed error = 0x%x",
                       OSAL_u32ErrorCode());
         }
      }

      //does error occured?
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         vThreadClose();
      }
   }
   else
   {
      s32RetVal = VDS_E_GYROTHREAD_NOT_UNINITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GyroThread:Init:Already initialized");
   }
   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.when thread is started, set state RUNNING.
//!   receive gyro-data from gyro-device when internal state is set to STOP,
//!   leave loop and set state INITIALIZED.
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGyroThread::vThreadMain
(
   //! void
   tPVoid
   )
{
   tU32 u32TempNumOfGyroData = 0;
   OSAL_trIOCtrl3dGyroData *parTempGyroData = OSAL_NULL;
   tU32 u32ErrorCount = 0;
   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempGyroData);

   if ( ( s32GetInternalState() == VDS_C_S32_THREAD_INITIALIZED ) &&
        ( OSAL_ERROR != SensorFd ) &&
        ( 0 < u32NumOfGyroData ) &&
        ( OSAL_NULL != parGyroData )
      )
   {
      //Thread is activated.
      vSetInternalState( VDS_C_S32_THREAD_RUNNING );

      //Read available data from gyro-device.
      if ( OSAL_ERROR  == OSAL_s32IOControl( SensorFd,
                                             OSAL_C_S32_IOCTRL_GYRO_GETCNT,
                                             (tLong)&u32TempNumOfGyroData ) )
      {
         //Read u32NumOfGyroData datas (=VDS_C_U16_GYROTHREAD_BLOCKREADSIZE)
         u32TempNumOfGyroData = u32NumOfGyroData;
      }

      //At the first time read,there would be more gyro-data than u32NumOfGyroData.
      if ( u32TempNumOfGyroData > u32NumOfGyroData )
      {
         parTempGyroData =  new OSAL_trIOCtrl3dGyroData[ u32TempNumOfGyroData ];

         if( OSAL_NULL == parTempGyroData )
         {
            //Creation of temp buffer failed, so use parGyroData buffer.
            parTempGyroData = parGyroData;
            u32TempNumOfGyroData = u32NumOfGyroData;
         }
      }
      else
      {
         //Buffer is big enough, use it.
         parTempGyroData = parGyroData;
      }

      //Read and store gyro data
      s32ReadGyroData( parTempGyroData, u32TempNumOfGyroData );

      //Delete temp-buffer.
      if ( parTempGyroData != parGyroData )
      {
         delete [] parTempGyroData;
         parTempGyroData = OSAL_NULL;
      }

      //Main Loop
      do
      {
         // consistence check
         if ( ( OSAL_ERROR != SensorFd ) && ( OSAL_NULL != parGyroData ) )
         {
            tS32 s32Ret = s32ReadGyroData( parGyroData, u32NumOfGyroData );

            if ( s32Ret == VDS_E_GYROTHREAD_IOREAD_ERROR )
            {
               if ( ++u32ErrorCount == 10 )
               {
                  et_vErrmemStringNormal( TR_COMP_SENSOR,
                                          "GyroThread:No GYRO data for 10 successive reads!");
               }
            }
            else
            {
               u32ErrorCount = 0;
            }
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "GyroThread:IO or Memory uninit." );

            vSetInternalState( VDS_C_S32_THREAD_ERROR );
         }

      }while ( s32GetInternalState() == VDS_C_S32_THREAD_RUNNING );

      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "GyroThread:Exit main Loop!");

      //set new state
      if ( s32GetInternalState() == VDS_C_S32_THREAD_STOP )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         //close handles, ...
         vThreadClose();
         vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
      }
   }

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTempGyroData);

   OSAL_vThreadExit();
   // else no error-handling needed, state is uninitialized
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   inherited from class tclSensorThread.closes handles, deletes semaphores,
//!   deletes allocated memory
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGyroThread::vThreadClose
(
   //! void
   )
{
   // if gyro-data exists, delete
   if ( OSAL_NULL != parGyroData )
   {
      delete [] parGyroData;
      parGyroData = OSAL_NULL;
   }

   // if gyro-storage exists, delete
   if ( OSAL_NULL != par3dGyroStorage )
   {
      delete [] par3dGyroStorage;
      par3dGyroStorage = OSAL_NULL;
   }

   u32NumOfGyroData = 0;
   // semaphore and device will be deleted in tclSensorThread
   // reset function-pointer

   tclSensorThread::vThreadClose();
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   reads gyro-data from gyro-device,
//! \return
//!   Number of Elements read
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROTHREAD_STORAGE_ERROR     : Storage error
//!   - \c VDS_E_GYROTHREAD_IOREAD_ERROR :  IO Read error
//!   - \c VDS_E_GYROTHREAD_NOT_INITIALIZED : Not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclGyroThread::s32ReadGyroData
(
   //! (O) : buffer for received gyro-data
   OSAL_trIOCtrl3dGyroData *pGyroData,
   //! (I) : max. number of gyro-data to be stored in buffer
   tU32 u32NumOfElem
   )
{

   tS32 s32NumOfElemRead = 0;

   // param-check
   if ( ( OSAL_NULL != pGyroData ) &&
        ( 0 < u32NumOfElem ) &&
        ( 10000 > u32NumOfElem ) )
   {
      // consistence-check
      if ( OSAL_ERROR != SensorFd )
      {
         //Read gyro-data
         tS32 s32GyroDataRead =
            OSAL_s32IORead( SensorFd,
                            (tPS8)pGyroData,
                            (u32NumOfElem * (tU32) sizeof( OSAL_trIOCtrl3dGyroData )) );

         s32NumOfElemRead =
            s32GyroDataRead /(tS32)sizeof( OSAL_trIOCtrl3dGyroData );

         if( (s32NumOfElemRead > 0)&&((tU32)s32NumOfElemRead <= u32NumOfElem) )
         {
            vTraceMsg( VDS_C_TRACELEVEL_DATA,
                        "GyroValues R=%d,S=%d,T=%d (ErrCnt=%d, TimeStamp=%d)",
                        pGyroData->u16Gyro_r,pGyroData->u16Gyro_s,pGyroData->u16Gyro_t,
                        pGyroData->u16ErrorCounter,pGyroData->u32TimeStamp);

            if ( VDS_E_NO_ERROR
                 !=
                 s32StoreGyroData( pGyroData,(tU32)s32NumOfElemRead ) )
            {
               s32NumOfElemRead = VDS_E_GYROTHREAD_STORAGE_ERROR;
            }
         }
         else
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR, "GyroThread: IORead failed" );
            s32NumOfElemRead = VDS_E_GYROTHREAD_IOREAD_ERROR;
         }
      }
      else
      {
         s32NumOfElemRead = VDS_E_GYROTHREAD_NOT_INITIALIZED;
      }
   }
   else
   {
      s32NumOfElemRead = VDS_E_INVALID_PARAMETER;
   }

   return s32NumOfElemRead;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   sends received gyro-data to tclMsgGyroIf
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GYROTHREAD_ALLOC_ERROR :   Allocation error
//!   - \c VDS_E_GYROTHREAD_STORAGE_ERROR :  Storage Error
//!   - \c VDS_E_GYROTHREAD_NOT_INITIALIZED : Not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid parameter
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclGyroThread::s32StoreGyroData
(
   //! (I) : buffer with received gyro-data
   const OSAL_trIOCtrl3dGyroData *pGyroData,
   //! (I) : number of gyro-data in buffer
   tU32 u32NumOfElem
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;
   sensor_fi_tcl_3dGyroData *parTemp3dGyroStorage = OSAL_NULL;
   tU32 u32GyroIndex = 0;

   VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dGyroStorage);

   //Param-check
   if ( ( OSAL_NULL != pGyroData ) && ( u32NumOfElem   > 0 ) )
   {
      //Consistence check
      if ( OSAL_NULL != par3dGyroStorage )
      {
         // if too much data, generate temp storage (expected for the first time)
         if ( u32NumOfElem > u32NumOfGyroData )
         {
            parTemp3dGyroStorage = new sensor_fi_tcl_3dGyroData[u32NumOfElem];

            if ( OSAL_NULL == parTemp3dGyroStorage )
            {
               s32RetVal = VDS_E_GYROTHREAD_ALLOC_ERROR;
               vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                          "GyroThread:s32Store3dGyroData: alloc error" );
            }
         }
         else
         {
            parTemp3dGyroStorage = par3dGyroStorage;
         }

         if ( OSAL_NULL != parTemp3dGyroStorage )
         {
            //copy gyro-device-format to gyro-vdsensor-format
            for ( u32GyroIndex = 0;
                  u32GyroIndex < u32NumOfElem;
                  u32GyroIndex++ )
            {
               //copy first 3d data plain to the 3d Buffer
               parTemp3dGyroStorage[u32GyroIndex].TimeStamp3dGyro =
                  pGyroData[u32GyroIndex].u32TimeStamp;

               parTemp3dGyroStorage[u32GyroIndex].RVal =
                  pGyroData[u32GyroIndex].u16Gyro_r;

               parTemp3dGyroStorage[u32GyroIndex].SVal =
                  pGyroData[u32GyroIndex].u16Gyro_s;

               parTemp3dGyroStorage[u32GyroIndex].TVal =
                  pGyroData[u32GyroIndex].u16Gyro_t;

               if( pGyroData[ u32GyroIndex ].u16ErrorCounter > 0 )
               {
                  parTemp3dGyroStorage[u32GyroIndex].RStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;

                  parTemp3dGyroStorage[u32GyroIndex].SStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;

                  parTemp3dGyroStorage[u32GyroIndex].TStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_INTERNALERROR;
               }
               else
               {
                  parTemp3dGyroStorage[u32GyroIndex].RStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_NORMAL;

                  parTemp3dGyroStorage[u32GyroIndex].SStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_NORMAL;

                  parTemp3dGyroStorage[u32GyroIndex].TStatus.enType =
                     sensor_fi_tcl_e16_GyroStatus::FI_EN_GYROSTATE_CONNECTED_NORMAL;
               }
            }

            if( ( pfs32Add3dGyroList != OSAL_NULL ) &&
                ( VDS_E_NO_ERROR != pfs32Add3dGyroList( parTemp3dGyroStorage,u32NumOfElem )) )
            {
               s32RetVal = VDS_E_GYROTHREAD_STORAGE_ERROR;
            }

            //Delete 3D temp buffer
            if ( parTemp3dGyroStorage != par3dGyroStorage )
            {
               delete [] parTemp3dGyroStorage;
               parTemp3dGyroStorage = OSAL_NULL;
            }
         }
         else
         {
            s32RetVal = VDS_E_GYROTHREAD_NOT_INITIALIZED;
         }
      }
      else
      {
         s32RetVal = VDS_E_INVALID_PARAMETER;
      }

      VDS_PREVIOUSLY_ASSIGNED_VALUE_INTENTIONALLY_UNUSED(parTemp3dGyroStorage);
   }
   else
   {
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   return s32RetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   get the gyro temperature values
//! \return
//!   - \c  tF32 : temperature in kelvin
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tF32 tclGyroThread::f32GetGyroTemperature
(
   //! (I) :void
   tVoid
   )
{
   if( bCalcTemperature( &f32TempKelvin ) )
   {
      return f32TempKelvin;
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,
                 "GyroThread:bCalcTemperature is failed" );
      return 0.f;
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Calculates the current temperature in Kelvin from temp sensor and ADC.
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of Failure\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGyroThread::bCalcTemperature
(
   //! (O) : Pointer to Temperature value
   tF32  *f32Temperature
   )
{
   tS32 s32Retval ; // Raw ADC value

   s32Retval =  OSAL_s32IOControl( SensorFd,
                                   OSAL_C_S32_IOCTRL_GYRO_GET_TEMP,
                                   (tLong)f32Temperature);
   if(s32Retval == OSAL_OK )
   {
      vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,
                 "GyroThread:GYRO Temp = %f K",(*f32Temperature) );
      return true;
   }
   else
   {
      *f32Temperature = 0.f;

      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GyroThread:Unable to get Gyro Temperature.");

      return false;
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   static wrapper for main-loop of instance
//! \return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGyroThread::vThreadMainWrapper
(
   //! (I) :  pointer to instance
   tPVoid pvThreadArg
   )
{
   // param-check
   if ( OSAL_NULL != pvThreadArg )
   {
      tclGyroThread *poGyroThread =
         static_cast<tclGyroThread *>( pvThreadArg );
      // Can not be NULL
      poGyroThread->vThreadMain( OSAL_NULL );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GyroThread:ThreadArgument NULL");
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns gyro-sensor-cycle-time\n
//! \return
//!      Sensor cycle time
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tU32 tclGyroThread::u32GetGyroCycleTime
(
   //! None
   tVoid
   )
{
   return u32GyroSensorCycleTimeInNS;
}


// ***************** F U N C T I O N  H E A D E R *****************************
//! \brief
//!       Start Gyro self test.
//! \return
//!   - \c TRUE  : Self test finished successfully.
//!   - \c FALSE : Unable to start self test.
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGyroThread::bStartSelftest
(
   //! None
   tS32 *ps32SelfTestResult
   )
{
   tBool bRetVal = FALSE;
   tS32 s32SelfTestResult = -1;

   if( OSAL_OK == OSAL_s32IOControl( SensorFd,
                                     OSAL_C_S32_IOCTRL_GYRO_SELF_TEST,
                                     (tLong)&s32SelfTestResult ) )
   {
      bRetVal = TRUE;
      *ps32SelfTestResult = s32SelfTestResult;
   }

   return bRetVal;
}

//End Of File.
