/*************************************************************************//**
 *  \file    tclGpsThread.h
 *  \par     SW_COMPONENT   VD-Sensor
 *  \brief   class-declaration: inherits from tclSensorThread,
 *           thread-main-loop, reads data from GPS device.
 *  \author  CM-DI/ESA1-Fischer
 *  \par     COPYRIGHT (c) 2002 Blaupunkt GmbH
 *----------------------------------------------------------------------------
 * \date 20.03.02  CM-DI/ESA1-Fischer  -    Rev. 1.0  Initial Revision.
 *----------------------------------------------------------------------------
 * \date 07.08.02  CM-CR/EES4-Bode     -    Adapted to SIRF-GPS on Tuareg.
 *----------------------------------------------------------------------------
 * \date 2005-2006     oej2hi          -    Ported to Paramount.
 *----------------------------------------------------------------------------
 * \date 30.09.09  RBEI/ECF1-Jeryn Mathew
 *       - Added doxygen headers for all classes/funcs
 *----------------------------------------------------------------------------
 * \date 23.09.10  RBEI/ECF1 - sak9kor - modifications related to calculation
 *    of Antenna Status and Temperature have been done for ADIT Gen2 platform
 *----------------------------------------------------------------------------
 * \date 16.12.10 Rev. 1.2 RBEI/ECF1 - Sainath Kalpuri
 *     - Diaglog related modifications have been done.
 *----------------------------------------------------------------------------
 * \date 04.02.10 RBEI/ECF1-sak9kor - tclGPIOif is put under TRITON macro
 *----------------------------------------------------------------------------
 * \date 04.05.11 RBEI CM-AI/PJ-CF31 - Sainath To fix MMS 294524
 *----------------------------------------------------------------------------
 * \date 09.11.12 RBEI/ECF5 - Sainath To fix issue NIKAI-231
 *----------------------------------------------------------------------------
 * \date 19.09.13 RBEI/ECF5-Sanajy G -
 * Modifications done to adopt VDSensor according to the CmbFi version-5.0.0
 *----------------------------------------------------------------------------
 * \date 10.07.14 RBEI/ECF5-Madhu Kiran - Enabled China encoder for VDS2
 *----------------------------------------------------------------------------
 * \date 03.14.16 Sanjay G(RBEI/ECF5) - Added methods vSetGpsThreadFlags() 
 * and vUpdateCrcCheckSumToReg(). Updated indentation and headers of all
 * methods to doxygen style. Removed unused code. Moved macros to the file
 * vds_defines.h.
 *----------------------------------------------------------------------------
 * \date 03.01.17 RBEI/ECF12-Arun Magi(amg6kor) - Removing the /dev/gps support and port to Gen4
 ****************************************************************************/

#ifndef tclGpsThread_h
#define tclGpsThread_h 1

#define OSAL_S_IMPORT_INTERFACE_THREADING
#include "osal_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#include "system_pif.h"

#include "vds_trace.h"
#include "vds_types.h"

/*!***************************************************************************
 * \brief Interface to a GPS coordinate encoder.
 *
 * The People's Republic of China provides digital map data with slightly
 * perturbed coordinates in order to protect its national security interests.
 * In order to use such map data in a navigation system, the GPS coordinates
 * have to be perturbed using the same algorithm.  The algorithm is provided
 * by the authorities of the PRC. The possibility that other countries may
 * require similar procedures is open.
 *
 * This interface is used by VD Sensor to encapsulate access to such encoders
 * In the case that no encoder is present in the system,the functions of this
 * interface will have no effect.
 ****************************************************************************/
class tclGpsEncoderIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Default constructor
      //! \param   void
      //! \return  void
      // *********************************************************************
      virtual ~tclGpsEncoderIf( ) {}

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Initialize china encoder. This function performs any
      //!          encoder-specific actions needed for initialization.
      //! \param   void
      //! \return  true  : In case of success.
      //!          false : In case of failure.
      //**********************************************************************
      virtual tBool bInitialize( tVoid ) = 0;

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Shuts down the encoder. This function performs any encoder
      //!          specific actions (if any) needed to shut down the encoder.
      //!          It is expected that if this function returns successfully,
      //!          bInitialize() can be called.
      //! \param   void
      //! \return  true : In case of success.
      //!          false: In case of failure.
      //**********************************************************************
      virtual tBool bShutdown( tVoid ) = 0;

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Checks an encoder available or not.
      //! \param  void
      //! \return true  : If an encoder is available.
      //!         false : If an encoder is not available.
      //**********************************************************************
      virtual tBool bAvailable( void ) = 0;

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Encodes given GPS position.
      //! \param  dRawLatitude
      //!         dRawLongitude
      //!         dRawAltitude
      //!         u16GpsWeek
      //!         u32GpsSecond
      //!         rfdConvertedLatitude
      //!         rfdConvertedLongitude
      //! \return true  : In case of success.
      //!         false : In case of failure.
      //**********************************************************************
      virtual tBool bEncode
      (
         //!  (I) : WGS84 Latitude in radians
         tDouble dRawLatitude,
         //!  (I ) : WGS84 Longitude in radians
         tDouble dRawLongitude,
         //! (I) : WGS84 Altitude in meters
         tDouble dRawAltitude,
         //! (I) : GPS week number
         tU16 u16GpsWeek,
         //! (I ) : Second within the GPS week
         tU32 u32GpsSecond,
         //! (O) : Encoded latitude. This is the value 
         //!       that should be sent to client applications
         tDouble &rfdConvertedLatitude,
         //! (O) : Encoded longitude. This is the value
         //!       that should be sent to client applications
         tDouble &rfdConvertedLongitude
      ) = 0;

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Enable resp. disable the china encoder. A requirement of
      //!          the production is to disable and enable the encoder
      //!          without a reset.
      //! \param   bInhibit: (I) : Inhibit, TRUE: inhibited; FALSE encoder
      //!                          is enabled.
      //! \return  true : In case of success.
      //!          false: In case of failure.
      //**********************************************************************
      virtual tBool bEnDisChinaEnc( tBool bInhibit ) = 0;
};

/*****************************************************************************
 *\brief Sensor thread class for gnss/gps.
 ****************************************************************************/
class tclGpsThread : public tclSensorThread
{

   public:
      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Default constructor
      //! \param   void
      //! \return  void
      // *********************************************************************
      tclGpsThread();

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Default destructor
      //! \param   void
      //! \return  void
      // *********************************************************************
      virtual ~tclGpsThread();

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  This method initialises all the necessary resources
      //!         like buffers, semaphores, events, opens aux clock and
      //!         finally creates the data collecting thread of gnss
      //!
      //! \param  (*pfs32AddGnssDataParam)(OSAL_trGnssFullRecord *) : pointer
      //!          to the function which updates gnss data to vd_sensor
      //!          gnss data ring buffer.
      //! \return  VDS_E_NO_ERROR : in case of success.
      //!          following error codes in case of failure.
      //!               VDS_E_INVALID_PARAMETER
      //!               VDS_E_GPSTHREAD_DEVICE_NOT_FOUND
      //!               VDS_E_GPSTHREAD_SEM_ERROR
      //!               VDS_E_GPSTHREAD_THREADCREATE_FAILED
      //!               VDS_E_GPSTHREAD_NOT_UNINITIALIZED
      //! \author  CM-DI/ESA1-Fischer   \date    21.Mar.2002
      //**********************************************************************
      tS32 s32ThreadInit( tS32
                          (*pfs32AddGnssDataParam)(OSAL_trGnssFullRecord *));


      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Inherited from class tclSensorThread. collects data from
      //!          gnss proxy and store it in vd_sensor ring buffer.
      //! \param   pvThreadArg: Thread start argument. not used( null ).
      //! \return  void
      //**********************************************************************
      tVoid vThreadMain( tPVoid pvThreadArg );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Updates checksum of gnss device firmware to registry.
      //! \param   void
      //! \return  void
      //! \author  Sanjay G(RBEI/ECF5)   \date    11.Mar.2016
      //**********************************************************************
      tVoid vUpdateCrcCheckSumToReg( tVoid );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Flushes gnss buffer on scc.
      //! \param   void
      //! \return  true  : incase of success.
      //!          false : incase of failure.
      //! \author  Sanjay G(RBEI/ECF5)   \date    11.Mar.2016
      //**********************************************************************
      tBool bFlushGnssBuffer( tVoid );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Static wrapper for main-loop of instance.
      //! \param   pvThreadArg: pointer to instance
      //! \return  void
      //! \author  CM-DI/ESA1-Fischer    \date    16.Apr.2002
      //**********************************************************************
      static tVoid vThreadMainWrapper( tPVoid pvThreadArg );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Inherited from class tclSensorThread. closes handles,
      //!          deletes semaphores, deletes allocated memory.
      //! \param   void
      //! \return  void
      //! \author  CM-DI/ESA1-Fischer    \date    09.Apr.2002
      //**********************************************************************
      virtual tVoid vThreadClose( tVoid );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Returns current current time of day (minutes in day)
      //! \param   rfu16TimeOfDay: updates time of the day in minutes.
      //! \return  true : if time of day is available.
      //!          false: if time of day is not available.
      //! \author  CM-CR/EES4-Bode       \date    08.Apr.2003
      //**********************************************************************
      tBool bGetTimeOfDay( tU16 &rfu16TimeOfDay );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Sets time of the day
      //! \param   rfu16TimeOfDay: time of the day in minutes( input )
      //! \return  true  : time of day changed
      //!          false : error while setting time of day or feature not
      //!                  available for this receiver
      //! \author  CM-CR/EES4-Bode       \date  08.Apr.2003 
      //**********************************************************************
      tBool bSetTimeOfDay ( const tU16 &rfu16TimeOfDay );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Starts gnss device.
      //! \param  void.
      //! \return true  : in case of success.
      //!         false : In case of failure.
      //**********************************************************************
      tBool bStartDevGnss();

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Stops dev gnss
      //! \param  void
      //! \return true : in case of success.
      //!         false: in case of failure.
      //**********************************************************************
      tBool bStopDevGnss( tVoid );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Sets gps thred event with the event provided.
      //! \param   u8Event: event to be set. It can take the following states.
      //!          VDS_GPS_EVENT_NOEVENT
      //!          VDS_GPS_EVENT_RESTART
      //!          VDS_GPS_EVENT_START
      //!          VDS_GPS_EVENT_STOP
      //!          VDS_GPS_EVENT_SEND_OSAL_TIME_UPDATE
      //!          VDS_GPS_EVENT_FRONT_END_TEST
      //!          VDS_GPS_EVENT_FACTORY_SETTINGS
      //! \return  void
      //**********************************************************************
      tVoid vSetGpsThreadEvent ( tU8 u8Event );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Sets the application state with the provided state
      //!          information.
      //! \param   u32AppState : Application state information to be set.
      //!          it can take any of the following values.
      //!          AMT_C_U32_STATE_INITIALIZED
      //!          AMT_C_U32_STATE_NORMAL
      //!          AMT_C_U32_STATE_DIAGNOSIS
      //!          AMT_C_U32_STATE_RECEIVE_READY
      //!          AMT_C_U32_STATE_PAUSE
      //!          AMT_C_U32_STATE_OFF
      //!          AMT_C_U32_STATE_PREPARE_DOWNLOAD
      //! \return  void
      //**********************************************************************
      tVoid vSetAppState( tU32 u32AppState );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Gets application current state information
      //! \param   void
      //! \return  current application state. it can take any of the
      //!          following states.
      //!          AMT_C_U32_STATE_INITIALIZED
      //!          AMT_C_U32_STATE_NORMAL
      //!          AMT_C_U32_STATE_DIAGNOSIS
      //!          AMT_C_U32_STATE_RECEIVE_READY
      //!          AMT_C_U32_STATE_PAUSE
      //!          AMT_C_U32_STATE_OFF
      //!          AMT_C_U32_STATE_PREPARE_DOWNLOAD
      //**********************************************************************
      tU32 u32GetAppState( tVoid ) const;

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Gets osal time status
      //! \param  void
      //! \return osal time status. it can return following status values.
      //!         VDS_C_U8_OSAL_CLOCK_NOT_SET
      //!         VDS_C_U8_OSAL_CLOCK_SET_TO_DEFAULT
      //!         VDS_C_U8_OSAL_CLOCK_SET_TO_USER_INPUT
      //!         VDS_C_U8_OSAL_CLOCK_ADJ_BY_USER_RTC
      //!         VDS_C_U8_OSAL_CLOCK_ADJ_BY_USER
      //!         VDS_C_U8_OSAL_CLOCK_ADJ_BY_GPS_RTC
      //!         VDS_C_U8_OSAL_CLOCK_ADJ_BY_GPS
      //! \note   some of the states are not used/not necessary. consider
      //!         cleaning unneeded status values.
      //**********************************************************************
      tU8 u8GetOsalTimeStatus( tVoid );

   protected:

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Reads timestamp from aux clock device and returns the lower
      //!         32bit of this timestamp ( resolution could be received from
      //!         aux clock device).
      //! \return tU32 - actual timestamp
      //! \author CM-DI/ESA1-Fischer       \date    21.Mar.2002
      //**********************************************************************
      tU32 u32GetTimestamp( tVoid );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Syncs system time with gps time.
      //! \param  pointer to the data of gps time.
      //! \return true : in case of success.
      //!         false: in case of failure.
      //**********************************************************************
      tBool bSyncTime( const OSAL_trGnssTimeUTC *rTimeUTC );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Sets the system time.
      //! \param  Pointer to the data of gps time.
      //! \return true : in case of success.
      //!         false: in case of failure.
      //**********************************************************************
      tBool bSetSystemTime( const OSAL_trTimeDate &rfTargetTimeDate );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Calculates the number of seconds and number of weeks based
      //!         on the parameter provided i.e. year, month, day,
      //!         hour, minute,second.
      //! \param  usYear   - Year
      //!         ucMonth  - Month
      //!         ucDay    - Day
      //!         ucHour   - Hour
      //!         ucMinute - Minute
      //!         ucSecond - Second
      //!         pusWn    - GPS week time stamp
      //!         pulTow   - GPS time of the week
      //! \return void
      //**********************************************************************
      tVoid vUtcToGps( tU16 usYear,   tU8  ucMonth,  tU8  ucDay,
                       tU8  ucHour,   tU8  ucMinute, tU8  ucSecond,
                       tU16 *pusWn,   tU32 *pulTow );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Converts the time in to day, month and year to gps time.
      //! \param   ulMsec    - Msec
      //!          usWeekNum - Week number
      //!          pusDay    - Day
      //!          pusMonth  - Month
      //!          pusYear   - Year
      //! \return  void
      //**********************************************************************
      tVoid vConvertGpsTime ( tU32  ulMsec,  tU16 usWeekNum,
                              tU16  *pusDay, tU16 *pusMonth, tU16 *pusYear );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Calculates the day, month, year, hour, minute and seconds
      //!         based on the parameters provided i.e. week time stamp and
      //!         time of the week.
      //! \param  usWn      - GPS week time stamp
      //!         ulTow     - GPS time of the week
      //!         pucDay    - Pointer to Day
      //!         pucMonth  - Pointer to Month
      //!         pusYear   - Pointer to Year
      //!         pucHour   - Pointer to Hour
      //!         pucMinute - Pointer to Minute
      //!         pucSecond - Pointer to Second
      //! \return void
      //**********************************************************************
      tVoid vGpsToUtc( tU16 usWn,    tU32 ulTow,
                       tU8 *pucDay,  tU8 *pucMonth,  tU16 *pusYear,
                       tU8 *pucHour, tU8 *pucMinute, tU8 *pucSecond );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Checks the Gps thread event and performs the respective
      //!         activity based on the event.
      //! \param  void
      //! \return true  : in case of success.
      //!         false : in case of failure.
      //**********************************************************************
      tBool bCheckGpsThreadEvent( tVoid );

   private:
      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Sends GNSS data to tclMsgGpsIf there it will be stored 
      //!         in ring buffer.
      //! \param  prGnssFullRecord : pointer which holds data needs to be
      //!                            copied to ring buffer.
      //! \return VDS_E_NO_ERROR : In case of success.
      //!         Returns following error codes in case of failure.
      //!             VDS_E_GPSTHREAD_STORAGE_ERROR.
      //!             VDS_E_GPSTHREAD_NOT_INITIALIZED.
      //! \note   Consider removing this method. Insted of calling this
      //!         function, directly call (*pfs32AddGnssData)().
      //**********************************************************************
      tS32 s32StoreGnssData(OSAL_trGnssFullRecord *prGnssFullRecord);

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Encodes given coordinates.
      //! \param  prGnssFullRecord: pointer which holds coordinates info
      //!                           which needs to be encoded.
      //! \return void
      //**********************************************************************
      tVoid vds_vEncodeToChinaPos( OSAL_trGnssFullRecord *prGnssFullRecord);

   public:
      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   The command input of trace is executed.
      //!          here in the command for destination of osal device
      //!          dev_gnss is carried out.
      //! \param   puchData : contains the byte buffer transmitted via trace.
      //! \return  OSAL_OK    : in case of success.
      //!          OSAL_ERROR : In case of error.
      //**********************************************************************
      tS32 bExecuteGpsTraceCommand( const tUChar* puchData );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   enable resp. disable the china encoder.
      //! \return  true : in case of success.
      //!          false: In case of failure.
      //! \note    requirement of the production is to disable and enable
      //!          the encoder without a reset.
      //**********************************************************************
      tBool bEnDisChinaEnc( tBool inhibit );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Gets the gnss satellite system currently used for fix.
      //! \param  void
      //! \return the actual used satellite systems.
      //!         if no satellite system could be determined it retruns 0.
      //!         value of type \c sensor_fi_tcl_b8_GnssSatSys.
      //**********************************************************************
      tU8 u8GetGnssSatSys ( tVoid );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Sets gnss satellite system to be used for position
      //!         calculation.
      //! \return if success returns the satellite system.
      // *********************************************************************
      tU8 u8SetGnssSatSys ( tU8 u8TargetGnssSatSys );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Sets the gnss satellite system used system used
      //!         NON-persistently. i.e. If you use this call to set satellite
      //!         system then it would be effective in the current power cycle
      //!         only.In the next power cycle previously configured satellite
      //!         system will be used.
      //! \return Method returns the actual used satellite systems.
      //!         If no satellite system could be determined it retruns 0.
      //!         value of type \c sensor_fi_tcl_b8_GnssSatSys.
      // *********************************************************************
      tU8 u8SetDiagGnssSatSys ( tU8 u8TargetGnssSatSys );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief Sets gnss epoch.
      //! \return  true  : incase of success.
      //!          false : incase of failure.
      //**********************************************************************
      tBool bSetWnEpoch( const OSAL_trGnssTimeUTC &rTargetEpochUTC,
                         OSAL_trGnssTimeUTC &rReturnActualDateUTC );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Gets the epoch.
      //! \return  true  : incase of success.
      //!          false : incase of failure.
      //**********************************************************************
      tBool bGetWnEpoch( const OSAL_trGnssTimeUTC &rActualDateUTC );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Sets the timemode.
      //! \return  true  : incase of success.
      //!          false : incase of failure.
      //**********************************************************************
      tBool bSetTimeMode ( const vds_tenTimeMode &enTargetTimeMode, 
                           const OSAL_trTimeDate &rfTargetTimeDate );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Gets actual time mode and actual time.
      //! \return  true  : incase of success.
      //!          false : incase of failure.
      // *********************************************************************
      tBool bGetTimeMode ( vds_tenTimeMode &rfenActualTimeMode );

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief   Get gnss configuration data.
      //! \return  true  : incase of success.
      //!          false : incase of failure.
      // *********************************************************************
      tBool bGetGnssConfigData(OSAL_trGnssConfigData *rGnssConfigData);

      // ***************** F U N C T I O N  H E A D E R **********************
      //! \brief  Sets different flags in tclGpsThread.
      //! \return void
      // *********************************************************************
      tVoid vSetGpsThreadFlags( vds_tenGpsThreadFlags  enFlag,
                                tBool                  bFlagStatus );

   private:
      //!*********************************************************************
      //!            Private data members for class attributes.
      //!*********************************************************************

      /*!
       * \brief Flag to indicate running status of gps/gnss main thread.
       *        true  - thread is running.
       *        false - thread is dead.
       */
      tBool                     bGpsRunning;

      /*!
       * \brief Flag to indicate whether gnss buffer on scc is flushed
       *        or not???
       *        Because of scarcity of memory on SCC( system communication
       *        controller ), gnss buffer on SCC is used to buffer start up
       *        data of inertial sensors( acc/gyro/odo/abs ). since any way
       *        GNSS fix takes some time and also initial fix info is not
       *        needed for navi. but the gyro data is needed. this flag can
       *        take following values.
       *        true  - buffer should be flushed.
       *        false - no need to flush buffer.
       */
      tBool                     bFlushGnssBuf;
      
      /*!
       * \brief  Holds different events of GpsThread. These events occur
       *         because of different state changes of vd_sensor. It can
       *         take the following states.
       *            VDS_GPS_EVENT_NOEVENT                0
       *            VDS_GPS_EVENT_RESTART                1
       *            VDS_GPS_EVENT_START                  2
       *            VDS_GPS_EVENT_STOP                   3
       *            VDS_GPS_EVENT_SEND_OSAL_TIME_UPDATE  4
       *            VDS_GPS_EVENT_FRONT_END_TEST         5
       *            VDS_GPS_EVENT_FACTORY_SETTINGS       6
       */
      tU8                       u8GpsThreadEvent;

      /*!
       * \brief Holds current application state. it can take any of the
       *        following states.
       *             AMT_C_U32_STATE_INITIALIZED
       *             AMT_C_U32_STATE_NORMAL
       *             AMT_C_U32_STATE_DIAGNOSIS
       *             AMT_C_U32_STATE_RECEIVE_READY
       *             AMT_C_U32_STATE_PAUSE
       *             AMT_C_U32_STATE_OFF
       *             AMT_C_U32_STATE_PREPARE_DOWNLOAD
       */
      tU32                      u32ApplicationState;

      /*!
       * \brief Holds the status of osal clock. It can take any of the
       *        following states.
       *             VDS_C_U8_OSAL_CLOCK_NOT_SET
       *             VDS_C_U8_OSAL_CLOCK_SET_TO_DEFAULT
       *             VDS_C_U8_OSAL_CLOCK_SET_TO_USER_INPUT
       *             VDS_C_U8_OSAL_CLOCK_ADJ_BY_USER_RTC
       *             VDS_C_U8_OSAL_CLOCK_ADJ_BY_USER
       *             VDS_C_U8_OSAL_CLOCK_ADJ_BY_GPS_RTC
       *             VDS_C_U8_OSAL_CLOCK_ADJ_BY_GPS
       */
      tU8                       u8OsalTimeStatus;

      //! \brief Holds time if it is set manually.
      OSAL_trTimeDate           rManualDateTime;

      /*!
       * \brief Holds mode of the time. It can take the following values from
       *        the enum vds_tenTimeMode. This info is used to decide how to
       *        set the system time.
       *        enTimeModeDefault = 0UL,
       *        enTimeModeAuto    = 1UL,
       *        enTimeModeManual  = 2UL,
       *        enTimeModeInvalid = 254UL
       */
      vds_tenTimeMode           enTimeMode;

      //! \brief The OSAL name used for hEvGpsState.
      tCString                  szEvGpsStateName;

      /*!
       * \brief An event group used to signal the on/off state of the
       * GPS receiver.
       *
       * The valid events are:
       * - VDS_C_GPSTHREAD_EV_ON
       * - VDS_C_GPSTHREAD_EV_OFF
       *
       * These events are set by the GPS thread when the receiver is
       * turned on or off, and are read by the AIL thread.
       *
       * NB: In transition states or immediately after initialization,
       * it is possible that none of the event flags is active.
       */
      OSAL_tEventHandle         hEvGpsState;

      //! \brief Holds KDS file descriptor.
      OSAL_tIODescriptor        KdsFd;

      //! \brief Holds Auxclock file descriptor.
      OSAL_tIODescriptor        AuxClockFd;

      //! \brief The gps encoder is accessed through this interface.
      tclGpsEncoderIf           *poGpsEncoderIf;

      //! \brief Are the factory settings being resetted?
      tBool                     bFactorySettingsReset;

      //! \brief Function pointer to hold the address of the method of
      //!        MsgGpsIf, which writes data to gps ring buffer.
      tS32                      (*pfs32AddGnssData)(OSAL_trGnssFullRecord *);

      //! \brief Flag to indidate the need to update sevice availabilty of
      //!        vdsesnor to spm. This flag will be set from tclAilVDSensor
      //!        class whenever there is a need to upadte the service
      //!        availabilty. If this flag is set then the service availability
      //!        is updated from VDS_GPS(GnssMain thread).
      tBool                     bInformVdsServiceAvailability;

      /*******************************************************************//**
       * \note CONSIDER REMOVING BELOW MEMBERS ALONG WITH THE METHODS WHICH
       *       ARE USING THESE. SINCE IN GEN3 THESE DOESN'T MAKE MUCH SENSE.
       **********************************************************************/
      tBool                     bCalcGps;
      tBool                     bCalcGpsOld;
      OSAL_trGPSRecordFields    trRecordFields;
      tU8                       aucRecBuf[sizeof(OSAL_trGPSRecordHeader)];
      tS32                      (*pfs32AddOsalTimeEvent)( tVoid );

};

#endif

//END OF FILE
