//*****************************************************************************
//* FILE:         tclGpsThread.cpp
//* SW_COMPONENT: VD-Sensor
//* DESCRIPTION:  class-definition: inherits from tclSensorThread,
//*               thread-main-loop, reads data from COM-device
//* AUTHOR:       CM-DI/ESA1-Fischer
//* COPYRIGHT:    (c) 2002 Blaupunkt GmbH
//* HISTORY:
//* 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer
//*          Initial Revision.
//* 14.11.02  CM-CR/EES4-Bode, Adapted to SIRF-GPS on Tuareg.
// * 28.08.09 RBEI/ECF1 - sak9kor - Trace Output related modifications bave been done
// * 08.09.09 RBEI/ECF1 - sak9kor - Trace level has been modified
// * 30.09.09 RBEI/ECF1 - jev1kor - Added doxygen headers for all classes/funcs
// * 29.01.10 RBEI/ECF1 - jev1kor - Removed usage of Goto in code. Lint prio 2.
// * 23.09.10 RBEI/ECF1 - sak9kor - modifications related to calculation of
// *                                Antenna Status and Temperature have been done
// *                                for ADIT Gen2 platform
// * 16.12.10 RBEI/ECF1 - sak9kor - Diaglog related modifications have been done
// * 04.02.11 RBEI/ECF1 - sak9kor - Gps Temeprature calculations have been modified
// *                                for ADIT Gen2 platform
// 29/03/2011 RBEI/ECF1 - rmm1kor   MMS-291715: bCLVStatus flag updated and is used to 
//                       check critical lowvoltage state in tclSensorMsgDispatcher.cpp
//                       and Get_GPS_TEMPARATURE error detail is supressed from logging 
//                       into error memory at CRITICAL LOW VOLTAGE.
// * 04.05.11 RBEI CM-AI/PJ-CF31 - Sainath MMS 294524 Low voltage has been consided 
// *                               while updating the Antnenna Status Diaglog
// * 08.07.11 RBEI CM-AI/PJ-CF31 - Sainath Fix for MMS GMNGA-10285 has been provided 
// * 15.07.11 RBEI CM-AI/PJ-CF31 - Sainath Fix for MMS GMNGA-12350 has been provided  
// *  26.09.12 RBEI/ECF5 - sak9kor - Fix for the issue NIKAI-231 has been provided.
// *                                 To fix this issue updated the bCalcTemperature function
// *  09.11.12 RBEI/ECF5 - sak9kor-  Fix for the issue NIKAI-231 is added           
// *  12.08.13 RBEI/ECF5 - sga5kor- Modifications done to adopt VDSensor according to 
//                                  the CmbFi version-5.0.0
// *  10.07.14 RBEI/ECF5 - rmm1kor- Enabled china encoder for VDS-2
// *  05.11.15 RBEI/ECF5 - rmm1kor- Added support for Diagnosis satellite system control
// *  10.11-15 RBEI/ECF5 - amg6kor - Adding Get operation for GnssEpoch property
// *  03.01.17 RBEI/ECF12 - amg6kor - Removing /dev/gps support and port to Gen4
//**********************************************************************************

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "math.h"

#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#define SYSTEM_S_IMPORT_INTERFACE_ERRMEM_DEF
#include "system_pif.h"

#define SENSOR_FI_S_IMPORT_INTERFACE_SENSOR_LOCATIONFI_TYPES
#include "sensor_fi_if.h"

#define VDS_S_IMPORT_INTERFACE_SENSOR_THREADS
#define VDS_S_IMPORT_INTERFACE_PERSISTENT_DATA
#define VDS_S_IMPORT_INTERFACE_TRACE
#include "vds_internal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#define CCA_S_IMPORT_INTERFACE_GENERIC // CCA_C_U16_APP_SENSOR
#include "cca_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC // scd_s32GetThreadConfiguration
#include "scd_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#include "regkeys.h"    // THREADNAME_C_STRING_SENSOR_VDS_GPS

#define VDS_REG_KEY_CRC_CHECKSUM              \
        "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/GNSS_DEVICE_FW_CRC_CHECKSUM"

/*MMS-291715::  bCLVStatus flag updated and is used to check critical lowvoltage
  state in tclSensorMsgDispatcher.cpp and Get_GPS_TEMPARATURE error detail is
  supressed from logging into error memory at CRITICAL LOW VOLTAGE.*/
tBool bCLVStatus = FALSE;
tU32 u32VDS_GPSnewTicks=0;

#define VDS_GNSS_PROXY_OPEN_RETRY_LIMIT    3
#define VDS_GNSS_READ_TIME_OUT_ERROR_LIMIT 30
#define VDS_GNSS_SCC_BUFFER_FLUSH_RETRY_LIMIT    3

extern tVoid vUpdateVdsServiceAvailability( tVoid );

//Local function declarations.
static inline tVoid vTraceGnssFullRecInfo( const OSAL_trGnssFullRecord &rGnssFullRecord,
                                           tU32 u32TimeStamp);

// tclGpsThread
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  to encode china co-ordinates
//!  Default constructor
//! \return
//!  None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tVoid tclGpsThread::vds_vEncodeToChinaPos( OSAL_trGnssFullRecord *prGnssFullRecord)
{
   // Handle GPS coordinate encoding (only if the GPS receiver has
   // delivered a 2D fix or better)
   if ( OSAL_NULL != prGnssFullRecord )
   {
      if( (prGnssFullRecord->rPVTData.rFixStatus.enMode >= GNSS_FIX_TYPE_2D ) &&
          (poGpsEncoderIf != OSAL_NULL) )
      {
         tDouble dConvertedLatitude = 0.0;
         tDouble dConvertedLongitude = 0.0;
         tDouble dRawLatitude = prGnssFullRecord->rPVTData.f64Latitude;
         tDouble dRawLongitude = prGnssFullRecord->rPVTData.f64Longitude;
         tDouble dRawAltitude = prGnssFullRecord->rPVTData.f32AltitudeWGS84;
         tU16 u16GpsWeek = 0;
         tU32 u32GpsSecond = 0;

         vUtcToGps( prGnssFullRecord->rPVTData.rTimeUTC.u16Year, 
                              prGnssFullRecord->rPVTData.rTimeUTC.u8Month,
                              prGnssFullRecord->rPVTData.rTimeUTC.u8Day, 
                              prGnssFullRecord->rPVTData.rTimeUTC.u8Hour,
                              prGnssFullRecord->rPVTData.rTimeUTC.u8Minute, 
                              prGnssFullRecord->rPVTData.rTimeUTC.u8Second,
                              &u16GpsWeek, 
                              &u32GpsSecond );

         if( poGpsEncoderIf->bEncode(
              dRawLatitude, dRawLongitude, dRawAltitude,
              u16GpsWeek, u32GpsSecond,
              dConvertedLatitude, dConvertedLongitude ) )
         {
           // Success
           prGnssFullRecord->rPVTData.f64Latitude = dConvertedLatitude;
           prGnssFullRecord->rPVTData.f64Longitude = dConvertedLongitude;

         }
         else
         {
           // Failure - the position must be voided
           prGnssFullRecord->rPVTData.rFixStatus.enMode = GNSS_FIX_TYPE_UNKNOWN;
           prGnssFullRecord->rPVTData.f64Latitude = 0.0;
           prGnssFullRecord->rPVTData.f64Longitude = 0.0;
         }
      }
      else if(poGpsEncoderIf != OSAL_NULL)
      {
         if(poGpsEncoderIf->bAvailable())
         {
          vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "Chenc avail but not used" );
         }
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL, "NULL pointer line %lu", __LINE__ );
   }
}


/*!
 * \brief Defines the interface for China GPS Enocder
 */
class tclChinaGpsEncoderIf : public tclGpsEncoderIf
{
private:
   /*!
    * \brief File descriptor for /dev/chenc if opened, OSAL_ERROR
    * otherwise.
    */
   OSAL_tIODescriptor hChenc;

   /*!
    * \brief TRUE if the China encoder has been initialized.
    *
    * Note that this is not done in bInitialize(), but the first time
    * bEncode() is called.
    */
   tBool bInitialized;

   /*!
    * \brief TRUE if the China encoder is inhibited.
    *
    * The China encoder is inhibited if a file /dev/ffs/chinasi.cfg
    * exists or if the chenc test mode is activated by function
    * vds_bGetChinaEncoderTestMode. The file /dev/ffs/chinasi.cfg
    * can be used to disable encoding for testing outside of China.
    * VD Sensor checks for the file when the GPS
    * receiver is started, which usually means when the application
    * enters the NORMAL or DIAGNOSIS state.
    */
   tBool bInhibited;

public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Default Constructor
      //!
      //! \return
      //!   None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
   tclChinaGpsEncoderIf( ) :
      hChenc( OSAL_ERROR ),
      bInitialized( false ),
      bInhibited( false )
   {
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
         "tclChinaGpsEncoderIf instantiated" );
   }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Perform any actions necessary to initialize the China encoder.
      //!
      //! \return
      //!   - \c  TRUE : In case of success\n
      //!   - \c FALSE : In case of failure\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tBool bInitialize
      (
         //!   None
         tVoid
      )
   {
      OSAL_tIODescriptor hCfgFile;

      bInitialized = false;

      hChenc = OSAL_IOOpen(OSAL_C_STRING_DEVICE_CHENC, OSAL_EN_READWRITE );

      if( hChenc != OSAL_ERROR )
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "tclChinaGpsEncoderIf::bInitialize( ) successful" );

         hCfgFile = OSAL_IOOpen( "/dev/ffs/chinasi.cfg", OSAL_EN_READONLY );
         if( hCfgFile != OSAL_ERROR )
         {
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
               "found /dev/ffs/chinasi.cfg, encoding inhibited" );

            bInhibited = true;
            (tVoid)OSAL_s32IOClose( hCfgFile );
         }
         else
         {
            bInhibited = false;
         }

         return TRUE;
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
            "tclChinaGpsEncoderIf::bInitialize( ) failed" );
         return FALSE;
      }
   }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Perform any actions necessary to shuts the China encoder.
      //!
      //! \return
      //!   - \c  TRUE : In case of success\n
      //!   - \c FALSE : In case of failure\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tBool bShutdown
      (
         //!   None
         tVoid
      )
   {
      if( hChenc != OSAL_ERROR)
      {
         (tVoid)OSAL_s32IOClose( hChenc );
         hChenc = OSAL_ERROR;
      }
      bInitialized = false;

      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
         "tclDummyGpsEncoderIf::bShutdown( )" );
         return TRUE;
      }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Checks the availability based on china encoder initialization and china encoder inhibition
      //!
      //! \return
      //!   - \c  TRUE : In case china encoder initialized  and not inhibited
      //!   - \c FALSE : In other cases
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tBool bAvailable
      (
         //!   None
         void
      )
   {
      return bInitialized && !bInhibited;
   }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Encodes china encoder
      //!
      //! \return
      //!   - \c  TRUE : In case of success\n
      //!   - \c FALSE : In case of failure\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tBool bEncode
      (
         //!  (I) : Raw Latitude
      tDouble dRawLatitude,
         //!  (I) : Raw Longitude
      tDouble dRawLongitude,
         //!  (I) : Raw Altitude
      tDouble dRawAltitude,
         //!  (I) : Gps Week information
      tU16 u16GpsWeek,
         //! (I) :  Gps Second information
      tU32 u32GpsSecond,
         //!  (O) : Converted Latitude
      tDouble &rfdConvertedLatitude,
         //! (O) :  Converted Longitude
         tDouble &rfdConvertedLongitude
      )
   {

      tS32 s32RetVal;

      OSAL_trChencConversionData rConversionData;

      // Encoding is inhibited or OSAL device not present.  Pretend
      // nothing happenend.
      if( bInhibited || (hChenc == OSAL_ERROR) )
      {
         rfdConvertedLongitude = dRawLongitude;
         rfdConvertedLatitude  = dRawLatitude;
         return true;
      }

      vTraceMsg( VDS_C_TRACELEVEL_MESSAGE,
         "bEncode: input la=%d lo=%d al=%d wk=%u sec=%lu",
         (tS32)(dRawLatitude* 1e6), (tS32)(dRawLongitude* 1e6), (tS32)(dRawAltitude),
         u16GpsWeek, u32GpsSecond );

      rConversionData.dLongitude = dRawLongitude;
      rConversionData.dLatitude = dRawLatitude;
      rConversionData.dAltitude = dRawAltitude;
      rConversionData.s32GpsWeek = u16GpsWeek;
      rConversionData.u32GpsSecond = u32GpsSecond;

      if( !bInitialized )
      {
         rConversionData.s32InitializationFlag = 0;

         s32RetVal = OSAL_s32IOControl(
            hChenc, OSAL_C_S32_IOCTRL_CHENC_CONVERT,
            (tLong) &rConversionData );

         if( s32RetVal != OSAL_OK )
         {
            rfdConvertedLatitude = rConversionData.dLatitude;
            rfdConvertedLongitude = rConversionData.dLongitude;

            return true;
         }

         if( rConversionData.u32ReturnCode != 0 )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
               "bEncode(): unable to initialize China encoder" );

            return false;
         }

         bInitialized = true;
      }

      rConversionData.s32InitializationFlag = 1;

      s32RetVal = OSAL_s32IOControl(
         hChenc, OSAL_C_S32_IOCTRL_CHENC_CONVERT,
         (tLong) &rConversionData );

      if( s32RetVal != OSAL_OK )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
            "bEncode(): China encoder does not encode" );

         return false;
      }

      if( rConversionData.u32ReturnCode != 0 )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
            "bEncode(): China encoder returned error code %x",
            rConversionData.u32ReturnCode );
      }
      else
      {
         tS32 s32ConvertedLatitude;
         tS32 s32ConvertedLongitude;

         rfdConvertedLatitude = rConversionData.dChinaLatitude;
         rfdConvertedLongitude = rConversionData.dChinaLongitude;

         s32ConvertedLatitude = (tS32)(rfdConvertedLatitude*RAD2DEG*1e6);
         s32ConvertedLongitude = (tS32)(rfdConvertedLongitude*RAD2DEG*1e6);

         vTraceMsg( VDS_C_TRACELEVEL_DATA,
            "bEncode: output la=%d lo=%d",
            s32ConvertedLatitude, s32ConvertedLongitude );
      }

      return TRUE;
   }

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   Enable resp. disable the china encoder
      //!   A requirement of the production is to disable and enable the encoder without a reset.
      //! \return
      //!   - \c  TRUE : In case of success\n
      //!   - \c FALSE : In case of failure\n
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tBool bEnDisChinaEnc
      (
         //!  Parameter to inhibit the china encoder
         tBool bInhibit
      )
      {
      bInhibited = bInhibit;

      vTraceMsg( VDS_C_TRACELEVEL_DATA,
         "bEnDisChinaEnc: bInhibited = %d",
         bInhibited
         );

      return TRUE;
   }
};

/*!
 * \brief Auxclock checker.
 *
 * This class is responsible for checking that /dev/auxclock is
 * ticking.  If it isn't, there is a serious problem, so we throw an
 * assert, which normally causes a system reset.
 */
class vds_tclAuxClockWatchDog
{
private:
   /*!
    * \brief Value of the OSAL ms counter when the last OK auxclock
    * value was given to the checker.
    */
   static OSAL_tMSecond msLastOK;

   /*!
    * \brief Last OK auxclock value.
    *
    * The definition of OK is "different from the previous value".
    */
   static tU32 u32LastTimeStamp;

   /*!
    * \brief Maximum time in ms before an assert is made if the
    * auxclock time stamp doesn't change.
    *
    * Since a system reset is a rather brutal measure, we wait 20
    * seconds to be really sure that we have a problem that isn't
    * going away by itself.
    */
   static const OSAL_tMSecond msMaxDiff = 20000;

public:

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!  Client interface for the auxclock checker
      //!
      //! \return
      //!  None
      //  HISTORY:
      // Date            |  Author                       | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      static void vCheckAuxClock
      (
         //! (I) : Current auxclock time stamp
         tU32 u32TimeStamp
      )
   {
      OSAL_tMSecond msTime = OSAL_ClockGetElapsedTime();

      if( u32TimeStamp != u32LastTimeStamp )
      {
         // Everything OK

         msLastOK = msTime;
         u32LastTimeStamp = u32TimeStamp;
      }
      else
      {
         if( (msTime - msLastOK) > msMaxDiff )
         {
            // Auxclock has been stopped for more than msMaxDiff ms.
            // This means that navigation isn't working. -> System reset
            /* after contact with Mr. Oester Jonas (CM-DI/PJ-CF14) (23.03.3007)
            FATAL_M_ASSERT_ALWAYS();
            */
         }
      }
   }
};

OSAL_tMSecond vds_tclAuxClockWatchDog::msLastOK = 0;
tU32 vds_tclAuxClockWatchDog::u32LastTimeStamp = 0;

/*!
 * \brief profiling hack event information
 *
 */
#if defined( VDS_PROFILING_HACK )
class vds_tclProfilingHack
{
public:
   enum  tenEvent
   {
      enReadStart,
      enReadGoto,
      enReadEnd,
      enParseStart,
      enParseEnd
   };

   vds_tclProfilingHack( ) : u32EvIndex(0)
   {
      }
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//  brief
// This function writes the event related information in to prof.dat file. If file is
//  is not available it creates the file write into it and closes it.
//
//  return
//  None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
   void vEvent( tenEvent ev );

private:
   static const tU32 cu32MaxEvs = 1000;

   struct
   {
      tenEvent ev;
      OSAL_tMSecond ms;
   } evs[cu32MaxEvs];

   tU32 u32EvIndex;
};

vds_tclProfilingHack *gpoProfiler = OSAL_NULL;
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//! This function writes the event related information in to prof.dat file. If file is
//! is not available it creates the file write into it and closes it.
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void vds_tclProfilingHack::vEvent
(
   //  (I):  Event Information i.e. enReadStart,enReadGoto,enReadEnd,enParseStart, enParseEnd
   vds_tclProfilingHack::tenEvent ev
)
{
   if( u32EvIndex < cu32MaxEvs )
   {
      evs[u32EvIndex].ev = ev;
      evs[u32EvIndex].ms = OSAL_ClockGetElapsedTime();
      u32EvIndex++;
   }

   if( u32EvIndex >= cu32MaxEvs  )
   {
      OSAL_tIODescriptor hFile;

      if( (hFile = OSAL_IOOpen( "/dev/ffs/prof.dat", OSAL_EN_READWRITE )) == OSAL_ERROR )
      {
         hFile = OSAL_IOCreate( "/dev/ffs/prof.dat", OSAL_EN_READWRITE );
      }

      if( hFile != OSAL_ERROR )
      {
         OSAL_s32IOWrite( hFile, (tPS8)evs, (tU32) sizeof( evs ) );
         OSAL_s32IOClose( hFile );
      }
      u32EvIndex = 0;
   }
}
#endif

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Default Constructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclGpsThread::tclGpsThread()
{   
    coszSensorSemName = "VDS_GPS_THREAD_SEM";   
    bGpsRunning = FALSE;
    bFlushGnssBuf = false;
    u8GpsThreadEvent = VDS_GPS_EVENT_NOEVENT; 
    u32ApplicationState =AMT_C_U32_STATE_INVALID;
    u8OsalTimeStatus = VDS_C_U8_OSAL_CLOCK_NOT_SET; 
    OSAL_pvMemorySet(&rManualDateTime,0,(tU32) sizeof(OSAL_trTimeDate));   
    enTimeMode =  enTimeModeAuto;
    szEvGpsStateName = "GPS_STATE_EV"; 
    hEvGpsState = OSAL_C_INVALID_HANDLE;   
    KdsFd = OSAL_ERROR;
    AuxClockFd = OSAL_ERROR;   
    poGpsEncoderIf = OSAL_NULL;   
    bFactorySettingsReset = FALSE;   
    pfs32AddGnssData = OSAL_NULL;   
    bInformVdsServiceAvailability = false;   
    bCalcGps = FALSE;   bCalcGpsOld = FALSE;   
    trRecordFields.u32FieldSpecifiers = 0;  
    trRecordFields.u32FieldTypes = 0;   
    aucRecBuf[0] = 0;  
    pfs32AddOsalTimeEvent = OSAL_NULL;   
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Default Destructor
//!
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tclGpsThread::~tclGpsThread()
{
   tclGpsThread::vThreadClose();
   
   szEvGpsStateName = OSAL_NULL;
   poGpsEncoderIf = OSAL_NULL;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  inherited from class tclSensorThread.\n
//!  Opens COM- and AuxClock-device, create semaphore, create gps-thread, set function-pointer,
//!  set internal state
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_INVALID_PARAMETER :   Invalid Parameter
//!   - \c VDS_E_GPSTHREAD_DEVICE_NOT_FOUND : Device not found
//!   - \c VDS_E_GPSTHREAD_SEM_ERROR : Semaphore error
//!   - \c VDS_E_GPSTHREAD_THREADCREATE_FAILED : GPS Thread creation failed
//!   - \c VDS_E_GPSTHREAD_NOT_UNINITIALIZED : GPS Thread Uninitialized
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclGpsThread::s32ThreadInit
(
   //! (I) :  function pointer
   tS32 (*pfs32AddGnssDataParam)(OSAL_trGnssFullRecord *)
)
{
   tS32 s32RetVal = VDS_E_NO_ERROR;

   if ( VDS_C_S32_THREAD_UNINITIALIZED == s32GetInternalState() )
   {

      if(OSAL_NULL != pfs32AddGnssDataParam)
      {
         this->pfs32AddGnssData = pfs32AddGnssDataParam;
      }
      else
      {
         s32RetVal = VDS_E_INVALID_PARAMETER;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GpsThread:Invalid argument");
      }

      //Open AuxClock Device
      AuxClockFd = OSAL_IOOpen( OSAL_C_STRING_DEVICE_AUXILIARY_CLOCK, OSAL_EN_READONLY );

      if( OSAL_ERROR  ==  AuxClockFd )
      {
         s32RetVal = VDS_E_GPSTHREAD_DEVICE_NOT_FOUND;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GpsThread:Opening DevAuxclock failed with error = 0x%x",
                    OSAL_u32ErrorCode());
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "GpsThread:DevAuxclock opened successfully.");
      }

      //Create semaphore.
      if( OSAL_ERROR == OSAL_s32SemaphoreCreate( coszSensorSemName,&hSensorSemaphore,1 ) )
      {
         s32RetVal = VDS_E_GPSTHREAD_SEM_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GpsThread:Creating Semaphore failed with error = 0x%x",
                    OSAL_u32ErrorCode());
      }

      //Create Event
      if( OSAL_ERROR == OSAL_s32EventCreate( szEvGpsStateName, &hEvGpsState ) )
      {
         s32RetVal = VDS_E_GPSTHREAD_SEM_ERROR;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GpsThread:Creating event failed with error = 0x%x",
                    OSAL_u32ErrorCode());
      }

      #ifndef VARIANT_S_FTR_DISABLE_CHINAENCODER
      //Instantiate China encoder interface
      poGpsEncoderIf = new tclChinaGpsEncoderIf;
      #endif

      //GNSS thread attributes.
      OSAL_trThreadAttribute rThreadAttribute = {0};
      rThreadAttribute.szName       = const_cast<tString>(THREADNAME_C_STRING_SENSOR_VDS_GPS);
      rThreadAttribute.pfEntry      = (OSAL_tpfThreadEntry) tclGpsThread::vThreadMainWrapper;
      rThreadAttribute.pvArg        = (tPVoid) this;

      //Get GNSS thread configuration.
      if ( OSAL_ERROR == scd_s32GetThreadConfiguration( CCA_C_U16_APP_SENSOR,
                                                        THREADNAME_C_STRING_SENSOR_VDS_GPS,
                                                        &rThreadAttribute.u32Priority,
                                                        &rThreadAttribute.s32StackSize ) )
      {
         s32RetVal = VDS_E_GPSTHREAD_THREADCREATE_FAILED;
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                    "GpsThread:Failed to get GPS/GNSS thread configuration.");
      }

      //Create GNSS thread.This thread collects data from GNSS Proxy.
      if( s32RetVal == VDS_E_NO_ERROR )
      {
         SensorThreadId = OSAL_ThreadCreate( &rThreadAttribute );
         if( OSAL_ERROR == SensorThreadId )
         {
            s32RetVal = VDS_E_GPSTHREAD_THREADCREATE_FAILED;
            vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                       "GpsThread:Creating GPS/GNSS thread failed with error = 0x%x",
                       OSAL_u32ErrorCode());
         }
      }

      //If no error occurred then change the state to initialized.
      if ( VDS_E_NO_ERROR == s32RetVal )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GpsThread:Unable to create GNSS thread");
         vThreadClose();
      }
   }
   else
   {
      s32RetVal = VDS_E_GPSTHREAD_NOT_UNINITIALIZED;
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GpsThread:Already initialized");
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Prints trace in readable form
//! \return
//   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
static inline tVoid vTraceGnssFullRecInfo( const OSAL_trGnssFullRecord &rGnssFullRecord,tU32 u32TimeStamp)
{

   ET_TRACE_INFO_BIN((int)TR_CLASS_VD_SENSOR, ET_EN_T8  _ 7
   _ ET_EN_T32 _ (tS32)(rGnssFullRecord.rPVTData.f64Latitude * RAD2DEG * 1e6)
   _ ET_EN_T32 _ (tS32)(rGnssFullRecord.rPVTData.f64Longitude * RAD2DEG *1e6)
   _ ET_EN_T8  _ (tU8)rGnssFullRecord.rPVTData.rFixStatus.enMode
   _ ET_EN_T8  _ rGnssFullRecord.rPVTData.u16Received
   _ ET_EN_T32 _ u32TimeStamp
   _ ET_EN_T16 _ rGnssFullRecord.rPVTData.rTimeUTC.u16Year
   _ ET_EN_T8  _ rGnssFullRecord.rPVTData.rTimeUTC.u8Month
   _ ET_EN_T8  _ rGnssFullRecord.rPVTData.rTimeUTC.u8Day
   _ ET_EN_T8  _ rGnssFullRecord.rPVTData.rTimeUTC.u8Hour
   _ ET_EN_T8  _ rGnssFullRecord.rPVTData.rTimeUTC.u8Minute
   _ ET_EN_T8  _ rGnssFullRecord.rPVTData.rTimeUTC.u8Second
   _ ET_EN_T16 _ rGnssFullRecord.rPVTData.rTimeUTC.u16Millisecond
   _ ET_EN_DONE);

   ET_TRACE_INFO_BIN((int)TR_CLASS_VD_SENSOR, ET_EN_T8  _ 8
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[0].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[0].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[1].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[1].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[2].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[2].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[3].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[3].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[4].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[4].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[5].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[5].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[6].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[6].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[7].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[7].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[8].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[8].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[9].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[9].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[10].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[10].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[11].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[11].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[12].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[12].u8CarrierToNoiseRatio)
   _ ET_EN_T16  _ (tU16)(rGnssFullRecord.rChannelStatus[13].u16SvID)
   _ ET_EN_T8   _ (rGnssFullRecord.rChannelStatus[13].u8CarrierToNoiseRatio)
   _ ET_EN_DONE);

}

// ***************** F U N C T I O N  H E A D E R ******************************
//
//  DESCRIPTION:
//
//! \brief
//!  inherited from class tclSensorThread.\n
//!  when thread is started, set state RUNNING.
//!  collects data from OSAL GNSS proxy and store data in GNSS Data Ring buffer.
//!  when internal state is set to STOP, leave loop and set state INITIALIZED.
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vThreadMain
(
   //! (I):None
   tPVoid
)
{

   OSAL_trGnssFullRecord rGnssFullRecord;
   OSAL_trIOCtrlAuxClockTicks rAuxClockTicks;
   tU32 u32TimeOutErrorCnt  = 0;
   tS32 s32RetVal = OSAL_OK;
   tU32 u32diffTicks = 0;
   tU32 u32lastTicks = 0;
   tU32 u32VDS_GNSSnewTicks = 0;

   //Set necessary flags which describes the state of the thread.
   bGpsRunning = FALSE;                   // GPS is not running at this time
   bCalcGps    = FALSE;                   // No pending GPS calculation
   bCalcGpsOld = FALSE;
   u8GpsThreadEvent = VDS_GPS_EVENT_NOEVENT;
   u8OsalTimeStatus = VDS_C_U8_OSAL_CLOCK_NOT_SET;


   /* This thread is initialized in "s32ThreadInit" function. In s32ThreadInit function all the
    * resources for this thread are created and once everthing is completed successfully
    * thread initialized flag "VDS_C_S32_THREAD_INITIALIZED" will be set.
    *
    * Check whether this thread is initialized or not.If not initialized proceeding further is
    * foolish so just EXIT from this thread.
    */

   if( s32GetInternalState() == VDS_C_S32_THREAD_INITIALIZED )
   {
      //Thread is activated and running so set thread running flag.
      vSetInternalState( VDS_C_S32_THREAD_RUNNING );
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "VDS_GPS: started running at time(ms) %u", OSAL_ClockGetElapsedTime() );

      /*
       * -> Wait until GNSS proxy started succesfully before collecting data records from it.
       * -> When GNSS proxy is successfully initialized "bGpsRunning" flag will be set.
       * -> AIL thread will set GPS start event(signal to start GNSS proxy) flag when application
       *    state changes to NORMAL.
       * -> bGpsRunning is set in bStartDevGnss(),which is called from bCheckGpsThreadEvent().
       * -> Since there is no guarantee that the state always remains in NORMAL so we have to 
       *    check explicitly(using bCheckGpsThreadEvent()) that anyone is trying to
       *    stop the thread.
       */

      //Wait till GNSS proxy initialized successfully.
      while( (!bGpsRunning) && (s32GetInternalState() != VDS_C_S32_THREAD_STOP) )
      {
         //Check Any GPS Event Occurred.
         (tVoid)bCheckGpsThreadEvent();
         (tVoid)OSAL_s32ThreadWait(100);
      }

      /* Skip entire main loop and exit the thread if the previous
       * loop was exited because of thread stop request(or event).
       */
      if( s32GetInternalState() != VDS_C_S32_THREAD_STOP )
      {
         /* Enter the main loop which collects data from GNSS proxy.
          * Stay in loop untill VD Sensor thread stop event occurrs.
          */
         do
         {
            //Check whether any thread state change events Occurred.
            bCheckGpsThreadEvent();

            //If GNSS proxy is running.
            if( bGpsRunning )
            {
               //Read data from GNSS Proxy.
               s32RetVal = OSAL_s32IORead( SensorFd,
                                           (tPS8)&rGnssFullRecord,
                                           (tU32) sizeof(OSAL_trGnssFullRecord));
               if( OSAL_ERROR != s32RetVal )
               {
                  //Read is successful so make time out count zero.
                  u32TimeOutErrorCnt = 0;

                  //Sync system time with GNSS time.
                  #ifndef VARIANT_S_FTR_ENABLE_CHERY_GNSS  //! for chery
                  /*! workitem-277276:
				   *! note: in chery, system time is handled by vdclock because of
                   *! some project specific requirements, so dont update system time
                   *! from vdsesnor if the project is chery */
                  bSyncTime(&rGnssFullRecord.rPVTData.rTimeUTC);
                  #endif

                  //Store collected record to ring buffer.
                  //Before storing,convert latitude and longitude from degrees to radians.
                  rGnssFullRecord.rPVTData.f64Latitude = rGnssFullRecord.rPVTData.f64Latitude*DEG2RAD;
                  rGnssFullRecord.rPVTData.f64Longitude = rGnssFullRecord.rPVTData.f64Longitude*DEG2RAD;

                  // Encode co-ordinates for china
                  vds_vEncodeToChinaPos( &rGnssFullRecord );

                  if(OSAL_ERROR == s32StoreGnssData( &rGnssFullRecord ))
                  {
                     vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                                "GpsThread:Storing GNSS Data To RingBuffer Failed");
                  }

                  if( true == bInformVdsServiceAvailability )
                  {
                     vUpdateVdsServiceAvailability();
                     bInformVdsServiceAvailability = false;
                  }
               }
               else if( OSAL_E_TIMEOUT == OSAL_u32ErrorCode() )
               {
                  /*This can happen and its not an error for some iterations.If timeout occurs for
                   *VDS_GNSS_READ_TIME_OUT_ERROR_LIMIT(30) times consecutively then restart GNSS Proxy.
                   */
                  u32TimeOutErrorCnt++;
                  vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                             "GpsThread:TimeOut error while reading from GNSS proxy (ErrCnt=%u)", u32TimeOutErrorCnt);

                  if(u32TimeOutErrorCnt >= VDS_GNSS_READ_TIME_OUT_ERROR_LIMIT)
                  {
                     vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                                "GpsThread:Time out error for %u iterations -> restarting GNSS proxy", u32TimeOutErrorCnt);
                     vSetGpsThreadEvent(VDS_GPS_EVENT_RESTART);
                  }
               }
               else
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                             "GpsThread:Reading from GNSS proxy failed with error 0x%x",
                             OSAL_u32ErrorCode());
               }

               /*Check whether this thread is running in regular
                *intervals or not i.e. Every One Second.
                */
               OSAL_s32IORead( AuxClockFd,(tPS8)&rAuxClockTicks,(tU32) sizeof(OSAL_trIOCtrlAuxClockTicks));
               u32VDS_GNSSnewTicks = rAuxClockTicks.u32Low;
               if (u32VDS_GNSSnewTicks >= u32lastTicks)
               {
                  u32diffTicks = u32VDS_GNSSnewTicks - u32lastTicks;
               }
               else
               {
                  //This should only happen if there is a wrap around.
                   u32diffTicks = ( OSAL_C_U32_MAX - u32lastTicks ) + u32VDS_GNSSnewTicks;
               }

               u32lastTicks = u32VDS_GNSSnewTicks;
               /* Just print a trace saying that there is no activitity
                * (thread might be blocked) in the last two seconds.
                */
               if( u32diffTicks > 2000)
               {
                  vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                             "GpsThread:GNSS overdue for more then 2s dt: %u ms",
                             u32diffTicks);
               }

               //Check Whether AuxClock Is Ticking Or Not.
               vds_tclAuxClockWatchDog::vCheckAuxClock( u32VDS_GNSSnewTicks );

               //If data is received from GNSS proxy without any error then trace it out.
               if(OSAL_ERROR != s32RetVal)
               {
                  vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                          "GpsThread:GPS mesg received after %ums from last msg",u32diffTicks);
                  vTraceGnssFullRecInfo(rGnssFullRecord,u32VDS_GNSSnewTicks);
               }
            }
            else
            {
               vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                          "GpsThread:GPS stopped running");
               //GNSS proxy is not running WAIT till it starts again before collecting GNSS data.
               while( !bGpsRunning &&  (VDS_C_S32_THREAD_RUNNING == s32GetInternalState()) )
               {
                  bCheckGpsThreadEvent();
                  (tVoid)OSAL_s32ThreadWait(100);
               }
               (tVoid)OSAL_s32ThreadWait(1000);
            }

         }while( VDS_C_S32_THREAD_RUNNING == s32GetInternalState() );
      }
      else
      {
         // This is not really an error, but it is so interesting
         // to see the trace message anyway...
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GpsThread:State NORMAL was never reached");
      }

      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"GpsThread:Exited Main Loop ....");
      
      //Set new state.
      if ( s32GetInternalState() == VDS_C_S32_THREAD_STOP )
      {
         vSetInternalState( VDS_C_S32_THREAD_INITIALIZED );
      }
      else
      {
         //Close handles, ...
         vThreadClose();
         vSetInternalState( VDS_C_S32_THREAD_UNINITIALIZED );
      }
   }
   //else
   //{
   //   No error-handling is needed, state is uninitialized
   //}

   OSAL_vThreadExit();
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  static wrapper for main-loop of instance.
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vThreadMainWrapper
(
   //! (I): void pointer to thread arg
   tPVoid pvThreadArg
)
{
   // param-check
   if ( OSAL_NULL != pvThreadArg )
   {
      tclGpsThread *poGpsThread = static_cast<tclGpsThread *>(pvThreadArg);
      poGpsThread->vThreadMain( NULL );
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Inherited from class tclSensorThread. closes handles,
//!   deletes semaphores, deletes allocated memory
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vThreadClose ()
{
   //Stop GNSS Proxy.
   bStopDevGnss();

   //Close AuxClock-Device
   if ( OSAL_ERROR != AuxClockFd )
   {
      if ( OSAL_ERROR == OSAL_s32IOClose( AuxClockFd ) )
      {
         AuxClockFd = OSAL_ERROR;
      }
   }

   // semaphore and device will be deleted in tclSensorThread

   // delete helper interfaces

   // reset function-pointers
   pfs32AddGnssData = NULL;
   pfs32AddOsalTimeEvent = OSAL_NULL;

   tclSensorThread::vThreadClose();

}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  send GNSS data to tclMsgGpsIf there it will be stored in ring buffer.
//! \return
//!   Succes code in case of success is:
//!   - \c  VDS_E_NO_ERROR : Success\n
//!   Error code in case of failure are:\n
//!   - \c VDS_E_GPSTHREAD_STORAGE_ERROR :   GPS thread storage error
//!   - \c VDS_E_GPSTHREAD_NOT_INITIALIZED : Gps thread not initialized
//!   - \c VDS_E_INVALID_PARAMETER : Invalid Parameter
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tS32 tclGpsThread::s32StoreGnssData(OSAL_trGnssFullRecord *prGnssFullRecord)
{
   
   tS32 s32RetVal = VDS_E_NO_ERROR;

   // param-check
   if ( NULL != prGnssFullRecord )
   {
      // consistence check
      if ( NULL != pfs32AddGnssData )
      {
         //Add GNSS data to ring buffer.
         if ( VDS_E_NO_ERROR != (*pfs32AddGnssData)( prGnssFullRecord ) )
         {
            s32RetVal = VDS_E_GPSTHREAD_STORAGE_ERROR;
         }
      }
      else
      {
         s32RetVal = VDS_E_GPSTHREAD_NOT_INITIALIZED;
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsThread:pfs32AddGnssData uninitialized" );
      }
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GpsThread:s32StoreGnssData:Invalid param" );
      s32RetVal = VDS_E_INVALID_PARAMETER;
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Sets the system time OSAL
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of Failure\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************

tBool 
tclGpsThread::bSetSystemTime( const OSAL_trTimeDate &rfTargetTimeDate )
{
   tBool bRet = FALSE;
   tS32 s32Ret = OSAL_s32ClockSetTime( &rfTargetTimeDate);
   if( OSAL_OK == s32Ret )
   {
      bRet = TRUE;
   }

   return bRet;
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Sets the system time (OSAL)
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of Failure\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool 
tclGpsThread::bSyncTime
(
   //! (I) : pointer to GPS Time
   const OSAL_trGnssTimeUTC *rTimeUTC
)
{
   // This is the synchronisation interval in ms.  If the system time
   // has not been synchronised with GPS for this time, we
   // resynchronise.
   const OSAL_tMSecond msSynchInterval = 3600000L;

   // This is the last time system and GPS time were synchronised.
   static OSAL_tMSecond msT0 = 0;
   // This is the current time.
   OSAL_tMSecond msT1;

   // If the date has changed, resynchronize immediately.  We do this
   // in order to heal the effects of a possible incorrect week number
   // from GPS as soon as the correct date is available.
   tBool bDateChanged = false;

   vds_tenTimeMode enActualTimeMode = enTimeModeInvalid;

   // get actual timemode
   bGetTimeMode( enActualTimeMode );

   OSAL_trTimeDate rTargetTimeDate = {0};
   OSAL_trTimeDate rActualTimeDate = {0};

   tBool bRet=false;

   // If the GPS time is not valid, return immediately.
   // If the following parameters are zero then time is not valid.
   if( (rTimeUTC->u16Year == 0)&&
       (rTimeUTC->u8Month == 0)&&
       (rTimeUTC->u8Day   == 0)&&
       (rTimeUTC->u8Hour  == 0)&&
       (rTimeUTC->u8Minute== 0)&&
       (rTimeUTC->u8Second== 0)
      )
   {
       return false;
   }
   
   // From this point, we know that the GPS time is valid
   msT1 = OSAL_ClockGetElapsedTime();

   if( msT1 < msT0 )
   {
       msT0 = msT1;
   }

   //Check whether date is changed or not.
   if( OSAL_OK == OSAL_s32ClockGetTime(&rActualTimeDate) )
   {
      if( (rActualTimeDate.s32Year  != (tS32)(rTimeUTC->u16Year - 1900) ) ||
          (rActualTimeDate.s32Month != (tS32)rTimeUTC->u8Month) ||
          (rActualTimeDate.s32Day   != (tS32)rTimeUTC->u8Day) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_DATA,
                    "GpsThread:GPS date not equal to system time date.");
         bDateChanged = true;
      }
   }

   if (enTimeModeAuto == enActualTimeMode)
   {
     if(  (0 == msT0)
          ||
          ((msT1 - msT0) > msSynchInterval )
          ||
          bDateChanged )
      {
         rTargetTimeDate.s32Second         = (tS32)rTimeUTC->u8Second;
         rTargetTimeDate.s32Minute         = (tS32)rTimeUTC->u8Minute;
         rTargetTimeDate.s32Hour           = (tS32)rTimeUTC->u8Hour;
         rTargetTimeDate.s32Day            = (tS32)rTimeUTC->u8Day;
         rTargetTimeDate.s32Month          = (tS32)rTimeUTC->u8Month;
         rTargetTimeDate.s32Year           = (tS32)rTimeUTC->u16Year - 1900;
         rTargetTimeDate.s32Weekday        = 0;
         rTargetTimeDate.s32Yearday        = 0;
         rTargetTimeDate.s32Daylightsaving = 0;

         bCheckGpsThreadEvent();

         bRet = bSetSystemTime(rTargetTimeDate);

         if ( true == bRet) 
         {
            msT0 = msT1;
         }
      }
   }
   else
   {
      bRet = true;
      vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                 "GpsThread:Time mode is not auto -> no time sync!" );
   }
   return bRet;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   returns current current time of day (minutes in day)
//! \return
//!   - \c  TRUE : time of day is available
//!   - \c  FALSE : time of day is not available
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bGetTimeOfDay
(
   //!  (O) : Time of day in minutes
   tU16 &rfu16TimeOfDay
)
{
   tS32               lResult;
   OSAL_trTimeDate    rTimeDate;

   lResult = OSAL_s32ClockGetTime(&rTimeDate);
   // VD Sensor does not support local time any more.
   if( lResult != OSAL_ERROR )
   {
      rfu16TimeOfDay = (tU16)( (60 * rTimeDate.s32Hour) + rTimeDate.s32Minute );
      return TRUE;
   }
   else
   {
      rfu16TimeOfDay = 0;
      return FALSE;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   set time of day
//! \return
//!   - \c  TRUE : time of day changed
//!   - \c  FALSE : error while setting time of day or feature not available for this receiver
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bSetTimeOfDay
(
   //! (I) : Timeofday in minustes
   const tU16 &rfu16TimeOfDay
)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( rfu16TimeOfDay );
   return FALSE;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Calculates the number of seconds and number of weeks based on the parameter provided
//!   i.e. year, month, day, hour, minute,second
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vUtcToGps
(
   //! (O) : Year
   tU16 usYear,
   //! (I) : Month
   tU8  ucMonth,
   //! (I) : Day
   tU8  ucDay,
   //! (I) : Hour
   tU8  ucHour,
   //! (I) : Minute
   tU8  ucMinute,
   //! (I) : Second
   tU8  ucSecond,
   //! (O) :  GPS week time stamp
   tU16 *pusWn,
   //! (I) : GPS time of the week
   tU32 *pulTow
)
{
   tU16 usDay;
   tU16 usLeapDays, usTotalDays, usLeftDays;
   tU16 usDaysInMonth[] = {0,0,31,59,90,120,151,181,212,243,273,304,334};

   usYear = usYear - 1980;
   usDay = usDaysInMonth[ucMonth];
   usDay += ucDay;

   if( (usYear % 4 == 0) && (ucMonth > 2) )
   {
      usDay += 1;
   }
   
   usLeapDays = ((usYear-1) / 4) + 1;

   usTotalDays = (tU16)(usYear*365 + usDay + usLeapDays - 6);
   *pusWn = usTotalDays / 7;
   usLeftDays = usTotalDays % 7;

   *pulTow =  (tU32)(   ((tF64)(usLeftDays) * 86400)
                      + ((tF64)ucHour * 3600)
                      + ((tF64)ucMinute * 60)
                      + ((tF64)ucSecond )  
                    );

   if(*pulTow >= 604800)
   {
      *pusWn += 1;
      *pulTow -= 604800;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Converts the time in to day, month and year
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vConvertGpsTime
(
   //! (O) :  MSec
   tU32 ulMsec,
   //! (O) :  Week Number
   tU16 usWeekNum,
   //! (O) :  Day
   tU16 *pusDay,
   //! (O) :  Month
   tU16 *pusMonth,
   //! (O) :  Year
   tU16 *pusYear
)
{
   #define START_WEEK       1024
   #define REF_WEEK         1024

   tU16 usMonth, usYear;
   static const tU8 ucDaysInMonth[] = {31, 30, 31, 30, 31, 31, 30, 31, 30, 31, 31, 28};

   ulMsec = ulMsec / (1000*86400);

   if (usWeekNum < START_WEEK)
   {
      usWeekNum += REF_WEEK;
   }

   ulMsec = ulMsec +7*usWeekNum - 54;
   usYear = 1980;

   while (ulMsec > 1461)
   {
      ulMsec -= 1461;
      usYear += 4;
   }

   if (ulMsec != 1461)
   {
      while (ulMsec > 365)
      {
         ulMsec -= 365;
         usYear += 1;
      }
      usMonth = 0;
      while (ulMsec > (tU32)(ucDaysInMonth[usMonth]))
      {
         ulMsec -= ucDaysInMonth[usMonth++];
      }

      if (usMonth > 9)
      {
         usYear += 1;
         usMonth -= 9;
      }
      else
      {
         usMonth += 3;
      }

      *pusDay = (tU16)ulMsec;
      *pusMonth = usMonth;
      *pusYear = usYear;
   }
   else
   {
      /* leap year */
      *pusDay = 29;
      *pusMonth = 2;
      *pusYear = (tU16)(usYear + 4);
   }
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Calculates the day, month, year, hour, minute and seconds based on the parameters
//!   provided i.e. week time stamp. and time of the week
//! \return
//!   None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vGpsToUtc
(
   //! (I) : GPS week time stamp
   tU16 usWn,
   //! (I) : GPS time of the week
   tU32 ulTow,
   //! (O) : Pointer to Day
   tU8 *pucDay,
   //! (O) : Pointer to Month
   tU8 *pucMonth,
   //! (O) : Pointer to Year
   tU16 *pusYear,
   //! (O) : Pointer to Hour
   tU8 *pucHour,
   //! (O) : Pointer to Minute
   tU8 *pucMinute,
   //! (O) : Pointer to Second
   tU8 *pucSecond
)
{
   tU32 ulTemp;
   tU16 usTemp;
   tU16 u16TempDay = (tU16)*pucDay;
   tU16 u16TempMonth = (tU16)*pucMonth;

   if (ulTow >= 604800)
   {
      ulTow = 0;
      usWn++;
   }
   vConvertGpsTime( ulTow*1000, usWn, &u16TempDay, &u16TempMonth, pusYear);

   ulTemp = ulTow;
   ulTemp %= 86400;

   *pucHour = (tU8)(ulTemp/3600);
   usTemp = (tU16)(ulTemp - (*pucHour)*3600);
   *pucMinute = (tU8)(usTemp/60);
   usTemp -= (tS16)(*pucMinute)*60;
   *pucSecond = (tU8)usTemp;            /* + dblTemp; */

   *pucDay = (tU8)u16TempDay;
   *pucMonth = (tU8)u16TempMonth;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Starts Dev GNSS.
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of Failure\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bStartDevGnss()
{

   tBool bRet = TRUE;
   tU16 u16RetryCount = 0;

   if( !bGpsRunning && (s32GetInternalState()==VDS_C_S32_THREAD_RUNNING) )
   {
      do
      {
         //Open GNSS OSAL Proxy.
         SensorFd = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS,OSAL_EN_READONLY );

         if ( OSAL_ERROR == SensorFd )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "GpsThread:Opening GNSS proxy failed with error = 0x%x",
                       OSAL_u32ErrorCode());
            bRet = FALSE;
            bGpsRunning = FALSE;
            //Trace error to Error Memory.
            et_vErrmemStringNormal( TR_COMP_SENSOR,
                                    "GpsThread:Opening GNSS Proxy failed with error = 0x%x",
                                    OSAL_u32ErrorCode());
         }
         else
         {
            bGpsRunning = TRUE;
            vTraceMsg( VDS_C_TRACELEVEL_COMPONENT, "GpsThread:GNSS proxy opened.");
         }

         u16RetryCount++;
       (tVoid)OSAL_s32ThreadWait(500);

      }while( (SensorFd == OSAL_ERROR) &&
              (u16RetryCount <= VDS_GNSS_PROXY_OPEN_RETRY_LIMIT) &&
              (s32GetInternalState()==VDS_C_S32_THREAD_RUNNING));
      
      if(SensorFd == OSAL_ERROR )
      {
         vTraceMsg( VDS_C_TRACELEVEL_FATAL,"GpsThread:GNSS proxy open failed!");
         et_vErrmemStringNormal( TR_COMP_SENSOR,"GpsThread:Opening GNSS Proxy failed!");
         //FATAL_M_ASSERT_ALWAYS();
      }
     else
     {
        if( poGpsEncoderIf != OSAL_NULL )
        {
          if( poGpsEncoderIf->bInitialize( ) != TRUE )
          {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,"GpsThread:Error initializing GPS Encoder.");
          }
        }
        else
        {
           vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,"poGpsEncoderIf is null");
        }
     }
   }
   else
   {
      bRet = FALSE;
   }

   //if everything is fine, flush buffer and update registry with CRC checksum.
   if( bRet )
   {
      //should flush gnss buffer? if yes go and flush.
      if( bFlushGnssBuf && !bFlushGnssBuffer() )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsThread:flushing gnss buffer on scc failed!!!");
      }
      //flushing buffer is not required. just print a trace saying so.
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "GpsThread:no need to flush gnss buffer!!!");
      }
      vUpdateCrcCheckSumToReg();
   }

   return bRet;
}

//***************** F U N C T I O N  H E A D E R *****************************
//! DESCRIPTION:
//! \brief  Sends flush request to GNSS driver on SCC.
//! \return true : In case of success.
//!         false: In case of failure.
//****************************************************************************
//! HISTORY:
//----------------------------------------------------------------------------
// Date          | Author     | MODIFICATION
// ---------------------------------------------------------------------------
// 07/Mar/2016   | Sanjay G   | Initial version
//****************************************************************************
tBool tclGpsThread::bFlushGnssBuffer( tVoid )
{
   tU8 u8RetryCount = 0;
   tS32 s32RetVal;
   tBool bRet = false;

   while( u8RetryCount < VDS_GNSS_SCC_BUFFER_FLUSH_RETRY_LIMIT )
   {
      if( OSAL_OK == ( s32RetVal = OSAL_s32IOControl( SensorFd,
                                  OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA,0 )) )
      {
         //flushing is successful. no need to 
         //try again so set retry count to maximum to exit loop.
         u8RetryCount  = VDS_GNSS_SCC_BUFFER_FLUSH_RETRY_LIMIT;
         bFlushGnssBuf = false;
         bRet = true;
      }
      u8RetryCount++;
   }

   return bRet;
}

//***************** F U N C T I O N  H E A D E R *****************************
//! DESCRIPTION:
//! \brief  Updates CRC checksum to registry.
//! \return void
//****************************************************************************
//! HISTORY:
//----------------------------------------------------------------------------
//! Date          | Author     | MODIFICATION
//!---------------------------------------------------------------------------
//! 07/Mar/2016   | Sanjay G   | Initial version
//****************************************************************************
tVoid tclGpsThread::vUpdateCrcCheckSumToReg( tVoid )
{

   reg_tclRegKey oRegistry;
   OSAL_trGnssConfigData rGnssConfigData = { GNSS_HW_UNKNOWN,0,0,0,0,0 };

   //open registry.
   if( !oRegistry.bOpen( VDS_REG_KEY_CRC_CHECKSUM ) )
   {
      //failed to open registry. print a trace.
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GpsThread:failed to open registry" );
   }
   //opening registry is successful.get config data
   //from driver, update crc checksum to registry.
   else
   {
      //get config data.
      if( !bGetGnssConfigData( &rGnssConfigData ) )
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsThread:failed to get config data" );
      }
      //update crc checksum value to registry.
      else if(oRegistry.bSetU32( "GNSS_FW_CRC_CHECKSUM",
                                 rGnssConfigData.u32GnssRecvFwCrc))
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "GpsThread:GNSS_FW_CRC_CHECKSUM = 0x%x",
                    rGnssConfigData.u32GnssRecvFwCrc );
      }
      //failed to update registry. print a trace.
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsThread:failed to set registry key" );
      }

      oRegistry.vClose();
   }

   return;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Stops Dev Gps
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of Failure\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bStopDevGnss
(
   //! None
   tVoid
)
{
   tS32   s32RetVal;
   tBool  bRet = TRUE;
   tU8    u8WaitCount = 0;

   // Wait until GPS calculation is finished or stop after 3 secs
   //   while( ( !bCalcGpsOld || bCalcGps ) && (++u8WaitCount < 30) )
   while( bCalcGps  && (++u8WaitCount < 30) )
   {
      (tVoid)OSAL_s32ThreadWait(100);
      bCalcGpsOld = bCalcGps;
   }
   bCalcGps = FALSE;
   bCalcGpsOld = FALSE;

   if( bGpsRunning && (s32GetInternalState()==VDS_C_S32_THREAD_RUNNING) )
   {
      bGpsRunning = FALSE;

      s32RetVal = OSAL_s32IOClose( SensorFd );
      if(s32RetVal != OSAL_OK)
      {
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsThread:Error in closing GPS/GNSS device.");
         bRet = FALSE;
      }
      else
      {
         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "GpsThread:GPS/GNSS closed successfully.");
      }

      if( poGpsEncoderIf != OSAL_NULL )
      {
         if( poGpsEncoderIf->bShutdown( ) != TRUE )
         {
            vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                       "GpsThread:Error shutting down GPS Encoder.");
         }
      }

      if( bRet )
      {
         SensorFd = OSAL_ERROR;
         bGpsRunning = FALSE;
      }
   }
   else
   {
      SensorFd = OSAL_ERROR;
      bRet = FALSE;
   }

   return bRet;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Sets GPS thred event with the event provided
//! \return
//!  None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vSetGpsThreadEvent
(
   //! (I) : Event
   tU8 u8Event
)
{
   u8GpsThreadEvent = u8Event;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Checks the Gps thread event and performs the respective activity based on the
//!   event
//! \return
//!   - \c  TRUE : In case of Success\n
//!   - \c  FALSE : In case of Failure\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bCheckGpsThreadEvent
(
   //! None
   tVoid
)
{
   tBool bRetVal = TRUE;
   tU8   u8InitialGpsThreadEvent;

   if(u8GpsThreadEvent != VDS_GPS_EVENT_NOEVENT)
   {
      do
      {
         u8InitialGpsThreadEvent = u8GpsThreadEvent;

         vTraceMsg( VDS_C_TRACELEVEL_COMPONENT,
                    "GpsThread:bCheckGpsThreadEvent: %d",
                    u8GpsThreadEvent);

         switch(u8GpsThreadEvent)
         {
            case VDS_GPS_EVENT_NOEVENT :
            {
               //Nothing;
            }
            break;

            case VDS_GPS_EVENT_RESTART :
            {
               bRetVal = bStopDevGnss();
               (tVoid)OSAL_s32ThreadWait(1000);
               bRetVal |= bStartDevGnss();
            }
            break;

            case VDS_GPS_EVENT_START :
            {
               bRetVal = bStartDevGnss();
            }
            break;

            case VDS_GPS_EVENT_STOP :
            {
               bRetVal = bStopDevGnss();
               (tVoid)OSAL_s32ThreadWait(1000);
            }
            break;

            case VDS_GPS_EVENT_SEND_OSAL_TIME_UPDATE :
            {
               if( pfs32AddOsalTimeEvent != NULL )
               {
                  if( VDS_E_NO_ERROR != (*pfs32AddOsalTimeEvent)() )
                  {
                     bRetVal = FALSE;
                  }
               }
            }
            break;

            case VDS_GPS_EVENT_FACTORY_SETTINGS:
            {
               bFactorySettingsReset = true;
               bRetVal = bStopDevGnss();
               (tVoid)OSAL_s32ThreadWait(1000);
               bRetVal |= bStartDevGnss();
               bFactorySettingsReset = false;
            }
            break;

            default:
            {
              //Nothing
            }
            break;
         }
         // Now check, whether a new GPS event occurred in the meantime.
         // If yes, work on the new event.
         // Abort loop in case of error.
      }while( (u8GpsThreadEvent != u8InitialGpsThreadEvent) && bRetVal );

      u8GpsThreadEvent = VDS_GPS_EVENT_NOEVENT;
   }

   return bRetVal;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Sets the application state with the provided state information
//!
//! \return
//!  None
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tVoid tclGpsThread::vSetAppState
(
   //! (I) : App state information
   tU32 u32AppState
)
{
   u32ApplicationState = u32AppState;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Gets the application state information
//!
//! \return
//!  tU32: Current Application state
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tU32 tclGpsThread::u32GetAppState( tVoid ) const
{
   return u32ApplicationState;
}
// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Gets the OSAL time status
//!
//! \return
//!  tU8: Osal Time Status
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tU8 tclGpsThread::u8GetOsalTimeStatus
(
   //! None
)
{
   return u8OsalTimeStatus;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   The command input of trace is executed.
//!   Herein the command for destination of osal device DEV_GPS is carried out.
//! \return
//!   - \c  OSAL_OK : In  case of Success\n
//!   - \c  OSAL_ERROR : In  case of eRROR\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tS32 tclGpsThread::bExecuteGpsTraceCommand
(
   //! (I) : puchData Contains the byte buffer transmitted via trace.
   const tUChar* puchData
)
{
   tS32 s32RetVal = OSAL_ERROR;

   if (OSAL_NULL != puchData)
   {
      s32RetVal = OSAL_s32IOControl( SensorFd,
                                     OSAL_C_S32_IOCTL_GPS_SET_TRACE_COMMAND,
                                     (tLong)puchData);
   }

   return s32RetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!   Enable resp. disable the china encoder\n
//!   A requirement of the production is to disable and enable the  encoder without a reset.
//! \return
//!   - \c  TRUE : In  case of Success\n
//!   - \c  FALSE : In  case of eRROR\n
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bEnDisChinaEnc
(
   //! (I) : Bool value to confirm whether inhibit or not
   tBool inhibit
)
{
   if (OSAL_NULL != poGpsEncoderIf)
   {
      return poGpsEncoderIf->bEnDisChinaEnc(inhibit);
   }
   else
   {
      return false;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief get the GNSS satellite system used
//!   
//! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
//  used satellite systems. If no satellite system could be determined it retruns 
//  0.
//******************************************************************************
tU8 tclGpsThread::u8GetGnssSatSys ( tVoid )
{
   tU8 u8ActGnssSatSys = (tU8) 0;
   tU32 u32ActGnssSatSys = 0;

   tS32 s32RetVal = OSAL_s32IOControl( SensorFd,
                                       OSAL_C_S32_IOCTRL_GNSS_GET_SAT_SYS,
                                       (tLong) &u32ActGnssSatSys);
   if ( OSAL_OK == s32RetVal )
   {
      u8ActGnssSatSys = (tU8) (u32ActGnssSatSys & (0x000000FF));
   }
   else
   {
      // u8ActGnssSatSys = 0;
      // TODO: remove this static setting. Check for implementation of ioctl
      u8ActGnssSatSys = 3;
   }

   return( u8ActGnssSatSys );
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the GNSS satellite system used persistently
//!   
//! \return value of type \c sensor_fi_tcl_b8_GnssSatSys. Method returns the actual
//  used satellite systems. If no satellite system could be determined it retruns 
//  0.
// ******************************************************************************
tU8 tclGpsThread::u8SetGnssSatSys( tU8 u8TargetGnssSatSys )
{
   tU32 u32TargetGnssSatSys = (tU32)u8TargetGnssSatSys;

   tS32 s32RetVal = OSAL_s32IOControl( SensorFd,
                                       OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS,
                                       (tLong)&u32TargetGnssSatSys );
   if ( OSAL_OK == s32RetVal )
   {
      return( (tU8)(u32TargetGnssSatSys & (0x000000FF)) );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "GpsThread:OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS failed!");
      return( u8GetGnssSatSys() );
   }
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the GNSS satellite system used non-persistantly
//!   
//! \return value of type \c sensor_fi_tcl_b8_DiagGnssSatSys. Method returns the actual
//  used satellite systems. If no satellite system could be determined it retruns 
//  0.
// ******************************************************************************
tU8 tclGpsThread::u8SetDiagGnssSatSys( tU8 u8TargetGnssSatSys )
{
   tU32 u32TargetGnssSatSys = (tU32)u8TargetGnssSatSys;

   tS32 s32RetVal = OSAL_s32IOControl( SensorFd,
                                       OSAL_C_S32_IOCTRL_GNSS_DIAG_SET_SAT_SYS,
                                       (tLong)&u32TargetGnssSatSys );
   if ( OSAL_OK == s32RetVal )
   {
      return( (tU8)(u32TargetGnssSatSys & (0x000000FF)) );
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_FATAL,
                 "GpsThread:OSAL_C_S32_IOCTRL_GNSS_DIAG_SET_SAT_SYS failed!");
      return( u8GetGnssSatSys() );
   }
}


// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION: 
//
//! \brief set the epoch WN
//!   
// ******************************************************************************
tBool tclGpsThread::bSetWnEpoch( const OSAL_trGnssTimeUTC &rfrTargetEpochUTC,
                                 OSAL_trGnssTimeUTC &rReturnActualDateUTC )
{
   tBool bRetVal = false;
   OSAL_trGnssTimeUTC rTargetEpochUTC = rfrTargetEpochUTC;

   tS32 s32RetVal =  OSAL_s32IOControl( SensorFd,
                                        OSAL_C_S32_IOCTL_GNSS_SET_EPOCH,
                                        (tLong)&rTargetEpochUTC );
   if ( OSAL_OK == s32RetVal )
   {
      rReturnActualDateUTC = rfrTargetEpochUTC;
         bRetVal = true;
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GpsThread:Setting epoch failed!");
   }

   return bRetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION: 
//
//! \brief Get the epoch WN
//!   
// ******************************************************************************
tBool tclGpsThread::bGetWnEpoch( const OSAL_trGnssTimeUTC &rActualDateUTC )
{
   tBool bRetVal = false;

   tS32 s32RetVal =  OSAL_s32IOControl( SensorFd,
                                        OSAL_C_S32_IOCTL_GNSS_GET_EPOCH,
                                        (tLong)&rActualDateUTC );
   if ( OSAL_OK == s32RetVal )
   {
      bRetVal = true;
   }
   else
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GpsThread:Getting epoch failed!");
   }

   return bRetVal;
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief set the timemode (auto/manual) and in case of manual the time
//!   
//! \return 
// ******************************************************************************
tBool tclGpsThread::bSetTimeMode
( 
   const vds_tenTimeMode &enTargetTimeMode, 
   const OSAL_trTimeDate &rfTargetTimeDate
)
{
   tBool bRet = false;

   if ( bCritSectionBegin () )
   {
      // update member variables
      enTimeMode = enTargetTimeMode;
      rManualDateTime = rfTargetTimeDate;

      // udate system time
      switch (enTargetTimeMode)
      {
         case enTimeModeManual:
            bRet = bSetSystemTime( rfTargetTimeDate );
         break;
         case enTimeModeAuto:
            // TODO: implement
         break;
         default:
         break;
      }

      vCritSectionEnd();
   }
   return (bRet);
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief get actualTimeMode 
//!   
//! \return 
// ******************************************************************************
tBool 
tclGpsThread::bGetTimeMode ( vds_tenTimeMode & rfenActualTimeMode )
{
   tBool bRet = false;

   if ( bCritSectionBegin () )
   {
      rfenActualTimeMode = enTimeMode;
      vCritSectionEnd ();
      bRet = true;
   }
   return (bRet);
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief Sets different flags in GpsThread.
//!   
//! \return 
// ******************************************************************************
tVoid
tclGpsThread::vSetGpsThreadFlags( vds_tenGpsThreadFlags enFlag, tBool bFlagStatus )
{

   switch( enFlag )
   {
      case enbFlushGnssBuf:
         bFlushGnssBuf = bFlagStatus;
      break;

      case enbInformVdsServiceAvail:
         bInformVdsServiceAvailability = bFlagStatus;
      break;

      default:
         vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                    "GpsThread::vSetGpsThreadFlags:invalid flag");
      break;
   }
}

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//! \brief
//!  Gets GNSS configuration data from OSAL GNSS Proxy.
//! \return
//!   Succes code in case of success is:
//!   - \c  TRUE : Success\n
//!   Error code in case of failure are:\n
//!   - \c FALSE : Failure 
//  HISTORY:
// Date            |  Author                       | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
tBool tclGpsThread::bGetGnssConfigData
(
   OSAL_trGnssConfigData *prGnssConfigData
)
{
   tBool bRetVal = TRUE;

   if(OSAL_OK != OSAL_s32IOControl( SensorFd,
                                    OSAL_C_S32_IOCTRL_GNSS_GET_CONFIG_DATA,
                                   (tLong)prGnssConfigData ))
   {
      vTraceMsg( VDS_C_TRACELEVEL_ERROR,
                 "GpsThread::bGetGnssConfigData:IOControl Failed");
      bRetVal = FALSE;
   }

   return bRetVal;
}

//EOF
