/*!
 * \file tclAilVDSensor.h
 *
 * \par SW_COMPONENT
 *   VD-Sensor
 *
 * \brief
 *    class-declaration: inherits from ail_tclAppInterface,
 *                 state- and message-handling
 *
 * \author CM-DI/ESA1-Fischer
 *
 * \par COPYRIGHT (c) 2002 Blaupunkt GmbH
 *
 * \date 20.03.02 Rev. 1.0 CM-DI/ESA1-Fischer Initial Revision.
 * \date 30.09.09 Rev. 1.1 RBEI/ECF1-Jeryn Mathew  Added doxygen headers for all classes/funcs
 * \date 16.12.10 Rev. 1.2 RBEI/ECF1-Sainath Kalpuri - Diaglog related modifications have been done
 * \date 10.03.11 Rev. 1.2 RBEI/ECF1-Sainath Kalpuri Accelerometer related modifications have been done
 * \date 04.05.11 RBEI CM-AI/PJ-CF31 - sainath - Low Voltage event handling related modifications have been done
 */                                    
#ifndef tclAilVDSensor_h
#define tclAilVDSensor_h 1

#include "vds_tcltracehandler.h"

#define VDS_E_DISPATCH_ERROR              -1
#define VDS_E_AILVDSENSOR_NOT_INITIALIZED -2
#define VDS_E_PARAMETER_INVALID           -3

// errors
typedef enum
{
   VDS_E_THREADCREATE_ERROR   =  ( tS32 ) - 1,
   VDS_E_AIL_SUBSCRIBER_ERROR =  ( tS32 ) - 2,
   VDS_E_AIL_SEMAPHORE_ERROR  =  ( tS32 ) - 3,
   VDS_E_AIL_STATE_NOT_NORMAL =  ( tS32 ) - 4,
   VDS_E_AIL_ALLOC_ERROR      =  ( tS32 ) - 5
} vds_tenAilError;

class amt_tclServiceData;
class tclSubscriberManager;
class tclMsgGpsIf;
class tclMsgGyroIf;
class tclMsgOdometerIf;
class vds_tclMsgAbsIf;
#ifdef VARIANT_S_FTR_IERROR_MESSAGE
class tclMsgIerrorIf;
#endif
class tclSensorMsgDispatcher;


/*!
 * \brief Abstract interface for notification about service
 * registration and state changes.
 */
class vds_tclServiceListenerIf
{
   public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Default Destructor
      //! \return
      //!    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual ~vds_tclServiceListenerIf
      (
         //! None
      ) {}

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Called when VD Sensor has registered as a client of a service.
      //! \return
      //!    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual void vNotifyRegistration
      (
         //! (I) : Service Id
         tU16 u16ServiceId,
         //! (I) : Register Id
         tU16 u16RegID
      ) = 0;

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Called when VD Sensor has received a service availability message.
      //! \return
      //!    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual void vNotifyServiceAvailability
      (
         //! (I) : Service Id
         tU16 u16ServiceId,
         //! (I) : Service State
         tU8 u8ServiceState
      ) = 0;

};

/*!
 * \brief Manages all service listeners.
 *
 * \bug Only one listener may be registered.
 */
class vds_tclServiceClientManager
{
      static const tU32 cu32MaxListeners = 10;
      vds_tclServiceListenerIf *paoListeners[cu32MaxListeners];
      tU32 u32NumListeners;

   public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Initializers num of listeners to OSAL_NULL
      //! \return
      //!    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      vds_tclServiceClientManager() : u32NumListeners( 0 )
      {
         /* Not necessary, satisfy Lint */
         for ( tU32 u32Ndx = 0; u32Ndx < cu32MaxListeners; ++u32Ndx )
         {
            paoListeners[u32Ndx] = OSAL_NULL;
         }
      }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Adds the listeners
      //! \return
      //!     TRUE : In Case of Success\n
      //!     FALSE : In Case of Falure\n
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tBool bAddListener
      (
         //! (O) : Pointer to Listener
         vds_tclServiceListenerIf *poListener
      )
      {
         if ( u32NumListeners >= cu32MaxListeners )
         {
            return false;
         }
         else
         {
            paoListeners[u32NumListeners] = poListener;
            ++u32NumListeners;
            return true;
         }
      }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Removes the listener
      //! \return
      //!     TRUE : In Case of Success\n
      //!     FALSE : In Case of Falure\n
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tBool bRemoveListener
      (
         //! (I) : Pointer to Listner
         const vds_tclServiceListenerIf *poListener
      )
      {
         for ( tU32 u32Idx = 0 ; u32Idx < u32NumListeners ; u32Idx++ )
         {
            if ( paoListeners[u32Idx] == poListener )
            {
               for ( tU32 u32Idx2 = u32Idx + 1 ; u32Idx2 < u32NumListeners ; u32Idx2++ )
               {
                  paoListeners[u32Idx2-1] = paoListeners[u32Idx2];
               }

               --u32NumListeners;
               return true;
            }
         }

         return false;
      }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Notifies the registration
      //! \return
      //!    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vNotifyRegistration
      (
         //! (I) : Pointer to Service Id
         tU16 u16ServiceId,
         //! (I) : Pointer to Register Id
         tU16 u16RegID
      )
      {
         for ( tU32 u32Idx = 0 ; u32Idx < u32NumListeners ; u32Idx++ )
         {
            paoListeners[u32Idx]->vNotifyRegistration( u16ServiceId, u16RegID );
         }
      }
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //  Parameters : None
      //! \brief
      //!    Notifies the service availability
      //! \return
      //!    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vNotifyServiceAvailability
      (
         //! (I) : Service Id
         tU16 u16ServiceId,
         //! (I) : Service State
         tU8 u8ServiceState
      )
      {
         for ( tU32 u32Idx = 0 ; u32Idx < u32NumListeners ; u32Idx++ )
         {
            paoListeners[u32Idx]->vNotifyServiceAvailability( u16ServiceId, u8ServiceState );
         }
      }
};

extern vds_tclServiceClientManager vds_goServiceClientManager;

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//  brief
//    Creates a VD Sensor application object
//  Parameters : None
//  return
//    returns the application interface
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
ail_tclAppInterface* vds_poCreateApplication( void );

// ***************** F U N C T I O N  H E A D E R *****************************
//
//  DESCRIPTION:
//
//  brief
//    Destroys a VD Sensor application object
//  Pointer : Pointer to application interface to be destroyed
//  return
//!   None
//  HISTORY:
// Date         |  Author              | MODIFICATION
// ----------------------------------------------------------------------------
//******************************************************************************
void vds_vDestroyApplication( ail_tclAppInterface* );

/*!
 * \brief Handles the main application logic for VD Sensor
 *
 * This class handles the communication with other CCA applications.
 * It starts and stops the other parts of VD Sensor.
 */
class tclAilVDSensor : public ail_tclAppInterface
{
   public:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Default Constructor
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tclAilVDSensor();
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Default Destructor
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual ~tclAilVDSensor();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Save availability information for the DiagLog service.
      //    Notifies the service client manager about changes in service availability.
      //  Parameters :
      //    tU16 u16ServiceId - Service Id
      //    tU16 u16ServerId - Server Id
      //    tU16 u16RegisterId - Register Id
      //    tU8 u8ServiceState - Service State
      //    tU16 u16SubId - SubId
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tVoid vOnServiceState (
         tU16 u16ServiceId,
         tU16 u16ServerId,
         tU16 u16RegisterId,
         tU8 u8ServiceState,
         tU16 u16SubId );

      #ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Register for Diaglog SendNextTestResultUpreg property
      //  Parameters
      //    u16DiaglogRegisterID - Diaglog service register ID
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      virtual tVoid vRegisterDiagLogSendNextTestResultUpReg
      (
         tU16 u16DiaglogRegisterID
      );
      #endif

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Handle application state changes.
      //    Inherited from class ail_tclAppInterfaceRestricted
      //    (The interface framework calls "vOnNewAppState()" whenever a
      //    new POWER-Message has been received. As default it mirrors all POWER
      //    commands to satisfy the LPM-interface. For specific application behavior,
      //    the user must overwrite this function. ) calls vSwitchFrom___ depending on
      //    u32OldAppState
      //  Parameters :
      //    tU32 u32OldAppState - the last confirmed application state
      //    tU32 u32AppState - the new requested application state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02       CM-DI/ESA1-Fischer   Initial Revision.
      //******************************************************************************
      virtual tVoid vOnNewAppState ( tU32 u32OldAppState, tU32 u32AppState );


      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Handle Service Data CCA messages
      //    Inherited from ail_tclAppInterfaceRestricted ( The interface
      //    framework calls "vOnNewMessage()" whenever a new Service
      //    Data-Message has been received. As default it releases the
      //    message buffer which belongs to "amt_tclBaseMessage*". The
      //    user has to overwrite this function to implement his own
      //    dipatcher here. Then it's his responsibility to free the
      //    message buffer. ) Handles Sensor- or SensorDiag-Message
      //    depending on ServiceId
      //  Parameters :
      //    amt_tclBaseMessage* poMessage - pointer to the received ServiceData-Message
      //  return
      //   None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      virtual tVoid vOnNewMessage ( amt_tclBaseMessage* poMessage );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Inform the application that a client has deregistered itself from a service.
      //    Inherited from ail_tclAppInterfaceRestricted ( The interface
      //    framework calls "vOnUnregister()" every time, when a valid
      //    Service Unregister-Message has been received. This function
      //    informs the application, that all requests label with this
      //    RegisterId has got obsolete. As default no corresponding
      //    action is done. The user has to overwrite this function and
      //    to clear the RegisterId from his application
      //    references. Please note, that the framework will assign a
      //    free RegisterId next time a service is requested by some
      //    client. ) All registered notifications will be deleted
      //  Parameters :
      //    tU16 u16ServiceId -  Service Id
      //    tU16 u16RegisterId - Register Id
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  28.05.02     CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      virtual tVoid vOnUnregister ( tU16 u16ServiceId, tU16 u16RegisterId );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Initialize the application.
      //    Inherited from class ail_tclAppInterfaceRestricted ( The
      //    interface framework calls "bOnInit()" before starting any
      //    communication action.  As default this function always
      //    returns TRUE. The user has to implement all his local
      //    application initialization ( creating required threads,
      //    allocate application memory, ... ). If returns TRUE,
      //    initializing could be successfully performed and framework
      //    will register application at the LocalPower Management, else,
      //    the application would end itself. )
      //    Initializes Sensor-Threads, Interfaces to persistent data,
      //    Interface to report-memory, ...
      //  Parameters :
      //    None
      //  return
      //     TRUE : application could be successfully initialized\n
      //     FALSE : application could be successfully not initialized\n
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      virtual tBool bOnInit ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Tells the application to close itself.
      //    Inherited from ail_tclAppInterfaceRestricted.  This function
      //    is called, when the application shall be immediatley closed.
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision
      //******************************************************************************
      virtual tVoid vOnApplicationClose ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Query supported service version.
      //    The server application has to override this method to report the
      //    actual version number to a given service id
      //  Parameters :
      //     tU16 u16MyServiceID - service id
      //     tU16 &rfu16MajorVersion - Major version
      //     tU16 &rfu16MinorVersion - Minor version
      //     tU16 &rfu16PatchVersion - Patch version
      //  return
      //    TRUE : server supports requested service\n
      //    FALSE : server does not support requested service\n
      //  HISTORY:
      // Date         |  Author        | MODIFICATION
      // ----------------------------------------------------------------------------
      //  11.02.02      PDP3-Andres       Initial version.
      //******************************************************************************
      virtual tBool bGetServiceVersion (
         tU16 u16MyServiceID,
         tU16 &rfu16MajorVersion,
         tU16 &rfu16MinorVersion,
         tU16 &rfu16PatchVersion );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Return application state of VD Sensor
      //  Parameters : None
      //  return
      //    application state (AMT-states)
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tU32 u32GetAppState ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Start sensor threads.
      //  Parameters ; None
      //  return
      //    Succes code in case of success is:
      //      VDS_E_NO_ERROR : Success\n
      //   Error code in case of failure are:\n
      //      VDS_E_THREADCREATE_ERROR : Thread Create Error
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      tS32 s32StartSensorThreads ();


      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Stop sensor threads.
      //  Parameters : None
      //  return
      //    Succes code in case of success is:
      //     VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //     VDS_E_THREADCREATE_ERROR : Thread Create Error
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02      CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      tS32 s32StopSensorThreads ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Delete sensor threads.
      //  Parameters : None
      //  return
      //    Succes code in case of success is:
      //      VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //      VDS_E_THREADCREATE_ERROR : Thread Create Error
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02      CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      tS32 s32DeleteSensorThreads ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   Sends message to subscribers
      //  return
      //    Succes code in case of success is:
      //       VDS_E_NO_ERROR : Success\n
      //    Error code in case of failure are:\n
      //       VDS_E_AILVDSENSOR_NOT_INITIALIZED : AilVdSensor not initialized
      //       VDS_E_AIL_STATE_NOT_NORMAL : AIL state is not normal
      //       VDS_E_AIL_ALLOC_ERROR : AIL Allocation error
      //       VDS_E_INVALID_PARAMETER : Invalid Parameter
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  08.04.02      CM-DI/ESA1-Fischer     Initial Revision
      //  23.04.02      CM-DI/ESA1-Fischer     Parameter changed to tclSubscriberManager*
      //******************************************************************************
      static tS32 s32SendSubscriberMessage (
         amt_tclServiceData *poServiceData,
         tclSubscriberManager *poSubscriber );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Send message (and delete if failed)
      //  Parameter :
      //    amt_tclServiceData *poServiceData - Pointer to Service Data
      //  return
      //     TRUE : Message Post Success\n
      //     FALSE : Message Post Failed\n
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  23.04.02       CM-DI/ESA1-Fischer     Initial Revision
      //******************************************************************************
      tBool bPostMessage ( amt_tclServiceData *poServiceData );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Simulate a toggle of DiagLog service availability. This is used to
      //    force the report memory module to resend the error states to VD DiagLog.
      //    If the Diaglog service is not available, this function does nothing.
      //  return
      //   None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vToggleDiagLog( void );
      
            // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    If Lowvoltage Event is occurred sets the internal member of tclAilVDSensor
      //  return
      //   None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
      void vSetLowVoltageEvent( tBool bCriticalLowVoltageEvent );
      #endif

   private:
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    close file-handles, free memory, close semaphores, ...
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  24.05.02       CM-DI/ESA1-Fischer    CM-DI/ESA1-Fischer
      //******************************************************************************
      tVoid vCloseVDSensor ();

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   read submessages of tablelist and generate
      //!   answermessage with tablelist of response submessages.
      //! \return
      //!   None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tVoid vDispatchSensorDiagMsg
      (
         //! (I) : Pointer to service data
         amt_tclServiceData* poServiceData
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Handle incoming CCA Service Data messages.
      //    handles following message-opcodes\n
      //    Get: collect data from Msg*If and send it back\n
      //    Set: set data in Msg*If and send status-msg back\n
      //  Check Subscriber :
      //    UpReg: create SubscriberElement
      //    RelUpReg: delete SubscriberElement
      //    MethodStart: calulate result, check for subscriber
      //    MethodAbort: stops calculation
      //  Parameter :
      //    amt_tclServiceData* poServiceData - Pointer to service data
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tVoid vDispatchSensorMsg( amt_tclServiceData* poServiceData );
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Register Service for Diag Log
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      void vRegisterDiagLog( void );
      tU16 u16DiagLogRegisterID;
      tU8 u8DiagLogServiceState;
      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Dispatches the messages received Diaglog (Server)
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
#ifndef VARIANT_S_FTR_DISABLE_DIAGLOG
      tVoid vDispatchDiaglogMsg( amt_tclServiceData* poServiceData );
#endif

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    Register Service for SPM.
      //  Parameters : None
      //  return
      //!   None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //******************************************************************************
      #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
      void vRegisterSPM( void );
      tU16 u16SPMRegisterID;
      tU8 u8SPMServiceState;
      #endif

      virtual tVoid vOnAsyncRegisterConf (tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId);

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    handles transition from state initialized to given state
      //  Parameters :
      //    tU32 u32NewAppState - switch to this application-state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02      CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      tVoid vSwitchFromInitialized ( tU32 u32NewAppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    handles transition from state normal to given state
      //  Parameters :
      //    tU32 u32NewAppState - switch to this application-state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02      CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      tVoid vSwitchFromNormal ( tU32 u32NewAppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    handles transition from state diagnosis to given state
      //  Parameters :
      //    tU32 u32NewAppState - switch to this application-state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02      CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      tVoid vSwitchFromDiagnosis ( tU32 u32NewAppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    handles transition from state ReceiveReady to given state
      //  Parameters :
      //     tU32 u32NewAppState - switch to this application-state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tVoid vSwitchFromReceiveReady ( tU32 u32NewAppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    handles transition from state pause to given state
      //  Paramters :
      //   tU32 u32NewAppState - switch to this application-state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  21.03.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tVoid vSwitchFromPause ( tU32 u32NewAppState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    handles transition from state off to given state
      //  Paramters :
      //   tU32 u32NewAppState - switch to this application-state
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  17.05.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tVoid vSwitchFromOff ( tU32 u32NewAppState );


      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //! \brief
      //!   creates table with supported commands and generates answermessage.
      //! \return
      //!   None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  25.04.02      CM-DI/ESA1-Fischer    Initial Revision
      //******************************************************************************
      tVoid vHandleMsgDiagListGet
      (
         //! (I) : pointer to message with ServiceId 'Diag'
         amt_tclServiceData *poServiceData
      );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //    sets new interface-state (normal or pending)
      //  Paramters :
      //   tS32 s32NewState - new state (VDS_C_S32_STATE_NORMAL or VDS_C_S32_STATE_PENDING)
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  28.05.02     CM-DI/ESA1-Fischer      Initial Revision.
      //******************************************************************************
      tVoid vSetInterfaceState ( tS32 s32NewState );

      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   locks access to critical sections.only when TRUE is returned, access
      //   is allowed
      //  Parameters : None
      //  return
      //     TRUE : access allowed
      //     FALSE : error, access is not allowed
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  22.05.02      CM-DI/ESA1-Fischer     Initial Revision.
      //******************************************************************************
      tBool bCritSectionBegin ();


      // ***************** F U N C T I O N  H E A D E R *****************************
      //
      //  DESCRIPTION:
      //
      //  brief
      //   unlocks access to critical sections.
      //  Parameters : None
      //  return
      //    None
      //  HISTORY:
      // Date         |  Author              | MODIFICATION
      // ----------------------------------------------------------------------------
      //  22.05.02      CM-DI/ESA1-Fischer    CM-DI/ESA1-Fischer
      //******************************************************************************
      tVoid vCritSectionEnd ();
      tBool bIsbOnInitFailed;
      tU32 u32AppState;
      tclMsgGpsIf *poMsgGpsIf;
      tclMsgGyroIf *poMsgGyroIf;
      tclMsgAccIf *poMsgAccIf;
      tclMsgOdometerIf *poMsgOdometerIf;
      vds_tclMsgAbsIf *poMsgAbsIf;
#ifdef VARIANT_S_FTR_IERROR_MESSAGE
      tclMsgIerrorIf *poMsgIerrorIf;
#endif
      tclSensorMsgDispatcher *poSensorMsgDispatcher;
      OSAL_tSemHandle hVDSensorSemaphore;
      static tCString coszVDSensorSemName;
      vds_tclTraceHandler *poTraceHandler;
      OSAL_tIODescriptor hTrace;
      #ifdef VDS_S_ENABLE_LOW_VOLTAGE_HANDLING
      tBool bIsLowVoltageEventTriggerred;
      #endif
      tBool bisDiaglogServiceavailable;
      tBool bRegisterAsyncRet;
   public:
      static tclAilVDSensor *poThisAppInterface;
};

#endif
