/*
 * ShellCmd.cpp
 *
 * See ShellCmd.h for API description.
 *
 * Created on May 7 2020
 *      Author: Maxime Chemin
 */

#include <regex>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <cerrno>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "ShellCmd.h"

#define DEFAULT_BUFFER_SIZE 4096

#define MAX_PID 32768

/**
 * Default environment used with the exec command in the class
 */
char * const ShellCmd::default_env[] = {
    "HOME=/",
    "PATH=/bin:/sbin:/usr/bin:/usr/sbin:/opt/bosch/base/bin",
    NULL
};

/**
 * Convert the vector string vector to a const char vector
 *
 * The string type does not look like a char array in memory and therefore
 * cannot be used by the exec call. This function will convert it to a const
 * char vector that can be used later for an exec call.
 */
std::vector<const char*> make_argv(std::vector<std::string> const& in){
    std::vector<const char*> argv;

    for (const auto& s : in) {
        argv.push_back( s.data() );
    }
    argv.push_back(NULL);
    argv.shrink_to_fit();

    return argv;
}

ShellCmd::ShellCmd(std::string program){
    int res;
    progname = program;
    outputToPipe = false;
    isBackgroundTask = false;
    cmdPid = -1;
    /* When calling exec you always need to have the program name as first
     * argument.
     */
    arguments.push_back(program);

    res = pipe(pipeLink);

    if (-1 == res) {
        perror("Pipe creation Failed");
    }
}

ShellCmd::~ShellCmd(){
    close(pipeLink[READ_PIPE]);
    close(pipeLink[WRITE_PIPE]);
}

void ShellCmd::addArgs(std::string args){
    size_t startPos, endPos, size;

    startPos = 0;
    endPos = args.find(' ');

    while (endPos != std::string::npos) {
        size = endPos - startPos;
        arguments.push_back(args.substr(startPos, size));

        startPos = endPos + 1;
        endPos = args.find(' ', startPos);
    }

    endPos = std::min(endPos, args.size());
    size = endPos - startPos;
    arguments.push_back(args.substr(startPos, size));
}


void ShellCmd::setOutputFile(std::string filepath){
    outputpath = filepath;
}

void ShellCmd::setOutputPipe(bool state){
    outputToPipe = state;
}

void ShellCmd::setBackgroundTask(bool state) {
    isBackgroundTask = state;
}

std::string ShellCmd::getFullCommandLine(){
    std::string cmdline = progname;
    auto it = std::begin(arguments);

    // skip first argument as that is the same as progname
    it++;
    cmdline += ":";

    for (auto end = std::end(arguments); it != end; it++) {
        cmdline += " ";
        cmdline += *it;
    }

    return cmdline;
}

pid_t ShellCmd::getRunningTaskPid() {
    return cmdPid;
}

void ShellCmd::redirectOutputToPipe() {
    int res;
    // Pipe stdout to the file descriptor
    res = dup2(pipeLink[WRITE_PIPE], STDOUT_FILENO);
    if (-1 == res) {
        perror("Stdout dup2 failed");
    }
}

void ShellCmd::redirectOutputToFile(const std::string &outPath){
    int fd, res;

    fd = open(outPath.c_str(), O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);

    if (fd != -1){
        // Pipe stdout and stderr to the file descriptor
        res = dup2(fd, STDOUT_FILENO);
        if (-1 == res) {
            perror("Stdout dup2 failed");
        }

        res = dup2(fd, STDERR_FILENO);
        if (-1 == res) {
            perror("Stderr dup2 failed");
        }
        // not required anymore
        close(fd);
    }
}
int ShellCmd::waitForChild(int pid){
    int status;

    while (-1 == waitpid(pid, &status, 0)){
        if(errno != EINTR){
            break;
        }
    }

    if (WIFEXITED(status)){
        return WEXITSTATUS(status);
    } else {
        return -1;
    }
}

std::string ShellCmd::getCommandOutput() {
    std::string output;
    char buffer[DEFAULT_BUFFER_SIZE];
    int nbytes;

    do {
        nbytes = read(pipeLink[READ_PIPE], buffer, DEFAULT_BUFFER_SIZE);
        if (nbytes > 0) {
            output.append(buffer, nbytes);
        }
    } while (nbytes == DEFAULT_BUFFER_SIZE );

    return output;
}

int ShellCmd::exec(){
    int pid;
    int res = 0;
    std::vector<const char*> argv = make_argv(arguments);

    pid = fork();
    /* Saving pid so you can kill it later if its a background task */
    cmdPid = pid;

    if (-1 == pid){ /* Fork failed */
        return -1;
    } else if ( pid != 0){ /* Parent Process */
        /* Only wait if the task is not meant to be run in background */
        if (!isBackgroundTask)
            res = waitForChild(pid);
    } else { /* Child process */
        if (!outputpath.empty()){
            redirectOutputToFile(outputpath);
        } else if (outputToPipe) {
            redirectOutputToPipe();
        }
        /*
         * const can be safely casted away as the standard guarantees that it
         * should not be modified.
         */
        res = execvpe(progname.c_str(),
                     const_cast<char * const *>(argv.data()),
                     default_env);
        /*
         * If this is reached the exec command could not be executed correctly.
         * Just check the error and exit with the appropriate exit code;
         */
        if (EACCES == errno){
            exit(CMD_NOT_EXECUTABLE);
        } else if (ENOENT == errno){
            exit(CMD_NOT_FOUND);
        }
    }

    return res;
}

/*
 * Following manual the standard signals range from 1 to 31:
 * https://man7.org/linux/man-pages/man7/signal.7.html
 *
 * In addition to the standard 31 signals there is a 33 signal range for real
 * time signals.
 */
bool isSignal(const std::string &number){
    int signal = -1;
    // Maximum RT signal, this symbol is not exported by the standard lib
    int SIGMAX = 64;
    bool res = false;
    std::regex re("[0-9]+?");

    try {
        if (!regex_match(number, re)) {
            return false;
        }

        signal = std::stoi(number);

        if (signal >= SIGHUP && signal <= SIGMAX){
            res = true;
        } else {
            res = false;
        }
    } catch(std::invalid_argument &e) {
        std::cerr << e.what() << std::endl;
    } catch(std::out_of_range &e) {
        std::cerr << e.what() << std::endl;
    } catch(...) {
        // Just in case catch, if we end up here the result will be false
    }

    return res;
}

bool isPid(const std::string &process_id){
    pid_t pid = -1;
    bool res = false;
    std::regex re("[0-9]+?");

    try {

        if (!regex_match(process_id, re)) {
            return false;
        }

        pid = std::stoi(process_id);

        if (pid >= 1 && pid <= MAX_PID){
            res = true;
        } else {
            res = false;
        }
    } catch(std::invalid_argument &e) {
        std::cerr << e.what() << std::endl;
    } catch(std::out_of_range &e) {
        std::cerr << e.what() << std::endl;
    } catch(...) {
        // Just in case catch, if we end up here the result will be false
    }
    return res;
}

bool isServiceName(const std::string &service_name){
    std::regex re("[a-zA-Z0-9_-]+(.service)?");

    if (std::regex_match(service_name, re)){
        return true;
    } else {
        return false;
    }
}

bool isPath(const std::string &path){
    std::regex re("[a-zA-Z0-9/._-]+?");

    if (std::regex_match(path, re)){
        return true;
    } else {
        return false;
    }
}

bool isTaskName(const std::string &task_name){
    std::regex re("[a-zA-Z0-9/._-]+?");

    if (std::regex_match(task_name, re)){
        return true;
    } else {
        return false;
    }
}

static bool invalidChar(char c) {
    return !(c < 128);
}

void stripNonASCII(std::string &str) {
    str.erase(remove_if(str.begin(), str.end(), invalidChar), str.end());
}

void splitStringAtSpace(const std::string &strIn, std::string &first, std::string &sec) {
   size_t pos;

   pos = strIn.find(' ');
   first = strIn.substr(0, pos);
   sec = strIn.substr(pos + 1, strIn.size());
}
