/*
 * RootDaemonServer.cpp
 *
 * see RootDaemonServer.h for description
 *
 *  Created on: Oct 27, 2014
 *      Author: Mohamed Mostafa
 */
  
#include <fcntl.h>
#include <algorithm>    // std::find
#include <dirent.h>
#include <dlfcn.h>
#include <pthread.h>
#include "RootDaemonServer.h"
#include "IClientSharedlib.h"
#include "RootDaemonHelper.h"
#include "RootDaemonTypes.h"
#include <grp.h>
#include <pwd.h>
 
//TODO deallocation of objects on failures

//Initialize default values for static members
int RootDaemonServer::isStarted = 0;
map<int, client_lib_handles_t> * RootDaemonServer::pipeCommandMap_ptr = new map<int, client_lib_handles_t>();

void RootDaemonServer::start()
{
	int err;
	if(isStarted == 1)
		return;
	
	//INITIALIZATION
	if(pipeCommandMap_ptr == NULL)
	{
		try{ //CID 830164
			pipeCommandMap_ptr = new map<int, client_lib_handles_t>();
		}
		catch(const std::bad_alloc& e){
		syslog(LOG_ERR, "caught bad_alloc exception on pipeCommandMap_ptr");
		return;
		}
		pipeCommandMap_ptr = new map<int, client_lib_handles_t>();
	}
	isStarted = 1;
	
	//load client shared libs and form pipe command lookup table
	err = loadClientSharedLibs();
	if(err)
	{
		 syslog(LOG_ERR, "%s: Loading Client Shared Libs from Path:[%s] failed: error=%s", ROOTD_SERVER, SHARED_LIB_PATH,  strerror(errno));
		 return;
	}
	//syslog(LOG_ERR, "%s: RootDaemonServer::loadClientSharedLibs(): %d client shared libs were loaded", ROOTD_SERVER, pipeCommandMap_ptr.size());
	
	if(pipeCommandMap_ptr->size() > 0)
	{
		err = waitForCommands();
		if(err)
		{
			 syslog(LOG_ERR, "%s: waitForCommands:() failed: error=%d", ROOTD_SERVER, err);
			 return;
		}
	}
	else
	{
		syslog(LOG_ERR, "%s: RootDaemonServer::loadClientSharedLibs(): 0 shared libs were loaded", ROOTD_SERVER);
		return;
	}
}

void RootDaemonServer::stop()
{
	if(pipeCommandMap_ptr != NULL)
	{
		releaseLibHandlesMap();
		delete pipeCommandMap_ptr;
	}
	isStarted = 0;
}
 
void* RootDaemonServer::thread_callback(void *thread_args)
{
   cmd_thread_params_t * args = (cmd_thread_params_t *)thread_args;	
	if(args)
	{
		if(args->cmd_func_ptr)
      {
         //Execute Client Command
         TRACE(LOG_INFO, "%s: thread_callback() executing command num[%d] with args [%s]", ROOTD_SERVER, args->command_num, args->command_args.c_str());
         CmdData command_result = (*(args->cmd_func_ptr))(args->command_num, args->command_args);
         TRACE(LOG_INFO, "%s: thread_callback() command result returned [%d]", ROOTD_SERVER, command_result.errorNo);
         
         //Write Result to Result Pipe
         TRACE(LOG_INFO, "%s: thread_callback() writing to result FIFO[%d], cmd_num[%d], mesage[%s]", ROOTD_SERVER, args->res_pipe_fd, args->command_num, command_result.message);
         if(RootDaemonHelper::writeCommandResultToFIFO(args->res_pipe_fd, ROOTD_SERVER, args->command_num, command_result.message) == 0)
         {
            TRACE(LOG_INFO, "%s: thread_callback, success writing to FIFO", ROOTD_SERVER);
         }
      }
      else
      {
         syslog(LOG_ERR, "%s:thread_callback() failed: args->cmd_func_ptr is NULL", ROOTD_SERVER);
      }
		delete args;
	}
	else
	{
		syslog(LOG_ERR, "%s:thread_callback() failed: thread_args is NULL", ROOTD_SERVER);
	}
	TRACE(LOG_INFO, "%s:thread_callback, calling pthread_exit()", ROOTD_SERVER);
	pthread_exit(NULL);
}

int RootDaemonServer::loadClientSharedLibs()
{
   int ret=0;
	string command_fifo_path;
	string cmd_result_fifo_path;
	string full_lib_path;
	DIR *dp;
    struct dirent *dirp;
	void * libHandle;
	char *error;
	commandPtr_t cmd_ptr;
	getNamePtr_t getName_ptr;
	const char * clientName;
    const char * tmp_name;
	int uid;
	int gid;
	int cmd_pipe_fd;
	int res_pipe_fd;
	
	//scan directory for shared libs
   if((dp  = opendir(SHARED_LIB_PATH)) == NULL) {
      ret = errno;
      syslog(LOG_ERR, "%s: error opening dir(%s) failed: error=%s", ROOTD_SERVER, SHARED_LIB_PATH, strerror(ret));
      return ret;
   }
   //create pipes folder
   ret = mkdir(tmp_rootdeamon_dir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
   if(ret < 0)
   {
      if(errno == EEXIST)
      {
         errno = 0;
      }
      else
      {
	ret = errno;
	syslog(LOG_ERR, "%s: mkdir(%s) failed: error=%s", ROOTD_SERVER, tmp_rootdeamon_dir, strerror(ret));
	if(dp != NULL)//CID 830231
	{
		closedir(dp);
	}
         return ret;
      }
   }
   while ((dirp = readdir(dp)) != NULL) {
      //check list of libraries allowed to be loaded
      if(strcmp(dirp->d_name, ".") == 0 || strcmp(dirp->d_name, "..") == 0)
         continue;
      if(find(allowed_libs_vec.begin(), allowed_libs_vec.end(), dirp->d_name) != allowed_libs_vec.end())
      {
         //load dynamic symbols from the root daemon client shared lib
         full_lib_path.clear();
         full_lib_path.append(SHARED_LIB_PATH);
         full_lib_path.append(dirp->d_name);
         TRACE(LOG_INFO, "%s:found lib(%s)", ROOTD_SERVER, full_lib_path.c_str());//debug
         libHandle = dlopen(full_lib_path.c_str(), RTLD_LAZY);
         if(!libHandle)
         {
            syslog(LOG_ERR, "%s: error loading lib{%s%s}: error=%s", ROOTD_SERVER, SHARED_LIB_PATH , dirp->d_name, dlerror());
            ret = -1;
            goto cleanup;
         }
         TRACE(LOG_INFO, "%s:loaded lib(%s)", ROOTD_SERVER, full_lib_path.c_str());//debug
         cmd_ptr = (commandPtr_t)dlsym(libHandle, "command");
         if((error = dlerror()) != NULL)
	{
		syslog(LOG_ERR, "%s: error loading symbol(command) in {%s%s}: error=%s", ROOTD_SERVER, SHARED_LIB_PATH, dirp->d_name, error);
		ret = -1;
		dlclose(libHandle);//CID 830120
            goto cleanup;
	}
         getName_ptr = (getNamePtr_t)dlsym(libHandle, "getClientName");
         if((error = dlerror()) != NULL)
         {
            syslog(LOG_ERR, "%s: error loading symbol(getClientName) in {%s%s}: error=%s", ROOTD_SERVER, SHARED_LIB_PATH, dirp->d_name, error);
            ret = -1;
	    dlclose(libHandle);//CID 830120
            goto cleanup;
         }
         clientName = (*getName_ptr)();
         
         getName_ptr = (getNamePtr_t)dlsym(libHandle, "getClientGroupName");
         if((error = dlerror()) != NULL)
         {
            syslog(LOG_ERR, "%s: error loading symbol(getClientGroupName) in {%s%s}: error=%s", ROOTD_SERVER, SHARED_LIB_PATH, dirp->d_name, error);
            ret = -1;
	    dlclose(libHandle);//CID 830120
            goto cleanup;
         }
         tmp_name = (*getName_ptr)();
         gid = getGidByName(tmp_name);
         
         getName_ptr = (getNamePtr_t)dlsym(libHandle, "getClientUserName");
         if((error = dlerror()) != NULL)
         {
            syslog(LOG_ERR, "%s: error loading symbol(getClientUserName) in {%s%s}: error=%s", ROOTD_SERVER, SHARED_LIB_PATH, dirp->d_name, error);
            ret = -1;
	    dlclose(libHandle);//CID 830120
            goto cleanup;
         }
         tmp_name = (*getName_ptr)();
         uid = getUidByName(tmp_name);
         
         TRACE(LOG_INFO, "%s: clientname(%s), gid(%d), uid(%d) ", ROOTD_SERVER, clientName, gid, uid);//debug
         
         command_fifo_path = RootDaemonHelper::getCmdPipePath(clientName);
         cmd_result_fifo_path = RootDaemonHelper::getResultPipePath(clientName);
         
         TRACE(LOG_INFO, "%s: creating command and command result pipes, command_fifo_path(%s), cmd_result_fifo_path(%s) ", ROOTD_SERVER, command_fifo_path.c_str(),cmd_result_fifo_path.c_str());//debug

         //Create Command pipe
         if((cmd_pipe_fd = RootDaemonServer::createClientServerPipe(command_fifo_path, uid, gid, O_RDWR| O_NONBLOCK)) < 0)
         {
            syslog(LOG_ERR, "%s: error creating command pipe fd[%d] ", ROOTD_SERVER, cmd_pipe_fd);//debug
            dlclose(libHandle);//CID 830120
	    ret = cmd_pipe_fd;
            goto cleanup;
         }
         TRACE(LOG_INFO, "%s: success creating command pipe, fd[%d] ", ROOTD_SERVER, cmd_pipe_fd);//debug
         //Create Result pipe
         if((res_pipe_fd = RootDaemonServer::createClientServerPipe(cmd_result_fifo_path, uid, gid, O_RDWR | O_NONBLOCK)) < 0)
         {
            syslog(LOG_ERR, "%s: error creating result pipe fd[%d] ", ROOTD_SERVER, res_pipe_fd);//debug
	    close(cmd_pipe_fd); //CID 830284
	    dlclose(libHandle);//CID 830120
            ret = res_pipe_fd;
            goto cleanup;
         }
         TRACE(LOG_INFO, "%s: success creating result pipe, fd[%d] ", ROOTD_SERVER, res_pipe_fd);//debug
         (*pipeCommandMap_ptr)[cmd_pipe_fd] = { .lib_handle = libHandle, 
                        .cmd_func_ptr = cmd_ptr,
                        .command_pipe_fd = cmd_pipe_fd,
                        .result_pipe_fd = res_pipe_fd,
                        .user_id = uid,
                        .group_id = gid,
                        .cmd_fifo_path = command_fifo_path,
                        .result_fifo_path = cmd_result_fifo_path};
      }
      else
      {
         continue;
      }
   }
	if(pipeCommandMap_ptr->empty())
		goto cleanup;
	
	//All succeeded 
	if(dp != NULL)
	  closedir(dp);
	return 0;

	//We have an error
	cleanup:
		if(dp != NULL)
		  closedir(dp);
		//TODO on error clear all handles or continue with partial working system
		releaseLibHandlesMap();
		pipeCommandMap_ptr->clear();
		return ret;
}
void RootDaemonServer::releaseLibHandlesMap()
{
	int err;
	
	for(map<int, client_lib_handles_t>::iterator iter = pipeCommandMap_ptr->begin(); iter != pipeCommandMap_ptr->end(); ++iter)
	{
		//close handle to the client shared lib
		dlclose(iter->second.lib_handle);
		
		//Close and unlink command FIFO
		err = close(iter->first);
		if (err < 0) {
			syslog(LOG_ERR, "%s: close(%d, %s) failed: error=%s", ROOTD_SERVER, iter->first, iter->second.cmd_fifo_path.c_str(), strerror(errno));
		}
		
		//Close and unlink command FIFO
		err = close(iter->second.command_pipe_fd);
		if (err < 0) {
			syslog(LOG_ERR, "%s: close(%d, %s) failed: error=%s", ROOTD_SERVER, iter->second.command_pipe_fd, iter->second.cmd_fifo_path.c_str(), strerror(errno));
		}
		/* remove the named pipe */
		err = unlink(iter->second.cmd_fifo_path.c_str());
		if (err < 0) {
			syslog(LOG_ERR, "%s: unlink(%s) failed: error=%s", ROOTD_SERVER, iter->second.cmd_fifo_path.c_str(), strerror(errno));
		}
		
		//Close and unlink result FIFO
		err = close(iter->second.result_pipe_fd);
		if (err < 0) {
			syslog(LOG_ERR, "%s: close(%d, %s) failed: error=%s", ROOTD_SERVER, iter->second.result_pipe_fd, iter->second.result_fifo_path.c_str(), strerror(errno));
		}
		/* remove the named pipe */
		err = unlink(iter->second.result_fifo_path.c_str());
		if (err < 0) {
			syslog(LOG_ERR, "%s: unlink(%s) failed: error=%s", ROOTD_SERVER, iter->second.result_fifo_path.c_str(), strerror(errno));
		}
	}
}
uid_t RootDaemonServer::getUidByName(const char *userName)
{
    if(userName) {
        struct passwd *pwd = getpwnam(userName); /* don't free, see getpwnam() for details */
        if(pwd) return pwd->pw_uid;
    }
    return -1;
}

uid_t RootDaemonServer::getGidByName(const char *groupName)
{
    if(groupName) {
        struct group *grp = getgrnam(groupName); /* don't free, see getgrnam() for details */
        if(grp) return grp->gr_gid;
    }
    return -1;
}

int RootDaemonServer::createClientServerPipe(const string & pipePath, const int uid, const int gid, int flags)
{
	TRACE(LOG_INFO,"entered RootDaemonServer::createClientServerPipe, creating [%s]",pipePath.c_str());
   int err = -1;
	int commandFd;
   
   //clear the default umask (needed to correctly set permissions for mkfifo)
   umask(0);
	/* create the pipe */
	unlink(pipePath.c_str());
   TRACE(LOG_INFO,"unlinked[%s]",pipePath.c_str());
	err = mkfifo(pipePath.c_str(), S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
	if (err < 0) {
		syslog(LOG_ERR, "%s: mkfifo(%s) failed: error=%s", ROOTD_SERVER, pipePath.c_str(), strerror(errno));
		goto error;
	}
   TRACE(LOG_INFO, "%s: mkfifo(%s) success", ROOTD_SERVER, pipePath.c_str());
   
	err = chown(pipePath.c_str(), uid, gid);
	if (err < 0) {
		syslog(LOG_ERR, "%s: chown(%s) failed: error=%s", ROOTD_SERVER, pipePath.c_str(), strerror(errno));
		goto error;
	}
   TRACE(LOG_INFO,"%s: owner changed for[%s]",ROOTD_SERVER, pipePath.c_str());
	
	/* open the pipe */
   TRACE(LOG_INFO,"opening pipe[%s]",pipePath.c_str());
	commandFd = open(pipePath.c_str(), flags);
	if (commandFd < 0) {
		syslog(LOG_ERR, "%s: open(%s, ...) failed: error=%s", ROOTD_SERVER, pipePath.c_str(), strerror(errno));
		goto error;
	}
   TRACE(LOG_INFO,"%s: pipe opened[%s] commandFd[%d]",ROOTD_SERVER, pipePath.c_str(), commandFd);

	return commandFd;
	
	error:
		unlink(pipePath.c_str());
		return -1;
}
 
int RootDaemonServer::waitForCommands()
{
	int res = -1;
	int commandsFound = 0;
	int rc;
	cmd_thread_params_t * threadParams;
	map<int, client_lib_handles_t>::iterator iter;
	fd_set commandFds;
	CmdData cmd_data;
	
	/* endless until end of daemon */
	while(1) {
	
		int maxFdNumber = 0;

		/* setup the select */
		FD_ZERO(&commandFds);
		for(iter = pipeCommandMap_ptr->begin(); iter != pipeCommandMap_ptr->end(); ++iter)
		{
			FD_SET(iter->first, &commandFds);
			maxFdNumber = (iter->first > maxFdNumber ? iter->first : maxFdNumber);
		}

		TRACE(LOG_INFO, "%s: starting select for commandFds, maxFdNumber=%d", ROOTD_SERVER, maxFdNumber);

		/* wait indefinitely for a command */
		res = select(maxFdNumber+1, &commandFds, NULL, NULL, NULL);
		if (res == -1) {
			syslog(LOG_ERR, "%s: select() failed: error=%s", ROOTD_SERVER, strerror(errno));
			if(errno != EINTR)
			break;
		} else if (res > 0) {
			//find which command pipe fd was set
			commandsFound = 0;
			for(iter = pipeCommandMap_ptr->begin(); iter != pipeCommandMap_ptr->end(); ++iter)
			{
				if (FD_ISSET(iter->first, &commandFds)) 
				{
					threadParams = new cmd_thread_params_t();
					if(threadParams != NULL) //CID 830011
					{
						FD_CLR(iter->first, &commandFds);	
						TRACE(LOG_INFO, "%s: waitForCommands(), reading data from command FIFO [%s]", ROOTD_SERVER, iter->second.cmd_fifo_path.c_str());
							
						cmd_data =  RootDaemonHelper::readDataFromFIFO(iter->first, ROOTD_SERVER);
					   
						TRACE(LOG_INFO, "%s: waitForCommands() found in FIFO command, num[%d], args[%s]", ROOTD_SERVER,cmd_data.cmdNo, cmd_data.message);
							
						threadParams->command_num = cmd_data.cmdNo;
						threadParams->command_args = string(cmd_data.message);
						threadParams->cmd_func_ptr = iter->second.cmd_func_ptr;
						threadParams->res_pipe_fd =  iter->second.result_pipe_fd;	
						//Kick off a new thread to execute the command
						pthread_t thread_id;
						pthread_attr_t attr;
						pthread_attr_init(&attr);
						pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
						rc = pthread_create(&thread_id, &attr, thread_callback, (void *)threadParams);
						if(rc)
						{
							syslog(LOG_ERR, "%s: pthread_create() failed for client FIFO[%s]: error=%s", ROOTD_SERVER, 
							iter->second.cmd_fifo_path.c_str(), strerror(rc));
							if(threadParams != NULL)
								delete threadParams;
						}
						else
						{
							TRACE(LOG_INFO, "%s: pthread_create() success for client FIFO[%s]", ROOTD_SERVER, iter->second.cmd_fifo_path.c_str());
						}
					}
					else
					{
						syslog(LOG_ERR, "Memory allocation failed for cmd_thread_params_t");
					}
						commandsFound++;
				}
			} //for
			if(commandsFound == 0)
			{	
				syslog(LOG_ERR, "%s: couldn't found the changed command pipe fd", ROOTD_SERVER);
			}
		}
	}

	return res;
}
 

int main(int argc, char **argv)
{
	try{	 
		(void)(argc);
		(void)(argv);
		RootDaemonServer::start();
		pthread_exit(NULL);
	 }
	 catch(...)
	 {
		syslog(LOG_ERR, "Caught exception"); 
	 }
}
