/*
 * RootDaemonHelper.h
 *
 * The RootDaemonHelper class has static methods usable from root daemon client side 
 * to send commands to the root daemon server and receive the result via Fifos
 *
 *  Created on: Oct 24, 2014
 *      Author: Mohamed Mostafa
 */

#ifndef _ROOT_DAEMON_HELPER_H_
#define _ROOT_DAEMON_HELPER_H_

//#define DEBUG //will enable syslog(LOG_INFO, ...) traces

#ifdef DEBUG
#define TRACE(...) syslog(__VA_ARGS__)
#else
#define TRACE(...)
#endif

using namespace std;

#include "RootDaemonTypes.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <syslog.h>
#include <pthread.h>
#include <errno.h>
#include <string>
#include <string.h>

static pthread_mutex_t lock = PTHREAD_MUTEX_INITIALIZER;
const char * tmp_rootdeamon_dir = "/tmp/rootdaemon/";

class RootDaemonHelper
{
public:
	static const string getCmdPipePath(const char * clientName);
	static const string getResultPipePath(const char * clientName);
	static CmdData performRootOp(const char * clientName, const int cmdNum, string args, const long timeOut = 10);
   static int writeCommandResultToFIFO(int fifo_fd, const char * processName, int cmdNum, char * cmd_result_payload);
	static int writeCommandRequestToFIFO(int fifo_fd, const char * processName, int cmdNum, string cmd_args);
	static CmdData readDataFromFIFO(int fifo_fd, const char * process_name, const long timeOut = 10);
};
/**
    Return the command FIFO path

    @param clientName: root daemon client name
    @return command FIFO path
*/
const string RootDaemonHelper::getCmdPipePath(const char * clientName)
{
	return string(tmp_rootdeamon_dir + string(clientName) + "_pipe.cmd");
}
/**
    Return the command result FIFO path

    @param clientName: root daemon client name
    @return command result FIFO path
*/
const string RootDaemonHelper::getResultPipePath(const char * clientName)
{
	return string(tmp_rootdeamon_dir + string(clientName) + "_pipe.result");
}
/**
   Performs an operation that requires root privilege by sending 
	command number and arguments to root daemon server and returning the result

   @param clientName: root daemon client name
	@param cmdNum: the number of command to execute
	@param args: the command arguments 
   @return The result of the command
*/
CmdData RootDaemonHelper::performRootOp(const char * clientName, const int cmdNum, string args, const long timeOut)
{
	CmdData cmd_result = {0,0,0,""};//CID 192462
	string cmd_pipe_path;
	string result_pipe_path;
	int cmdFd = -1;
	int resultFd = -1;
	
	pthread_mutex_lock(&lock);
	
	//TODO busywait keep trying to open cmd,result pipes till server create them?
	
	//Open command and result FIFOs
	cmd_pipe_path = getCmdPipePath(clientName);
	cmdFd = open(cmd_pipe_path.c_str(), O_WRONLY);
    if (cmdFd < 0) {
		syslog(LOG_ERR, "%s: open(%s, ...) failed: error=%s", clientName, cmd_pipe_path.c_str(), strerror(errno));
        cmd_result.errorNo = ERR_NO_CMD_PIPE;
		goto cleanup;
    }
	result_pipe_path = getResultPipePath(clientName);
	resultFd = open(getResultPipePath(clientName).c_str(), O_RDONLY);
    if (resultFd < 0) {
        syslog(LOG_ERR, "%s: open(%s, ...) failed: error=%s", clientName, result_pipe_path.c_str(), strerror(errno));
		cmd_result.errorNo = ERR_NO_RESULT_PIPE;
		goto cleanup;
    }
   
	//1.Send the Command to Server Root Daemon
	if(writeCommandRequestToFIFO(cmdFd, clientName, cmdNum, args) < 0 )
	{
		cmd_result.errorNo = ERR_WRITE_TO_CMD_PIPE;
		goto cleanup;
	}
	TRACE(LOG_INFO, "%s: before readDataFromFIFO, resultfd[%d]", clientName,resultFd );
   
	//2.Read command result from result FIFO (Will timeout if no response within 5 seconds)
	cmd_result = readDataFromFIFO(resultFd, clientName, timeOut);
	if (cmd_result.errorNo != ERR_NONE) {
        syslog(LOG_ERR, "%s: RootDaemonHelper::performRootOp(), readDataFromFIFO(%d, %s) failed: cmd_result_error=%d", clientName, resultFd, result_pipe_path.c_str(), cmd_result.errorNo);
        cmd_result.errorNo = ERR_READ_FROM_RESULT_PIPE;
		goto cleanup;
    }
	
	cleanup: 
	if(cmdFd != -1)
	{
		close(cmdFd);
		cmdFd = -1;
	}
	if(resultFd != -1)
	{
		close(resultFd);
		resultFd = -1;
	}
	
	pthread_mutex_unlock(&lock);
	
	return cmd_result;
}
/**
   send command result data over a FIFO in the form of CmdData

   @param fifo_fd: FIFO handle to write to
   @param processName: the calling process name to show in logs
   @param cmdNum: the command number
   @param cmd_result_payload: command result as an array of bytes to write to the FIFO
   @return 0 on success, errno on failure
*/
int RootDaemonHelper::writeCommandResultToFIFO(int fifo_fd, const char * processName, int cmdNum, char * cmd_result_payload)
{
   TRACE(LOG_INFO, "%s: entered RootDaemonHelper::writeCommandResultToFIFO", processName);
   CmdData fifo_data = {0,0,0,""}; //CID 17608
   int errorNum;
   if(NULL == cmd_result_payload)
   {
      syslog(LOG_ERR, "%s:writeCommandResultToFIFO: failed to write to fifo{%d}: cmd_result_payload is NULL", processName, fifo_fd);
      return -1;
   }
   //build up the fifo_data and argument message
   fifo_data.cmdNo = cmdNum;
   strncpy(fifo_data.message, cmd_result_payload, (sizeof(fifo_data.message) - 1));//CID 10119
   fifo_data.message[sizeof(fifo_data.message) -1 ] = '\0';
   fifo_data.cmdDataSize = sizeof(CmdData);
   TRACE(LOG_INFO, "%s: writeCommandResultToFIFO: sizeof(CmdData)[%ld], sizeof(fifo_data.message)[%ld]", processName, sizeof(CmdData), sizeof(fifo_data.message));

   //write the fifo_data to the FIFO
   ssize_t sentBytes;
   sentBytes = write(fifo_fd, &fifo_data, fifo_data.cmdDataSize);
   errorNum = errno;
   if (sentBytes != fifo_data.cmdDataSize) {
      //couldn't write all bytes
      syslog(LOG_ERR, "%s: write to fifo{%d} failed: error=%s, commandNum=%d, args=%s", processName, fifo_fd, 
             strerror(errorNum), fifo_data.cmdNo, fifo_data.message);
      return errorNum;
   }
   else
   {
      //success
      return 0;
   }
}
/**
   send command request data over a FIFO in the form of CmdData

   @param fifo_fd: FIFO handle to write to
   @param processName: the calling process name to show in logs
   @param cmdNum: the command number
   @param args: arguments(string) of command to execute
   @return 0 on success, errno on failure
*/
int RootDaemonHelper::writeCommandRequestToFIFO(int fifo_fd, const char * processName, int cmdNum, string cmd_args)
{
   TRACE(LOG_INFO, "%s: entered RootDaemonHelper::writeCommandRequestToFIFO", processName);
   CmdData fifo_data = {0,0,0,""}; //CID 30063
   int errorNum;
   //build up the fifo_data and argument message
   fifo_data.cmdNo = cmdNum;
   strncpy(fifo_data.message, cmd_args.c_str(), sizeof(fifo_data.message));
   fifo_data.message[sizeof(fifo_data.message)-1] = 0;
   fifo_data.cmdDataSize = (int)(sizeof(CmdData) - sizeof(fifo_data.message) + strlen(fifo_data.message) +1);
   
   TRACE(LOG_INFO, "%s:writeCommandRequestToFIFO:sizeof(CmdData)[%ld], sizeof(fifo_data.message)[%ld], strlen(fifo_data.message)[%ld]", processName, sizeof(CmdData), sizeof(fifo_data.message), strlen(fifo_data.message));

   //write the fifo_data to the FIFO
   ssize_t sentBytes;
   sentBytes = write(fifo_fd, &fifo_data, fifo_data.cmdDataSize);
   errorNum = errno;
   if (sentBytes != fifo_data.cmdDataSize) {
      //couldn't write all bytes
      syslog(LOG_ERR, "%s: write to fifo{%d} failed: error=%s, commandNum=%d, cmd_args=%s", processName, fifo_fd, 
             strerror(errorNum), fifo_data.cmdNo, fifo_data.message);
      return errorNum;
   }
   else
   {
      //success
      return 0;
   }
}
/**
   read data (command data or command result) from a FIFO in the form of CmdData

   @param fifo_fd: FIFO handle to read from
   @param processName: the calling process name to show in logs
   @return data read from FIFO (command data or command result)
*/
CmdData RootDaemonHelper::readDataFromFIFO(int fifo_fd, const char * processName, const long timeOut)
{
   TRACE(LOG_INFO, "%s: entered RootDaemonHelper::readDataFromFIFO", processName); 
   int ret = ERR_PIPE_SELECT;
   CmdData cmd_data;
   ssize_t maxReadLen = sizeof(cmd_data);
   ssize_t nBytesRead;
   unsigned char *readPtr = (unsigned char *)&cmd_data;
   int i = 0;
   struct timeval tv; // timeout for reading the fifo data parts
   int error=0;

   tv.tv_sec = timeOut;
   tv.tv_usec = 0;
	
	//parse cmd_data from the pipe
   while(1) {

      /* setup the select */
      fd_set fds;
      FD_ZERO(&fds);
      FD_SET(fifo_fd, &fds);

      //TRACE(LOG_INFO, "%s: start select for fifo_fd=%d", processName, fifo_fd);

      //wait for a (part of) of the FIFO data
      //timeout value should exist between reading further parts of FIFO data
      if (-1 == timeOut)
      {
         ret = select(fifo_fd+1, &fds, NULL, NULL, NULL);
      }
      else
      {
         ret = select(fifo_fd+1, &fds, NULL, NULL, &tv);
      }
      error = (ret == -1) ? errno : 0;
      TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO() found part[%d] of command, parts timeout[%ld], ret[%d], error[%d]", processName, i, tv.tv_sec, ret, error);
      
      if (ret == -1) {
         syslog(LOG_ERR, "%s: RootDaemonHelper::readDataFromFIFO(), select() failed: error=%s", processName, strerror(error));
         cmd_data.errorNo = ERR_PIPE_SELECT;
         return cmd_data;
      //break;
      } else if (ret > 0) { // data available
      } else {
         syslog(LOG_ERR, "%s: RootDaemonHelper::readDataFromFIFO(), select() timeout: no more data", processName);
         cmd_data.errorNo = ERR_INCOMPLETE_DATA_IN_PIPE;
         return cmd_data;
      }

      TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), data available reading FIFO fd[%d]", processName, fifo_fd);
      /* do a read (must not hang because select was successful before) */
      nBytesRead = read(fifo_fd, readPtr, maxReadLen);
      error = errno;
      TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), success reading FIFO with nBytesRead[%ld] maxReadLen[%ld]", processName, nBytesRead, maxReadLen);
      /* end of file? */
      if (nBytesRead == 0) { // EOF
         TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), breaking, read FIFO with nBytesRead[%ld]", processName, nBytesRead);
         break;
      }

      /* error and NOT a signal? */
      if (nBytesRead < 0 && error != EINTR) {
         TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), breaking, nBytesRead[%ld], error[%d]", processName, nBytesRead, error);
         break;
      }

      /* was the message len read in? */
      if (nBytesRead >= (ssize_t)sizeof(cmd_data.cmdDataSize)) {
         TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), nBytesRead[%ld]", processName, nBytesRead);
         /* use this nBytesRead to check if the complete message was read in */
         if (nBytesRead >= cmd_data.cmdDataSize) {
            TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), breaking, nBytesRead[%ld] >= cmd_data.cmdDataSize[%d]", processName, nBytesRead, cmd_data.cmdDataSize);
            /* complete message read, break read loop */
            break;
         }
      }

      /* not all read in, try next */
      readPtr += nBytesRead;
      maxReadLen -= nBytesRead;
      i++;
   }

   /* read error? */
   if (nBytesRead <= 0) {
      syslog(LOG_ERR, "%s: RootDaemonHelper::readDataFromFIFO(), read() failed: nBytesRead[%zu]", processName, nBytesRead);
      cmd_data.errorNo = ERR_PIPE_SELECT;
   }
   else
   {
      TRACE(LOG_INFO, "%s: RootDaemonHelper::readDataFromFIFO(), returning, no errors ", processName);
      cmd_data.errorNo = ERR_NONE;
   }
   return cmd_data;
}

#endif //#ifndef _ROOT_DAEMON_HELPER_H_
